"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sns = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [sns](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsns.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sns extends shared_1.PolicyStatement {
    /**
     * Grants permission to add a statement to a topic's access control policy, granting access for the specified AWS accounts to the specified actions
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_AddPermission.html
     */
    toAddPermission() {
        return this.to('AddPermission');
    }
    /**
     * Grants permission to accept a phone number and indicate whether the phone holder has opted out of receiving SMS messages from your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CheckIfPhoneNumberIsOptedOut.html
     */
    toCheckIfPhoneNumberIsOptedOut() {
        return this.to('CheckIfPhoneNumberIsOptedOut');
    }
    /**
     * Grants permission to verify an endpoint owner's intent to receive messages by validating the token sent to the endpoint by an earlier Subscribe action
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ConfirmSubscription.html
     */
    toConfirmSubscription() {
        return this.to('ConfirmSubscription');
    }
    /**
     * Grants permission to create a platform application object for one of the supported push notification services, such as APNS and GCM, to which devices and mobile apps may register
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformApplication.html
     */
    toCreatePlatformApplication() {
        return this.to('CreatePlatformApplication');
    }
    /**
     * Grants permission to create an endpoint for a device and mobile app on one of the supported push notification services, such as GCM and APNS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreatePlatformEndpoint.html
     */
    toCreatePlatformEndpoint() {
        return this.to('CreatePlatformEndpoint');
    }
    /**
     * Grants permission to add a destination phone number and send a one-time password (OTP) to that phone number for an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreateSMSSandboxPhoneNumber.html
     */
    toCreateSMSSandboxPhoneNumber() {
        return this.to('CreateSMSSandboxPhoneNumber');
    }
    /**
     * Grants permission to create a topic to which notifications can be published
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_CreateTopic.html
     */
    toCreateTopic() {
        return this.to('CreateTopic');
    }
    /**
     * Grants permission to delete the endpoint for a device and mobile app from Amazon SNS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteEndpoint.html
     */
    toDeleteEndpoint() {
        return this.to('DeleteEndpoint');
    }
    /**
     * Grants permission to delete a platform application object for one of the supported push notification services, such as APNS and GCM
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeletePlatformApplication.html
     */
    toDeletePlatformApplication() {
        return this.to('DeletePlatformApplication');
    }
    /**
     * Grants permission to delete an AWS account's verified or pending phone number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteSMSSandboxPhoneNumber.html
     */
    toDeleteSMSSandboxPhoneNumber() {
        return this.to('DeleteSMSSandboxPhoneNumber');
    }
    /**
     * Grants permission to delete a topic and all its subscriptions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_DeleteTopic.html
     */
    toDeleteTopic() {
        return this.to('DeleteTopic');
    }
    /**
     * Grants permission to return the data protection policy of the topic
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetDataProtectionPolicy.html
     */
    toGetDataProtectionPolicy() {
        return this.to('GetDataProtectionPolicy');
    }
    /**
     * Grants permission to retrieve the endpoint attributes for a device on one of the supported push notification services, such as GCM and APNS
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetEndpointAttributes.html
     */
    toGetEndpointAttributes() {
        return this.to('GetEndpointAttributes');
    }
    /**
     * Grants permission to retrieve the attributes of the platform application object for the supported push notification services, such as APNS and GCM
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetPlatformApplicationAttributes.html
     */
    toGetPlatformApplicationAttributes() {
        return this.to('GetPlatformApplicationAttributes');
    }
    /**
     * Grants permission to return the settings for sending SMS messages from your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSMSAttributes.html
     */
    toGetSMSAttributes() {
        return this.to('GetSMSAttributes');
    }
    /**
     * Grants permission to retrieve the sandbox status for the calling account in the target region
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSMSSandboxAccountStatus.html
     */
    toGetSMSSandboxAccountStatus() {
        return this.to('GetSMSSandboxAccountStatus');
    }
    /**
     * Grants permission to return all of the properties of a subscription
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetSubscriptionAttributes.html
     */
    toGetSubscriptionAttributes() {
        return this.to('GetSubscriptionAttributes');
    }
    /**
     * Grants permission to return all of the properties of a topic
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_GetTopicAttributes.html
     */
    toGetTopicAttributes() {
        return this.to('GetTopicAttributes');
    }
    /**
     * Grants permission to list the endpoints and endpoint attributes for devices in a supported push notification service, such as GCM and APNS
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListEndpointsByPlatformApplication.html
     */
    toListEndpointsByPlatformApplication() {
        return this.to('ListEndpointsByPlatformApplication');
    }
    /**
     * Grants permission to list all origination numbers, and their metadata
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListOriginationNumbers.html
     */
    toListOriginationNumbers() {
        return this.to('ListOriginationNumbers');
    }
    /**
     * Grants permission to return a list of phone numbers that are opted out, meaning you cannot send SMS messages to them
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPhoneNumbersOptedOut.html
     */
    toListPhoneNumbersOptedOut() {
        return this.to('ListPhoneNumbersOptedOut');
    }
    /**
     * Grants permission to list the platform application objects for the supported push notification services, such as APNS and GCM
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListPlatformApplications.html
     */
    toListPlatformApplications() {
        return this.to('ListPlatformApplications');
    }
    /**
     * Grants permission to list the calling account's current pending and verified destination phone numbers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSMSSandboxPhoneNumbers.html
     */
    toListSMSSandboxPhoneNumbers() {
        return this.to('ListSMSSandboxPhoneNumbers');
    }
    /**
     * Grants permission to return a list of the requester's subscriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptions.html
     */
    toListSubscriptions() {
        return this.to('ListSubscriptions');
    }
    /**
     * Grants permission to return a list of the subscriptions to a specific topic
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListSubscriptionsByTopic.html
     */
    toListSubscriptionsByTopic() {
        return this.to('ListSubscriptionsByTopic');
    }
    /**
     * Grants permission to list all tags added to the specified Amazon SNS topic
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to return a list of the requester's topics
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_ListTopics.html
     */
    toListTopics() {
        return this.to('ListTopics');
    }
    /**
     * Grants permission to opt in a phone number that is currently opted out, which enables you to resume sending SMS messages to the number
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_OptInPhoneNumber.html
     */
    toOptInPhoneNumber() {
        return this.to('OptInPhoneNumber');
    }
    /**
     * Grants permission to send a message to all of a topic's subscribed endpoints
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Publish.html
     */
    toPublish() {
        return this.to('Publish');
    }
    /**
     * Grants permission to allow a topic owner to set the data protection policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_PutDataProtectionPolicy.html
     */
    toPutDataProtectionPolicy() {
        return this.to('PutDataProtectionPolicy');
    }
    /**
     * Grants permission to remove a statement from a topic's access control policy
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_RemovePermission.html
     */
    toRemovePermission() {
        return this.to('RemovePermission');
    }
    /**
     * Grants permission to set the attributes for an endpoint for a device on one of the supported push notification services, such as GCM and APNS
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetEndpointAttributes.html
     */
    toSetEndpointAttributes() {
        return this.to('SetEndpointAttributes');
    }
    /**
     * Grants permission to set the attributes of the platform application object for the supported push notification services, such as APNS and GCM
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetPlatformApplicationAttributes.html
     */
    toSetPlatformApplicationAttributes() {
        return this.to('SetPlatformApplicationAttributes');
    }
    /**
     * Grants permission to set the default settings for sending SMS messages and receiving daily SMS usage reports
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSMSAttributes.html
     */
    toSetSMSAttributes() {
        return this.to('SetSMSAttributes');
    }
    /**
     * Grants permission to allow a subscription owner to set an attribute of the topic to a new value
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetSubscriptionAttributes.html
     */
    toSetSubscriptionAttributes() {
        return this.to('SetSubscriptionAttributes');
    }
    /**
     * Grants permission to allow a topic owner to set an attribute of the topic to a new value
     *
     * Access Level: Permissions management
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_SetTopicAttributes.html
     */
    toSetTopicAttributes() {
        return this.to('SetTopicAttributes');
    }
    /**
     * Grants permission to prepare to subscribe an endpoint by sending the endpoint a confirmation message
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifEndpoint()
     * - .ifProtocol()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Subscribe.html
     */
    toSubscribe() {
        return this.to('Subscribe');
    }
    /**
     * Grants permission to add tags to the specified Amazon SNS topic
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to delete a subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_Unsubscribe.html
     */
    toUnsubscribe() {
        return this.to('Unsubscribe');
    }
    /**
     * Grants permission to remove tags from the specified Amazon SNS topic
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to verify a destination phone number with a one-time password (OTP) for an AWS account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/sns/latest/api/API_VerifySMSSandboxPhoneNumber.html
     */
    toVerifySMSSandboxPhoneNumber() {
        return this.to('VerifySMSSandboxPhoneNumber');
    }
    /**
     * Adds a resource of type topic to the statement
     *
     * https://docs.aws.amazon.com/sns/latest/dg/CreateTopic.html
     *
     * @param topicName - Identifier for the topicName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTopic(topicName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:sns:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${topicName}`);
    }
    /**
     * Filters access by tags from request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateTopic()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - topic
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys from request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateTopic()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the URL, email address, or ARN from a Subscribe request or a previously confirmed subscription
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * Applies to actions:
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifEndpoint(value, operator) {
        return this.if(`Endpoint`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the protocol value from a Subscribe request or a previously confirmed subscription
     *
     * https://docs.aws.amazon.com/sns/latest/dg/UsingIAMwithSNS.html#w2ab1c11c23c19
     *
     * Applies to actions:
     * - .toSubscribe()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifProtocol(value, operator) {
        return this.if(`Protocol`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [sns](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonsns.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'sns';
        this.accessLevelList = {
            'Permissions management': [
                'AddPermission',
                'RemovePermission',
                'SetTopicAttributes'
            ],
            Read: [
                'CheckIfPhoneNumberIsOptedOut',
                'GetDataProtectionPolicy',
                'GetEndpointAttributes',
                'GetPlatformApplicationAttributes',
                'GetSMSAttributes',
                'GetSMSSandboxAccountStatus',
                'GetSubscriptionAttributes',
                'GetTopicAttributes',
                'ListPhoneNumbersOptedOut',
                'ListTagsForResource'
            ],
            Write: [
                'ConfirmSubscription',
                'CreatePlatformApplication',
                'CreatePlatformEndpoint',
                'CreateSMSSandboxPhoneNumber',
                'CreateTopic',
                'DeleteEndpoint',
                'DeletePlatformApplication',
                'DeleteSMSSandboxPhoneNumber',
                'DeleteTopic',
                'OptInPhoneNumber',
                'Publish',
                'PutDataProtectionPolicy',
                'SetEndpointAttributes',
                'SetPlatformApplicationAttributes',
                'SetSMSAttributes',
                'SetSubscriptionAttributes',
                'Subscribe',
                'Unsubscribe',
                'VerifySMSSandboxPhoneNumber'
            ],
            List: [
                'ListEndpointsByPlatformApplication',
                'ListOriginationNumbers',
                'ListPlatformApplications',
                'ListSMSSandboxPhoneNumbers',
                'ListSubscriptions',
                'ListSubscriptionsByTopic',
                'ListTopics'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Sns = Sns;
//# sourceMappingURL=data:application/json;base64,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