"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Storagegateway = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [storagegateway](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsstoragegateway.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Storagegateway extends shared_1.PolicyStatement {
    /**
     * Grants permission to activate the gateway you previously deployed on your host
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ActivateGateway.html
     */
    toActivateGateway() {
        return this.to('ActivateGateway');
    }
    /**
     * Grants permission to configure one or more gateway local disks as cache for a cached-volume gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddCache.html
     */
    toAddCache() {
        return this.to('AddCache');
    }
    /**
     * Grants permission to add one or more tags to the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        return this.to('AddTagsToResource');
    }
    /**
     * Grants permission to configure one or more gateway local disks as upload buffer for a specified gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddUploadBuffer.html
     */
    toAddUploadBuffer() {
        return this.to('AddUploadBuffer');
    }
    /**
     * Grants permission to configure one or more gateway local disks as working storage for a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddWorkingStorage.html
     */
    toAddWorkingStorage() {
        return this.to('AddWorkingStorage');
    }
    /**
     * Grants permission to move a tape to the target pool specified
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AssignTapePool.html
     */
    toAssignTapePool() {
        return this.to('AssignTapePool');
    }
    /**
     * Grants permission to associate an Amazon FSx file system with the Amazon FSx file gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     * - ec2:DescribeNetworkInterfaces
     * - fsx:DescribeFileSystems
     * - iam:CreateServiceLinkedRole
     * - logs:CreateLogDelivery
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     * - logs:UpdateLogDelivery
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AssociateFileSystem.html
     */
    toAssociateFileSystem() {
        return this.to('AssociateFileSystem');
    }
    /**
     * Grants permission to connect a volume to an iSCSI connection and then attaches the volume to the specified gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AttachVolume.html
     */
    toAttachVolume() {
        return this.to('AttachVolume');
    }
    /**
     * Grants permission to allow the governance retention lock on a pool to be bypassed
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/CreatingCustomTapePool.html#TapeRetentionLock
     */
    toBypassGovernanceRetention() {
        return this.to('BypassGovernanceRetention');
    }
    /**
     * Grants permission to cancel archiving of a virtual tape to the virtual tape shelf (VTS) after the archiving process is initiated
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CancelArchival.html
     */
    toCancelArchival() {
        return this.to('CancelArchival');
    }
    /**
     * Grants permission to cancel a cache report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CancelCacheReport.html
     */
    toCancelCacheReport() {
        return this.to('CancelCacheReport');
    }
    /**
     * Grants permission to cancel retrieval of a virtual tape from the virtual tape shelf (VTS) to a gateway after the retrieval process is initiated
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CancelRetrieval.html
     */
    toCancelRetrieval() {
        return this.to('CancelRetrieval');
    }
    /**
     * Grants permission to create a cached volume on a specified cached gateway. This operation is supported only for the gateway-cached volume architecture
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateCachediSCSIVolume.html
     */
    toCreateCachediSCSIVolume() {
        return this.to('CreateCachediSCSIVolume');
    }
    /**
     * Grants permission to create a NFS file share on an existing file gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateNFSFileShare.html
     */
    toCreateNFSFileShare() {
        return this.to('CreateNFSFileShare');
    }
    /**
     * Grants permission to create a SMB file share on an existing file gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateSMBFileShare.html
     */
    toCreateSMBFileShare() {
        return this.to('CreateSMBFileShare');
    }
    /**
     * Grants permission to initiate a snapshot of a volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        return this.to('CreateSnapshot');
    }
    /**
     * Grants permission to initiate a snapshot of a gateway from a volume recovery point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateSnapshotFromVolumeRecoveryPoint.html
     */
    toCreateSnapshotFromVolumeRecoveryPoint() {
        return this.to('CreateSnapshotFromVolumeRecoveryPoint');
    }
    /**
     * Grants permission to create a volume on a specified gateway
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateStorediSCSIVolume.html
     */
    toCreateStorediSCSIVolume() {
        return this.to('CreateStorediSCSIVolume');
    }
    /**
     * Grants permission to create a tape pool
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateTapePool.html
     */
    toCreateTapePool() {
        return this.to('CreateTapePool');
    }
    /**
     * Grants permission to create a virtual tape by using your own barcode
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateTapeWithBarcode.html
     */
    toCreateTapeWithBarcode() {
        return this.to('CreateTapeWithBarcode');
    }
    /**
     * Grants permission to create one or more virtual tapes. You write data to the virtual tapes and then archive the tapes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateTapes.html
     */
    toCreateTapes() {
        return this.to('CreateTapes');
    }
    /**
     * Grants permission to delete the automatic tape creation policy configured on a gateway-VTL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteAutomaticTapeCreationPolicy.html
     */
    toDeleteAutomaticTapeCreationPolicy() {
        return this.to('DeleteAutomaticTapeCreationPolicy');
    }
    /**
     * Grants permission to delete the bandwidth rate limits of a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteBandwidthRateLimit.html
     */
    toDeleteBandwidthRateLimit() {
        return this.to('DeleteBandwidthRateLimit');
    }
    /**
     * Grants permission to delete the metadata associated with a cache report
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteCacheReport.html
     */
    toDeleteCacheReport() {
        return this.to('DeleteCacheReport');
    }
    /**
     * Grants permission to delete Challenge-Handshake Authentication Protocol (CHAP) credentials for a specified iSCSI target and initiator pair
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteChapCredentials.html
     */
    toDeleteChapCredentials() {
        return this.to('DeleteChapCredentials');
    }
    /**
     * Grants permission to delete a file share from a file gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteFileShare.html
     */
    toDeleteFileShare() {
        return this.to('DeleteFileShare');
    }
    /**
     * Grants permission to delete a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteGateway.html
     */
    toDeleteGateway() {
        return this.to('DeleteGateway');
    }
    /**
     * Grants permission to delete a snapshot of a volume
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteSnapshotSchedule.html
     */
    toDeleteSnapshotSchedule() {
        return this.to('DeleteSnapshotSchedule');
    }
    /**
     * Grants permission to delete the specified virtual tape
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteTape.html
     */
    toDeleteTape() {
        return this.to('DeleteTape');
    }
    /**
     * Grants permission to delete the specified virtual tape from the virtual tape shelf (VTS)
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteTapeArchive.html
     */
    toDeleteTapeArchive() {
        return this.to('DeleteTapeArchive');
    }
    /**
     * Grants permission to delete the specified tape pool
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteTapePool.html
     */
    toDeleteTapePool() {
        return this.to('DeleteTapePool');
    }
    /**
     * Grants permission to delete the specified gateway volume that you previously created using the CreateCachediSCSIVolume or CreateStorediSCSIVolume API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteVolume.html
     */
    toDeleteVolume() {
        return this.to('DeleteVolume');
    }
    /**
     * Grants permission to get the information about the most recent high availability monitoring test that was performed on the gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeAvailabilityMonitorTest.html
     */
    toDescribeAvailabilityMonitorTest() {
        return this.to('DescribeAvailabilityMonitorTest');
    }
    /**
     * Grants permission to get the bandwidth rate limits of a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeBandwidthRateLimit.html
     */
    toDescribeBandwidthRateLimit() {
        return this.to('DescribeBandwidthRateLimit');
    }
    /**
     * Grants permission to get the bandwidth rate limit schedule of a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeBandwidthRateLimitSchedule.html
     */
    toDescribeBandwidthRateLimitSchedule() {
        return this.to('DescribeBandwidthRateLimitSchedule');
    }
    /**
     * Grants permission to get information about the cache of a gateway. This operation is supported only for the gateway-cached volume architecture
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeCache.html
     */
    toDescribeCache() {
        return this.to('DescribeCache');
    }
    /**
     * Grants permission to get a description of a cache report
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeCacheReport.html
     */
    toDescribeCacheReport() {
        return this.to('DescribeCacheReport');
    }
    /**
     * Grants permission to get a description of the gateway volumes specified in the request. This operation is supported only for the gateway-cached volume architecture
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeCachediSCSIVolumes.html
     */
    toDescribeCachediSCSIVolumes() {
        return this.to('DescribeCachediSCSIVolumes');
    }
    /**
     * Grants permission to get an array of Challenge-Handshake Authentication Protocol (CHAP) credentials information for a specified iSCSI target, one for each target-initiator pair
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeChapCredentials.html
     */
    toDescribeChapCredentials() {
        return this.to('DescribeChapCredentials');
    }
    /**
     * Grants permission to get a description for one or more file system associations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeFileSystemAssociations.html
     */
    toDescribeFileSystemAssociations() {
        return this.to('DescribeFileSystemAssociations');
    }
    /**
     * Grants permission to get metadata about a gateway such as its name, network interfaces, configured time zone, and the state (whether the gateway is running or not)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeGatewayInformation.html
     */
    toDescribeGatewayInformation() {
        return this.to('DescribeGatewayInformation');
    }
    /**
     * Grants permission to get your gateway's weekly maintenance start time including the day and time of the week
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeMaintenanceStartTime.html
     */
    toDescribeMaintenanceStartTime() {
        return this.to('DescribeMaintenanceStartTime');
    }
    /**
     * Grants permission to get a description for one or more file shares from a file gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeNFSFileShares.html
     */
    toDescribeNFSFileShares() {
        return this.to('DescribeNFSFileShares');
    }
    /**
     * Grants permission to get a description for one or more file shares from a file gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeSMBFileShares.html
     */
    toDescribeSMBFileShares() {
        return this.to('DescribeSMBFileShares');
    }
    /**
     * Grants permission to get a description of a Server Message Block (SMB) file share settings from a file gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeSMBSettings.html
     */
    toDescribeSMBSettings() {
        return this.to('DescribeSMBSettings');
    }
    /**
     * Grants permission to describe the snapshot schedule for the specified gateway volume
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeSnapshotSchedule.html
     */
    toDescribeSnapshotSchedule() {
        return this.to('DescribeSnapshotSchedule');
    }
    /**
     * Grants permission to get the description of the gateway volumes specified in the request
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeStorediSCSIVolumes.html
     */
    toDescribeStorediSCSIVolumes() {
        return this.to('DescribeStorediSCSIVolumes');
    }
    /**
     * Grants permission to get a description of specified virtual tapes in the virtual tape shelf (VTS)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeTapeArchives.html
     */
    toDescribeTapeArchives() {
        return this.to('DescribeTapeArchives');
    }
    /**
     * Grants permission to get a list of virtual tape recovery points that are available for the specified gateway-VTL
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeTapeRecoveryPoints.html
     */
    toDescribeTapeRecoveryPoints() {
        return this.to('DescribeTapeRecoveryPoints');
    }
    /**
     * Grants permission to get a description of the specified Amazon Resource Name (ARN) of virtual tapes
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeTapes.html
     */
    toDescribeTapes() {
        return this.to('DescribeTapes');
    }
    /**
     * Grants permission to get information about the upload buffer of a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeUploadBuffer.html
     */
    toDescribeUploadBuffer() {
        return this.to('DescribeUploadBuffer');
    }
    /**
     * Grants permission to get a description of virtual tape library (VTL) devices for the specified gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeVTLDevices.html
     */
    toDescribeVTLDevices() {
        return this.to('DescribeVTLDevices');
    }
    /**
     * Grants permission to get information about the working storage of a gateway
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeWorkingStorage.html
     */
    toDescribeWorkingStorage() {
        return this.to('DescribeWorkingStorage');
    }
    /**
     * Grants permission to disconnect a volume from an iSCSI connection and then detaches the volume from the specified gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DetachVolume.html
     */
    toDetachVolume() {
        return this.to('DetachVolume');
    }
    /**
     * Grants permission to disable a gateway when the gateway is no longer functioning
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DisableGateway.html
     */
    toDisableGateway() {
        return this.to('DisableGateway');
    }
    /**
     * Grants permission to disassociate an Amazon FSx file system from an Amazon FSx file gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DisassociateFileSystem.html
     */
    toDisassociateFileSystem() {
        return this.to('DisassociateFileSystem');
    }
    /**
     * Grants permission to clean a share's cache of file entries that are failing upload to Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_EvictFilesFailingUpload.html
     */
    toEvictFilesFailingUpload() {
        return this.to('EvictFilesFailingUpload');
    }
    /**
     * Grants permission to enable you to join an Active Directory Domain
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_JoinDomain.html
     */
    toJoinDomain() {
        return this.to('JoinDomain');
    }
    /**
     * Grants permission to list the automatic tape creation policies configured on the specified gateway-VTL or all gateway-VTLs owned by your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListAutomaticTapeCreationPolicies.html
     */
    toListAutomaticTapeCreationPolicies() {
        return this.to('ListAutomaticTapeCreationPolicies');
    }
    /**
     * Grants permission to get a list of the cache reports owned by your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListCacheReports.html
     */
    toListCacheReports() {
        return this.to('ListCacheReports');
    }
    /**
     * Grants permission to get a list of the file shares for a specific file gateway, or the list of file shares owned by your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListFileShares.html
     */
    toListFileShares() {
        return this.to('ListFileShares');
    }
    /**
     * Grants permission to get a list of the file system associations for the specified gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListFileSystemAssociations.html
     */
    toListFileSystemAssociations() {
        return this.to('ListFileSystemAssociations');
    }
    /**
     * Grants permission to list gateways owned by an AWS account in a region specified in the request. The returned list is ordered by gateway Amazon Resource Name (ARN)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListGateways.html
     */
    toListGateways() {
        return this.to('ListGateways');
    }
    /**
     * Grants permission to get a list of the gateway's local disks
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListLocalDisks.html
     */
    toListLocalDisks() {
        return this.to('ListLocalDisks');
    }
    /**
     * Grants permission to get the tags that have been added to the specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list tape pools owned by your AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListTapePools.html
     */
    toListTapePools() {
        return this.to('ListTapePools');
    }
    /**
     * Grants permission to list virtual tapes in your virtual tape library (VTL) and your virtual tape shelf (VTS)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListTapes.html
     */
    toListTapes() {
        return this.to('ListTapes');
    }
    /**
     * Grants permission to list iSCSI initiators that are connected to a volume
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListVolumeInitiators.html
     */
    toListVolumeInitiators() {
        return this.to('ListVolumeInitiators');
    }
    /**
     * Grants permission to list the recovery points for a specified gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListVolumeRecoveryPoints.html
     */
    toListVolumeRecoveryPoints() {
        return this.to('ListVolumeRecoveryPoints');
    }
    /**
     * Grants permission to list the iSCSI stored volumes of a gateway
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListVolumes.html
     */
    toListVolumes() {
        return this.to('ListVolumes');
    }
    /**
     * Grants permission to send you a notification through CloudWatch Events when all files written to your NFS file share have been uploaded to Amazon S3
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_NotifyWhenUploaded.html
     */
    toNotifyWhenUploaded() {
        return this.to('NotifyWhenUploaded');
    }
    /**
     * Grants permission to refresh the cache for the specified file share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RefreshCache.html
     */
    toRefreshCache() {
        return this.to('RefreshCache');
    }
    /**
     * Grants permission to remove one or more tags from the specified resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        return this.to('RemoveTagsFromResource');
    }
    /**
     * Grants permission to reset all cache disks that have encountered a error and makes the disks available for reconfiguration as cache storage
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ResetCache.html
     */
    toResetCache() {
        return this.to('ResetCache');
    }
    /**
     * Grants permission to retrieve an archived virtual tape from the virtual tape shelf (VTS) to a gateway-VTL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RetrieveTapeArchive.html
     */
    toRetrieveTapeArchive() {
        return this.to('RetrieveTapeArchive');
    }
    /**
     * Grants permission to retrieve the recovery point for the specified virtual tape
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RetrieveTapeRecoveryPoint.html
     */
    toRetrieveTapeRecoveryPoint() {
        return this.to('RetrieveTapeRecoveryPoint');
    }
    /**
     * Grants permission to set the password for your VM local console
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_SetLocalConsolePassword.html
     */
    toSetLocalConsolePassword() {
        return this.to('SetLocalConsolePassword');
    }
    /**
     * Grants permission to set the password for SMB Guest user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_SetSMBGuestPassword.html
     */
    toSetSMBGuestPassword() {
        return this.to('SetSMBGuestPassword');
    }
    /**
     * Grants permission to shut down a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ShutdownGateway.html
     */
    toShutdownGateway() {
        return this.to('ShutdownGateway');
    }
    /**
     * Grants permission to start a test that verifies that the specified gateway is configured for High Availability monitoring in your host environment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_StartAvailabilityMonitorTest.html
     */
    toStartAvailabilityMonitorTest() {
        return this.to('StartAvailabilityMonitorTest');
    }
    /**
     * Grants permission to start a cache report for an existing file share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_StartCacheReport.html
     */
    toStartCacheReport() {
        return this.to('StartCacheReport');
    }
    /**
     * Grants permission to start a gateway that you previously shut down
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_StartGateway.html
     */
    toStartGateway() {
        return this.to('StartGateway');
    }
    /**
     * Grants permission to update the automatic tape creation policy configured on a gateway-VTL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateAutomaticTapeCreationPolicy.html
     */
    toUpdateAutomaticTapeCreationPolicy() {
        return this.to('UpdateAutomaticTapeCreationPolicy');
    }
    /**
     * Grants permission to update the bandwidth rate limits of a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateBandwidthRateLimit.html
     */
    toUpdateBandwidthRateLimit() {
        return this.to('UpdateBandwidthRateLimit');
    }
    /**
     * Grants permission to update the bandwidth rate limit schedule of a gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateBandwidthRateLimitSchedule.html
     */
    toUpdateBandwidthRateLimitSchedule() {
        return this.to('UpdateBandwidthRateLimitSchedule');
    }
    /**
     * Grants permission to update the Challenge-Handshake Authentication Protocol (CHAP) credentials for a specified iSCSI target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateChapCredentials.html
     */
    toUpdateChapCredentials() {
        return this.to('UpdateChapCredentials');
    }
    /**
     * Grants permission to update a file system association
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - logs:CreateLogDelivery
     * - logs:DeleteLogDelivery
     * - logs:GetLogDelivery
     * - logs:ListLogDeliveries
     * - logs:UpdateLogDelivery
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateFileSystemAssociation.html
     */
    toUpdateFileSystemAssociation() {
        return this.to('UpdateFileSystemAssociation');
    }
    /**
     * Grants permission to update a gateway's metadata, which includes the gateway's name and time zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateGatewayInformation.html
     */
    toUpdateGatewayInformation() {
        return this.to('UpdateGatewayInformation');
    }
    /**
     * Grants permission to update the gateway virtual machine (VM) software
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateGatewaySoftwareNow.html
     */
    toUpdateGatewaySoftwareNow() {
        return this.to('UpdateGatewaySoftwareNow');
    }
    /**
     * Grants permission to update a gateway's weekly maintenance start time information, including day and time of the week. The maintenance time is the time in your gateway's time zone
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateMaintenanceStartTime.html
     */
    toUpdateMaintenanceStartTime() {
        return this.to('UpdateMaintenanceStartTime');
    }
    /**
     * Grants permission to update a NFS file share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateNFSFileShare.html
     */
    toUpdateNFSFileShare() {
        return this.to('UpdateNFSFileShare');
    }
    /**
     * Grants permission to update a SMB file share
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateSMBFileShare.html
     */
    toUpdateSMBFileShare() {
        return this.to('UpdateSMBFileShare');
    }
    /**
     * Grants permission to update whether the shares on a gateway are visible in a net view or browse list
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateSMBFileShareVisibility.html
     */
    toUpdateSMBFileShareVisibility() {
        return this.to('UpdateSMBFileShareVisibility');
    }
    /**
     * Grants permission to update the list of Active Directory users and groups that have special permissions for SMB file shares on the gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateSMBLocalGroups.html
     */
    toUpdateSMBLocalGroups() {
        return this.to('UpdateSMBLocalGroups');
    }
    /**
     * Grants permission to update the SMB security strategy on a file gateway
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateSMBSecurityStrategy.html
     */
    toUpdateSMBSecurityStrategy() {
        return this.to('UpdateSMBSecurityStrategy');
    }
    /**
     * Grants permission to update a snapshot schedule configured for a gateway volume
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateSnapshotSchedule.html
     */
    toUpdateSnapshotSchedule() {
        return this.to('UpdateSnapshotSchedule');
    }
    /**
     * Grants permission to update the type of medium changer in a gateway-VTL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateVTLDeviceType.html
     */
    toUpdateVTLDeviceType() {
        return this.to('UpdateVTLDeviceType');
    }
    /**
     * Adds a resource of type cache-report to the statement
     *
     * https://docs.aws.amazon.com/filegateway/latest/files3/cache-report.html
     *
     * @param shareId - Identifier for the shareId.
     * @param cacheReportId - Identifier for the cacheReportId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCacheReport(shareId, cacheReportId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:storagegateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:share/${shareId}/cache-report/${cacheReportId}`);
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/resource_vtl-devices.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param vtldevice - Identifier for the vtldevice.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDevice(gatewayId, vtldevice, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:storagegateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gateway/${gatewayId}/device/${vtldevice}`);
    }
    /**
     * Adds a resource of type fs-association to the statement
     *
     * https://docs.aws.amazon.com/filegateway/latest/filefsxw/attach-fsxw-filesystem.html
     *
     * @param fsaId - Identifier for the fsaId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onFsAssociation(fsaId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:storagegateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:fs-association/${fsaId}`);
    }
    /**
     * Adds a resource of type gateway to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/StorageGatewayConcepts.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGateway(gatewayId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:storagegateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gateway/${gatewayId}`);
    }
    /**
     * Adds a resource of type share to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/GettingStartedCreateFileShare.html
     *
     * @param shareId - Identifier for the shareId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onShare(shareId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:storagegateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:share/${shareId}`);
    }
    /**
     * Adds a resource of type tape to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/StorageGatewayConcepts.html#storage-gateway-vtl-concepts
     *
     * @param tapeBarcode - Identifier for the tapeBarcode.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTape(tapeBarcode, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:storagegateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:tape/${tapeBarcode}`);
    }
    /**
     * Adds a resource of type tapepool to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/CreatingCustomTapePool.html
     *
     * @param poolId - Identifier for the poolId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTapepool(poolId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:storagegateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:tapepool/${poolId}`);
    }
    /**
     * Adds a resource of type target to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/GettingStartedCreateVolumes.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param iscsiTarget - Identifier for the iscsiTarget.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onTarget(gatewayId, iscsiTarget, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:storagegateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gateway/${gatewayId}/target/${iscsiTarget}`);
    }
    /**
     * Adds a resource of type volume to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/StorageGatewayConcepts.html#volume-gateway-concepts
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVolume(gatewayId, volumeId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:storagegateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:gateway/${gatewayId}/volume/${volumeId}`);
    }
    /**
     * Filters access by the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toActivateGateway()
     * - .toAddTagsToResource()
     * - .toAssociateFileSystem()
     * - .toCreateCachediSCSIVolume()
     * - .toCreateNFSFileShare()
     * - .toCreateSMBFileShare()
     * - .toCreateSnapshot()
     * - .toCreateSnapshotFromVolumeRecoveryPoint()
     * - .toCreateStorediSCSIVolume()
     * - .toCreateTapePool()
     * - .toCreateTapeWithBarcode()
     * - .toCreateTapes()
     * - .toStartCacheReport()
     * - .toUpdateSnapshotSchedule()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to resource types:
     * - cache-report
     * - fs-association
     * - gateway
     * - share
     * - tape
     * - tapepool
     * - volume
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toActivateGateway()
     * - .toAddTagsToResource()
     * - .toAssociateFileSystem()
     * - .toCreateCachediSCSIVolume()
     * - .toCreateNFSFileShare()
     * - .toCreateSMBFileShare()
     * - .toCreateSnapshot()
     * - .toCreateSnapshotFromVolumeRecoveryPoint()
     * - .toCreateStorediSCSIVolume()
     * - .toCreateTapePool()
     * - .toCreateTapeWithBarcode()
     * - .toCreateTapes()
     * - .toRemoveTagsFromResource()
     * - .toStartCacheReport()
     * - .toUpdateSnapshotSchedule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [storagegateway](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsstoragegateway.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'storagegateway';
        this.accessLevelList = {
            Write: [
                'ActivateGateway',
                'AddCache',
                'AddUploadBuffer',
                'AddWorkingStorage',
                'AssignTapePool',
                'AssociateFileSystem',
                'AttachVolume',
                'BypassGovernanceRetention',
                'CancelArchival',
                'CancelCacheReport',
                'CancelRetrieval',
                'CreateCachediSCSIVolume',
                'CreateNFSFileShare',
                'CreateSMBFileShare',
                'CreateSnapshot',
                'CreateSnapshotFromVolumeRecoveryPoint',
                'CreateStorediSCSIVolume',
                'CreateTapePool',
                'CreateTapeWithBarcode',
                'CreateTapes',
                'DeleteAutomaticTapeCreationPolicy',
                'DeleteBandwidthRateLimit',
                'DeleteCacheReport',
                'DeleteChapCredentials',
                'DeleteFileShare',
                'DeleteGateway',
                'DeleteSnapshotSchedule',
                'DeleteTape',
                'DeleteTapeArchive',
                'DeleteTapePool',
                'DeleteVolume',
                'DetachVolume',
                'DisableGateway',
                'DisassociateFileSystem',
                'EvictFilesFailingUpload',
                'JoinDomain',
                'NotifyWhenUploaded',
                'RefreshCache',
                'ResetCache',
                'RetrieveTapeArchive',
                'RetrieveTapeRecoveryPoint',
                'SetLocalConsolePassword',
                'SetSMBGuestPassword',
                'ShutdownGateway',
                'StartAvailabilityMonitorTest',
                'StartCacheReport',
                'StartGateway',
                'UpdateAutomaticTapeCreationPolicy',
                'UpdateBandwidthRateLimit',
                'UpdateBandwidthRateLimitSchedule',
                'UpdateChapCredentials',
                'UpdateFileSystemAssociation',
                'UpdateGatewayInformation',
                'UpdateGatewaySoftwareNow',
                'UpdateMaintenanceStartTime',
                'UpdateNFSFileShare',
                'UpdateSMBFileShare',
                'UpdateSMBFileShareVisibility',
                'UpdateSMBLocalGroups',
                'UpdateSMBSecurityStrategy',
                'UpdateSnapshotSchedule',
                'UpdateVTLDeviceType'
            ],
            Tagging: [
                'AddTagsToResource',
                'RemoveTagsFromResource'
            ],
            Read: [
                'DescribeAvailabilityMonitorTest',
                'DescribeBandwidthRateLimit',
                'DescribeBandwidthRateLimitSchedule',
                'DescribeCache',
                'DescribeCacheReport',
                'DescribeCachediSCSIVolumes',
                'DescribeChapCredentials',
                'DescribeFileSystemAssociations',
                'DescribeGatewayInformation',
                'DescribeMaintenanceStartTime',
                'DescribeNFSFileShares',
                'DescribeSMBFileShares',
                'DescribeSMBSettings',
                'DescribeSnapshotSchedule',
                'DescribeStorediSCSIVolumes',
                'DescribeTapeArchives',
                'DescribeTapeRecoveryPoints',
                'DescribeTapes',
                'DescribeUploadBuffer',
                'DescribeVTLDevices',
                'DescribeWorkingStorage'
            ],
            List: [
                'ListAutomaticTapeCreationPolicies',
                'ListCacheReports',
                'ListFileShares',
                'ListFileSystemAssociations',
                'ListGateways',
                'ListLocalDisks',
                'ListTagsForResource',
                'ListTapePools',
                'ListTapes',
                'ListVolumeInitiators',
                'ListVolumeRecoveryPoints',
                'ListVolumes'
            ]
        };
    }
}
exports.Storagegateway = Storagegateway;
//# sourceMappingURL=data:application/json;base64,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