"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Scn = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [scn](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Scn extends shared_1.PolicyStatement {
    /**
     * Grants permission to add AWS Supply Chain administrator permission to federated user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toAssignAdminPermissionsToUser() {
        return this.to('AssignAdminPermissionsToUser');
    }
    /**
     * Grants permission to create a BillOfMaterialsImportJob which will import a CSV file of BillOfMaterials records
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toCreateBillOfMaterialsImportJob() {
        return this.to('CreateBillOfMaterialsImportJob');
    }
    /**
     * Grants permission to create DataIntegrationFlow that can transform from multiple sources to one target
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toCreateDataIntegrationFlow() {
        return this.to('CreateDataIntegrationFlow');
    }
    /**
     * Grants permission to create the data lake dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toCreateDataLakeDataset() {
        return this.to('CreateDataLakeDataset');
    }
    /**
     * Grants permission to create the data lake namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toCreateDataLakeNamespace() {
        return this.to('CreateDataLakeNamespace');
    }
    /**
     * Grants permission to create a new AWS Supply Chain instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toCreateInstance() {
        return this.to('CreateInstance');
    }
    /**
     * Grants permission to create IAM Identity Center application for a AWS Supply Chain instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toCreateSSOApplication() {
        return this.to('CreateSSOApplication');
    }
    /**
     * Grants permission to delete the DataIntegrationFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toDeleteDataIntegrationFlow() {
        return this.to('DeleteDataIntegrationFlow');
    }
    /**
     * Grants permission to delete the data lake dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toDeleteDataLakeDataset() {
        return this.to('DeleteDataLakeDataset');
    }
    /**
     * Grants permission to delete the data lake namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toDeleteDataLakeNamespace() {
        return this.to('DeleteDataLakeNamespace');
    }
    /**
     * Grants permission to delete an AWS Supply Chain instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toDeleteInstance() {
        return this.to('DeleteInstance');
    }
    /**
     * Grants permission to delete IAM Identity Center application of the AWS Supply Chain instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toDeleteSSOApplication() {
        return this.to('DeleteSSOApplication');
    }
    /**
     * Grants permission to view details of an AWS Supply Chain instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toDescribeInstance() {
        return this.to('DescribeInstance');
    }
    /**
     * Grants permission to view status and details of a BillOfMaterialsImportJob
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toGetBillOfMaterialsImportJob() {
        return this.to('GetBillOfMaterialsImportJob');
    }
    /**
     * Grants permission to get a DataIntegrationEvent
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toGetDataIntegrationEvent() {
        return this.to('GetDataIntegrationEvent');
    }
    /**
     * Grants permission to get the DataIntegrationFlow details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toGetDataIntegrationFlow() {
        return this.to('GetDataIntegrationFlow');
    }
    /**
     * Grants permission to get a particular execution of one specified DataIntegrationFlow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toGetDataIntegrationFlowExecution() {
        return this.to('GetDataIntegrationFlowExecution');
    }
    /**
     * Grants permission to get the dataset details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toGetDataLakeDataset() {
        return this.to('GetDataLakeDataset');
    }
    /**
     * Grants permission to get the namespace details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toGetDataLakeNamespace() {
        return this.to('GetDataLakeNamespace');
    }
    /**
     * Grants permission to view details of an AWS Supply Chain instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toGetInstance() {
        return this.to('GetInstance');
    }
    /**
     * Grants permission to list AWS Supply Chain administrators of an instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toListAdminUsers() {
        return this.to('ListAdminUsers');
    }
    /**
     * Grants permission to list all DataIntegrationEvents under an instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toListDataIntegrationEvents() {
        return this.to('ListDataIntegrationEvents');
    }
    /**
     * Grants permission to list all executions of one specified DataIntegrationFlow
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toListDataIntegrationFlowExecutions() {
        return this.to('ListDataIntegrationFlowExecutions');
    }
    /**
     * Grants permission to list all the DataIntegrationFlows in a paginated way
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toListDataIntegrationFlows() {
        return this.to('ListDataIntegrationFlows');
    }
    /**
     * Grants permission to list the data lake datasets under specific instance and namespace
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toListDataLakeDatasets() {
        return this.to('ListDataLakeDatasets');
    }
    /**
     * Grants permission to list the data lake namespaces under specific instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toListDataLakeNamespaces() {
        return this.to('ListDataLakeNamespaces');
    }
    /**
     * Grants permission to view the AWS Supply Chain instances associated with an AWS account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toListInstances() {
        return this.to('ListInstances');
    }
    /**
     * Grants permission to list tags for an AWS Supply Chain resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to remove AWS Supply Chain administrator permission from federated user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toRemoveAdminPermissionsForUser() {
        return this.to('RemoveAdminPermissionsForUser');
    }
    /**
     * Grants permission to create a DataIntegrationEvent which will ingest data in real-time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toSendDataIntegrationEvent() {
        return this.to('SendDataIntegrationEvent');
    }
    /**
     * Grants permission to tag an AWS Supply Chain resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tag from an AWS Supply Chain resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the DataIntegrationFlow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toUpdateDataIntegrationFlow() {
        return this.to('UpdateDataIntegrationFlow');
    }
    /**
     * Grants permission to update the data lake dataset
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toUpdateDataLakeDataset() {
        return this.to('UpdateDataLakeDataset');
    }
    /**
     * Grants permission to update the data lake namespace
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toUpdateDataLakeNamespace() {
        return this.to('UpdateDataLakeNamespace');
    }
    /**
     * Grants permission to update an AWS Supply Chain instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     */
    toUpdateInstance() {
        return this.to('UpdateInstance');
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onInstance(instanceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:scn:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:instance/${instanceId}`);
    }
    /**
     * Adds a resource of type bill-of-materials-import-job to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onBillOfMaterialsImportJob(instanceId, jobId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:scn:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:instance/${instanceId}/bill-of-materials-import-job/${jobId}`);
    }
    /**
     * Adds a resource of type data-integration-flow to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param flowName - Identifier for the flowName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDataIntegrationFlow(instanceId, flowName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:scn:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:instance/${instanceId}/data-integration-flows/${flowName}`);
    }
    /**
     * Adds a resource of type namespace to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param namespace - Identifier for the namespace.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onNamespace(instanceId, namespace, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:scn:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:instance/${instanceId}/namespaces/${namespace}`);
    }
    /**
     * Adds a resource of type dataset to the statement
     *
     * https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param namespace - Identifier for the namespace.
     * @param datasetName - Identifier for the datasetName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onDataset(instanceId, namespace, datasetName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:scn:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:instance/${instanceId}/namespaces/${namespace}/datasets/${datasetName}`);
    }
    /**
     * Filters access by using tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by using tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by using tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [scn](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awssupplychain.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'scn';
        this.accessLevelList = {
            Write: [
                'AssignAdminPermissionsToUser',
                'CreateBillOfMaterialsImportJob',
                'CreateDataIntegrationFlow',
                'CreateDataLakeDataset',
                'CreateDataLakeNamespace',
                'CreateInstance',
                'CreateSSOApplication',
                'DeleteDataIntegrationFlow',
                'DeleteDataLakeDataset',
                'DeleteDataLakeNamespace',
                'DeleteInstance',
                'DeleteSSOApplication',
                'RemoveAdminPermissionsForUser',
                'SendDataIntegrationEvent',
                'UpdateDataIntegrationFlow',
                'UpdateDataLakeDataset',
                'UpdateDataLakeNamespace',
                'UpdateInstance'
            ],
            Read: [
                'DescribeInstance',
                'GetBillOfMaterialsImportJob',
                'GetDataIntegrationEvent',
                'GetDataIntegrationFlow',
                'GetDataIntegrationFlowExecution',
                'GetDataLakeDataset',
                'GetDataLakeNamespace',
                'GetInstance'
            ],
            List: [
                'ListAdminUsers',
                'ListDataIntegrationEvents',
                'ListDataIntegrationFlowExecutions',
                'ListDataIntegrationFlows',
                'ListDataLakeDatasets',
                'ListDataLakeNamespaces',
                'ListInstances',
                'ListTagsForResource'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Scn = Scn;
//# sourceMappingURL=data:application/json;base64,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