"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transcribe = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [transcribe](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazontranscribe.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Transcribe extends shared_1.PolicyStatement {
    /**
     * Grants permission to create an analytics category. Amazon Transcribe applies the conditions specified by your analytics categories to your call analytics jobs
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateCallAnalyticsCategory.html
     */
    toCreateCallAnalyticsCategory() {
        return this.to('CreateCallAnalyticsCategory');
    }
    /**
     * Grants permission to create a new custom language model
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     * - s3:ListBucket
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateLanguageModel.html
     */
    toCreateLanguageModel() {
        return this.to('CreateLanguageModel');
    }
    /**
     * Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe Medical handles transcription of an audio file
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateMedicalVocabulary.html
     */
    toCreateMedicalVocabulary() {
        return this.to('CreateMedicalVocabulary');
    }
    /**
     * Grants permission to create a new custom vocabulary that you can use to change the way Amazon Transcribe handles transcription of an audio file
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabulary.html
     */
    toCreateVocabulary() {
        return this.to('CreateVocabulary');
    }
    /**
     * Grants permission to create a new vocabulary filter that you can use to filter out words from the transcription of an audio file generated by Amazon Transcribe
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabularyFilter.html
     */
    toCreateVocabularyFilter() {
        return this.to('CreateVocabularyFilter');
    }
    /**
     * Grants permission to delete a call analytics category using its name from Amazon Transcribe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteCallAnalyticsCategory.html
     */
    toDeleteCallAnalyticsCategory() {
        return this.to('DeleteCallAnalyticsCategory');
    }
    /**
     * Grants permission to delete a previously submitted call analytics job along with any other generated results such as the transcription, models, and so on
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteCallAnalyticsJob.html
     */
    toDeleteCallAnalyticsJob() {
        return this.to('DeleteCallAnalyticsJob');
    }
    /**
     * Grants permission to delete a previously created custom language model
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteLanguageModel.html
     */
    toDeleteLanguageModel() {
        return this.to('DeleteLanguageModel');
    }
    /**
     * Grants permission to delete a previously submitted Medical Scribe job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalScribeJob.html
     */
    toDeleteMedicalScribeJob() {
        return this.to('DeleteMedicalScribeJob');
    }
    /**
     * Grants permission to delete a previously submitted medical transcription job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalTranscriptionJob.html
     */
    toDeleteMedicalTranscriptionJob() {
        return this.to('DeleteMedicalTranscriptionJob');
    }
    /**
     * Grants permission to delete a medical vocabulary from Amazon Transcribe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteMedicalVocabulary.html
     */
    toDeleteMedicalVocabulary() {
        return this.to('DeleteMedicalVocabulary');
    }
    /**
     * Grants permission to delete a previously submitted transcription job along with any other generated results such as the transcription, models, and so on
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteTranscriptionJob.html
     */
    toDeleteTranscriptionJob() {
        return this.to('DeleteTranscriptionJob');
    }
    /**
     * Grants permission to delete a vocabulary from Amazon Transcribe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabulary.html
     */
    toDeleteVocabulary() {
        return this.to('DeleteVocabulary');
    }
    /**
     * Grants permission to delete a vocabulary filter from Amazon Transcribe
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DeleteVocabularyFilter.html
     */
    toDeleteVocabularyFilter() {
        return this.to('DeleteVocabularyFilter');
    }
    /**
     * Grants permission to return information about a custom language model
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_DescribeLanguageModel.html
     */
    toDescribeLanguageModel() {
        return this.to('DescribeLanguageModel');
    }
    /**
     * Grants permission to retrieve information about a call analytics category
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetCallAnalyticsCategory.html
     */
    toGetCallAnalyticsCategory() {
        return this.to('GetCallAnalyticsCategory');
    }
    /**
     * Grants permission to return information about a call analytics job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetCallAnalyticsJob.html
     */
    toGetCallAnalyticsJob() {
        return this.to('GetCallAnalyticsJob');
    }
    /**
     * Grants permission to return information about a Medical Scribe job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalScribeJob.html
     */
    toGetMedicalScribeJob() {
        return this.to('GetMedicalScribeJob');
    }
    /**
     * Grants permission to get information about the specified AWS HealthScribe streaming session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_GetMedicalScribeStream.html
     */
    toGetMedicalScribeStream() {
        return this.to('GetMedicalScribeStream');
    }
    /**
     * Grants permission to return information about a medical transcription job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalTranscriptionJob.html
     */
    toGetMedicalTranscriptionJob() {
        return this.to('GetMedicalTranscriptionJob');
    }
    /**
     * Grants permission to get information about a medical vocabulary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetMedicalVocabulary.html
     */
    toGetMedicalVocabulary() {
        return this.to('GetMedicalVocabulary');
    }
    /**
     * Grants permission to return information about a transcription job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetTranscriptionJob.html
     */
    toGetTranscriptionJob() {
        return this.to('GetTranscriptionJob');
    }
    /**
     * Grants permission to to get information about a vocabulary
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabulary.html
     */
    toGetVocabulary() {
        return this.to('GetVocabulary');
    }
    /**
     * Grants permission to get information about a vocabulary filter
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_GetVocabularyFilter.html
     */
    toGetVocabularyFilter() {
        return this.to('GetVocabularyFilter');
    }
    /**
     * Grants permission to list call analytics categories that has been created
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListCallAnalyticsCategories.html
     */
    toListCallAnalyticsCategories() {
        return this.to('ListCallAnalyticsCategories');
    }
    /**
     * Grants permission to list call analytics jobs with the specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListCallAnalyticsJobs.html
     */
    toListCallAnalyticsJobs() {
        return this.to('ListCallAnalyticsJobs');
    }
    /**
     * Grants permission to list custom language models
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListLanguageModels.html
     */
    toListLanguageModels() {
        return this.to('ListLanguageModels');
    }
    /**
     * Grants permission to list Medical Scribe jobs with the specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalScribeJobs.html
     */
    toListMedicalScribeJobs() {
        return this.to('ListMedicalScribeJobs');
    }
    /**
     * Grants permission to list medical transcription jobs with the specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalTranscriptionJobs.html
     */
    toListMedicalTranscriptionJobs() {
        return this.to('ListMedicalTranscriptionJobs');
    }
    /**
     * Grants permission to return a list of medical vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListMedicalVocabularies.html
     */
    toListMedicalVocabularies() {
        return this.to('ListMedicalVocabularies');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list transcription jobs with the specified status
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListTranscriptionJobs.html
     */
    toListTranscriptionJobs() {
        return this.to('ListTranscriptionJobs');
    }
    /**
     * Grants permission to return a list of vocabularies that match the specified criteria. If no criteria are specified, returns the entire list of vocabularies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularies.html
     */
    toListVocabularies() {
        return this.to('ListVocabularies');
    }
    /**
     * Grants permission to return a list of vocabulary filters that match the specified criteria. If no criteria are specified, returns the at most 5 vocabulary filters
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_ListVocabularyFilters.html
     */
    toListVocabularyFilters() {
        return this.to('ListVocabularyFilters');
    }
    /**
     * Grants permission to start an asynchronous analytics job that not only transcribes the audio recording of a caller and agent, but also returns additional insights
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputEncryptionKMSKeyId()
     * - .ifOutputLocation()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartCallAnalyticsJob.html
     */
    toStartCallAnalyticsJob() {
        return this.to('StartCallAnalyticsJob');
    }
    /**
     * Grants permission to start a protocol where audio is streamed to Transcribe Call Analytics and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartCallAnalyticsStreamTranscription.html
     */
    toStartCallAnalyticsStreamTranscription() {
        return this.to('StartCallAnalyticsStreamTranscription');
    }
    /**
     * Grants permission to start a WebSocket where audio is streamed to Transcribe Call Analytics and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartCallAnalyticsStreamTranscriptionWebSocket.html
     */
    toStartCallAnalyticsStreamTranscriptionWebSocket() {
        return this.to('StartCallAnalyticsStreamTranscriptionWebSocket');
    }
    /**
     * Grants permission to start an asynchronous job to transcribe patient-clinician conversations and generates clinical notes
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputBucketName()
     * - .ifOutputEncryptionKMSKeyId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartMedicalScribeJob.html
     */
    toStartMedicalScribeJob() {
        return this.to('StartMedicalScribeJob');
    }
    /**
     * Grants permission to start a bidirectional HTTP2 stream where audio is streamed to AWS HealthScribe and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalScribeStream.html
     */
    toStartMedicalScribeStream() {
        return this.to('StartMedicalScribeStream');
    }
    /**
     * Grants permission to start a protocol where audio is streamed to Transcribe Medical and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscription.html
     */
    toStartMedicalStreamTranscription() {
        return this.to('StartMedicalStreamTranscription');
    }
    /**
     * Grants permission to start a WebSocket where audio is streamed to Transcribe Medical and the transcription results are streamed to your application
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartMedicalStreamTranscriptionWebSocket.html
     */
    toStartMedicalStreamTranscriptionWebSocket() {
        return this.to('StartMedicalStreamTranscriptionWebSocket');
    }
    /**
     * Grants permission to start an asynchronous job to transcribe medical speech to text
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputBucketName()
     * - .ifOutputEncryptionKMSKeyId()
     * - .ifOutputKey()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartMedicalTranscriptionJob.html
     */
    toStartMedicalTranscriptionJob() {
        return this.to('StartMedicalTranscriptionJob');
    }
    /**
     * Grants permission to start a bidirectional HTTP2 stream to transcribe speech to text in real time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscription.html
     */
    toStartStreamTranscription() {
        return this.to('StartStreamTranscription');
    }
    /**
     * Grants permission to start a websocket stream to transcribe speech to text in real time
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_streaming_StartStreamTranscriptionWebSocket.html
     */
    toStartStreamTranscriptionWebSocket() {
        return this.to('StartStreamTranscriptionWebSocket');
    }
    /**
     * Grants permission to start an asynchronous job to transcribe speech to text
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifOutputBucketName()
     * - .ifOutputEncryptionKMSKeyId()
     * - .ifOutputKey()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_StartTranscriptionJob.html
     */
    toStartTranscriptionJob() {
        return this.to('StartTranscriptionJob');
    }
    /**
     * Grants permission to tag a resource with given key value pairs
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a resource with given key
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update the call analytics category with new values. The UpdateCallAnalyticsCategory operation overwrites all of the existing information with the values that you provide in the request
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateCallAnalyticsCategory.html
     */
    toUpdateCallAnalyticsCategory() {
        return this.to('UpdateCallAnalyticsCategory');
    }
    /**
     * Grants permission to update an existing medical vocabulary with new values. The UpdateMedicalVocabulary operation overwrites all of the existing information with the values that you provide in the request
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateMedicalVocabulary.html
     */
    toUpdateMedicalVocabulary() {
        return this.to('UpdateMedicalVocabulary');
    }
    /**
     * Grants permission to update an existing vocabulary with new values. The UpdateVocabulary operation overwrites all of the existing information with the values that you provide in the request
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabulary.html
     */
    toUpdateVocabulary() {
        return this.to('UpdateVocabulary');
    }
    /**
     * Grants permission to update an existing vocabulary filter with new values. The UpdateVocabularyFilter operation overwrites all of the existing information with the values that you provide in the request
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_UpdateVocabularyFilter.html
     */
    toUpdateVocabularyFilter() {
        return this.to('UpdateVocabularyFilter');
    }
    /**
     * Adds a resource of type transcriptionjob to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_TranscriptionJob.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTranscriptionjob(jobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:transcription-job/${jobName}`);
    }
    /**
     * Adds a resource of type vocabulary to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabulary.html
     *
     * @param vocabularyName - Identifier for the vocabularyName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVocabulary(vocabularyName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vocabulary/${vocabularyName}`);
    }
    /**
     * Adds a resource of type vocabularyfilter to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateVocabularyFilter.html
     *
     * @param vocabularyFilterName - Identifier for the vocabularyFilterName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVocabularyfilter(vocabularyFilterName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:vocabulary-filter/${vocabularyFilterName}`);
    }
    /**
     * Adds a resource of type languagemodel to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_LanguageModel.html
     *
     * @param modelName - Identifier for the modelName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLanguagemodel(modelName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:language-model/${modelName}`);
    }
    /**
     * Adds a resource of type medicaltranscriptionjob to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_MedicalTranscriptionJob.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMedicaltranscriptionjob(jobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:medical-transcription-job/${jobName}`);
    }
    /**
     * Adds a resource of type medicalvocabulary to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateMedicalVocabulary.html
     *
     * @param vocabularyName - Identifier for the vocabularyName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMedicalvocabulary(vocabularyName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:medical-vocabulary/${vocabularyName}`);
    }
    /**
     * Adds a resource of type callanalyticsjob to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CallAnalyticsJob.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCallanalyticsjob(jobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:analytics/${jobName}`);
    }
    /**
     * Adds a resource of type callanalyticscategory to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_CreateCallAnalyticsCategory.html
     *
     * @param categoryName - Identifier for the categoryName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCallanalyticscategory(categoryName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:analytics-category/${categoryName}`);
    }
    /**
     * Adds a resource of type medicalscribejob to the statement
     *
     * https://docs.aws.amazon.com/transcribe/latest/dg/API_MedicalScribeJob.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMedicalscribejob(jobName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:transcribe:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:medical-scribe-job/${jobName}`);
    }
    /**
     * Filters access by requiring tag values present in a resource creation request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateCallAnalyticsCategory()
     * - .toCreateLanguageModel()
     * - .toCreateMedicalVocabulary()
     * - .toCreateVocabulary()
     * - .toCreateVocabularyFilter()
     * - .toStartCallAnalyticsJob()
     * - .toStartMedicalScribeJob()
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by requiring tag value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to resource types:
     * - transcriptionjob
     * - vocabulary
     * - vocabularyfilter
     * - languagemodel
     * - medicaltranscriptionjob
     * - medicalvocabulary
     * - callanalyticsjob
     * - callanalyticscategory
     * - medicalscribejob
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by requiring the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-globally-available
     *
     * Applies to actions:
     * - .toCreateCallAnalyticsCategory()
     * - .toCreateLanguageModel()
     * - .toCreateMedicalVocabulary()
     * - .toCreateVocabulary()
     * - .toCreateVocabularyFilter()
     * - .toStartCallAnalyticsJob()
     * - .toStartMedicalScribeJob()
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the output bucket name included in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html#amazontranscribe-policy-keys
     *
     * Applies to actions:
     * - .toStartMedicalScribeJob()
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputBucketName(value, operator) {
        return this.if(`OutputBucketName`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the KMS key id included in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html#amazontranscribe-policy-keys
     *
     * Applies to actions:
     * - .toStartCallAnalyticsJob()
     * - .toStartMedicalScribeJob()
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputEncryptionKMSKeyId(value, operator) {
        return this.if(`OutputEncryptionKMSKeyId`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the output key included in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html#amazontranscribe-policy-keys
     *
     * Applies to actions:
     * - .toStartMedicalTranscriptionJob()
     * - .toStartTranscriptionJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputKey(value, operator) {
        return this.if(`OutputKey`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access based on the output location included in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazontranscribe.html#amazontranscribe-policy-keys
     *
     * Applies to actions:
     * - .toStartCallAnalyticsJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifOutputLocation(value, operator) {
        return this.if(`OutputLocation`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [transcribe](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazontranscribe.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'transcribe';
        this.accessLevelList = {
            Write: [
                'CreateCallAnalyticsCategory',
                'CreateLanguageModel',
                'CreateMedicalVocabulary',
                'CreateVocabulary',
                'CreateVocabularyFilter',
                'DeleteCallAnalyticsCategory',
                'DeleteCallAnalyticsJob',
                'DeleteLanguageModel',
                'DeleteMedicalScribeJob',
                'DeleteMedicalTranscriptionJob',
                'DeleteMedicalVocabulary',
                'DeleteTranscriptionJob',
                'DeleteVocabulary',
                'DeleteVocabularyFilter',
                'StartCallAnalyticsJob',
                'StartCallAnalyticsStreamTranscription',
                'StartCallAnalyticsStreamTranscriptionWebSocket',
                'StartMedicalScribeJob',
                'StartMedicalScribeStream',
                'StartMedicalStreamTranscription',
                'StartMedicalStreamTranscriptionWebSocket',
                'StartMedicalTranscriptionJob',
                'StartStreamTranscription',
                'StartStreamTranscriptionWebSocket',
                'StartTranscriptionJob',
                'UpdateCallAnalyticsCategory',
                'UpdateMedicalVocabulary',
                'UpdateVocabulary',
                'UpdateVocabularyFilter'
            ],
            Read: [
                'DescribeLanguageModel',
                'GetCallAnalyticsCategory',
                'GetCallAnalyticsJob',
                'GetMedicalScribeJob',
                'GetMedicalScribeStream',
                'GetMedicalTranscriptionJob',
                'GetMedicalVocabulary',
                'GetTranscriptionJob',
                'GetVocabulary',
                'GetVocabularyFilter',
                'ListTagsForResource'
            ],
            List: [
                'ListCallAnalyticsCategories',
                'ListCallAnalyticsJobs',
                'ListLanguageModels',
                'ListMedicalScribeJobs',
                'ListMedicalTranscriptionJobs',
                'ListMedicalVocabularies',
                'ListTranscriptionJobs',
                'ListVocabularies',
                'ListVocabularyFilters'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Transcribe = Transcribe;
//# sourceMappingURL=data:application/json;base64,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