"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wafv2 = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [wafv2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awswafv2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Wafv2 extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a WebACL with a resource
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - amplify:AssociateWebACL
     * - apigateway:SetWebACL
     * - apprunner:AssociateWebAcl
     * - appsync:SetWebACL
     * - cognito-idp:AssociateWebACL
     * - ec2:AssociateVerifiedAccessInstanceWebAcl
     * - elasticloadbalancing:SetWebAcl
     * - wafv2:GetPermissionPolicy
     * - wafv2:PutPermissionPolicy
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_AssociateWebACL.html
     */
    toAssociateWebACL() {
        return this.to('AssociateWebACL');
    }
    /**
     * Grants permission to calculate web ACL capacity unit (WCU) requirements for a specified scope and set of rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CheckCapacity.html
     */
    toCheckCapacity() {
        return this.to('CheckCapacity');
    }
    /**
     * Grants permission to create an API key for use in the integration of the CAPTCHA API in your JavaScript client applications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateAPIKey.html
     */
    toCreateAPIKey() {
        return this.to('CreateAPIKey');
    }
    /**
     * Grants permission to create an IPSet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - wafv2:TagResource
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateIPSet.html
     */
    toCreateIPSet() {
        return this.to('CreateIPSet');
    }
    /**
     * Grants permission to create a RegexPatternSet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - wafv2:TagResource
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRegexPatternSet.html
     */
    toCreateRegexPatternSet() {
        return this.to('CreateRegexPatternSet');
    }
    /**
     * Grants permission to create a RuleGroup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - wafv2:TagResource
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRuleGroup.html
     */
    toCreateRuleGroup() {
        return this.to('CreateRuleGroup');
    }
    /**
     * Grants permission to create a WebACL
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - wafv2:TagResource
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateWebACL.html
     */
    toCreateWebACL() {
        return this.to('CreateWebACL');
    }
    /**
     * Grants permission to delete an API key
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteAPIKey.html
     */
    toDeleteAPIKey() {
        return this.to('DeleteAPIKey');
    }
    /**
     * Grants permission to delete FirewallManagedRulesGroups from a WebACL if not managed by Firewall Manager anymore
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteFirewallManagerRuleGroups.html
     */
    toDeleteFirewallManagerRuleGroups() {
        return this.to('DeleteFirewallManagerRuleGroups');
    }
    /**
     * Grants permission to delete an IPSet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteIPSet.html
     */
    toDeleteIPSet() {
        return this.to('DeleteIPSet');
    }
    /**
     * Grants permission to delete the LoggingConfiguration from a WebACL
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLogScope()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteLoggingConfiguration.html
     */
    toDeleteLoggingConfiguration() {
        return this.to('DeleteLoggingConfiguration');
    }
    /**
     * Grants permission to delete the PermissionPolicy on a RuleGroup
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeletePermissionPolicy.html
     */
    toDeletePermissionPolicy() {
        return this.to('DeletePermissionPolicy');
    }
    /**
     * Grants permission to delete a RegexPatternSet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRegexPatternSet.html
     */
    toDeleteRegexPatternSet() {
        return this.to('DeleteRegexPatternSet');
    }
    /**
     * Grants permission to delete a RuleGroup
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRuleGroup.html
     */
    toDeleteRuleGroup() {
        return this.to('DeleteRuleGroup');
    }
    /**
     * Grants permission to delete a WebACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteWebACL.html
     */
    toDeleteWebACL() {
        return this.to('DeleteWebACL');
    }
    /**
     * Grants permission to retrieve product information for a managed rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DescribeAllManagedProducts.html
     */
    toDescribeAllManagedProducts() {
        return this.to('DescribeAllManagedProducts');
    }
    /**
     * Grants permission to retrieve product information for a managed rule group by a given vendor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DescribeManagedProductsByVendor.html
     */
    toDescribeManagedProductsByVendor() {
        return this.to('DescribeManagedProductsByVendor');
    }
    /**
     * Grants permission to retrieve high-level information for a managed rule group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DescribeManagedRuleGroup.html
     */
    toDescribeManagedRuleGroup() {
        return this.to('DescribeManagedRuleGroup');
    }
    /**
     * Grants permission to disassociate Firewall Manager from a WebACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateFirewallManager.html
     */
    toDisassociateFirewallManager() {
        return this.to('DisassociateFirewallManager');
    }
    /**
     * Grants permission to disassociate a WebACL from an application resource
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - amplify:DisassociateWebACL
     * - apigateway:SetWebACL
     * - apprunner:DisassociateWebAcl
     * - appsync:SetWebACL
     * - cognito-idp:DisassociateWebACL
     * - ec2:DisassociateVerifiedAccessInstanceWebAcl
     * - elasticloadbalancing:SetWebAcl
     * - wafv2:PutPermissionPolicy
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateWebACL.html
     */
    toDisassociateWebACL() {
        return this.to('DisassociateWebACL');
    }
    /**
     * Grants permission to generate a presigned download URL for the specified release of the mobile SDK
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GenerateMobileSdkReleaseUrl.html
     */
    toGenerateMobileSdkReleaseUrl() {
        return this.to('GenerateMobileSdkReleaseUrl');
    }
    /**
     * Grants permission to return your API key in decrypted form. Use this to check the token domains that you have defined for the key
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetDecryptedAPIKey.html
     */
    toGetDecryptedAPIKey() {
        return this.to('GetDecryptedAPIKey');
    }
    /**
     * Grants permission to retrieve details about an IPSet
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetIPSet.html
     */
    toGetIPSet() {
        return this.to('GetIPSet');
    }
    /**
     * Grants permission to retrieve LoggingConfiguration for a WebACL
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifLogScope()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetLoggingConfiguration.html
     */
    toGetLoggingConfiguration() {
        return this.to('GetLoggingConfiguration');
    }
    /**
     * Grants permission to retrieve details about a ManagedRuleSet
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetManagedRuleSet.html
     */
    toGetManagedRuleSet() {
        return this.to('GetManagedRuleSet');
    }
    /**
     * Grants permission to retrieve information for the specified mobile SDK release, including release notes and tags
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetMobileSdkRelease.html
     */
    toGetMobileSdkRelease() {
        return this.to('GetMobileSdkRelease');
    }
    /**
     * Grants permission to retrieve a PermissionPolicy for a RuleGroup
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetPermissionPolicy.html
     */
    toGetPermissionPolicy() {
        return this.to('GetPermissionPolicy');
    }
    /**
     * Grants permission to retrieve the keys that are currently blocked by a rate-based rule
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRateBasedStatementManagedKeys.html
     */
    toGetRateBasedStatementManagedKeys() {
        return this.to('GetRateBasedStatementManagedKeys');
    }
    /**
     * Grants permission to retrieve details about a RegexPatternSet
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRegexPatternSet.html
     */
    toGetRegexPatternSet() {
        return this.to('GetRegexPatternSet');
    }
    /**
     * Grants permission to retrieve details about a RuleGroup
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRuleGroup.html
     */
    toGetRuleGroup() {
        return this.to('GetRuleGroup');
    }
    /**
     * Grants permission to retrieve detailed information about a sampling of web requests
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetSampledRequests.html
     */
    toGetSampledRequests() {
        return this.to('GetSampledRequests');
    }
    /**
     * Grants permission to retrieve details about a WebACL
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACL.html
     */
    toGetWebACL() {
        return this.to('GetWebACL');
    }
    /**
     * Grants permission to retrieve the WebACL that's associated with a resource
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - amplify:GetWebACLForResource
     * - apprunner:DescribeWebAclForService
     * - cognito-idp:GetWebACLForResource
     * - ec2:GetVerifiedAccessInstanceWebAcl
     * - wafv2:GetWebACL
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACLForResource.html
     */
    toGetWebACLForResource() {
        return this.to('GetWebACLForResource');
    }
    /**
     * Grants permission to retrieve a list of the API keys that you've defined for the specified scope
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListAPIKeys.html
     */
    toListAPIKeys() {
        return this.to('ListAPIKeys');
    }
    /**
     * Grants permission to retrieve an array of managed rule group versions that are available for you to use
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListAvailableManagedRuleGroupVersions.html
     */
    toListAvailableManagedRuleGroupVersions() {
        return this.to('ListAvailableManagedRuleGroupVersions');
    }
    /**
     * Grants permission to retrieve an array of managed rule groups that are available for you to use
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListAvailableManagedRuleGroups.html
     */
    toListAvailableManagedRuleGroups() {
        return this.to('ListAvailableManagedRuleGroups');
    }
    /**
     * Grants permission to retrieve an array of IPSetSummary objects for the IP sets that you manage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListIPSets.html
     */
    toListIPSets() {
        return this.to('ListIPSets');
    }
    /**
     * Grants permission to retrieve an array of your LoggingConfiguration objects
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifLogScope()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListLoggingConfigurations.html
     */
    toListLoggingConfigurations() {
        return this.to('ListLoggingConfigurations');
    }
    /**
     * Grants permission to retrieve an array of your ManagedRuleSet objects
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListManagedRuleSets.html
     */
    toListManagedRuleSets() {
        return this.to('ListManagedRuleSets');
    }
    /**
     * Grants permission to retrieve a list of the available releases for the mobile SDK and the specified device platform
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListMobileSdkReleases.html
     */
    toListMobileSdkReleases() {
        return this.to('ListMobileSdkReleases');
    }
    /**
     * Grants permission to retrieve an array of RegexPatternSetSummary objects for the regex pattern sets that you manage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRegexPatternSets.html
     */
    toListRegexPatternSets() {
        return this.to('ListRegexPatternSets');
    }
    /**
     * Grants permission to retrieve an array of the Amazon Resource Names (ARNs) for the resources that are associated with a web ACL
     *
     * Access Level: List
     *
     * Dependent actions:
     * - amplify:ListResourcesForWebACL
     * - apprunner:ListAssociatedServicesForWebAcl
     * - cognito-idp:ListResourcesForWebACL
     * - ec2:DescribeVerifiedAccessInstanceWebAclAssociations
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListResourcesForWebACL.html
     */
    toListResourcesForWebACL() {
        return this.to('ListResourcesForWebACL');
    }
    /**
     * Grants permission to retrieve an array of RuleGroupSummary objects for the rule groups that you manage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRuleGroups.html
     */
    toListRuleGroups() {
        return this.to('ListRuleGroups');
    }
    /**
     * Grants permission to list tags for a resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to retrieve an array of WebACLSummary objects for the web ACLs that you manage
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListWebACLs.html
     */
    toListWebACLs() {
        return this.to('ListWebACLs');
    }
    /**
     * Grants permission to create FirewallManagedRulesGroups in a WebACL
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutFirewallManagerRuleGroups.html
     */
    toPutFirewallManagerRuleGroups() {
        return this.to('PutFirewallManagerRuleGroups');
    }
    /**
     * Grants permission to enable a LoggingConfiguration, to start logging for a web ACL
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifLogScope()
     * - .ifLogDestinationResource()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutLoggingConfiguration.html
     */
    toPutLoggingConfiguration() {
        return this.to('PutLoggingConfiguration');
    }
    /**
     * Grants permission to enable create a new or update an existing version of a ManagedRuleSet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutManagedRuleSetVersions.html
     */
    toPutManagedRuleSetVersions() {
        return this.to('PutManagedRuleSetVersions');
    }
    /**
     * Grants permission to attach an IAM policy to a resource, used to share rule groups between accounts
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutPermissionPolicy.html
     */
    toPutPermissionPolicy() {
        return this.to('PutPermissionPolicy');
    }
    /**
     * Grants permission to associate tags with a AWS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to disassociate tags from an AWS resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update an IPSet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateIPSet.html
     */
    toUpdateIPSet() {
        return this.to('UpdateIPSet');
    }
    /**
     * Grants permission to update the expiry date of a version in ManagedRuleSet
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateManagedRuleSetVersionExpiryDate.html
     */
    toUpdateManagedRuleSetVersionExpiryDate() {
        return this.to('UpdateManagedRuleSetVersionExpiryDate');
    }
    /**
     * Grants permission to update a RegexPatternSet
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRegexPatternSet.html
     */
    toUpdateRegexPatternSet() {
        return this.to('UpdateRegexPatternSet');
    }
    /**
     * Grants permission to update a RuleGroup
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRuleGroup.html
     */
    toUpdateRuleGroup() {
        return this.to('UpdateRuleGroup');
    }
    /**
     * Grants permission to update a WebACL
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateWebACL.html
     */
    toUpdateWebACL() {
        return this.to('UpdateWebACL');
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebacl(scope, name, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wafv2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${scope}/webacl/${name}/${id}`);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_IPSet.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIpset(scope, name, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wafv2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${scope}/ipset/${name}/${id}`);
    }
    /**
     * Adds a resource of type managedruleset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ManagedRuleSet.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onManagedruleset(scope, name, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wafv2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${scope}/managedruleset/${name}/${id}`);
    }
    /**
     * Adds a resource of type rulegroup to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_RuleGroup.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRulegroup(scope, name, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wafv2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${scope}/rulegroup/${name}/${id}`);
    }
    /**
     * Adds a resource of type regexpatternset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_RegexPatternSet.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRegexpatternset(scope, name, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wafv2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:${scope}/regexpatternset/${name}/${id}`);
    }
    /**
     * Adds a resource of type loadbalancer/app/ to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onLoadbalancerApp(loadBalancerName, loadBalancerId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:elasticloadbalancing:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:loadbalancer/app/${loadBalancerName}/${loadBalancerId}`);
    }
    /**
     * Adds a resource of type apigateway to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param apiId - Identifier for the apiId.
     * @param stageName - Identifier for the stageName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApigateway(apiId, stageName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apigateway:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:/restapis/${apiId}/stages/${stageName}`);
    }
    /**
     * Adds a resource of type appsync to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAppsync(graphQLAPIId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:appsync:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apis/${graphQLAPIId}`);
    }
    /**
     * Adds a resource of type userpool to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param userPoolId - Identifier for the userPoolId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onUserpool(userPoolId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:cognito-idp:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:userpool/${userPoolId}`);
    }
    /**
     * Adds a resource of type apprunner to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param serviceName - Identifier for the serviceName.
     * @param serviceId - Identifier for the serviceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onApprunner(serviceName, serviceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:apprunner:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:service/${serviceName}/${serviceId}`);
    }
    /**
     * Adds a resource of type verified-access-instance to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param verifiedAccessInstanceId - Identifier for the verifiedAccessInstanceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onVerifiedAccessInstance(verifiedAccessInstanceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:ec2:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:verified-access-instance/${verifiedAccessInstanceId}`);
    }
    /**
     * Adds a resource of type amplify-app to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param appId - Identifier for the appId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onAmplifyApp(appId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:amplify:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:apps/${appId}`);
    }
    /**
     * Filters access by the allowed set of values for each of the tags
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateIPSet()
     * - .toCreateRegexPatternSet()
     * - .toCreateRuleGroup()
     * - .toCreateWebACL()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag-value associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toGetIPSet()
     * - .toGetLoggingConfiguration()
     * - .toGetRateBasedStatementManagedKeys()
     * - .toGetRegexPatternSet()
     * - .toGetRuleGroup()
     * - .toGetWebACL()
     * - .toListTagsForResource()
     * - .toTagResource()
     * - .toUpdateIPSet()
     * - .toUpdateRegexPatternSet()
     * - .toUpdateRuleGroup()
     * - .toUpdateWebACL()
     *
     * Applies to resource types:
     * - webacl
     * - ipset
     * - rulegroup
     * - regexpatternset
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the presence of mandatory tags in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateIPSet()
     * - .toCreateRegexPatternSet()
     * - .toCreateRuleGroup()
     * - .toCreateWebACL()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by log destination ARN for PutLoggingConfiguration API
     *
     * https://docs.aws.amazon.com/waf/latest/developerguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toPutLoggingConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifLogDestinationResource(value, operator) {
        return this.if(`LogDestinationResource`, value, operator ?? 'ArnLike');
    }
    /**
     * Filters access by log scope for Logging Configuration API
     *
     * https://docs.aws.amazon.com/waf/latest/developerguide/security_iam_service-with-iam.html#security_iam_service-with-iam-id-based-policies-conditionkeys
     *
     * Applies to actions:
     * - .toDeleteLoggingConfiguration()
     * - .toGetLoggingConfiguration()
     * - .toListLoggingConfigurations()
     * - .toPutLoggingConfiguration()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLogScope(value, operator) {
        return this.if(`LogScope`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [wafv2](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awswafv2.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'wafv2';
        this.accessLevelList = {
            Write: [
                'AssociateWebACL',
                'CreateAPIKey',
                'CreateIPSet',
                'CreateRegexPatternSet',
                'CreateRuleGroup',
                'CreateWebACL',
                'DeleteAPIKey',
                'DeleteFirewallManagerRuleGroups',
                'DeleteIPSet',
                'DeleteLoggingConfiguration',
                'DeleteRegexPatternSet',
                'DeleteRuleGroup',
                'DeleteWebACL',
                'DisassociateFirewallManager',
                'DisassociateWebACL',
                'PutFirewallManagerRuleGroups',
                'PutLoggingConfiguration',
                'PutManagedRuleSetVersions',
                'UpdateIPSet',
                'UpdateManagedRuleSetVersionExpiryDate',
                'UpdateRegexPatternSet',
                'UpdateRuleGroup',
                'UpdateWebACL'
            ],
            Read: [
                'CheckCapacity',
                'DescribeAllManagedProducts',
                'DescribeManagedProductsByVendor',
                'DescribeManagedRuleGroup',
                'GenerateMobileSdkReleaseUrl',
                'GetDecryptedAPIKey',
                'GetIPSet',
                'GetLoggingConfiguration',
                'GetManagedRuleSet',
                'GetMobileSdkRelease',
                'GetPermissionPolicy',
                'GetRateBasedStatementManagedKeys',
                'GetRegexPatternSet',
                'GetRuleGroup',
                'GetSampledRequests',
                'GetWebACL',
                'GetWebACLForResource',
                'ListTagsForResource'
            ],
            'Permissions management': [
                'DeletePermissionPolicy',
                'PutPermissionPolicy'
            ],
            List: [
                'ListAPIKeys',
                'ListAvailableManagedRuleGroupVersions',
                'ListAvailableManagedRuleGroups',
                'ListIPSets',
                'ListLoggingConfigurations',
                'ListManagedRuleSets',
                'ListMobileSdkReleases',
                'ListRegexPatternSets',
                'ListResourcesForWebACL',
                'ListRuleGroups',
                'ListWebACLs'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Wafv2 = Wafv2;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid2FmdjIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ3YWZ2Mi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSx5Q0FBeUQ7QUFHekQ7Ozs7R0FJRztBQUNILE1BQWEsS0FBTSxTQUFRLHdCQUFlO0lBR3hDOzs7Ozs7Ozs7Ozs7Ozs7OztPQWlCRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUM7SUFDbEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQ2pDLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxpQkFBaUI7UUFDdEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlDQUFpQztRQUN0QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztJQUNwRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDRCQUE0QjtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwwQkFBMEI7UUFDL0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSxvQkFBb0I7UUFDekIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ3ZDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxVQUFVO1FBQ2YsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0kseUJBQXlCO1FBQzlCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0lBQzVDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7SUFDdEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxrQ0FBa0M7UUFDdkMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGtDQUFrQyxDQUFDLENBQUM7SUFDckQsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9CQUFvQjtRQUN6QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN2QyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksV0FBVztRQUNoQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxzQkFBc0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1Q0FBdUM7UUFDNUMsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVDQUF1QyxDQUFDLENBQUM7SUFDMUQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdDQUFnQztRQUNyQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksWUFBWTtRQUNqQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLENBQUMsQ0FBQztJQUM5QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNCQUFzQjtRQUMzQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0ksd0JBQXdCO1FBQzdCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLHFCQUFxQjtRQUMxQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUN4QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDhCQUE4QjtRQUNuQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLHlCQUF5QjtRQUM5QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7SUFDeEMsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksdUNBQXVDO1FBQzVDLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSx1QkFBdUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDMUMsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGlCQUFpQjtRQUN0QixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksY0FBYztRQUNuQixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDakMsQ0FBQztJQXVFRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNJLFFBQVEsQ0FBQyxLQUFhLEVBQUUsSUFBWSxFQUFFLEVBQVUsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM1RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixVQUFXLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxJQUFLLEtBQU0sV0FBWSxJQUFLLElBQUssRUFBRyxFQUFFLENBQUMsQ0FBQztJQUNoTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxPQUFPLENBQUMsS0FBYSxFQUFFLElBQVksRUFBRSxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0csT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsVUFBVyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsSUFBSyxLQUFNLFVBQVcsSUFBSyxJQUFLLEVBQUcsRUFBRSxDQUFDLENBQUM7SUFDL0ssQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksZ0JBQWdCLENBQUMsS0FBYSxFQUFFLElBQVksRUFBRSxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDcEgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsVUFBVyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsSUFBSyxLQUFNLG1CQUFvQixJQUFLLElBQUssRUFBRyxFQUFFLENBQUMsQ0FBQztJQUN4TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSxXQUFXLENBQUMsS0FBYSxFQUFFLElBQVksRUFBRSxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDL0csT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsVUFBVyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsSUFBSyxLQUFNLGNBQWUsSUFBSyxJQUFLLEVBQUcsRUFBRSxDQUFDLENBQUM7SUFDbkwsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0ksaUJBQWlCLENBQUMsS0FBYSxFQUFFLElBQVksRUFBRSxFQUFVLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsVUFBVyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsSUFBSyxLQUFNLG9CQUFxQixJQUFLLElBQUssRUFBRyxFQUFFLENBQUMsQ0FBQztJQUN6TCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGlCQUFpQixDQUFDLGdCQUF3QixFQUFFLGNBQXNCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDOUgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIseUJBQTBCLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxxQkFBc0IsZ0JBQWlCLElBQUssY0FBZSxFQUFFLENBQUMsQ0FBQztJQUN0TixDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFlBQVksQ0FBQyxLQUFhLEVBQUUsU0FBaUIsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixlQUFnQixNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsY0FBZSxLQUFNLFdBQVksU0FBVSxFQUFFLENBQUMsQ0FBQztJQUM1TCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksU0FBUyxDQUFDLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDMUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsWUFBYSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsU0FBVSxZQUFhLEVBQUUsQ0FBQyxDQUFDO0lBQ3JLLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxVQUFVLENBQUMsVUFBa0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixnQkFBaUIsTUFBTSxJQUFJLElBQUksQ0FBQyxhQUFjLElBQUssT0FBTyxJQUFJLElBQUksQ0FBQyxjQUFlLGFBQWMsVUFBVyxFQUFFLENBQUMsQ0FBQztJQUMzSyxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFdBQVcsQ0FBQyxXQUFtQixFQUFFLFNBQWlCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDOUcsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsY0FBZSxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsWUFBYSxXQUFZLElBQUssU0FBVSxFQUFFLENBQUMsQ0FBQztJQUN4TCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksd0JBQXdCLENBQUMsd0JBQWdDLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckgsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLE9BQVEsU0FBUyxJQUFJLElBQUksQ0FBQyxnQkFBaUIsUUFBUyxNQUFNLElBQUksSUFBSSxDQUFDLGFBQWMsSUFBSyxPQUFPLElBQUksSUFBSSxDQUFDLGNBQWUsNkJBQThCLHdCQUF5QixFQUFFLENBQUMsQ0FBQztJQUNqTSxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksWUFBWSxDQUFDLEtBQWEsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN0RixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBUSxTQUFTLElBQUksSUFBSSxDQUFDLGdCQUFpQixZQUFhLE1BQU0sSUFBSSxJQUFJLENBQUMsYUFBYyxJQUFLLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBZSxTQUFVLEtBQU0sRUFBRSxDQUFDLENBQUM7SUFDOUosQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLGVBQWUsQ0FBQyxNQUFjLEVBQUUsS0FBd0IsRUFBRSxRQUE0QjtRQUMzRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQW1CLE1BQU8sRUFBRSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09BNEJHO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYyxFQUFFLEtBQXdCLEVBQUUsUUFBNEI7UUFDNUYsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFvQixNQUFPLEVBQUUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxZQUFZLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUN4RSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsYUFBYSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDakUsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSx3QkFBd0IsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3BGLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFNBQVMsQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksVUFBVSxDQUFDLEtBQXdCLEVBQUUsUUFBNEI7UUFDdEUsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQzlELENBQUM7SUFFRDs7O09BR0c7SUFDSCxZQUFZLEtBQWdDO1FBQzFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztRQXBsQ1Isa0JBQWEsR0FBRyxPQUFPLENBQUM7UUErdEJyQixvQkFBZSxHQUFvQjtZQUMzQyxLQUFLLEVBQUU7Z0JBQ0wsaUJBQWlCO2dCQUNqQixjQUFjO2dCQUNkLGFBQWE7Z0JBQ2IsdUJBQXVCO2dCQUN2QixpQkFBaUI7Z0JBQ2pCLGNBQWM7Z0JBQ2QsY0FBYztnQkFDZCxpQ0FBaUM7Z0JBQ2pDLGFBQWE7Z0JBQ2IsNEJBQTRCO2dCQUM1Qix1QkFBdUI7Z0JBQ3ZCLGlCQUFpQjtnQkFDakIsY0FBYztnQkFDZCw2QkFBNkI7Z0JBQzdCLG9CQUFvQjtnQkFDcEIsOEJBQThCO2dCQUM5Qix5QkFBeUI7Z0JBQ3pCLDJCQUEyQjtnQkFDM0IsYUFBYTtnQkFDYix1Q0FBdUM7Z0JBQ3ZDLHVCQUF1QjtnQkFDdkIsaUJBQWlCO2dCQUNqQixjQUFjO2FBQ2Y7WUFDRCxJQUFJLEVBQUU7Z0JBQ0osZUFBZTtnQkFDZiw0QkFBNEI7Z0JBQzVCLGlDQUFpQztnQkFDakMsMEJBQTBCO2dCQUMxQiw2QkFBNkI7Z0JBQzdCLG9CQUFvQjtnQkFDcEIsVUFBVTtnQkFDVix5QkFBeUI7Z0JBQ3pCLG1CQUFtQjtnQkFDbkIscUJBQXFCO2dCQUNyQixxQkFBcUI7Z0JBQ3JCLGtDQUFrQztnQkFDbEMsb0JBQW9CO2dCQUNwQixjQUFjO2dCQUNkLG9CQUFvQjtnQkFDcEIsV0FBVztnQkFDWCxzQkFBc0I7Z0JBQ3RCLHFCQUFxQjthQUN0QjtZQUNELHdCQUF3QixFQUFFO2dCQUN4Qix3QkFBd0I7Z0JBQ3hCLHFCQUFxQjthQUN0QjtZQUNELElBQUksRUFBRTtnQkFDSixhQUFhO2dCQUNiLHVDQUF1QztnQkFDdkMsZ0NBQWdDO2dCQUNoQyxZQUFZO2dCQUNaLDJCQUEyQjtnQkFDM0IscUJBQXFCO2dCQUNyQix1QkFBdUI7Z0JBQ3ZCLHNCQUFzQjtnQkFDdEIsd0JBQXdCO2dCQUN4QixnQkFBZ0I7Z0JBQ2hCLGFBQWE7YUFDZDtZQUNELE9BQU8sRUFBRTtnQkFDUCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBbVRGLENBQUM7Q0FDRjtBQXZsQ0Qsc0JBdWxDQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gJy4uLy4uL3NoYXJlZC9hY2Nlc3MtbGV2ZWwnO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBPcGVyYXRvciB9IGZyb20gJy4uLy4uL3NoYXJlZCc7XG5pbXBvcnQgeyBhd3NfaWFtIGFzIGlhbSB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbd2FmdjJdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZXJ2aWNlLWF1dGhvcml6YXRpb24vbGF0ZXN0L3JlZmVyZW5jZS9saXN0X2F3c3dhZnYyLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgV2FmdjIgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICd3YWZ2Mic7XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSBhIFdlYkFDTCB3aXRoIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBhbXBsaWZ5OkFzc29jaWF0ZVdlYkFDTFxuICAgKiAtIGFwaWdhdGV3YXk6U2V0V2ViQUNMXG4gICAqIC0gYXBwcnVubmVyOkFzc29jaWF0ZVdlYkFjbFxuICAgKiAtIGFwcHN5bmM6U2V0V2ViQUNMXG4gICAqIC0gY29nbml0by1pZHA6QXNzb2NpYXRlV2ViQUNMXG4gICAqIC0gZWMyOkFzc29jaWF0ZVZlcmlmaWVkQWNjZXNzSW5zdGFuY2VXZWJBY2xcbiAgICogLSBlbGFzdGljbG9hZGJhbGFuY2luZzpTZXRXZWJBY2xcbiAgICogLSB3YWZ2MjpHZXRQZXJtaXNzaW9uUG9saWN5XG4gICAqIC0gd2FmdjI6UHV0UGVybWlzc2lvblBvbGljeVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Fzc29jaWF0ZVdlYkFDTC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9Bc3NvY2lhdGVXZWJBQ0woKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Fzc29jaWF0ZVdlYkFDTCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNhbGN1bGF0ZSB3ZWIgQUNMIGNhcGFjaXR5IHVuaXQgKFdDVSkgcmVxdWlyZW1lbnRzIGZvciBhIHNwZWNpZmllZCBzY29wZSBhbmQgc2V0IG9mIHJ1bGVzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NoZWNrQ2FwYWNpdHkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ2hlY2tDYXBhY2l0eSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ2hlY2tDYXBhY2l0eScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhbiBBUEkga2V5IGZvciB1c2UgaW4gdGhlIGludGVncmF0aW9uIG9mIHRoZSBDQVBUQ0hBIEFQSSBpbiB5b3VyIEphdmFTY3JpcHQgY2xpZW50IGFwcGxpY2F0aW9uc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlQVBJS2V5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUFQSUtleSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlQVBJS2V5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGFuIElQU2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gd2FmdjI6VGFnUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVJUFNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVJUFNldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlSVBTZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBjcmVhdGUgYSBSZWdleFBhdHRlcm5TZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSB3YWZ2MjpUYWdSZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZVJlZ2V4UGF0dGVyblNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVSZWdleFBhdHRlcm5TZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0NyZWF0ZVJlZ2V4UGF0dGVyblNldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGNyZWF0ZSBhIFJ1bGVHcm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHdhZnYyOlRhZ1Jlc291cmNlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlUnVsZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVJ1bGVHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnQ3JlYXRlUnVsZUdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIGEgV2ViQUNMXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gd2FmdjI6VGFnUmVzb3VyY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVXZWJBQ0wuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlV2ViQUNMKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdDcmVhdGVXZWJBQ0wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gQVBJIGtleVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlQVBJS2V5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUFQSUtleSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlQVBJS2V5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIEZpcmV3YWxsTWFuYWdlZFJ1bGVzR3JvdXBzIGZyb20gYSBXZWJBQ0wgaWYgbm90IG1hbmFnZWQgYnkgRmlyZXdhbGwgTWFuYWdlciBhbnltb3JlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVGaXJld2FsbE1hbmFnZXJSdWxlR3JvdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUZpcmV3YWxsTWFuYWdlclJ1bGVHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZUZpcmV3YWxsTWFuYWdlclJ1bGVHcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYW4gSVBTZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUlQU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUlQU2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVJUFNldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgTG9nZ2luZ0NvbmZpZ3VyYXRpb24gZnJvbSBhIFdlYkFDTFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmTG9nU2NvcGUoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUxvZ2dpbmdDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUxvZ2dpbmdDb25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVMb2dnaW5nQ29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSB0aGUgUGVybWlzc2lvblBvbGljeSBvbiBhIFJ1bGVHcm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFBlcm1pc3Npb25zIG1hbmFnZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVQZXJtaXNzaW9uUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVBlcm1pc3Npb25Qb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVBlcm1pc3Npb25Qb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkZWxldGUgYSBSZWdleFBhdHRlcm5TZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZVJlZ2V4UGF0dGVyblNldC5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVSZWdleFBhdHRlcm5TZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0RlbGV0ZVJlZ2V4UGF0dGVyblNldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRlbGV0ZSBhIFJ1bGVHcm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlUnVsZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVJ1bGVHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVsZXRlUnVsZUdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gZGVsZXRlIGEgV2ViQUNMXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVXZWJBQ0wuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlV2ViQUNMKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZWxldGVXZWJBQ0wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBwcm9kdWN0IGluZm9ybWF0aW9uIGZvciBhIG1hbmFnZWQgcnVsZSBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUFsbE1hbmFnZWRQcm9kdWN0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUFsbE1hbmFnZWRQcm9kdWN0cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnRGVzY3JpYmVBbGxNYW5hZ2VkUHJvZHVjdHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBwcm9kdWN0IGluZm9ybWF0aW9uIGZvciBhIG1hbmFnZWQgcnVsZSBncm91cCBieSBhIGdpdmVuIHZlbmRvclxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU1hbmFnZWRQcm9kdWN0c0J5VmVuZG9yLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlTWFuYWdlZFByb2R1Y3RzQnlWZW5kb3IoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0Rlc2NyaWJlTWFuYWdlZFByb2R1Y3RzQnlWZW5kb3InKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBoaWdoLWxldmVsIGluZm9ybWF0aW9uIGZvciBhIG1hbmFnZWQgcnVsZSBncm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZU1hbmFnZWRSdWxlR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVNYW5hZ2VkUnVsZUdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEZXNjcmliZU1hbmFnZWRSdWxlR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgRmlyZXdhbGwgTWFuYWdlciBmcm9tIGEgV2ViQUNMXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVGaXJld2FsbE1hbmFnZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlRmlyZXdhbGxNYW5hZ2VyKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVGaXJld2FsbE1hbmFnZXInKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBkaXNhc3NvY2lhdGUgYSBXZWJBQ0wgZnJvbSBhbiBhcHBsaWNhdGlvbiByZXNvdXJjZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGFtcGxpZnk6RGlzYXNzb2NpYXRlV2ViQUNMXG4gICAqIC0gYXBpZ2F0ZXdheTpTZXRXZWJBQ0xcbiAgICogLSBhcHBydW5uZXI6RGlzYXNzb2NpYXRlV2ViQWNsXG4gICAqIC0gYXBwc3luYzpTZXRXZWJBQ0xcbiAgICogLSBjb2duaXRvLWlkcDpEaXNhc3NvY2lhdGVXZWJBQ0xcbiAgICogLSBlYzI6RGlzYXNzb2NpYXRlVmVyaWZpZWRBY2Nlc3NJbnN0YW5jZVdlYkFjbFxuICAgKiAtIGVsYXN0aWNsb2FkYmFsYW5jaW5nOlNldFdlYkFjbFxuICAgKiAtIHdhZnYyOlB1dFBlcm1pc3Npb25Qb2xpY3lcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EaXNhc3NvY2lhdGVXZWJBQ0wuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGlzYXNzb2NpYXRlV2ViQUNMKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdEaXNhc3NvY2lhdGVXZWJBQ0wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBnZW5lcmF0ZSBhIHByZXNpZ25lZCBkb3dubG9hZCBVUkwgZm9yIHRoZSBzcGVjaWZpZWQgcmVsZWFzZSBvZiB0aGUgbW9iaWxlIFNES1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZW5lcmF0ZU1vYmlsZVNka1JlbGVhc2VVcmwuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2VuZXJhdGVNb2JpbGVTZGtSZWxlYXNlVXJsKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZW5lcmF0ZU1vYmlsZVNka1JlbGVhc2VVcmwnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXR1cm4geW91ciBBUEkga2V5IGluIGRlY3J5cHRlZCBmb3JtLiBVc2UgdGhpcyB0byBjaGVjayB0aGUgdG9rZW4gZG9tYWlucyB0aGF0IHlvdSBoYXZlIGRlZmluZWQgZm9yIHRoZSBrZXlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0RGVjcnlwdGVkQVBJS2V5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERlY3J5cHRlZEFQSUtleSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0RGVjcnlwdGVkQVBJS2V5Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgZGV0YWlscyBhYm91dCBhbiBJUFNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0SVBTZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0SVBTZXQoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldElQU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgTG9nZ2luZ0NvbmZpZ3VyYXRpb24gZm9yIGEgV2ViQUNMXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZMb2dTY29wZSgpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0TG9nZ2luZ0NvbmZpZ3VyYXRpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0TG9nZ2luZ0NvbmZpZ3VyYXRpb24oKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldExvZ2dpbmdDb25maWd1cmF0aW9uJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgZGV0YWlscyBhYm91dCBhIE1hbmFnZWRSdWxlU2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldE1hbmFnZWRSdWxlU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldE1hbmFnZWRSdWxlU2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRNYW5hZ2VkUnVsZVNldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGluZm9ybWF0aW9uIGZvciB0aGUgc3BlY2lmaWVkIG1vYmlsZSBTREsgcmVsZWFzZSwgaW5jbHVkaW5nIHJlbGVhc2Ugbm90ZXMgYW5kIHRhZ3NcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0TW9iaWxlU2RrUmVsZWFzZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRNb2JpbGVTZGtSZWxlYXNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRNb2JpbGVTZGtSZWxlYXNlJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBQZXJtaXNzaW9uUG9saWN5IGZvciBhIFJ1bGVHcm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXRQZXJtaXNzaW9uUG9saWN5Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFBlcm1pc3Npb25Qb2xpY3koKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFBlcm1pc3Npb25Qb2xpY3knKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSB0aGUga2V5cyB0aGF0IGFyZSBjdXJyZW50bHkgYmxvY2tlZCBieSBhIHJhdGUtYmFzZWQgcnVsZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0UmF0ZUJhc2VkU3RhdGVtZW50TWFuYWdlZEtleXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0UmF0ZUJhc2VkU3RhdGVtZW50TWFuYWdlZEtleXMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0dldFJhdGVCYXNlZFN0YXRlbWVudE1hbmFnZWRLZXlzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgZGV0YWlscyBhYm91dCBhIFJlZ2V4UGF0dGVyblNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0UmVnZXhQYXR0ZXJuU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFJlZ2V4UGF0dGVyblNldCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0UmVnZXhQYXR0ZXJuU2V0Jyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgZGV0YWlscyBhYm91dCBhIFJ1bGVHcm91cFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0UnVsZUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFJ1bGVHcm91cCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0UnVsZUdyb3VwJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgZGV0YWlsZWQgaW5mb3JtYXRpb24gYWJvdXQgYSBzYW1wbGluZyBvZiB3ZWIgcmVxdWVzdHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0U2FtcGxlZFJlcXVlc3RzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNhbXBsZWRSZXF1ZXN0cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0U2FtcGxlZFJlcXVlc3RzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgZGV0YWlscyBhYm91dCBhIFdlYkFDTFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0V2ViQUNMLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFdlYkFDTCgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnR2V0V2ViQUNMJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgdGhlIFdlYkFDTCB0aGF0J3MgYXNzb2NpYXRlZCB3aXRoIGEgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGFtcGxpZnk6R2V0V2ViQUNMRm9yUmVzb3VyY2VcbiAgICogLSBhcHBydW5uZXI6RGVzY3JpYmVXZWJBY2xGb3JTZXJ2aWNlXG4gICAqIC0gY29nbml0by1pZHA6R2V0V2ViQUNMRm9yUmVzb3VyY2VcbiAgICogLSBlYzI6R2V0VmVyaWZpZWRBY2Nlc3NJbnN0YW5jZVdlYkFjbFxuICAgKiAtIHdhZnYyOkdldFdlYkFDTFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFdlYkFDTEZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFdlYkFDTEZvclJlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdHZXRXZWJBQ0xGb3JSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGEgbGlzdCBvZiB0aGUgQVBJIGtleXMgdGhhdCB5b3UndmUgZGVmaW5lZCBmb3IgdGhlIHNwZWNpZmllZCBzY29wZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0QVBJS2V5cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QVBJS2V5cygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEFQSUtleXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhbiBhcnJheSBvZiBtYW5hZ2VkIHJ1bGUgZ3JvdXAgdmVyc2lvbnMgdGhhdCBhcmUgYXZhaWxhYmxlIGZvciB5b3UgdG8gdXNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBdmFpbGFibGVNYW5hZ2VkUnVsZUdyb3VwVmVyc2lvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEF2YWlsYWJsZU1hbmFnZWRSdWxlR3JvdXBWZXJzaW9ucygpIHtcbiAgICByZXR1cm4gdGhpcy50bygnTGlzdEF2YWlsYWJsZU1hbmFnZWRSdWxlR3JvdXBWZXJzaW9ucycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGFuIGFycmF5IG9mIG1hbmFnZWQgcnVsZSBncm91cHMgdGhhdCBhcmUgYXZhaWxhYmxlIGZvciB5b3UgdG8gdXNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RBdmFpbGFibGVNYW5hZ2VkUnVsZUdyb3Vwcy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QXZhaWxhYmxlTWFuYWdlZFJ1bGVHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RBdmFpbGFibGVNYW5hZ2VkUnVsZUdyb3VwcycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGFuIGFycmF5IG9mIElQU2V0U3VtbWFyeSBvYmplY3RzIGZvciB0aGUgSVAgc2V0cyB0aGF0IHlvdSBtYW5hZ2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdElQU2V0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0SVBTZXRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0SVBTZXRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYW4gYXJyYXkgb2YgeW91ciBMb2dnaW5nQ29uZmlndXJhdGlvbiBvYmplY3RzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkxvZ1Njb3BlKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0TG9nZ2luZ0NvbmZpZ3VyYXRpb25zLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RMb2dnaW5nQ29uZmlndXJhdGlvbnMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RMb2dnaW5nQ29uZmlndXJhdGlvbnMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhbiBhcnJheSBvZiB5b3VyIE1hbmFnZWRSdWxlU2V0IG9iamVjdHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdE1hbmFnZWRSdWxlU2V0cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0TWFuYWdlZFJ1bGVTZXRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0TWFuYWdlZFJ1bGVTZXRzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gcmV0cmlldmUgYSBsaXN0IG9mIHRoZSBhdmFpbGFibGUgcmVsZWFzZXMgZm9yIHRoZSBtb2JpbGUgU0RLIGFuZCB0aGUgc3BlY2lmaWVkIGRldmljZSBwbGF0Zm9ybVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0TW9iaWxlU2RrUmVsZWFzZXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdE1vYmlsZVNka1JlbGVhc2VzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0TW9iaWxlU2RrUmVsZWFzZXMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhbiBhcnJheSBvZiBSZWdleFBhdHRlcm5TZXRTdW1tYXJ5IG9iamVjdHMgZm9yIHRoZSByZWdleCBwYXR0ZXJuIHNldHMgdGhhdCB5b3UgbWFuYWdlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogTGlzdFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RSZWdleFBhdHRlcm5TZXRzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RSZWdleFBhdHRlcm5TZXRzKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdMaXN0UmVnZXhQYXR0ZXJuU2V0cycpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHJldHJpZXZlIGFuIGFycmF5IG9mIHRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZXMgKEFSTnMpIGZvciB0aGUgcmVzb3VyY2VzIHRoYXQgYXJlIGFzc29jaWF0ZWQgd2l0aCBhIHdlYiBBQ0xcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGFtcGxpZnk6TGlzdFJlc291cmNlc0ZvcldlYkFDTFxuICAgKiAtIGFwcHJ1bm5lcjpMaXN0QXNzb2NpYXRlZFNlcnZpY2VzRm9yV2ViQWNsXG4gICAqIC0gY29nbml0by1pZHA6TGlzdFJlc291cmNlc0ZvcldlYkFDTFxuICAgKiAtIGVjMjpEZXNjcmliZVZlcmlmaWVkQWNjZXNzSW5zdGFuY2VXZWJBY2xBc3NvY2lhdGlvbnNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0UmVzb3VyY2VzRm9yV2ViQUNMLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RSZXNvdXJjZXNGb3JXZWJBQ0woKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RSZXNvdXJjZXNGb3JXZWJBQ0wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhbiBhcnJheSBvZiBSdWxlR3JvdXBTdW1tYXJ5IG9iamVjdHMgZm9yIHRoZSBydWxlIGdyb3VwcyB0aGF0IHlvdSBtYW5hZ2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFJ1bGVHcm91cHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFJ1bGVHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RSdWxlR3JvdXBzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gbGlzdCB0YWdzIGZvciBhIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9MaXN0VGFnc0ZvclJlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUYWdzRm9yUmVzb3VyY2UoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byByZXRyaWV2ZSBhbiBhcnJheSBvZiBXZWJBQ0xTdW1tYXJ5IG9iamVjdHMgZm9yIHRoZSB3ZWIgQUNMcyB0aGF0IHlvdSBtYW5hZ2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFdlYkFDTHMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFdlYkFDTHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ0xpc3RXZWJBQ0xzJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gY3JlYXRlIEZpcmV3YWxsTWFuYWdlZFJ1bGVzR3JvdXBzIGluIGEgV2ViQUNMXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9QdXRGaXJld2FsbE1hbmFnZXJSdWxlR3JvdXBzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dEZpcmV3YWxsTWFuYWdlclJ1bGVHcm91cHMoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1B1dEZpcmV3YWxsTWFuYWdlclJ1bGVHcm91cHMnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byBlbmFibGUgYSBMb2dnaW5nQ29uZmlndXJhdGlvbiwgdG8gc3RhcnQgbG9nZ2luZyBmb3IgYSB3ZWIgQUNMXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZMb2dTY29wZSgpXG4gICAqIC0gLmlmTG9nRGVzdGluYXRpb25SZXNvdXJjZSgpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpDcmVhdGVTZXJ2aWNlTGlua2VkUm9sZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dExvZ2dpbmdDb25maWd1cmF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1B1dExvZ2dpbmdDb25maWd1cmF0aW9uKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQdXRMb2dnaW5nQ29uZmlndXJhdGlvbicpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGVuYWJsZSBjcmVhdGUgYSBuZXcgb3IgdXBkYXRlIGFuIGV4aXN0aW5nIHZlcnNpb24gb2YgYSBNYW5hZ2VkUnVsZVNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0TWFuYWdlZFJ1bGVTZXRWZXJzaW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9QdXRNYW5hZ2VkUnVsZVNldFZlcnNpb25zKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdQdXRNYW5hZ2VkUnVsZVNldFZlcnNpb25zJyk7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHBlcm1pc3Npb24gdG8gYXR0YWNoIGFuIElBTSBwb2xpY3kgdG8gYSByZXNvdXJjZSwgdXNlZCB0byBzaGFyZSBydWxlIGdyb3VwcyBiZXR3ZWVuIGFjY291bnRzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUGVybWlzc2lvbnMgbWFuYWdlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1B1dFBlcm1pc3Npb25Qb2xpY3kuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvUHV0UGVybWlzc2lvblBvbGljeSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnUHV0UGVybWlzc2lvblBvbGljeScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGFzc29jaWF0ZSB0YWdzIHdpdGggYSBBV1MgcmVzb3VyY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1RhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdUYWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIGRpc2Fzc29jaWF0ZSB0YWdzIGZyb20gYW4gQVdTIHJlc291cmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VudGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVW50YWdSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy50bygnVW50YWdSZXNvdXJjZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSBhbiBJUFNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZUlQU2V0Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUlQU2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVJUFNldCcpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwZXJtaXNzaW9uIHRvIHVwZGF0ZSB0aGUgZXhwaXJ5IGRhdGUgb2YgYSB2ZXJzaW9uIGluIE1hbmFnZWRSdWxlU2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVNYW5hZ2VkUnVsZVNldFZlcnNpb25FeHBpcnlEYXRlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZU1hbmFnZWRSdWxlU2V0VmVyc2lvbkV4cGlyeURhdGUoKSB7XG4gICAgcmV0dXJuIHRoaXMudG8oJ1VwZGF0ZU1hbmFnZWRSdWxlU2V0VmVyc2lvbkV4cGlyeURhdGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBSZWdleFBhdHRlcm5TZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVSZWdleFBhdHRlcm5TZXQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUmVnZXhQYXR0ZXJuU2V0KCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVSZWdleFBhdHRlcm5TZXQnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBSdWxlR3JvdXBcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVSdWxlR3JvdXAuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUnVsZUdyb3VwKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVSdWxlR3JvdXAnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHcmFudHMgcGVybWlzc2lvbiB0byB1cGRhdGUgYSBXZWJBQ0xcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVXZWJBQ0wuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlV2ViQUNMKCkge1xuICAgIHJldHVybiB0aGlzLnRvKCdVcGRhdGVXZWJBQ0wnKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBhY2Nlc3NMZXZlbExpc3Q6IEFjY2Vzc0xldmVsTGlzdCA9IHtcbiAgICBXcml0ZTogW1xuICAgICAgJ0Fzc29jaWF0ZVdlYkFDTCcsXG4gICAgICAnQ3JlYXRlQVBJS2V5JyxcbiAgICAgICdDcmVhdGVJUFNldCcsXG4gICAgICAnQ3JlYXRlUmVnZXhQYXR0ZXJuU2V0JyxcbiAgICAgICdDcmVhdGVSdWxlR3JvdXAnLFxuICAgICAgJ0NyZWF0ZVdlYkFDTCcsXG4gICAgICAnRGVsZXRlQVBJS2V5JyxcbiAgICAgICdEZWxldGVGaXJld2FsbE1hbmFnZXJSdWxlR3JvdXBzJyxcbiAgICAgICdEZWxldGVJUFNldCcsXG4gICAgICAnRGVsZXRlTG9nZ2luZ0NvbmZpZ3VyYXRpb24nLFxuICAgICAgJ0RlbGV0ZVJlZ2V4UGF0dGVyblNldCcsXG4gICAgICAnRGVsZXRlUnVsZUdyb3VwJyxcbiAgICAgICdEZWxldGVXZWJBQ0wnLFxuICAgICAgJ0Rpc2Fzc29jaWF0ZUZpcmV3YWxsTWFuYWdlcicsXG4gICAgICAnRGlzYXNzb2NpYXRlV2ViQUNMJyxcbiAgICAgICdQdXRGaXJld2FsbE1hbmFnZXJSdWxlR3JvdXBzJyxcbiAgICAgICdQdXRMb2dnaW5nQ29uZmlndXJhdGlvbicsXG4gICAgICAnUHV0TWFuYWdlZFJ1bGVTZXRWZXJzaW9ucycsXG4gICAgICAnVXBkYXRlSVBTZXQnLFxuICAgICAgJ1VwZGF0ZU1hbmFnZWRSdWxlU2V0VmVyc2lvbkV4cGlyeURhdGUnLFxuICAgICAgJ1VwZGF0ZVJlZ2V4UGF0dGVyblNldCcsXG4gICAgICAnVXBkYXRlUnVsZUdyb3VwJyxcbiAgICAgICdVcGRhdGVXZWJBQ0wnXG4gICAgXSxcbiAgICBSZWFkOiBbXG4gICAgICAnQ2hlY2tDYXBhY2l0eScsXG4gICAgICAnRGVzY3JpYmVBbGxNYW5hZ2VkUHJvZHVjdHMnLFxuICAgICAgJ0Rlc2NyaWJlTWFuYWdlZFByb2R1Y3RzQnlWZW5kb3InLFxuICAgICAgJ0Rlc2NyaWJlTWFuYWdlZFJ1bGVHcm91cCcsXG4gICAgICAnR2VuZXJhdGVNb2JpbGVTZGtSZWxlYXNlVXJsJyxcbiAgICAgICdHZXREZWNyeXB0ZWRBUElLZXknLFxuICAgICAgJ0dldElQU2V0JyxcbiAgICAgICdHZXRMb2dnaW5nQ29uZmlndXJhdGlvbicsXG4gICAgICAnR2V0TWFuYWdlZFJ1bGVTZXQnLFxuICAgICAgJ0dldE1vYmlsZVNka1JlbGVhc2UnLFxuICAgICAgJ0dldFBlcm1pc3Npb25Qb2xpY3knLFxuICAgICAgJ0dldFJhdGVCYXNlZFN0YXRlbWVudE1hbmFnZWRLZXlzJyxcbiAgICAgICdHZXRSZWdleFBhdHRlcm5TZXQnLFxuICAgICAgJ0dldFJ1bGVHcm91cCcsXG4gICAgICAnR2V0U2FtcGxlZFJlcXVlc3RzJyxcbiAgICAgICdHZXRXZWJBQ0wnLFxuICAgICAgJ0dldFdlYkFDTEZvclJlc291cmNlJyxcbiAgICAgICdMaXN0VGFnc0ZvclJlc291cmNlJ1xuICAgIF0sXG4gICAgJ1Blcm1pc3Npb25zIG1hbmFnZW1lbnQnOiBbXG4gICAgICAnRGVsZXRlUGVybWlzc2lvblBvbGljeScsXG4gICAgICAnUHV0UGVybWlzc2lvblBvbGljeSdcbiAgICBdLFxuICAgIExpc3Q6IFtcbiAgICAgICdMaXN0QVBJS2V5cycsXG4gICAgICAnTGlzdEF2YWlsYWJsZU1hbmFnZWRSdWxlR3JvdXBWZXJzaW9ucycsXG4gICAgICAnTGlzdEF2YWlsYWJsZU1hbmFnZWRSdWxlR3JvdXBzJyxcbiAgICAgICdMaXN0SVBTZXRzJyxcbiAgICAgICdMaXN0TG9nZ2luZ0NvbmZpZ3VyYXRpb25zJyxcbiAgICAgICdMaXN0TWFuYWdlZFJ1bGVTZXRzJyxcbiAgICAgICdMaXN0TW9iaWxlU2RrUmVsZWFzZXMnLFxuICAgICAgJ0xpc3RSZWdleFBhdHRlcm5TZXRzJyxcbiAgICAgICdMaXN0UmVzb3VyY2VzRm9yV2ViQUNMJyxcbiAgICAgICdMaXN0UnVsZUdyb3VwcycsXG4gICAgICAnTGlzdFdlYkFDTHMnXG4gICAgXSxcbiAgICBUYWdnaW5nOiBbXG4gICAgICAnVGFnUmVzb3VyY2UnLFxuICAgICAgJ1VudGFnUmVzb3VyY2UnXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB3ZWJhY2wgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1dlYkFDTC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzY29wZS5cbiAgICogQHBhcmFtIG5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgbmFtZS5cbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25XZWJhY2woc2NvcGU6IHN0cmluZywgbmFtZTogc3RyaW5nLCBpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OndhZnYyOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTokeyBzY29wZSB9L3dlYmFjbC8keyBuYW1lIH0vJHsgaWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGlwc2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9JUFNldC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzY29wZS5cbiAgICogQHBhcmFtIG5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgbmFtZS5cbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25JcHNldChzY29wZTogc3RyaW5nLCBuYW1lOiBzdHJpbmcsIGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06d2FmdjI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OiR7IHNjb3BlIH0vaXBzZXQvJHsgbmFtZSB9LyR7IGlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBtYW5hZ2VkcnVsZXNldCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTWFuYWdlZFJ1bGVTZXQuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgLSBJZGVudGlmaWVyIGZvciB0aGUgc2NvcGUuXG4gICAqIEBwYXJhbSBuYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIG5hbWUuXG4gICAqIEBwYXJhbSBpZCAtIElkZW50aWZpZXIgZm9yIHRoZSBpZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25NYW5hZ2VkcnVsZXNldChzY29wZTogc3RyaW5nLCBuYW1lOiBzdHJpbmcsIGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06d2FmdjI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OiR7IHNjb3BlIH0vbWFuYWdlZHJ1bGVzZXQvJHsgbmFtZSB9LyR7IGlkIH1gKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBydWxlZ3JvdXAgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1J1bGVHcm91cC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSAtIElkZW50aWZpZXIgZm9yIHRoZSBzY29wZS5cbiAgICogQHBhcmFtIG5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgbmFtZS5cbiAgICogQHBhcmFtIGlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqL1xuICBwdWJsaWMgb25SdWxlZ3JvdXAoc2NvcGU6IHN0cmluZywgbmFtZTogc3RyaW5nLCBpZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OndhZnYyOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTokeyBzY29wZSB9L3J1bGVncm91cC8keyBuYW1lIH0vJHsgaWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHJlZ2V4cGF0dGVybnNldCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUmVnZXhQYXR0ZXJuU2V0Lmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHNjb3BlIC0gSWRlbnRpZmllciBmb3IgdGhlIHNjb3BlLlxuICAgKiBAcGFyYW0gbmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBuYW1lLlxuICAgKiBAcGFyYW0gaWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblJlZ2V4cGF0dGVybnNldChzY29wZTogc3RyaW5nLCBuYW1lOiBzdHJpbmcsIGlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06d2FmdjI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OiR7IHNjb3BlIH0vcmVnZXhwYXR0ZXJuc2V0LyR7IG5hbWUgfS8keyBpZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgbG9hZGJhbGFuY2VyL2FwcC8gdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1dlYkFDTC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBsb2FkQmFsYW5jZXJOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGxvYWRCYWxhbmNlck5hbWUuXG4gICAqIEBwYXJhbSBsb2FkQmFsYW5jZXJJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBsb2FkQmFsYW5jZXJJZC5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25Mb2FkYmFsYW5jZXJBcHAobG9hZEJhbGFuY2VyTmFtZTogc3RyaW5nLCBsb2FkQmFsYW5jZXJJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmVsYXN0aWNsb2FkYmFsYW5jaW5nOiR7IHJlZ2lvbiA/PyB0aGlzLmRlZmF1bHRSZWdpb24gfTokeyBhY2NvdW50ID8/IHRoaXMuZGVmYXVsdEFjY291bnQgfTpsb2FkYmFsYW5jZXIvYXBwLyR7IGxvYWRCYWxhbmNlck5hbWUgfS8keyBsb2FkQmFsYW5jZXJJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXBpZ2F0ZXdheSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfV2ViQUNMLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIGFwaUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGFwaUlkLlxuICAgKiBAcGFyYW0gc3RhZ2VOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHN0YWdlTmFtZS5cbiAgICogQHBhcmFtIGFjY291bnQgLSBBY2NvdW50IG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBhY2NvdW50LlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gYCpgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLCB1bmxlc3MgdXNpbmcgdGhlIENESywgd2hlcmUgdGhlIGRlZmF1bHQgaXMgdGhlIGN1cnJlbnQgU3RhY2sncyBwYXJ0aXRpb24uXG4gICAqL1xuICBwdWJsaWMgb25BcGlnYXRld2F5KGFwaUlkOiBzdHJpbmcsIHN0YWdlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFwaWdhdGV3YXk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9Oi9yZXN0YXBpcy8keyBhcGlJZCB9L3N0YWdlcy8keyBzdGFnZU5hbWUgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGFwcHN5bmMgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1dlYkFDTC5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBncmFwaFFMQVBJSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZ3JhcGhRTEFQSUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkFwcHN5bmMoZ3JhcGhRTEFQSUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMub24oYGFybjokeyBwYXJ0aXRpb24gPz8gdGhpcy5kZWZhdWx0UGFydGl0aW9uIH06YXBwc3luYzokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06YXBpcy8keyBncmFwaFFMQVBJSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHVzZXJwb29sIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9XZWJBQ0wuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gdXNlclBvb2xJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB1c2VyUG9vbElkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvblVzZXJwb29sKHVzZXJQb29sSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTpjb2duaXRvLWlkcDokeyByZWdpb24gPz8gdGhpcy5kZWZhdWx0UmVnaW9uIH06JHsgYWNjb3VudCA/PyB0aGlzLmRlZmF1bHRBY2NvdW50IH06dXNlcnBvb2wvJHsgdXNlclBvb2xJZCB9YCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgYXBwcnVubmVyIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9XZWJBQ0wuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gc2VydmljZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgc2VydmljZU5hbWUuXG4gICAqIEBwYXJhbSBzZXJ2aWNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgc2VydmljZUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIGFjY291bnQuXG4gICAqIEBwYXJhbSByZWdpb24gLSBSZWdpb24gb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBgKmAsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHJlZ2lvbi5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AsIHVubGVzcyB1c2luZyB0aGUgQ0RLLCB3aGVyZSB0aGUgZGVmYXVsdCBpcyB0aGUgY3VycmVudCBTdGFjaydzIHBhcnRpdGlvbi5cbiAgICovXG4gIHB1YmxpYyBvbkFwcHJ1bm5lcihzZXJ2aWNlTmFtZTogc3RyaW5nLCBzZXJ2aWNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTphcHBydW5uZXI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnNlcnZpY2UvJHsgc2VydmljZU5hbWUgfS8keyBzZXJ2aWNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIHZlcmlmaWVkLWFjY2Vzcy1pbnN0YW5jZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfV2ViQUNMLmh0bWxcbiAgICpcbiAgICogQHBhcmFtIHZlcmlmaWVkQWNjZXNzSW5zdGFuY2VJZCAtIElkZW50aWZpZXIgZm9yIHRoZSB2ZXJpZmllZEFjY2Vzc0luc3RhbmNlSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uVmVyaWZpZWRBY2Nlc3NJbnN0YW5jZSh2ZXJpZmllZEFjY2Vzc0luc3RhbmNlSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5vbihgYXJuOiR7IHBhcnRpdGlvbiA/PyB0aGlzLmRlZmF1bHRQYXJ0aXRpb24gfTplYzI6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OnZlcmlmaWVkLWFjY2Vzcy1pbnN0YW5jZS8keyB2ZXJpZmllZEFjY2Vzc0luc3RhbmNlSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGFtcGxpZnktYXBwIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9XZWJBQ0wuaHRtbFxuICAgKlxuICAgKiBAcGFyYW0gYXBwSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgYXBwSWQuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgYWNjb3VudC5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGAqYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcmVnaW9uLlxuICAgKiBAcGFyYW0gcGFydGl0aW9uIC0gUGFydGl0aW9uIG9mIHRoZSBBV1MgYWNjb3VudCBbYXdzLCBhd3MtY24sIGF3cy11cy1nb3ZdOyBkZWZhdWx0cyB0byBgYXdzYCwgdW5sZXNzIHVzaW5nIHRoZSBDREssIHdoZXJlIHRoZSBkZWZhdWx0IGlzIHRoZSBjdXJyZW50IFN0YWNrJ3MgcGFydGl0aW9uLlxuICAgKi9cbiAgcHVibGljIG9uQW1wbGlmeUFwcChhcHBJZDogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLm9uKGBhcm46JHsgcGFydGl0aW9uID8/IHRoaXMuZGVmYXVsdFBhcnRpdGlvbiB9OmFtcGxpZnk6JHsgcmVnaW9uID8/IHRoaXMuZGVmYXVsdFJlZ2lvbiB9OiR7IGFjY291bnQgPz8gdGhpcy5kZWZhdWx0QWNjb3VudCB9OmFwcHMvJHsgYXBwSWQgfWApO1xuICB9XG5cbiAgLyoqXG4gICAqIEZpbHRlcnMgYWNjZXNzIGJ5IHRoZSBhbGxvd2VkIHNldCBvZiB2YWx1ZXMgZm9yIGVhY2ggb2YgdGhlIHRhZ3NcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19jb25kaXRpb24ta2V5cy5odG1sI2NvbmRpdGlvbi1rZXlzLXJlcXVlc3R0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0NyZWF0ZUlQU2V0KClcbiAgICogLSAudG9DcmVhdGVSZWdleFBhdHRlcm5TZXQoKVxuICAgKiAtIC50b0NyZWF0ZVJ1bGVHcm91cCgpXG4gICAqIC0gLnRvQ3JlYXRlV2ViQUNMKClcbiAgICogLSAudG9UYWdSZXNvdXJjZSgpXG4gICAqXG4gICAqIEBwYXJhbSB0YWdLZXkgVGhlIHRhZyBrZXkgdG8gY2hlY2tcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBd3NSZXF1ZXN0VGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlcXVlc3RUYWcvJHsgdGFnS2V5IH1gLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSB0YWctdmFsdWUgYXNzb2NpYXRlZCB3aXRoIHRoZSByZXNvdXJjZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2NvbmRpdGlvbi1rZXlzLmh0bWwjY29uZGl0aW9uLWtleXMtcmVzb3VyY2V0YWdcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b0dldElQU2V0KClcbiAgICogLSAudG9HZXRMb2dnaW5nQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvR2V0UmF0ZUJhc2VkU3RhdGVtZW50TWFuYWdlZEtleXMoKVxuICAgKiAtIC50b0dldFJlZ2V4UGF0dGVyblNldCgpXG4gICAqIC0gLnRvR2V0UnVsZUdyb3VwKClcbiAgICogLSAudG9HZXRXZWJBQ0woKVxuICAgKiAtIC50b0xpc3RUYWdzRm9yUmVzb3VyY2UoKVxuICAgKiAtIC50b1RhZ1Jlc291cmNlKClcbiAgICogLSAudG9VcGRhdGVJUFNldCgpXG4gICAqIC0gLnRvVXBkYXRlUmVnZXhQYXR0ZXJuU2V0KClcbiAgICogLSAudG9VcGRhdGVSdWxlR3JvdXAoKVxuICAgKiAtIC50b1VwZGF0ZVdlYkFDTCgpXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gcmVzb3VyY2UgdHlwZXM6XG4gICAqIC0gd2ViYWNsXG4gICAqIC0gaXBzZXRcbiAgICogLSBydWxlZ3JvdXBcbiAgICogLSByZWdleHBhdHRlcm5zZXRcbiAgICpcbiAgICogQHBhcmFtIHRhZ0tleSBUaGUgdGFnIGtleSB0byBjaGVja1xuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1Jlc291cmNlVGFnKHRhZ0tleTogc3RyaW5nLCB2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYXdzOlJlc291cmNlVGFnLyR7IHRhZ0tleSB9YCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgdGhlIHByZXNlbmNlIG9mIG1hbmRhdG9yeSB0YWdzIGluIHRoZSByZXF1ZXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfY29uZGl0aW9uLWtleXMuaHRtbCNjb25kaXRpb24ta2V5cy10YWdrZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9DcmVhdGVJUFNldCgpXG4gICAqIC0gLnRvQ3JlYXRlUmVnZXhQYXR0ZXJuU2V0KClcbiAgICogLSAudG9DcmVhdGVSdWxlR3JvdXAoKVxuICAgKiAtIC50b0NyZWF0ZVdlYkFDTCgpXG4gICAqIC0gLnRvVGFnUmVzb3VyY2UoKVxuICAgKiAtIC50b1VudGFnUmVzb3VyY2UoKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkF3c1RhZ0tleXModmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGF3czpUYWdLZXlzYCwgdmFsdWUsIG9wZXJhdG9yID8/ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogRmlsdGVycyBhY2Nlc3MgYnkgbG9nIGRlc3RpbmF0aW9uIEFSTiBmb3IgUHV0TG9nZ2luZ0NvbmZpZ3VyYXRpb24gQVBJXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS93YWYvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL3NlY3VyaXR5X2lhbV9zZXJ2aWNlLXdpdGgtaWFtLmh0bWwjc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0taWQtYmFzZWQtcG9saWNpZXMtY29uZGl0aW9ua2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUHV0TG9nZ2luZ0NvbmZpZ3VyYXRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFthcm4gb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX0FSTikuICoqRGVmYXVsdDoqKiBgQXJuTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkxvZ0Rlc3RpbmF0aW9uUmVzb3VyY2UodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYExvZ0Rlc3RpbmF0aW9uUmVzb3VyY2VgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ0Fybkxpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGaWx0ZXJzIGFjY2VzcyBieSBsb2cgc2NvcGUgZm9yIExvZ2dpbmcgQ29uZmlndXJhdGlvbiBBUElcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3dhZi9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvc2VjdXJpdHlfaWFtX3NlcnZpY2Utd2l0aC1pYW0uaHRtbCNzZWN1cml0eV9pYW1fc2VydmljZS13aXRoLWlhbS1pZC1iYXNlZC1wb2xpY2llcy1jb25kaXRpb25rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9EZWxldGVMb2dnaW5nQ29uZmlndXJhdGlvbigpXG4gICAqIC0gLnRvR2V0TG9nZ2luZ0NvbmZpZ3VyYXRpb24oKVxuICAgKiAtIC50b0xpc3RMb2dnaW5nQ29uZmlndXJhdGlvbnMoKVxuICAgKiAtIC50b1B1dExvZ2dpbmdDb25maWd1cmF0aW9uKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZMb2dTY29wZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgTG9nU2NvcGVgLCB2YWx1ZSwgb3BlcmF0b3IgPz8gJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3dhZnYyXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2VydmljZS1hdXRob3JpemF0aW9uL2xhdGVzdC9yZWZlcmVuY2UvbGlzdF9hd3N3YWZ2Mi5odG1sKS5cbiAgICpcbiAgICovXG4gIGNvbnN0cnVjdG9yKHByb3BzPzogaWFtLlBvbGljeVN0YXRlbWVudFByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuICB9XG59XG4iXX0=