"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wellarchitected = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [wellarchitected](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awswell-architectedtool.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Wellarchitected extends shared_1.PolicyStatement {
    /**
     * Grants permission to associate a lens to the specified workload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_AssociateLenses.html
     */
    toAssociateLenses() {
        return this.to('AssociateLenses');
    }
    /**
     * Grants permission to associate a profile to the specified workload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_AssociateProfiles.html
     */
    toAssociateProfiles() {
        return this.to('AssociateProfiles');
    }
    /**
     * Grants permission to configure the integration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/setting-up-jira.html
     */
    toConfigureIntegration() {
        return this.to('ConfigureIntegration');
    }
    /**
     * Grants permission to an owner of a lens to share with other AWS accounts and IAM users
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_CreateLensShare.html
     */
    toCreateLensShare() {
        return this.to('CreateLensShare');
    }
    /**
     * Grants permission to create a new lens version
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_CreateLensVersion.html
     */
    toCreateLensVersion() {
        return this.to('CreateLensVersion');
    }
    /**
     * Grants permission to create a new milestone for the specified workload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_CreateMilestone.html
     */
    toCreateMilestone() {
        return this.to('CreateMilestone');
    }
    /**
     * Grants permission to create a new profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_CreateProfile.html
     */
    toCreateProfile() {
        return this.to('CreateProfile');
    }
    /**
     * Grants permission to an owner of a profile to share with other AWS accounts and IAM users
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_CreateProfileShare.html
     */
    toCreateProfileShare() {
        return this.to('CreateProfileShare');
    }
    /**
     * Grants permission to create a new review template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_CreateReviewTemplate.html
     */
    toCreateReviewTemplate() {
        return this.to('CreateReviewTemplate');
    }
    /**
     * Grants permission to an owner of a review template to share with other AWS accounts and IAM users
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_CreateTemplateShare.html
     */
    toCreateTemplateShare() {
        return this.to('CreateTemplateShare');
    }
    /**
     * Grants permission to create a new workload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifJiraProjectKey()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_CreateWorkload.html
     */
    toCreateWorkload() {
        return this.to('CreateWorkload');
    }
    /**
     * Grants permission to share a workload with another account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_CreateWorkloadShare.html
     */
    toCreateWorkloadShare() {
        return this.to('CreateWorkloadShare');
    }
    /**
     * Grants permission to delete a lens
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DeleteLens.html
     */
    toDeleteLens() {
        return this.to('DeleteLens');
    }
    /**
     * Grants permission to delete an existing lens share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DeleteLensShare.html
     */
    toDeleteLensShare() {
        return this.to('DeleteLensShare');
    }
    /**
     * Grants permission to delete a profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DeleteProfile.html
     */
    toDeleteProfile() {
        return this.to('DeleteProfile');
    }
    /**
     * Grants permission to delete an existing profile share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DeleteProfileShare.html
     */
    toDeleteProfileShare() {
        return this.to('DeleteProfileShare');
    }
    /**
     * Grants permission to delete an existing review template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DeleteReviewTemplate.html
     */
    toDeleteReviewTemplate() {
        return this.to('DeleteReviewTemplate');
    }
    /**
     * Grants permission to delete an existing review template share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DeleteTemplateShare.html
     */
    toDeleteTemplateShare() {
        return this.to('DeleteTemplateShare');
    }
    /**
     * Grants permission to delete an existing workload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DeleteWorkload.html
     */
    toDeleteWorkload() {
        return this.to('DeleteWorkload');
    }
    /**
     * Grants permission to delete an existing workload share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DeleteWorkloadShare.html
     */
    toDeleteWorkloadShare() {
        return this.to('DeleteWorkloadShare');
    }
    /**
     * Grants permission to disassociate a lens from the specified workload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DisassociateLenses.html
     */
    toDisassociateLenses() {
        return this.to('DisassociateLenses');
    }
    /**
     * Grants permission to disassociate a profile from the specified workload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DisassociateProfiles.html
     */
    toDisassociateProfiles() {
        return this.to('DisassociateProfiles');
    }
    /**
     * Grants permission to export an existing lens
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ExportLens.html
     */
    toExportLens() {
        return this.to('ExportLens');
    }
    /**
     * Grants permission to retrieve the specified answer from the specified lens review
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetAnswer.html
     */
    toGetAnswer() {
        return this.to('GetAnswer');
    }
    /**
     * Grants permission to get consolidated report metrics or to generate the consolidated report PDF in this account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetConsolidatedReport.html
     */
    toGetConsolidatedReport() {
        return this.to('GetConsolidatedReport');
    }
    /**
     * Grants permission to get all settings for the account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetGlobalSettings.html
     */
    toGetGlobalSettings() {
        return this.to('GetGlobalSettings');
    }
    /**
     * Grants permission to get an existing lens
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_DeleteLensShare.html
     */
    toGetLens() {
        return this.to('GetLens');
    }
    /**
     * Grants permission to retrieve the specified lens review of the specified workload
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetLensReview.html
     */
    toGetLensReview() {
        return this.to('GetLensReview');
    }
    /**
     * Grants permission to retrieve the report for the specified lens review
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetLensReviewReport.html
     */
    toGetLensReviewReport() {
        return this.to('GetLensReviewReport');
    }
    /**
     * Grants permission to get the difference between the specified lens version and latest available lens version
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetLensVersionDifference.html
     */
    toGetLensVersionDifference() {
        return this.to('GetLensVersionDifference');
    }
    /**
     * Grants permission to retrieve the specified milestone of the specified workload
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetMilestone.html
     */
    toGetMilestone() {
        return this.to('GetMilestone');
    }
    /**
     * Grants permission to retrieve the specified profile
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetProfile.html
     */
    toGetProfile() {
        return this.to('GetProfile');
    }
    /**
     * Grants permission to retrieve the specified profile template
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetProfileTemplate.html
     */
    toGetProfileTemplate() {
        return this.to('GetProfileTemplate');
    }
    /**
     * Grants permission to retrieve the specified review template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetReviewTemplate.html
     */
    toGetReviewTemplate() {
        return this.to('GetReviewTemplate');
    }
    /**
     * Grants permission to retrieve the specified answer from the specified review template lens review
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetReviewTemplateAnswer.html
     */
    toGetReviewTemplateAnswer() {
        return this.to('GetReviewTemplateAnswer');
    }
    /**
     * Grants permission to retrieve the specified lens review of the specified review template
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetReviewTemplateLensReview.html
     */
    toGetReviewTemplateLensReview() {
        return this.to('GetReviewTemplateLensReview');
    }
    /**
     * Grants permission to retrieve the specified workload
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_GetWorkload.html
     */
    toGetWorkload() {
        return this.to('GetWorkload');
    }
    /**
     * Grants permission to import a new lens
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ImportLens.html
     */
    toImportLens() {
        return this.to('ImportLens');
    }
    /**
     * Grants permission to list the answers from the specified lens review
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListAnswers.html
     */
    toListAnswers() {
        return this.to('ListAnswers');
    }
    /**
     * Grants permission to list the check-details for the workload
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListCheckDetails.html
     */
    toListCheckDetails() {
        return this.to('ListCheckDetails');
    }
    /**
     * Grants permission to list the check-summaries for the workload
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListCheckSummaries.html
     */
    toListCheckSummaries() {
        return this.to('ListCheckSummaries');
    }
    /**
     * Grants permission to list the improvements of the specified lens review
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListLensReviewImprovements.html
     */
    toListLensReviewImprovements() {
        return this.to('ListLensReviewImprovements');
    }
    /**
     * Grants permission to list the lens reviews of the specified workload
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListLensReviews.html
     */
    toListLensReviews() {
        return this.to('ListLensReviews');
    }
    /**
     * Grants permission to list all shares created for a lens
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListLensShares.html
     */
    toListLensShares() {
        return this.to('ListLensShares');
    }
    /**
     * Grants permission to list the lenses available to this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListLenses.html
     */
    toListLenses() {
        return this.to('ListLenses');
    }
    /**
     * Grants permission to list the milestones of the specified workload
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListMilestones.html
     */
    toListMilestones() {
        return this.to('ListMilestones');
    }
    /**
     * Grants permission to list notifications related to the account or specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListNotifications.html
     */
    toListNotifications() {
        return this.to('ListNotifications');
    }
    /**
     * Grants permission to list profile notifications related to specified resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListProfileNotifications.html
     */
    toListProfileNotifications() {
        return this.to('ListProfileNotifications');
    }
    /**
     * Grants permission to list all shares created for a profile
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListProfileShares.html
     */
    toListProfileShares() {
        return this.to('ListProfileShares');
    }
    /**
     * Grants permission to list the profiles available to this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListProfiles.html
     */
    toListProfiles() {
        return this.to('ListProfiles');
    }
    /**
     * Grants permission to list the answers from the specified review template lens review
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListReviewTemplateAnswers.html
     */
    toListReviewTemplateAnswers() {
        return this.to('ListReviewTemplateAnswers');
    }
    /**
     * Grants permission to list the review templates available to this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListReviewTemplates.html
     */
    toListReviewTemplates() {
        return this.to('ListReviewTemplates');
    }
    /**
     * Grants permission to list the workload share invitations of the specified account or user
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListShareInvitations.html
     */
    toListShareInvitations() {
        return this.to('ListShareInvitations');
    }
    /**
     * Grants permission to list tags for a Well-Architected resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to list all shares created for a review template
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListTemplateShares.html
     */
    toListTemplateShares() {
        return this.to('ListTemplateShares');
    }
    /**
     * Grants permission to list the workload shares of the specified workload
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListWorkloadShares.html
     */
    toListWorkloadShares() {
        return this.to('ListWorkloadShares');
    }
    /**
     * Grants permission to list the workloads in this account
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ListWorkloads.html
     */
    toListWorkloads() {
        return this.to('ListWorkloads');
    }
    /**
     * Grants permission to tag a Well-Architected resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to untag a Well-Architected resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update properties of the specified answer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateAnswer.html
     */
    toUpdateAnswer() {
        return this.to('UpdateAnswer');
    }
    /**
     * Grants permission to manage all settings for the account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifJiraProjectKey()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateGlobalSettings.html
     */
    toUpdateGlobalSettings() {
        return this.to('UpdateGlobalSettings');
    }
    /**
     * Grants permission to update properties of the integration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateIntegration.html
     */
    toUpdateIntegration() {
        return this.to('UpdateIntegration');
    }
    /**
     * Grants permission to update properties of the specified lens review
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateLensReview.html
     */
    toUpdateLensReview() {
        return this.to('UpdateLensReview');
    }
    /**
     * Grants permission to update properties of the specified profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateProfile.html
     */
    toUpdateProfile() {
        return this.to('UpdateProfile');
    }
    /**
     * Grants permission to update properties of the specified review template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateReviewTemplate.html
     */
    toUpdateReviewTemplate() {
        return this.to('UpdateReviewTemplate');
    }
    /**
     * Grants permission to update properties of the specified review template answer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateReviewTemplateAnswer.html
     */
    toUpdateReviewTemplateAnswer() {
        return this.to('UpdateReviewTemplateAnswer');
    }
    /**
     * Grants permission to update properties of the specified review template lens review
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateReviewTemplateLensReview.html
     */
    toUpdateReviewTemplateLensReview() {
        return this.to('UpdateReviewTemplateLensReview');
    }
    /**
     * Grants permission to update status of the specified workload share invitation
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateShareInvitation.html
     */
    toUpdateShareInvitation() {
        return this.to('UpdateShareInvitation');
    }
    /**
     * Grants permission to update properties of the specified workload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifJiraProjectKey()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateWorkload.html
     */
    toUpdateWorkload() {
        return this.to('UpdateWorkload');
    }
    /**
     * Grants permission to update properties of the specified workload share
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpdateWorkloadShare.html
     */
    toUpdateWorkloadShare() {
        return this.to('UpdateWorkloadShare');
    }
    /**
     * Grants permission to upgrade the specified lens review to use the latest version of the associated lens
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpgradeLensReview.html
     */
    toUpgradeLensReview() {
        return this.to('UpgradeLensReview');
    }
    /**
     * Grants permission to upgrade the specified workload to use the latest version of the associated profile
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpgradeProfileVersion.html
     */
    toUpgradeProfileVersion() {
        return this.to('UpgradeProfileVersion');
    }
    /**
     * Grants permission to upgrade the specified lens review of the specified review template
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_UpgradeReviewTemplateLensReview.html
     */
    toUpgradeReviewTemplateLensReview() {
        return this.to('UpgradeReviewTemplateLensReview');
    }
    /**
     * Adds a resource of type workload to the statement
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_Workload.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkload(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wellarchitected:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:workload/${resourceId}`);
    }
    /**
     * Adds a resource of type lens to the statement
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_Lens.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLens(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wellarchitected:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:lens/${resourceId}`);
    }
    /**
     * Adds a resource of type profile to the statement
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_Profile.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onProfile(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wellarchitected:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:profile/${resourceId}`);
    }
    /**
     * Adds a resource of type review-template to the statement
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/APIReference/API_ReviewTemplate.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onReviewTemplate(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:wellarchitected:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:review-template/${resourceId}`);
    }
    /**
     * Filters access by tag key-value pairs in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateProfile()
     * - .toCreateReviewTemplate()
     * - .toCreateWorkload()
     * - .toImportLens()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag key-value pairs attached to the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toAssociateLenses()
     * - .toAssociateProfiles()
     * - .toCreateLensShare()
     * - .toCreateLensVersion()
     * - .toCreateMilestone()
     * - .toCreateProfileShare()
     * - .toCreateTemplateShare()
     * - .toCreateWorkloadShare()
     * - .toDeleteLens()
     * - .toDeleteLensShare()
     * - .toDeleteProfile()
     * - .toDeleteProfileShare()
     * - .toDeleteReviewTemplate()
     * - .toDeleteTemplateShare()
     * - .toDeleteWorkload()
     * - .toDeleteWorkloadShare()
     * - .toDisassociateLenses()
     * - .toDisassociateProfiles()
     * - .toExportLens()
     * - .toGetAnswer()
     * - .toGetLens()
     * - .toGetLensReview()
     * - .toGetLensReviewReport()
     * - .toGetLensVersionDifference()
     * - .toGetMilestone()
     * - .toGetProfile()
     * - .toGetReviewTemplate()
     * - .toGetReviewTemplateAnswer()
     * - .toGetReviewTemplateLensReview()
     * - .toGetWorkload()
     * - .toListAnswers()
     * - .toListCheckDetails()
     * - .toListCheckSummaries()
     * - .toListLensReviewImprovements()
     * - .toListLensReviews()
     * - .toListLensShares()
     * - .toListMilestones()
     * - .toListProfileShares()
     * - .toListReviewTemplateAnswers()
     * - .toListTagsForResource()
     * - .toListTemplateShares()
     * - .toListWorkloadShares()
     * - .toUpdateAnswer()
     * - .toUpdateIntegration()
     * - .toUpdateLensReview()
     * - .toUpdateProfile()
     * - .toUpdateReviewTemplate()
     * - .toUpdateReviewTemplateAnswer()
     * - .toUpdateReviewTemplateLensReview()
     * - .toUpdateWorkload()
     * - .toUpdateWorkloadShare()
     * - .toUpgradeLensReview()
     * - .toUpgradeProfileVersion()
     * - .toUpgradeReviewTemplateLensReview()
     *
     * Applies to resource types:
     * - workload
     * - lens
     * - profile
     * - review-template
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by tag keys in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateProfile()
     * - .toCreateReviewTemplate()
     * - .toCreateWorkload()
     * - .toImportLens()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by project key
     *
     * https://docs.aws.amazon.com/wellarchitected/latest/userguide/security_iam_id-based-policy-examples.html
     *
     * Applies to actions:
     * - .toCreateWorkload()
     * - .toUpdateGlobalSettings()
     * - .toUpdateWorkload()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJiraProjectKey(value, operator) {
        return this.if(`JiraProjectKey`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [wellarchitected](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awswell-architectedtool.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'wellarchitected';
        this.accessLevelList = {
            Write: [
                'AssociateLenses',
                'AssociateProfiles',
                'ConfigureIntegration',
                'CreateLensShare',
                'CreateLensVersion',
                'CreateMilestone',
                'CreateProfile',
                'CreateProfileShare',
                'CreateReviewTemplate',
                'CreateTemplateShare',
                'CreateWorkload',
                'CreateWorkloadShare',
                'DeleteLens',
                'DeleteLensShare',
                'DeleteProfile',
                'DeleteProfileShare',
                'DeleteReviewTemplate',
                'DeleteTemplateShare',
                'DeleteWorkload',
                'DeleteWorkloadShare',
                'DisassociateLenses',
                'DisassociateProfiles',
                'ImportLens',
                'UpdateAnswer',
                'UpdateGlobalSettings',
                'UpdateIntegration',
                'UpdateLensReview',
                'UpdateProfile',
                'UpdateReviewTemplate',
                'UpdateReviewTemplateAnswer',
                'UpdateReviewTemplateLensReview',
                'UpdateShareInvitation',
                'UpdateWorkload',
                'UpdateWorkloadShare',
                'UpgradeLensReview',
                'UpgradeProfileVersion',
                'UpgradeReviewTemplateLensReview'
            ],
            Read: [
                'ExportLens',
                'GetAnswer',
                'GetConsolidatedReport',
                'GetGlobalSettings',
                'GetLens',
                'GetLensReview',
                'GetLensReviewReport',
                'GetLensVersionDifference',
                'GetMilestone',
                'GetProfile',
                'GetProfileTemplate',
                'GetReviewTemplate',
                'GetReviewTemplateAnswer',
                'GetReviewTemplateLensReview',
                'GetWorkload',
                'ListTagsForResource'
            ],
            List: [
                'ListAnswers',
                'ListCheckDetails',
                'ListCheckSummaries',
                'ListLensReviewImprovements',
                'ListLensReviews',
                'ListLensShares',
                'ListLenses',
                'ListMilestones',
                'ListNotifications',
                'ListProfileNotifications',
                'ListProfileShares',
                'ListProfiles',
                'ListReviewTemplateAnswers',
                'ListReviewTemplates',
                'ListShareInvitations',
                'ListTemplateShares',
                'ListWorkloadShares',
                'ListWorkloads'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Wellarchitected = Wellarchitected;
//# sourceMappingURL=data:application/json;base64,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