"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workdocs = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [workdocs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkdocs.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Workdocs extends shared_1.PolicyStatement {
    /**
     * Grants permission to abort the upload of the specified document version that was previously initiated by InitiateDocumentVersionUpload
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_AbortDocumentVersionUpload.html
     */
    toAbortDocumentVersionUpload() {
        return this.to('AbortDocumentVersionUpload');
    }
    /**
     * Grants permission to activate the specified user. Only active users can access Amazon WorkDocs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_ActivateUser.html
     */
    toActivateUser() {
        return this.to('ActivateUser');
    }
    /**
     * Grants permission to add principals that are allowed to call notification subscription APIs for a given WorkDocs site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/manage-notifications.html
     */
    toAddNotificationPermissions() {
        return this.to('AddNotificationPermissions');
    }
    /**
     * Grants permission to create a set of permissions for the specified folder or document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_AddResourcePermissions.html
     */
    toAddResourcePermissions() {
        return this.to('AddResourcePermissions');
    }
    /**
     * Grants permission to add a user to a group
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/manage_set_admin.html
     */
    toAddUserToGroup() {
        return this.to('AddUserToGroup');
    }
    /**
     * Grants permission to check an alias
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/cloud_quick_start.html
     */
    toCheckAlias() {
        return this.to('CheckAlias');
    }
    /**
     * Grants permission to add a new comment to the specified document version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_CreateComment.html
     */
    toCreateComment() {
        return this.to('CreateComment');
    }
    /**
     * Grants permission to add one or more custom properties to the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_CreateCustomMetadata.html
     */
    toCreateCustomMetadata() {
        return this.to('CreateCustomMetadata');
    }
    /**
     * Grants permission to create a folder with the specified name and parent folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_CreateFolder.html
     */
    toCreateFolder() {
        return this.to('CreateFolder');
    }
    /**
     * Grants permission to create an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/getting_started.html
     */
    toCreateInstance() {
        return this.to('CreateInstance');
    }
    /**
     * Grants permission to add labels to the given resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_CreateLabels.html
     */
    toCreateLabels() {
        return this.to('CreateLabels');
    }
    /**
     * Grants permission to configure WorkDocs to use Amazon SNS notifications
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_CreateNotificationSubscription.html
     */
    toCreateNotificationSubscription() {
        return this.to('CreateNotificationSubscription');
    }
    /**
     * Grants permission to create a user in a Simple AD or Microsoft AD directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_CreateUser.html
     */
    toCreateUser() {
        return this.to('CreateUser');
    }
    /**
     * Grants permission to deactivate the specified user, which revokes the user's access to Amazon WorkDocs
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeactivateUser.html
     */
    toDeactivateUser() {
        return this.to('DeactivateUser');
    }
    /**
     * Grants permission to delete the specified comment from the document version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeleteComment.html
     */
    toDeleteComment() {
        return this.to('DeleteComment');
    }
    /**
     * Grants permission to delete custom metadata from the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeleteCustomMetadata.html
     */
    toDeleteCustomMetadata() {
        return this.to('DeleteCustomMetadata');
    }
    /**
     * Grants permission to permanently delete the specified document and its associated metadata
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeleteDocument.html
     */
    toDeleteDocument() {
        return this.to('DeleteDocument');
    }
    /**
     * Grants permission to delete versions of a specified document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeleteDocumentVersion.html
     */
    toDeleteDocumentVersion() {
        return this.to('DeleteDocumentVersion');
    }
    /**
     * Grants permission to permanently delete the specified folder and its contents
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeleteFolder.html
     */
    toDeleteFolder() {
        return this.to('DeleteFolder');
    }
    /**
     * Grants permission to delete the contents of the specified folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeleteFolderContents.html
     */
    toDeleteFolderContents() {
        return this.to('DeleteFolderContents');
    }
    /**
     * Grants permission to delete an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/manage-sites.html#delete_site
     */
    toDeleteInstance() {
        return this.to('DeleteInstance');
    }
    /**
     * Grants permission to delete one or more labels from a resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeleteLabels.html
     */
    toDeleteLabels() {
        return this.to('DeleteLabels');
    }
    /**
     * Grants permission to delete principals that are allowed to call notification subscription APIs for a given WorkDocs site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/manage-notifications.html
     */
    toDeleteNotificationPermissions() {
        return this.to('DeleteNotificationPermissions');
    }
    /**
     * Grants permission to delete the specified subscription from the specified organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeleteNotificationSubscription.html
     */
    toDeleteNotificationSubscription() {
        return this.to('DeleteNotificationSubscription');
    }
    /**
     * Grants permission to delete the specified user from a Simple AD or Microsoft AD directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser() {
        return this.to('DeleteUser');
    }
    /**
     * Grants permission to deregister a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/manage-sites.html#delete_site
     */
    toDeregisterDirectory() {
        return this.to('DeregisterDirectory');
    }
    /**
     * Grants permission to fetch user activities in a specified time period
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DescribeActivities.html
     */
    toDescribeActivities() {
        return this.to('DescribeActivities');
    }
    /**
     * Grants permission to describe available directories
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/getting_started.html
     */
    toDescribeAvailableDirectories() {
        return this.to('DescribeAvailableDirectories');
    }
    /**
     * Grants permission to list all the comments for the specified document version
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DescribeComments.html
     */
    toDescribeComments() {
        return this.to('DescribeComments');
    }
    /**
     * Grants permission to retrieve the document versions for the specified document
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DescribeDocumentVersions.html
     */
    toDescribeDocumentVersions() {
        return this.to('DescribeDocumentVersions');
    }
    /**
     * Grants permission to describe the contents of the specified folder, including its documents and sub-folders
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DescribeFolderContents.html
     */
    toDescribeFolderContents() {
        return this.to('DescribeFolderContents');
    }
    /**
     * Grants permission to describe the user groups
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DescribeGroups.html
     */
    toDescribeGroups() {
        return this.to('DescribeGroups');
    }
    /**
     * Grants permission to describe the export history for an instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/migration-tool.html
     */
    toDescribeInstanceExports() {
        return this.to('DescribeInstanceExports');
    }
    /**
     * Grants permission to describe instances
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/getting_started.html
     */
    toDescribeInstances() {
        return this.to('DescribeInstances');
    }
    /**
     * Grants permission to describe principals that are allowed to call notification subscription APIs for a given WorkDocs site
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/manage-notifications.html
     */
    toDescribeNotificationPermissions() {
        return this.to('DescribeNotificationPermissions');
    }
    /**
     * Grants permission to list the specified notification subscriptions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DescribeNotificationSubscriptions.html
     */
    toDescribeNotificationSubscriptions() {
        return this.to('DescribeNotificationSubscriptions');
    }
    /**
     * Grants permission to view a description of a specified resource's permissions
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DescribeResourcePermissions.html
     */
    toDescribeResourcePermissions() {
        return this.to('DescribeResourcePermissions');
    }
    /**
     * Grants permission to describe the root folders
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DescribeRootFolders.html
     */
    toDescribeRootFolders() {
        return this.to('DescribeRootFolders');
    }
    /**
     * Grants permission to view a description of the specified users. You can describe all users or filter the results (for example, by status or organization)
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_DescribeUsers.html
     */
    toDescribeUsers() {
        return this.to('DescribeUsers');
    }
    /**
     * Grants permission to download a specified document version
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_GetDocumentVersion.html
     */
    toDownloadDocumentVersion() {
        return this.to('DownloadDocumentVersion');
    }
    /**
     * Grants permission to retrieve the details of the current user
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_GetCurrentUser.html
     */
    toGetCurrentUser() {
        return this.to('GetCurrentUser');
    }
    /**
     * Grants permission to retrieve the specified document object
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_GetDocument.html
     */
    toGetDocument() {
        return this.to('GetDocument');
    }
    /**
     * Grants permission to retrieve the path information (the hierarchy from the root folder) for the requested document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_GetDocumentPath.html
     */
    toGetDocumentPath() {
        return this.to('GetDocumentPath');
    }
    /**
     * Grants permission to retrieve version metadata for the specified document
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_GetDocumentVersion.html
     */
    toGetDocumentVersion() {
        return this.to('GetDocumentVersion');
    }
    /**
     * Grants permission to retrieve the metadata of the specified folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_GetFolder.html
     */
    toGetFolder() {
        return this.to('GetFolder');
    }
    /**
     * Grants permission to retrieve the path information (the hierarchy from the root folder) for the specified folder
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_GetFolderPath.html
     */
    toGetFolderPath() {
        return this.to('GetFolderPath');
    }
    /**
     * Grants permission to retrieve details for the specified group
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_Operations.html
     */
    toGetGroup() {
        return this.to('GetGroup');
    }
    /**
     * Grants permission to get a collection of resources
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_GetResources.html
     */
    toGetResources() {
        return this.to('GetResources');
    }
    /**
     * Grants permission to create a new document object and version object
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_InitiateDocumentVersionUpload.html
     */
    toInitiateDocumentVersionUpload() {
        return this.to('InitiateDocumentVersionUpload');
    }
    /**
     * Grants permission to register a directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/existing-dir-setup.html
     */
    toRegisterDirectory() {
        return this.to('RegisterDirectory');
    }
    /**
     * Grants permission to remove all the permissions from the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_RemoveAllResourcePermissions.html
     */
    toRemoveAllResourcePermissions() {
        return this.to('RemoveAllResourcePermissions');
    }
    /**
     * Grants permission to remove the permission for the specified principal from the specified resource
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_RemoveResourcePermission.html
     */
    toRemoveResourcePermission() {
        return this.to('RemoveResourcePermission');
    }
    /**
     * Grants permission to restore versions of a specified document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_RestoreDocumentVersions.html
     */
    toRestoreDocumentVersions() {
        return this.to('RestoreDocumentVersions');
    }
    /**
     * Grants permission to search metadata and the content of resources
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_SearchResources.html
     */
    toSearchResources() {
        return this.to('SearchResources');
    }
    /**
     * Grants permission to start an export for an instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/migration-tool.html
     */
    toStartInstanceExport() {
        return this.to('StartInstanceExport');
    }
    /**
     * Grants permission to update the specified attributes of the specified document
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_UpdateDocument.html
     */
    toUpdateDocument() {
        return this.to('UpdateDocument');
    }
    /**
     * Grants permission to change the status of the document version to ACTIVE
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_UpdateDocumentVersion.html
     */
    toUpdateDocumentVersion() {
        return this.to('UpdateDocumentVersion');
    }
    /**
     * Grants permission to update the specified attributes of the specified folder
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_UpdateFolder.html
     */
    toUpdateFolder() {
        return this.to('UpdateFolder');
    }
    /**
     * Grants permission to update an instance alias
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/getting_started.html
     */
    toUpdateInstanceAlias() {
        return this.to('UpdateInstanceAlias');
    }
    /**
     * Grants permission to update the specified attributes of the specified user, and grants or revokes administrative privileges to the Amazon WorkDocs site
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/APIReference/API_UpdateUser.html
     */
    toUpdateUser() {
        return this.to('UpdateUser');
    }
    /**
     * Grants permission to update the administrative settings for a user
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/migration.html
     */
    toUpdateUserAdministrativeSettings() {
        return this.to('UpdateUserAdministrativeSettings');
    }
    /**
     * Adds a resource of type organization to the statement
     *
     * https://docs.aws.amazon.com/workdocs/latest/adminguide/migration-tool.html
     *
     * @param resourceId - Identifier for the resourceId.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     */
    onOrganization(resourceId, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:workdocs:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:organization/${resourceId}`);
    }
    /**
     * Statement provider for service [workdocs](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonworkdocs.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'workdocs';
        this.accessLevelList = {
            Write: [
                'AbortDocumentVersionUpload',
                'ActivateUser',
                'AddNotificationPermissions',
                'AddResourcePermissions',
                'AddUserToGroup',
                'CreateComment',
                'CreateCustomMetadata',
                'CreateFolder',
                'CreateInstance',
                'CreateLabels',
                'CreateNotificationSubscription',
                'CreateUser',
                'DeactivateUser',
                'DeleteComment',
                'DeleteCustomMetadata',
                'DeleteDocument',
                'DeleteDocumentVersion',
                'DeleteFolder',
                'DeleteFolderContents',
                'DeleteInstance',
                'DeleteLabels',
                'DeleteNotificationPermissions',
                'DeleteNotificationSubscription',
                'DeleteUser',
                'DeregisterDirectory',
                'InitiateDocumentVersionUpload',
                'RegisterDirectory',
                'RemoveAllResourcePermissions',
                'RemoveResourcePermission',
                'RestoreDocumentVersions',
                'StartInstanceExport',
                'UpdateDocument',
                'UpdateDocumentVersion',
                'UpdateFolder',
                'UpdateInstanceAlias',
                'UpdateUser',
                'UpdateUserAdministrativeSettings'
            ],
            Read: [
                'CheckAlias',
                'DownloadDocumentVersion',
                'GetCurrentUser',
                'GetDocument',
                'GetDocumentPath',
                'GetDocumentVersion',
                'GetFolder',
                'GetFolderPath',
                'GetGroup',
                'GetResources'
            ],
            List: [
                'DescribeActivities',
                'DescribeAvailableDirectories',
                'DescribeComments',
                'DescribeDocumentVersions',
                'DescribeFolderContents',
                'DescribeGroups',
                'DescribeInstanceExports',
                'DescribeInstances',
                'DescribeNotificationPermissions',
                'DescribeNotificationSubscriptions',
                'DescribeResourcePermissions',
                'DescribeRootFolders',
                'DescribeUsers',
                'SearchResources'
            ]
        };
    }
}
exports.Workdocs = Workdocs;
//# sourceMappingURL=data:application/json;base64,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