"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Xray = void 0;
const shared_1 = require("../../shared");
/**
 * Statement provider for service [xray](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsx-ray.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Xray extends shared_1.PolicyStatement {
    /**
     * Grants permission to retrieve metadata for a list of traces specified by ID
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-console
     */
    toBatchGetTraceSummaryById() {
        return this.to('BatchGetTraceSummaryById');
    }
    /**
     * Grants permission to retrieve a list of traces specified by ID. Each trace is a collection of segment documents that originates from a single request. Use GetTraceSummaries to get a list of trace IDs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_BatchGetTraces.html
     */
    toBatchGetTraces() {
        return this.to('BatchGetTraces');
    }
    /**
     * Grants permission to cancel an ongoing trace retrieval job initiated by StartTraceRetrieval using the provided RetrievalToken. A successful cancellation will return an HTTP 200 response
     *
     * Access Level: Read
     */
    toCancelTraceRetrieval() {
        return this.to('CancelTraceRetrieval');
    }
    /**
     * Grants permission to create a group resource with a name and a filter expression
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateGroup.html
     */
    toCreateGroup() {
        return this.to('CreateGroup');
    }
    /**
     * Grants permission to create a rule to control sampling behavior for instrumented applications
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_CreateSamplingRule.html
     */
    toCreateSamplingRule() {
        return this.to('CreateSamplingRule');
    }
    /**
     * Grants permission to delete a group resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteGroup.html
     */
    toDeleteGroup() {
        return this.to('DeleteGroup');
    }
    /**
     * Grants permission to delete resource policies
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteResourcePolicy.html
     */
    toDeleteResourcePolicy() {
        return this.to('DeleteResourcePolicy');
    }
    /**
     * Grants permission to delete a sampling rule
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_DeleteSamplingRule.html
     */
    toDeleteSamplingRule() {
        return this.to('DeleteSamplingRule');
    }
    /**
     * Grants permission to retrieve distinct service graphs for one or more specific trace IDs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-console
     */
    toGetDistinctTraceGraphs() {
        return this.to('GetDistinctTraceGraphs');
    }
    /**
     * Grants permission to retrieve the current encryption configuration for X-Ray data
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetEncryptionConfig.html
     */
    toGetEncryptionConfig() {
        return this.to('GetEncryptionConfig');
    }
    /**
     * Grants permission to retrieve group resource details
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroup.html
     */
    toGetGroup() {
        return this.to('GetGroup');
    }
    /**
     * Grants permission to retrieve all active group details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetGroups.html
     */
    toGetGroups() {
        return this.to('GetGroups');
    }
    /**
     * Grants permission to retrieve all indexing rules. Indexing rules are used to determine the server-side sampling rate for spans ingested through the CloudWatchLogs destination and indexed by X-Ray
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetIndexingRules.html
     */
    toGetIndexingRules() {
        return this.to('GetIndexingRules');
    }
    /**
     * Grants permission to retrieve the details of a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsight.html
     */
    toGetInsight() {
        return this.to('GetInsight');
    }
    /**
     * Grants permission to retrieve the events of a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightEvents.html
     */
    toGetInsightEvents() {
        return this.to('GetInsightEvents');
    }
    /**
     * Grants permission to retrieve the part of the service graph which is impacted for a specific insight
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightImpactGraph.html
     */
    toGetInsightImpactGraph() {
        return this.to('GetInsightImpactGraph');
    }
    /**
     * Grants permission to retrieve the summary of all insights for a group and time range with optional filters
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetInsightSummaries.html
     */
    toGetInsightSummaries() {
        return this.to('GetInsightSummaries');
    }
    /**
     * Grants permission to retrieve a service graph for traces based on the specified RetrievalToken from the Transaction Search CloudWatch log group
     *
     * Access Level: Read
     */
    toGetRetrievedTracesGraph() {
        return this.to('GetRetrievedTracesGraph');
    }
    /**
     * Grants permission to retrieve all sampling rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingRules.html
     */
    toGetSamplingRules() {
        return this.to('GetSamplingRules');
    }
    /**
     * Grants permission to retrieve information about recent sampling results for all sampling rules
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingStatisticSummaries.html
     */
    toGetSamplingStatisticSummaries() {
        return this.to('GetSamplingStatisticSummaries');
    }
    /**
     * Grants permission to request a sampling quota for rules that the service is using to sample requests
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetSamplingTargets.html
     */
    toGetSamplingTargets() {
        return this.to('GetSamplingTargets');
    }
    /**
     * Grants permission to retrieve a document that describes services that process incoming requests, and downstream services that they call as a result
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetServiceGraph.html
     */
    toGetServiceGraph() {
        return this.to('GetServiceGraph');
    }
    /**
     * Grants permission to retrieve an aggregation of service statistics defined by a specific time range bucketed into time intervals
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTimeSeriesServiceStatistics.html
     */
    toGetTimeSeriesServiceStatistics() {
        return this.to('GetTimeSeriesServiceStatistics');
    }
    /**
     * Grants permission to retrieve a service graph for one or more specific trace IDs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceGraph.html
     */
    toGetTraceGraph() {
        return this.to('GetTraceGraph');
    }
    /**
     * Grants permission to retrieve the current destination of data sent to PutTraceSegments and OpenTelemetry API
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceSegmentDestination.html
     */
    toGetTraceSegmentDestination() {
        return this.to('GetTraceSegmentDestination');
    }
    /**
     * Grants permission to retrieve IDs and metadata for traces available for a specified time frame using an optional filter. To get the full traces, pass the trace IDs to BatchGetTraces
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_GetTraceSummaries.html
     */
    toGetTraceSummaries() {
        return this.to('GetTraceSummaries');
    }
    /**
     * Grants permission to share X-Ray resources with a monitoring account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch-Unified-Cross-Account-Setup.html#CloudWatch-Unified-Cross-Account-Setup-permissions
     */
    toLink() {
        return this.to('Link');
    }
    /**
     * Grants permission to list resource policies
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_ListResourcePolicies.html
     */
    toListResourcePolicies() {
        return this.to('ListResourcePolicies');
    }
    /**
     * Grants permission to retrieve a list of traces for a given RetrievalToken from the Transaction Search CloudWatch log group
     *
     * Access Level: List
     */
    toListRetrievedTraces() {
        return this.to('ListRetrievedTraces');
    }
    /**
     * Grants permission to list tags for an X-Ray resource
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        return this.to('ListTagsForResource');
    }
    /**
     * Grants permission to update the encryption configuration for X-Ray data
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutEncryptionConfig.html
     */
    toPutEncryptionConfig() {
        return this.to('PutEncryptionConfig');
    }
    /**
     * Grants permission to create or update resource policies
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutResourcePolicy.html
     */
    toPutResourcePolicy() {
        return this.to('PutResourcePolicy');
    }
    /**
     * Grants permission to upload OpenTelemetry spans to AWS X-Ray
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutSpans.html
     */
    toPutSpans() {
        return this.to('PutSpans');
    }
    /**
     * Grants permission to upload spans to AWS X-Ray to be indexed
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-console
     */
    toPutSpansForIndexing() {
        return this.to('PutSpansForIndexing');
    }
    /**
     * Grants permission to send AWS X-Ray daemon telemetry to the service
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTelemetryRecords.html
     */
    toPutTelemetryRecords() {
        return this.to('PutTelemetryRecords');
    }
    /**
     * Grants permission to upload segment documents to AWS X-Ray. The X-Ray SDK generates segment documents and sends them to the X-Ray daemon, which uploads them in batches
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_PutTraceSegments.html
     */
    toPutTraceSegments() {
        return this.to('PutTraceSegments');
    }
    /**
     * Grants permission to initiate a trace retrieval process using the specified time range and for the given trace IDs on the Transaction Search CloudWatch log group
     *
     * Access Level: Read
     */
    toStartTraceRetrieval() {
        return this.to('StartTraceRetrieval');
    }
    /**
     * Grants permission to add tags to an X-Ray resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_TagResource.html
     */
    toTagResource() {
        return this.to('TagResource');
    }
    /**
     * Grants permission to remove tags from an X-Ray resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        return this.to('UntagResource');
    }
    /**
     * Grants permission to update a group resource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateGroup.html
     */
    toUpdateGroup() {
        return this.to('UpdateGroup');
    }
    /**
     * Grants permission to modify an indexing rule's configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateIndexingRule.html
     */
    toUpdateIndexingRule() {
        return this.to('UpdateIndexingRule');
    }
    /**
     * Grants permission to modify a sampling rule's configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateSamplingRule.html
     */
    toUpdateSamplingRule() {
        return this.to('UpdateSamplingRule');
    }
    /**
     * Grants permission to modify the destination of data sent to PutTraceSegments and OpenTelemetry API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/xray/latest/api/API_UpdateTraceSegmentDestination.html
     */
    toUpdateTraceSegmentDestination() {
        return this.to('UpdateTraceSegmentDestination');
    }
    /**
     * Adds a resource of type group to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-groups
     *
     * @param groupName - Identifier for the groupName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGroup(groupName, id, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:xray:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:group/${groupName}/${id}`);
    }
    /**
     * Adds a resource of type sampling-rule to the statement
     *
     * https://docs.aws.amazon.com/xray/latest/devguide/xray-concepts.html#xray-concepts-sampling
     *
     * @param samplingRuleName - Identifier for the samplingRuleName.
     * @param account - Account of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's account.
     * @param region - Region of the resource; defaults to `*`, unless using the CDK, where the default is the current Stack's region.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`, unless using the CDK, where the default is the current Stack's partition.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSamplingRule(samplingRuleName, account, region, partition) {
        return this.on(`arn:${partition ?? this.defaultPartition}:xray:${region ?? this.defaultRegion}:${account ?? this.defaultAccount}:sampling-rule/${samplingRuleName}`);
    }
    /**
     * Filters access by the tags that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * Applies to actions:
     * - .toCreateGroup()
     * - .toCreateSamplingRule()
     * - .toTagResource()
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(tagKey, value, operator) {
        return this.if(`aws:RequestTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tags associated with the resource
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * Applies to actions:
     * - .toDeleteGroup()
     * - .toDeleteSamplingRule()
     * - .toGetGroup()
     * - .toUpdateGroup()
     * - .toUpdateSamplingRule()
     *
     * Applies to resource types:
     * - group
     * - sampling-rule
     *
     * @param tagKey The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(tagKey, value, operator) {
        return this.if(`aws:ResourceTag/${tagKey}`, value, operator ?? 'StringLike');
    }
    /**
     * Filters access by the tag keys that are passed in the request
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * Applies to actions:
     * - .toCreateGroup()
     * - .toCreateSamplingRule()
     * - .toTagResource()
     * - .toUntagResource()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if(`aws:TagKeys`, value, operator ?? 'StringLike');
    }
    /**
     * Statement provider for service [xray](https://docs.aws.amazon.com/service-authorization/latest/reference/list_awsx-ray.html).
     *
     */
    constructor(props) {
        super(props);
        this.servicePrefix = 'xray';
        this.accessLevelList = {
            Read: [
                'BatchGetTraceSummaryById',
                'CancelTraceRetrieval',
                'GetDistinctTraceGraphs',
                'GetEncryptionConfig',
                'GetGroup',
                'GetGroups',
                'GetIndexingRules',
                'GetInsight',
                'GetInsightEvents',
                'GetInsightImpactGraph',
                'GetInsightSummaries',
                'GetRetrievedTracesGraph',
                'GetSamplingRules',
                'GetSamplingStatisticSummaries',
                'GetSamplingTargets',
                'GetServiceGraph',
                'GetTimeSeriesServiceStatistics',
                'GetTraceGraph',
                'GetTraceSegmentDestination',
                'GetTraceSummaries',
                'StartTraceRetrieval'
            ],
            List: [
                'BatchGetTraces',
                'ListResourcePolicies',
                'ListRetrievedTraces',
                'ListTagsForResource'
            ],
            Write: [
                'CreateGroup',
                'CreateSamplingRule',
                'DeleteGroup',
                'DeleteResourcePolicy',
                'DeleteSamplingRule',
                'Link',
                'PutResourcePolicy',
                'PutSpans',
                'PutSpansForIndexing',
                'PutTelemetryRecords',
                'PutTraceSegments',
                'UpdateGroup',
                'UpdateIndexingRule',
                'UpdateSamplingRule',
                'UpdateTraceSegmentDestination'
            ],
            'Permissions management': [
                'PutEncryptionConfig'
            ],
            Tagging: [
                'TagResource',
                'UntagResource'
            ]
        };
    }
}
exports.Xray = Xray;
//# sourceMappingURL=data:application/json;base64,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