"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithCondition = void 0;
const operators_1 = require("../operators");
const _1_base_1 = require("./1-base");
/**
 * Adds "condition" functionality to the Policy Statement
 */
class PolicyStatementWithCondition extends _1_base_1.PolicyStatementBase {
    constructor() {
        super(...arguments);
        this.floydConditions = {};
        this.cdkConditionsApplied = false;
    }
    /**
     * Injects conditions into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyConditions();
            return super.toJSON();
        }
        const statement = super.toJSON();
        if (this.hasConditions()) {
            statement.Condition = this.floydConditions;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyConditions();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    freeze() {
        // @ts-ignore only available after swapping 1-base
        if (!this.frozen) {
            this.cdkApplyConditions();
        }
        // @ts-ignore only available after swapping 1-base
        return super.freeze();
    }
    cdkApplyConditions() {
        if (this.hasConditions() && !this.cdkConditionsApplied) {
            Object.keys(this.floydConditions).forEach((operator) => {
                Object.keys(this.floydConditions[operator]).forEach((key) => {
                    const condition = {};
                    condition[key] = this.floydConditions[operator][key];
                    // @ts-ignore only available after swapping 1-base
                    this.addCondition(operator, condition);
                });
            });
            this.cdkConditionsApplied = true;
        }
    }
    /**
     * Checks weather a condition was applied to the policy.
     */
    hasConditions() {
        return Object.keys(this.floydConditions).length > 0;
    }
    /**
     * Adds a condition to the statement
     *
     * @param key The key of the condition
     * @param value The value(s) to check for
     * @param operator [Operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html) of the condition. **Default:** `StringLike`
     */
    if(key, value, operator) {
        if (this.servicePrefix.length && key.indexOf(':') < 0) {
            key = `${this.servicePrefix}:${key}`;
        }
        if (typeof operator === 'undefined') {
            operator = operators_1.Operator.stringLike;
        }
        let op = '';
        if (typeof operator === 'string') {
            op = operator;
        }
        else {
            op = operator.toString();
        }
        // For boolean/number operators, IAM accepts both, booleans/numbers and
        // their string representation. To be consistent with how the IAM console
        // displays/generates the values, we convert them to strings.
        if (['boolean', 'number'].includes(typeof value)) {
            value = value.toString();
        }
        if (!(op in this.floydConditions)) {
            this.floydConditions[op] = {};
        }
        this.floydConditions[op][key] = value;
        return this;
    }
    /**
     * Compare the services with the services that made requests on behalf of the IAM principal (user or role). When a principal makes a request to an AWS service, that service might use the principal's credentials to make subsequent requests to other services.
     *
     * The `aws:CalledVia` key contains an ordered list of each service in the chain that made requests on the principal's behalf.
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringEquals`
     */
    ifAwsCalledVia(value, operator) {
        return this.if('aws:CalledVia', value, operator ?? new operators_1.Operator().forAnyValue().stringEquals());
    }
    /**
     * Compare the services with the first service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledviafirst
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaFirst(value, operator) {
        return this.if('aws:CalledViaFirst', value, operator);
    }
    /**
     * Compare the services with the last service that made a request on behalf of the IAM principal (user or role). For more information, see [aws:CalledVia](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvia).
     *
     * **Availability:** This key is present in the request when a service that supports aws:CalledVia uses the credentials of an IAM principal to make a request to another service. This key is not present if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. This key is also not present when the principal makes the call directly.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-calledvialast
     *
     * @param value The service(s) to check for
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsCalledViaLast(value, operator) {
        return this.if('aws:CalledViaLast', value, operator);
    }
    /**
     * Compare the date and time of the request with the date and time that you specify. To view an example policy that uses this condition key, see [AWS: Allows Access Within Specific Dates](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws-dates.html).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-currenttime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateLessThanEquals`
     */
    ifAwsCurrentTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:CurrentTime', value, operator ?? operators_1.Operator.dateLessThanEquals);
    }
    /**
     * This key identifies the VPC to which Amazon EC2 IAM role credentials were delivered to. You can use this key in a policy with the [aws:SourceVPC](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpc) global key to check if a call is made from a VPC (`aws:SourceVPC`) that matches the VPC where a credential was delivered to (`aws:Ec2InstanceSourceVpc`).
     *
     * **Availability:** This key is included in the request context whenever the requester is signing requests with an Amazon EC2 role credential. It can be used in IAM policies, service control policies, VPC endpoint policies, and resource policies.
     *
     * This key can be used with VPC identifier values, but is most useful when used as a variable combined with the `aws:SourceVpc` context key. The `aws:SourceVpc` context key is included in the request context only if the requester uses a VPC endpoint to make the request. Using `aws:Ec2InstanceSourceVpc` with `aws:SourceVpc` allows you to use `aws:Ec2InstanceSourceVpc` more broadly since it compares values that typically change together.
     *
     * **Note:** This condition key is not available in EC2-Classic.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-ec2instancesourcevpc
     *
     * @param value The VPS ID
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsEc2InstanceSourceVpc(value, operator) {
        return this.if('aws:Ec2InstanceSourceVpc', value, operator);
    }
    /**
     * This key identifies the private IPv4 address of the primary elastic network interface to which Amazon EC2 IAM role credentials were delivered. You must use this condition key with its companion key `aws:Ec2InstanceSourceVpc` to ensure that you have a globally unique combination of VPC ID and source private IP. Use this key with `aws:Ec2InstanceSourceVpc` to ensure that a request was made from the same private IP address that the credentials were delivered to.
     *
     * **Availability:** This key is included in the request context whenever the requester is signing requests with an Amazon EC2 role credential. It can be used in IAM policies, service control policies, VPC endpoint policies, and resource policies.
     *
     * **Note:** This condition key is not available in EC2-Classic.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-vpcsourceip
     *
     * @param value The private IPv4 address
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsEc2InstanceSourcePrivateIPv4(value, operator) {
        return this.if('aws:Ec2InstanceSourcePrivateIPv4', value, operator ?? operators_1.Operator.ipAddress);
    }
    /**
     * Compare the date and time of the request in epoch or Unix time with the value that you specify. This key also accepts the number of seconds since January 1, 1970.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-epochtime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date) and [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `DateLessThanEquals`
     */
    ifAwsEpochTime(value, operator) {
        if (typeof value.getMonth === 'function') {
            value = value.toISOString();
        }
        else if (Array.isArray(value)) {
            value = value.map((item) => {
                if (typeof item.getMonth === 'function') {
                    item = item.toISOString();
                }
                return item;
            });
        }
        return this.if('aws:EpochTime', value, operator ?? operators_1.Operator.dateLessThanEquals);
    }
    /**
     * Use this key to compare the principal's issuing identity provider (IdP) with the IdP that you specify in the policy. This means that an IAM role was assumed using the `AssumeRoleWithWebIdentity` or `AssumeRoleWithSAML` AWS STS operations. When the resulting role session's temporary credentials are used to make a request, the request context identifies the IdP that authenticated the original federated identity.
     *
     * **Availability:** This key is present when the principal is a role session principal and that session was issued using a third-party identity provider.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-federatedprovider
     *
     * @param value The principal's issuing identity provider (IdP)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsFederatedProvider(value, operator) {
        return this.if('aws:FederatedProvider', value, operator);
    }
    /**
     * Compare the number of seconds since the requesting principal was authorized using MFA with the number that you specify. For more information about MFA, see [Using Multi-Factor Authentication (MFA) in AWS](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_mfa.html).
     *
     * **Availability:** This key is included in the request context only if the principal was authenticated using MFA. If MFA was not used, this key is not present.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthage
     *
     * @param value Number of seconds
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericLessThan`
     */
    ifAwsMultiFactorAuthAge(value, operator) {
        return this.if('aws:MultiFactorAuthAge', value, operator ?? operators_1.Operator.numericLessThan);
    }
    /**
     * Check whether multi-factor authentication (MFA) was used to validate the temporary security credentials that made the request.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. The key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using long-term credentials.
     *
     * Temporary credentials are used to authenticate IAM roles, federated users, IAM users with temporary tokens from `sts:GetSessionToken`, and users of the AWS Management Console. IAM users in the AWS Management Console unknowingly use temporary credentials. Users sign into the console using their user name and password, which are long-term credentials. However, in the background, the console generates temporary credentials on behalf of the user. To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * The `aws:MultiFactorAuthPresent` key is not present when an API or CLI command is called with long-term credentials, such as user access key pairs. Therefore we recommend that when you check for this key that you use the [...IfExists](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IfExists) versions of the condition operators.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-multifactorauthpresent
     *
     * @param value Weather the MFA should be present or absent. **Default:** `true`
     */
    ifAwsMultiFactorAuthPresent(value) {
        return this.if(`aws:MultiFactorAuthPresent`, typeof value !== 'undefined' ? value : true, operators_1.Operator.bool);
    }
    /**
     * Compare the account to which the requesting principal belongs with the account identifier that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalaccount
     *
     * @param value Account identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalAccount(value, operator) {
        return this.if('aws:PrincipalAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) (ARN) of the principal that made the request with the ARN that you specify.
     *
     * For IAM roles, the request context returns the ARN of the role, not the ARN of the user that assumed the role. To learn which types of principals you can specify in this condition key, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn
     *
     * @param value Principle ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsPrincipalArn(value, operator) {
        return this.if('aws:PrincipalArn', value, operator ?? operators_1.Operator.arnLike);
    }
    /**
     * Check whether the call to your resource is being made directly by an AWS [service principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services). For example, AWS CloudTrail uses the service principal `cloudtrail.amazonaws.com` to write logs to your Amazon S3 bucket. The request context key is set to true when a service uses a service principal to perform a direct action on your resources. The context key is set to false if the service uses the credentials of an IAM principal to make a request on the principal's behalf. It is also set to false if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf.
     *
     * **Availability:** This key is present in the request context for all signed API requests that use AWS credentials.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalisawsservice
     *
     * @param value Weather the call to your resource is being made directly by an AWS service principal. **Default:** `true`
     */
    ifAwsPrincipalIsAWSService(value) {
        return this.if(`aws:PrincipalIsAWSService`, typeof value !== 'undefined' ? value : true, operators_1.Operator.bool);
    }
    /**
     * Compare the identifier of the organization in AWS Organizations to which the requesting principal belongs with the identifier you specify.
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * This global key provides an alternative to listing all the account IDs for all AWS accounts in an organization. You can use this condition key to simplify specifying the `Principal` element in a resource-based policy. You can specify the organization ID in the condition element. When you add and remove accounts, policies that include the `aws:PrincipalOrgID` key automatically include the correct accounts and don't require manual updating.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgid
     *
     * @param value Organization ID(s) in format `o-xxxxxxxxxxx`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalOrgID(value, operator) {
        return this.if('aws:PrincipalOrgID', value, operator);
    }
    /**
     * Compare the AWS Organizations path for the principal who is making the request to the path you provide. That principal can be an IAM user, IAM role, federated user, or AWS account root user.
     *
     * This condition ensures that the requester is an account member within the specified organization root or organizational units (OUs) in AWS Organizations. An AWS Organizations path is a text representation of the structure of an Organizations entity. For more information about using and understanding paths, see Understand the [AWS Organizations Entity Path](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path).
     *
     * **Availability:** This key is included in the request context only if the principal is a member of an organization.
     *
     * **Note:** Organization IDs are globally unique but OU IDs and root IDs are unique only within an organization. This means that no two organizations share the same organization ID. However, another organization might have an OU or root with the same ID as yours. We recommend that you always include the organization ID when you specify an OU or root.
     *
     * `aws:PrincipalOrgPaths` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. When you use multiple values with the `ForAnyValue:` condition operator, the principal's path must match one of the paths provided. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array (`"Key":["Value1", "Value2"]`). You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Creating a Condition with Multiple Keys or Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalorgpaths
     *
     * @param value Organization path(s) in the format of `o-xxxxxxxxxxx/r-xxxxxxxxxx/ou-xxxx-xxxxxxxx/ou-xxxx-xxxxxxxx/`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalOrgPaths(value, operator) {
        return this.if('aws:PrincipalOrgPaths', value, operator);
    }
    /**
     * Compare the [service principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services) name in the policy with the service principal that is making requests to your resources. You can use this key to check whether this call is made by a specific service principal. When a service principal makes a direct request to your resource, the `aws:PrincipalServiceName` key contains the name of the service principal. For example, the AWS CloudTrail service principal name is `cloudtrail.amazonaws.com`.
     *
     * **Availability:** This key is present in the request when the call is made by an AWS service principal. This key is not present in any other situation, including the following:
     *
     * - If the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf.
     * - If the service uses the credentials of an IAM principal to make a request on the principal's behalf.
     * - If the call is made directly by an IAM principal.
     *
     * You can use this condition key to limit access to your trusted identities and expected network locations, while safely granting access to an AWS service.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalservicename
     *
     * @param value AWS service name, e.g. `cloudtrail.amazonaws.com`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalServiceName(value, operator) {
        return this.if('aws:PrincipalServiceName', value, operator);
    }
    /**
     * This key provides a list of all [service principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#principal-services) names that belong to the service. This is an advanced condition key. You can use it to restrict the service from accessing your resource from a specific Region only. Some services may create Regional service principals to indicate a particular instance of the service within a specific Region. You can limit access to a resource to a particular instance of the service. When a service principal makes a direct request to your resource, the `aws:PrincipalServiceNamesList` contains an unordered list of all service principal names associated with the Regional instance of the service.
     *
     * **Availability:** This key is present in the request when the call is made by an AWS service principal. This key is not present in any other situation, including the following:
     *
     * - If the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf.
     * - If the service uses the credentials of an IAM principal to make a request on the principal's behalf.
     * - If the call is made directly by an IAM principal.
     *
     * `aws:PrincipalServiceNamesList` is a multivalued condition key. Multivalued keys include one or more values in a list format. The result is a logical `OR`. You must use the `ForAnyValue` or `ForAllValues` set operators with the `StringLike` [condition operator](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String) when you use this key. For policies that include multiple values for a single key, you must enclose the conditions within brackets like an array, such as `("Key":["Value1", "Value2"])`. You should also include these brackets when there is a single value. For more information about multivalued condition keys, see [Using multiple keys and values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalservicenameslist
     *
     * @param value AWS service name list
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ForAnyValue:StringLike`
     */
    ifAwsPrincipalServiceNamesList(value, operator) {
        return this.if('aws:PrincipalServiceNamesList', value, operator ?? new operators_1.Operator().stringLike().forAnyValue());
    }
    /**
     * Compare the tag attached to the principal making the request with the tag that you specify. If the principal has more than one tag attached, the request context includes one aws:PrincipalTag key for each attached tag key.
     *
     * **Availability:** This key is included in the request context if the principal is using an IAM user with attached tags. It is included for a principal using an IAM role with attached tags or [session tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_session-tags.html).
     *
     * You can add custom attributes to a user or role in the form of a key-value pair. For more information about IAM tags, see [Tagging IAM Users and Roles](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_tags.html). You can use `aws:PrincipalTag` to [control access](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_iam-tags.html#access_iam-tags_control-resources) for AWS principals.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsPrincipalTag(key, value, operator) {
        return this.if(`aws:PrincipalTag/${key}`, value, operator);
    }
    /**
     * Compare the type of principal making the request with the principal type that you specify. For details about how the information appears in the request context for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principaltype
     *
     * @param value The principal type(s). Any of `Account`, `User`, `FederatedUser`, `AssumedRole`, `Anonymous`
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsPrincipalType(value, operator) {
        return this.if('aws:PrincipalType', value, operator);
    }
    /**
     * Compare who referred the request in the client browser with the referer that you specify. The `aws:referer` request context value is provided by the caller in an HTTP header.
     *
     * **Availability:** This key is included in the request context only if the request was invoked using a URL in the browser.
     *
     * For example, you can call [Amazon S3 API operations directly using a web browser](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies.html#example-bucket-policies-use-case-4). This means that you can view S3 objects, such as images and documents, directly through a web browser. The `aws:referer` condition allows you to restrict access to specific values in the HTTP or HTTPS request based on the value of the referrer header.
     *
     * **Warning:** This condition should be used carefully. It is dangerous to include a publicly known referer header value. Unauthorized parties can use modified or custom browsers to provide any `aws:referer` value that they choose. As a result, `aws:referer` should not be used to prevent unauthorized parties from making direct AWS requests. It is offered only to allow customers to protect their digital content, such as content stored in Amazon S3, from being referenced on unauthorized third-party sites.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-referer
     *
     * @param value The referer url(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsReferer(value, operator) {
        return this.if('aws:Referer', value, operator);
    }
    /**
     * Compare the AWS Region that was called in the request with the Region that you specify. You can use this global condition key to control which Regions can be requested. To view the AWS Regions for each service, see [Service endpoints and quotas](https://docs.aws.amazon.com/general/latest/gr/aws-service-information.html) in the Amazon Web Services General Reference.
     *
     * **Availability:** This key is always included in the request context.
     *
     * Some global services, such as IAM, have a single endpoint. Because this endpoint is physically located in the US East (N. Virginia) Region, IAM calls are always made to the us-east-1 Region. For example, if you create a policy that denies access to all services if the requested Region is not us-west-2, then IAM calls always fail. To view an example of how to work around this, see [NotAction with Deny](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notaction.html).
     *
     * **Note:** The `aws:RequestedRegion` condition key allows you to control which endpoint of a service is invoked but does not control the impact of the operation. Some services have cross-Region impacts. For example, Amazon S3 has API operations that control cross-Region replication. You can invoke `s3:PutBucketReplication` in one Region (which is affected by the `aws:RequestedRegion` condition key), but other Regions are affected based on the replications configuration settings.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requestedregion
     *
     * @param value The region(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsRequestedRegion(value, operator) {
        return this.if('aws:RequestedRegion', value, operator);
    }
    /**
     * Compare the tag key-value pair that was passed in the request with the tag pair that you specify. For example, you could check whether the request includes the tag key `Dept` and that it has the value `Accounting`. For more information, see [Controlling Access During AWS Requests](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-requests).
     *
     * **Availability:** This key is included in the request context when tags are passed in the request. When multiple tags are passed in the request, there is one context key for each tag key-value pair.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a multivalued request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see [Using Multiple Keys and Values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-requesttag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsRequestTag(key, value, operator) {
        return this.if(`aws:RequestTag/${key}`, value, operator);
    }
    /**
     * Use this key to compare the requested resource owner's [AWS account ID](https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html) with the resource account in the policy. You can then allow or deny access to that resource based on the account that owns the resource.
     *
     * This key is equal to the AWS account ID for the account with the resources evaluated in the request.
     *
     * For most resources in your account, the ARN contains the owner account ID for that resource. For certain resources, such as Amazon S3 buckets, the resource ARN does not include the account ID. The following two examples show the difference between a resource with an account ID in the ARN, and an Amazon S3 ARN without an account ID:
     *
     * - `arn:aws:iam::123456789012:role/AWSExampleRole` - IAM role created and owned within the account 123456789012.
     * - `arn:aws:s3:::DOC-EXAMPLE-BUCKET2` - Amazon S3 bucket created and owned within the account 111122223333, not displayed in the ARN.
     *
     * **Availability:** This key is always included in the request context for most service actions. The following actions don't support this key:
     *
     *   - Amazon Elastic Block Store - All actions
     *   - Amazon EC2
     *     - `ec2:CopyFpgaImage`
     *     - `ec2:CopyImage`
     *     - `ec2:CopySnapshot`
     *     - `ec2:CreateTransitGatewayPeeringAttachment`
     *     - `ec2:CreateVolume`
     *     - `ec2:CreateVpcPeeringConnection`
     *   - Amazon EventBridge - All actions
     *   - Amazon WorkSpaces
     *     - `workspaces:CopyWorkspaceImage`
     *     - `workspaces:DescribeWorkspaceImages`
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourceaccount
     *
     * @param value The account ID
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceAccount(value, operator) {
        return this.if('aws:ResourceAccount', value, operator);
    }
    /**
     * Use this key to compare the identifier of the organization in AWS Organizations to which the requested resource belongs with the identifier specified in the policy.
     *
     * This global key returns the resource organization ID for a given request. It allows you to create rules that apply to all resources in an organization that are specified in the Resource element of an [identity-based policy](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_identity-vs-resource.html). You can specify the [organization ID](https://docs.aws.amazon.com/organizations/latest/userguide/orgs_manage_org_details.html) in the condition element. When you add and remove accounts, policies that include the aws:ResourceOrgID key automatically include the correct accounts and you don't have to manually update it.
     *
     * **Note:** Some AWS services require access to AWS owned resources that are hosted in another AWS account. Using `aws:ResourceOrgID` in your identity-based policies might impact your identity's ability to access these resources.
     *
     * **Availability:** This key is included in the request context only if the account that owns the resource is a member of an organization. This global condition key does not support the following actions:
     *
     * - Amazon Elastic Block Store - All actions
     * - Amazon EC2
     *   - `ec2:CopyFpgaImage`
     *   - `ec2:CopyImage`
     *   - `ec2:CopySnapshot`
     *   - `ec2:CreateTransitGatewayPeeringAttachment`
     *   - `ec2:CreateVolume`
     *   - `ec2:CreateVpcPeeringConnection`
     * - Amazon EventBridge - All actions
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourceorgid
     *
     * @param value ID of an organization
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceOrgID(value, operator) {
        return this.if('aws:ResourceOrgID', value, operator);
    }
    /**
     * Use this key to compare the AWS Organizations path for the accessed resource to the path in the policy. In a policy, this condition key ensures that the resource belongs to an account member within the specified organization root or organizational units (OUs) in AWS Organizations. An AWS Organizations path is a text representation of the structure of an Organizations entity. For more information about using and understanding paths, see [Understand the AWS Organizations entity path](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor-view-data-orgs.html#access_policies_access-advisor-viewing-orgs-entity-path).
     *
     * `aws:ResourceOrgPaths` is a multivalued condition key. Multivalued keys can have multiple values in the request context. You must use the `ForAnyValue` or `ForAllValues` set operators with [string condition operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String) for this key. For more information about multivalued condition keys, see [Using multiple keys and values](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html#reference_policies_multi-key-or-value-conditions).
     *
     * **Note:** Some AWS services require access to AWS owned resources that are hosted in another AWS account. Using aws:ResourceOrgPaths in your identity-based policies might impact your identity's ability to access these resources.
     *
     * **Availability:** This key is included in the request context only if the account that owns the resource is a member of an organization. This global condition key does not support the following actions:
     *
     * - Amazon Elastic Block Store - All actions
     * - Amazon EC2
     *   - `ec2:CopyFpgaImage`
     *   - `ec2:CopyImage`
     *   - `ec2:CopySnapshot`
     *   - `ec2:CreateTransitGatewayPeeringAttachment`
     *   - `ec2:CreateVolume`
     *   - `ec2:CreateVpcPeeringConnection`
     * - Amazon EventBridge - All actions
     * - Amazon WorkSpaces
     *   - `workspaces:CopyWorkspaceImage`
     *   - `workspaces:DescribeWorkspaceImages`
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourceorgpaths
     *
     * @param value The path of an organization
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceOrgPaths(value, operator) {
        return this.if('aws:ResourceOrgPaths', value, operator);
    }
    /**
     * Compare the tag key-value pair that you specify with the key-value pair that is attached to the resource. For example, you could require that access to a resource is allowed only if the resource has the attached tag key `Dept` with the value `Marketing`. For more information, see [Controlling Access to AWS Resources](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-resources).
     *
     * **Availability:** This key is included in the request context when the requested resource already has attached tags. This key is returned only for resources that [support authorization based on tags](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html). There is one context key for each tag key-value pair.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-resourcetag
     *
     * @param key The tag key to check
     * @param value The tag value(s) to check against
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsResourceTag(key, value, operator) {
        return this.if(`aws:ResourceTag/${key}`, value, operator);
    }
    /**
     * Check whether the request was sent using SSL. The request context returns `true` or `false`. In a policy, you can allow specific actions only if the request is sent using SSL.
     *
     * **Availability:** This key is always included in the request context.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-securetransport
     *
     * @param value Weather request was sent using SSL. **Default:** `true`
     */
    ifAwsSecureTransport(value) {
        return this.if(`aws:SecureTransport`, typeof value !== 'undefined' ? value : true, operators_1.Operator.bool);
    }
    /**
     * Compare the account ID of the resource making a service-to-service request with the account ID that you specify.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the resource ARN of the source to the called service. This ARN includes the source account ID.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the account ID from the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceaccount
     *
     * @param value The account ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceAccount(value, operator) {
        return this.if('aws:SourceAccount', value, operator);
    }
    /**
     * Compare the [Amazon Resource Name (ARN)](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-arns) of the resource making a service-to-service request with the ARN that you specify.
     *
     * This key does not work with the ARN of the principal making the request. Instead, use [aws:PrincipalArn](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-principalarn). The source's ARN includes the account ID, so it is not necessary to use `aws:SourceAccount` with `aws:SourceArn`.
     *
     * **Availability:** This key is included in the request context only if accessing a resource triggers an AWS service to call another service on behalf of the resource owner. The calling service must pass the ARN of the original resource to the called service.
     *
     * You can use this condition key to check that Amazon S3 is not being used as a [confused deputy](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_create_for-user_externalid.html#confused-deputy). For example, when an Amazon S3 bucket update triggers an Amazon SNS topic post, the Amazon S3 service invokes the `sns:Publish` API operation. The bucket is considered the source of the SNS request and the value of the key is the bucket's ARN.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcearn
     *
     * @param value The source ARN(s)
     * @param operator Works with [ARN operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN) and [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `ArnLike`
     */
    ifAwsSourceArn(value, operator) {
        return this.if('aws:SourceArn', value, operator ?? operators_1.Operator.arnLike);
    }
    /**
     * Compare the source identity that was set by the principal with the source identity that you specify in the policy.
     *
     * **Availability:** This key is included in the request context after a source identity has been set when a role is assumed using any AWS STS assume-role CLI command, or AWS STS `AssumeRole` API operation.
     *
     * You can use this key in a policy to allow actions in AWS by principals that have set a source identity when assuming a role. Activity for the role's specified source identity appears in [AWS CloudTrail](https://docs.aws.amazon.com/IAM/latest/UserGuide/cloudtrail-integration.html#cloudtrail-integration_signin-tempcreds). This makes it easier for administrators to determine who or what performed actions with a role in AWS.
     *
     * Unlike [sts:RoleSessionName](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_rolesessionname), after the source identity is set, the value cannot be changed. It is present in the request context for all actions taken by the role. The value persists into subsequent role sessions when you use the session credentials to assume another role. Assuming one role from another is called [role chaining](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-role-chaining).
     *
     * The [sts:SourceIdentity](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_iam-condition-keys.html#ck_sourceidentity) key is present in the request when the principal initially sets a source identity while assuming a role using any AWS STS assume-role CLI command, or AWS STS `AssumeRole` API operation. The `aws:SourceIdentity` key is present in the request for any actions that are taken with a role session that has a source identity set.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceidentity
     *
     * @param value The source identity
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceIdentity(value, operator) {
        return this.if('aws:SourceIdentity', value, operator);
    }
    /**
     * Compare the requester's IP address with the IP address that you specify.
     *
     * **Availability:** This key is included in the request context, except when the requester uses a VPC endpoint to make the request.
     *
     * The `aws:SourceIp` condition key can be used in a policy to allow principals to make requests only from within a specified IP range. However, this policy denies access if an AWS service makes calls on the principal's behalf. In this case, you can use `aws:SourceIp` with the [aws:ViaAWSService](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice) key to ensure that the source IP restriction applies only to requests made directly by a principal.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourceip
     *
     * @param value The source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsSourceIp(value, operator) {
        return this.if('aws:SourceIp', value, operator ?? operators_1.Operator.ipAddress);
    }
    /**
     * Check whether the request comes from the VPC that you specify. In a policy, you can use this condition to allow access to only a specific VPC. For more information, see [Restricting Access to a Specific VPC](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpc
     *
     * @param value The VPS ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringEquals`
     */
    ifAwsSourceVpc(value, operator) {
        return this.if('aws:SourceVpc', value, operator);
    }
    /**
     * Compare the VPC endpoint identifier of the request with the endpoint ID that you specify. In a policy, you can use this condition to restrict access to a specific VPC endpoint. For more information, see [Restricting Access to a Specific VPC Endpoint](https://docs.aws.amazon.com/AmazonS3/latest/dev/example-bucket-policies-vpc-endpoint.html#example-bucket-policies-restrict-access-vpc-endpoint) in the *Amazon Simple Storage Service Developer Guide*.
     *
     * **Availability:** This key is included in the request context only if the requester uses a VPC endpoint to make the request.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-sourcevpce
     *
     * @param value The VPC Endpoint ID(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsSourceVpce(value, operator) {
        return this.if('aws:SourceVpce', value, operator);
    }
    /**
     * Compare the tag keys in a request with the keys that you specify. As a best practice when you use policies to control access using tags, use the `aws:TagKeys` condition key to define what tag keys are allowed. For example policies and more information, see [Controlling Access Based on Tag Keys](https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html#access_tags_control-tag-keys).
     *
     * **Availability:** This key is included in the request context only if the operation supports attaching tags to resources.
     *
     * Because you can include multiple tag key-value pairs in a request, the request content could be a [multivalued](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_multi-value-conditions.html) request. In this case, you should consider using the `ForAllValues` or `ForAnyValue` set operators. For more information, see Using Multiple Keys and Values.
     *
     * Some services support tagging with resource operations, such as creating, modifying, or deleting a resource. To allow tagging and operations as a single call, you must create a policy that includes both the tagging action and the resource-modifying action. You can then use the `aws:TagKeys` condition key to enforce using specific tag keys in the request. For example, to limit tags when someone creates an Amazon EC2 snapshot, you must include the `ec2:CreateSnapshot` creation action ***and*** the `ec2:CreateTags` tagging action in the policy. To view a policy for this scenario that uses `aws:TagKeys`, see [Creating a Snapshot with Tags](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ExamplePolicies_EC2.html#iam-creating-snapshot-with-tags) in the *Amazon EC2 User Guide for Linux Instances*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tagkeys
     *
     * @param value The tag key(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsTagKeys(value, operator) {
        return this.if('aws:TagKeys', value, operator);
    }
    /**
     * Compare the date and time that temporary security credentials were issued with the date and time that you specify.
     *
     * **Availability:** This key is included in the request context only when the principal uses temporary credentials to make the request. They key is not present in AWS CLI, AWS API, or AWS SDK requests that are made using access keys.
     *
     * To learn which services support using temporary credentials, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-tokenissuetime
     *
     * @param value The date and time to check for. Can be a string in one of the [W3C implementations of the ISO 8601 date](https://www.w3.org/TR/NOTE-datetime) (e.g. `2020-04-01T00:00:00Z`) or in epoch (UNIX) time or a `Date()` object (JavaScript only)
     * @param operator Works with [date operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Date). **Default:** `DateGreaterThanEquals`
     */
    ifAwsTokenIssueTime(value, operator) {
        return this.if('aws:TokenIssueTime', dateToString(value), operator ?? new operators_1.Operator().dateGreaterThanEquals());
    }
    /**
     * Compare the requester's client application with the application that you specify.
     *
     * **Availability:** This key is always included in the request context.
     *
     * **Warning:** This key should be used carefully. Since the `aws:UserAgent` value is provided by the caller in an HTTP header, unauthorized parties can use modified or custom browsers to provide any `aws:UserAgent` value that they choose. As a result, `aws:UserAgent` should not be used to prevent unauthorized parties from making direct AWS requests. You can use it to allow only specific client applications, and only after testing your policy.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-useragent
     *
     * @param value The User Agent string(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserAgent(value, operator) {
        return this.if('aws:UserAgent', value, operator);
    }
    /**
     * Compare the requester's principal identifier with the ID that you specify. For IAM users, the request context value is the user ID. For IAM roles, this value format can vary. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is included in the request context for all signed requests. Anonymous requests do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-userid
     *
     * @param value The principal identifier(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUserid(value, operator) {
        return this.if('aws:userid', value, operator);
    }
    /**
     * Compare the requester's user name with the user name that you specify. For details about how the information appears for different principals, see [Specifying a Principal](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html#Principal_specifying).
     *
     * **Availability:** This key is always included in the request context for IAM users. Anonymous requests and requests that are made using the AWS account root user or IAM roles do not include this key.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-username
     *
     * @param value The user name(s)
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAwsUsername(value, operator) {
        return this.if('aws:username', value, operator);
    }
    /**
     * Check whether an AWS service makes a request to another service on your behalf.
     *
     * The request context key returns `true` when a service uses the credentials of an IAM principal to make a request on behalf of the principal. The context key returns `false` if the service uses a [service role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-role) or [service-linked role](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role) to make a call on the principal's behalf. The request context key also returns `false` when the principal makes the call directly.
     *
     * **Availability:** This key is always included in the request context for most services.
     *
     *The following services do not currently support `aws:ViaAWSService`:
     *- Amazon EC2
     *- AWS Glue
     *- AWS Lake Formation
     *- AWS OpsWorks
     *
     * You can use this condition key to allow or deny access based on whether a request was made by a service. To view an example policy, see [AWS: Denies Access to AWS Based on the Source IP](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_examples_aws_deny-ip.html).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-viaawsservice
     *
     * @param value Whether a request was made by a service. **Default:** `true`
     */
    ifAwsViaAWSService(value) {
        return this.if(`aws:ViaAWSService`, typeof value !== 'undefined' ? value : true, operators_1.Operator.bool);
    }
    /**
     * Compare the IP address from which a request was made with the IP address that you specify. In a policy, the key matches only if the request originates from the specified IP address and it goes through a VPC endpoint.
     *
     * **Availability:** This key is included in the request context only if the request is made using a VPC endpoint.
     *
     * For more information, see [Controlling Access to Services with VPC Endpoints](https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-access.html) in the *Amazon VPC User Guide*.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_condition-keys.html#condition-keys-vpcsourceip
     *
     * @param value The VPC source IP(s)
     * @param operator Works with IP [address operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_IPAddress). **Default:** `IpAddress`
     */
    ifAwsVpcSourceIp(value, operator) {
        return this.if('aws:VpcSourceIp', value, operator ?? operators_1.Operator.ipAddress);
    }
}
exports.PolicyStatementWithCondition = PolicyStatementWithCondition;
function dateToString(value) {
    if (typeof value.getMonth === 'function') {
        value = value.toISOString();
    }
    return value;
}
//# sourceMappingURL=data:application/json;base64,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