# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN

|hland_96p| combines `HBV96's` :cite:t:`ref-Lindstrom1997HBV96` and `PREVAH's`
:cite:t:`ref-Viviroli2009PREVAH` process equations.  We implemented it on behalf of the
`German Federal Institute of Hydrology (BfG)`_ as an alternative to |hland_96| for
modelling large river basins in central Europe.  All processes "above the soil" (input
data correction, interception, snowmelt) and "inside the soil" (evaporation, generation
of effective precipitation), as well as the handling of water areas, are identical with
|hland_96| (and so with HBV96). Most processes "below the soil" agree with PREVAH
(runoff generation and runoff concentration).

The following figure shows the general structure of |hland_96p|:

.. image:: HydPy-H-HBV96-PREVAH.png

Comparing the above figure with the analogue figure of |hland_96| reveals that
|hland_96p| models five instead of two runoff components, requiring a larger number of
vertically arranged storages.  The two upper storages (|SUZ|, related to the soil, and
|SG1|, related to dynamic groundwater aquifers) are zone-specific.  In comparison, the
upper zone layer (|UZ|) of |hland_96| is subbasin-specific.  |hland_96p| models the two
lower storages (|SG2| and |SG3|), dealing with slow and slowest groundwater responses,
slightly different from PREVAH.  First, their outflows (|RG2| and |RG3|) are
individually accessible (PREVAH handles their outflow as one runoff component).  Second,
it represents internal lakes as an "extension" of the groundwater (like in
|hland_96|/HBV96).  Like all |hland| application models, |hland_96p| optionally allows
using a submodel for additional consideration of runoff concentration. While the figure
of |hland_96| relies on |rconc_uh|, the figure of |hland_96p| selects |rconc_nash| as
an example.

Integration tests
=================

.. how_to_understand_integration_tests::

We define the settings of the following test calculations as similar to the ones of
application model |hland_96| as possible.  Hence, comparing the test results gives a
good impression of the functional differences of both models.

The following settings are identical:

>>> from hydpy import pub
>>> pub.timegrids = "01.01.2000", "05.01.2000", "1h"

>>> from hydpy.models.hland_96p import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> land = Element("land", outlets=outlet)
>>> land.model = model
>>> nmbzones(1)
>>> sclass(1)
>>> area(1.0)
>>> zonearea(1.0)
>>> zonez(1.0)
>>> psi(1.0)

We assign identical values to all parameters, besides those that are specific to
|hland_96p| (|SGR|, |SG1Max|, |K0|, |K1|, |K2|, and |K3|). Also, we use the same
submodels to simulate evaporation but select |rconc_nash| instead of |rconc_uh| for
extending the simulation of runoff concentration processes:

>>> zonetype(FIELD)
>>> pcorr(1.1)
>>> pcalt(0.1)
>>> rfcf(1.1)
>>> sfcf(1.3)
>>> tcorr(0.6)
>>> tcalt(0.6)
>>> icmax(2.0)
>>> sfdist(1.0)
>>> smax(inf)
>>> sred(0.0)
>>> tt(0.0)
>>> ttint(2.0)
>>> dttm(1.0)
>>> cfmax(0.5)
>>> cfvar(0.1)
>>> gmelt(1.0)
>>> gvar(0.2)
>>> cfr(0.1)
>>> whc(0.4)
>>> fc(200.0)
>>> beta(2.0)
>>> percmax(2.0)
>>> sgr(20.0)
>>> sg1max(50.0)
>>> k0(2.0)
>>> k1(10.0)
>>> k2(20.0)
>>> k3(100.0)

>>> with model.add_aetmodel_v1("evap_aet_hbv96"):
...     temperaturethresholdice(0.0)
...     soilmoisturelimit(0.8)
...     excessreduction(0.5)
...     with model.add_petmodel_v1("evap_pet_hbv96"):
...         evapotranspirationfactor(0.7)
...         airtemperaturefactor(0.1)
...         altitudefactor(-0.1)
...         precipitationfactor(0.1)

>>> with model.add_rconcmodel_v1("rconc_nash"):
...     retentiontime(3.0)
...     nmbstorages(5)
...     nmbsteps(100)

>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = (
...     inputs.p, fluxes.rs, fluxes.ri, fluxes.rg1, fluxes.rg2, fluxes.rg3, fluxes.rt)
>>> IntegrationTest.plotting_options.axis2 = inputs.t
>>> test = IntegrationTest(land)
>>> test.dateformat = '%d/%m %H:00'

The same holds for the initial conditions.

>>> test.inits = ((states.ic, 0.0),
...               (states.sp, 0.0),
...               (states.wc, 0.0),
...               (states.sm, 100.0),
...               (states.suz, 0.0),
...               (states.sg1, 10.0),
...               (states.sg2, 10.0),
...               (states.sg3, 10.0),
...               (model.rconcmodel.sequences.states.sc, 0.05))

All input time series are identical:

>>> inputs.p.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0, 2.1, 10.4, 3.5, 3.4,
...     1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1, 20.7, 37.9, 8.2, 3.6, 7.5, 18.5,
...     15.4, 6.3, 1.9, 4.9, 2.7, 0.5, 0.2, 0.5, 2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6,
...     0.7, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     1.3, 0.0, 0.0, 0.0, 0.7, 0.4, 0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
>>> inputs.t.series = (
...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2, 35.2, 37.1,
...     31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8, 19.6, 19.2, 19.2, 19.2,
...     18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8, 19.0, 19.2, 19.3, 19.0, 18.8, 18.7,
...     17.8, 17.4, 17.3, 16.8, 16.5, 16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3,
...     14.9, 15.7, 16.0, 16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3,
...     18.1, 16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1, 11.9,
...     12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8, 22.2, 20.1, 17.8,
...     15.2, 14.5, 12.4, 11.7, 11.9)
>>> petinputs = model.aetmodel.petmodel.sequences.inputs
>>> petinputs.normalairtemperature.series = inputs.t.series - 1.0
>>> petinputs.normalevapotranspiration.series = (
...     0.100707, 0.097801, 0.096981, 0.09599, 0.096981, 0.102761, 0.291908, 1.932875,
...     4.369536, 7.317556, 8.264362, 9.369867, 5.126178, 6.62503, 7.397619, 2.39151,
...     1.829834, 1.136569, 0.750986, 0.223895, 0.099425, 0.098454, 0.098128, 0.097474,
...     0.097474, 0.097474, 0.096981, 0.096652, 0.096321, 0.09599, 0.187298, 1.264612,
...     3.045538, 1.930758, 2.461001, 6.215945, 3.374783, 8.821555, 4.046025, 2.110757,
...     2.239257, 2.877848, 1.591452, 0.291604, 0.092622, 0.092451, 0.091248, 0.089683,
...     0.089858, 0.089683, 0.088805, 0.089157, 0.090207, 0.091593, 0.154861, 0.470369,
...     1.173726, 4.202296, 4.359715, 5.305753, 5.376027, 4.658915, 7.789594, 4.851567,
...     5.30692, 3.286036, 1.506216, 0.274762, 0.087565, 0.085771, 0.084317, 0.083215,
...     0.082289, 0.0845, 0.084864, 0.083584, 0.0834, 0.084864, 0.310229, 1.391958,
...     3.195876, 5.191651, 7.155036, 8.391432, 8.391286, 10.715238, 9.383394, 7.861915,
...     6.298329, 2.948416, 1.309232, 0.32955, 0.089508, 0.085771, 0.0845, 0.084864)

>>> test.reset_inits()
>>> conditions = model.conditions

.. _hland_96p_field:

field
_____

|hland_96p| neither implements an approach analogue to HBV96's contributing area
concept nor a substep mechanism controlling the numerical accuracy of the runoff
generation module.  Hence, we provide only a single "field" example that is comparable
both with the :ref:`first <hland_96_field>` and the :ref:`second <hland_96_resparea>`
example of |hland_96|:

.. integration-test::

    >>> test('hland_96p_field')
    |        date |    p |    t |   tc | fracrain | rfc | sfc |    cfact | swe | gact |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       dp |  el |       rs |       ri |      gr1 |      rg1 |      gr2 |      rg2 |      gr3 |      rg3 |      inrc |    outrc |       rt |       qt |       ic |  sp |  wc |         sm |       suz |       sg1 |       sg2 |       sg3 |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.048465 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.487706 |      0.0 | 0.099502 |      0.0 | 0.011105 |  0.598312 | 0.202814 | 0.202814 | 0.056337 |      0.0 | 0.0 | 0.0 |  99.951535 |       0.0 |  9.512294 |  9.900498 |  9.988895 | 0.056337 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.047044 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.46392 |      0.0 | 0.098512 |      0.0 | 0.011093 |  0.573524 | 0.414293 | 0.414293 | 0.115081 |      0.0 | 0.0 | 0.0 |  99.904491 |       0.0 |  9.048374 |  9.801987 |  9.977802 | 0.115081 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046628 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.441294 |      0.0 | 0.097531 |      0.0 |  0.01108 |  0.549906 | 0.547167 | 0.547167 | 0.151991 |      0.0 | 0.0 | 0.0 |  99.857863 |       0.0 |   8.60708 |  9.704455 |  9.966722 | 0.151991 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04613 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.419772 |      0.0 | 0.096561 |      0.0 | 0.011068 |  0.527401 | 0.556325 | 0.556325 | 0.154535 |      0.0 | 0.0 | 0.0 |  99.811734 |       0.0 |  8.187308 |  9.607894 |  9.955654 | 0.154535 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046584 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.3993 |      0.0 |   0.0956 |      0.0 | 0.011056 |  0.505956 | 0.537334 | 0.537334 |  0.14926 |      0.0 | 0.0 | 0.0 |   99.76515 |       0.0 |  7.788008 |  9.512294 |  9.944598 |  0.14926 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.049338 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.379826 |      0.0 | 0.094649 |      0.0 | 0.011043 |  0.485518 | 0.515762 | 0.515762 | 0.143267 |      0.0 | 0.0 | 0.0 |  99.715812 |       0.0 |  7.408182 |  9.417645 |  9.933555 | 0.143267 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.140081 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.361301 |      0.0 | 0.093707 |      0.0 | 0.011031 |   0.46604 | 0.494888 | 0.494888 | 0.137469 |      0.0 | 0.0 | 0.0 |   99.57573 |       0.0 |  7.046881 |  9.323938 |  9.922524 | 0.137469 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.92625 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.34368 |      0.0 | 0.092775 |      0.0 | 0.011019 |  0.447474 | 0.474971 | 0.474971 | 0.131936 |      0.0 | 0.0 | 0.0 |  98.649481 |       0.0 |    6.7032 |  9.231163 |  9.911505 | 0.131936 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  2.07444 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.326919 |      0.0 | 0.091852 |      0.0 | 0.011007 |  0.429777 | 0.455987 | 0.455987 | 0.126663 |      0.0 | 0.0 | 0.0 |  96.575041 |       0.0 |  6.376282 |  9.139312 |  9.900498 | 0.126663 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.400961 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.310975 |      0.0 | 0.090938 |      0.0 | 0.010994 |  0.412907 | 0.437892 | 0.437892 | 0.121637 |      0.0 | 0.0 | 0.0 |   93.17408 |       0.0 |  6.065307 |  9.048374 |  9.889504 | 0.121637 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.705742 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.295808 |      0.0 | 0.090033 |      0.0 | 0.010982 |  0.396824 | 0.420642 | 0.420642 | 0.116845 |      0.0 | 0.0 | 0.0 |  89.468338 |       0.0 |  5.769498 |  8.958341 |  9.878522 | 0.116845 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.03435 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.281382 |      0.0 | 0.089137 |      0.0 |  0.01097 |  0.381489 | 0.404198 | 0.404198 | 0.112277 |      0.0 | 0.0 | 0.0 |  85.433988 |       0.0 |  5.488116 |  8.869204 |  9.867552 | 0.112277 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.107634 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.267659 |      0.0 |  0.08825 |      0.0 | 0.010958 |  0.366867 | 0.388519 | 0.388519 | 0.107922 |      0.0 | 0.0 | 0.0 |  83.326354 |       0.0 |  5.220458 |  8.780954 |  9.856594 | 0.107922 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.656691 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.254605 |      0.0 | 0.087372 |      0.0 | 0.010946 |  0.352922 |  0.37357 |  0.37357 |  0.10377 |      0.0 | 0.0 | 0.0 |  80.669664 |       0.0 |  4.965853 |  8.693582 |  9.845648 |  0.10377 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.871924 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.242188 |      0.0 | 0.086503 |      0.0 | 0.010934 |  0.339624 | 0.359315 | 0.359315 |  0.09981 |      0.0 | 0.0 | 0.0 |   77.79774 |       0.0 |  4.723666 |   8.60708 |  9.834715 |  0.09981 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  0.242 |    0.242 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.873977 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.230376 |      0.0 | 0.085642 |      0.0 | 0.010921 |  0.326939 |  0.34572 |  0.34572 | 0.096033 |      0.0 | 0.0 | 0.0 |  76.923763 |       0.0 |   4.49329 |  8.521438 |  9.823793 | 0.096033 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.677397 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.21914 |      0.0 |  0.08479 |      0.0 | 0.010909 |  0.314839 | 0.332754 | 0.332754 | 0.092432 |      0.0 | 0.0 | 0.0 |  76.246366 |       0.0 |  4.274149 |  8.436648 |  9.812884 | 0.092432 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.417048 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.208453 |      0.0 | 0.083946 |      0.0 | 0.010897 |  0.303296 | 0.320386 | 0.320386 | 0.088996 |      0.0 | 0.0 | 0.0 |  75.829319 |       0.0 |  4.065697 |  8.352702 |  9.801987 | 0.088996 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  1.573 | 0.494092 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.117083 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.198286 |      0.0 | 0.083111 |      0.0 | 0.010885 |  0.292282 | 0.308587 | 0.308587 | 0.085719 | 1.078908 | 0.0 | 0.0 |  75.712235 |       0.0 |   3.86741 |  8.269591 |  9.791102 | 0.085719 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  6.776 |  0.08755 |  5.854908 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.854908 |  0.839058 | 0.0 | 0.022087 | 0.839058 | 0.0 |      0.0 |      0.0 | 0.839058 | 0.209247 |      0.0 | 0.082284 |      0.0 | 0.010873 |  0.302404 | 0.298803 | 0.298803 | 0.083001 |  1.91245 | 0.0 | 0.0 |  80.705998 |       0.0 |  4.497222 |  8.187308 |  9.780229 | 0.083001 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  3.509 |   0.0539 |   3.42145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.42145 |  0.557137 | 0.0 | 0.014076 | 0.557137 | 0.0 |      0.0 |      0.0 | 0.557137 | 0.233031 |      0.0 | 0.081465 |      0.0 | 0.010861 |  0.325357 | 0.300746 | 0.300746 | 0.083541 |   1.9461 | 0.0 | 0.0 |  83.556234 |       0.0 |  4.821327 |  8.105842 |  9.769368 | 0.083541 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  5.929 | 0.041902 |    5.8751 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    5.8751 |  1.025446 | 0.0 | 0.011576 | 1.025446 | 0.0 |      0.0 |      0.0 | 1.025446 | 0.260353 |      0.0 | 0.080654 |      0.0 | 0.010849 |  0.351856 | 0.316706 | 0.316706 | 0.087974 | 1.958098 | 0.0 | 0.0 |  88.394311 |       0.0 |   5.58642 |  8.025188 |  9.758519 | 0.087974 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 | 12.826 | 0.020954 | 12.784098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.784098 |  2.497231 | 0.0 | 0.006462 |      2.0 | 0.0 |      0.0 | 0.047318 |      2.0 |  0.32163 |      0.0 | 0.079852 |      0.0 | 0.010837 |  0.459636 | 0.346221 | 0.346221 | 0.096172 | 1.979046 | 0.0 | 0.0 |  98.674717 |  0.449913 |  7.264791 |  7.945336 |  9.747682 | 0.096172 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  0.121 | 0.074152 |  0.100046 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.100046 |  0.024353 | 0.0 | 0.022883 | 0.474266 | 0.0 |      0.0 |      0.0 | 0.474266 |  0.36597 |      0.0 | 0.079057 |      0.0 | 0.010825 |  0.455852 | 0.410819 | 0.410819 | 0.114117 | 1.925848 | 0.0 | 0.0 |  98.727527 |       0.0 |  7.373087 |  7.866279 |  9.736857 | 0.114117 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.847 |  0.06896 |  0.772848 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.772848 |  0.188326 | 0.0 | 0.021402 | 0.188326 | 0.0 |      0.0 |      0.0 | 0.188326 |  0.36422 |      0.0 | 0.078271 |      0.0 | 0.010813 |  0.453304 | 0.448084 | 0.448084 | 0.124468 |  1.93104 | 0.0 | 0.0 |  99.290647 |       0.0 |  7.197193 |  7.788008 |  9.726045 | 0.124468 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   3.63 | 0.052207 |   3.56104 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.56104 |  0.877675 | 0.0 | 0.016637 | 0.877675 | 0.0 |      0.0 |      0.0 | 0.877675 | 0.372592 |      0.0 | 0.077492 |      0.0 | 0.010801 |  0.460885 | 0.454178 | 0.454178 | 0.126161 | 1.947793 | 0.0 | 0.0 | 101.957376 |       0.0 |  7.702276 |  7.710516 |  9.715244 | 0.126161 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  2.541 | 0.057919 |  2.488793 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.488793 |  0.646794 | 0.0 | 0.018787 | 0.646794 | 0.0 |      0.0 |      0.0 | 0.646794 | 0.391548 |      0.0 | 0.076721 |      0.0 | 0.010789 |  0.479058 | 0.459092 | 0.459092 | 0.127526 | 1.942081 | 0.0 | 0.0 | 103.780587 |       0.0 |  7.957522 |  7.633795 |  9.704455 | 0.127526 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 12.584 | 0.021144 | 12.526081 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.526081 |  3.372776 | 0.0 | 0.007462 |      2.0 | 0.0 |      0.0 | 0.130637 |      2.0 |  0.43727 |      0.0 | 0.075958 |      0.0 | 0.010777 |  0.654641 | 0.483207 | 0.483207 | 0.134224 | 1.978856 | 0.0 | 0.0 |  112.92643 |  1.242139 |  9.520252 |  7.557837 |  9.693679 | 0.134224 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.235 | 0.048561 |  4.213856 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.213856 |  1.343417 | 0.0 | 0.017573 |      2.0 | 0.0 |      0.0 | 0.055723 |      2.0 | 0.513485 |      0.0 | 0.075202 |      0.0 | 0.010765 |  0.655175 | 0.577967 | 0.577967 | 0.160546 | 1.951439 | 0.0 | 0.0 | 115.779297 |  0.529833 | 11.006767 |  7.482636 |  9.682914 | 0.160546 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.114 | 0.048983 |  4.065439 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.065439 |  1.362415 | 0.0 | 0.018136 | 1.892248 | 0.0 |      0.0 |      0.0 | 1.892248 | 0.583334 |      0.0 | 0.074453 |      0.0 | 0.010753 |   0.66854 | 0.641174 | 0.641174 | 0.178104 | 1.951017 | 0.0 | 0.0 | 118.464185 |       0.0 | 12.315681 |  7.408182 |  9.672161 | 0.178104 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  1.452 | 0.124728 |  1.403017 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.403017 |   0.49224 | 0.0 | 0.046529 |  0.49224 | 0.0 |      0.0 |      0.0 |  0.49224 | 0.612746 |      0.0 | 0.073713 |      0.0 | 0.010741 |    0.6972 | 0.663102 | 0.663102 | 0.184195 | 1.875272 | 0.0 | 0.0 | 119.328432 |       0.0 | 12.195174 |   7.33447 |   9.66142 | 0.184195 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.121 |  0.96204 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.358746 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.594766 |      0.0 | 0.072979 |      0.0 | 0.010729 |  0.678474 | 0.682336 | 0.682336 | 0.189538 | 1.034232 | 0.0 | 0.0 | 118.969686 |       0.0 | 11.600409 |   7.26149 |  9.650691 | 0.189538 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.0 | 1.034232 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.527265 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.565759 |      0.0 | 0.072253 |      0.0 | 0.010717 |  0.648729 | 0.681849 | 0.681849 | 0.189402 |      0.0 | 0.0 | 0.0 | 117.442421 |       0.0 |  11.03465 |  7.189237 |  9.639974 | 0.189402 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.091248 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.538166 |      0.0 | 0.071534 |      0.0 | 0.010705 |  0.620405 | 0.660696 | 0.660696 | 0.183527 |      0.0 | 0.0 | 0.0 | 116.351173 |       0.0 | 10.496484 |  7.117703 |  9.629269 | 0.183527 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.312905 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.51192 |      0.0 | 0.070822 |      0.0 | 0.010693 |  0.593435 | 0.633221 | 0.633221 | 0.175895 |      0.0 | 0.0 | 0.0 | 115.038267 |       0.0 |  9.984564 |  7.046881 |  9.618576 | 0.175895 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.399893 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.486953 |      0.0 | 0.070118 |      0.0 | 0.010681 |  0.567752 | 0.605792 | 0.605792 | 0.168276 |      0.0 | 0.0 | 0.0 | 111.638374 |       0.0 |  9.497611 |  6.976763 |  9.607894 | 0.168276 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.356 | 1.680962 |     2.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     2.356 |  0.734078 | 0.0 | 0.594957 | 0.734078 | 0.0 |      0.0 |      0.0 | 0.734078 | 0.481254 |      0.0 |  0.06942 |      0.0 |  0.01067 |  0.561343 |  0.58082 |  0.58082 | 0.161339 | 0.319038 | 0.0 | 0.0 | 112.665339 |       0.0 |  9.750436 |  6.907343 |  9.597225 | 0.161339 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  7.139 |      2.0 |  5.458038 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.458038 |  1.732037 | 0.0 | 1.873211 | 1.732037 | 0.0 |      0.0 |      0.0 | 1.732037 | 0.518123 |      0.0 | 0.068729 |      0.0 | 0.010658 |  0.597509 | 0.569083 | 0.569083 | 0.158078 |      0.0 | 0.0 | 0.0 | 114.518129 |       0.0 |  10.96435 |  6.838614 |  9.586567 | 0.158078 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  1.331 |    1.331 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.674068 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.534738 |      0.0 | 0.068045 |      0.0 | 0.010646 |  0.613429 | 0.583975 | 0.583975 | 0.162215 |      0.0 | 0.0 | 0.0 | 112.844061 |       0.0 | 10.429613 |  6.770569 |  9.575921 | 0.162215 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 25.047 | 0.132786 |    23.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    23.047 |  7.336887 | 0.0 | 0.053344 |      2.0 | 0.0 |      0.0 | 0.507872 | 1.978519 | 0.557307 | 0.019094 | 0.067463 | 0.002387 | 0.010635 |  1.143278 | 0.641527 | 0.641527 | 0.178202 | 1.867214 | 0.0 | 0.0 | 128.500829 |  4.829015 | 11.850825 |  6.722199 |  9.567673 | 0.178202 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 45.859 | 0.017578 | 45.726214 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.726214 | 18.876311 | 0.0 | 0.008533 |      2.0 | 0.0 | 0.670993 | 2.065535 | 1.907459 | 0.624873 | 0.082259 | 0.067297 | 0.010282 | 0.010631 |  3.439329 | 1.078774 | 1.078774 |  0.29966 | 1.982422 | 0.0 | 0.0 | 155.342199 | 18.968797 | 13.133411 |  6.737161 |  9.567325 |  0.29966 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  9.922 | 0.821583 |  9.904422 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.904422 |   5.97514 | 0.0 | 0.408921 |      2.0 | 0.0 | 1.158349 | 2.183404 | 1.843329 | 0.685849 | 0.139263 |  0.06773 | 0.017408 | 0.010634 |  4.105966 | 2.460315 | 2.460315 | 0.683421 | 1.178417 | 0.0 | 0.0 | 158.862561 | 19.602184 | 14.290891 |  6.808694 |  9.574098 | 0.683421 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.356 | 0.792694 |  3.534417 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.534417 |   2.22998 | 0.0 | 0.396347 |      2.0 | 0.0 |      0.0 |  1.88728 | 1.785455 | 0.740877 | 0.190706 | 0.068698 | 0.023838 | 0.010645 |    2.7075 | 3.525529 | 3.525529 | 0.979313 | 1.207306 | 0.0 | 0.0 | 159.770651 | 17.944883 |  15.33547 |  6.930702 |  9.587291 | 0.979313 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  9.075 | 0.090607 |  8.282306 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.282306 |   5.28549 | 0.0 | 0.045304 |      2.0 | 0.0 | 0.484114 | 2.020337 | 1.733226 | 0.790537 | 0.237132 | 0.070143 | 0.029642 | 0.010663 |  3.375795 | 3.275671 | 3.275671 | 0.909909 | 1.909393 | 0.0 | 0.0 | 162.722163 | 18.725922 |  16.27816 |  7.097691 |   9.60627 | 0.909909 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 22.385 | 0.007604 | 22.294393 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.294393 | 14.758053 | 0.0 | 0.003802 |      2.0 | 0.0 | 4.518592 | 2.996096 | 1.686092 | 0.835354 | 0.279029 | 0.072014 | 0.034879 | 0.010687 |  8.432743 | 3.558158 | 3.558158 | 0.988377 | 1.992396 | 0.0 | 0.0 | 170.254701 | 23.969287 | 17.128898 |  7.304707 |  9.630461 | 0.988377 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 18.634 | 0.011044 | 18.626396 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.626396 | 13.497927 | 0.0 | 0.005522 |      2.0 | 0.0 | 6.085874 | 3.375152 | 1.643555 | 0.875799 |  0.31684 | 0.074262 | 0.039605 | 0.010717 | 10.421803 | 6.368261 | 6.368261 | 1.768961 | 1.988956 | 0.0 | 0.0 | 175.377648 | 26.006188 | 17.896654 |  7.547284 |   9.65935 | 1.768961 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  7.623 | 0.032783 |  7.611956 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.611956 |  5.853084 | 0.0 | 0.016392 |      2.0 | 0.0 | 3.879321 | 2.841485 | 1.605167 | 0.912299 | 0.350962 | 0.076846 |  0.04387 | 0.010751 |  7.720702 | 8.953614 | 8.953614 | 2.487115 | 1.967217 | 0.0 | 0.0 | 177.120128 | 23.138465 | 18.589523 |  7.821401 |  9.692469 | 2.487115 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  2.299 | 0.054873 |  2.266217 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.266217 |  1.777368 | 0.0 | 0.027436 |      2.0 | 0.0 | 1.147291 |  2.18073 | 1.570524 | 0.945239 | 0.381757 | 0.079727 |  0.04772 |  0.01079 |  4.363776 | 8.422601 | 8.422601 | 2.339611 | 1.945127 | 0.0 | 0.0 | 177.581541 | 19.587812 | 19.214808 |  8.123431 |  9.729399 | 2.339611 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  5.929 | 0.038243 |  5.874127 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.874127 |  4.631045 | 0.0 | 0.019122 |      2.0 | 0.0 | 0.873052 | 2.114404 |  1.53926 | 0.974965 | 0.409547 |  0.08287 | 0.051193 | 0.010833 |  4.056125 |  5.97839 |  5.97839 | 1.660664 | 1.961757 | 0.0 | 0.0 | 178.805502 | 19.231401 | 19.779102 |  8.450108 |  9.769759 | 1.660664 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  3.267 |  0.04981 |  3.228757 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.228757 |  2.580697 | 0.0 | 0.024905 |      2.0 | 0.0 |      0.0 |  1.88537 | 1.511045 | 1.001793 | 0.434627 | 0.086246 | 0.054328 | 0.010879 |  2.984288 | 4.413537 | 4.413537 | 1.225983 |  1.95019 | 0.0 | 0.0 | 179.428656 | 17.926728 | 20.288355 |  8.798489 |  9.813208 | 1.225983 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.605 | 0.064366 |   0.55519 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.55519 |  0.446854 | 0.0 | 0.032183 |      2.0 | 0.0 |      0.0 | 1.558152 | 1.485582 | 1.026003 |  0.45726 | 0.089825 | 0.057158 | 0.010929 |   2.68491 | 3.481809 | 3.481809 | 0.967169 | 1.935634 | 0.0 | 0.0 |  179.50481 | 14.815429 | 20.747934 |  9.165924 |  9.859436 | 0.967169 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.242 | 0.067009 |  0.177634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.177634 |  0.143093 | 0.0 | 0.033505 |      2.0 | 0.0 |      0.0 | 1.233166 | 1.462603 | 1.047852 | 0.477686 | 0.093583 | 0.059711 | 0.010982 |  2.385583 | 2.886447 | 2.886447 | 0.801791 | 1.932991 | 0.0 | 0.0 | 179.505846 | 11.725356 | 21.162685 |  9.550027 |  9.908165 | 0.801791 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.605 | 0.065382 |  0.537991 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.537991 |  0.433383 | 0.0 | 0.032691 |      2.0 | 0.0 |      0.0 | 0.966732 | 1.441866 |  1.06757 | 0.496119 | 0.097497 | 0.062015 | 0.011037 |  2.142836 |  2.53203 |  2.53203 | 0.703342 | 1.934618 | 0.0 | 0.0 | 179.577763 |  9.192007 | 21.536982 |  9.948649 |  9.959143 | 0.703342 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  2.904 | 0.052751 |  2.838618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.838618 |  2.288506 | 0.0 | 0.026376 |      2.0 | 0.0 |      0.0 |  0.90219 | 1.423151 | 1.085364 | 0.512755 | 0.101546 | 0.064094 | 0.011095 |  2.100195 | 2.272317 | 2.272317 | 0.631199 | 1.947249 | 0.0 | 0.0 | 180.101499 |  8.578324 | 21.874768 | 10.359858 | 10.012142 | 0.631199 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.484 | 0.113609 |  0.431249 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.431249 |  0.349705 | 0.0 | 0.056805 |      2.0 | 0.0 |      0.0 | 0.659289 | 1.406262 | 1.101423 | 0.527767 | 0.105712 | 0.065971 | 0.011155 |  1.877579 | 2.125889 | 2.125889 | 0.590525 | 1.886391 | 0.0 | 0.0 | 180.126238 |   6.26874 | 22.179607 | 10.781913 | 10.066958 | 0.590525 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.242 | 0.353524 |  0.128391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.128391 |  0.104143 | 0.0 | 0.176762 |      2.0 | 0.0 |      0.0 | 0.416135 |  1.39102 | 1.115915 | 0.541316 | 0.109979 | 0.067664 | 0.011217 |  1.653246 | 1.962995 | 1.962995 | 0.545276 | 1.646476 | 0.0 | 0.0 | 179.973724 |  3.956748 | 22.454711 |  11.21325 | 10.123405 | 0.545276 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 | 0.903769 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.451885 |      2.0 | 0.0 |      0.0 | 0.186209 | 1.377264 | 1.128994 | 0.553543 | 0.114332 | 0.069193 |  0.01128 |  1.440816 | 1.753161 | 1.753161 | 0.486989 | 0.742707 | 0.0 | 0.0 | 179.521839 |  1.770538 | 22.702982 |  11.65246 | 10.181318 | 0.486989 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 | 0.742707 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.864415 | 1.770538 | 0.0 |      0.0 |      0.0 | 1.364851 | 1.140797 | 0.360611 | 0.117741 | 0.045076 | 0.011331 |  1.269869 | 1.540313 | 1.540313 | 0.427865 |      0.0 | 0.0 | 0.0 | 176.657424 |       0.0 | 22.927036 |  11.89533 | 10.215063 | 0.427865 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.055807 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.118165 |      0.0 | 0.118361 |      0.0 | 0.011344 |  1.247869 | 1.361418 | 1.361418 | 0.378172 |      0.0 | 0.0 | 0.0 | 173.601617 |       0.0 | 21.808871 |  11.77697 | 10.203719 | 0.378172 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     1.146 |   0.86344 | 0.0 | 1.982694 |  0.86344 | 0.0 |      0.0 |      0.0 |  0.86344 | 1.084862 |      0.0 | 0.117183 |      0.0 | 0.011331 |  1.213376 | 1.271249 | 1.271249 | 0.353125 |      0.0 | 0.0 | 0.0 | 171.901483 |       0.0 | 21.587449 | 11.659787 | 10.192388 | 0.353125 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.37986 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.052832 |      0.0 | 0.116017 |      0.0 | 0.011319 |  1.180168 | 1.229444 | 1.229444 | 0.341512 |      0.0 | 0.0 | 0.0 | 168.521623 |       0.0 | 20.534617 |  11.54377 | 10.181069 | 0.341512 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.277978 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.001485 |      0.0 | 0.114862 |      0.0 | 0.011306 |  1.127654 |  1.19386 |  1.19386 | 0.331628 |      0.0 | 0.0 | 0.0 | 165.243645 |       0.0 | 19.533132 | 11.428908 | 10.169763 | 0.331628 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.602665 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.952642 |      0.0 |  0.11372 |      0.0 | 0.011293 |  1.077655 | 1.149672 | 1.149672 | 0.319353 |      0.0 | 0.0 | 0.0 |  159.64098 |       0.0 |  18.58049 | 11.315188 |  10.15847 | 0.319353 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.727324 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.906181 |      0.0 | 0.112588 |      0.0 | 0.011281 |   1.03005 | 1.100356 | 1.100356 | 0.305654 |      0.0 | 0.0 | 0.0 | 155.913656 |       0.0 | 17.674308 |   11.2026 | 10.147189 | 0.305654 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.981965 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.861986 |      0.0 | 0.111468 |      0.0 | 0.011268 |  0.984722 | 1.051868 | 1.051868 | 0.292185 |      0.0 | 0.0 | 0.0 | 151.931691 |       0.0 | 16.812322 | 11.091133 | 10.135921 | 0.292185 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.402655 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.819947 |      0.0 | 0.110359 |      0.0 | 0.011256 |  0.941561 | 1.005514 | 1.005514 | 0.279309 |      0.0 | 0.0 | 0.0 | 149.529036 |       0.0 | 15.992376 | 10.980774 | 10.124665 | 0.279309 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.083886 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.779957 |      0.0 | 0.109261 |      0.0 | 0.011243 |  0.900461 |  0.96136 |  0.96136 | 0.267044 |      0.0 | 0.0 | 0.0 |  148.44515 |       0.0 | 15.212418 | 10.871513 | 10.113421 | 0.267044 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.196288 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.741918 |      0.0 | 0.108173 |      0.0 | 0.011231 |  0.861323 | 0.919314 | 0.919314 | 0.255365 |      0.0 | 0.0 | 0.0 | 148.248862 |       0.0 |   14.4705 |  10.76334 |  10.10219 | 0.255365 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062473 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.705735 |      0.0 | 0.107097 |      0.0 | 0.011218 |   0.82405 | 0.879276 | 0.879276 | 0.244243 |      0.0 | 0.0 | 0.0 | 148.186389 |       0.0 | 13.764765 | 10.656243 | 10.090972 | 0.244243 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061167 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.671316 |      0.0 | 0.106031 |      0.0 | 0.011206 |  0.788553 | 0.841147 | 0.841147 | 0.233652 |      0.0 | 0.0 | 0.0 | 148.125222 |       0.0 |  13.09345 | 10.550212 | 10.079766 | 0.233652 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060106 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.638575 |      0.0 | 0.104976 |      0.0 | 0.011194 |  0.754745 | 0.804835 | 0.804835 | 0.223565 |      0.0 | 0.0 | 0.0 | 148.065116 |       0.0 | 12.454875 | 10.445235 | 10.068572 | 0.223565 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.059296 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.607431 |      0.0 | 0.103932 |      0.0 | 0.011181 |  0.722544 | 0.770252 | 0.770252 | 0.213959 |      0.0 | 0.0 | 0.0 |  148.00582 |       0.0 | 11.847443 | 10.341303 | 10.057391 | 0.213959 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.058613 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.577807 |      0.0 | 0.102898 |      0.0 | 0.011169 |  0.691873 | 0.737314 | 0.737314 | 0.204809 |      0.0 | 0.0 | 0.0 | 147.947208 |       0.0 | 11.269637 | 10.238406 | 10.046223 | 0.204809 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060164 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.549627 |      0.0 | 0.101874 |      0.0 | 0.011156 |  0.662657 | 0.705941 | 0.705941 | 0.196095 |      0.0 | 0.0 | 0.0 | 147.887044 |       0.0 |  10.72001 | 10.136532 | 10.035066 | 0.196095 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060398 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.522821 |      0.0 |  0.10086 |      0.0 | 0.011144 |  0.634825 | 0.676057 | 0.676057 | 0.187794 |      0.0 | 0.0 | 0.0 | 147.826646 |       0.0 | 10.197189 | 10.035672 | 10.023923 | 0.187794 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  1.573 | 0.054992 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.025404 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.497323 |      0.0 | 0.099857 |      0.0 | 0.011132 |  0.608311 |  0.64759 |  0.64759 | 0.179886 | 1.518008 | 0.0 | 0.0 | 147.801242 |       0.0 |  9.699866 |  9.935815 | 10.012791 | 0.179886 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 | 0.064218 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.029661 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.473068 |      0.0 | 0.098863 |      0.0 | 0.011119 |   0.58305 | 0.620471 | 0.620471 | 0.172353 |  1.45379 | 0.0 | 0.0 | 147.771581 |       0.0 |  9.226798 |  9.836952 | 10.001672 | 0.172353 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 | 0.065345 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.030176 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.449996 |      0.0 | 0.097879 |      0.0 | 0.011107 |  0.558982 | 0.594635 | 0.594635 | 0.165176 | 1.388445 | 0.0 | 0.0 | 147.741405 |       0.0 |  8.776802 |  9.739073 |  9.990565 | 0.165176 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 | 0.238876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.110287 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.42805 |      0.0 | 0.096905 |      0.0 | 0.011094 |   0.53605 |  0.57002 |  0.57002 | 0.158339 | 1.149568 | 0.0 | 0.0 | 147.631118 |       0.0 |  8.348752 |  9.642167 |  9.979471 | 0.158339 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.847 | 0.984764 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.454318 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.407173 |      0.0 | 0.095941 |      0.0 | 0.011082 |  0.514197 | 0.546566 | 0.546566 | 0.151824 | 1.011805 | 0.0 | 0.0 |   147.1768 |       0.0 |  7.941579 |  9.546226 |  9.968389 | 0.151824 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.484 | 1.495805 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.502702 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.387315 |      0.0 | 0.094987 |      0.0 |  0.01107 |  0.493372 | 0.524218 | 0.524218 | 0.145616 |      0.0 | 0.0 | 0.0 | 145.674097 |       0.0 |  7.554263 |   9.45124 |  9.957319 | 0.145616 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.595867 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.368426 |      0.0 | 0.094041 |      0.0 | 0.011058 |  0.473525 | 0.502921 | 0.502921 |   0.1397 |      0.0 | 0.0 | 0.0 |  142.07823 |       0.0 |  7.185838 |  9.357198 |  9.946261 |   0.1397 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.66112 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.350457 |      0.0 | 0.093106 |      0.0 | 0.011045 |  0.454608 | 0.482626 | 0.482626 | 0.134063 |      0.0 | 0.0 | 0.0 | 137.417111 |       0.0 |   6.83538 |  9.264093 |  9.935216 | 0.134063 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.549421 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.333365 |      0.0 | 0.092179 |      0.0 | 0.011033 |  0.436578 | 0.463282 | 0.463282 |  0.12869 |      0.0 | 0.0 | 0.0 |  131.86769 |       0.0 |  6.502015 |  9.171913 |  9.924183 |  0.12869 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.325221 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.317107 |      0.0 | 0.091262 |      0.0 | 0.011021 |   0.41939 | 0.444845 | 0.444845 | 0.123568 |      0.0 | 0.0 | 0.0 | 126.542469 |       0.0 |  6.184908 |  9.080651 |  9.913162 | 0.123568 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.525426 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.301642 |      0.0 | 0.090354 |      0.0 | 0.011009 |  0.403004 | 0.427271 | 0.427271 | 0.118686 |      0.0 | 0.0 | 0.0 | 120.017043 |       0.0 |  5.883266 |  8.990297 |  9.902154 | 0.118686 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  5.41968 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.28693 |      0.0 | 0.089455 |      0.0 | 0.010996 |  0.387382 | 0.410517 | 0.410517 | 0.114032 |      0.0 | 0.0 | 0.0 | 114.597363 |       0.0 |  5.596336 |  8.900842 |  9.891157 | 0.114032 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.335845 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.272937 |      0.0 | 0.088565 |      0.0 | 0.010984 |  0.372485 | 0.394544 | 0.394544 | 0.109596 |      0.0 | 0.0 | 0.0 | 110.261519 |       0.0 |  5.323399 |  8.812277 |  9.880173 | 0.109596 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.342105 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.259625 |      0.0 | 0.087684 |      0.0 | 0.010972 |  0.358281 | 0.379315 | 0.379315 | 0.105365 |      0.0 | 0.0 | 0.0 | 106.919414 |       0.0 |  5.063774 |  8.724594 |  9.869201 | 0.105365 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.517107 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.246963 |      0.0 | 0.086811 |      0.0 |  0.01096 |  0.344734 | 0.364793 | 0.364793 | 0.101331 |      0.0 | 0.0 | 0.0 | 105.402307 |       0.0 |  4.816811 |  8.637783 |  9.858242 | 0.101331 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.664106 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.234919 |      0.0 | 0.085947 |      0.0 | 0.010948 |  0.331814 | 0.350944 | 0.350944 | 0.097485 |      0.0 | 0.0 | 0.0 | 104.738201 |       0.0 |  4.581892 |  8.551835 |  9.847294 | 0.097485 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.166111 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.223462 |      0.0 | 0.085092 |      0.0 | 0.010935 |  0.319489 | 0.337737 | 0.337737 | 0.093816 |      0.0 | 0.0 | 0.0 | 104.572091 |       0.0 |  4.358431 |  8.466743 |  9.836359 | 0.093816 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.045045 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.212563 |      0.0 | 0.084246 |      0.0 | 0.010923 |  0.307732 | 0.325139 | 0.325139 | 0.090316 |      0.0 | 0.0 | 0.0 | 104.527046 |       0.0 |  4.145868 |  8.382498 |  9.825436 | 0.090316 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043146 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.202196 |      0.0 | 0.083407 |      0.0 | 0.010911 |  0.296515 | 0.313121 | 0.313121 | 0.086978 |      0.0 | 0.0 | 0.0 |   104.4839 |       0.0 |  3.943671 |   8.29909 |  9.814525 | 0.086978 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042489 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.192335 |      0.0 | 0.082577 |      0.0 | 0.010899 |  0.285811 | 0.301656 | 0.301656 | 0.083793 |      0.0 | 0.0 | 0.0 | 104.441411 |       0.0 |  3.751336 |  8.216513 |  9.803626 | 0.083793 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042655 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.182955 |      0.0 | 0.081756 |      0.0 | 0.010887 |  0.275597 | 0.290717 | 0.290717 | 0.080755 |      0.0 | 0.0 | 0.0 | 104.398756 |       0.0 |  3.568381 |  8.134757 |  9.792739 | 0.080755 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96p_ilake:

internal lake
_____________

Like in the :ref:`hland_96_ilake` example of |hland_96|, we now focus on |ILAKE| zones.
We understand the combination of the slow-response groundwater reservoirs |SG2| and
|SG3| as the surrogate for HBV96's lower zone storage (|LZ|).  Hence, we add
precipitation falling on "internal lakes" to both storages and remove lake evaporation
from both storages.  As for percolation, parameter |FSG| determines how |hland_96p|
divides precipitation and evaporation between both storages:

.. integration-test::

    >>> zonetype(ILAKE)
    >>> model.aetmodel.parameters.control.temperaturethresholdice(13.0)
    >>> test('hland_96p_ilake')
    |        date |    p |    t |   tc | fracrain | rfc | sfc | cfact | swe | gact |     pc |  ei |     tf | spl | wcl | spg | wcg | glmelt | melt | refr |    in_ |      r |  sr |  ea |  dp |       el |  rs |  ri | gr1 | rg1 |       gr2 |      rg2 |      gr3 |      rg3 |     inrc |    outrc |       rt |       qt |  ic |  sp |  wc |  sm | suz | sg1 |        sg2 |       sg3 |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.077544 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.099502 |      0.0 | 0.011105 | 0.110607 | 0.168002 | 0.168002 | 0.046667 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    9.83157 |  9.980279 | 0.046667 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.075307 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.097826 |      0.0 | 0.011083 | 0.108909 | 0.136934 | 0.136934 | 0.038037 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.666805 |  9.960829 | 0.038037 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.074675 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.096186 |      0.0 | 0.011061 | 0.107248 | 0.114636 | 0.114636 | 0.031843 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    9.50424 |   9.94147 | 0.031843 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.073912 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.094569 |      0.0 |  0.01104 | 0.105609 | 0.108582 | 0.108582 | 0.030162 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.343972 |  9.922217 | 0.030162 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.074675 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.092974 |      0.0 | 0.011019 | 0.103993 | 0.106402 | 0.106402 | 0.029556 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    9.18462 |  9.902902 | 0.029556 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.079126 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.091388 |      0.0 | 0.010997 | 0.102386 | 0.104727 | 0.104727 | 0.029091 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.022897 |  9.883113 | 0.029091 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.224769 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.089779 |      0.0 | 0.010975 | 0.100754 | 0.103108 | 0.103108 | 0.028641 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.733323 |  9.847163 | 0.028641 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 1.488314 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.086898 |      0.0 | 0.010935 | 0.097833 | 0.101394 | 0.101394 | 0.028165 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   7.323479 |   9.67086 | 0.028165 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 3.364543 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.07287 |      0.0 | 0.010739 | 0.083609 |  0.09822 |  0.09822 | 0.027283 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   4.259905 |  9.286282 | 0.027283 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 5.634518 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.042387 |      0.0 | 0.010312 | 0.052699 | 0.087749 | 0.087749 | 0.024375 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -0.790942 |  8.649912 | 0.024375 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 6.363559 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.009606 | 0.009606 | 0.064058 | 0.064058 | 0.017794 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -6.447439 |  7.933245 | 0.017794 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 7.214798 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.00881 |  0.00881 |  0.03069 |  0.03069 | 0.008525 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -12.860593 |  7.122791 | 0.008525 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 3.947157 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.00791 |  0.00791 | 0.012968 | 0.012968 | 0.003602 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -16.369177 |  6.676308 | 0.003602 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 5.101273 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.007414 | 0.007414 | 0.008771 | 0.008771 | 0.002436 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -20.903642 |  6.102086 | 0.002436 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 5.696167 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.006776 | 0.006776 | 0.007702 | 0.007702 | 0.002139 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -25.966901 |  5.462402 | 0.002139 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.242 |  0.242 | 0.0 | 0.0 | 0.0 | 1.797434 | 0.0 | 0.0 | 0.0 | 0.0 |  0.215111 |      0.0 | 0.026889 | 0.006081 | 0.006081 | 0.007053 | 0.007053 | 0.001959 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -27.349509 |  5.283495 | 0.001959 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 1.408972 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.005867 | 0.005867 | 0.006422 | 0.006422 | 0.001784 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -28.601929 |  5.121075 | 0.001784 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.875158 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.005687 | 0.005687 |  0.00601 |  0.00601 | 0.001669 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -29.379847 |  5.018149 | 0.001669 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.573 |  1.573 | 0.0 | 0.0 | 0.0 | 0.494092 | 0.0 | 0.0 | 0.0 | 0.0 |  1.398222 |      0.0 | 0.174778 |  0.00567 |  0.00567 | 0.005788 | 0.005788 | 0.001608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -28.420818 |  5.132358 | 0.001608 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  6.776 | 0.0 |  6.776 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  6.776 |  6.776 | 0.0 | 0.0 | 0.0 |  0.08755 | 0.0 | 0.0 | 0.0 | 0.0 |  6.023111 |      0.0 | 0.752889 | 0.006118 | 0.006118 | 0.005727 | 0.005727 | 0.001591 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -22.475529 |  5.869401 | 0.001591 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  3.509 | 0.0 |  3.509 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.509 |  3.509 | 0.0 | 0.0 | 0.0 |   0.0539 | 0.0 | 0.0 | 0.0 | 0.0 |  3.119111 |      0.0 | 0.389889 | 0.006734 | 0.006734 | 0.005973 | 0.005973 | 0.001659 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -19.40433 |  6.246567 | 0.001659 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  5.929 | 0.0 |  5.929 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.929 |  5.929 | 0.0 | 0.0 | 0.0 | 0.041902 | 0.0 | 0.0 | 0.0 | 0.0 |  5.270222 |      0.0 | 0.658778 | 0.007303 | 0.007303 | 0.006474 | 0.006474 | 0.001798 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -14.171353 |  6.893386 | 0.001798 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 12.826 | 0.0 | 12.826 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.826 | 12.826 | 0.0 | 0.0 | 0.0 | 0.020954 | 0.0 | 0.0 | 0.0 | 0.0 | 11.400889 |      0.0 | 1.425111 | 0.008446 | 0.008446 | 0.007086 | 0.007086 | 0.001968 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   -2.78909 |  8.307722 | 0.001968 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 | 0.0 | 0.074152 | 0.0 | 0.0 | 0.0 | 0.0 |  0.107556 |      0.0 | 0.013444 | 0.009233 | 0.009233 |  0.00796 |  0.00796 | 0.002211 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -2.747447 |  8.303695 | 0.002211 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.847 |  0.847 | 0.0 | 0.0 | 0.0 |  0.06896 | 0.0 | 0.0 | 0.0 | 0.0 |  0.752889 |      0.0 | 0.094111 | 0.009273 | 0.009273 | 0.008799 | 0.008799 | 0.002444 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -2.055856 |   8.38087 | 0.002444 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |   3.63 | 0.0 |   3.63 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.63 |   3.63 | 0.0 | 0.0 | 0.0 | 0.052207 | 0.0 | 0.0 | 0.0 | 0.0 |  3.226667 | 0.003722 | 0.403333 | 0.009531 | 0.013253 | 0.009463 | 0.009463 | 0.002629 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   1.120682 |  8.768872 | 0.002629 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  2.541 | 0.0 |  2.541 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.541 |  2.541 | 0.0 | 0.0 | 0.0 | 0.057919 | 0.0 | 0.0 | 0.0 | 0.0 |  2.258667 | 0.022407 | 0.282333 | 0.009895 | 0.032301 | 0.012898 | 0.012898 | 0.003583 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   3.305458 |  9.034875 | 0.003583 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 12.584 | 0.0 | 12.584 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.584 | 12.584 | 0.0 | 0.0 | 0.0 | 0.021144 | 0.0 | 0.0 | 0.0 | 0.0 | 11.185778 | 0.088633 | 1.398222 |  0.01081 | 0.099442 | 0.028612 | 0.028612 | 0.007948 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.383809 | 10.419938 | 0.007948 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.235 | 0.0 |  4.235 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.235 |  4.235 | 0.0 | 0.0 | 0.0 | 0.048561 | 0.0 | 0.0 | 0.0 | 0.0 |  3.764444 | 0.161881 | 0.470556 | 0.011833 | 0.173714 | 0.074396 | 0.074396 | 0.020666 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  17.943207 | 10.873265 | 0.020666 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.114 | 0.0 |  4.114 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.114 |  4.114 | 0.0 | 0.0 | 0.0 | 0.048983 | 0.0 | 0.0 | 0.0 | 0.0 |  3.656889 | 0.196762 | 0.457111 | 0.012329 |  0.20909 | 0.138707 | 0.138707 |  0.03853 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  21.359794 | 11.312605 |  0.03853 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.452 | 0.0 |  1.452 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.452 |  1.452 | 0.0 | 0.0 | 0.0 | 0.124728 | 0.0 | 0.0 | 0.0 | 0.0 |  1.290667 | 0.218965 | 0.161333 | 0.012652 | 0.231618 | 0.188739 | 0.188739 | 0.052428 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  22.320626 | 11.447428 | 0.052428 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 | 0.0 |  0.96204 | 0.0 | 0.0 | 0.0 | 0.0 |  0.107556 |  0.22263 | 0.013444 |  0.01272 |  0.23535 | 0.218534 | 0.218534 | 0.060704 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  21.350405 | 11.341259 | 0.060704 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 2.345064 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.21244 |      0.0 | 0.012594 | 0.225034 | 0.230739 | 0.230739 | 0.064094 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  19.053463 | 11.068102 | 0.064094 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 1.486684 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.189585 |      0.0 | 0.012291 | 0.201876 | 0.227233 | 0.227233 |  0.06312 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  17.542381 | 10.890624 |  0.06312 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 | 0.0 | 1.805438 | 0.0 | 0.0 | 0.0 | 0.0 |  0.430222 | 0.176694 | 0.053778 | 0.012124 | 0.188817 |  0.21165 |  0.21165 | 0.058792 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  16.191076 | 10.731674 | 0.058792 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 | 0.0 | 4.728713 | 0.0 | 0.0 | 0.0 | 0.0 |  0.107556 |  0.16164 | 0.013444 | 0.011925 | 0.173565 |  0.19533 |  0.19533 | 0.054258 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  11.933691 | 10.207781 | 0.054258 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.356 | 0.0 |  4.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.356 |  4.356 | 0.0 | 0.0 | 0.0 | 1.680962 | 0.0 | 0.0 | 0.0 | 0.0 |     3.872 | 0.138038 |    0.484 | 0.011604 | 0.149642 | 0.179699 | 0.179699 | 0.049916 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.173465 | 10.493403 | 0.049916 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  7.139 | 0.0 |  7.139 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.139 |  7.139 | 0.0 | 0.0 | 0.0 | 3.326542 | 0.0 | 0.0 | 0.0 | 0.0 |  6.345778 | 0.172652 | 0.793222 | 0.012093 | 0.184745 | 0.163795 | 0.163795 | 0.045499 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  17.389665 | 10.904916 | 0.045499 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.331 | 0.0 |  1.331 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.331 |  1.331 | 0.0 | 0.0 | 0.0 | 2.727186 | 0.0 | 0.0 | 0.0 | 0.0 |  1.183111 | 0.178926 | 0.147889 | 0.012192 | 0.191118 | 0.172321 | 0.172321 | 0.047867 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  15.969685 | 10.737592 | 0.047867 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 25.047 | 0.0 | 25.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 25.047 | 25.047 | 0.0 | 0.0 | 0.0 | 0.132786 | 0.0 | 0.0 | 0.0 | 0.0 |    22.264 | 0.269851 |    2.783 |  0.01347 |  0.28332 | 0.192311 | 0.192311 |  0.05342 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  37.845802 | 13.492368 |  0.05342 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 45.859 | 0.0 | 45.859 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.859 | 45.859 | 0.0 | 0.0 | 0.0 | 0.017578 | 0.0 | 0.0 | 0.0 | 0.0 | 40.763556 | 0.579712 | 5.095444 | 0.017813 | 0.597525 | 0.265446 | 0.265446 | 0.073735 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  78.014021 | 18.568047 | 0.073735 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  9.922 | 0.0 |  9.922 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.922 |  9.922 | 0.0 | 0.0 | 0.0 | 0.821583 | 0.0 | 0.0 | 0.0 | 0.0 |  8.819556 | 0.820204 | 1.102444 | 0.021232 | 0.841436 | 0.472753 | 0.472753 |  0.13132 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  85.283077 | 19.557972 |  0.13132 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.356 | 0.0 |  4.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.356 |  4.356 | 0.0 | 0.0 | 0.0 | 0.792694 | 0.0 | 0.0 | 0.0 | 0.0 |     3.872 | 0.867876 |    0.484 | 0.021988 | 0.889864 | 0.713589 | 0.713589 | 0.198219 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  87.582583 | 19.931907 | 0.198219 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  9.075 | 0.0 |  9.075 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.075 |  9.075 | 0.0 | 0.0 | 0.0 | 0.090607 | 0.0 | 0.0 | 0.0 | 0.0 |  8.066667 |  0.91166 | 1.008333 | 0.022694 | 0.934355 | 0.848983 | 0.848983 | 0.235829 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   94.65705 | 20.907479 | 0.235829 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 22.385 | 0.0 | 22.385 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.385 | 22.385 | 0.0 | 0.0 | 0.0 | 0.007604 | 0.0 | 0.0 | 0.0 | 0.0 | 19.897778 | 1.041011 | 2.487222 | 0.024599 |  1.06561 | 0.918032 | 0.918032 | 0.255009 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 113.507057 | 23.369257 | 0.255009 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 18.634 | 0.0 | 18.634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.634 | 18.634 | 0.0 | 0.0 | 0.0 | 0.011044 | 0.0 | 0.0 | 0.0 | 0.0 | 16.563556 | 1.211956 | 2.070444 | 0.027101 | 1.239058 | 1.017441 | 1.017441 | 0.282622 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 128.848839 | 25.411374 | 0.282622 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  7.623 | 0.0 |  7.623 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.623 |  7.623 | 0.0 | 0.0 | 0.0 | 0.032783 | 0.0 | 0.0 | 0.0 | 0.0 |     6.776 | 1.315835 |    0.847 |  0.02869 | 1.344524 | 1.161002 | 1.161002 | 0.322501 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 134.279864 | 26.226041 | 0.322501 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  2.299 | 0.0 |  2.299 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.299 |  2.299 | 0.0 | 0.0 | 0.0 | 0.054873 | 0.0 | 0.0 | 0.0 | 0.0 |  2.043556 | 1.346291 | 0.255444 | 0.029266 | 1.375556 | 1.285925 | 1.285925 | 0.357201 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 134.928353 | 26.446123 | 0.357201 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  5.929 | 0.0 |  5.929 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.929 |  5.929 | 0.0 | 0.0 | 0.0 | 0.038243 | 0.0 | 0.0 | 0.0 | 0.0 |  5.270222 | 1.368823 | 0.658778 | 0.029734 | 1.398557 | 1.353384 | 1.353384 |  0.37594 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 138.795758 | 27.070918 |  0.37594 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  3.267 | 0.0 |  3.267 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.267 |  3.267 | 0.0 | 0.0 | 0.0 |  0.04981 | 0.0 | 0.0 | 0.0 | 0.0 |     2.904 | 1.395513 |    0.363 | 0.030264 | 1.425776 | 1.386907 | 1.386907 | 0.385252 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  140.25997 | 27.398119 | 0.385252 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.605 |  0.605 | 0.0 | 0.0 | 0.0 | 0.064366 | 0.0 | 0.0 | 0.0 | 0.0 |  0.537778 |  1.39829 | 0.067222 | 0.030463 | 1.428753 | 1.412033 | 1.412033 | 0.392231 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 139.342244 | 27.427727 | 0.392231 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.242 |  0.242 | 0.0 | 0.0 | 0.0 | 0.067009 | 0.0 | 0.0 | 0.0 | 0.0 |  0.215111 |  1.38755 | 0.026889 | 0.030473 | 1.418024 | 1.424201 | 1.424201 | 0.395612 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  138.11024 | 27.416697 | 0.395612 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.605 |  0.605 | 0.0 | 0.0 | 0.0 | 0.065382 | 0.0 | 0.0 | 0.0 | 0.0 |  0.537778 |   1.3769 | 0.067222 | 0.030483 | 1.407383 | 1.421323 | 1.421323 | 0.394812 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 137.213001 | 27.446171 | 0.394812 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  2.904 | 0.0 |  2.904 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.904 |  2.904 | 0.0 | 0.0 | 0.0 | 0.052751 | 0.0 | 0.0 | 0.0 | 0.0 |  2.581333 | 1.378156 | 0.322667 | 0.030658 | 1.408814 | 1.412874 | 1.412874 | 0.392465 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 138.369289 | 27.732319 | 0.392465 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 | 0.0 | 0.113609 | 0.0 | 0.0 | 0.0 | 0.0 |  0.430222 | 1.378941 | 0.053778 | 0.030826 | 1.409768 | 1.409251 | 1.409251 | 0.391459 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 137.319584 | 27.742647 | 0.391459 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.242 |  0.242 | 0.0 | 0.0 | 0.0 | 0.353524 | 0.0 | 0.0 | 0.0 | 0.0 |  0.215111 | 1.367425 | 0.026889 | 0.030823 | 1.398248 | 1.408531 | 1.408531 | 0.391259 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 135.853026 | 27.699432 | 0.391259 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.903769 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  1.35176 |      0.0 |  0.03076 |  1.38252 | 1.401972 | 1.401972 | 0.389437 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 133.697916 | 27.568254 | 0.389437 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 3.235768 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 1.330316 |      0.0 | 0.030614 | 1.360931 | 1.388654 | 1.388654 | 0.385737 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 129.491361 | 27.178109 | 0.385737 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.363 |  0.363 | 0.0 | 0.0 | 0.0 | 3.237307 | 0.0 | 0.0 | 0.0 | 0.0 |  0.322667 | 1.290069 | 0.040333 | 0.030204 | 1.320272 | 1.368681 | 1.368681 | 0.380189 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 125.646353 | 26.828538 | 0.380189 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  3.146 | 0.0 |  3.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.146 |  3.146 | 0.0 | 0.0 | 0.0 | 2.982694 | 0.0 | 0.0 | 0.0 | 0.0 |  2.796444 | 1.264138 | 0.349556 | 0.029987 | 1.294125 | 1.337755 | 1.337755 | 0.371599 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 124.527376 | 26.816696 | 0.371599 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.847 |  0.847 | 0.0 | 0.0 | 0.0 |  3.80336 | 0.0 | 0.0 | 0.0 | 0.0 |  0.752889 |  1.24282 | 0.094111 | 0.029832 | 1.272652 | 1.307373 | 1.307373 | 0.363159 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  120.65668 |  26.45838 | 0.363159 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.363 |  0.363 | 0.0 | 0.0 | 0.0 | 3.459478 | 0.0 | 0.0 | 0.0 | 0.0 |  0.322667 | 1.202162 | 0.040333 | 0.029404 | 1.231566 | 1.281488 | 1.281488 | 0.355969 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 116.702093 | 26.084923 | 0.355969 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.363 |  0.363 | 0.0 | 0.0 | 0.0 | 5.784165 | 0.0 | 0.0 | 0.0 | 0.0 |  0.322667 | 1.162813 | 0.040333 |  0.02899 | 1.191803 | 1.248378 | 1.248378 | 0.346772 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.720467 | 25.453581 | 0.346772 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 3.735707 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 1.101687 |      0.0 | 0.028266 | 1.129953 |  1.20804 |  1.20804 | 0.335567 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 106.298151 | 25.010237 | 0.335567 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 4.086328 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 1.057684 |      0.0 | 0.027774 | 1.085458 | 1.156956 | 1.156956 | 0.321377 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 101.608175 | 24.528427 | 0.321377 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 2.530248 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 1.011018 |      0.0 | 0.027239 | 1.038257 | 1.106675 | 1.106675 |  0.30741 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  98.348048 | 24.220049 |  0.30741 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 1.159786 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.978579 |      0.0 | 0.026896 | 1.005476 | 1.060438 | 1.060438 | 0.294566 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  96.338547 | 24.064288 | 0.294566 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.211567 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.958585 |      0.0 | 0.026723 | 0.985308 | 1.022461 | 1.022461 | 0.284017 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  95.191903 | 24.014057 | 0.284017 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.067425 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.947175 |      0.0 | 0.026667 | 0.973843 | 0.996335 | 0.996335 |  0.27676 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  94.184795 | 23.979898 |  0.27676 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.937154 |      0.0 |  0.02663 | 0.963784 | 0.980063 | 0.980063 |  0.27224 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   93.24764 | 23.953269 |  0.27224 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.92783 |      0.0 |   0.0266 | 0.954429 | 0.968597 | 0.968597 | 0.269055 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  92.319811 | 23.926669 | 0.269055 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.918597 |      0.0 |  0.02657 | 0.945168 | 0.958735 | 0.958735 | 0.266315 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  91.401213 | 23.900098 | 0.266315 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.909457 |      0.0 | 0.026541 | 0.935998 |  0.94936 |  0.94936 | 0.263711 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  90.491756 | 23.873557 | 0.263711 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.900408 |      0.0 | 0.026511 | 0.926919 | 0.940141 | 0.940141 |  0.26115 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  89.591348 | 23.847046 |  0.26115 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.891449 |      0.0 | 0.026482 | 0.917931 | 0.931021 | 0.931021 | 0.258617 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  88.699899 | 23.820564 | 0.258617 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.573 |  1.573 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.398222 | 0.889547 | 0.174778 |  0.02655 | 0.916096 | 0.922496 | 0.922496 | 0.256249 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  89.208575 | 23.968792 | 0.256249 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.88764 |      0.0 | 0.026617 | 0.914257 | 0.917591 | 0.917591 | 0.254886 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  88.320935 | 23.942175 | 0.254886 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.878808 |      0.0 | 0.026588 | 0.905396 | 0.914659 | 0.914659 | 0.254072 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  87.442127 | 23.915587 | 0.254072 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.870064 |      0.0 | 0.026558 | 0.896622 |  0.90874 |  0.90874 | 0.252428 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  86.572063 | 23.889029 | 0.252428 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.847 |  0.847 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.752889 | 0.865158 | 0.094111 | 0.026581 | 0.891739 | 0.900782 | 0.900782 | 0.250217 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  86.459794 | 23.956559 | 0.250217 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.430222 | 0.862433 | 0.053778 | 0.026633 | 0.889067 | 0.894452 | 0.894452 | 0.248459 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  86.027583 | 23.983703 | 0.248459 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.107556 | 0.856525 | 0.013444 | 0.026641 | 0.883166 | 0.890286 | 0.890286 | 0.247302 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  85.278613 | 23.970507 | 0.247302 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 | 0.0 | 5.249074 | 0.0 | 0.0 | 0.0 | 0.0 |  0.430222 |  0.85068 | 0.053778 | 0.026649 | 0.877329 | 0.885551 | 0.885551 | 0.245986 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  80.192312 | 23.414405 | 0.245986 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 6.461403 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.797927 |      0.0 | 0.026002 | 0.823928 | 0.876535 | 0.876535 | 0.243482 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  73.650916 |  22.67047 | 0.243482 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  6.46129 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.732839 |      0.0 | 0.025175 | 0.758014 | 0.842618 | 0.842618 | 0.234061 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  67.174708 | 21.927373 | 0.234061 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 8.250733 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |   0.6684 |      0.0 |  0.02435 |  0.69275 | 0.786047 | 0.786047 | 0.218346 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  59.172323 | 20.986275 | 0.218346 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 7.225213 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.588774 |      0.0 | 0.023305 |  0.61208 | 0.720903 | 0.720903 | 0.200251 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.161137 | 20.160168 | 0.200251 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 6.053675 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.519012 |      0.0 | 0.022388 |   0.5414 | 0.647647 | 0.647647 | 0.179902 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  46.261081 |  19.46515 | 0.179902 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 4.849713 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.460305 |      0.0 | 0.021616 | 0.481921 | 0.574775 | 0.574775 |  0.15966 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  41.489919 | 18.904677 |  0.15966 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |  2.27028 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.412832 |      0.0 | 0.020994 | 0.433825 | 0.510638 | 0.510638 | 0.141844 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  39.059061 |  18.63143 | 0.141844 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 1.008109 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.388644 |      0.0 |  0.02069 | 0.409334 | 0.458344 | 0.458344 | 0.127318 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   37.77432 | 18.498728 | 0.127318 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.253754 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.375861 |      0.0 | 0.020543 | 0.396403 | 0.423518 | 0.423518 | 0.117644 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  37.172901 |  18.44999 | 0.117644 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.068921 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.369877 |      0.0 | 0.020489 | 0.390365 | 0.404057 | 0.404057 | 0.112238 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  36.741761 | 18.421844 | 0.112238 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.365587 |      0.0 | 0.020457 | 0.386044 | 0.393995 | 0.393995 | 0.109443 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  36.376174 | 18.401387 | 0.109443 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.361949 |      0.0 | 0.020435 | 0.382384 | 0.388282 | 0.388282 | 0.107856 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  36.014225 | 18.380952 | 0.107856 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 | 0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 0.358348 |      0.0 | 0.020412 | 0.378759 | 0.384121 | 0.384121 |   0.1067 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  35.655878 |  18.36054 |   0.1067 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96p_snow_classes:

snow classes
____________

Regarding snow simulation, all functionalities of |hland_96p| are identical with the
ones of |hland_96|.  Hence, all snow data in the following table agrees with the
:ref:`hland_96_snow_classes` example of |hland_96|:

.. integration-test::

    >>> sclass(2)
    >>> sfdist(linear=0.2)
    >>> zonetype(FIELD)
    >>> t_series = inputs.t.series.copy()
    >>> tn_series = petinputs.normalairtemperature.series.copy()
    >>> inputs.t.series[:48] = -20.0
    >>> inputs.t.series[48:] = 20.0
    >>> petinputs.normalairtemperature.series = inputs.t.series
    >>> test("hland_96p_snow_classes")
    |        date |    p |     t |    tc | fracrain | rfc | sfc |    cfact |                    swe | gact |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt |               melt |      refr |       in_ |        r |  sr |       ea |       dp |  el |       rs |       ri |      gr1 |      rg1 |      gr2 |      rg2 |      gr3 |      rg3 |     inrc |    outrc |       rt |       qt |       ic |                     sp |                   wc |         sm |       suz |       sg1 |       sg2 |       sg3 |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.044059 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.487706 |      0.0 | 0.099502 |      0.0 | 0.011105 | 0.598312 | 0.202814 | 0.202814 | 0.056337 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.955941 |       0.0 |  9.512294 |  9.900498 |  9.988895 | 0.056337 |
    | 01/01 01:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.042769 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.46392 |      0.0 | 0.098512 |      0.0 | 0.011093 | 0.573524 | 0.414293 | 0.414293 | 0.115081 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.913172 |       0.0 |  9.048374 |  9.801987 |  9.977802 | 0.115081 |
    | 01/01 02:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.042392 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.441294 |      0.0 | 0.097531 |      0.0 |  0.01108 | 0.549906 | 0.547167 | 0.547167 | 0.151991 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.870779 |       0.0 |   8.60708 |  9.704455 |  9.966722 | 0.151991 |
    | 01/01 03:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.041941 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.419772 |      0.0 | 0.096561 |      0.0 | 0.011068 | 0.527401 | 0.556325 | 0.556325 | 0.154535 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.828838 |       0.0 |  8.187308 |  9.607894 |  9.955654 | 0.154535 |
    | 01/01 04:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.042357 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |   0.3993 |      0.0 |   0.0956 |      0.0 | 0.011056 | 0.505956 | 0.537334 | 0.537334 |  0.14926 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.786481 |       0.0 |  7.788008 |  9.512294 |  9.944598 |  0.14926 |
    | 01/01 05:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.044862 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.379826 |      0.0 | 0.094649 |      0.0 | 0.011043 | 0.485518 | 0.515762 | 0.515762 | 0.143267 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.741619 |       0.0 |  7.408182 |  9.417645 |  9.933555 | 0.143267 |
    | 01/01 06:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  0.12738 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.361301 |      0.0 | 0.093707 |      0.0 | 0.011031 |  0.46604 | 0.494888 | 0.494888 | 0.137469 |      0.0 |        0.0         0.0 |       0.0        0.0 |   99.61424 |       0.0 |  7.046881 |  9.323938 |  9.922524 | 0.137469 |
    | 01/01 07:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.842371 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.34368 |      0.0 | 0.092775 |      0.0 | 0.011019 | 0.447474 | 0.474971 | 0.474971 | 0.131936 |      0.0 |        0.0         0.0 |       0.0        0.0 |  98.771869 |       0.0 |    6.7032 |  9.231163 |  9.911505 | 0.131936 |
    | 01/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.888194 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.326919 |      0.0 | 0.091852 |      0.0 | 0.011007 | 0.429777 | 0.455987 | 0.455987 | 0.126663 |      0.0 |        0.0         0.0 |       0.0        0.0 |  96.883675 |       0.0 |  6.376282 |  9.139312 |  9.900498 | 0.126663 |
    | 01/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.101664 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.310975 |      0.0 | 0.090938 |      0.0 | 0.010994 | 0.412907 | 0.437892 | 0.437892 | 0.121637 |      0.0 |        0.0         0.0 |       0.0        0.0 |  93.782011 |       0.0 |  6.065307 |  9.048374 |  9.889504 | 0.121637 |
    | 01/01 10:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.390837 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.295808 |      0.0 | 0.090033 |      0.0 | 0.010982 | 0.396824 | 0.420642 | 0.420642 | 0.116845 |      0.0 |        0.0         0.0 |       0.0        0.0 |  90.391174 |       0.0 |  5.769498 |  8.958341 |  9.878522 | 0.116845 |
    | 01/01 11:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.705421 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.281382 |      0.0 | 0.089137 |      0.0 |  0.01097 | 0.381489 | 0.404198 | 0.404198 | 0.112277 |      0.0 |        0.0         0.0 |       0.0        0.0 |  86.685753 |       0.0 |  5.488116 |  8.869204 |  9.867552 | 0.112277 |
    | 01/01 12:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.944104 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.267659 |      0.0 |  0.08825 |      0.0 | 0.010958 | 0.366867 | 0.388519 | 0.388519 | 0.107922 |      0.0 |        0.0         0.0 |       0.0        0.0 |  84.741649 |       0.0 |  5.220458 |  8.780954 |  9.856594 | 0.107922 |
    | 01/01 13:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.456195 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.254605 |      0.0 | 0.087372 |      0.0 | 0.010946 | 0.352922 |  0.37357 |  0.37357 |  0.10377 |      0.0 |        0.0         0.0 |       0.0        0.0 |  82.285455 |       0.0 |  4.965853 |  8.693582 |  9.845648 |  0.10377 |
    | 01/01 14:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.663134 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.242188 |      0.0 | 0.086503 |      0.0 | 0.010934 | 0.339624 | 0.359315 | 0.359315 |  0.09981 |      0.0 |        0.0         0.0 |       0.0        0.0 |   79.62232 |       0.0 |  4.723666 |   8.60708 |  9.834715 |  0.09981 |
    | 01/01 15:00 |  0.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |  0.286 |    0.286 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.809588 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.230376 |      0.0 | 0.085642 |      0.0 | 0.010921 | 0.326939 |  0.34572 |  0.34572 | 0.096033 |      0.0 |        0.0         0.0 |       0.0        0.0 |  78.812732 |       0.0 |   4.49329 |  8.521438 |  9.823793 | 0.096033 |
    | 01/01 16:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.630937 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.21914 |      0.0 |  0.08479 |      0.0 | 0.010909 | 0.314839 | 0.332754 | 0.332754 | 0.092432 |      0.0 |        0.0         0.0 |       0.0        0.0 |  78.181795 |       0.0 |  4.274149 |  8.436648 |  9.812884 | 0.092432 |
    | 01/01 17:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.388758 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.208453 |      0.0 | 0.083946 |      0.0 | 0.010897 | 0.303296 | 0.320386 | 0.320386 | 0.088996 |      0.0 |        0.0         0.0 |       0.0        0.0 |  77.793036 |       0.0 |  4.065697 |  8.352702 |  9.801987 | 0.088996 |
    | 01/01 18:00 |  1.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |  1.859 |  0.43651 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.106117 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.198286 |      0.0 | 0.083111 |      0.0 | 0.010885 | 0.292282 | 0.308587 | 0.308587 | 0.085719 |  1.42249 |        0.0         0.0 |       0.0        0.0 |  77.686919 |       0.0 |   3.86741 |  8.269591 |  9.791102 | 0.085719 |
    | 01/01 19:00 |  5.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   5.944392    8.916588 |  0.0 |  8.008 | 0.070365 |   7.43049 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.188616 |      0.0 | 0.082284 |      0.0 | 0.010873 | 0.281773 |  0.29733 |  0.29733 | 0.082592 | 1.929635 |   5.944392    8.916588 |       0.0        0.0 |  77.686919 |       0.0 |  3.678794 |  8.187308 |  9.780229 | 0.082592 |
    | 01/01 20:00 |  2.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   9.205699   13.808549 |  0.0 |  4.147 | 0.045972 |  4.076635 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.179417 |      0.0 | 0.081465 |      0.0 | 0.010861 | 0.271743 | 0.286589 | 0.286589 | 0.079608 | 1.954028 |   9.205699   13.808549 |       0.0        0.0 |  77.686919 |       0.0 |  3.499377 |  8.105842 |  9.769368 | 0.079608 |
    | 01/01 21:00 |  4.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  14.774522   22.161783 |  0.0 |  7.007 |   0.0342 |  6.961028 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.170667 |      0.0 | 0.080654 |      0.0 | 0.010849 |  0.26217 |  0.27634 |  0.27634 | 0.076761 |   1.9658 |  14.774522   22.161783 |       0.0        0.0 |  77.686919 |       0.0 |  3.328711 |  8.025188 |  9.758519 | 0.076761 |
    | 01/01 22:00 | 10.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.873562   40.310343 |  0.0 | 15.158 | 0.015086 |   15.1238 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.162343 |      0.0 | 0.079852 |      0.0 | 0.010837 | 0.253032 | 0.266557 | 0.266557 | 0.074044 | 1.984914 |  26.873562   40.310343 |       0.0        0.0 |  77.686919 |       0.0 |  3.166368 |  7.945336 |  9.747682 | 0.074044 |
    | 01/01 23:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.975893    40.46384 |  0.0 |  0.143 | 0.067263 |  0.127914 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.154426 |      0.0 | 0.079057 |      0.0 | 0.010825 | 0.244308 |  0.25722 |  0.25722 |  0.07145 | 1.932737 |  26.975893    40.46384 |       0.0        0.0 |  77.686919 |       0.0 |  3.011942 |  7.866279 |  9.736857 |  0.07145 |
    | 02/01 00:00 |  0.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  27.722883   41.584324 |  0.0 |  1.001 | 0.061733 |  0.933737 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.146894 |      0.0 | 0.078271 |      0.0 | 0.010813 | 0.235978 | 0.248305 | 0.248305 | 0.068974 | 1.938267 |  27.722883   41.584324 |       0.0        0.0 |  77.686919 |       0.0 |  2.865048 |  7.788008 |  9.726045 | 0.068974 |
    | 02/01 01:00 |  3.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  31.105497   46.658245 |  0.0 |   4.29 |  0.04443 |  4.228267 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.13973 |      0.0 | 0.077492 |      0.0 | 0.010801 | 0.228023 | 0.239795 | 0.239795 |  0.06661 |  1.95557 |  31.105497   46.658245 |       0.0        0.0 |  77.686919 |       0.0 |  2.725318 |  7.710516 |  9.715244 |  0.06661 |
    | 02/01 02:00 |  2.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  33.472353   50.208529 |  0.0 |  3.003 | 0.050277 |   2.95857 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.132915 |      0.0 | 0.076721 |      0.0 | 0.010789 | 0.220425 | 0.231668 | 0.231668 | 0.064352 | 1.949723 |  33.472353   50.208529 |       0.0        0.0 |  77.686919 |       0.0 |  2.592403 |  7.633795 |  9.704455 | 0.064352 |
    | 02/01 03:00 | 10.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  45.329731   67.994597 |  0.0 | 14.872 | 0.015291 | 14.821723 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.126433 |      0.0 | 0.075958 |      0.0 | 0.010777 | 0.213167 | 0.223906 | 0.223906 | 0.062196 | 1.984709 |  45.329731   67.994597 |       0.0        0.0 |  77.686919 |       0.0 |   2.46597 |  7.557837 |  9.693679 | 0.062196 |
    | 02/01 04:00 |  3.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  49.321499   73.982248 |  0.0 |  5.005 | 0.040875 |  4.989709 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.120267 |      0.0 | 0.075202 |      0.0 | 0.010765 | 0.206233 | 0.216492 | 0.216492 | 0.060137 | 1.959125 |  49.321499   73.982248 |       0.0        0.0 |  77.686919 |       0.0 |  2.345703 |  7.482636 |  9.682914 | 0.060137 |
    | 02/01 05:00 |  3.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.178399   79.767599 |  0.0 |  4.862 | 0.041321 |  4.821125 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.114401 |      0.0 | 0.074453 |      0.0 | 0.010753 | 0.199608 |  0.20941 |  0.20941 | 0.058169 | 1.958679 |  53.178399   79.767599 |       0.0        0.0 |  77.686919 |       0.0 |  2.231302 |  7.408182 |  9.672161 | 0.058169 |
    | 02/01 06:00 |  1.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.518142   81.777214 |  0.0 |  1.716 | 0.110435 |  1.674679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.108822 |      0.0 | 0.073713 |      0.0 | 0.010741 | 0.193275 | 0.202643 | 0.202643 |  0.05629 | 1.889565 |  54.518142   81.777214 |       0.0        0.0 |  77.686919 |       0.0 |   2.12248 |   7.33447 |   9.66142 |  0.05629 |
    | 02/01 07:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |  0.143 |  0.87266 |  0.032565 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.103515 |      0.0 | 0.072979 |      0.0 | 0.010729 | 0.187223 | 0.196176 | 0.196176 | 0.054493 |  1.12734 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  2.018965 |   7.26149 |  9.650691 | 0.054493 |
    | 02/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |    0.0 |  1.12734 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.098466 |      0.0 | 0.072253 |      0.0 | 0.010717 | 0.181436 | 0.189995 | 0.189995 | 0.052776 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  1.920499 |  7.189237 |  9.639974 | 0.052776 |
    | 02/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.093664 |      0.0 | 0.071534 |      0.0 | 0.010705 | 0.175903 | 0.184086 | 0.184086 | 0.051135 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  1.826835 |  7.117703 |  9.629269 | 0.051135 |
    | 02/01 10:00 |  0.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |  0.572 |    0.572 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.089096 |      0.0 | 0.070822 |      0.0 | 0.010693 | 0.170611 | 0.178437 | 0.178437 | 0.049566 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  1.737739 |  7.046881 |  9.618576 | 0.049566 |
    | 02/01 11:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |  0.143 |    0.143 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.084751 |      0.0 | 0.070118 |      0.0 | 0.010681 |  0.16555 | 0.173035 | 0.173035 | 0.048065 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |  1.652989 |  6.976763 |  9.607894 | 0.048065 |
    | 02/01 12:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  57.062595   85.593892 |  0.0 |  5.148 | 1.411787 |     3.148 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.080617 |      0.0 |  0.06942 |      0.0 |  0.01067 | 0.160707 | 0.167867 | 0.167867 |  0.04663 | 0.588213 |  57.062595   85.593892 |       0.0        0.0 |  77.686919 |       0.0 |  1.572372 |  6.907343 |  9.597225 |  0.04663 |
    | 02/01 13:00 |  5.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  62.682765   94.024148 |  0.0 |  8.437 |      2.0 |  7.025213 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.076685 |      0.0 | 0.068729 |      0.0 | 0.010658 | 0.156072 | 0.162924 | 0.162924 | 0.045257 |      0.0 |  62.682765   94.024148 |       0.0        0.0 |  77.686919 |       0.0 |  1.495686 |  6.838614 |  9.586567 | 0.045257 |
    | 02/01 14:00 |  1.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  62.682765   94.024148 |  0.0 |  1.573 |    1.573 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.072945 |      0.0 | 0.068045 |      0.0 | 0.010646 | 0.151637 | 0.158194 | 0.158194 | 0.043943 |      0.0 |  62.682765   94.024148 |       0.0        0.0 |  77.686919 |       0.0 |  1.422741 |  6.770569 |  9.575921 | 0.043943 |
    | 02/01 15:00 | 20.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  84.763565  127.145348 |  0.0 | 29.601 | 0.076556 |    27.601 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.069388 |      0.0 | 0.067368 |      0.0 | 0.010634 |  0.14739 | 0.153667 | 0.153667 | 0.042685 | 1.923444 |  84.763565  127.145348 |       0.0        0.0 |  77.686919 |       0.0 |  1.353353 |    6.7032 |  9.565287 | 0.042685 |
    | 02/01 16:00 | 37.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  128.05992   192.08988 |  0.0 | 54.197 | 0.006942 | 54.120444 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.066004 |      0.0 | 0.066698 |      0.0 | 0.010622 | 0.143324 | 0.149334 | 0.149334 | 0.041482 | 1.993058 |  128.05992   192.08988 |       0.0        0.0 |  77.686919 |       0.0 |  1.287349 |  6.636503 |  9.554665 | 0.041482 |
    | 02/01 17:00 |  8.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 137.435167   206.15275 |  0.0 | 11.726 | 0.623609 | 11.719058 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.062785 |      0.0 | 0.066034 |      0.0 |  0.01061 | 0.139429 | 0.145185 | 0.145185 | 0.040329 | 1.376391 | 137.435167   206.15275 |       0.0        0.0 |  77.686919 |       0.0 |  1.224564 |  6.570468 |  9.544055 | 0.040329 |
    | 02/01 18:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  141.05468   211.58202 |  0.0 |  5.148 | 0.665759 |  4.524391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.059723 |      0.0 | 0.065377 |      0.0 | 0.010599 | 0.135699 | 0.141212 | 0.141212 | 0.039226 | 1.334241 |  141.05468   211.58202 |       0.0        0.0 |  77.686919 |       0.0 |  1.164842 |  6.505091 |  9.533456 | 0.039226 |
    | 02/01 19:00 |  7.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 149.102073  223.653109 |  0.0 | 10.725 | 0.069841 | 10.059241 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.05681 |      0.0 | 0.064727 |      0.0 | 0.010587 | 0.132124 | 0.137406 | 0.137406 | 0.038168 | 1.930159 | 149.102073  223.653109 |       0.0        0.0 |  77.686919 |       0.0 |  1.108032 |  6.440364 |  9.522869 | 0.038168 |
    | 02/01 20:00 | 18.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   170.2102    255.3153 |  0.0 | 26.455 | 0.004601 | 26.385159 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.054039 |      0.0 | 0.064083 |      0.0 | 0.010575 | 0.128697 | 0.133759 | 0.133759 | 0.037155 | 1.995399 |   170.2102    255.3153 |       0.0        0.0 |  77.686919 |       0.0 |  1.053992 |  6.376282 |  9.512294 | 0.037155 |
    | 02/01 21:00 | 15.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 187.824119  281.736179 |  0.0 | 22.022 | 0.007155 | 22.017399 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.051404 |      0.0 | 0.063445 |      0.0 | 0.010563 | 0.125412 | 0.130265 | 0.130265 | 0.036185 | 1.992845 | 187.824119  281.736179 |       0.0        0.0 |  77.686919 |       0.0 |  1.002588 |  6.312836 |  9.501731 | 0.036185 |
    | 02/01 22:00 |  6.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 195.025595  292.538393 |  0.0 |  9.009 | 0.025946 |  9.001845 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.048897 |      0.0 | 0.062814 |      0.0 | 0.010552 | 0.122262 | 0.126915 | 0.126915 | 0.035254 | 1.974054 | 195.025595  292.538393 |       0.0        0.0 |  77.686919 |       0.0 |  0.953692 |  6.250023 |  9.491179 | 0.035254 |
    | 02/01 23:00 |  1.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 197.178439  295.767658 |  0.0 |  2.717 | 0.047842 |  2.691054 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.046512 |      0.0 | 0.062189 |      0.0 |  0.01054 | 0.119241 | 0.123703 | 0.123703 | 0.034362 | 1.952158 | 197.178439  295.767658 |       0.0        0.0 |  77.686919 |       0.0 |   0.90718 |  6.187834 |  9.480639 | 0.034362 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 201.883365  302.825047 |  0.0 |  5.929 | 0.034767 |  5.881158 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.044244 |      0.0 |  0.06157 |      0.0 | 0.010528 | 0.116342 | 0.120623 | 0.120623 | 0.033506 | 1.965233 | 188.332116  286.921335 | 13.551249  15.903712 |  77.686919 |       0.0 |  0.862936 |  6.126264 |  9.470111 | 0.033506 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 204.469152  306.703727 |  0.0 |  3.267 | 0.045282 |  3.232233 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.042086 |      0.0 | 0.060957 |      0.0 | 0.010517 |  0.11356 | 0.117668 | 0.117668 | 0.032685 | 1.954718 | 179.485793  278.075012 | 24.983359  28.628715 |  77.686919 |       0.0 |   0.82085 |  6.065307 |  9.459595 | 0.032685 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 204.916926  307.375389 |  0.0 |  0.605 | 0.058514 |  0.559718 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.040033 |      0.0 | 0.060351 |      0.0 | 0.010505 | 0.110889 | 0.114832 | 0.114832 | 0.031898 | 1.941486 |  170.63947  269.228689 | 34.277456    38.1467 |  77.686919 |       0.0 |  0.780817 |  6.004956 |   9.44909 | 0.031898 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 205.063715  307.595572 |  0.0 |  0.242 | 0.060918 |  0.183486 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.038081 |      0.0 |  0.05975 |      0.0 | 0.010493 | 0.108324 |  0.11211 |  0.11211 | 0.031142 | 1.939082 | 161.793147  260.382367 | 43.270568  47.213206 |  77.686919 |       0.0 |  0.742736 |  5.945205 |  9.438597 | 0.031142 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 205.498981  308.248471 |  0.0 |  0.605 | 0.059438 |  0.544082 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.036224 |      0.0 | 0.059156 |      0.0 | 0.010482 | 0.105861 | 0.109497 | 0.109497 | 0.030416 | 1.940562 | 152.946824  251.536044 | 52.552156  56.712427 |  77.686919 |       0.0 |  0.706512 |   5.88605 |  9.428115 | 0.030416 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 201.740702  311.661946 |  0.0 |  2.904 | 0.047956 |  2.844562 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  3.016964 | 0.455204 | 0.0 |      0.0 | 0.455204 | 0.0 |      0.0 |      0.0 | 0.455204 |  0.04565 |      0.0 | 0.058567 |      0.0 |  0.01047 | 0.114687 | 0.107786 | 0.107786 | 0.029941 | 1.952044 | 144.100502  242.689721 | 57.640201  68.972225 |   80.24868 |       0.0 |  1.116066 |  5.827483 |  9.417645 | 0.029941 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  189.35585  312.185199 |  0.0 |  0.484 | 0.103281 |  0.436044 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.366844 | 1.025038 | 0.0 |      0.0 | 1.025038 | 0.0 |      0.0 |      0.0 | 1.025038 | 0.079635 |      0.0 | 0.057984 |      0.0 | 0.010458 | 0.148078 | 0.113525 | 0.113525 | 0.031535 | 1.896719 | 135.254179  233.843398 | 54.101671  78.341801 |  85.590485 |       0.0 |  2.061469 |  5.769498 |  9.407187 | 0.031535 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 176.970998  312.351661 |  0.0 |  0.242 | 0.321386 |  0.138719 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.247914 | 1.144263 | 0.0 |      0.0 | 1.144263 | 0.0 |      0.0 |      0.0 | 1.144263 | 0.128675 |      0.0 | 0.057407 |      0.0 | 0.010447 | 0.196529 | 0.136543 | 0.136543 | 0.037929 | 1.678614 | 126.407856  224.997075 | 50.563142  87.354586 |  90.694135 |       0.0 |  3.077058 |  5.712091 |   9.39674 | 0.037929 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 164.586146  302.611053 |  0.0 |    0.0 | 0.821608 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  11.06273 | 2.274892 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.026159 |      2.0 | 0.199247 |      0.0 | 0.056836 |      0.0 | 0.010435 | 0.292678 |  0.17981 |  0.17981 | 0.049947 | 0.857006 | 117.561533  216.150752 | 47.024613  86.460301 |  99.481974 |  0.248732 |  4.877811 |  5.655254 |  9.386305 | 0.049947 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 152.201294  290.226201 |  0.0 |    0.0 | 0.857006 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.064218 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.124944 |      2.0 | 0.287071 |      0.0 | 0.056271 |      0.0 | 0.010423 | 0.478708 | 0.260468 | 0.260468 | 0.072352 |      0.0 |  108.71521   207.30443 | 43.486084  82.921772 | 108.802608 |  1.188006 |   6.59074 |  5.598984 |  9.375882 | 0.072352 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 139.816442  277.841349 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.665299 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.271528 |      2.0 | 0.370611 |      0.0 | 0.055711 |      0.0 | 0.010412 | 0.708262 | 0.407024 | 0.407024 | 0.113062 |      0.0 |  99.868887  198.458107 | 39.947555  79.383243 | 117.522161 |  2.581778 |  8.220129 |  5.543273 |   9.36547 | 0.113062 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  127.43159  265.456497 |  0.0 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.530852 |  4.67202 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.499965 |      2.0 | 0.450077 |      0.0 | 0.055156 |      0.0 |   0.0104 | 1.015599 | 0.615583 | 0.615583 | 0.170995 |      0.0 |  91.022565  189.611784 | 36.409026  75.844714 | 126.380993 |  4.753833 |  9.770052 |  5.488116 |   9.35507 | 0.170995 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 115.046738  253.071645 |  0.0 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 4.945319 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 0.732671 |      2.0 | 0.525668 |      0.0 | 0.054608 |      0.0 | 0.010389 | 1.323336 | 0.885144 | 0.885144 | 0.245873 |      0.0 |  82.176242  180.765461 | 32.870497  72.306184 | 133.820525 |  6.966481 | 11.244384 |  5.433509 |  9.344681 | 0.245873 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 102.661886  240.686793 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 5.544677 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 1.000269 | 1.937781 | 0.596042 | 0.055306 |  0.05434 | 0.006913 | 0.010381 | 1.661032 | 1.187903 | 1.187903 | 0.329973 |      0.0 |  73.329919  171.919138 | 29.331968  68.767655 |   140.6607 |  9.510889 | 12.586122 |  5.434475 |  9.341213 | 0.329973 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  90.277035  228.301941 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 6.125991 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 | 1.297721 | 1.870694 |  0.65983 | 0.114939 | 0.054647 | 0.014367 | 0.010381 | 2.022579 |  1.51365 |  1.51365 | 0.420458 |      0.0 |  64.483596  163.072815 | 25.793438  65.229126 |  146.91956 |  12.33916 | 13.796986 |  5.494767 |  9.345199 | 0.420458 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  77.892183   215.91709 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 6.683286 | 0.0 |      0.0 |      2.0 | 0.0 |      0.0 |   1.6199 | 1.810151 | 0.717396 | 0.168755 | 0.055515 | 0.021094 |  0.01039 |   2.4032 | 1.863798 | 1.863798 | 0.517722 |      0.0 |  55.637273  154.226493 | 22.254909  61.690597 | 152.621126 | 15.402546 | 14.889741 |  5.608007 |  9.355904 | 0.517722 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  65.507331  203.532238 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.212073 | 0.0 |      0.0 |      2.0 | 0.0 | 0.241834 |  1.96174 | 1.755513 | 0.769347 | 0.217322 | 0.056884 | 0.027165 | 0.010405 | 3.040209 | 2.252286 | 2.252286 | 0.625635 |      0.0 |   46.79095   145.38017 |  18.71638  58.152068 | 157.793905 | 18.411045 | 15.875907 |  5.768445 |  9.372665 | 0.625635 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  53.122479  191.147386 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.709235 | 0.0 |      0.0 |      2.0 | 0.0 | 1.621204 | 2.295348 | 1.706205 |  0.81623 | 0.261151 | 0.058698 | 0.032644 | 0.010426 | 4.801907 | 2.858024 | 2.858024 | 0.793896 |      0.0 |  37.944628  136.533847 | 15.177851  54.613539 | 162.469522 | 20.203728 | 16.765882 |  5.970898 |  9.394882 | 0.793896 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  40.737627  178.762534 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.172871 | 0.0 |      0.0 |      2.0 | 0.0 | 2.508996 | 2.510065 | 1.661706 | 0.858541 | 0.300706 |  0.06091 | 0.037588 | 0.010454 | 5.948966 | 4.073128 | 4.073128 | 1.131425 |      0.0 |  29.098305  127.687524 | 11.639322   51.07501 | 166.681503 | 21.357537 | 17.569047 |  6.210694 |  9.422017 | 1.131425 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  28.352775  166.377682 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.602123 | 0.0 |      0.0 |      2.0 | 0.0 | 3.131882 | 2.660713 | 1.621548 | 0.896724 | 0.336402 | 0.063474 |  0.04205 | 0.010486 |  6.76328 | 5.363783 | 5.363783 |  1.48994 |      0.0 |  20.251982  118.841201 |  8.100793   47.53648 | 170.464232 | 22.167065 | 18.293871 |  6.483622 |   9.45358 |  1.48994 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  15.967923   153.99283 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.996993 | 0.0 |      0.0 |      2.0 | 0.0 | 3.605776 | 2.775327 | 1.585306 | 0.931183 | 0.368616 |  0.06635 | 0.046077 | 0.010524 | 7.389159 | 6.346809 | 6.346809 | 1.763003 |      0.0 |  11.405659  109.994878 |  4.562264  43.997951 | 173.852091 | 22.782955 | 18.947994 |  6.785889 |  9.489134 | 1.763003 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   3.583071  141.607978 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 9.358164 | 0.0 |      0.0 |      2.0 | 0.0 | 3.990219 | 2.868307 |   1.5526 | 0.962281 | 0.397689 | 0.069503 | 0.049711 | 0.010565 | 7.900875 | 7.077728 | 7.077728 | 1.966036 |      0.0 |   2.559336  101.148556 |  1.023735  40.459422 | 176.878779 | 23.282593 | 19.538314 |  7.114075 |   9.52828 | 1.966036 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  129.223126 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 2.559336  8.846323 | 0.0   0.0 |  7.983961 | 6.244676 | 0.0 | 0.029511 |      2.0 | 0.0 | 2.961749 | 2.619566 | 1.523084 | 0.990345 | 0.423925 | 0.072899 | 0.052991 | 0.010611 |  6.65517 | 7.532144 | 7.532144 | 2.092262 |      0.0 |        0.0   92.302233 |       0.0  36.920893 | 178.588554 | 21.945953 | 20.071053 |  7.465101 |   9.57066 | 2.092262 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  116.838274 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.846323 | 0.0   0.0 |  6.192426 | 4.937511 | 0.0 | 0.029125 |      2.0 | 0.0 | 1.921493 | 2.367975 | 1.496447 | 1.015672 | 0.447602 |  0.07651 |  0.05595 | 0.010659 | 5.392309 | 7.048852 | 7.048852 | 1.958014 |      0.0 |        0.0    83.45591 |       0.0  33.382364 | 179.814343 | 20.593996 | 20.551828 |  7.836194 |  9.615951 | 1.958014 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  104.447803 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.007794 | 0.0 | 0.028801 |      2.0 | 0.0 | 1.417194 | 2.246007 | 1.472409 | 1.038529 |  0.46897 | 0.080308 | 0.058621 | 0.010711 |  4.79275 | 5.987543 | 5.987543 | 1.663206 |      0.0 |        0.0   74.605574 |       0.0  29.842229 | 180.972983 | 19.938589 | 20.985708 |  8.224855 |  9.663861 | 1.663206 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   92.057332 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.072538 | 0.0 | 0.029575 |      2.0 | 0.0 | 1.184786 | 2.189798 | 1.450715 | 1.059156 | 0.488254 | 0.084272 | 0.061032 | 0.010766 | 4.528778 | 5.146691 | 5.146691 | 1.429636 |      0.0 |        0.0   65.755237 |       0.0  26.302095 | 182.066106 | 19.636543 | 21.377267 |  8.628837 |  9.714127 | 1.429636 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   79.666862 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.134002 | 0.0 | 0.029702 |      2.0 | 0.0 | 1.090124 | 2.166904 | 1.431137 | 1.077771 | 0.505656 | 0.088378 | 0.063207 | 0.010823 |    4.434 | 4.697922 | 4.697922 | 1.304978 |      0.0 |        0.0   56.904901 |       0.0   22.76196 | 183.097637 | 19.513516 | 21.730632 |  9.046115 |  9.766512 | 1.304978 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   67.276391 |  0.0 |  1.573 | 0.049993 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.192342 | 0.0 | 0.012498 |      2.0 | 0.0 | 1.064672 | 2.160748 | 1.413468 |  1.09457 | 0.521361 | 0.092608 |  0.06517 | 0.010882 | 4.423481 | 4.502645 | 4.502645 | 1.250735 | 1.523007 |        0.0   48.054565 |       0.0  19.221826 | 184.088032 | 19.480438 |  22.04953 |  9.474868 |    9.8208 | 1.250735 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    54.88592 |  0.0 |    0.0 |  0.05838 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.248666 | 0.0 | 0.014595 |      2.0 | 0.0 | 1.073819 |  2.16296 | 1.397523 | 1.109731 | 0.535535 | 0.096945 | 0.066942 | 0.010943 | 4.454399 | 4.441198 | 4.441198 | 1.233666 | 1.464627 |        0.0   39.204229 |       0.0  15.681691 | 185.020006 | 19.492325 | 22.337322 |  9.913458 |  9.876799 | 1.233666 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   42.495449 |  0.0 |    0.0 | 0.059405 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.301945 | 0.0 | 0.014851 |      2.0 | 0.0 | 1.099459 | 2.169162 | 1.383134 | 1.123413 | 0.548325 | 0.101373 | 0.068541 | 0.011006 | 4.504414 | 4.446798 | 4.446798 | 1.235222 | 1.405222 |        0.0   30.353892 |       0.0  12.141557 | 185.898446 | 19.525649 | 22.597043 |  10.36041 |  9.934333 | 1.235222 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   30.104979 |  0.0 |    0.0 |  0.21716 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 |  5.35241 | 0.0 |  0.05429 |      2.0 | 0.0 | 1.132428 | 2.177135 | 1.370148 | 1.135761 | 0.559869 | 0.105878 | 0.069984 | 0.011071 | 4.562272 | 4.484782 | 4.484782 | 1.245773 | 1.188062 |        0.0   21.503556 |       0.0   8.601422 | 186.686981 | 19.568495 |  22.83143 | 10.814401 |  9.993246 | 1.245773 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   17.714508 |  0.0 |  0.847 |  0.89524 |  0.035062 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.230298 | 5.428463 | 0.0 |  0.22381 |      2.0 | 0.0 | 1.179211 |  2.18845 | 1.358428 | 1.146904 | 0.570286 | 0.110447 | 0.071286 | 0.011137 | 4.636149 |  4.53836 |  4.53836 | 1.260655 |  1.10476 |        0.0    12.65322 |       0.0   5.061288 | 187.265006 | 19.629297 | 23.042955 |  11.27424 | 10.053394 | 1.260655 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    5.324037 |  0.0 |  0.484 |  1.58876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.431391 | 0.0 | 0.668518 |      2.0 | 0.0 | 1.204287 | 2.194515 | 1.347852 |  1.15696 | 0.579687 | 0.115069 | 0.072461 | 0.011204 | 4.682035 | 4.602508 | 4.602508 | 1.278474 |      0.0 |        0.0    3.802884 |       0.0   1.521153 | 187.360333 | 19.661887 | 23.233847 | 11.738857 | 10.114651 | 1.278474 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  3.802884 | 0.0   0.0 |  2.662019 |  2.33618 | 0.0 | 3.529947 |      2.0 | 0.0 |      0.0 | 1.903068 | 1.338308 | 1.166035 | 0.588171 | 0.119735 | 0.073521 | 0.011273 | 3.200111 | 4.549894 | 4.549894 |  1.26386 |      0.0 |        0.0         0.0 |       0.0        0.0 | 184.156223 | 18.094999 |  23.40612 | 12.207293 | 10.176899 |  1.26386 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.529885 |      2.0 | 0.0 |      0.0 | 1.531642 | 1.329694 | 1.174225 | 0.595828 | 0.124434 | 0.074478 | 0.011343 | 2.841643 | 3.804097 | 3.804097 | 1.056693 |      0.0 |        0.0         0.0 |       0.0        0.0 | 179.626338 | 14.563358 | 23.561589 | 12.678687 | 10.240035 | 1.056693 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 5.874002 |      2.0 | 0.0 |      0.0 | 1.195562 | 1.321921 | 1.181616 | 0.602737 | 0.129159 | 0.075342 | 0.011413 |  2.51775 | 3.094159 | 3.094159 | 0.859489 |      0.0 |        0.0         0.0 |       0.0        0.0 | 173.752336 | 11.367796 | 23.701893 | 13.152266 | 10.303964 | 0.859489 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |   5.8739 |      2.0 | 0.0 |      0.0 | 0.891464 | 1.314905 | 1.188287 | 0.608973 | 0.133902 | 0.076122 | 0.011485 | 2.225137 | 2.679498 | 2.679498 | 0.744305 |      0.0 |        0.0         0.0 |       0.0        0.0 | 167.878435 |  8.476332 | 23.828512 | 13.627337 | 10.368601 | 0.744305 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 7.500667 |      2.0 | 0.0 |      0.0 | 0.616305 | 1.308574 | 1.194306 | 0.614601 | 0.138657 | 0.076825 | 0.011557 | 1.960825 | 2.363055 | 2.363055 | 0.656404 |      0.0 |        0.0         0.0 |       0.0        0.0 | 160.377769 |  5.860028 |  23.94278 |  14.10328 | 10.433869 | 0.656404 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.568376 |      2.0 | 0.0 |      0.0 |  0.36733 | 1.302861 | 1.199739 | 0.619679 | 0.143418 |  0.07746 |  0.01163 | 1.722117 |  2.08467 |  2.08467 | 0.579075 |      0.0 |        0.0         0.0 |       0.0        0.0 | 153.809393 |  3.492698 | 24.045903 | 14.579541 | 10.499699 | 0.579075 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 5.290409 |      2.0 | 0.0 |      0.0 | 0.142049 | 1.297705 | 1.204641 | 0.624262 |  0.14818 | 0.078033 | 0.011703 | 1.506573 | 1.833918 | 1.833918 | 0.509422 |      0.0 |        0.0         0.0 |       0.0        0.0 | 148.518984 |  1.350649 | 24.138966 | 15.055624 | 10.566028 | 0.509422 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.092469 | 1.350649 | 0.0 |      0.0 |      0.0 | 1.293052 | 1.209065 | 0.051197 | 0.150061 |   0.0064 | 0.011737 | 1.370864 | 1.611732 | 1.611732 | 0.447703 |      0.0 |        0.0         0.0 |       0.0        0.0 | 144.426515 |       0.0 | 24.222953 |  14.95676 | 10.560691 | 0.447703 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.863004 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.181367 |      0.0 | 0.148822 |      0.0 | 0.011728 | 1.341917 | 1.447287 | 1.447287 | 0.402024 |      0.0 |        0.0         0.0 |       0.0        0.0 | 142.563511 |       0.0 | 23.041585 | 14.807938 | 10.548963 | 0.402024 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.816588 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.123751 |      0.0 | 0.147341 |      0.0 | 0.011715 | 1.282807 |  1.36364 |  1.36364 | 0.378789 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.746923 |       0.0 | 21.917834 | 14.660597 | 10.537249 | 0.378789 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.204368 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.068945 |      0.0 | 0.145875 |      0.0 | 0.011702 | 1.226522 | 1.307999 | 1.307999 | 0.363333 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.542555 |       0.0 | 20.848889 | 14.514721 | 10.525547 | 0.363333 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.055428 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 1.016812 |      0.0 | 0.144424 |      0.0 | 0.011689 | 1.172925 |  1.25209 |  1.25209 | 0.347803 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.487127 |       0.0 | 19.832076 | 14.370297 | 10.513859 | 0.347803 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.053093 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.967222 |      0.0 | 0.142987 |      0.0 | 0.011676 | 1.121884 | 1.197487 | 1.197487 | 0.332635 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.434035 |       0.0 | 18.864855 |  14.22731 | 10.502183 | 0.332635 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.052286 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |  0.92005 |      0.0 | 0.141564 |      0.0 | 0.011663 | 1.073277 | 1.145295 | 1.145295 | 0.318137 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.381748 |       0.0 | 17.944805 | 14.085746 | 10.490521 | 0.318137 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.052492 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 | 0.875178 |      0.0 | 0.140156 |      0.0 |  0.01165 | 1.026984 | 1.095572 | 1.095572 | 0.304326 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.329256 |       0.0 | 17.069626 | 13.945591 | 10.478871 | 0.304326 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96p_glacier:

glacier
_______

Regarding glacier simulation, all functionalities of |hland_96p| are identical with the
ones of |hland_96|.  Hence, all snow data in the following table agrees with the
:ref:`hland_96_glacier` example of |hland_96|:

.. integration-test::

    >>> zonetype(GLACIER)
    >>> test('hland_96p_glacier')
    |        date |    p |     t |    tc | fracrain | rfc | sfc |    cfact |                    swe |     gact |     pc |  ei |     tf | spl | wcl | spg | wcg |    glmelt |               melt |      refr |       in_ |         r |  sr |  ea |       dp |  el |        rs |       ri |      gr1 |      rg1 |      gr2 |      rg2 |      gr3 |      rg3 |      inrc |     outrc |        rt |       qt |  ic |                     sp |                   wc |  sm |       suz |       sg1 |       sg2 |       sg3 |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.487706 |      0.0 | 0.099502 |      0.0 | 0.011105 |  0.598312 |  0.202814 |  0.202814 | 0.056337 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.512294 |  9.900498 |  9.988895 | 0.056337 |
    | 01/01 01:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.46392 |      0.0 | 0.098512 |      0.0 | 0.011093 |  0.573524 |  0.414293 |  0.414293 | 0.115081 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.048374 |  9.801987 |  9.977802 | 0.115081 |
    | 01/01 02:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.441294 |      0.0 | 0.097531 |      0.0 |  0.01108 |  0.549906 |  0.547167 |  0.547167 | 0.151991 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   8.60708 |  9.704455 |  9.966722 | 0.151991 |
    | 01/01 03:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.419772 |      0.0 | 0.096561 |      0.0 | 0.011068 |  0.527401 |  0.556325 |  0.556325 | 0.154535 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  8.187308 |  9.607894 |  9.955654 | 0.154535 |
    | 01/01 04:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |   0.3993 |      0.0 |   0.0956 |      0.0 | 0.011056 |  0.505956 |  0.537334 |  0.537334 |  0.14926 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  7.788008 |  9.512294 |  9.944598 |  0.14926 |
    | 01/01 05:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.379826 |      0.0 | 0.094649 |      0.0 | 0.011043 |  0.485518 |  0.515762 |  0.515762 | 0.143267 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  7.408182 |  9.417645 |  9.933555 | 0.143267 |
    | 01/01 06:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.361301 |      0.0 | 0.093707 |      0.0 | 0.011031 |   0.46604 |  0.494888 |  0.494888 | 0.137469 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  7.046881 |  9.323938 |  9.922524 | 0.137469 |
    | 01/01 07:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.34368 |      0.0 | 0.092775 |      0.0 | 0.011019 |  0.447474 |  0.474971 |  0.474971 | 0.131936 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |    6.7032 |  9.231163 |  9.911505 | 0.131936 |
    | 01/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.326919 |      0.0 | 0.091852 |      0.0 | 0.011007 |  0.429777 |  0.455987 |  0.455987 | 0.126663 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  6.376282 |  9.139312 |  9.900498 | 0.126663 |
    | 01/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.310975 |      0.0 | 0.090938 |      0.0 | 0.010994 |  0.412907 |  0.437892 |  0.437892 | 0.121637 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  6.065307 |  9.048374 |  9.889504 | 0.121637 |
    | 01/01 10:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.295808 |      0.0 | 0.090033 |      0.0 | 0.010982 |  0.396824 |  0.420642 |  0.420642 | 0.116845 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  5.769498 |  8.958341 |  9.878522 | 0.116845 |
    | 01/01 11:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.281382 |      0.0 | 0.089137 |      0.0 |  0.01097 |  0.381489 |  0.404198 |  0.404198 | 0.112277 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  5.488116 |  8.869204 |  9.867552 | 0.112277 |
    | 01/01 12:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.267659 |      0.0 |  0.08825 |      0.0 | 0.010958 |  0.366867 |  0.388519 |  0.388519 | 0.107922 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  5.220458 |  8.780954 |  9.856594 | 0.107922 |
    | 01/01 13:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.254605 |      0.0 | 0.087372 |      0.0 | 0.010946 |  0.352922 |   0.37357 |   0.37357 |  0.10377 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  4.965853 |  8.693582 |  9.845648 |  0.10377 |
    | 01/01 14:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.242188 |      0.0 | 0.086503 |      0.0 | 0.010934 |  0.339624 |  0.359315 |  0.359315 |  0.09981 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  4.723666 |   8.60708 |  9.834715 |  0.09981 |
    | 01/01 15:00 |  0.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     0.2288      0.3432 | 0.901953 |  0.286 | 0.0 |  0.286 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.230376 |      0.0 | 0.085642 |      0.0 | 0.010921 |  0.326939 |   0.34572 |   0.34572 | 0.096033 | 0.0 |     0.2288      0.3432 |       0.0        0.0 | 0.0 |       0.0 |   4.49329 |  8.521438 |  9.823793 | 0.096033 |
    | 01/01 16:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     0.2288      0.3432 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.21914 |      0.0 |  0.08479 |      0.0 | 0.010909 |  0.314839 |  0.332754 |  0.332754 | 0.092432 | 0.0 |     0.2288      0.3432 |       0.0        0.0 | 0.0 |       0.0 |  4.274149 |  8.436648 |  9.812884 | 0.092432 |
    | 01/01 17:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     0.2288      0.3432 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.208453 |      0.0 | 0.083946 |      0.0 | 0.010897 |  0.303296 |  0.320386 |  0.320386 | 0.088996 | 0.0 |     0.2288      0.3432 |       0.0        0.0 | 0.0 |       0.0 |  4.065697 |  8.352702 |  9.801987 | 0.088996 |
    | 01/01 18:00 |  1.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |      1.716       2.574 | 0.901953 |  1.859 | 0.0 |  1.859 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.198286 |      0.0 | 0.083111 |      0.0 | 0.010885 |  0.292282 |  0.308587 |  0.308587 | 0.085719 | 0.0 |      1.716       2.574 |       0.0        0.0 | 0.0 |       0.0 |   3.86741 |  8.269591 |  9.791102 | 0.085719 |
    | 01/01 19:00 |  5.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     8.1224     12.1836 | 0.901953 |  8.008 | 0.0 |  8.008 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.188616 |      0.0 | 0.082284 |      0.0 | 0.010873 |  0.281773 |   0.29733 |   0.29733 | 0.082592 | 0.0 |     8.1224     12.1836 |       0.0        0.0 | 0.0 |       0.0 |  3.678794 |  8.187308 |  9.780229 | 0.082592 |
    | 01/01 20:00 |  2.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |      11.44       17.16 | 0.901953 |  4.147 | 0.0 |  4.147 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.179417 |      0.0 | 0.081465 |      0.0 | 0.010861 |  0.271743 |  0.286589 |  0.286589 | 0.079608 | 0.0 |      11.44       17.16 |       0.0        0.0 | 0.0 |       0.0 |  3.499377 |  8.105842 |  9.769368 | 0.079608 |
    | 01/01 21:00 |  4.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    17.0456     25.5684 | 0.901953 |  7.007 | 0.0 |  7.007 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.170667 |      0.0 | 0.080654 |      0.0 | 0.010849 |   0.26217 |   0.27634 |   0.27634 | 0.076761 | 0.0 |    17.0456     25.5684 |       0.0        0.0 | 0.0 |       0.0 |  3.328711 |  8.025188 |  9.758519 | 0.076761 |
    | 01/01 22:00 | 10.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     29.172      43.758 | 0.901953 | 15.158 | 0.0 | 15.158 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.162343 |      0.0 | 0.079852 |      0.0 | 0.010837 |  0.253032 |  0.266557 |  0.266557 | 0.074044 | 0.0 |     29.172      43.758 |       0.0        0.0 | 0.0 |       0.0 |  3.166368 |  7.945336 |  9.747682 | 0.074044 |
    | 01/01 23:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    29.2864     43.9296 | 0.901953 |  0.143 | 0.0 |  0.143 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.154426 |      0.0 | 0.079057 |      0.0 | 0.010825 |  0.244308 |   0.25722 |   0.25722 |  0.07145 | 0.0 |    29.2864     43.9296 |       0.0        0.0 | 0.0 |       0.0 |  3.011942 |  7.866279 |  9.736857 |  0.07145 |
    | 02/01 00:00 |  0.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    30.0872     45.1308 | 0.902305 |  1.001 | 0.0 |  1.001 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.146894 |      0.0 | 0.078271 |      0.0 | 0.010813 |  0.235978 |  0.248305 |  0.248305 | 0.068974 | 0.0 |    30.0872     45.1308 |       0.0        0.0 | 0.0 |       0.0 |  2.865048 |  7.788008 |  9.726045 | 0.068974 |
    | 02/01 01:00 |  3.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    33.5192     50.2788 | 0.902305 |   4.29 | 0.0 |   4.29 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.13973 |      0.0 | 0.077492 |      0.0 | 0.010801 |  0.228023 |  0.239795 |  0.239795 |  0.06661 | 0.0 |    33.5192     50.2788 |       0.0        0.0 | 0.0 |       0.0 |  2.725318 |  7.710516 |  9.715244 |  0.06661 |
    | 02/01 02:00 |  2.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    35.9216     53.8824 | 0.902305 |  3.003 | 0.0 |  3.003 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.132915 |      0.0 | 0.076721 |      0.0 | 0.010789 |  0.220425 |  0.231668 |  0.231668 | 0.064352 | 0.0 |    35.9216     53.8824 |       0.0        0.0 | 0.0 |       0.0 |  2.592403 |  7.633795 |  9.704455 | 0.064352 |
    | 02/01 03:00 | 10.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    47.8192     71.7288 | 0.902305 | 14.872 | 0.0 | 14.872 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.126433 |      0.0 | 0.075958 |      0.0 | 0.010777 |  0.213167 |  0.223906 |  0.223906 | 0.062196 | 0.0 |    47.8192     71.7288 |       0.0        0.0 | 0.0 |       0.0 |   2.46597 |  7.557837 |  9.693679 | 0.062196 |
    | 02/01 04:00 |  3.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    51.8232     77.7348 | 0.902305 |  5.005 | 0.0 |  5.005 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.120267 |      0.0 | 0.075202 |      0.0 | 0.010765 |  0.206233 |  0.216492 |  0.216492 | 0.060137 | 0.0 |    51.8232     77.7348 |       0.0        0.0 | 0.0 |       0.0 |  2.345703 |  7.482636 |  9.682914 | 0.060137 |
    | 02/01 05:00 |  3.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    55.7128     83.5692 | 0.902305 |  4.862 | 0.0 |  4.862 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.114401 |      0.0 | 0.074453 |      0.0 | 0.010753 |  0.199608 |   0.20941 |   0.20941 | 0.058169 | 0.0 |    55.7128     83.5692 |       0.0        0.0 | 0.0 |       0.0 |  2.231302 |  7.408182 |  9.672161 | 0.058169 |
    | 02/01 06:00 |  1.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    57.0856     85.6284 | 0.902305 |  1.716 | 0.0 |  1.716 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.108822 |      0.0 | 0.073713 |      0.0 | 0.010741 |  0.193275 |  0.202643 |  0.202643 |  0.05629 | 0.0 |    57.0856     85.6284 |       0.0        0.0 | 0.0 |       0.0 |   2.12248 |   7.33447 |   9.66142 |  0.05629 |
    | 02/01 07:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |       57.2        85.8 | 0.902305 |  0.143 | 0.0 |  0.143 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.103515 |      0.0 | 0.072979 |      0.0 | 0.010729 |  0.187223 |  0.196176 |  0.196176 | 0.054493 | 0.0 |       57.2        85.8 |       0.0        0.0 | 0.0 |       0.0 |  2.018965 |   7.26149 |  9.650691 | 0.054493 |
    | 02/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |       57.2        85.8 | 0.902305 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.098466 |      0.0 | 0.072253 |      0.0 | 0.010717 |  0.181436 |  0.189995 |  0.189995 | 0.052776 | 0.0 |       57.2        85.8 |       0.0        0.0 | 0.0 |       0.0 |  1.920499 |  7.189237 |  9.639974 | 0.052776 |
    | 02/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |       57.2        85.8 | 0.902305 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.093664 |      0.0 | 0.071534 |      0.0 | 0.010705 |  0.175903 |  0.184086 |  0.184086 | 0.051135 | 0.0 |       57.2        85.8 |       0.0        0.0 | 0.0 |       0.0 |  1.826835 |  7.117703 |  9.629269 | 0.051135 |
    | 02/01 10:00 |  0.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    57.6576     86.4864 | 0.902305 |  0.572 | 0.0 |  0.572 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.089096 |      0.0 | 0.070822 |      0.0 | 0.010693 |  0.170611 |  0.178437 |  0.178437 | 0.049566 | 0.0 |    57.6576     86.4864 |       0.0        0.0 | 0.0 |       0.0 |  1.737739 |  7.046881 |  9.618576 | 0.049566 |
    | 02/01 11:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |     57.772      86.658 | 0.902305 |  0.143 | 0.0 |  0.143 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.084751 |      0.0 | 0.070118 |      0.0 | 0.010681 |   0.16555 |  0.173035 |  0.173035 | 0.048065 | 0.0 |     57.772      86.658 |       0.0        0.0 | 0.0 |       0.0 |  1.652989 |  6.976763 |  9.607894 | 0.048065 |
    | 02/01 12:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    61.8904     92.8356 | 0.902305 |  5.148 | 0.0 |  5.148 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.080617 |      0.0 |  0.06942 |      0.0 |  0.01067 |  0.160707 |  0.167867 |  0.167867 |  0.04663 | 0.0 |    61.8904     92.8356 |       0.0        0.0 | 0.0 |       0.0 |  1.572372 |  6.907343 |  9.597225 |  0.04663 |
    | 02/01 13:00 |  5.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      68.64      102.96 | 0.902305 |  8.437 | 0.0 |  8.437 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.076685 |      0.0 | 0.068729 |      0.0 | 0.010658 |  0.156072 |  0.162924 |  0.162924 | 0.045257 | 0.0 |      68.64      102.96 |       0.0        0.0 | 0.0 |       0.0 |  1.495686 |  6.838614 |  9.586567 | 0.045257 |
    | 02/01 14:00 |  1.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    69.8984    104.8476 | 0.902305 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.072945 |      0.0 | 0.068045 |      0.0 | 0.010646 |  0.151637 |  0.158194 |  0.158194 | 0.043943 | 0.0 |    69.8984    104.8476 |       0.0        0.0 | 0.0 |       0.0 |  1.422741 |  6.770569 |  9.575921 | 0.043943 |
    | 02/01 15:00 | 20.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    93.5792    140.3688 | 0.902305 | 29.601 | 0.0 | 29.601 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.069388 |      0.0 | 0.067368 |      0.0 | 0.010634 |   0.14739 |  0.153667 |  0.153667 | 0.042685 | 0.0 |    93.5792    140.3688 |       0.0        0.0 | 0.0 |       0.0 |  1.353353 |    6.7032 |  9.565287 | 0.042685 |
    | 02/01 16:00 | 37.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   136.9368    205.4052 | 0.902305 | 54.197 | 0.0 | 54.197 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.066004 |      0.0 | 0.066698 |      0.0 | 0.010622 |  0.143324 |  0.149334 |  0.149334 | 0.041482 | 0.0 |   136.9368    205.4052 |       0.0        0.0 | 0.0 |       0.0 |  1.287349 |  6.636503 |  9.554665 | 0.041482 |
    | 02/01 17:00 |  8.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   146.3176    219.4764 | 0.902305 | 11.726 | 0.0 | 11.726 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.062785 |      0.0 | 0.066034 |      0.0 |  0.01061 |  0.139429 |  0.145185 |  0.145185 | 0.040329 | 0.0 |   146.3176    219.4764 |       0.0        0.0 | 0.0 |       0.0 |  1.224564 |  6.570468 |  9.544055 | 0.040329 |
    | 02/01 18:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    150.436     225.654 | 0.902305 |  5.148 | 0.0 |  5.148 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.059723 |      0.0 | 0.065377 |      0.0 | 0.010599 |  0.135699 |  0.141212 |  0.141212 | 0.039226 | 0.0 |    150.436     225.654 |       0.0        0.0 | 0.0 |       0.0 |  1.164842 |  6.505091 |  9.533456 | 0.039226 |
    | 02/01 19:00 |  7.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    159.016     238.524 | 0.902305 | 10.725 | 0.0 | 10.725 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 |  0.05681 |      0.0 | 0.064727 |      0.0 | 0.010587 |  0.132124 |  0.137406 |  0.137406 | 0.038168 | 0.0 |    159.016     238.524 |       0.0        0.0 | 0.0 |       0.0 |  1.108032 |  6.440364 |  9.522869 | 0.038168 |
    | 02/01 20:00 | 18.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |     180.18      270.27 | 0.902305 | 26.455 | 0.0 | 26.455 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.054039 |      0.0 | 0.064083 |      0.0 | 0.010575 |  0.128697 |  0.133759 |  0.133759 | 0.037155 | 0.0 |     180.18      270.27 |       0.0        0.0 | 0.0 |       0.0 |  1.053992 |  6.376282 |  9.512294 | 0.037155 |
    | 02/01 21:00 | 15.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   197.7976    296.6964 | 0.902305 | 22.022 | 0.0 | 22.022 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.051404 |      0.0 | 0.063445 |      0.0 | 0.010563 |  0.125412 |  0.130265 |  0.130265 | 0.036185 | 0.0 |   197.7976    296.6964 |       0.0        0.0 | 0.0 |       0.0 |  1.002588 |  6.312836 |  9.501731 | 0.036185 |
    | 02/01 22:00 |  6.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   205.0048    307.5072 | 0.902305 |  9.009 | 0.0 |  9.009 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.048897 |      0.0 | 0.062814 |      0.0 | 0.010552 |  0.122262 |  0.126915 |  0.126915 | 0.035254 | 0.0 |   205.0048    307.5072 |       0.0        0.0 | 0.0 |       0.0 |  0.953692 |  6.250023 |  9.491179 | 0.035254 |
    | 02/01 23:00 |  1.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   207.1784    310.7676 | 0.902305 |  2.717 | 0.0 |  2.717 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.046512 |      0.0 | 0.062189 |      0.0 |  0.01054 |  0.119241 |  0.123703 |  0.123703 | 0.034362 | 0.0 |   207.1784    310.7676 |       0.0        0.0 | 0.0 |       0.0 |   0.90718 |  6.187834 |  9.480639 | 0.034362 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   211.9216    317.8824 | 0.902686 |  5.929 | 0.0 |  5.929 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.044244 |      0.0 |  0.06157 |      0.0 | 0.010528 |  0.116342 |  0.120623 |  0.120623 | 0.033506 | 0.0 | 198.332077  301.921277 | 13.589523  15.961123 | 0.0 |       0.0 |  0.862936 |  6.126264 |  9.470111 | 0.033506 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   214.5352    321.8028 | 0.902686 |  3.267 | 0.0 |  3.267 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.042086 |      0.0 | 0.060957 |      0.0 | 0.010517 |   0.11356 |  0.117668 |  0.117668 | 0.032685 | 0.0 | 189.485754  293.074954 | 25.049446  28.727846 | 0.0 |       0.0 |   0.82085 |  6.065307 |  9.459595 | 0.032685 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   215.0192    322.5288 | 0.902686 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.040033 |      0.0 | 0.060351 |      0.0 | 0.010505 |  0.110889 |  0.114832 |  0.114832 | 0.031898 | 0.0 | 180.639432  284.228632 | 34.379768  38.300168 | 0.0 |       0.0 |  0.780817 |  6.004956 |   9.44909 | 0.031898 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   215.2128    322.8192 | 0.902686 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.038081 |      0.0 |  0.05975 |      0.0 | 0.010493 |  0.108324 |   0.11211 |   0.11211 | 0.031142 | 0.0 | 171.793109  275.382309 | 43.419691  47.436891 | 0.0 |       0.0 |  0.742736 |  5.945205 |  9.438597 | 0.031142 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   215.6968    323.5452 | 0.902686 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |      0.0 | 0.0 |       0.0 |      0.0 |      0.0 | 0.036224 |      0.0 | 0.059156 |      0.0 | 0.010482 |  0.105861 |  0.109497 |  0.109497 | 0.030416 | 0.0 | 162.946786  266.535986 | 52.750014  57.009214 | 0.0 |       0.0 |  0.706512 |   5.88605 |  9.428115 | 0.030416 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 215.740648      327.03 | 0.902686 |  2.904 | 0.0 |  2.904 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  1.139676 |  1.139676 | 0.0 | 0.0 | 1.139676 | 0.0 |       0.0 |      0.0 | 1.139676 |  0.06248 |      0.0 | 0.058567 |      0.0 |  0.01047 |  0.131517 |  0.108987 |  0.108987 | 0.030274 | 0.0 | 154.100463  257.689663 | 61.640185  69.340337 | 0.0 |       0.0 |  1.783708 |  5.827483 |  9.417645 | 0.030274 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 203.355796    327.6108 | 0.902686 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.386026 |  6.386026 | 0.0 | 0.0 |      2.0 | 0.0 |       0.0 | 0.417386 |      2.0 | 0.136169 |      0.0 | 0.057984 |      0.0 | 0.010458 |  0.621998 |   0.15578 |   0.15578 | 0.043272 | 0.0 |  145.25414   248.84334 | 58.101656   78.76746 | 0.0 |   3.96864 |  3.647539 |  5.769498 |  9.407187 | 0.043272 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 190.970944    327.9012 | 0.902686 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.289226 |  6.289226 | 0.0 | 0.0 |      2.0 | 0.0 |       0.0 |  0.78584 |      2.0 |  0.22707 |      0.0 | 0.057407 |      0.0 | 0.010447 |  1.080764 |  0.442759 |  0.442759 | 0.122989 | 0.0 | 136.407817  239.997017 | 54.563127  87.904183 | 0.0 |  7.472027 |  5.420469 |  5.712091 |   9.39674 | 0.122989 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 178.586092  323.610972 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |   8.33754 |   8.33754 | 0.0 | 0.0 |      2.0 | 0.0 |       0.0 | 1.314154 |      2.0 | 0.313536 |      0.0 | 0.056836 |      0.0 | 0.010435 |  1.694962 |  0.887415 |  0.887415 | 0.246504 | 0.0 | 127.561495  231.150695 | 51.024598  92.460278 | 0.0 | 12.495412 |  7.106933 |  5.655254 |  9.386305 | 0.246504 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  166.20124   311.22612 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  1.133296 | 2.177345 |      2.0 | 0.395786 |      0.0 | 0.056271 |      0.0 | 0.010423 |  3.773121 |  1.537689 |  1.537689 | 0.427136 | 0.0 | 118.715172  222.304372 | 47.486069  88.921749 | 0.0 | 19.569624 |  8.711147 |  5.598984 |  9.375882 | 0.427136 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 153.816388  298.841268 | 0.902686 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.747852 | 12.747852 | 0.0 | 0.0 |      2.0 | 0.0 |   4.05961 | 2.885089 |      2.0 | 0.474025 |      0.0 | 0.055711 |      0.0 | 0.010412 |  7.484847 |   3.09303 |   3.09303 | 0.859175 | 0.0 | 109.868849  213.458049 |  43.94754   85.38322 | 0.0 | 23.372776 | 10.237122 |  5.543273 |   9.36547 | 0.859175 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 141.431536  286.456416 | 0.902686 |  3.146 | 0.0 |  3.146 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 15.530852 | 15.530852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.651059 | 3.511845 | 1.988144 | 0.548156 | 0.010539 | 0.055209 | 0.001317 | 0.010401 |  10.77667 |  5.952973 |  5.952973 | 1.653604 | 0.0 | 101.022526  204.611726 |  40.40901   81.84469 | 0.0 | 26.740724 |  11.67711 |  5.498603 |  9.356386 | 1.653604 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 129.046684  274.071564 | 0.902686 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.231852 | 13.231852 | 0.0 | 0.0 |      2.0 | 0.0 |  7.071658 | 3.613568 | 1.916144 | 0.616614 | 0.074538 | 0.055083 | 0.009317 | 0.010395 | 11.367319 |  9.082014 |  9.082014 | 2.522782 | 0.0 |  92.176203  195.765403 | 36.870481  78.306161 | 0.0 |  27.28735 |  12.97664 |  5.518057 |  9.355308 | 2.522782 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 116.661833  261.686713 | 0.902686 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.747852 | 12.747852 | 0.0 | 0.0 |      2.0 | 0.0 |  7.096299 | 3.619528 | 1.851168 | 0.678396 | 0.132295 | 0.055565 | 0.016537 | 0.010398 | 11.460186 | 10.810681 | 10.810681 | 3.002967 | 0.0 |   83.32988   186.91908 | 33.331952  74.767632 | 0.0 | 27.319375 | 14.149412 |  5.594788 |  9.361447 | 3.002967 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 104.276981  249.301861 | 0.902686 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.747852 | 12.747852 | 0.0 | 0.0 |      2.0 | 0.0 |    7.1089 | 3.622576 | 1.792529 | 0.734151 | 0.184418 | 0.056588 | 0.023052 | 0.010409 | 11.532623 | 11.341437 | 11.341437 | 3.150399 | 0.0 |  74.483558  178.072758 | 29.793423  71.229103 | 0.0 | 27.335751 | 15.207791 |  5.722618 |  9.374091 | 3.150399 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  91.892129  236.917009 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.972514 |  3.58959 |  1.73961 | 0.784467 | 0.231457 | 0.058094 | 0.028932 | 0.010426 | 11.415092 | 11.476761 | 11.476761 | 3.187989 | 0.0 |  65.637235  169.226435 | 26.254894  67.690574 | 0.0 | 27.158499 | 16.162935 |  5.895981 |  9.392597 | 3.187989 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  79.507277  224.532157 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.902771 | 3.572722 | 1.691853 | 0.829876 | 0.273908 | 0.060031 | 0.034239 | 0.010449 | 11.375849 | 11.454958 | 11.454958 | 3.181933 | 0.0 |  56.790912  160.380112 | 22.716365  64.152045 | 0.0 | 27.067858 | 17.024912 |  6.109858 |  9.416386 | 3.181933 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  67.122425  212.147305 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.867106 | 3.564097 | 1.648754 | 0.870855 | 0.312218 |  0.06235 | 0.039027 | 0.010479 | 11.374887 | 11.401475 | 11.401475 | 3.167076 | 0.0 |  47.944589  151.533789 | 19.177836  60.613516 | 0.0 | 27.021507 | 17.802811 |  6.359726 |  9.444935 | 3.167076 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  54.737573  199.762453 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.848869 | 3.559686 | 1.609859 | 0.907837 | 0.346792 | 0.065009 | 0.043349 | 0.010513 | 11.391913 | 11.380835 | 11.380835 | 3.161343 | 0.0 |  39.098266  142.687466 | 15.639306  57.074986 | 0.0 | 26.997805 | 18.504833 |  6.641509 |  9.477771 | 3.161343 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  42.352721  187.377601 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.839542 |  3.55743 | 1.574758 | 0.941212 | 0.377993 | 0.067968 | 0.047249 | 0.010551 | 11.416704 | 11.386952 | 11.386952 | 3.163042 | 0.0 |  30.251943  133.841143 | 12.100777  53.536457 | 0.0 | 26.985684 | 19.138379 |  6.951534 |  9.514469 | 3.163042 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  29.967869  174.992749 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.834773 | 3.556277 | 1.543081 | 0.971332 |  0.40615 | 0.071193 | 0.050769 | 0.010594 | 11.444169 | 11.406695 | 11.406695 | 3.168526 | 0.0 |  21.405621  124.994821 |  8.562248  49.997928 | 0.0 | 26.979486 | 19.710129 |  7.286491 |  9.554644 | 3.168526 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  17.583017  162.607897 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.832335 | 3.555687 | 1.514494 | 0.998513 | 0.431561 | 0.074652 | 0.053945 |  0.01064 | 11.471828 | 11.432175 | 11.432175 | 3.175604 | 0.0 |  12.559298  116.148498 |  5.023719  46.459399 | 0.0 | 26.976316 | 20.226109 |    7.6434 |  9.597949 | 3.175604 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   5.198165  150.223045 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |      2.0 | 0.0 |  6.831087 | 3.555385 | 1.488695 | 1.023044 | 0.454494 | 0.078318 | 0.056812 |  0.01069 | 11.498525 | 11.459368 | 11.459368 | 3.183158 | 0.0 |   3.712975  107.302175 |   1.48519   42.92087 | 0.0 | 26.974696 |  20.69176 |  8.019576 |   9.64407 | 3.183158 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  137.838193 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.846323 | 3.712975  8.846323 | 0.0   0.0 | 17.637831 | 17.637831 | 0.0 | 0.0 |      2.0 | 0.0 |  8.897336 | 4.055118 | 1.465412 | 1.045181 | 0.475189 | 0.082164 | 0.059399 | 0.010743 | 14.090542 | 11.669538 | 11.669538 | 3.241538 | 0.0 |        0.0   98.455852 |       0.0  39.382341 | 0.0 | 29.660073 |  21.11199 |  8.412601 |  9.692726 | 3.241538 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  125.447722 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  8.933975 | 4.063979 |   1.4444 |  1.06516 | 0.493866 | 0.086168 | 0.061733 | 0.010798 |  14.16008 | 12.983827 | 12.983827 | 3.606619 | 0.0 |        0.0   89.605516 |       0.0  35.842206 | 0.0 |  29.70769 | 21.491231 |    8.8203 |  9.743662 | 3.606619 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  113.057251 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  8.952711 | 4.068511 | 1.425438 | 1.083189 | 0.510721 | 0.090309 |  0.06384 | 0.010856 | 14.205575 | 13.918581 | 13.918581 | 3.866273 | 0.0 |        0.0    80.75518 |       0.0  32.302072 | 0.0 |  29.73204 | 21.833481 |  9.240712 |  9.796646 | 3.866273 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  100.666781 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  8.962292 | 4.070828 | 1.408326 |  1.09946 | 0.525932 | 0.094568 | 0.065742 | 0.010916 | 14.238063 | 14.157479 | 14.157479 | 3.932633 | 0.0 |        0.0   71.904843 |       0.0  28.761937 | 0.0 | 29.744492 | 22.142347 |  9.672077 |  9.851472 | 3.932633 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    88.27631 | 0.903096 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 16.618572 | 16.618572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.586119 | 4.221704 | 1.392883 | 1.114144 |  0.53966 | 0.098928 | 0.067457 | 0.010977 | 15.031872 | 14.274166 | 14.274166 | 3.965046 | 0.0 |        0.0   63.054507 |       0.0  25.221803 | 0.0 | 30.555241 | 22.421085 | 10.112809 |  9.907952 | 3.965046 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   75.885839 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.286196 | 4.149166 | 1.378946 | 1.127395 | 0.552048 | 0.103375 | 0.069006 | 0.011041 | 14.677174 | 14.663513 | 14.663513 | 4.073198 | 0.0 |        0.0   54.204171 |       0.0  21.681668 | 0.0 | 30.165451 | 22.672636 | 10.561482 |  9.965917 | 4.073198 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   63.495368 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.132826 | 4.112072 | 1.366368 | 1.139354 | 0.563228 | 0.107895 | 0.070404 | 0.011106 | 14.503254 | 14.750419 | 14.750419 | 4.097339 | 0.0 |        0.0   45.353835 |       0.0  18.141534 | 0.0 | 29.966124 | 22.899649 | 11.016815 | 10.025214 | 4.097339 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   51.104898 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.054397 | 4.093104 | 1.355018 | 1.150147 | 0.573318 | 0.112476 | 0.071665 | 0.011173 | 14.421297 | 14.593471 | 14.593471 | 4.053742 | 0.0 |        0.0   36.503498 |       0.0  14.601399 | 0.0 | 29.864195 |  23.10452 | 11.477657 | 10.085706 | 4.053742 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   38.714427 | 0.903096 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.892572 | 15.892572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.347559 | 4.164007 | 1.344774 | 1.159887 | 0.582423 | 0.117107 | 0.072803 | 0.011241 |   14.7998 | 14.500555 | 14.500555 | 4.027932 | 0.0 |        0.0   27.653162 |       0.0  11.061265 | 0.0 |   30.2452 | 23.289407 | 11.942973 | 10.147269 | 4.027932 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   26.323956 | 0.903096 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.529572 | 15.529572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.354644 |  4.16572 |  1.33553 | 1.168676 |  0.59064 | 0.121778 |  0.07383 | 0.011309 | 14.822128 | 14.648155 | 14.648155 | 4.068932 | 0.0 |        0.0   18.802826 |       0.0    7.52113 | 0.0 | 30.254408 | 23.456261 | 12.411835 | 10.209789 | 4.068932 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   13.933485 | 0.903096 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.166572 | 15.166572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.215437 | 4.132053 | 1.327187 | 1.176609 | 0.598056 |  0.12648 | 0.074757 | 0.011379 | 14.661958 | 14.770751 | 14.770751 | 4.102986 | 0.0 |        0.0     9.95249 |       0.0   3.980996 | 0.0 |  30.07349 | 23.606839 | 12.883411 | 10.273167 | 4.102986 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    1.543015 | 0.903096 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.529572 | 15.529572 | 0.0 | 0.0 |      2.0 | 0.0 |  9.287081 |  4.14938 | 1.319658 | 1.183768 | 0.604748 | 0.131206 | 0.075594 |  0.01145 | 14.762884 |  14.73226 |  14.73226 | 4.092295 | 0.0 |        0.0    1.102153 |       0.0   0.440861 | 0.0 |   30.1666 | 23.742729 | 13.356954 |  10.33731 | 4.092295 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0  1.102153 | 0.0   0.0 |  18.47218 |  18.47218 | 0.0 | 0.0 |      2.0 | 0.0 | 10.481543 | 4.438267 | 1.312864 | 1.190228 | 0.610788 | 0.135948 | 0.076348 | 0.011522 | 16.257508 | 14.839116 | 14.839116 | 4.121977 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |  31.71897 | 23.865365 | 13.831794 | 10.402137 | 4.121977 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 10.788789 | 4.512576 | 1.306732 | 1.196058 | 0.616238 |   0.1407 |  0.07703 | 0.011594 | 16.649716 | 15.639078 | 15.639078 | 4.344188 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.118278 | 23.976038 | 14.307333 | 10.467572 | 4.344188 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 10.945904 | 4.550575 | 1.301198 |  1.20132 | 0.621157 | 0.145456 | 0.077645 | 0.011667 | 16.854922 |  16.37188 |  16.37188 | 4.547744 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.322472 | 24.075917 | 14.783034 | 10.533549 | 4.547744 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.026248 | 4.570006 | 1.296204 | 1.206068 | 0.625596 | 0.150211 |   0.0782 | 0.011741 | 16.964275 | 16.727843 | 16.727843 | 4.646623 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |  32.42689 | 24.166053 | 15.258419 | 10.600008 | 4.646623 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.067333 | 4.579943 | 1.291697 | 1.210353 | 0.629602 | 0.154961 |   0.0787 | 0.011815 | 17.024406 | 16.899687 | 16.899687 | 4.694357 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.480286 | 24.247397 |  15.73306 | 10.666893 | 4.694357 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.088343 | 4.585024 |  1.28763 |  1.21422 | 0.633218 | 0.159702 | 0.079152 | 0.011889 | 17.059179 | 16.989678 | 16.989678 | 4.719355 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.507591 | 24.320807 | 16.206576 | 10.734156 | 4.719355 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.099087 | 4.587623 |  1.28396 |  1.21771 |  0.63648 |  0.16443 |  0.07956 | 0.011964 | 17.080814 | 17.039595 | 17.039595 | 4.733221 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.521554 | 24.387056 | 16.678626 | 10.801752 | 4.733221 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.104581 | 4.588952 | 1.280647 |  1.22086 | 0.639425 | 0.169142 | 0.079928 |  0.01204 | 17.095574 | 17.069038 | 17.069038 | 4.741399 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.528694 | 24.446843 | 17.148909 |  10.86964 | 4.741399 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 |  11.10739 | 4.589631 | 1.277658 | 1.223702 | 0.642082 | 0.173834 |  0.08026 | 0.012115 | 17.106673 | 17.087861 | 17.087861 | 4.746628 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.532345 | 24.500799 | 17.617157 | 10.937785 | 4.746628 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.108827 | 4.589979 |  1.27496 | 1.226267 |  0.64448 | 0.178505 |  0.08056 | 0.012191 | 17.115769 | 17.101106 | 17.101106 | 4.750307 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.534212 | 24.549492 | 18.083132 | 11.006154 | 4.750307 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.109562 | 4.590156 | 1.272525 | 1.228582 | 0.646644 | 0.183153 | 0.080831 | 0.012267 |  17.12372 | 17.111361 | 17.111361 | 4.753156 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.535167 | 24.593435 | 18.546623 | 11.074718 | 4.753156 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.109937 | 4.590247 | 1.270328 | 1.230671 | 0.648597 | 0.187774 | 0.081075 | 0.012343 | 17.130973 |  17.11996 |  17.11996 | 4.755544 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.535655 | 24.633092 | 19.007446 | 11.143449 | 4.755544 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |      2.0 | 0.0 | 11.110129 | 4.590294 | 1.268345 | 1.232557 |  0.65036 | 0.192368 | 0.081295 |  0.01242 | 17.137768 | 17.127597 | 17.127597 | 4.757666 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 32.535905 |  24.66888 | 19.465437 | 11.212324 | 4.757666 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96p_multiple_zones:

multiple zones
______________

This example focuses on the spatial configuration of |hland_96p|.  The most marked
difference to |hland_96| is the separate modelling of the faster runoff components
(|RS|, |RI|, |RG1|) for each zone.  Hence, there are five distinct values for each
runoff component in the following table.  In contrast, the table of the corresponding
:ref:`hland_96_multiple_zones` example of |hland_96| contains only a single value for
the only fast runoff component |hland_fluxes.Q0| of |hland_96|:

.. integration-test::

    >>> name2value = {par.name: par.value for par in control}
    >>> nmbzones(5)
    >>> sclass(1)
    >>> area(15.0)
    >>> zonearea(5.0, 4.0, 3.0, 2.0, 1.0)
    >>> zonetype(FIELD, FOREST, GLACIER, ILAKE, SEALED)
    >>> sfdist(1.0)
    >>> for name, value in name2value.items():
    ...     if name not in ("nmbzones", "sclass", "area", "zonearea", "zonetype", "sfdist"):
    ...         control[name].value = value
    >>> model.add_aetmodel_v1.update(model, model.aetmodel, refresh=True)
    >>> model.aetmodel.add_petmodel_v1.update(model.aetmodel, model.aetmodel.petmodel, refresh=True)
    >>> aetcontrol = model.aetmodel.parameters.control
    >>> aetcontrol.temperaturethresholdice(0.0)
    >>> aetcontrol.soilmoisturelimit(0.8)
    >>> aetcontrol.excessreduction(0.5)
    >>> petcontrol = model.aetmodel.petmodel.parameters.control
    >>> petcontrol.evapotranspirationfactor(0.7)
    >>> petcontrol.airtemperaturefactor(0.1)
    >>> petcontrol.altitudefactor(-0.1)
    >>> petcontrol.precipitationfactor(0.1)
    >>> inputs.t.series = t_series
    >>> petinputs.normalairtemperature.series = tn_series
    >>> test("hland_96p_multiple_zones",
    ...      axis1=(inputs.p, fluxes.rt, fluxes.qt),
    ...      axis2=inputs.t)
    |        date |    p |    t |                           tc |                     fracrain |                     rfc |                     sfc |                                       cfact |                     swe |                          gact |                                     pc |                                     ei |                                              tf |                     spl |                     wcl |                     spg |                     wcg |                           glmelt |                     melt |                     refr |                                                in_ |                                                  r |                            sr |                                ea |                                dp |                           el |                                      rs |                                     ri |                                    gr1 |                                    rg1 |      gr2 |      rg2 |      gr3 |      rg3 |      inrc |     outrc |        rt |        qt |                                     ic |                      sp |                      wc |                                    sm |                                       suz |                                       sg1 |       sg2 |       sg3 |    outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.760624     0.0        0.0 |       0.0        0.0  18.760624     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.048465  0.048465  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.077544  0.0 |      0.0       0.0        0.0  0.0  0.0 |      0.0       0.0  1.594984  0.0  0.0 |      0.0       0.0       2.0  0.0  0.0 | 0.487706  0.487706  0.536883  0.0  0.0 |      0.0 | 0.099502 |      0.0 | 0.011105 |   0.82223 |  0.218798 |  0.218798 |  0.911656 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.951535   99.951535  0.0  0.0  0.0 |       0.0        0.0   15.16564  0.0  0.0 |  9.512294   9.512294  11.463117  0.0  0.0 |  9.890651 |  9.987664 |  0.911656 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0  20.0  20.0  20.0  20.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.137109  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.137109     0.0        0.0 |       0.0        0.0  17.137109     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.047044  0.047044  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.075307  0.0 |      0.0       0.0   4.053816  0.0  0.0 |      0.0       0.0  2.883688  0.0  0.0 |      0.0       0.0  1.926844  0.0  0.0 |  0.46392   0.46392  0.606441  0.0  0.0 | 0.013934 | 0.098483 | 0.001742 | 0.011092 |  1.889411 |  0.620336 |  0.620336 |  2.584734 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.904491   99.904491  0.0  0.0  0.0 |       0.0        0.0  23.365245  0.0  0.0 |  9.048374   9.048374   12.78352  0.0  0.0 |   9.79654 |  9.977118 |  2.584734 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132     0.0        0.0 |       0.0        0.0  16.686132     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.046628  0.046628  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.074675  0.0 |      0.0       0.0   7.102663  0.0  0.0 |      0.0       0.0  3.621067  0.0  0.0 |      0.0       0.0  1.860824  0.0  0.0 | 0.441294  0.441294  0.669214  0.0  0.0 |  0.02651 | 0.097609 | 0.003314 | 0.011081 |   2.64481 |  1.432838 |  1.432838 |  5.970157 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.857863   99.857863  0.0  0.0  0.0 |       0.0        0.0  27.327646  0.0  0.0 |   8.60708    8.60708   13.97513  0.0  0.0 |  9.715958 |  9.968165 |  5.970157 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   16.14496  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   16.14496     0.0        0.0 |       0.0        0.0   16.14496     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.04613   0.04613  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.073912  0.0 |      0.0       0.0   8.448812  0.0  0.0 |      0.0       0.0   3.94664  0.0  0.0 |      0.0       0.0  1.801244  0.0  0.0 | 0.419772  0.419772  0.725865  0.0  0.0 | 0.037858 | 0.096864 | 0.004732 | 0.011072 |  2.976867 |  2.250557 |  2.250557 |  9.377319 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.811734   99.811734  0.0  0.0  0.0 |       0.0        0.0  29.077154  0.0  0.0 |  8.187308   8.187308  15.050508  0.0  0.0 |  9.647567 |  9.960652 |  9.377319 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132     0.0        0.0 |       0.0        0.0  16.686132     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.046584  0.046584  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.074675  0.0 |      0.0       0.0   9.350124  0.0  0.0 |      0.0       0.0  4.164627  0.0  0.0 |      0.0       0.0  1.747475  0.0  0.0 |   0.3993    0.3993   0.77699  0.0  0.0 |   0.0481 | 0.096235 | 0.006013 | 0.011065 |  3.198074 |  2.777077 |  2.777077 | 11.571155 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   99.76515    99.76515  0.0  0.0  0.0 |       0.0        0.0  30.248534  0.0  0.0 |  7.788008   7.788008  16.020993  0.0  0.0 |  9.589949 |  9.954415 | 11.571155 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1  23.1  23.1  23.1  23.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.933163  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.933163     0.0        0.0 |       0.0        0.0  19.933163     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.049338  0.049338  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.079126  0.0 |      0.0       0.0  11.088634  0.0  0.0 |      0.0       0.0  4.585095  0.0  0.0 |      0.0       0.0   1.69895  0.0  0.0 | 0.379826  0.379826  0.823128  0.0  0.0 | 0.057343 | 0.095707 | 0.007168 | 0.011058 |  3.626915 |  3.098236 |  3.098236 | 12.909319 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.715812   99.715812  0.0  0.0  0.0 |       0.0        0.0  32.507969  0.0  0.0 |  7.408182   7.408182  16.896816  0.0  0.0 |  9.541537 |  9.949269 | 12.909319 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7  25.7  25.7  25.7  25.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.278241  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.278241     0.0        0.0 |       0.0        0.0  22.278241     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.140081  0.140081  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.224769  0.0 |      0.0       0.0  12.900368  0.0  0.0 |      0.0       0.0  5.023272  0.0  0.0 |      0.0       0.0  1.655159  0.0  0.0 | 0.361301  0.361301  0.864765  0.0  0.0 | 0.065684 | 0.095267 |  0.00821 | 0.011053 |  4.073694 |  3.453036 |  3.453036 |  14.38765 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   99.57573    99.57573  0.0  0.0  0.0 |       0.0        0.0   34.86257  0.0  0.0 |  7.046881   7.046881   17.68721  0.0  0.0 |  9.483412 |  9.942858 |  14.38765 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9  28.9  28.9  28.9  28.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  25.164491  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  25.164491     0.0        0.0 |       0.0        0.0  25.164491     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.92625   0.92625  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.488314  0.0 |      0.0       0.0  14.962482  0.0  0.0 |      0.0       0.0  5.522005  0.0  0.0 |      0.0       0.0   1.61564  0.0  0.0 |  0.34368   0.34368  0.902342  0.0  0.0 | 0.073212 | 0.094726 | 0.009151 | 0.011047 |  4.582295 |  3.880808 |  3.880808 | 16.170033 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.649481   98.649481  0.0  0.0  0.0 |       0.0        0.0  37.542573  0.0  0.0 |    6.7032     6.7032  18.400508  0.0  0.0 |  9.272905 |  9.917339 | 16.170033 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4  28.4  28.4  28.4  28.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  24.713515  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  24.713515     0.0        0.0 |       0.0        0.0  24.713515     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  2.07444   2.07444  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  3.364543  0.0 |      0.0       0.0  15.839536  0.0  0.0 |      0.0       0.0  5.734125  0.0  0.0 |      0.0       0.0  1.579975  0.0  0.0 | 0.326919  0.326919  0.936253  0.0  0.0 | 0.080005 | 0.092666 | 0.010001 | 0.011019 |  4.794906 |  4.338447 |  4.338447 | 18.076864 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  96.575041   96.575041  0.0  0.0  0.0 |       0.0        0.0  38.682427  0.0  0.0 |  6.376282   6.376282   19.04423  0.0  0.0 |     8.833 |  9.862915 | 18.076864 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0  32.0  32.0  32.0  32.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.960546  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.960546     0.0        0.0 |       0.0        0.0  27.960546     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.400961  3.400961  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  5.634518  0.0 |      0.0       0.0  17.565641  0.0  0.0 |      0.0       0.0  6.151592  0.0  0.0 |      0.0       0.0  1.547789  0.0  0.0 | 0.310975  0.310975  0.966856  0.0  0.0 | 0.086136 | 0.088319 | 0.010767 | 0.010959 |  5.216062 |  4.687141 |  4.687141 | 19.529752 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   93.17408    93.17408  0.0  0.0  0.0 |       0.0        0.0  40.925739  0.0  0.0 |  6.065307   6.065307  19.625163  0.0  0.0 |  8.115322 |  9.773287 | 19.529752 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8  32.8  32.8  32.8  32.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  28.682108  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  28.682108     0.0        0.0 |       0.0        0.0  28.682108     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.705742  3.705742  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  6.363559  0.0 |      0.0       0.0  18.732228  0.0  0.0 |      0.0       0.0  6.433737  0.0  0.0 |      0.0       0.0  1.518742  0.0  0.0 | 0.295808  0.295808  0.994474  0.0  0.0 | 0.091668 | 0.081206 | 0.011459 |  0.01086 |  5.495501 |   5.03283 |   5.03283 | 20.970125 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  89.468338   89.468338  0.0  0.0  0.0 |       0.0        0.0  42.441882  0.0  0.0 |  5.769498   5.769498   20.14943  0.0  0.0 |  7.317714 |  9.672877 | 20.970125 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8  35.8  35.8  35.8  35.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  31.387967  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  31.387967     0.0        0.0 |       0.0        0.0  31.387967     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  4.03435   4.03435  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  7.214798  0.0 |      0.0       0.0  20.393457  0.0  0.0 |      0.0       0.0  6.835514  0.0  0.0 |      0.0       0.0  1.492528  0.0  0.0 | 0.281382  0.281382  1.019398  0.0  0.0 | 0.096661 | 0.073294 | 0.012083 | 0.010748 |  5.896943 |  5.367165 |  5.367165 | 22.363188 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  85.433988   85.433988  0.0  0.0  0.0 |       0.0        0.0  44.600879  0.0  0.0 |  5.488116   5.488116  20.622561  0.0  0.0 |  6.424916 |  9.559691 | 22.363188 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7  37.7  37.7  37.7  37.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  33.101678  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  33.101678     0.0        0.0 |       0.0        0.0  33.101678     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.107634  2.107634  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  3.947157  0.0 |      0.0       0.0  21.917248  0.0  0.0 |      0.0       0.0  7.204051  0.0  0.0 |      0.0       0.0  1.468872  0.0  0.0 | 0.267659  0.267659  1.041891  0.0  0.0 | 0.101167 | 0.064433 | 0.012646 | 0.010623 |  6.263286 |  5.723857 |  5.723857 | 23.849404 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  83.326354   83.326354  0.0  0.0  0.0 |       0.0        0.0  46.581258  0.0  0.0 |  5.220458   5.220458  21.049541  0.0  0.0 |  5.960424 |   9.49906 | 23.849404 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8  31.8  31.8  31.8  31.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.780155  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.780155     0.0        0.0 |       0.0        0.0  27.780155     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.656691  2.656691  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  5.101273  0.0 |      0.0       0.0  20.602611  0.0  0.0 |      0.0       0.0  6.886099  0.0  0.0 |      0.0       0.0  1.447523  0.0  0.0 | 0.254605  0.254605  1.062191  0.0  0.0 | 0.105234 | 0.059832 | 0.013154 | 0.010556 |  5.928638 |  6.046293 |  6.046293 | 25.192888 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  80.669664   80.669664  0.0  0.0  0.0 |       0.0        0.0  44.872703  0.0  0.0 |  4.965853   4.965853  21.434873  0.0  0.0 |  5.358046 |  9.420686 | 25.192888 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9  24.9  24.9  24.9  24.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.556679  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.556679     0.0        0.0 |       0.0        0.0  21.556679     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.871924  2.871924  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  5.696167  0.0 |      0.0       0.0    17.4816  0.0  0.0 |      0.0       0.0  6.131266  0.0  0.0 |      0.0       0.0  1.428256  0.0  0.0 | 0.242188  0.242188   1.08051  0.0  0.0 | 0.108904 | 0.053856 | 0.013613 | 0.010469 |  5.144025 |  5.981326 |  5.981326 | 24.922194 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   77.79774    77.79774  0.0  0.0  0.0 |       0.0        0.0  40.816516  0.0  0.0 |  4.723666   4.723666   21.78262  0.0  0.0 |   4.68977 |  9.333414 | 24.922194 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0  26.0  26.0  26.0  26.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.242   0.242   0.242   0.242   0.242 |    0.242     0.242  0.0  0.0     0.242 |       0.0        0.0   0.242   0.242        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.548827  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.790827   0.242        0.0 |       0.0        0.0  22.790827   0.242        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.873977  0.873977  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.797434  0.0 |      0.0       0.0  16.371214  0.0  0.0 |      0.0       0.0  5.862714  0.0  0.0 |      0.0       0.0  1.410869  0.0  0.0 | 0.230376  0.230376  1.097042  0.0  0.0 | 0.142946 | 0.047376 | 0.017868 | 0.010375 |   4.85832 |  5.483184 |  5.483184 |   22.8466 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.923763   76.923763  0.0  0.0  0.0 |       0.0        0.0  39.373416  0.0  0.0 |   4.49329    4.49329  22.096447  0.0  0.0 |  4.557094 |  9.312377 |   22.8466 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5  26.5  26.5  26.5  26.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.999804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.999804     0.0        0.0 |       0.0        0.0  22.999804     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.677397  0.677397  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.408972  0.0 |      0.0       0.0  15.885624  0.0  0.0 |      0.0       0.0  5.745271  0.0  0.0 |      0.0       0.0  1.395178  0.0  0.0 |  0.21914   0.21914  1.111962  0.0  0.0 | 0.115204 | 0.045918 | 0.014401 | 0.010349 |  4.732572 |  5.039505 |  5.039505 | 20.997935 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.246366   76.246366  0.0  0.0  0.0 |       0.0        0.0  38.742324  0.0  0.0 |  4.274149   4.274149  22.379663  0.0  0.0 |  4.447463 |  9.294064 | 20.997935 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3  24.3  24.3  24.3  24.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.015507  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.015507     0.0        0.0 |       0.0        0.0  21.015507     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.417048  0.417048  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.875158  0.0 |      0.0       0.0  14.856549  0.0  0.0 |      0.0       0.0  5.496384  0.0  0.0 |      0.0       0.0  1.381017  0.0  0.0 | 0.208453  0.208453  1.125426  0.0  0.0 | 0.117902 | 0.044841 | 0.014738 | 0.010329 |  4.472235 |  4.799343 |  4.799343 | 19.997261 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  75.829319   75.829319  0.0  0.0  0.0 |       0.0        0.0  37.404898  0.0  0.0 |  4.065697   4.065697  22.635253  0.0  0.0 |  4.409393 |  9.284581 | 19.997261 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2  22.2  22.2  22.2  22.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  1.573   1.573   1.573   1.573   1.573 | 0.494092  0.494092  0.0  0.0  0.494092 |       0.0        0.0   1.573   1.573        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.121405  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  20.694405   1.573        0.0 |       0.0        0.0  20.694405   1.573        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.117083  0.117083  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.494092  0.0 |      0.0       0.0  14.203969  0.0  0.0 |      0.0       0.0  5.338555  0.0  0.0 |      0.0       0.0  1.368237  0.0  0.0 | 0.198286  0.198286  1.137577  0.0  0.0 | 0.320082 | 0.045469 |  0.04001 | 0.010333 |  4.307074 |  4.585727 |  4.585727 | 19.107198 | 1.078908  1.078908  0.0  0.0  1.078908 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  75.712235   75.712235  0.0  0.0  0.0 |       0.0        0.0   36.55678  0.0  0.0 |   3.86741    3.86741  22.865914  0.0  0.0 |  4.621264 |  9.306416 | 19.107198 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  6.776   6.776   6.776   6.776   6.776 |  0.08755   0.08755  0.0  0.0   0.08755 |  5.854908   5.854908   6.776   6.776   5.854908 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.854908   5.854908  25.536624   6.776   5.854908 |  0.839058   0.839058  25.536624   6.776   5.854908 | 0.0  0.0  0.0  0.0   5.854908 | 0.022087  0.022087  0.0  0.0  0.0 | 0.839058  0.839058  2.0  0.0  0.0 | 0.0  0.0  0.0   0.08755  0.0 |      0.0       0.0  15.775525  0.0  0.0 |      0.0       0.0  5.718643  0.0  0.0 | 0.839058  0.839058  1.356704  0.0  0.0 | 0.209247  0.209247  1.148543  0.0  0.0 | 0.982977 | 0.050881 | 0.122872 | 0.010403 |  5.101616 |  4.457543 |  4.457543 | 18.573096 |  1.91245   1.91245  0.0  0.0   1.91245 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  80.705998   80.705998  0.0  0.0  0.0 |       0.0        0.0  38.599235  0.0  0.0 |  4.497222   4.497222  23.074075  0.0  0.0 |  5.542242 |  9.417495 | 18.573096 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0  21.0  21.0  21.0  21.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  3.509   3.509   3.509   3.509   3.509 |   0.0539    0.0539  0.0  0.0    0.0539 |   3.42145    3.42145   3.509   3.509    3.42145 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.039062  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   3.42145    3.42145  21.548062   3.509    3.42145 |  0.557137   0.557137  21.548062   3.509    3.42145 | 0.0  0.0  0.0  0.0    3.42145 | 0.014076  0.014076  0.0  0.0  0.0 | 0.557137  0.557137  2.0  0.0  0.0 | 0.0  0.0  0.0    0.0539  0.0 |      0.0       0.0  15.009792  0.0  0.0 |      0.0       0.0  5.533447  0.0  0.0 | 0.557137  0.557137  1.346296  0.0  0.0 | 0.233031  0.233031  1.158439  0.0  0.0 | 0.570102 | 0.057987 | 0.071263 | 0.010498 |   4.77217 |  4.754437 |  4.754437 | 19.810153 |   1.9461    1.9461  0.0  0.0    1.9461 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  83.556234   83.556234  0.0  0.0  0.0 |       0.0        0.0  37.604058  0.0  0.0 |  4.821327   4.821327  23.261932  0.0  0.0 |  6.047513 |  9.477405 | 19.810153 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4  20.4  20.4  20.4  20.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  5.929   5.929   5.929   5.929   5.929 | 0.041902  0.041902  0.0  0.0  0.041902 |    5.8751     5.8751   5.929   5.929     5.8751 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   17.49789  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    5.8751     5.8751   23.42689   5.929     5.8751 |  1.025446   1.025446   23.42689   5.929     5.8751 | 0.0  0.0  0.0  0.0     5.8751 | 0.011576  0.011576  0.0  0.0  0.0 | 1.025446  1.025446  2.0  0.0  0.0 | 0.0  0.0  0.0  0.041902  0.0 |      0.0       0.0  15.357481  0.0  0.0 |      0.0       0.0  5.617537  0.0  0.0 | 1.025446  1.025446  1.336903  0.0  0.0 | 0.260353  0.260353   1.16737  0.0  0.0 | 0.879193 | 0.064555 | 0.109899 | 0.010586 |  5.046494 |  4.868748 |  4.868748 |  20.28645 | 1.958098  1.958098  0.0  0.0  1.958098 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  88.394311   88.394311  0.0  0.0  0.0 |       0.0        0.0  38.055929  0.0  0.0 |   5.58642    5.58642  23.431465  0.0  0.0 |   6.85683 |  9.576053 |  20.28645 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2  20.2  20.2  20.2  20.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 | 12.826  12.826  12.826  12.826  12.826 | 0.020954  0.020954  0.0  0.0  0.020954 | 12.784098  12.784098  12.826  12.826  12.784098 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.317499  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.784098  12.784098  30.143499  12.826  12.784098 |  2.497231   2.497231  30.143499  12.826  12.784098 | 0.0  0.0  0.0  0.0  12.784098 | 0.006462  0.006462  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.020954  0.0 |      0.0       0.0  18.178059  0.0  0.0 | 0.047318  0.047318  6.299709  0.0  0.0 |      2.0       2.0  1.328427  0.0  0.0 |  0.32163   0.32163   1.17543  0.0  0.0 | 1.756617 |  0.07698 | 0.219577 | 0.010756 |  6.286169 |  5.037327 |  5.037327 | 20.988863 | 1.979046  1.979046  0.0  0.0  1.979046 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.674717   98.674717  0.0  0.0  0.0 |  0.449913   0.449913  41.721661  0.0  0.0 |  7.264791   7.264791  23.584462  0.0  0.0 |  8.533806 |  9.784542 | 20.988863 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.121   0.121   0.121   0.121   0.121 | 0.074152  0.074152  0.0  0.0  0.074152 |  0.100046   0.100046   0.121   0.121   0.100046 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.956718  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.100046   0.100046  17.077718   0.121   0.100046 |  0.024353   0.024353  17.077718   0.121   0.100046 | 0.0  0.0  0.0  0.0   0.100046 | 0.022883  0.022883  0.0  0.0  0.0 | 0.474266  0.474266  2.0  0.0  0.0 | 0.0  0.0  0.0  0.074152  0.0 |      0.0       0.0  14.479428  0.0  0.0 |      0.0       0.0  5.405176  0.0  0.0 | 0.474266  0.474266  1.320777  0.0  0.0 |  0.36597   0.36597  1.182704  0.0  0.0 | 0.144741 | 0.085634 | 0.018093 | 0.010876 |  4.529789 |  5.610127 |  5.610127 | 23.375528 | 1.925848  1.925848  0.0  0.0  1.925848 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.727527   98.727527  0.0  0.0  0.0 |       0.0        0.0  36.914776  0.0  0.0 |  7.373087   7.373087  23.722535  0.0  0.0 |  8.583496 |  9.790581 | 23.375528 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.847   0.847   0.847   0.847   0.847 |  0.06896   0.06896  0.0  0.0   0.06896 |  0.772848   0.772848   0.847   0.847   0.772848 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.772848   0.772848  17.810337   0.847   0.772848 |  0.188326   0.188326  17.810337   0.847   0.772848 | 0.0  0.0  0.0  0.0   0.772848 | 0.021402  0.021402  0.0  0.0  0.0 | 0.188326  0.188326  2.0  0.0  0.0 | 0.0  0.0  0.0   0.06896  0.0 |      0.0       0.0  12.876329  0.0  0.0 |      0.0       0.0  5.017458  0.0  0.0 | 0.188326  0.188326  1.313873  0.0  0.0 |  0.36422   0.36422  1.189268  0.0  0.0 | 0.238246 | 0.086594 | 0.029781 | 0.010889 |  4.177651 |  5.151125 |  5.151125 | 21.463021 |  1.93104   1.93104  0.0  0.0   1.93104 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.290647   99.290647  0.0  0.0  0.0 |       0.0        0.0  34.831327  0.0  0.0 |  7.197193   7.197193   23.84714  0.0  0.0 |  8.726391 |  9.808379 | 21.463021 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |   3.63    3.63    3.63    3.63    3.63 | 0.052207  0.052207  0.0  0.0  0.052207 |   3.56104    3.56104    3.63    3.63    3.56104 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   3.56104    3.56104  20.593337    3.63    3.56104 |  0.877675   0.877675  20.593337    3.63    3.56104 | 0.0  0.0  0.0  0.0    3.56104 | 0.016637  0.016637  0.0  0.0  0.0 | 0.877675  0.877675  2.0  0.0  0.0 | 0.0  0.0  0.0  0.052207  0.0 |      0.0       0.0   13.15158  0.0  0.0 |      0.0       0.0  5.084029  0.0  0.0 | 0.877675  0.877675  1.307643  0.0  0.0 | 0.372592  0.372592  1.195192  0.0  0.0 |  0.59283 | 0.089783 | 0.074104 | 0.010933 |   4.44112 |  4.480379 |  4.480379 | 18.668246 | 1.947793  1.947793  0.0  0.0  1.947793 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 101.957376  101.957376  0.0  0.0  0.0 |       0.0        0.0  35.189054  0.0  0.0 |  7.702276   7.702276  23.959592  0.0  0.0 |  9.222808 |   9.87072 | 18.668246 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  2.541   2.541   2.541   2.541   2.541 | 0.057919  0.057919  0.0  0.0  0.057919 |  2.488793   2.488793   2.541   2.541   2.488793 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.692645  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.488793   2.488793  19.233645   2.541   2.488793 |  0.646794   0.646794  19.233645   2.541   2.488793 | 0.0  0.0  0.0  0.0   2.488793 | 0.018787  0.018787  0.0  0.0  0.0 | 0.646794  0.646794  2.0  0.0  0.0 | 0.0  0.0  0.0  0.057919  0.0 |      0.0       0.0  12.757338  0.0  0.0 |      0.0       0.0  4.988679  0.0  0.0 | 0.646794  0.646794   1.30202  0.0  0.0 | 0.391548  0.391548  1.200538  0.0  0.0 | 0.455615 | 0.094039 | 0.056952 | 0.010993 |  4.288189 |  4.362606 |  4.362606 | 18.177524 | 1.942081  1.942081  0.0  0.0  1.942081 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 103.780587  103.780587  0.0  0.0  0.0 |       0.0        0.0  34.676682  0.0  0.0 |  7.957522   7.957522  24.061074  0.0  0.0 |   9.57703 |   9.91576 | 18.177524 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 12.584  12.584  12.584  12.584  12.584 | 0.021144  0.021144  0.0  0.0  0.021144 | 12.526081  12.526081  12.584  12.584  12.526081 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.526081  12.526081  29.096184  12.584  12.526081 |  3.372776   3.372776  29.096184  12.584  12.526081 | 0.0  0.0  0.0  0.0  12.526081 | 0.007462  0.007462  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.021144  0.0 |      0.0       0.0  16.436342  0.0  0.0 | 0.130637  0.130637  5.878465  0.0  0.0 |      2.0       2.0  1.296946  0.0  0.0 |  0.43727   0.43727  1.205362  0.0  0.0 | 1.731883 | 0.103924 | 0.216485 | 0.011132 |  5.987235 |  4.457923 |  4.457923 | 18.574681 | 1.978856  1.978856  0.0  0.0  1.978856 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  112.92643   112.92643  0.0  0.0  0.0 |  1.242139   1.242139  39.458059  0.0  0.0 |  9.520252   9.520252  24.152658  0.0  0.0 | 11.202304 | 10.120778 | 18.574681 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.235   4.235   4.235   4.235   4.235 | 0.048561  0.048561  0.0  0.0  0.048561 |  4.213856   4.213856   4.235   4.235   4.213856 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  4.213856   4.213856  20.566723   4.235   4.213856 |  1.343417   1.343417  20.566723   4.235   4.213856 | 0.0  0.0  0.0  0.0   4.213856 | 0.017573  0.017573  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.048561  0.0 |      0.0       0.0  14.961586  0.0  0.0 | 0.055723  0.055723  5.521788  0.0  0.0 |      2.0       2.0  1.292367  0.0  0.0 | 0.513485  0.513485  1.209716  0.0  0.0 | 0.672565 | 0.114816 | 0.084071 | 0.011286 |  5.078762 |  5.205849 |  5.205849 | 21.691039 | 1.951439  1.951439  0.0  0.0  1.951439 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.779297  115.779297  0.0  0.0  0.0 |  0.529833   0.529833  37.541408  0.0  0.0 | 11.006767  11.006767  24.235309  0.0  0.0 | 11.753886 | 10.192792 | 21.691039 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.114   4.114   4.114   4.114   4.114 | 0.048983  0.048983  0.0  0.0  0.048983 |  4.065439   4.065439   4.114   4.114   4.065439 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.151262  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  4.065439   4.065439  20.265262   4.114   4.065439 |  1.362415   1.362415  20.265262   4.114   4.065439 | 0.0  0.0  0.0  0.0   4.065439 | 0.018136  0.018136  0.0  0.0  0.0 | 1.892248  1.892248  2.0  0.0  0.0 | 0.0  0.0  0.0  0.048983  0.0 |      0.0       0.0  14.088827  0.0  0.0 |      0.0       0.0  5.310707  0.0  0.0 | 1.892248  1.892248  1.288235  0.0  0.0 | 0.583334  0.583334  1.213646  0.0  0.0 | 0.657987 | 0.120232 | 0.082248 | 0.011365 |  4.866489 |  5.312915 |  5.312915 | 22.137144 | 1.951017  1.951017  0.0  0.0  1.951017 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 118.464185  118.464185  0.0  0.0  0.0 |       0.0        0.0  36.407136  0.0  0.0 | 12.315681  12.315681  24.309898  0.0  0.0 | 12.285421 | 10.262898 | 22.137144 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.452   1.452   1.452   1.452   1.452 | 0.124728  0.124728  0.0  0.0  0.124728 |  1.403017   1.403017   1.452   1.452   1.403017 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.403017   1.403017  17.783723   1.452   1.403017 |   0.49224    0.49224  17.783723   1.452   1.403017 | 0.0  0.0  0.0  0.0   1.403017 | 0.046529  0.046529  0.0  0.0  0.0 |  0.49224   0.49224  2.0  0.0  0.0 | 0.0  0.0  0.0  0.124728  0.0 |      0.0       0.0  12.666116  0.0  0.0 |      0.0       0.0  4.966617  0.0  0.0 |  0.49224   0.49224  1.284505  0.0  0.0 | 0.612746  0.612746  1.217192  0.0  0.0 | 0.320666 |  0.12384 | 0.040083 | 0.011419 |  4.357409 |   4.97986 |   4.97986 | 20.749418 | 1.875272  1.875272  0.0  0.0  1.875272 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 119.328432  119.328432  0.0  0.0  0.0 |       0.0        0.0  34.558126  0.0  0.0 | 12.195174  12.195174  24.377211  0.0  0.0 | 12.466409 | 10.289583 | 20.749418 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.121   0.121   0.121   0.121 |  0.96204   0.96204  0.0  0.0   0.96204 |       0.0        0.0   0.121   0.121        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.723415   0.121        0.0 |       0.0        0.0  16.723415   0.121        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.358746  0.358746  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0   0.96204  0.0 |      0.0       0.0  11.521389  0.0  0.0 |      0.0       0.0  4.689759  0.0  0.0 |      0.0       0.0  1.281139  0.0  0.0 | 0.594766  0.594766  1.220392  0.0  0.0 | 0.152291 | 0.124802 | 0.019036 | 0.011437 |  3.970324 |  4.572586 |  4.572586 |  19.05244 | 1.034232  1.034232  0.0  0.0  1.034232 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 118.969686  118.969686  0.0  0.0  0.0 |       0.0        0.0  33.070394  0.0  0.0 | 11.600409  11.600409  24.437959  0.0  0.0 | 12.371734 | 10.281911 |  19.05244 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 1.034232  1.034232  0.0  0.0  1.034232 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.602415     0.0        0.0 |       0.0        0.0  16.602415     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.527265  1.527265  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  2.345064  0.0 |      0.0       0.0  10.888402  0.0  0.0 |      0.0       0.0  4.536668  0.0  0.0 |      0.0       0.0  1.278102  0.0  0.0 | 0.565759  0.565759   1.22328  0.0  0.0 | 0.137504 | 0.123786 | 0.017188 | 0.011428 |  3.795324 |  4.168372 |  4.168372 | 17.368218 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 117.442421  117.442421  0.0  0.0  0.0 |       0.0        0.0  32.247739  0.0  0.0 |  11.03465   11.03465  24.492781  0.0  0.0 | 12.087666 | 10.250449 | 17.368218 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.782876     0.0        0.0 |       0.0        0.0  16.782876     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.091248  1.091248  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.486684  0.0 |      0.0       0.0  10.635718  0.0  0.0 |      0.0       0.0  4.475555  0.0  0.0 |      0.0       0.0  1.275361  0.0  0.0 | 0.538166  0.538166  1.225886  0.0  0.0 | 0.138026 | 0.120962 | 0.017253 | 0.011393 |  3.713863 |  3.901632 |  3.901632 | 16.256798 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 116.351173  116.351173  0.0  0.0  0.0 |       0.0        0.0  31.919342  0.0  0.0 | 10.496484  10.496484  24.542256  0.0  0.0 | 11.915945 | 10.232711 | 16.256798 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.484   0.484   0.484   0.484   0.484 |    0.484     0.484  0.0  0.0     0.484 |       0.0        0.0   0.484   0.484        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.447337   0.484        0.0 |       0.0        0.0  17.447337   0.484        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.312905  1.312905  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.805438  0.0 |      0.0       0.0  10.767949  0.0  0.0 |      0.0       0.0  4.507535  0.0  0.0 |      0.0       0.0  1.272887  0.0  0.0 |  0.51192   0.51192  1.228238  0.0  0.0 | 0.199958 | 0.119562 | 0.024995 | 0.011377 |  3.730106 |  3.768585 |  3.768585 | 15.702439 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.038267  115.038267  0.0  0.0  0.0 |       0.0        0.0  32.091194  0.0  0.0 |  9.984564   9.984564  24.586904  0.0  0.0 | 11.767079 | 10.217671 | 15.702439 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9  19.9  19.9  19.9  19.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.121   0.121   0.121   0.121 |    0.121     0.121  0.0  0.0     0.121 |       0.0        0.0   0.121   0.121        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.053567  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.174567   0.121        0.0 |       0.0        0.0  17.174567   0.121        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.399893  3.399893  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  4.728713  0.0 |      0.0       0.0  10.728241  0.0  0.0 |      0.0       0.0  4.497932  0.0  0.0 |      0.0       0.0  1.270655  0.0  0.0 | 0.486953  0.486953  1.230361  0.0  0.0 | 0.154288 | 0.117853 | 0.019286 | 0.011357 |  3.704075 |  3.730107 |  3.730107 | 15.542112 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 111.638374  111.638374  0.0  0.0  0.0 |       0.0        0.0  32.039588  0.0  0.0 |  9.497611   9.497611  24.627198  0.0  0.0 | 11.203043 | 10.150541 | 15.542112 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   4.356   4.356   4.356   4.356 | 1.680962  1.680962  0.0  0.0  1.680962 |     2.356      2.356   4.356   4.356      2.356 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |     2.356      2.356  21.138876   4.356      2.356 |  0.734078   0.734078  21.138876   4.356      2.356 | 0.0  0.0  0.0  0.0      2.356 | 0.594957  0.594957  0.0  0.0  0.0 | 0.734078  0.734078  2.0  0.0  0.0 | 0.0  0.0  0.0  1.680962  0.0 |      0.0       0.0   12.26777  0.0  0.0 |      0.0       0.0  4.870275  0.0  0.0 | 0.734078  0.734078   1.26864  0.0  0.0 | 0.481254  0.481254  1.232277  0.0  0.0 |  0.69245 | 0.114923 | 0.086556 |  0.01132 |   4.23771 |  3.752926 |  3.752926 | 15.637192 | 0.319038  0.319038  0.0  0.0  0.319038 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 112.665339  112.665339  0.0  0.0  0.0 |       0.0        0.0   34.04042  0.0  0.0 |  9.750436   9.750436  24.663562  0.0  0.0 | 11.567114 | 10.199095 | 15.637192 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.139   7.139   7.139   7.139   7.139 |      2.0       2.0  0.0  0.0       2.0 |  5.458038   5.458038   7.139   7.139   5.458038 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.458038   5.458038  23.741415   7.139   5.458038 |  1.732037   1.732037  23.741415   7.139   5.458038 | 0.0  0.0  0.0  0.0   5.458038 | 1.873211  1.873211  0.0  0.0  0.0 | 1.732037  1.732037  2.0  0.0  0.0 | 0.0  0.0  0.0  3.326542  0.0 |      0.0       0.0  14.079055  0.0  0.0 |      0.0       0.0  5.308343  0.0  0.0 | 1.732037  1.732037  1.266822  0.0  0.0 | 0.518123  0.518123  1.234005  0.0  0.0 | 1.046193 | 0.120308 | 0.130774 | 0.011399 |   4.92195 |  4.060279 |  4.060279 | 16.917829 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 114.518129  114.518129  0.0  0.0  0.0 |       0.0        0.0  36.394436  0.0  0.0 |  10.96435   10.96435  24.696378  0.0  0.0 |  12.07058 | 10.265668 | 16.917829 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.331   1.331   1.331   1.331   1.331 |    1.331     1.331  0.0  0.0     1.331 |       0.0        0.0   1.331   1.331        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.843184   1.331        0.0 |       0.0        0.0  17.843184   1.331        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.674068  1.674068  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  2.727186  0.0 |      0.0       0.0  12.684515  0.0  0.0 |      0.0       0.0  4.971067  0.0  0.0 |      0.0       0.0  1.265181  0.0  0.0 | 0.534738  0.534738  1.235565  0.0  0.0 | 0.308981 | 0.121644 | 0.038623 | 0.011421 |  4.223267 |  4.535321 |  4.535321 |  18.89717 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 112.844061  112.844061  0.0  0.0  0.0 |       0.0        0.0  34.582038  0.0  0.0 | 10.429613  10.429613  24.725994  0.0  0.0 | 11.911608 | 10.249581 |  18.89717 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 25.047  25.047  25.047  25.047  25.047 | 0.132786  0.132786  0.0  0.0  0.132786 |    23.047     23.047  25.047  25.047     23.047 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   15.70011  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    23.047     23.047   40.74711  25.047     23.047 |  7.336887   7.336887   40.74711  25.047     23.047 | 0.0  0.0  0.0  0.0     23.047 | 0.053344  0.053344  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.132786  0.0 |      0.0       0.0  20.983385  0.0  0.0 | 0.507872  0.507872  6.978191  0.0  0.0 | 1.978519  1.978519    1.2637  0.0  0.0 | 0.557307  0.557307  1.236973  0.0  0.0 | 3.333094 | 0.135133 | 0.416637 | 0.011613 |  8.152247 |   4.74086 |   4.74086 | 19.753584 | 1.867214  1.867214  0.0  0.0  1.867214 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 128.500829  128.500829  0.0  0.0  0.0 |  4.829015   4.829015  45.367572  0.0  0.0 | 11.850825  11.850825  24.752721  0.0  0.0 | 15.092707 | 10.652496 | 19.753584 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0  18.0  18.0  18.0  18.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 45.859  45.859  45.859  45.859  45.859 | 0.017578  0.017578  0.0  0.0  0.017578 | 45.726214  45.726214  45.859  45.859  45.726214 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.339188  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 45.726214  45.726214  61.198188  45.859  45.726214 | 18.876311  18.876311  61.198188  45.859  45.726214 | 0.0  0.0  0.0  0.0  45.726214 | 0.008533  0.008533  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.017578  0.0 | 0.670993  0.670993  33.274034  0.0  0.0 | 2.065535  2.065535  9.950748  0.0  0.0 | 1.907459  1.907459  1.262364  0.0  0.0 | 0.624873  0.624873  1.238244  0.0  0.0 | 6.016748 | 0.180159 | 0.752093 | 0.012247 | 14.137439 |  6.948961 |  6.948961 | 28.954003 | 1.982422  1.982422  0.0  0.0  1.982422 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 155.342199  155.342199  0.0  0.0  0.0 | 18.968797  18.968797  61.340978  0.0  0.0 | 13.133411  13.133411  24.776841  0.0  0.0 | 20.927065 | 11.392063 | 28.954003 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9  17.9  17.9  17.9  17.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  9.922   9.922   9.922   9.922   9.922 | 0.821583  0.821583  0.0  0.0  0.821583 |  9.904422   9.904422   9.922   9.922   9.904422 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.248957  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  9.904422   9.904422  25.170957   9.922   9.904422 |   5.97514    5.97514  25.170957   9.922   9.904422 | 0.0  0.0  0.0  0.0   9.904422 | 0.408921  0.408921  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.821583  0.0 | 1.158349  1.158349  25.383469  0.0  0.0 | 2.183404  2.183404  8.042374  0.0  0.0 | 1.843329  1.843329  1.261158  0.0  0.0 | 0.685849  0.685849  1.239391  0.0  0.0 | 1.490194 | 0.215654 | 0.186274 | 0.012754 | 10.223084 | 10.977571 | 10.977571 | 45.739877 | 1.178417  1.178417  0.0  0.0  1.178417 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 158.862561  158.862561  0.0  0.0  0.0 | 19.602184  19.602184  51.086093  0.0  0.0 | 14.290891  14.290891  24.798608  0.0  0.0 | 22.097277 | 11.552542 | 45.739877 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4  17.4  17.4  17.4  17.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   4.356   4.356   4.356   4.356 | 0.792694  0.792694  0.0  0.0  0.792694 |  3.534417   3.534417   4.356   4.356   3.534417 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.797804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.534417   3.534417  19.153804   4.356   3.534417 |   2.22998    2.22998  19.153804   4.356   3.534417 | 0.0  0.0  0.0  0.0   3.534417 | 0.396347  0.396347  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.792694  0.0 |      0.0       0.0  18.980921  0.0  0.0 |  1.88728   1.88728  6.493885  0.0  0.0 | 1.785455  1.785455   1.26007  0.0  0.0 | 0.740877  0.740877  1.240426  0.0  0.0 | 0.816679 | 0.223941 | 0.102085 | 0.012886 |  7.376607 | 11.161569 | 11.161569 | 46.506536 | 1.207306  1.207306  0.0  0.0  1.207306 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 159.770651  159.770651  0.0  0.0  0.0 | 17.944883  17.944883  42.765092  0.0  0.0 |  15.33547   15.33547  24.818252  0.0  0.0 | 22.589355 | 11.629159 | 46.506536 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  9.075   9.075   9.075   9.075   9.075 | 0.090607  0.090607  0.0  0.0  0.090607 |  8.282306   8.282306   9.075   9.075   8.282306 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.527113  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  8.282306   8.282306  23.602113   9.075   8.282306 |   5.28549    5.28549  23.602113   9.075   8.282306 | 0.0  0.0  0.0  0.0   8.282306 | 0.045304  0.045304  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.090607  0.0 | 0.484114  0.484114  17.457135  0.0  0.0 | 2.020337  2.020337  6.125349  0.0  0.0 | 1.733226  1.733226  1.259087  0.0  0.0 | 0.790537  0.790537  1.241359  0.0  0.0 | 1.445949 | 0.231974 | 0.180744 | 0.013014 |  7.722571 |  8.882188 |  8.882188 | 37.009117 | 1.909393  1.909393  0.0  0.0  1.909393 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 162.722163  162.722163  0.0  0.0  0.0 | 18.725922  18.725922  40.784721  0.0  0.0 |  16.27816   16.27816   24.83598  0.0  0.0 | 23.791825 |  11.79545 | 37.009117 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 22.385  22.385  22.385  22.385  22.385 | 0.007604  0.007604  0.0  0.0  0.007604 | 22.294393  22.294393  22.385  22.385  22.294393 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.346652  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 22.294393  22.294393  36.731652  22.385  22.294393 | 14.758053  14.758053  36.731652  22.385  22.294393 | 0.0  0.0  0.0  0.0  22.294393 | 0.003802  0.003802  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.007604  0.0 | 4.518592  4.518592   21.84399  0.0  0.0 | 2.996096  2.996096  7.186333  0.0  0.0 | 1.686092  1.686092  1.258201  0.0  0.0 | 0.835354  0.835354  1.242202  0.0  0.0 | 3.163211 | 0.252496 | 0.395401 | 0.013318 | 12.798917 |  8.205307 |  8.205307 | 34.188781 | 1.992396  1.992396  0.0  0.0  1.992396 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 170.254701  170.254701  0.0  0.0  0.0 | 23.969287  23.969287  46.486049  0.0  0.0 | 17.128898  17.128898  24.851979  0.0  0.0 | 26.701574 | 12.177412 | 34.188781 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 18.634  18.634  18.634  18.634  18.634 | 0.011044  0.011044  0.0  0.0  0.011044 | 18.626396  18.626396  18.634  18.634  18.626396 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.256421  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 18.626396  18.626396  32.890421  18.634  18.626396 | 13.497927  13.497927  32.890421  18.634  18.626396 | 0.0  0.0  0.0  0.0  18.626396 | 0.005522  0.005522  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.011044  0.0 | 6.085874  6.085874  22.575882  0.0  0.0 | 3.375152  3.375152  7.363345  0.0  0.0 | 1.643555  1.643555  1.257401  0.0  0.0 | 0.875799  0.875799  1.242963  0.0  0.0 | 2.711352 | 0.279197 | 0.338919 | 0.013711 | 13.953673 | 10.711003 | 10.711003 | 44.629178 | 1.988956  1.988956  0.0  0.0  1.988956 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 175.377648  175.377648  0.0  0.0  0.0 | 26.006188  26.006188  47.437244  0.0  0.0 | 17.896654  17.896654  24.866417  0.0  0.0 | 29.132327 | 12.502445 | 44.629178 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1  16.1  16.1  16.1  16.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.623   7.623   7.623   7.623   7.623 | 0.032783  0.032783  0.0  0.0  0.032783 |  7.611956   7.611956   7.623   7.623   7.611956 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.624808  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  7.611956   7.611956  21.247808   7.623   7.611956 |  5.853084   5.853084  21.247808   7.623   7.611956 | 0.0  0.0  0.0  0.0   7.611956 | 0.016392  0.016392  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.032783  0.0 | 3.879321  3.879321  18.369136  0.0  0.0 | 2.841485  2.841485  6.345922  0.0  0.0 | 1.605167  1.605167  1.256679  0.0  0.0 | 0.912299  0.912299  1.243649  0.0  0.0 | 1.335204 | 0.296525 |   0.1669 | 0.013977 |  10.56887 | 12.796724 | 12.796724 | 53.319685 | 1.967217  1.967217  0.0  0.0  1.967217 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.120128  177.120128  0.0  0.0  0.0 | 23.138465  23.138465  41.969993  0.0  0.0 | 18.589523  18.589523  24.879447  0.0  0.0 | 30.166842 | 12.654848 | 53.319685 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  2.299   2.299   2.299   2.299   2.299 | 0.054873  0.054873  0.0  0.0  0.054873 |  2.266217   2.266217   2.299   2.299   2.266217 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.812733  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.266217   2.266217  15.111733   2.299   2.266217 |  1.777368   1.777368  15.111733   2.299   2.266217 | 0.0  0.0  0.0  0.0   2.266217 | 0.027436  0.027436  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.054873  0.0 | 1.147291  1.147291  13.803584  0.0  0.0 |  2.18073   2.18073  5.241719  0.0  0.0 | 1.570524  1.570524  1.256028  0.0  0.0 | 0.945239  0.945239  1.244269  0.0  0.0 |  0.67906 | 0.303549 | 0.084883 |   0.0141 |  7.069424 | 11.622468 | 11.622468 |  48.42695 | 1.945127  1.945127  0.0  0.0  1.945127 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.581541  177.581541  0.0  0.0  0.0 | 19.587812  19.587812  36.036423  0.0  0.0 | 19.214808  19.214808  24.891206  0.0  0.0 | 30.535386 |  12.72476 |  48.42695 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3  15.3  15.3  15.3  15.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  5.929   5.929   5.929   5.929   5.929 | 0.038243  0.038243  0.0  0.0  0.038243 |  5.874127   5.874127   5.929   5.929   5.874127 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   12.90841  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.874127   5.874127   18.83741   5.929   5.874127 |  4.631045   4.631045   18.83741   5.929   5.874127 | 0.0  0.0  0.0  0.0   5.874127 | 0.019122  0.019122  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.038243  0.0 | 0.873052  0.873052  12.934845  0.0  0.0 | 2.114404  2.114404   5.03161  0.0  0.0 |  1.53926   1.53926   1.25544  0.0  0.0 | 0.974965  0.974965  1.244828  0.0  0.0 |  1.15799 | 0.309603 | 0.144749 | 0.014211 |  6.913545 |  8.820601 |  8.820601 | 36.752503 | 1.961757  1.961757  0.0  0.0  1.961757 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.805502  178.805502  0.0  0.0  0.0 | 19.231401  19.231401  34.907377  0.0  0.0 | 19.779102  19.779102  24.901817  0.0  0.0 | 31.378917 |  12.85469 | 36.752503 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  3.267   3.267   3.267   3.267   3.267 |  0.04981   0.04981  0.0  0.0   0.04981 |  3.228757   3.228757   3.267   3.267   3.228757 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.818141  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.228757   3.228757  16.085141   3.267   3.228757 |  2.580697   2.580697  16.085141   3.267   3.228757 | 0.0  0.0  0.0  0.0   3.228757 | 0.024905  0.024905  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0   0.04981  0.0 |      0.0       0.0  11.407667  0.0  0.0 |  1.88537   1.88537  4.662255  0.0  0.0 | 1.511045  1.511045  1.254909  0.0  0.0 | 1.001793  1.001793  1.245332  0.0  0.0 | 0.836182 | 0.316392 | 0.104523 | 0.014333 |  5.719276 |  7.216427 |  7.216427 | 30.068445 |  1.95019   1.95019  0.0  0.0   1.95019 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.428656  179.428656  0.0  0.0  0.0 | 17.926728  17.926728  32.922597  0.0  0.0 | 20.288355  20.288355  24.911394  0.0  0.0 | 31.892381 | 12.944089 | 30.068445 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7  14.7  14.7  14.7  14.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.605   0.605   0.605   0.605   0.605 | 0.064366  0.064366  0.0  0.0  0.064366 |   0.55519    0.55519   0.605   0.605    0.55519 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.366798  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   0.55519    0.55519  12.971798   0.605    0.55519 |  0.446854   0.446854  12.971798   0.605    0.55519 | 0.0  0.0  0.0  0.0    0.55519 | 0.032183  0.032183  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.064366  0.0 |      0.0       0.0   9.401712  0.0  0.0 | 1.558152  1.558152  4.177104  0.0  0.0 | 1.485582  1.485582   1.25443  0.0  0.0 | 1.026003  1.026003  1.245788  0.0  0.0 | 0.512792 |  0.31989 | 0.064099 |  0.01441 |   4.86444 |  6.218987 |  6.218987 | 25.912445 | 1.935634  1.935634  0.0  0.0  1.935634 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  179.50481   179.50481  0.0  0.0  0.0 | 14.815429  14.815429  30.315579  0.0  0.0 | 20.747934  20.747934  24.920037  0.0  0.0 |  32.07711 | 12.992757 | 25.912445 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9  14.9  14.9  14.9  14.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.242   0.242   0.242   0.242 | 0.067009  0.067009  0.0  0.0  0.067009 |  0.177634   0.177634   0.242   0.242   0.177634 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.547335  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.177634   0.177634  12.789335   0.242   0.177634 |  0.143093   0.143093  12.789335   0.242   0.177634 | 0.0  0.0  0.0  0.0   0.177634 | 0.033505  0.033505  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.067009  0.0 |      0.0       0.0   8.304137  0.0  0.0 | 1.233166  1.233166   3.91165  0.0  0.0 | 1.462603  1.462603  1.253998  0.0  0.0 | 1.047852  1.047852  1.246198  0.0  0.0 |  0.47991 | 0.321564 | 0.059989 | 0.014462 |  4.386474 |  5.299753 |  5.299753 | 22.082304 | 1.932991  1.932991  0.0  0.0  1.932991 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.505846  179.505846  0.0  0.0  0.0 | 11.725356  11.725356  28.889128  0.0  0.0 | 21.162685  21.162685  24.927837  0.0  0.0 | 32.226946 |  13.03722 | 22.082304 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5  15.5  15.5  15.5  15.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.605   0.605   0.605   0.605   0.605 | 0.065382  0.065382  0.0  0.0  0.065382 |  0.537991   0.537991   0.605   0.605   0.537991 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.088947  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.537991   0.537991  13.693947   0.605   0.537991 |  0.433383   0.433383  13.693947   0.605   0.537991 | 0.0  0.0  0.0  0.0   0.537991 | 0.032691  0.032691  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.065382  0.0 |      0.0       0.0   8.098809  0.0  0.0 | 0.966732  0.966732   3.86199  0.0  0.0 | 1.441866  1.441866  1.253608  0.0  0.0 |  1.06757   1.06757  1.246569  0.0  0.0 | 0.537929 | 0.323344 | 0.067241 | 0.014515 |  4.213256 |  4.661408 |  4.661408 | 19.422533 | 1.934618  1.934618  0.0  0.0  1.934618 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.577763  179.577763  0.0  0.0  0.0 |  9.192007   9.192007  28.622276  0.0  0.0 | 21.536982  21.536982  24.934876  0.0  0.0 | 32.433229 | 13.088908 | 19.422533 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3  16.3  16.3  16.3  16.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  2.904   2.904   2.904   2.904   2.904 | 0.052751  0.052751  0.0  0.0  0.052751 |  2.838618   2.838618   2.904   2.904   2.838618 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.811096  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.838618   2.838618  16.715096   2.904   2.838618 |  2.288506   2.288506  16.715096   2.904   2.838618 | 0.0  0.0  0.0  0.0   2.838618 | 0.026376  0.026376  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.052751  0.0 |      0.0       0.0    9.18254  0.0  0.0 |  0.90219   0.90219  4.124096  0.0  0.0 | 1.423151  1.423151  1.253256  0.0  0.0 | 1.085364  1.085364  1.246904  0.0  0.0 | 0.840627 | 0.326905 | 0.105078 | 0.014594 |  4.611214 |  4.364719 |  4.364719 | 18.186329 | 1.947249  1.947249  0.0  0.0  1.947249 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 180.101499  180.101499  0.0  0.0  0.0 |  8.578324   8.578324  30.030735  0.0  0.0 | 21.874768  21.874768  24.941228  0.0  0.0 | 32.940252 | 13.178556 | 18.186329 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6  16.6  16.6  16.6  16.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.484   0.484   0.484   0.484   0.484 | 0.113609  0.113609  0.0  0.0  0.113609 |  0.431249   0.431249   0.484   0.484   0.431249 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.081902  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.431249   0.431249  14.565902   0.484   0.431249 |  0.349705   0.349705  14.565902   0.484   0.431249 | 0.0  0.0  0.0  0.0   0.431249 | 0.056805  0.056805  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.113609  0.0 |      0.0       0.0   8.891084  0.0  0.0 | 0.659289  0.659289  4.053606  0.0  0.0 | 1.406262  1.406262  1.252939  0.0  0.0 | 1.101423  1.101423  1.247206  0.0  0.0 | 0.543037 | 0.330467 |  0.06788 | 0.014672 |  4.245686 |  4.434622 |  4.434622 | 18.477593 | 1.886391  1.886391  0.0  0.0  1.886391 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 180.126238  180.126238  0.0  0.0  0.0 |   6.26874    6.26874  29.651947  0.0  0.0 | 22.179607  22.179607  24.946961  0.0  0.0 | 33.138395 |  13.22996 | 18.477593 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.242   0.242   0.242   0.242 | 0.353524  0.353524  0.0  0.0  0.353524 |  0.128391   0.128391   0.242   0.242   0.128391 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.128391   0.128391  14.955782   0.242   0.128391 |  0.104143   0.104143  14.955782   0.242   0.128391 | 0.0  0.0  0.0  0.0   0.128391 | 0.176762  0.176762  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.353524  0.0 |      0.0       0.0   8.895448  0.0  0.0 | 0.416135  0.416135  4.054662  0.0  0.0 |  1.39102   1.39102  1.252652  0.0  0.0 | 1.115915  1.115915  1.247478  0.0  0.0 | 0.521071 | 0.332329 | 0.065134 | 0.014728 |  4.091227 |  4.360261 |  4.360261 | 18.167754 | 1.646476  1.646476  0.0  0.0  1.646476 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.973724  179.973724  0.0  0.0  0.0 |  3.956748   3.956748  29.657619  0.0  0.0 | 22.454711  22.454711  24.952134  0.0  0.0 | 33.282245 | 13.274754 | 18.167754 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.903769  0.903769  0.0  0.0  0.903769 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.074856  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.074856     0.0        0.0 |       0.0        0.0  15.074856     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.451885  0.451885  0.0  0.0  0.0 |      2.0       2.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.903769  0.0 |      0.0       0.0   8.944532  0.0  0.0 | 0.186209  0.186209  4.066533  0.0  0.0 | 1.377264  1.377264  1.252393  0.0  0.0 | 1.128994  1.128994  1.247724  0.0  0.0 |  0.49825 | 0.333647 | 0.062281 | 0.014776 |  3.966075 |  4.174986 |  4.174986 | 17.395774 | 0.742707  0.742707  0.0  0.0  0.742707 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.521839  179.521839  0.0  0.0  0.0 |  1.770538   1.770538  29.721411  0.0  0.0 | 22.702982  22.702982  24.956803  0.0  0.0 | 33.332084 | 13.307914 | 17.395774 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.742707  0.742707  0.0  0.0  0.742707 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.262439  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.262439     0.0        0.0 |       0.0        0.0  14.262439     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.864415  2.864415  0.0  0.0  0.0 | 1.770538  1.770538  2.0  0.0  0.0 | 0.0  0.0  0.0  3.235768  0.0 |      0.0       0.0   8.649971  0.0  0.0 |      0.0       0.0  3.995292  0.0  0.0 | 1.364851  1.364851   1.25216  0.0  0.0 | 1.140797  1.140797  1.247946  0.0  0.0 | 0.374267 | 0.333525 | 0.046783 | 0.014804 |  3.788227 |  4.022951 |  4.022951 | 16.762297 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 176.657424  176.657424  0.0  0.0  0.0 |       0.0        0.0  29.338587  0.0  0.0 | 22.927036  22.927036  24.961016  0.0  0.0 | 32.961935 | 13.288531 | 16.762297 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5  16.5  16.5  16.5  16.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.363   0.363   0.363   0.363 |    0.363     0.363  0.0  0.0     0.363 |       0.0        0.0   0.363   0.363        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.991633  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.354633   0.363        0.0 |       0.0        0.0  14.354633   0.363        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.055807  3.055807  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  3.237307  0.0 |      0.0       0.0   8.535617  0.0  0.0 |      0.0       0.0  3.967634  0.0  0.0 |      0.0       0.0  1.251949  0.0  0.0 | 1.118165  1.118165  1.248147  0.0  0.0 | 0.188581 | 0.328917 | 0.023573 |  0.01477 |  3.741952 |  3.873182 |  3.873182 |  16.13826 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 173.601617  173.601617  0.0  0.0  0.0 |       0.0        0.0  29.189969  0.0  0.0 | 21.808871  21.808871  24.964819  0.0  0.0 | 32.410513 | 13.245948 |  16.13826 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  3.146   3.146   3.146   3.146   3.146 |      2.0       2.0  0.0  0.0       2.0 |     1.146      1.146   3.146   3.146      1.146 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |     1.146      1.146  17.498707   3.146      1.146 |   0.86344    0.86344  17.498707   3.146      1.146 | 0.0  0.0  0.0  0.0      1.146 | 1.982694  1.982694  0.0  0.0  0.0 |  0.86344   0.86344  2.0  0.0  0.0 | 0.0  0.0  0.0  2.982694  0.0 |      0.0       0.0   9.714237  0.0  0.0 |      0.0       0.0   4.25269  0.0  0.0 |  0.86344   0.86344  1.251759  0.0  0.0 | 1.084862  1.084862  1.248327  0.0  0.0 | 0.542014 | 0.325191 | 0.067752 | 0.014747 |  4.087644 |  3.802529 |  3.802529 | 15.843872 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 171.901483  171.901483  0.0  0.0  0.0 |       0.0        0.0  30.721749  0.0  0.0 | 21.587449  21.587449  24.968251  0.0  0.0 | 32.248581 | 13.251608 | 15.843872 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.847   0.847   0.847   0.847   0.847 |    0.847     0.847  0.0  0.0     0.847 |       0.0        0.0   0.847   0.847        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.199707   0.847        0.0 |       0.0        0.0  15.199707   0.847        0.0 | 0.0  0.0  0.0  0.0        0.0 |  3.37986   3.37986  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0   3.80336  0.0 |      0.0       0.0    9.41236  0.0  0.0 |      0.0       0.0  4.179679  0.0  0.0 |      0.0       0.0  1.251587  0.0  0.0 | 1.052832  1.052832  1.248491  0.0  0.0 |  0.25011 | 0.322125 | 0.031264 | 0.014733 |  3.914206 |  3.933013 |  3.933013 | 16.387555 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 168.521623  168.521623  0.0  0.0  0.0 |       0.0        0.0  30.329418  0.0  0.0 | 20.534617  20.534617  24.971347  0.0  0.0 |   31.6936 | 13.207768 | 16.387555 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0  17.0  17.0  17.0  17.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.363   0.363   0.363   0.363 |    0.363     0.363  0.0  0.0     0.363 |       0.0        0.0   0.363   0.363        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.442976  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.805976   0.363        0.0 |       0.0        0.0  14.805976   0.363        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.277978  3.277978  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  3.459478  0.0 |      0.0       0.0   9.103068  0.0  0.0 |      0.0       0.0  4.104875  0.0  0.0 |      0.0       0.0  1.251433  0.0  0.0 | 1.001485  1.001485  1.248638  0.0  0.0 | 0.188679 | 0.316297 | 0.023585 |  0.01468 |  3.801119 |  3.952326 |  3.952326 | 16.468026 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 165.243645  165.243645  0.0  0.0  0.0 |       0.0        0.0   29.92745  0.0  0.0 | 19.533132  19.533132  24.974142  0.0  0.0 | 31.126684 | 13.161761 | 16.468026 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.363   0.363   0.363   0.363 |    0.363     0.363  0.0  0.0     0.363 |       0.0        0.0   0.363   0.363        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.533245  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.896245   0.363        0.0 |       0.0        0.0  14.896245   0.363        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.602665  5.602665  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  5.784165  0.0 |      0.0       0.0   8.980424  0.0  0.0 |      0.0       0.0  4.075213  0.0  0.0 |      0.0       0.0  1.251293  0.0  0.0 | 0.952642  0.952642  1.248771  0.0  0.0 | 0.188706 | 0.310656 | 0.023588 | 0.014629 |  3.736066 |   3.85593 |   3.85593 | 16.066373 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  159.64098   159.64098  0.0  0.0  0.0 |       0.0        0.0  29.768057  0.0  0.0 |  18.58049   18.58049  24.976664  0.0  0.0 | 30.270237 | 13.078907 | 16.066373 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0  19.0  19.0  19.0  19.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.248348  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.248348     0.0        0.0 |       0.0        0.0  16.248348     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.727324  3.727324  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  3.735707  0.0 |      0.0       0.0   9.449719  0.0  0.0 |      0.0       0.0  4.188715  0.0  0.0 |      0.0       0.0  1.251167  0.0  0.0 | 0.906181  0.906181  1.248891  0.0  0.0 | 0.142635 | 0.301905 | 0.017829 | 0.014534 |  3.816516 |  3.780406 |  3.780406 | 15.751692 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 155.913656  155.913656  0.0  0.0  0.0 |       0.0        0.0  30.377971  0.0  0.0 | 17.674308  17.674308  24.978941  0.0  0.0 | 29.636591 | 13.022906 | 15.751692 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.158079  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.158079     0.0        0.0 |       0.0        0.0  16.158079     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.981965  3.981965  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  4.086328  0.0 |      0.0       0.0   9.654184  0.0  0.0 |      0.0       0.0  4.238166  0.0  0.0 |      0.0       0.0  1.251053  0.0  0.0 | 0.861986  0.861986  1.248999  0.0  0.0 | 0.142657 |   0.2956 | 0.017832 | 0.014472 |  3.834862 |  3.790013 |  3.790013 |  15.79172 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 151.931691  151.931691  0.0  0.0  0.0 |       0.0        0.0  30.643702  0.0  0.0 | 16.812322  16.812322  24.980995  0.0  0.0 | 28.964749 | 12.961404 |  15.79172 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7  18.7  18.7  18.7  18.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.977542  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.977542     0.0        0.0 |       0.0        0.0  15.977542     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.402655  2.402655  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  2.530248  0.0 |      0.0       0.0   9.687705  0.0  0.0 |      0.0       0.0  4.246273  0.0  0.0 |      0.0       0.0   1.25095  0.0  0.0 | 0.819947  0.819947  1.249096  0.0  0.0 | 0.142676 | 0.288915 | 0.017835 | 0.014403 |   3.81168 |   3.81943 |   3.81943 | 15.914292 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 149.529036  149.529036  0.0  0.0  0.0 |       0.0        0.0  30.687266  0.0  0.0 | 15.992376  15.992376  24.982849  0.0  0.0 | 28.497209 | 12.924672 | 15.914292 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.713782     0.0        0.0 |       0.0        0.0  14.713782     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.083886  1.083886  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.159786  0.0 |      0.0       0.0   9.207595  0.0  0.0 |      0.0       0.0  4.130156  0.0  0.0 |      0.0       0.0  1.250858  0.0  0.0 | 0.779957  0.779957  1.249185  0.0  0.0 | 0.142694 | 0.284263 | 0.017837 | 0.014363 |  3.664079 |  3.808827 |  3.808827 | 15.870111 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  148.44515   148.44515  0.0  0.0  0.0 |       0.0        0.0  30.063297  0.0  0.0 | 15.212418  15.212418  24.984522  0.0  0.0 | 28.208365 | 12.909737 | 15.870111 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.359753  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.359753     0.0        0.0 |       0.0        0.0  13.359753     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.196288  0.196288  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.211567  0.0 |      0.0       0.0   8.429313  0.0  0.0 |      0.0       0.0  3.941924  0.0  0.0 |      0.0       0.0  1.250774  0.0  0.0 | 0.741918  0.741918  1.249264  0.0  0.0 |  0.14271 | 0.281389 | 0.017839 | 0.014346 |  3.445271 |    3.7133 |    3.7133 | 15.472083 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.248862  148.248862  0.0  0.0  0.0 |       0.0        0.0  29.051812  0.0  0.0 |   14.4705    14.4705  24.986031  0.0  0.0 |  28.04282 | 12.909871 | 15.472083 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0  14.0  14.0  14.0  14.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.734918  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.734918     0.0        0.0 |       0.0        0.0  11.734918     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.062473  0.062473  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.067425  0.0 |      0.0       0.0   7.392002  0.0  0.0 |      0.0       0.0  3.691045  0.0  0.0 |      0.0       0.0  1.250698  0.0  0.0 | 0.705735  0.705735  1.249336  0.0  0.0 | 0.142724 | 0.279742 | 0.017841 | 0.014346 |    3.1644 |  3.531285 |  3.531285 | 14.713689 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.186389  148.186389  0.0  0.0  0.0 |       0.0        0.0  27.703683  0.0  0.0 | 13.764765  13.764765  24.987394  0.0  0.0 |  27.89724 | 12.912296 | 14.713689 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.832232  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.832232     0.0        0.0 |       0.0        0.0  10.832232     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.061167  0.061167  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.066044  0.0 |      0.0       0.0   6.506375  0.0  0.0 |      0.0       0.0  3.476852  0.0  0.0 |      0.0       0.0   1.25063  0.0  0.0 | 0.671316  0.671316  1.249401  0.0  0.0 | 0.142737 | 0.278293 | 0.017842 | 0.014349 |  2.922448 |  3.286895 |  3.286895 | 13.695397 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.125222  148.125222  0.0  0.0  0.0 |       0.0        0.0  26.552687  0.0  0.0 |  13.09345   13.09345  24.988623  0.0  0.0 | 27.753298 |  12.91474 | 13.695397 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.110083  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.110083     0.0        0.0 |       0.0        0.0  10.110083     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.060106  0.060106  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.064924  0.0 |      0.0       0.0   5.769351  0.0  0.0 |      0.0       0.0  3.298599  0.0  0.0 |      0.0       0.0  1.250569  0.0  0.0 | 0.638575  0.638575  1.249459  0.0  0.0 | 0.142749 | 0.276861 | 0.017844 | 0.014352 |  2.718425 |  3.036937 |  3.036937 | 12.653906 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.065116  148.065116  0.0  0.0  0.0 |       0.0        0.0  25.594821  0.0  0.0 | 12.454875  12.454875  24.989733  0.0  0.0 | 27.610941 | 12.917202 | 12.653906 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6  11.6  11.6  11.6  11.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.568472  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.568472     0.0        0.0 |       0.0        0.0   9.568472     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.059296  0.059296  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.064076  0.0 |      0.0       0.0   5.179352  0.0  0.0 |      0.0       0.0  3.155905  0.0  0.0 |      0.0       0.0  1.250513  0.0  0.0 | 0.607431  0.607431  1.249512  0.0  0.0 | 0.142759 | 0.275445 | 0.017845 | 0.014354 |  2.551892 |  2.816808 |  2.816808 | 11.736699 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  148.00582   148.00582  0.0  0.0  0.0 |       0.0        0.0  24.828036  0.0  0.0 | 11.847443  11.847443  24.990735  0.0  0.0 | 27.470119 | 12.919675 | 11.736699 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1  11.1  11.1  11.1  11.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.121265  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.121265     0.0        0.0 |       0.0        0.0   9.121265     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.058613  0.058613  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.063363  0.0 |      0.0       0.0   4.701684  0.0  0.0 |      0.0       0.0  3.040378  0.0  0.0 |      0.0       0.0  1.250463  0.0  0.0 | 0.577807  0.577807   1.24956  0.0  0.0 | 0.142769 | 0.274044 | 0.017846 | 0.014357 |  2.414182 |  2.632828 |  2.632828 | 10.970115 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.947208  147.947208  0.0  0.0  0.0 |       0.0        0.0  24.207239  0.0  0.0 | 11.269637  11.269637  24.991638  0.0  0.0 | 27.330798 | 12.922158 | 10.970115 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498     0.0        0.0 |       0.0        0.0   10.20498     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.060164  0.060164  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.065065  0.0 |      0.0       0.0   4.883828  0.0  0.0 |      0.0       0.0   3.08443  0.0  0.0 |      0.0       0.0  1.250418  0.0  0.0 | 0.549627  0.549627  1.249602  0.0  0.0 | 0.142778 | 0.272657 | 0.017847 |  0.01436 |  2.441231 |  2.491042 |  2.491042 |  10.37934 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.887044  147.887044  0.0  0.0  0.0 |       0.0        0.0  24.443961  0.0  0.0 |  10.72001   10.72001  24.992454  0.0  0.0 | 27.192656 | 12.924613 |  10.37934 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599     0.0        0.0 |       0.0        0.0  10.385599     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.060398  0.060398  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.065345  0.0 |      0.0       0.0   5.048038  0.0  0.0 |      0.0       0.0  3.124146  0.0  0.0 |      0.0       0.0  1.250377  0.0  0.0 | 0.522821  0.522821  1.249641  0.0  0.0 | 0.142785 | 0.271283 | 0.017848 | 0.014363 |   2.46466 |  2.444543 |  2.444543 | 10.185594 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.826646  147.826646  0.0  0.0  0.0 |       0.0        0.0  24.657376  0.0  0.0 | 10.197189  10.197189   24.99319  0.0  0.0 |  27.05586 | 12.927061 | 10.185594 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8  11.8  11.8  11.8  11.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  1.573   1.573   1.573   1.573   1.573 | 0.054992  0.054992  0.0  0.0  0.054992 |       0.0        0.0   1.573   1.573        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.753432  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.326432   1.573        0.0 |       0.0        0.0  11.326432   1.573        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.025404  0.025404  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.054992  0.0 |      0.0       0.0   5.502199  0.0  0.0 |      0.0       0.0  3.233987  0.0  0.0 |      0.0       0.0   1.25034  0.0  0.0 | 0.497323  0.497323  1.249676  0.0  0.0 | 0.342538 | 0.270917 | 0.042817 | 0.014379 |  2.561843 |  2.460823 |  2.460823 | 10.253431 | 1.518008  1.518008  0.0  0.0  1.518008 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.801242  147.801242  0.0  0.0  0.0 |       0.0        0.0  25.247621  0.0  0.0 |  9.699866   9.699866  24.993854  0.0  0.0 | 27.120498 | 12.954627 | 10.253431 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7  11.7  11.7  11.7  11.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.064218  0.064218  0.0  0.0  0.064218 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.663122  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.663122     0.0        0.0 |       0.0        0.0   9.663122     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.029661  0.029661  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.064218  0.0 |      0.0       0.0   5.079982  0.0  0.0 |      0.0       0.0  3.131871  0.0  0.0 |      0.0       0.0  1.250307  0.0  0.0 | 0.473068  0.473068  1.249708  0.0  0.0 | 0.142799 | 0.270565 |  0.01785 | 0.014396 |  2.442117 |  2.509636 |  2.509636 | 10.456817 |  1.45379   1.45379  0.0  0.0   1.45379 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.771581  147.771581  0.0  0.0  0.0 |       0.0        0.0   24.69889  0.0  0.0 |  9.226798   9.226798  24.994454  0.0  0.0 | 26.984577 | 12.957061 | 10.456817 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.065345  0.065345  0.0  0.0  0.065345 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599     0.0        0.0 |       0.0        0.0  10.385599     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.030176  0.030176  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.065345  0.0 |      0.0       0.0   5.148345  0.0  0.0 |      0.0       0.0  3.148405  0.0  0.0 |      0.0       0.0  1.250277  0.0  0.0 | 0.449996  0.449996  1.249736  0.0  0.0 | 0.142804 | 0.269213 | 0.017851 | 0.014399 |  2.443999 |  2.485193 |  2.485193 | 10.354971 | 1.388445  1.388445  0.0  0.0  1.388445 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.741405  147.741405  0.0  0.0  0.0 |       0.0        0.0  24.787738  0.0  0.0 |  8.776802   8.776802  24.994995  0.0  0.0 | 26.849871 | 12.959476 | 10.354971 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8  12.8  12.8  12.8  12.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.238876  0.238876  0.0  0.0  0.238876 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.656527  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.656527     0.0        0.0 |       0.0        0.0  10.656527     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.110287  0.110287  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.238876  0.0 |      0.0       0.0   5.289906  0.0  0.0 |      0.0       0.0  3.182643  0.0  0.0 |      0.0       0.0   1.25025  0.0  0.0 |  0.42805   0.42805  1.249762  0.0  0.0 | 0.142809 | 0.267872 | 0.017851 | 0.014401 |  2.464747 |  2.453627 |  2.453627 | 10.223447 | 1.149568  1.149568  0.0  0.0  1.149568 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.631118  147.631118  0.0  0.0  0.0 |       0.0        0.0  24.971717  0.0  0.0 |  8.348752   8.348752  24.995483  0.0  0.0 | 26.694474 | 12.959134 | 10.223447 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4  12.4  12.4  12.4  12.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.847   0.847   0.847   0.847   0.847 | 0.984764  0.984764  0.0  0.0  0.984764 |       0.0        0.0   0.847   0.847        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.295289  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.142289   0.847        0.0 |       0.0        0.0  11.142289   0.847        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.454318  0.454318  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.984764  0.0 |      0.0       0.0   5.553429  0.0  0.0 |      0.0       0.0  3.246377  0.0  0.0 |      0.0       0.0  1.250226  0.0  0.0 | 0.407173  0.407173  1.249785  0.0  0.0 |  0.25037 | 0.266862 | 0.031296 | 0.014408 |  2.516741 |  2.460502 |  2.460502 | 10.252092 | 1.011805  1.011805  0.0  0.0  1.011805 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   147.1768    147.1768  0.0  0.0  0.0 |       0.0        0.0    25.3142  0.0  0.0 |  7.941579   7.941579  24.995924  0.0  0.0 | 26.552932 | 12.960391 | 10.252092 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0  12.0  12.0  12.0  12.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   0.484   0.484   0.484   0.484 | 1.495805  1.495805  0.0  0.0  1.495805 |       0.0        0.0   0.484   0.484        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.934051  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.418051   0.484        0.0 |       0.0        0.0  10.418051   0.484        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.502702  1.502702  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  2.344557  0.0 |      0.0       0.0    5.40322  0.0  0.0 |      0.0       0.0  3.210048  0.0  0.0 |      0.0       0.0  1.250204  0.0  0.0 | 0.387315  0.387315  1.249806  0.0  0.0 | 0.204279 | 0.265224 | 0.025535 | 0.014407 |  2.465993 |  2.489229 |  2.489229 | 10.371787 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 145.674097  145.674097  0.0  0.0  0.0 |       0.0        0.0  25.118983  0.0  0.0 |  7.554263   7.554263  24.996321  0.0  0.0 | 26.194265 | 12.934304 | 10.371787 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.121   0.121   0.121   0.121   0.121 |    0.121     0.121  0.0  0.0     0.121 |       0.0        0.0   0.121   0.121        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.11467  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.23567   0.121        0.0 |       0.0        0.0   10.23567   0.121        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.595867  3.595867  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  3.949492  0.0 |      0.0       0.0   5.254647  0.0  0.0 |      0.0       0.0  3.174115  0.0  0.0 |      0.0       0.0  1.250184  0.0  0.0 | 0.368426  0.368426  1.249825  0.0  0.0 | 0.158187 | 0.261426 | 0.019773 | 0.014374 |  2.414186 |  2.479508 |  2.479508 | 10.331285 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  142.07823   142.07823  0.0  0.0  0.0 |       0.0        0.0  24.925892  0.0  0.0 |  7.185838   7.185838   24.99668  0.0  0.0 | 25.589504 | 12.877012 | 10.331285 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6  13.6  13.6  13.6  13.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   0.484   0.484   0.484   0.484 |    0.484     0.484  0.0  0.0     0.484 |       0.0        0.0   0.484   0.484        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.379004  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.863004   0.484        0.0 |       0.0        0.0  11.863004   0.484        0.0 | 0.0  0.0  0.0  0.0        0.0 |  4.66112   4.66112  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  5.249074  0.0 |      0.0       0.0   5.818977  0.0  0.0 |      0.0       0.0  3.310601  0.0  0.0 |      0.0       0.0  1.250166  0.0  0.0 | 0.350457  0.350457  1.249842  0.0  0.0 | 0.204286 | 0.255638 | 0.025536 | 0.014314 |  2.538114 |  2.448883 |  2.448883 | 10.203678 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 137.417111  137.417111  0.0  0.0  0.0 |       0.0        0.0  25.659317  0.0  0.0 |   6.83538    6.83538  24.997004  0.0  0.0 | 24.871603 | 12.804915 | 10.203678 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.081695  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.081695     0.0        0.0 |       0.0        0.0  15.081695     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.549421  5.549421  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  6.461403  0.0 |      0.0       0.0   7.374014  0.0  0.0 |      0.0       0.0  3.686695  0.0  0.0 |      0.0       0.0   1.25015  0.0  0.0 | 0.333365  0.333365  1.249858  0.0  0.0 | 0.142829 | 0.248188 | 0.017854 |  0.01423 |  2.907056 |  2.516156 |  2.516156 | 10.483983 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  131.86769   131.86769  0.0  0.0  0.0 |       0.0        0.0  27.680304  0.0  0.0 |  6.502015   6.502015  24.997296  0.0  0.0 | 23.945748 | 12.705977 | 10.483983 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8  18.8  18.8  18.8  18.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.075101  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.075101     0.0        0.0 |       0.0        0.0  16.075101     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.325221  5.325221  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0   6.46129  0.0 |      0.0       0.0   8.560085  0.0  0.0 |      0.0       0.0  3.973552  0.0  0.0 |      0.0       0.0  1.250135  0.0  0.0 | 0.317107  0.317107  1.249871  0.0  0.0 | 0.142831 | 0.238976 | 0.017854 |  0.01412 |  3.183189 |  2.759166 |  2.759166 | 11.496527 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 126.542469  126.542469  0.0  0.0  0.0 |       0.0        0.0  29.221768  0.0  0.0 |  6.184908   6.184908   24.99756  0.0  0.0 | 23.029122 | 12.607151 | 11.496527 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0  23.0  23.0  23.0  23.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.868102  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.868102     0.0        0.0 |       0.0        0.0  19.868102     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 6.525426  6.525426  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  8.250733  0.0 |      0.0       0.0  10.659033  0.0  0.0 |      0.0       0.0  4.481194  0.0  0.0 |      0.0       0.0  1.250122  0.0  0.0 | 0.301642  0.301642  1.249884  0.0  0.0 | 0.142834 | 0.229855 | 0.017854 |  0.01401 |  3.686615 |  3.069244 |  3.069244 | 12.788517 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 120.017043  120.017043  0.0  0.0  0.0 |       0.0        0.0  31.949643  0.0  0.0 |  5.883266   5.883266  24.997798  0.0  0.0 | 21.894388 | 12.480031 | 12.788517 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0  22.0  22.0  22.0  22.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.965006  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.965006     0.0        0.0 |       0.0        0.0  18.965006     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  5.41968   5.41968  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  7.225213  0.0 |      0.0       0.0  11.377028  0.0  0.0 |      0.0       0.0  4.654844  0.0  0.0 |      0.0       0.0   1.25011  0.0  0.0 |  0.28693   0.28693  1.249895  0.0  0.0 | 0.142836 | 0.218565 | 0.017855 | 0.013869 |   3.84545 |   3.45597 |   3.45597 | 14.399875 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 114.597363  114.597363  0.0  0.0  0.0 |       0.0        0.0  32.882777  0.0  0.0 |  5.596336   5.596336  24.998013  0.0  0.0 | 20.901172 | 12.369331 | 14.399875 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4  22.4  22.4  22.4  22.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.326244  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.326244     0.0        0.0 |       0.0        0.0  19.326244     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.335845  4.335845  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  6.053675  0.0 |      0.0       0.0  11.886324  0.0  0.0 |      0.0       0.0   4.77802  0.0  0.0 |      0.0       0.0  1.250099  0.0  0.0 | 0.272937  0.272937  1.249906  0.0  0.0 | 0.142838 | 0.208682 | 0.017855 | 0.013746 |  3.954211 |  3.740541 |  3.740541 | 15.585589 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 110.261519  110.261519  0.0  0.0  0.0 |       0.0        0.0  33.544678  0.0  0.0 |  5.323399   5.323399  24.998206  0.0  0.0 | 20.066608 |  12.27735 | 15.585589 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8  22.8  22.8  22.8  22.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.687483  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.687483     0.0        0.0 |       0.0        0.0  19.687483     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.342105  3.342105  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  4.849713  0.0 |      0.0       0.0  12.288897  0.0  0.0 |      0.0       0.0  4.875385  0.0  0.0 |      0.0       0.0   1.25009  0.0  0.0 | 0.259625  0.259625  1.249915  0.0  0.0 |  0.14284 | 0.200378 | 0.017855 | 0.013644 |  4.038368 |  3.895374 |  3.895374 | 16.230726 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 106.919414  106.919414  0.0  0.0  0.0 |       0.0        0.0  34.067878  0.0  0.0 |  5.063774   5.063774  24.998381  0.0  0.0 | 19.393234 | 12.204581 | 16.230726 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7  20.7  20.7  20.7  20.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.790982  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.790982     0.0        0.0 |       0.0        0.0  17.790982     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.517107  1.517107  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0   2.27028  0.0 |      0.0       0.0  11.748546  0.0  0.0 |      0.0       0.0  4.744698  0.0  0.0 |      0.0       0.0  1.250081  0.0  0.0 | 0.246963  0.246963  1.249923  0.0  0.0 | 0.142842 | 0.193678 | 0.017855 | 0.013563 |  3.890236 |  3.980783 |  3.980783 | 16.586595 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 105.402307  105.402307  0.0  0.0  0.0 |       0.0        0.0  33.365616  0.0  0.0 |  4.816811   4.816811  24.998539  0.0  0.0 | 19.054108 | 12.172837 | 16.586595 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.713862  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.713862     0.0        0.0 |       0.0        0.0  15.713862     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.664106  0.664106  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  1.008109  0.0 |      0.0       0.0  10.654945  0.0  0.0 |      0.0       0.0  4.480205  0.0  0.0 |      0.0       0.0  1.250073  0.0  0.0 | 0.234919  0.234919  1.249931  0.0  0.0 | 0.142843 | 0.190303 | 0.017855 | 0.013528 |   3.60821 |  3.925299 |  3.925299 | 16.355411 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.738201  104.738201  0.0  0.0  0.0 |       0.0        0.0  31.944329  0.0  0.0 |  4.581892   4.581892  24.998682  0.0  0.0 | 18.878634 | 12.161163 | 16.355411 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.365814  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.365814     0.0        0.0 |       0.0        0.0  13.365814     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.166111  0.166111  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.253754  0.0 |      0.0       0.0   9.171827  0.0  0.0 |      0.0       0.0  4.121505  0.0  0.0 |      0.0       0.0  1.250066  0.0  0.0 | 0.223462  0.223462  1.249937  0.0  0.0 | 0.142845 | 0.188557 | 0.017856 | 0.013515 |  3.231331 |  3.713341 |  3.713341 | 15.472254 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.572091  104.572091  0.0  0.0  0.0 |       0.0        0.0  30.016811  0.0  0.0 |  4.358431   4.358431   24.99881  0.0  0.0 | 18.800699 | 12.161476 | 15.472254 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1  15.1  15.1  15.1  15.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.733647  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  12.733647     0.0        0.0 |       0.0        0.0  12.733647     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.045045  0.045045  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.068921  0.0 |      0.0       0.0   8.164669  0.0  0.0 |      0.0       0.0  3.877919  0.0  0.0 |      0.0       0.0  1.250059  0.0  0.0 | 0.212563  0.212563  1.249943  0.0  0.0 | 0.142846 | 0.187782 | 0.017856 | 0.013515 |  2.973922 |  3.398951 |  3.398951 | 14.162295 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.527046  104.527046  0.0  0.0  0.0 |       0.0        0.0   28.70787  0.0  0.0 |  4.145868   4.145868  24.998926  0.0  0.0 | 18.747011 | 12.164723 | 14.162295 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.837146  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.837146     0.0        0.0 |       0.0        0.0  10.837146     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.043146  0.043146  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.066044  0.0 |      0.0       0.0   6.903426  0.0  0.0 |      0.0       0.0  3.572881  0.0  0.0 |      0.0       0.0  1.250054  0.0  0.0 | 0.202196  0.202196  1.249949  0.0  0.0 | 0.142847 | 0.187248 | 0.017856 | 0.013519 |  2.653951 |  3.095309 |  3.095309 | 12.897121 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   104.4839    104.4839  0.0  0.0  0.0 |       0.0        0.0   27.06871  0.0  0.0 |  3.943671   3.943671  24.999031  0.0  0.0 | 18.694223 | 12.168011 | 12.897121 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498     0.0        0.0 |       0.0        0.0   10.20498     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.042489  0.042489  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.065065  0.0 |      0.0       0.0   6.009729  0.0  0.0 |      0.0       0.0  3.356735  0.0  0.0 |      0.0       0.0  1.250048  0.0  0.0 | 0.192335  0.192335  1.249954  0.0  0.0 | 0.142848 | 0.186722 | 0.017856 | 0.013522 |   2.42558 |  2.799802 |  2.799802 |  11.66584 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.441411  104.441411  0.0  0.0  0.0 |       0.0        0.0  25.907226  0.0  0.0 |  3.751336   3.751336  24.999126  0.0  0.0 | 18.642087 | 12.171312 |  11.66584 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599     0.0        0.0 |       0.0        0.0  10.385599     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.042655  0.042655  0.0  0.0  0.0 |      0.0       0.0  2.0  0.0  0.0 | 0.0  0.0  0.0  0.065345  0.0 |      0.0       0.0   5.623788  0.0  0.0 |      0.0       0.0  3.263394  0.0  0.0 |      0.0       0.0  1.250044  0.0  0.0 | 0.182955  0.182955  1.249958  0.0  0.0 | 0.142849 | 0.186204 | 0.017856 | 0.013526 |  2.323615 |  2.545369 |  2.545369 | 10.605703 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.398756  104.398756  0.0  0.0  0.0 |       0.0        0.0  25.405642  0.0  0.0 |  3.568381   3.568381  24.999211  0.0  0.0 | 18.590434 | 12.174605 | 10.605703 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96p_snow_redistribution:

snow redistribution
___________________

Regarding snow redistribution, all functionalities of |hland_96p| are identical with
the ones of |hland_96|.  Hence, all snow data in the following table agrees with the
:ref:`hland_96_snow_redistribution_1` example of |hland_96|:

.. integration-test::

    >>> zonez(0.0, 10.0, 30.0, 0.0, 20.0)
    >>> smax(200.0)
    >>> sred(n_zones=1)
    >>> inputs.t.series = -0.4
    >>> pcorr *= 2.0
    >>> pcalt(0.05)
    >>> test("hland_96p_snow_redistribution",
    ...      axis1=(states.sp, states.wc), axis2=(factors.tc, fluxes.pc))
    |        date |    p |    t |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                 swe |                          gact |                                        pc |                                ei |                                          tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                          melt |                               refr |                                    in_ |                                    r |                      sr |                      ea |                                dp |                      el |                      rs |                                ri |                               gr1 |                                    rg1 |      gr2 |      rg2 |      gr3 |      rg3 |     inrc |    outrc |       rt |       qt |                                  ic |                                                  sp |                                                  wc |                                    sm |                               suz |                                      sg1 |       sg2 |       sg3 |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.487706  0.487706  0.487706  0.0  0.0 |      0.0 | 0.099502 |      0.0 | 0.011105 | 0.493397 | 0.195326 | 0.195326 | 0.813856 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.512294   9.512294  9.512294  0.0  0.0 |  9.900498 |  9.988895 | 0.813856 |
    | 01/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.46392   0.46392   0.46392  0.0  0.0 |      0.0 | 0.098512 |      0.0 | 0.011093 | 0.473433 | 0.354617 | 0.354617 | 1.477572 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.048374   9.048374  9.048374  0.0  0.0 |  9.801987 |  9.977802 | 1.477572 |
    | 01/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.441294  0.441294  0.441294  0.0  0.0 |      0.0 | 0.097531 |      0.0 |  0.01108 | 0.454406 |  0.45403 |  0.45403 | 1.891792 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   8.60708    8.60708   8.60708  0.0  0.0 |  9.704455 |  9.966722 | 1.891792 |
    | 01/01 03:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.419772  0.419772  0.419772  0.0  0.0 |      0.0 | 0.096561 |      0.0 | 0.011068 | 0.436271 | 0.459792 | 0.459792 | 1.915798 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.187308   8.187308  8.187308  0.0  0.0 |  9.607894 |  9.955654 | 1.915798 |
    | 01/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   0.3993    0.3993    0.3993  0.0  0.0 |      0.0 |   0.0956 |      0.0 | 0.011056 | 0.418985 | 0.444294 | 0.444294 | 1.851225 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.788008   7.788008  7.788008  0.0  0.0 |  9.512294 |  9.944598 | 1.851225 |
    | 01/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.379826  0.379826  0.379826  0.0  0.0 |      0.0 | 0.094649 |      0.0 | 0.011043 | 0.402507 |  0.42689 |  0.42689 | 1.778708 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.408182   7.408182  7.408182  0.0  0.0 |  9.417645 |  9.933555 | 1.778708 |
    | 01/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.361301  0.361301  0.361301  0.0  0.0 |      0.0 | 0.093707 |      0.0 | 0.011031 | 0.386797 | 0.410061 | 0.410061 | 1.708587 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.046881   7.046881  7.046881  0.0  0.0 |  9.323938 |  9.922524 | 1.708587 |
    | 01/01 07:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.34368   0.34368   0.34368  0.0  0.0 |      0.0 | 0.092775 |      0.0 | 0.011019 | 0.371818 |    0.394 |    0.394 | 1.641665 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |    6.7032     6.7032    6.7032  0.0  0.0 |  9.231163 |  9.911505 | 1.641665 |
    | 01/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.326919  0.326919  0.326919  0.0  0.0 |      0.0 | 0.091852 |      0.0 | 0.011007 | 0.357536 | 0.378686 | 0.378686 | 1.577857 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.376282   6.376282  6.376282  0.0  0.0 |  9.139312 |  9.900498 | 1.577857 |
    | 01/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.310975  0.310975  0.310975  0.0  0.0 |      0.0 | 0.090938 |      0.0 | 0.010994 | 0.343917 | 0.364084 | 0.364084 | 1.517017 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.065307   6.065307  6.065307  0.0  0.0 |  9.048374 |  9.889504 | 1.517017 |
    | 01/01 10:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.295808  0.295808  0.295808  0.0  0.0 |      0.0 | 0.090033 |      0.0 | 0.010982 | 0.330928 |  0.35016 |  0.35016 | 1.459002 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.769498   5.769498  5.769498  0.0  0.0 |  8.958341 |  9.878522 | 1.459002 |
    | 01/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.281382  0.281382  0.281382  0.0  0.0 |      0.0 | 0.089137 |      0.0 |  0.01097 | 0.318539 | 0.336882 | 0.336882 | 1.403675 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.488116   5.488116  5.488116  0.0  0.0 |  8.869204 |  9.867552 | 1.403675 |
    | 01/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.267659  0.267659  0.267659  0.0  0.0 |      0.0 |  0.08825 |      0.0 | 0.010958 | 0.306721 | 0.324218 | 0.324218 | 1.350908 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.220458   5.220458  5.220458  0.0  0.0 |  8.780954 |  9.856594 | 1.350908 |
    | 01/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.254605  0.254605  0.254605  0.0  0.0 |      0.0 | 0.087372 |      0.0 | 0.010946 | 0.295447 | 0.312138 | 0.312138 | 1.300575 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.965853   4.965853  4.965853  0.0  0.0 |  8.693582 |  9.845648 | 1.300575 |
    | 01/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.242188  0.242188  0.242188  0.0  0.0 |      0.0 | 0.086503 |      0.0 | 0.010934 |  0.28469 | 0.300615 | 0.300615 | 1.252561 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.723666   4.723666  4.723666  0.0  0.0 |   8.60708 |  9.834715 | 1.252561 |
    | 01/01 15:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0    1.144   0.242       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.230376  0.230376  0.230376  0.0  0.0 |  0.03073 | 0.085795 | 0.003841 | 0.010924 | 0.274571 | 0.289631 | 0.289631 | 1.206796 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   4.49329    4.49329   4.49329  0.0  0.0 |  8.552015 |  9.827632 | 1.206796 |
    | 01/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.21914   0.21914   0.21914  0.0  0.0 |      0.0 | 0.085094 |      0.0 | 0.010914 | 0.264919 | 0.279224 | 0.279224 | 1.163433 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.274149   4.274149  4.274149  0.0  0.0 |  8.466921 |  9.816719 | 1.163433 |
    | 01/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.208453  0.208453  0.208453  0.0  0.0 |      0.0 | 0.084247 |      0.0 | 0.010901 | 0.255568 | 0.269335 | 0.269335 | 1.122229 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.065697   4.065697  4.065697  0.0  0.0 |  8.382674 |  9.805817 | 1.122229 |
    | 01/01 18:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    1.224502        8.58  0.0       4.435 | 0.0  0.0  0.901953  0.0   0.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |    0.0     1.894    7.436   1.573     4.435 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0   0.669498  0.0   1.573  0.0 |      0.0  0.167375  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.167375  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0  0.167375  0.0  0.0  0.0 | 0.198286  0.202402  0.198286  0.0  0.0 | 0.199746 | 0.084404 | 0.024968 | 0.010903 |  0.24868 | 0.259985 | 0.259985 | 1.083273 | 1.815       2.0  0.0  0.0       2.0 |        0.0    0.874644        8.58  0.0       4.435 |        0.0    0.349858         0.0  0.0         0.0 |      100.0  100.502124  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   3.86741   4.030669   3.86741  0.0  0.0 |  8.498015 |  9.819882 | 1.083273 |
    | 01/01 19:00 |  5.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   10.249462      40.612  0.0      28.459 | 0.0  0.0  0.901953  0.0   0.0 |  6.776  14.5376   32.032   6.776   24.024 | 0.0       0.0  0.0  0.0       0.0 |  6.591   14.5376   32.032   6.776    24.024 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     6.591    5.51264  0.0   6.776  0.0 |  1.64775  1.392035  0.0   6.776  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.64775  1.392035  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  1.64775  1.392035  0.0  0.0  0.0 | 0.229132  0.230806  0.188616  0.0  0.0 | 0.860444 | 0.088845 | 0.107556 | 0.010965 | 0.268804 | 0.253942 | 0.253942 |  1.05809 |   2.0       2.0  0.0  0.0       2.0 |        0.0    7.321044      40.612  0.0      28.459 |        0.0    2.928418         0.0  0.0         0.0 |  104.94325  104.622729  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.286029   5.191898  3.678794  0.0  0.0 |  9.269615 |  9.916473 |  1.05809 |
    | 01/01 20:00 |  2.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   14.950102        57.2  0.0        40.9 | 0.0  0.0  0.901953  0.0   0.0 |  3.509   7.5284   16.588   3.509   12.441 | 0.0       0.0  0.0  0.0       0.0 |  3.509    7.5284   16.588   3.509    12.441 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     3.509    2.82776  0.0   3.509  0.0 | 0.966123  0.773811  0.0   3.509  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.966123  0.773811  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.966123  0.773811  0.0  0.0  0.0 | 0.281558  0.272239  0.179417  0.0  0.0 | 0.445587 | 0.094455 | 0.055698 | 0.011043 | 0.300798 | 0.263147 | 0.263147 | 1.096444 |   2.0       2.0  0.0  0.0       2.0 |        0.0   10.678644        57.2  0.0        40.9 |        0.0    4.271458         0.0  0.0         0.0 | 107.486127  106.676678  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.970594    5.69347  3.499377  0.0  0.0 |  9.620748 |  9.961129 | 1.096444 |
    | 01/01 21:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   22.853942      85.228  0.0      61.921 | 0.0  0.0  0.901953  0.0   0.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     5.929    4.81656  0.0   5.929  0.0 | 1.712483  1.370301  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.712483  1.370301  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.712483  1.370301  0.0  0.0  0.0 | 0.333297  0.311367  0.170667  0.0  0.0 | 0.752889 |  0.09948 | 0.094111 | 0.011114 | 0.331485 | 0.287706 | 0.287706 | 1.198775 |   2.0       2.0  0.0  0.0       2.0 |        0.0   16.324244      85.228  0.0      61.921 |        0.0    6.529698         0.0  0.0         0.0 | 111.702644  110.122937  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   7.34978   6.752403  3.328711  0.0  0.0 | 10.274157 | 10.044126 | 1.198775 |
    | 01/01 22:00 | 10.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   39.886902      145.86  0.0     107.395 | 0.0  0.0  0.901953  0.0   0.0 | 12.826  27.5176   60.632  12.826   45.474 | 0.0       0.0  0.0  0.0       0.0 | 12.826   27.5176   60.632  12.826    45.474 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.826   10.48464  0.0  12.826  0.0 | 4.000904  3.178697  0.0  12.826  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.190411  0.112168  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 |  0.40763  0.378496  0.162343  0.0  0.0 | 1.628698 | 0.110346 | 0.203587 | 0.011267 | 0.476165 | 0.325841 | 0.325841 | 1.357672 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.490644      145.86  0.0     107.395 |        0.0   11.396258         0.0  0.0         0.0 |  120.52774  117.428881  0.0  0.0  0.0 | 1.810493  1.066529  0.0  0.0  0.0 |   8.94215   8.373908  3.166368  0.0  0.0 | 11.792509 | 10.236446 | 1.357672 |
    | 01/01 23:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   40.103062     146.432  0.0     107.824 | 0.0  0.0  0.901953  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121    0.04344  0.0   0.121  0.0 | 0.043944  0.014975  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.854437  1.081504  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.854437  1.081504  0.0  0.0  0.0 | 0.481712  0.434993  0.154426  0.0  0.0 | 0.015365 | 0.117414 | 0.001921 | 0.011369 | 0.427651 | 0.407921 | 0.407921 | 1.699672 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.645044     146.432  0.0     107.824 |        0.0   11.458018         0.0  0.0         0.0 | 120.604796  117.457345  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.314875   9.020419  3.011942  0.0  0.0 |  11.69046 | 10.226998 | 1.699672 |
    | 02/01 00:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   41.280182     150.436  0.0     110.827 | 0.0  0.0  0.902305  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.847    0.64008  0.0   0.847  0.0 | 0.308001  0.220767  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.308001  0.220767  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.308001  0.220767  0.0  0.0  0.0 | 0.510636  0.445359  0.146894  0.0  0.0 | 0.107556 | 0.116858 | 0.013444 | 0.011364 | 0.438028 | 0.436841 | 0.436841 |  1.82017 |   2.0       2.0  0.0  0.0       2.0 |        0.0   29.485844     150.436  0.0     110.827 |        0.0   11.794338         0.0  0.0         0.0 | 121.143795  117.876658  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.112241   8.795827  2.865048  0.0  0.0 | 11.681158 | 10.229078 |  1.82017 |
    | 02/01 01:00 |  3.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   46.140982     167.596  0.0     123.697 | 0.0  0.0  0.902305  0.0   0.0 |   3.63    7.788    17.16    3.63    12.87 | 0.0       0.0  0.0  0.0       0.0 |   3.63     7.788    17.16    3.63     12.87 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |      3.63     2.9272  0.0    3.63  0.0 | 1.331831  1.016829  0.0    3.63  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.331831  1.016829  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.331831  1.016829  0.0  0.0  0.0 | 0.525928   0.45398   0.13973  0.0  0.0 | 0.460952 | 0.118527 | 0.057619 | 0.011391 | 0.445573 | 0.436712 | 0.436712 | 1.819631 |   2.0       2.0  0.0  0.0       2.0 |        0.0   32.957844     167.596  0.0     123.697 |        0.0   13.183138         0.0  0.0         0.0 | 123.441964  119.787029  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.918144   9.358677  2.725318  0.0  0.0 | 12.023584 | 10.275306 | 1.819631 |
    | 02/01 02:00 |  2.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   49.560342     179.608  0.0     132.706 | 0.0  0.0  0.902305  0.0   0.0 |  2.541   5.4516   12.012   2.541    9.009 | 0.0       0.0  0.0  0.0       0.0 |  2.541    5.4516   12.012   2.541     9.009 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     2.541    2.03224  0.0   2.541  0.0 | 0.967989  0.729012  0.0   2.541  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.967989  0.729012  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.967989  0.729012  0.0  0.0  0.0 | 0.556286  0.474353  0.132915  0.0  0.0 | 0.322667 | 0.121245 | 0.040333 | 0.011433 | 0.462338 | 0.443027 | 0.443027 | 1.845945 |   2.0       2.0  0.0  0.0       2.0 |        0.0   35.400244     179.608  0.0     132.706 |        0.0   14.160098         0.0  0.0         0.0 | 125.014975  121.090257  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 11.329847   9.613335  2.592403  0.0  0.0 | 12.225006 | 10.304206 | 1.845945 |
    | 02/01 03:00 | 10.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   99.386482       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 | 12.584  26.9984   59.488  12.584   44.616 | 0.0       0.0  0.0  0.0       0.0 | 12.584   26.9984   59.488  12.584    44.616 |        0.0         0.0      39.096  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0     23.6525         0.0  0.0      22.678 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.584    0.82476  0.0  12.584  0.0 | 4.916803  0.302333  0.0  12.584  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0  0.302333  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.27757       0.0  0.0  0.0  0.0 | 1.933508  0.302333  0.0  0.0  0.0 | 0.600105  0.476282  0.126433  0.0  0.0 | 1.619077 | 0.129709 | 0.202385 | 0.011555 |   0.5767 | 0.462634 | 0.462634 | 1.927641 |   2.0       2.0  0.0  0.0       2.0 |        0.0   70.990344       200.0  0.0       200.0 |        0.0   28.396138         0.0  0.0         0.0 | 132.682172  121.612683  0.0  0.0  0.0 | 2.639232       0.0  0.0  0.0  0.0 |  12.66325   9.439387   2.46597  0.0  0.0 | 13.714373 | 10.495035 | 1.927641 |
    | 02/01 04:00 |  3.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  127.241232       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.235    9.086    20.02   4.235   15.015 | 0.0       0.0  0.0  0.0       0.0 |  4.235     9.086    20.02   4.235    15.015 |        0.0         0.0       20.02  0.0      15.015 |       0.0        0.0        0.0  0.0        0.0 |        0.0    18.76875         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.235        0.0  0.0   4.235  0.0 | 1.863883       0.0  0.0   4.235  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.238203       0.0  0.0  0.0  0.0 | 1.866838       0.0  0.0  0.0  0.0 | 0.663497  0.460364  0.120267  0.0  0.0 | 0.580052 | 0.139351 | 0.072506 | 0.011695 |  0.58836 | 0.527127 | 0.527127 | 2.196361 |   2.0       2.0  0.0  0.0       2.0 |        0.0   93.803094       200.0  0.0       200.0 |        0.0   33.438138         0.0  0.0         0.0 |  135.05329  121.612683  0.0  0.0  0.0 | 2.264912       0.0  0.0  0.0  0.0 |  13.86659   8.979022  2.345703  0.0  0.0 | 14.155074 | 10.555847 | 2.196361 |
    | 02/01 05:00 |  3.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  154.300132       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.114   8.8264   19.448   4.114   14.586 | 0.0       0.0  0.0  0.0       0.0 |  4.114    8.8264   19.448   4.114    14.586 |        0.0         0.0      19.448  0.0      14.586 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.2325         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.114        0.0  0.0   4.114  0.0 | 1.875921       0.0  0.0   4.114  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.203727       0.0  0.0  0.0  0.0 |  1.80667       0.0  0.0  0.0  0.0 | 0.720705  0.437912  0.114401  0.0  0.0 | 0.583787 | 0.143755 | 0.072973 | 0.011763 |  0.59295 | 0.574056 | 0.574056 | 2.391898 |   2.0       2.0  0.0  0.0       2.0 |        0.0  115.965194       200.0  0.0       200.0 |        0.0   38.334938         0.0  0.0         0.0 | 137.291368  121.612683  0.0  0.0  0.0 | 1.937106       0.0  0.0  0.0  0.0 | 14.952556    8.54111  2.231302  0.0  0.0 | 14.595106 | 10.617058 | 2.391898 |
    | 02/01 06:00 |  1.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  163.850332       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.452   3.1152    6.864   1.452    5.148 | 0.0       0.0  0.0  0.0       0.0 |  1.452    3.1152    6.864   1.452     5.148 |        0.0         0.0       6.864  0.0       5.148 |       0.0        0.0        0.0  0.0        0.0 |        0.0       6.435         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     1.452        0.0  0.0   1.452  0.0 | 0.684216       0.0  0.0   1.452  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.059127       0.0  0.0  0.0  0.0 | 1.752372       0.0  0.0  0.0  0.0 | 0.772333  0.416555  0.108822  0.0  0.0 | 0.262993 | 0.146534 | 0.032874 | 0.011808 | 0.557785 | 0.586385 | 0.586385 | 2.443271 |   2.0       2.0  0.0  0.0       2.0 |        0.0  123.812994       200.0  0.0       200.0 |        0.0   40.037338         0.0  0.0         0.0 | 138.059152  121.612683  0.0  0.0  0.0 | 0.562195       0.0  0.0  0.0  0.0 | 15.932595   8.124555   2.12248  0.0  0.0 | 14.711565 | 10.638123 | 2.443271 |
    | 02/01 07:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 | 0.057657       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.619853       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.619853       0.0  0.0  0.0  0.0 | 0.792283  0.396239  0.103515  0.0  0.0 | 0.015365 | 0.146459 | 0.001921 | 0.011815 | 0.538183 | 0.570842 | 0.570842 | 2.378508 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.503644       200.0  0.0       200.0 |        0.0   40.142538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 15.760165   7.728316  2.018965  0.0  0.0 | 14.580471 | 10.628229 | 2.378508 |
    | 02/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.768632  0.376914  0.098466  0.0  0.0 |      0.0 | 0.145078 |      0.0 | 0.011803 | 0.522836 | 0.548327 | 0.548327 | 2.284696 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.543644       200.0  0.0       200.0 |        0.0   40.102538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.991533   7.351402  1.920499  0.0  0.0 | 14.435393 | 10.616427 | 2.284696 |
    | 02/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.731146  0.358532  0.093664  0.0  0.0 |      0.0 | 0.143635 |      0.0 | 0.011789 | 0.503119 | 0.529956 | 0.529956 | 2.208151 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.583644       200.0  0.0       200.0 |        0.0   40.062538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.260387    6.99287  1.826835  0.0  0.0 | 14.291758 | 10.604637 | 2.208151 |
    | 02/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  167.829582       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 |        0.0         0.0       2.288  0.0       1.716 |       0.0        0.0        0.0  0.0        0.0 |        0.0       2.145         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.484        0.0  0.0   0.484  0.0 | 0.230842       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.230842       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.230842       0.0  0.0  0.0  0.0 | 0.701163  0.341046  0.089096  0.0  0.0 |  0.06146 | 0.142512 | 0.007683 | 0.011781 | 0.486492 | 0.511861 | 0.511861 | 2.132753 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.226244       200.0  0.0       200.0 |        0.0   40.603338         0.0  0.0         0.0 | 138.375653  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.790065   6.651823  1.737739  0.0  0.0 | 14.210707 | 10.600539 | 2.132753 |
    | 02/01 11:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  168.625432       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 | 0.057922       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.057922       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.057922       0.0  0.0  0.0  0.0 | 0.673974  0.324413  0.084751  0.0  0.0 | 0.015365 | 0.141475 | 0.001921 | 0.011773 |  0.47115 | 0.494313 | 0.494313 | 2.059638 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.916894       200.0  0.0       200.0 |        0.0   40.708538         0.0  0.0         0.0 | 138.438731  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.174014    6.32741  1.652989  0.0  0.0 | 14.084597 | 10.590687 | 2.059638 |
    | 02/01 12:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  197.276032       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |        0.0         0.0      20.592  0.0      15.444 |       0.0        0.0        0.0  0.0        0.0 |        0.0      19.305         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.356        0.0  0.0   4.356  0.0 | 2.087099       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.008289       0.0  0.0  0.0  0.0 | 1.841299       0.0  0.0  0.0  0.0 | 0.687779  0.308591  0.080617  0.0  0.0 | 0.603524 | 0.143152 | 0.075441 | 0.011803 | 0.475061 | 0.479577 | 0.479577 |  1.99824 |   2.0       2.0  0.0  0.0       2.0 |        0.0  151.380294       200.0  0.0       200.0 |        0.0   45.895738         0.0  0.0         0.0 | 140.707632  121.612683  0.0  0.0  0.0 | 0.078811       0.0  0.0  0.0  0.0 | 14.327534   6.018819  1.572372  0.0  0.0 | 14.544969 | 10.654325 |  1.99824 |
    | 02/01 13:00 |  5.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  42.523945       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.139  15.3164   33.748   7.139   25.311 | 0.0       0.0  0.0  0.0       0.0 |  7.139   15.3164   33.748   7.139    25.311 |        0.0    9.366467      33.748  0.0      25.311 |       0.0   3.225965        0.0  0.0        0.0 |  32.804173         0.0         0.0  0.0         0.0 |  2.580772        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0   7.139  0.0 |      0.0       0.0  0.0   7.139  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.078811       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.078811       0.0  0.0  0.0  0.0 |   0.7007  0.293541  0.076685  0.0  0.0 |  0.90654 | 0.149242 | 0.113317 | 0.011894 | 0.477576 | 0.475017 | 0.475017 | 1.979238 |   2.0       2.0  0.0  0.0       2.0 |   30.45818  148.803428       200.0  0.0       200.0 |  12.065766   51.196572         0.0  0.0         0.0 | 140.707632  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.705645   5.725277  1.495686  0.0  0.0 | 15.302266 | 10.755748 | 1.979238 |
    | 02/01 14:00 |  1.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  48.329573       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.331   2.8556    6.292   1.331    4.719 | 0.0       0.0  0.0  0.0       0.0 |  1.331    2.8556    6.292   1.331     4.719 |        0.0    2.112422       6.292  0.0       4.719 |       0.0   0.743178        0.0  0.0        0.0 |   6.408937         0.0         0.0  0.0         0.0 |  0.594543        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |  2.528852        0.0  0.0   1.331  0.0 | 1.251696       0.0  0.0   1.331  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.251696       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.251696       0.0  0.0  0.0  0.0 |  0.69921  0.279225  0.072945  0.0  0.0 | 0.169016 | 0.153102 | 0.021127 | 0.011956 | 0.476173 |  0.47624 |  0.47624 | 1.984334 |   2.0       2.0  0.0  0.0       2.0 |  34.521124  147.989406       200.0  0.0       200.0 |  13.808449   52.010594         0.0  0.0         0.0 | 141.984788  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.258131   5.446052  1.422741  0.0  0.0 |  15.31818 | 10.764919 | 1.984334 |
    | 02/01 15:00 | 20.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 201.988205  191.984574  201.988205  0.0  201.988205 | 0.0  0.0  0.902305  0.0   0.0 | 25.047  53.7372  118.404  25.047   88.803 | 0.0       0.0  0.0  0.0       0.0 | 25.047   53.7372  118.404  25.047    88.803 |        0.0   36.356814     118.404  0.0      88.803 |       0.0  17.380386        0.0  0.0        0.0 | 115.042936    1.778447    1.778447  0.0    1.778447 | 13.568696   0.209758   0.209758  0.0   0.209758 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.209758  0.0  0.209758 |       0.0  10.003631  0.0  25.047  0.0 |      0.0  3.698754  0.0  25.047  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.161658  0.0  0.0  0.0 |      0.0       2.0  0.0  0.0  0.0 | 0.695377  0.314784  0.069388  0.0  0.0 | 3.180571 | 0.168268 | 0.397571 | 0.012175 | 0.541135 | 0.481178 | 0.481178 |  2.00491 |   2.0       2.0  0.0  0.0       2.0 | 147.218067  137.131839  201.988205  0.0  201.988205 |  54.770138   54.852736         0.0  0.0         0.0 | 141.984788  127.917561  0.0  0.0  0.0 |      0.0  1.537096  0.0  0.0  0.0 | 13.562754   7.131268  1.353353  0.0  0.0 | 18.330483 | 11.150315 |  2.00491 |
    | 02/01 16:00 | 37.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 290.938862  302.004664  309.356596  0.0  309.356596 | 0.0  0.0  0.902305  0.0   0.0 | 45.859  98.3884  216.788  45.859  162.591 | 0.0       0.0  0.0  0.0       0.0 | 45.859   98.3884  216.788  45.859   162.591 |  28.420627   56.173848  218.776205  0.0  164.579205 | 19.426578  34.199126        0.0  0.0        0.0 |  91.362027   91.362027   91.362027  0.0   91.362027 | 17.994569  17.994569  17.994569  0.0  17.994569 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 | 18.417734   7.351931  0.0  45.859  0.0 |  9.28239  3.007473  0.0  45.859  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.693011  0.242148  0.0  0.0  0.0 | 1.821862       2.0  0.0  0.0  0.0 |  0.70626  0.396973  0.066004  0.0  0.0 | 5.879917 | 0.211693 |  0.73499 | 0.012791 | 0.859575 | 0.536156 | 0.536156 | 2.233985 |   2.0       2.0  0.0  0.0       2.0 | 207.813473  215.717617  292.002027  0.0  291.702027 |  83.125389   86.287047   17.354569  0.0   17.654569 | 151.120131  132.262019  0.0  0.0  0.0 | 6.589379  2.302421  0.0  0.0  0.0 | 14.678356   8.734295  1.287349  0.0  0.0 | 23.998706 | 11.872514 | 2.233985 |
    | 02/01 17:00 |  8.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 323.906781  323.906781  323.906781  0.0  323.906781 | 0.0  0.0  0.902305  0.0   0.0 |  9.922  21.2872   46.904   9.922   35.178 | 0.0       0.0  0.0  0.0       0.0 |  9.922   21.2872   46.904   9.922    35.178 |  69.667573   85.844423  148.648653  0.0  137.128384 | 31.193289  37.447441   7.611943  0.0   7.406211 |  98.060761   98.060761   98.060761  0.0   98.060761 | 25.846019  25.846019  25.846019  0.0  25.846019 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.922  0.0 |      0.0       0.0  0.0   9.922  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.436737  0.028779  0.0  0.0  0.0 | 1.766082       2.0  0.0  0.0  0.0 | 0.759297  0.475154  0.062785  0.0  0.0 | 1.334196 |  0.24544 | 0.166775 | 0.013277 | 0.787086 | 0.712764 | 0.712764 | 2.969852 |   2.0       2.0  0.0  0.0       2.0 | 233.860668  237.354756  288.958135  0.0  288.152404 |  90.046113   86.552025   34.948645  0.0   35.754377 | 151.120131  132.262019  0.0  0.0  0.0 | 4.152642  0.273642  0.0  0.0  0.0 | 15.685141  10.259142  1.224564  0.0  0.0 | 25.087463 | 12.026011 | 2.969852 |
    | 02/01 18:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 334.397735  334.397735  334.397735  0.0  334.397735 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |  91.376852   96.554068  129.839697  0.0  124.668627 | 36.885929  36.698313  14.659084  0.0  14.682154 | 104.406788  104.406788  104.406788  0.0  104.406788 | 29.990946  29.990946  29.990946  0.0  29.990946 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   4.356  0.0 |      0.0       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0  0.273642  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.204851       0.0  0.0  0.0  0.0 | 1.715743  0.273642  0.0  0.0  0.0 | 0.807161  0.507073  0.059723  0.0  0.0 | 0.643383 | 0.252831 | 0.080423 | 0.013399 | 0.732983 | 0.787237 | 0.787237 | 3.280153 |   2.0       2.0  0.0  0.0       2.0 | 244.544611  249.365877  284.757227  0.0  283.674565 |  89.853124   85.031858   49.640507  0.0   50.723169 | 151.120131  132.262019  0.0  0.0  0.0 | 1.947791       0.0  0.0  0.0  0.0 | 16.593723  10.025711  1.164842  0.0  0.0 | 25.478015 | 12.093035 | 3.280153 |
    | 02/01 19:00 |  7.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 356.253888  356.253888  356.253888  0.0  356.253888 | 0.0  0.0  0.902305  0.0   0.0 |  9.075    19.47     42.9   9.075   32.175 | 0.0       0.0  0.0  0.0       0.0 |  9.075     19.47     42.9   9.075    32.175 | 102.149255  112.159273  153.970933  0.0  143.523838 | 41.323479  41.708462  23.326802  0.0  23.048897 | 120.370769  120.370769  120.370769  0.0  120.370769 | 35.883119  35.883119  35.883119  0.0  35.883119 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.075  0.0 |      0.0       0.0  0.0   9.075  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.947791       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.670314       0.0  0.0  0.0  0.0 | 0.850356   0.48896   0.05681  0.0  0.0 | 1.240469 | 0.259692 | 0.155059 | 0.013515 | 0.680197 | 0.755225 | 0.755225 | 3.146769 |   2.0       2.0  0.0  0.0       2.0 | 260.420132  266.197374  294.697064  0.0  293.036497 |  95.833757   90.056515   61.556824  0.0   63.217392 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 17.413681   9.536751  1.108032  0.0  0.0 | 26.458792 | 12.234578 | 3.146769 |
    | 02/01 20:00 | 18.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 410.165735  410.165735  410.165735  0.0  410.165735 | 0.0  0.0  0.902305  0.0   0.0 | 22.385   48.026   105.82  22.385   79.365 | 0.0       0.0  0.0  0.0       0.0 | 22.385    48.026   105.82  22.385    79.365 | 122.864197  145.201755  227.160779  0.0   201.42567 | 55.774692  59.078133   34.91311  0.0  34.193218 | 159.848924  159.848924  159.848924  0.0  159.848924 | 50.316811  50.316811  50.316811  0.0  50.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  22.385  0.0 |      0.0       0.0  0.0  22.385  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.849275  0.465113  0.054039  0.0  0.0 |  2.84254 | 0.277435 | 0.355317 | 0.013784 | 0.689734 | 0.708375 | 0.708375 | 2.951563 |   2.0       2.0  0.0  0.0       2.0 | 295.058865  302.048542  333.845209  0.0   331.16475 | 115.106869  108.117192   76.320526  0.0   79.000985 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 16.564406   9.071639  1.053992  0.0  0.0 | 29.023897 | 12.576112 | 2.951563 |
    | 02/01 21:00 | 15.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 455.043704  455.043704  455.043704  0.0  455.043704 | 0.0  0.0  0.902305  0.0   0.0 | 18.634  39.9784   88.088  18.634   66.066 | 0.0       0.0  0.0  0.0       0.0 | 18.634   39.9784   88.088  18.634    66.066 | 157.438041  177.637793  252.568414  0.0  230.408289 | 71.361694  72.506342  45.685321  0.0  45.823446 | 191.219608  191.219608  191.219608  0.0  191.219608 | 63.824096  63.824096  63.824096  0.0  63.824096 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  18.634  0.0 |      0.0       0.0  0.0  18.634  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.807856  0.442429  0.051404  0.0  0.0 | 2.366222 | 0.300584 | 0.295778 |  0.01413 |  0.69128 | 0.690702 | 0.690702 | 2.877927 |   2.0       2.0  0.0  0.0       2.0 | 326.494439  333.287958  361.224403  0.0  358.382069 | 128.549264  121.755746     93.8193  0.0   96.661635 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  15.75655    8.62921  1.002588  0.0  0.0 | 31.089535 |  12.85776 | 2.877927 |
    | 02/01 22:00 |  6.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 473.402873  473.402873  473.402873  0.0  473.402873 | 0.0  0.0  0.902305  0.0   0.0 |  7.623  16.3548   36.036   7.623   27.027 | 0.0       0.0  0.0  0.0       0.0 |  7.623   16.3548   36.036   7.623    27.027 | 185.358526  196.033453  235.469802  0.0  225.511749 | 77.308178  75.365051  55.609902  0.0  56.558955 | 203.295969  203.295969  203.295969  0.0  203.295969 | 70.106904  70.106904  70.106904  0.0  70.106904 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   7.623  0.0 |      0.0       0.0  0.0   7.623  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.768456  0.420852  0.048897  0.0  0.0 |    0.968 |  0.31417 |    0.121 | 0.014346 | 0.684773 | 0.689926 | 0.689926 | 2.874691 |   2.0       2.0  0.0  0.0       2.0 | 342.085889  347.797674   365.72657  0.0  363.533289 | 131.316984  125.605199  107.676303  0.0  109.869584 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.988094   8.208358  0.953692  0.0  0.0 | 31.743365 | 12.964414 | 2.874691 |
    | 02/01 23:00 |  1.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 478.939765  478.939765  478.939765  0.0  478.939765 | 0.0  0.0  0.902305  0.0   0.0 |  2.299   4.9324   10.868   2.299    8.151 | 0.0       0.0  0.0  0.0       0.0 |  2.299    4.9324   10.868   2.299     8.151 | 198.262243  203.642415  221.064023  0.0  217.315563 |  77.43963  74.692858   63.20685  0.0   64.23831 |  206.64527   206.64527   206.64527  0.0   206.64527 | 72.294496  72.294496  72.294496  0.0  72.294496 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.299  0.0 |      0.0       0.0  0.0   2.299  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.730978  0.400326  0.046512  0.0  0.0 | 0.291937 | 0.317307 | 0.036492 | 0.014417 | 0.669324 | 0.686283 | 0.686283 | 2.859512 |   2.0       2.0  0.0  0.0       2.0 | 348.122922  353.014129  362.815817  0.0  361.353996 | 130.816843  125.925637  116.123948  0.0   117.58577 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.257116   7.808032   0.90718  0.0  0.0 | 31.717995 | 12.986489 | 2.859512 |
    | 03/01 00:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 493.219119  493.219119  493.219119  0.0  493.219119 | 0.0  0.0  0.902686  0.0   0.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 | 204.528223  212.738588  236.654993  0.0  229.412995 | 80.340542  78.921577  70.312772  0.0  70.547771 | 216.382573  216.382573  216.382573  0.0  216.382573 | 76.836547  76.836547  76.836547  0.0  76.836547 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   5.929  0.0 |      0.0       0.0  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.695328  0.380802  0.044244  0.0  0.0 | 0.752889 | 0.319351 | 0.094111 | 0.014474 | 0.653742 | 0.675342 | 0.675342 | 2.813926 |   2.0       2.0  0.0  0.0       2.0 | 357.630288  362.303713  371.211396  0.0  369.684574 | 135.588831  130.915406  122.007723  0.0  123.534546 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.561788    7.42723  0.862936  0.0  0.0 | 32.151532 | 13.066126 | 2.813926 |
    | 03/01 01:00 |  2.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 501.087335  501.087335  501.087335  0.0  501.087335 | 0.0  0.0  0.902686  0.0   0.0 |  3.267   7.0092   15.444   3.267   11.583 | 0.0       0.0  0.0  0.0       0.0 |  3.267    7.0092   15.444   3.267    11.583 | 213.565721  219.302218  234.627313  0.0  230.211324 | 82.920398  80.926101  74.035807  0.0  74.590795 | 221.471596  221.471596  221.471596  0.0  221.471596 | 79.615739  79.615739  79.615739  0.0  79.615739 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   3.267  0.0 |      0.0       0.0  0.0   3.267  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.661416   0.36223  0.042086  0.0  0.0 | 0.414857 |  0.32198 | 0.051857 | 0.014539 | 0.639569 | 0.660739 | 0.660739 | 2.753081 |   2.0       2.0  0.0  0.0       2.0 | 363.189179   367.60189  374.139679  0.0  372.867845 | 137.898156  133.485444  126.947655  0.0   128.21949 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.900372   7.064999   0.82085  0.0  0.0 | 32.244409 | 13.103445 | 2.753081 |
    | 03/01 02:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 502.544412  502.544412  502.544412  0.0  502.544412 | 0.0  0.0  0.902686  0.0   0.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 218.403559  221.603573  227.380998  0.0  225.971212 | 83.288776  80.781762  76.566337  0.0  77.261123 | 222.042022  222.042022  222.042022  0.0  222.042022 | 80.502389  80.502389  80.502389  0.0  80.502389 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.629159  0.344564  0.040033  0.0  0.0 | 0.076825 |  0.32122 | 0.009603 | 0.014557 | 0.623002 | 0.645897 | 0.645897 | 2.691238 |   2.0       2.0  0.0  0.0       2.0 | 364.480659   368.65234  372.300704  0.0  371.423655 | 138.063753  133.892072  130.243708  0.0  131.120756 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.271214   6.720435  0.780817  0.0  0.0 | 32.000014 | 13.098491 | 2.691238 |
    | 03/01 03:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 503.127242  503.127242  503.127242  0.0  503.127242 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 219.496367  222.227265  225.160687  0.0  224.375469 | 83.290045  80.836347  78.527724  0.0  79.026943 |  222.01911   222.01911   222.01911  0.0   222.01911 | 81.108133  81.108133  81.108133  0.0  81.108133 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.598474  0.327759  0.038081  0.0  0.0 |  0.03073 | 0.318559 | 0.003841 | 0.014548 |  0.60541 | 0.630174 | 0.630174 | 2.625725 |   2.0       2.0  0.0  0.0       2.0 | 364.656418  368.712985  370.943126  0.0  370.265296 | 138.470825  134.414258  132.184116  0.0  132.861946 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  11.67274   6.392676  0.742736  0.0  0.0 | 31.712186 | 13.087785 | 2.625725 |
    | 03/01 04:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 504.584319  504.584319  504.584319  0.0  504.584319 | 0.0  0.0  0.902686  0.0   0.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 219.874572  222.866862   226.05113  0.0  225.000538 |  83.85767   81.55838  79.936113  0.0  80.271705 | 222.614941  222.614941  222.614941  0.0  222.614941 | 81.969378  81.969378  81.969378  0.0  81.969378 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.569286  0.311774  0.036224  0.0  0.0 | 0.076825 | 0.315924 | 0.009603 | 0.014539 | 0.588579 |  0.61326 |  0.61326 | 2.555251 |   2.0       2.0  0.0  0.0       2.0 | 365.049803  369.073064  371.006938  0.0    370.3647 | 139.534516  135.511255  133.577381  0.0  134.219619 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 11.103453   6.080901  0.706512  0.0  0.0 | 31.473087 | 13.082849 | 2.555251 |
    | 03/01 05:00 |  2.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 511.578288  511.578288  511.578288  0.0  511.578288 | 0.0  0.0  0.902686  0.0   0.0 |  2.904   6.2304   13.728   2.904   10.296 | 0.0       0.0  0.0  0.0       0.0 |  2.904    6.2304   13.728   2.904    10.296 |   221.1845  226.239983  236.278139  0.0   232.79694 | 86.303819  84.574736   82.03418  0.0  82.083379 | 227.116446  227.116446  227.116446  0.0  227.116446 | 84.461843  84.461843  84.461843  0.0  84.461843 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.904  0.0 |      0.0       0.0  0.0   2.904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.541522  0.296569  0.034457  0.0  0.0 | 0.368762 |    0.315 | 0.046095 | 0.014554 | 0.574068 | 0.596377 | 0.596377 | 2.484903 |   2.0       2.0  0.0  0.0       2.0 | 368.634765  372.735126  376.213244  0.0  375.320206 | 142.943524  138.843162  135.365044  0.0  136.258083 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.561931   5.784332  0.672055  0.0  0.0 | 31.526849 |  13.11439 | 2.484903 |
    | 03/01 06:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  512.74395   512.74395   512.74395  0.0   512.74395 | 0.0  0.0  0.902686  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 224.654326  227.589692  231.186173  0.0  230.127809 | 87.407963  85.026997  82.680115  0.0   83.16648 | 227.485902  227.485902  227.485902  0.0  227.485902 | 85.258048  85.258048  85.258048  0.0  85.258048 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.484  0.0 |      0.0       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.515111  0.282105  0.032777  0.0  0.0 |  0.06146 | 0.314004 | 0.007683 | 0.014568 | 0.560154 | 0.580864 | 0.580864 | 2.420268 |   2.0       2.0  0.0  0.0       2.0 | 369.119357  373.128936  375.440973  0.0  374.734299 | 143.624593  139.615014  137.302977  0.0  138.009651 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  10.04682   5.502227  0.639279  0.0  0.0 | 31.274305 | 13.107505 | 2.420268 |
    | 03/01 07:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 225.209233  227.873792  230.021905  0.0  229.334171 | 87.776717  85.389358  83.866045  0.0  84.267779 | 227.457017  227.457017  227.457017  0.0  227.457017 | 85.869764  85.869764  85.869764  0.0  85.869764 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.489989  0.268347  0.031178  0.0  0.0 |  0.03073 | 0.311338 | 0.003841 | 0.014558 | 0.545294 | 0.566429 | 0.566429 | 2.360119 |   2.0       2.0  0.0  0.0       2.0 | 369.020157  372.980961  374.660085  0.0  374.055145 | 144.306624   140.34582  138.666696  0.0  139.271636 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.556831    5.23388  0.608101  0.0  0.0 | 30.993698 | 13.096788 | 2.360119 |
    | 03/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.244235  227.661848   228.68676  0.0  228.317514 | 88.082546  85.664932  84.640021  0.0  85.009267 | 227.018951  227.018951  227.018951  0.0  227.018951 |  86.30783   86.30783   86.30783  0.0   86.30783 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.466092  0.255259  0.029657  0.0  0.0 |      0.0 | 0.308392 |      0.0 | 0.014544 | 0.530772 | 0.551925 | 0.551925 | 2.299687 |   2.0       2.0  0.0  0.0       2.0 |  368.44789  372.378063  373.632276  0.0  373.096582 | 144.878891  140.948718  139.694505  0.0  140.230199 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.090739   4.978621  0.578443  0.0  0.0 | 30.685305 | 13.082244 | 2.299687 |
    | 03/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.894931  227.293849  228.059401  0.0  227.732422 | 88.431849  86.032932   85.26738  0.0  85.594359 | 226.581591  226.581591  226.581591  0.0  226.581591 |  86.74519   86.74519   86.74519  0.0   86.74519 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.443361   0.24281  0.028211  0.0  0.0 |      0.0 | 0.305324 |      0.0 | 0.014528 | 0.516707 | 0.537362 | 0.537362 | 2.239007 |   2.0       2.0  0.0  0.0       2.0 | 367.787565  371.705805  372.794465  0.0  372.285751 | 145.539216  141.620976  140.532315  0.0   141.04103 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.647378   4.735811  0.550232  0.0  0.0 | 30.379981 | 13.067716 | 2.239007 |
    | 03/01 10:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.201027  514.201027  514.201027  0.0  514.201027 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 |  224.59314  227.312585  229.081589  0.0  228.387074 | 89.096641  86.792995  85.961192  0.0  86.226707 | 226.757529  226.757529  226.757529  0.0  226.757529 | 87.443498  87.443498  87.443498  0.0  87.443498 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.363  0.0 |      0.0       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.421738  0.230968  0.026835  0.0  0.0 | 0.046095 | 0.302516 | 0.005762 | 0.014515 | 0.503433 | 0.523138 | 0.523138 | 2.179741 |   2.0       2.0  0.0  0.0       2.0 | 367.604971  371.533949  372.826406  0.0  372.283206 | 146.596056  142.667078  141.374621  0.0  141.917821 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.225641   4.504842  0.523397  0.0  0.0 | 30.123561 | 13.058963 | 2.179741 |
    | 03/01 11:00 |  2.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 519.961252  521.777827  521.777827  0.0  521.777827 | 0.0  0.0  0.902686  0.0   0.0 |  3.146   6.7496   14.872   3.146   11.154 | 0.0       0.0  0.0  0.0       0.0 |  3.146    6.7496   14.872   3.146    11.154 | 225.493409  230.729522  241.140791  0.0  237.464602 | 91.853618  90.221105  87.932236  0.0  87.890425 | 231.636316  231.636316  231.636316  0.0  231.636316 | 90.141511  90.141511  90.141511  0.0  90.141511 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.816574        0.0  0.0   3.146  0.0 | 1.037141       0.0  0.0   3.146  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.037141       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.037141       0.0  0.0  0.0  0.0 | 0.426671  0.219704  0.025526  0.0  0.0 | 0.399492 | 0.301725 | 0.049937 |  0.01453 | 0.501088 | 0.510265 | 0.510265 | 2.126105 |   2.0       2.0  0.0  0.0       2.0 | 371.400895  375.455142  378.833931  0.0   377.94892 | 148.560358  146.322685  142.943896  0.0  143.828907 | 151.899565  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.836111   4.285139  0.497871  0.0  0.0 | 30.221328 |  13.09437 | 2.126105 |
    | 03/01 12:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.773236  523.119052  523.119052  0.0  523.119052 | 0.0  0.0  0.902686  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 228.776083   232.53573  237.211776  0.0  235.766768 |  92.03217  91.059297  88.570051  0.0  89.014059 | 232.417341  232.417341  232.417341  0.0  232.417341 | 90.701711  90.701711  90.701711  0.0  90.701711 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.345816        0.0  0.0   0.847  0.0 | 0.776316       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.776316       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.776316       0.0  0.0  0.0  0.0 | 0.450031  0.208989  0.024281  0.0  0.0 | 0.107556 | 0.301243 | 0.013444 | 0.014549 | 0.505336 | 0.503644 | 0.503644 | 2.098516 |   2.0       2.0  0.0  0.0       2.0 | 372.695169  376.177553  378.683497  0.0  377.942492 | 149.078068  146.941499  144.435556  0.0   145.17656 | 152.469064  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.162396    4.07615  0.473589  0.0  0.0 |  30.02764 | 13.093266 | 2.098516 |
    | 03/01 13:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.992672  523.475677  523.475677  0.0  523.475677 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 229.937343  232.782522  235.439858  0.0  234.597488 | 92.198893   91.11533  89.395194  0.0  89.808564 | 232.441067  232.441067  232.441067  0.0  232.441067 |  91.03461   91.03461   91.03461  0.0   91.03461 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.483005        0.0  0.0   0.363  0.0 | 0.861878       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.861878       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.861878       0.0  0.0  0.0  0.0 | 0.468048  0.198796  0.023097  0.0  0.0 | 0.046095 |  0.29901 | 0.005762 | 0.014543 | 0.506297 | 0.503923 | 0.503923 | 2.099678 |   2.0       2.0  0.0  0.0       2.0 | 372.851909  376.219298  378.040705  0.0  377.413071 | 149.140763  147.256379  145.434972  0.0  146.062606 | 153.090191  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.556227   3.877354  0.450492  0.0  0.0 | 29.774726 | 13.084485 | 2.099678 |
    | 03/01 14:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 522.008863  523.779537  523.779537  0.0  523.779537 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 230.094041  232.906119  235.140283  0.0  234.368133 | 92.261631  91.348357  90.051394  0.0  90.394544 |  232.45259   232.45259   232.45259  0.0   232.45259 | 91.326947  91.326947  91.326947  0.0  91.326947 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.770674        0.0  0.0   0.363  0.0 | 1.037465       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.037465       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.037465       0.0  0.0  0.0  0.0 | 0.491572  0.189101  0.021971  0.0  0.0 | 0.046095 | 0.296493 | 0.005762 | 0.014533 |  0.50897 |  0.50576 |  0.50576 | 2.107332 |   2.0       2.0  0.0  0.0       2.0 | 372.863473  376.148968  377.709011  0.0  377.124528 | 149.145389  147.630568  146.070525  0.0  146.655009 |   153.8234  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.102119   3.688253  0.428521  0.0  0.0 | 29.524328 | 13.075713 | 2.107332 |
    | 03/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.265413  523.098508  523.098508  0.0  523.098508 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 230.006331  232.520231  233.484587  0.0  233.123282 | 92.002532  91.259306   90.29495  0.0  90.656254 | 231.822278  231.822278  231.822278  0.0  231.822278 |  91.27623   91.27623   91.27623  0.0   91.27623 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.833096        0.0  0.0     0.0  0.0 | 1.084351       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.084351       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.084351       0.0  0.0  0.0  0.0 | 0.519349  0.179878  0.020899  0.0  0.0 |      0.0 | 0.293772 |      0.0 | 0.014521 | 0.513003 | 0.507996 | 0.507996 |  2.11665 |   2.0       2.0  0.0  0.0       2.0 | 372.332438  375.491016  376.686703  0.0  376.163524 | 148.932975  147.607492  146.411805  0.0  146.934984 | 154.572144  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.667122   3.508375  0.407622  0.0  0.0 | 29.230556 | 13.061193 |  2.11665 |
    | 03/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 520.375077  522.393471  522.393471  0.0  522.393471 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 229.475295  231.926846  232.665377  0.0  232.342229 | 91.790118  91.171662  90.433131  0.0  90.756279 | 231.186324  231.186324  231.186324  0.0  231.186324 | 91.207147  91.207147  91.207147  0.0  91.207147 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.018394        0.0  0.0     0.0  0.0 | 1.205615       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.205615       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.205615       0.0  0.0  0.0  0.0 | 0.549886  0.171105   0.01988  0.0  0.0 |      0.0 | 0.290849 |      0.0 | 0.014504 | 0.517896 | 0.511392 | 0.511392 | 2.130802 |   2.0       2.0  0.0  0.0       2.0 | 371.696484  374.790495   375.84765  0.0  375.347619 | 148.678593  147.602977  146.545821  0.0  147.045852 | 155.384924  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  11.32285   3.337269  0.387742  0.0  0.0 | 28.939707 | 13.046688 | 2.130802 |
    | 03/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  519.45772  521.617166  521.617166  0.0  521.617166 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.839341  231.300763  231.953183  0.0  231.644591 | 91.535736  91.092708  90.440288  0.0  90.748881 | 230.531069  230.531069  230.531069  0.0  230.531069 | 91.086097  91.086097  91.086097  0.0  91.086097 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.159446        0.0  0.0     0.0  0.0 | 1.303467       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.303467       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.303467       0.0  0.0  0.0  0.0 | 0.584272  0.162761   0.01891  0.0  0.0 |      0.0 | 0.287955 |      0.0 | 0.014488 | 0.524223 | 0.515893 | 0.515893 | 2.149553 |   2.0       2.0  0.0  0.0       2.0 | 371.041228    374.0608  375.065536  0.0  374.574097 | 148.416491  147.556365   146.55163  0.0  147.043069 | 156.240903  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.042046   3.174509  0.368832  0.0  0.0 | 28.651752 |   13.0322 | 2.149553 |
    | 03/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.519709   520.78661   520.78661  0.0   520.78661 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.184085    230.6373  231.256796  0.0  230.953787 | 91.273634  90.979866   90.36037  0.0  90.663379 | 229.861062  229.861062  229.861062  0.0  229.861062 | 90.925548  90.925548  90.925548  0.0  90.925548 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |    2.2669        0.0  0.0     0.0  0.0 | 1.383445       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.383445       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.383445       0.0  0.0  0.0  0.0 | 0.621314  0.154823  0.017988  0.0  0.0 |      0.0 |  0.28509 |      0.0 | 0.014472 |  0.53158 | 0.521594 | 0.521594 | 2.173307 |   2.0       2.0  0.0  0.0       2.0 | 370.371221  373.324563  374.309801  0.0  373.821372 | 148.148488  147.462047  146.476809  0.0  146.965237 | 157.124358  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.804176   3.019686  0.350844  0.0  0.0 | 28.366662 | 13.017728 | 2.173307 |
    | 03/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 517.565962  519.914725  519.914725  0.0  519.914725 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 227.514078  229.955069  230.561942  0.0  230.261087 | 91.005631  90.831541  90.224668  0.0  90.525523 | 229.179814  229.179814  229.179814  0.0  229.179814 | 90.734911  90.734911  90.734911  0.0  90.734911 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.348763        0.0  0.0     0.0  0.0 |  1.44966       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.44966       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  1.44966       0.0  0.0  0.0  0.0 | 0.660112  0.147272  0.017111  0.0  0.0 |      0.0 | 0.282253 |      0.0 | 0.014456 |  0.53966 | 0.528433 | 0.528433 | 2.201804 |   2.0       2.0  0.0  0.0       2.0 | 369.689973  372.589307  373.567673  0.0  373.080099 | 147.875989  147.325418  146.347052  0.0  146.834625 | 158.023461  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.593724   2.872414  0.333733  0.0  0.0 | 28.084409 | 13.003272 | 2.201804 |
    | 03/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 516.600219  519.011355  519.011355  0.0  519.011355 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  226.83283  229.262223  229.864233  0.0  229.564218 | 90.733132  90.652502  90.050492  0.0  90.350507 | 228.489997  228.489997  228.489997  0.0  228.489997 | 90.521358  90.521358  90.521358  0.0  90.521358 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.411136        0.0  0.0     0.0  0.0 | 1.505237       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.505237       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.505237       0.0  0.0  0.0  0.0 | 0.699985  0.140089  0.016276  0.0  0.0 |      0.0 | 0.279445 |      0.0 |  0.01444 | 0.548233 | 0.536139 | 0.536139 | 2.233913 |   2.0       2.0  0.0  0.0       2.0 | 369.000156  371.857081  372.833437  0.0  372.345878 | 147.600062  147.154274  146.177917  0.0  146.665476 |  158.92936  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.398976   2.732325  0.317456  0.0  0.0 | 27.804965 | 12.988832 | 2.233913 |
    | 03/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 515.625325  518.083995  518.083995  0.0  518.083995 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 226.143013   228.56269  229.162809  0.0   228.86313 | 90.457205  90.448665  89.848546  0.0  90.148225 | 227.793645  227.793645  227.793645  0.0  227.793645 |  90.29035   90.29035   90.29035  0.0   90.29035 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.458669        0.0  0.0     0.0  0.0 |  1.55256       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.55256       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  1.55256       0.0  0.0  0.0  0.0 | 0.740421  0.133257  0.015483  0.0  0.0 |      0.0 | 0.276664 |      0.0 | 0.014424 | 0.557121 | 0.544453 | 0.544453 | 2.268553 |   2.0       2.0  0.0  0.0       2.0 | 368.303804  371.128036  372.104273  0.0  371.616393 | 147.321522  146.955959  145.979722  0.0  146.467601 | 159.835469  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 15.211114   2.599068  0.301974  0.0  0.0 | 27.528301 | 12.974408 | 2.268553 |
    | 03/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.643451  517.138353  517.138353  0.0  517.138353 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.446661  227.858589  228.457962  0.0  228.158422 | 90.178664  90.225405  89.626033  0.0  89.925572 | 227.092305  227.092305  227.092305  0.0  227.092305 | 90.046047  90.046047  90.046047  0.0  90.046047 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.494902        0.0  0.0     0.0  0.0 | 1.593455       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.593455       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.593455       0.0  0.0  0.0  0.0 | 0.781035  0.126758  0.014727  0.0  0.0 |      0.0 | 0.273911 |      0.0 | 0.014408 | 0.566191 |  0.55317 |  0.55317 | 2.304875 |   2.0       2.0  0.0  0.0       2.0 | 367.602465  370.401752  371.378616  0.0  370.890276 | 147.040986  146.736601  145.759736  0.0  146.248076 | 160.736916  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 16.023534    2.47231  0.287246  0.0  0.0 |  27.25439 |     12.96 | 2.304875 |
    | 03/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.656246  516.178775  516.178775  0.0  516.178775 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.745322  227.151208  227.750276  0.0  227.450799 | 89.898129  89.987145  89.388076  0.0  89.687554 | 226.387159  226.387159  226.387159  0.0  226.387159 | 89.791616  89.791616  89.791616  0.0  89.791616 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.522529        0.0  0.0     0.0  0.0 | 1.629324       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.629324       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.629324       0.0  0.0  0.0  0.0 |  0.82154  0.120576  0.014009  0.0  0.0 |      0.0 | 0.271186 |      0.0 | 0.014392 | 0.575341 | 0.562136 | 0.562136 | 2.342233 |   2.0       2.0  0.0  0.0       2.0 | 366.897318  369.677703  370.655499  0.0  370.166637 | 146.758927  146.501072  145.523276  0.0  146.012138 | 161.630121  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 16.831319   2.351734  0.273237  0.0  0.0 | 26.983204 | 12.945608 | 2.342233 |
    | 04/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.663479  515.208571  515.208571  0.0  515.208571 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0  0.000576  0.0  0.0  0.001152 |    0.0       0.0      0.0     0.0       0.0 | 224.040175  226.441398  227.040334  0.0  226.740888 |  89.61607  89.737377   89.13844  0.0  89.437887 | 225.679105  225.679105  225.679105  0.0  225.679105 | 89.529467  89.529467  89.529467  0.0  89.529467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.545093        0.0  0.0     0.0  0.0 | 1.662219       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.662219       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.658434       0.0  0.0  0.0  0.0 | 0.861651  0.114695  0.013326  0.0  0.0 | 0.001202 | 0.268493 |  0.00015 | 0.014376 | 0.584479 | 0.571232 | 0.571232 | 2.380135 |   2.0  1.999424  0.0  0.0  1.998848 | 366.188199   368.95541   369.93427  0.0  369.444854 |  146.47528  146.253161  145.274302  0.0  145.763718 | 162.512995  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 17.628101   2.237039  0.259911  0.0  0.0 | 26.715912 | 12.931382 | 2.380135 |
    | 04/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.668518   514.22969   514.22969  0.0   514.22969 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.331056  225.729761  226.328635  0.0  226.029206 | 89.332422   89.47881  88.879937  0.0  89.179365 | 224.968418  224.968418  224.968418  0.0  224.968418 | 89.261271  89.261271  89.261271  0.0  89.261271 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.561172        0.0  0.0     0.0  0.0 | 1.691044       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.691044       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.618595       0.0  0.0  0.0  0.0 | 0.899531  0.109102  0.012676  0.0  0.0 |    0.023 | 0.265942 | 0.002875 | 0.014362 |  0.59309 | 0.580329 | 0.580329 | 2.418039 |   2.0  1.999424  0.0  0.0  1.998848 | 365.477513  368.234067  369.214053  0.0  368.724065 | 146.191005  145.995622  145.015637  0.0  145.505624 | 163.383123  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 18.347165   2.127937  0.247235  0.0  0.0 | 26.472969 | 12.919895 | 2.418039 |
    | 04/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.671177  513.244623  513.244623  0.0  513.244623 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  222.62037  225.016328  225.615167  0.0  225.315751 | 89.048148  89.213361  88.614523  0.0  88.913939 | 224.256032  224.256032  224.256032  0.0  224.256032 | 88.988592  88.988592  88.988592  0.0  88.988592 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.573447        0.0  0.0     0.0  0.0 | 1.717393       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.717393       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.582642       0.0  0.0  0.0  0.0 | 0.933717  0.103781  0.012058  0.0  0.0 | 0.042778 | 0.263624 | 0.005347 |  0.01435 | 0.600768 | 0.589099 | 0.589099 |  2.45458 |   2.0  1.999424  0.0  0.0  1.998848 | 364.765126  367.513771  368.494918  0.0  368.004346 | 145.906051  145.730853  144.749706  0.0  145.240277 | 164.239177  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  18.99609   2.024156  0.235177  0.0  0.0 | 26.252124 | 12.910892 |  2.45458 |
    | 04/01 03:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.766676   516.04297   516.04297  0.0   516.04297 | 0.0  0.0  0.903096  0.0   0.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |  1.573  3.374224    7.436   1.573  5.575848 | 222.346669  226.148508  231.531107  0.0   229.56884 | 89.897507  90.470339  89.149517  0.0  89.251632 | 226.191503  226.191503  226.191503  0.0  226.191503 | 89.851467  89.851467  89.851467  0.0  89.851467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  3.276294        0.0  0.0   1.573  0.0 |  2.20941       0.0  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019928       0.0  0.0  0.0  0.0 | 1.550196       0.0  0.0  0.0  0.0 | 0.964567  0.098719   0.01147  0.0  0.0 | 0.342541 |  0.26292 | 0.042818 | 0.014361 |  0.61558 | 0.597729 | 0.597729 | 2.490538 |   2.0       2.0  0.0  0.0       2.0 | 366.261912  369.083712  371.231314  0.0  370.542858 | 146.504765  146.959259  144.811656  0.0  145.500113 | 165.306061  132.262019  0.0  0.0  0.0 | 0.189482       0.0  0.0  0.0  0.0 | 19.581718   1.925437  0.223708  0.0  0.0 | 26.331745 | 12.939348 | 2.490538 |
    | 04/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.270864  514.782857  514.782857  0.0  514.782857 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.404769  226.039922  227.355189  0.0  226.933554 | 89.361908  90.003049  88.687781  0.0  89.109416 | 225.398666  225.398666  225.398666  0.0  225.398666 | 89.384192  89.384192  89.384192  0.0  89.384192 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.511993        0.0  0.0     0.0  0.0 | 1.716074       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.905557       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.520914       0.0  0.0  0.0  0.0 | 0.992409  0.093905   0.01091  0.0  0.0 | 0.122109 | 0.262614 | 0.015264 | 0.014378 | 0.616551 | 0.608583 | 0.608583 | 2.535762 |   2.0       2.0  0.0  0.0       2.0 |  365.90776  368.482456   369.91479  0.0  369.347969 | 146.363104  146.300402  144.868067  0.0  145.434888 |  166.10198  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 20.110224   1.831532  0.212797  0.0  0.0 |  26.19124 | 12.940234 | 2.535762 |
    | 04/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  511.28054  513.816706  513.816706  0.0  513.816706 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.050617  225.322111  226.197965  0.0  225.851361 | 89.220247  89.460746  88.584892  0.0  88.931496 | 224.691291  224.691291  224.691291  0.0  224.691291 | 89.125415  89.125415  89.125415  0.0  89.125415 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536166        0.0  0.0     0.0  0.0 | 1.749312       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.749312       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.494489       0.0  0.0  0.0  0.0 | 1.017534  0.089325  0.010378  0.0  0.0 | 0.080896 |  0.26101 | 0.010112 | 0.014376 | 0.622101 | 0.615222 | 0.615222 | 2.563427 |   2.0       2.0  0.0  0.0       2.0 | 365.200386  367.891636  369.048117  0.0  368.527899 | 146.080154   145.92507  144.768589  0.0  145.288807 | 166.888834  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 20.587178   1.742207  0.202419  0.0  0.0 | 26.011126 | 12.935971 | 2.563427 |
    | 04/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.286928  512.841258  512.841258  0.0  512.841258 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 222.343243  224.692074  225.398402  0.0  225.080676 | 88.937297  89.124632  88.418304  0.0   88.73603 | 223.981569  223.981569  223.981569  0.0  223.981569 | 88.859689  88.859689  88.859689  0.0  88.859689 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554329        0.0  0.0     0.0  0.0 | 1.778572       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.778572       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.470641       0.0  0.0  0.0  0.0 | 1.040209  0.084968  0.009872  0.0  0.0 | 0.097756 | 0.259302 | 0.012219 | 0.014372 | 0.626798 |  0.61983 |  0.61983 | 2.582625 |   2.0       2.0  0.0  0.0       2.0 | 364.490663   367.22113  368.271283  0.0  367.768792 | 145.796265  145.620127  144.569974  0.0  145.072466 | 167.664591  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  21.01761   1.657239  0.192547  0.0  0.0 | 25.849579 | 12.933818 | 2.582625 |
    | 04/01 07:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.955818  513.898731  513.898731  0.0  513.898731 | 0.0  0.0  0.903096  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 221.870249  225.005946  228.218517  0.0  227.018407 | 89.263679  89.652512  88.626741  0.0  88.825851 | 224.696076  224.696076  224.696076  0.0  224.696076 | 89.202655  89.202655  89.202655  0.0  89.202655 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.942913        0.0  0.0   0.847  0.0 | 2.068236       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.006494       0.0  0.0  0.0  0.0 |  1.44912       0.0  0.0  0.0  0.0 | 1.060673  0.080824  0.009391  0.0  0.0 | 0.282438 | 0.258615 | 0.035305 | 0.014383 | 0.633951 | 0.624817 | 0.624817 | 2.603402 |   2.0       2.0  0.0  0.0       2.0 | 364.968442   367.75206  369.392842  0.0  368.789461 | 145.987377  146.146671  144.505889  0.0   145.10927 | 168.539268  132.262019  0.0  0.0  0.0 | 0.061743       0.0  0.0  0.0  0.0 | 21.406057   1.576414  0.183156  0.0  0.0 | 25.873403 |  12.95474 | 2.603402 |
    | 04/01 08:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.181898  513.932503  513.932503  0.0  513.932503 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 222.246491  225.198159  227.670615  0.0   226.79139 | 89.193327  89.738972  88.516116  0.0  88.823341 | 224.756025  224.756025  224.756025  0.0  224.756025 | 89.176478  89.176478  89.176478  0.0  89.176478 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.750605        0.0  0.0   0.484  0.0 | 1.953307       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.001432       0.0  0.0  0.0  0.0 | 1.429697       0.0  0.0  0.0  0.0 |  1.07914  0.076883  0.008933  0.0  0.0 | 0.242509 | 0.258653 | 0.030314 | 0.014403 | 0.637332 | 0.630687 | 0.630687 | 2.627862 |   2.0       2.0  0.0  0.0       2.0 | 365.129927  367.807526  369.406253  0.0  368.810097 | 146.051971  146.124977   144.52625  0.0  145.122406 | 169.336566  132.262019  0.0  0.0  0.0 | 0.013617       0.0  0.0  0.0  0.0 | 21.756614   1.499532  0.174224  0.0  0.0 | 25.857258 | 12.970651 | 2.627862 |
    | 04/01 09:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.581776  513.165993  513.165993  0.0  513.165993 | 0.0  0.0  0.903096  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 | 222.306591  224.815129  226.159005  0.0  225.667272 | 88.996307  89.376974  88.345498  0.0  88.694231 | 224.225981  224.225981  224.225981  0.0  224.225981 | 88.940012  88.940012  88.940012  0.0  88.940012 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.584217        0.0  0.0   0.121  0.0 | 1.852552       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.86617       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.412169       0.0  0.0  0.0  0.0 | 1.095806  0.073133  0.008497  0.0  0.0 | 0.159492 | 0.258079 | 0.019937 | 0.014415 | 0.640798 |  0.63549 |  0.63549 | 2.647873 |   2.0       2.0  0.0  0.0       2.0 | 364.701269  367.372779  368.685229  0.0  368.137806 | 145.880507  145.793214  144.480764  0.0  145.028188 | 170.068231  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.072978   1.426399  0.165727  0.0  0.0 | 25.758672 | 12.976172 | 2.647873 |
    | 04/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.579336  513.337725  513.337725  0.0  513.337725 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 221.979417  224.762609  227.032845  0.0   226.18834 | 89.086359  89.441784  88.421148  0.0  88.693653 | 224.325723  224.325723  224.325723  0.0  224.325723 | 89.012002  89.012002  89.012002  0.0  89.012002 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.758389        0.0  0.0   0.484  0.0 | 1.994536       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.994536       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.396351       0.0  0.0  0.0  0.0 | 1.110846  0.069566  0.008083  0.0  0.0 |  0.25136 | 0.257556 |  0.03142 | 0.014427 |   0.6443 | 0.639211 | 0.639211 | 2.663378 |   2.0       2.0  0.0  0.0       2.0 | 364.699526  367.433493  368.906107  0.0  368.331188 |  145.87981  145.904232  144.431618  0.0  145.006537 | 170.832084  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.358483   1.356832  0.157644  0.0  0.0 | 25.752476 | 12.993165 | 2.663378 |
    | 04/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  509.74056  512.276806  512.276806  0.0  512.276806 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.842383  224.278811  225.177685  0.0  224.826758 | 88.736953  89.058914  88.160041  0.0  88.510967 | 223.591306  223.591306  223.591306  0.0  223.591306 | 88.685501  88.685501  88.685501  0.0  88.685501 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536246        0.0  0.0     0.0  0.0 |  1.85042       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.85042       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.382076       0.0  0.0  0.0  0.0 | 1.124419  0.066174  0.007688  0.0  0.0 | 0.148681 | 0.256982 | 0.018585 | 0.014439 | 0.647317 |  0.64269 |  0.64269 | 2.677877 |   2.0       2.0  0.0  0.0       2.0 |   364.1004  366.785987  367.959728  0.0  367.435736 |  145.64016  145.490819  144.317078  0.0  144.841071 |  171.51791  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.616139   1.290659  0.149956  0.0  0.0 | 25.644174 | 12.997311 | 2.677877 |
    | 04/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 508.746874  511.301327  511.301327  0.0  511.301327 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.243257  223.587629  224.303125  0.0  223.983707 | 88.497303  88.689177  87.973681  0.0  88.293099 |  222.88153   222.88153   222.88153  0.0   222.88153 | 88.419797  88.419797  88.419797  0.0  88.419797 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554453        0.0  0.0     0.0  0.0 | 1.878698       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.878698       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.369193       0.0  0.0  0.0  0.0 | 1.136669  0.062946  0.007313  0.0  0.0 | 0.161748 |  0.25597 | 0.020218 | 0.014445 | 0.649525 | 0.645857 | 0.645857 | 2.691069 |   2.0       2.0  0.0  0.0       2.0 | 363.390624  366.119888  367.178132  0.0  366.673558 |  145.35625  145.181439  144.123195  0.0  144.627769 | 172.193665  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.848664   1.227713  0.142642  0.0  0.0 | 25.549952 | 13.003085 | 2.691069 |
    | 04/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 507.750567  510.318845  510.318845  0.0  510.318845 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 220.533481  222.908881  223.553184  0.0  223.245979 | 88.213393  88.392446  87.748143  0.0  88.055348 | 222.169882  222.169882  222.169882  0.0  222.169882 | 88.148963  88.148963  88.148963  0.0  88.148963 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.568278        0.0  0.0     0.0  0.0 | 1.903778       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.903778       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.357567       0.0  0.0  0.0  0.0 | 1.147723  0.059876  0.006957  0.0  0.0 | 0.173401 |  0.25509 | 0.021675 | 0.014452 | 0.651505 | 0.648434 | 0.648434 |  2.70181 |   2.0       2.0  0.0  0.0       2.0 | 362.678976  365.420888   366.43483  0.0  365.937461 | 145.071591  144.897957  143.884015  0.0  144.381384 | 172.858165  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 23.058508   1.167837  0.135686  0.0  0.0 | 25.468262 | 13.010308 |  2.70181 |
    | 04/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 506.752198  509.331046  509.331046  0.0  509.331046 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.821833  222.208114   222.82468  0.0  222.522236 | 87.928733  88.110731  87.494165  0.0  87.796609 | 221.456761  221.456761  221.456761  0.0  221.456761 | 87.874284  87.874284  87.874284  0.0  87.874284 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.578848        0.0  0.0     0.0  0.0 | 1.926396       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.926396       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.347075       0.0  0.0  0.0  0.0 | 1.157699  0.056956  0.006617  0.0  0.0 | 0.183911 |  0.25433 | 0.022989 | 0.014461 | 0.653283 | 0.650569 | 0.650569 | 2.710706 |   2.0       2.0  0.0  0.0       2.0 | 361.965856  364.709536  365.706911  0.0  365.211987 | 144.786342   144.62151  143.624134  0.0  144.119059 | 173.510617  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 23.247883    1.11088  0.129068  0.0  0.0 | 25.397844 | 13.018836 | 2.710706 |
    | 04/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 505.752249  508.339181  508.339181  0.0  508.339181 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.108713  221.498342  222.104076  0.0  221.803495 | 87.643485  87.832704   87.22697  0.0  87.527551 | 220.742512  220.742512  220.742512  0.0  220.742512 | 87.596669  87.596669  87.596669  0.0  87.596669 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.586932        0.0  0.0     0.0  0.0 |  1.94705       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.94705       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.337606       0.0  0.0  0.0  0.0 | 1.166702  0.054178  0.006295  0.0  0.0 | 0.193474 | 0.253677 | 0.024184 | 0.014471 | 0.654878 | 0.652448 | 0.652448 | 2.718532 |   2.0       2.0  0.0  0.0       2.0 | 361.251606  363.993706  364.985347  0.0  364.491004 | 144.500643  144.345475  143.353834  0.0  143.848177 | 174.150499  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 23.418787   1.056702  0.122773  0.0  0.0 | 25.337641 |  13.02855 | 2.718532 |
    | 04/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 504.751087  507.344207  507.344207  0.0  507.344207 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 218.394464  220.784715  221.386207  0.0  221.086357 | 87.357785  87.554466  86.952974  0.0  87.252824 | 220.027397  220.027397  220.027397  0.0  220.027397 | 87.316811  87.316811  87.316811  0.0  87.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.59312        0.0  0.0     0.0  0.0 | 1.966129       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.966129       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.329061       0.0  0.0  0.0  0.0 | 1.174827  0.051536  0.005988  0.0  0.0 | 0.202244 | 0.253122 |  0.02528 | 0.014482 | 0.656313 | 0.654129 | 0.654129 |  2.72554 |   2.0       2.0  0.0  0.0       2.0 | 360.536491  363.276387  364.266537  0.0  363.772044 | 144.214596   144.06782   143.07767  0.0  143.572164 |  174.77749  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  23.57302   1.005166  0.116786  0.0  0.0 | 25.286764 | 13.039348 |  2.72554 |
    | 04/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 503.748995  506.346853  506.346853  0.0  506.346853 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 217.679348  220.069317   220.66914  0.0  220.369581 | 87.071739   87.27489  86.675067  0.0  86.974626 | 219.311616  219.311616  219.311616  0.0  219.311616 | 87.035237  87.035237  87.035237  0.0  87.035237 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.597858        0.0  0.0     0.0  0.0 | 1.983931       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.983931       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.321349       0.0  0.0  0.0  0.0 |  1.18216  0.049023  0.005696  0.0  0.0 | 0.210343 | 0.252656 | 0.026293 | 0.014495 | 0.657605 |  0.65564 |  0.65564 | 2.731834 |   2.0       2.0  0.0  0.0       2.0 | 359.820711  362.558686  363.549013  0.0  363.054079 | 143.928284  143.788167  142.797841  0.0  143.292775 | 175.391417  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 23.712209   0.956144   0.11109  0.0  0.0 | 25.244451 | 13.051146 | 2.731834 |
    | 04/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 502.746187  505.347677  505.347677  0.0  505.347677 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.963568  219.353023  219.952184  0.0  219.652742 | 86.785427  86.993831   86.39467  0.0  86.694112 | 218.595325  218.595325  218.595325  0.0  218.595325 | 86.752352  86.752352  86.752352  0.0  86.752352 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.60149        0.0  0.0     0.0  0.0 | 2.000686       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.000065       0.0  0.0  0.0  0.0 |  1.31439       0.0  0.0  0.0  0.0 | 1.188777  0.046632  0.005418  0.0  0.0 | 0.217654 | 0.252271 | 0.027207 | 0.014508 | 0.658794 | 0.657001 | 0.657001 | 2.737505 |   2.0       2.0  0.0  0.0       2.0 | 359.104419  361.840989  362.832154  0.0  362.336662 | 143.641768  143.506688  142.515523  0.0  143.011015 | 175.992222  132.262019  0.0  0.0  0.0 |  0.00062       0.0  0.0  0.0  0.0 | 23.837822   0.909512  0.105672  0.0  0.0 | 25.209834 | 13.063845 | 2.737505 |
    | 04/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 501.742828  504.347104  504.347104  0.0  504.347104 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.247276  218.636219  219.235114  0.0  218.935721 | 86.498911  86.711458  86.112564  0.0  86.411956 |  217.87864   217.87864   217.87864  0.0   217.87864 | 86.468464  86.468464  86.468464  0.0  86.468464 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.604276        0.0  0.0     0.0  0.0 | 2.016573       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.001636       0.0  0.0  0.0  0.0 | 1.308109       0.0  0.0  0.0  0.0 | 1.194749  0.044357  0.005154  0.0  0.0 | 0.219648 | 0.251937 | 0.027456 | 0.014523 |  0.66035 | 0.658274 | 0.658274 | 2.742807 |   2.0       2.0  0.0  0.0       2.0 | 358.387734   361.12341   362.11568  0.0  361.619581 | 143.355094  143.223694  142.231424  0.0  142.727523 | 176.579925  132.262019  0.0  0.0  0.0 | 0.015557       0.0  0.0  0.0  0.0 | 23.951182   0.865155  0.100518  0.0  0.0 | 25.177546 | 13.076778 | 2.742807 |
    | 04/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 500.739045   503.34546   503.34546  0.0   503.34546 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 215.530592  217.919094  218.517877  0.0  218.218507 | 86.212237   86.42801  85.829227  0.0  86.128597 | 217.161652  217.161652  217.161652  0.0  217.161652 | 86.183808  86.183808  86.183808  0.0  86.183808 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.606415        0.0  0.0     0.0  0.0 | 2.031731       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   0.0045       0.0  0.0  0.0  0.0 | 1.302441       0.0  0.0  0.0  0.0 | 1.200138  0.042194  0.004902  0.0  0.0 | 0.221447 | 0.251624 | 0.027681 | 0.014537 | 0.662196 | 0.659703 | 0.659703 | 2.748763 |   2.0       2.0  0.0  0.0       2.0 | 357.670746  360.405968  361.399455  0.0  360.902725 | 143.068299  142.939492  141.946005  0.0  142.442734 | 177.154609  132.262019  0.0  0.0  0.0 | 0.042788       0.0  0.0  0.0  0.0 | 24.053485    0.82296  0.095616  0.0  0.0 | 25.147369 | 13.089922 | 2.748763 |
    | 04/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 499.734933  502.342992  502.342992  0.0  502.342992 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.813603  217.201748  217.800482  0.0  217.501124 | 85.925441  86.143711  85.544978  0.0  85.844336 | 216.444429  216.444429  216.444429  0.0  216.444429 | 85.898563  85.898563  85.898563  0.0  85.898563 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.608059        0.0  0.0     0.0  0.0 | 2.046267       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.008475       0.0  0.0  0.0  0.0 | 1.297326       0.0  0.0  0.0  0.0 | 1.205002  0.040136  0.004663  0.0  0.0 | 0.223071 | 0.251332 | 0.027884 | 0.014552 | 0.664286 | 0.661413 | 0.661413 | 2.755888 |   2.0       2.0  0.0  0.0       2.0 | 356.953524  359.688648  360.683402  0.0  360.186031 | 142.781409  142.654344   141.65959  0.0  142.156961 | 177.716401  132.262019  0.0  0.0  0.0 | 0.080581       0.0  0.0  0.0  0.0 | 24.145809   0.782824  0.090953  0.0  0.0 | 25.119108 | 13.103254 | 2.755888 |
    | 04/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 498.730567  501.339893  501.339893  0.0  501.339893 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.096381  216.484243  217.082951  0.0    216.7836 | 85.638552   85.85875  85.260042  0.0  85.559392 | 215.727025  215.727025  215.727025  0.0  215.727025 | 85.612867  85.612867  85.612867  0.0  85.612867 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.609325        0.0  0.0     0.0  0.0 | 2.060266       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.013403       0.0  0.0  0.0  0.0 |  1.29271       0.0  0.0  0.0  0.0 | 1.209391  0.038179  0.004436  0.0  0.0 | 0.224537 | 0.251058 | 0.028067 | 0.014567 | 0.666583 | 0.663386 | 0.663386 |  2.76411 |   2.0       2.0  0.0  0.0       2.0 |  356.23612  358.971431  359.967476  0.0  359.469456 | 142.494448  142.368462  141.372416  0.0  141.870437 |  178.26546  132.262019  0.0  0.0  0.0 | 0.127443       0.0  0.0  0.0  0.0 | 24.229128   0.744645  0.086517  0.0  0.0 | 25.092586 | 13.116755 |  2.76411 |
    | 04/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 497.726004  500.336306  500.336306  0.0  500.336306 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 213.378977  215.766617  216.365309  0.0  216.065964 | 85.351591  85.573276  84.974584  0.0  85.273929 |  215.00948   215.00948   215.00948  0.0   215.00948 | 85.326826  85.326826  85.326826  0.0  85.326826 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.610302        0.0  0.0     0.0  0.0 | 2.073792       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01915       0.0  0.0  0.0  0.0 | 1.288544       0.0  0.0  0.0  0.0 | 1.213352  0.036317  0.004219  0.0  0.0 | 0.225859 | 0.250801 | 0.028232 | 0.014582 | 0.669053 | 0.665584 | 0.665584 | 2.773266 |   2.0       2.0  0.0  0.0       2.0 | 355.518575  358.254294  359.251648  0.0  358.752972 |  142.20743  142.082012  141.084658  0.0  141.583334 |  178.80197  132.262019  0.0  0.0  0.0 | 0.182085       0.0  0.0  0.0  0.0 |  24.30432   0.708329  0.082297  0.0  0.0 | 25.067644 | 13.130405 | 2.773266 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

The submodel rconcmodel is optional. We repeat the last integration test, but without
a submodel for runoff concentration.

>>> model.rconcmodel = None

.. integration-test::

    >>> test("hland_96p_snow_redistribution",
    ...      axis1=(states.sp, states.wc), axis2=(factors.tc, fluxes.pc))
    |        date |    p |    t |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                 swe |                          gact |                                        pc |                                ei |                                          tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                          melt |                               refr |                                    in_ |                                    r |                      sr |                      ea |                                dp |                      el |                      rs |                                ri |                               gr1 |                                    rg1 |      gr2 |      rg2 |      gr3 |      rg3 |     inrc |    outrc |       rt |       qt |                                  ic |                                                  sp |                                                  wc |                                    sm |                               suz |                                      sg1 |       sg2 |       sg3 |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.487706  0.487706  0.487706  0.0  0.0 |      0.0 | 0.099502 |      0.0 | 0.011105 | 0.493397 | 0.493397 | 0.493397 | 2.055823 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.512294   9.512294  9.512294  0.0  0.0 |  9.900498 |  9.988895 | 2.055823 |
    | 01/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.46392   0.46392   0.46392  0.0  0.0 |      0.0 | 0.098512 |      0.0 | 0.011093 | 0.473433 | 0.473433 | 0.473433 | 1.972639 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.048374   9.048374  9.048374  0.0  0.0 |  9.801987 |  9.977802 | 1.972639 |
    | 01/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.441294  0.441294  0.441294  0.0  0.0 |      0.0 | 0.097531 |      0.0 |  0.01108 | 0.454406 | 0.454406 | 0.454406 |  1.89336 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   8.60708    8.60708   8.60708  0.0  0.0 |  9.704455 |  9.966722 |  1.89336 |
    | 01/01 03:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.419772  0.419772  0.419772  0.0  0.0 |      0.0 | 0.096561 |      0.0 | 0.011068 | 0.436271 | 0.436271 | 0.436271 | 1.817798 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.187308   8.187308  8.187308  0.0  0.0 |  9.607894 |  9.955654 | 1.817798 |
    | 01/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   0.3993    0.3993    0.3993  0.0  0.0 |      0.0 |   0.0956 |      0.0 | 0.011056 | 0.418985 | 0.418985 | 0.418985 | 1.745772 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.788008   7.788008  7.788008  0.0  0.0 |  9.512294 |  9.944598 | 1.745772 |
    | 01/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.379826  0.379826  0.379826  0.0  0.0 |      0.0 | 0.094649 |      0.0 | 0.011043 | 0.402507 | 0.402507 | 0.402507 | 1.677111 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.408182   7.408182  7.408182  0.0  0.0 |  9.417645 |  9.933555 | 1.677111 |
    | 01/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.361301  0.361301  0.361301  0.0  0.0 |      0.0 | 0.093707 |      0.0 | 0.011031 | 0.386797 | 0.386797 | 0.386797 | 1.611653 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  7.046881   7.046881  7.046881  0.0  0.0 |  9.323938 |  9.922524 | 1.611653 |
    | 01/01 07:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.34368   0.34368   0.34368  0.0  0.0 |      0.0 | 0.092775 |      0.0 | 0.011019 | 0.371818 | 0.371818 | 0.371818 | 1.549243 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |    6.7032     6.7032    6.7032  0.0  0.0 |  9.231163 |  9.911505 | 1.549243 |
    | 01/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.326919  0.326919  0.326919  0.0  0.0 |      0.0 | 0.091852 |      0.0 | 0.011007 | 0.357536 | 0.357536 | 0.357536 | 1.489734 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.376282   6.376282  6.376282  0.0  0.0 |  9.139312 |  9.900498 | 1.489734 |
    | 01/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.310975  0.310975  0.310975  0.0  0.0 |      0.0 | 0.090938 |      0.0 | 0.010994 | 0.343917 | 0.343917 | 0.343917 | 1.432986 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  6.065307   6.065307  6.065307  0.0  0.0 |  9.048374 |  9.889504 | 1.432986 |
    | 01/01 10:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.295808  0.295808  0.295808  0.0  0.0 |      0.0 | 0.090033 |      0.0 | 0.010982 | 0.330928 | 0.330928 | 0.330928 | 1.378865 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.769498   5.769498  5.769498  0.0  0.0 |  8.958341 |  9.878522 | 1.378865 |
    | 01/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.281382  0.281382  0.281382  0.0  0.0 |      0.0 | 0.089137 |      0.0 |  0.01097 | 0.318539 | 0.318539 | 0.318539 | 1.327244 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.488116   5.488116  5.488116  0.0  0.0 |  8.869204 |  9.867552 | 1.327244 |
    | 01/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.267659  0.267659  0.267659  0.0  0.0 |      0.0 |  0.08825 |      0.0 | 0.010958 | 0.306721 | 0.306721 | 0.306721 | 1.278004 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.220458   5.220458  5.220458  0.0  0.0 |  8.780954 |  9.856594 | 1.278004 |
    | 01/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.254605  0.254605  0.254605  0.0  0.0 |      0.0 | 0.087372 |      0.0 | 0.010946 | 0.295447 | 0.295447 | 0.295447 | 1.231029 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.965853   4.965853  4.965853  0.0  0.0 |  8.693582 |  9.845648 | 1.231029 |
    | 01/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.242188  0.242188  0.242188  0.0  0.0 |      0.0 | 0.086503 |      0.0 | 0.010934 |  0.28469 |  0.28469 |  0.28469 |  1.18621 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.723666   4.723666  4.723666  0.0  0.0 |   8.60708 |  9.834715 |  1.18621 |
    | 01/01 15:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0    1.144   0.242       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.230376  0.230376  0.230376  0.0  0.0 |  0.03073 | 0.085795 | 0.003841 | 0.010924 | 0.274571 | 0.274571 | 0.274571 | 1.144047 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   4.49329    4.49329   4.49329  0.0  0.0 |  8.552015 |  9.827632 | 1.144047 |
    | 01/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  0.21914   0.21914   0.21914  0.0  0.0 |      0.0 | 0.085094 |      0.0 | 0.010914 | 0.264919 | 0.264919 | 0.264919 |  1.10383 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.274149   4.274149  4.274149  0.0  0.0 |  8.466921 |  9.816719 |  1.10383 |
    | 01/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.208453  0.208453  0.208453  0.0  0.0 |      0.0 | 0.084247 |      0.0 | 0.010901 | 0.255568 | 0.255568 | 0.255568 | 1.064865 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.065697   4.065697  4.065697  0.0  0.0 |  8.382674 |  9.805817 | 1.064865 |
    | 01/01 18:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    1.224502        8.58  0.0       4.435 | 0.0  0.0  0.901953  0.0   0.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |    0.0     1.894    7.436   1.573     4.435 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0   0.669498  0.0   1.573  0.0 |      0.0  0.167375  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.167375  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0  0.167375  0.0  0.0  0.0 | 0.198286  0.202402  0.198286  0.0  0.0 | 0.199746 | 0.084404 | 0.024968 | 0.010903 |  0.24868 |  0.24868 |  0.24868 | 1.036168 | 1.815       2.0  0.0  0.0       2.0 |        0.0    0.874644        8.58  0.0       4.435 |        0.0    0.349858         0.0  0.0         0.0 |      100.0  100.502124  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   3.86741   4.030669   3.86741  0.0  0.0 |  8.498015 |  9.819882 | 1.036168 |
    | 01/01 19:00 |  5.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   10.249462      40.612  0.0      28.459 | 0.0  0.0  0.901953  0.0   0.0 |  6.776  14.5376   32.032   6.776   24.024 | 0.0       0.0  0.0  0.0       0.0 |  6.591   14.5376   32.032   6.776    24.024 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     6.591    5.51264  0.0   6.776  0.0 |  1.64775  1.392035  0.0   6.776  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.64775  1.392035  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  1.64775  1.392035  0.0  0.0  0.0 | 0.229132  0.230806  0.188616  0.0  0.0 | 0.860444 | 0.088845 | 0.107556 | 0.010965 | 0.268804 | 0.268804 | 0.268804 | 1.120016 |   2.0       2.0  0.0  0.0       2.0 |        0.0    7.321044      40.612  0.0      28.459 |        0.0    2.928418         0.0  0.0         0.0 |  104.94325  104.622729  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.286029   5.191898  3.678794  0.0  0.0 |  9.269615 |  9.916473 | 1.120016 |
    | 01/01 20:00 |  2.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   14.950102        57.2  0.0        40.9 | 0.0  0.0  0.901953  0.0   0.0 |  3.509   7.5284   16.588   3.509   12.441 | 0.0       0.0  0.0  0.0       0.0 |  3.509    7.5284   16.588   3.509    12.441 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     3.509    2.82776  0.0   3.509  0.0 | 0.966123  0.773811  0.0   3.509  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.966123  0.773811  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.966123  0.773811  0.0  0.0  0.0 | 0.281558  0.272239  0.179417  0.0  0.0 | 0.445587 | 0.094455 | 0.055698 | 0.011043 | 0.300798 | 0.300798 | 0.300798 | 1.253324 |   2.0       2.0  0.0  0.0       2.0 |        0.0   10.678644        57.2  0.0        40.9 |        0.0    4.271458         0.0  0.0         0.0 | 107.486127  106.676678  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  5.970594    5.69347  3.499377  0.0  0.0 |  9.620748 |  9.961129 | 1.253324 |
    | 01/01 21:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   22.853942      85.228  0.0      61.921 | 0.0  0.0  0.901953  0.0   0.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     5.929    4.81656  0.0   5.929  0.0 | 1.712483  1.370301  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.712483  1.370301  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.712483  1.370301  0.0  0.0  0.0 | 0.333297  0.311367  0.170667  0.0  0.0 | 0.752889 |  0.09948 | 0.094111 | 0.011114 | 0.331485 | 0.331485 | 0.331485 | 1.381186 |   2.0       2.0  0.0  0.0       2.0 |        0.0   16.324244      85.228  0.0      61.921 |        0.0    6.529698         0.0  0.0         0.0 | 111.702644  110.122937  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |   7.34978   6.752403  3.328711  0.0  0.0 | 10.274157 | 10.044126 | 1.381186 |
    | 01/01 22:00 | 10.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   39.886902      145.86  0.0     107.395 | 0.0  0.0  0.901953  0.0   0.0 | 12.826  27.5176   60.632  12.826   45.474 | 0.0       0.0  0.0  0.0       0.0 | 12.826   27.5176   60.632  12.826    45.474 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.826   10.48464  0.0  12.826  0.0 | 4.000904  3.178697  0.0  12.826  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.190411  0.112168  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 |  0.40763  0.378496  0.162343  0.0  0.0 | 1.628698 | 0.110346 | 0.203587 | 0.011267 | 0.476165 | 0.476165 | 0.476165 |  1.98402 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.490644      145.86  0.0     107.395 |        0.0   11.396258         0.0  0.0         0.0 |  120.52774  117.428881  0.0  0.0  0.0 | 1.810493  1.066529  0.0  0.0  0.0 |   8.94215   8.373908  3.166368  0.0  0.0 | 11.792509 | 10.236446 |  1.98402 |
    | 01/01 23:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   40.103062     146.432  0.0     107.824 | 0.0  0.0  0.901953  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121    0.04344  0.0   0.121  0.0 | 0.043944  0.014975  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.854437  1.081504  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.854437  1.081504  0.0  0.0  0.0 | 0.481712  0.434993  0.154426  0.0  0.0 | 0.015365 | 0.117414 | 0.001921 | 0.011369 | 0.427651 | 0.427651 | 0.427651 | 1.781878 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.645044     146.432  0.0     107.824 |        0.0   11.458018         0.0  0.0         0.0 | 120.604796  117.457345  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.314875   9.020419  3.011942  0.0  0.0 |  11.69046 | 10.226998 | 1.781878 |
    | 02/01 00:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   41.280182     150.436  0.0     110.827 | 0.0  0.0  0.902305  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.847    0.64008  0.0   0.847  0.0 | 0.308001  0.220767  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.308001  0.220767  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.308001  0.220767  0.0  0.0  0.0 | 0.510636  0.445359  0.146894  0.0  0.0 | 0.107556 | 0.116858 | 0.013444 | 0.011364 | 0.438028 | 0.438028 | 0.438028 | 1.825115 |   2.0       2.0  0.0  0.0       2.0 |        0.0   29.485844     150.436  0.0     110.827 |        0.0   11.794338         0.0  0.0         0.0 | 121.143795  117.876658  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.112241   8.795827  2.865048  0.0  0.0 | 11.681158 | 10.229078 | 1.825115 |
    | 02/01 01:00 |  3.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   46.140982     167.596  0.0     123.697 | 0.0  0.0  0.902305  0.0   0.0 |   3.63    7.788    17.16    3.63    12.87 | 0.0       0.0  0.0  0.0       0.0 |   3.63     7.788    17.16    3.63     12.87 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |      3.63     2.9272  0.0    3.63  0.0 | 1.331831  1.016829  0.0    3.63  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.331831  1.016829  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.331831  1.016829  0.0  0.0  0.0 | 0.525928   0.45398   0.13973  0.0  0.0 | 0.460952 | 0.118527 | 0.057619 | 0.011391 | 0.445573 | 0.445573 | 0.445573 | 1.856555 |   2.0       2.0  0.0  0.0       2.0 |        0.0   32.957844     167.596  0.0     123.697 |        0.0   13.183138         0.0  0.0         0.0 | 123.441964  119.787029  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.918144   9.358677  2.725318  0.0  0.0 | 12.023584 | 10.275306 | 1.856555 |
    | 02/01 02:00 |  2.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   49.560342     179.608  0.0     132.706 | 0.0  0.0  0.902305  0.0   0.0 |  2.541   5.4516   12.012   2.541    9.009 | 0.0       0.0  0.0  0.0       0.0 |  2.541    5.4516   12.012   2.541     9.009 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     2.541    2.03224  0.0   2.541  0.0 | 0.967989  0.729012  0.0   2.541  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.967989  0.729012  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.967989  0.729012  0.0  0.0  0.0 | 0.556286  0.474353  0.132915  0.0  0.0 | 0.322667 | 0.121245 | 0.040333 | 0.011433 | 0.462338 | 0.462338 | 0.462338 |  1.92641 |   2.0       2.0  0.0  0.0       2.0 |        0.0   35.400244     179.608  0.0     132.706 |        0.0   14.160098         0.0  0.0         0.0 | 125.014975  121.090257  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 11.329847   9.613335  2.592403  0.0  0.0 | 12.225006 | 10.304206 |  1.92641 |
    | 02/01 03:00 | 10.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   99.386482       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 | 12.584  26.9984   59.488  12.584   44.616 | 0.0       0.0  0.0  0.0       0.0 | 12.584   26.9984   59.488  12.584    44.616 |        0.0         0.0      39.096  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0     23.6525         0.0  0.0      22.678 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.584    0.82476  0.0  12.584  0.0 | 4.916803  0.302333  0.0  12.584  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0  0.302333  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.27757       0.0  0.0  0.0  0.0 | 1.933508  0.302333  0.0  0.0  0.0 | 0.600105  0.476282  0.126433  0.0  0.0 | 1.619077 | 0.129709 | 0.202385 | 0.011555 |   0.5767 |   0.5767 |   0.5767 | 2.402919 |   2.0       2.0  0.0  0.0       2.0 |        0.0   70.990344       200.0  0.0       200.0 |        0.0   28.396138         0.0  0.0         0.0 | 132.682172  121.612683  0.0  0.0  0.0 | 2.639232       0.0  0.0  0.0  0.0 |  12.66325   9.439387   2.46597  0.0  0.0 | 13.714373 | 10.495035 | 2.402919 |
    | 02/01 04:00 |  3.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  127.241232       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.235    9.086    20.02   4.235   15.015 | 0.0       0.0  0.0  0.0       0.0 |  4.235     9.086    20.02   4.235    15.015 |        0.0         0.0       20.02  0.0      15.015 |       0.0        0.0        0.0  0.0        0.0 |        0.0    18.76875         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.235        0.0  0.0   4.235  0.0 | 1.863883       0.0  0.0   4.235  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.238203       0.0  0.0  0.0  0.0 | 1.866838       0.0  0.0  0.0  0.0 | 0.663497  0.460364  0.120267  0.0  0.0 | 0.580052 | 0.139351 | 0.072506 | 0.011695 |  0.58836 |  0.58836 |  0.58836 | 2.451499 |   2.0       2.0  0.0  0.0       2.0 |        0.0   93.803094       200.0  0.0       200.0 |        0.0   33.438138         0.0  0.0         0.0 |  135.05329  121.612683  0.0  0.0  0.0 | 2.264912       0.0  0.0  0.0  0.0 |  13.86659   8.979022  2.345703  0.0  0.0 | 14.155074 | 10.555847 | 2.451499 |
    | 02/01 05:00 |  3.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  154.300132       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.114   8.8264   19.448   4.114   14.586 | 0.0       0.0  0.0  0.0       0.0 |  4.114    8.8264   19.448   4.114    14.586 |        0.0         0.0      19.448  0.0      14.586 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.2325         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.114        0.0  0.0   4.114  0.0 | 1.875921       0.0  0.0   4.114  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.203727       0.0  0.0  0.0  0.0 |  1.80667       0.0  0.0  0.0  0.0 | 0.720705  0.437912  0.114401  0.0  0.0 | 0.583787 | 0.143755 | 0.072973 | 0.011763 |  0.59295 |  0.59295 |  0.59295 | 2.470626 |   2.0       2.0  0.0  0.0       2.0 |        0.0  115.965194       200.0  0.0       200.0 |        0.0   38.334938         0.0  0.0         0.0 | 137.291368  121.612683  0.0  0.0  0.0 | 1.937106       0.0  0.0  0.0  0.0 | 14.952556    8.54111  2.231302  0.0  0.0 | 14.595106 | 10.617058 | 2.470626 |
    | 02/01 06:00 |  1.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  163.850332       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.452   3.1152    6.864   1.452    5.148 | 0.0       0.0  0.0  0.0       0.0 |  1.452    3.1152    6.864   1.452     5.148 |        0.0         0.0       6.864  0.0       5.148 |       0.0        0.0        0.0  0.0        0.0 |        0.0       6.435         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     1.452        0.0  0.0   1.452  0.0 | 0.684216       0.0  0.0   1.452  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.059127       0.0  0.0  0.0  0.0 | 1.752372       0.0  0.0  0.0  0.0 | 0.772333  0.416555  0.108822  0.0  0.0 | 0.262993 | 0.146534 | 0.032874 | 0.011808 | 0.557785 | 0.557785 | 0.557785 | 2.324106 |   2.0       2.0  0.0  0.0       2.0 |        0.0  123.812994       200.0  0.0       200.0 |        0.0   40.037338         0.0  0.0         0.0 | 138.059152  121.612683  0.0  0.0  0.0 | 0.562195       0.0  0.0  0.0  0.0 | 15.932595   8.124555   2.12248  0.0  0.0 | 14.711565 | 10.638123 | 2.324106 |
    | 02/01 07:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 | 0.057657       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.619853       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.619853       0.0  0.0  0.0  0.0 | 0.792283  0.396239  0.103515  0.0  0.0 | 0.015365 | 0.146459 | 0.001921 | 0.011815 | 0.538183 | 0.538183 | 0.538183 |  2.24243 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.503644       200.0  0.0       200.0 |        0.0   40.142538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 15.760165   7.728316  2.018965  0.0  0.0 | 14.580471 | 10.628229 |  2.24243 |
    | 02/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.768632  0.376914  0.098466  0.0  0.0 |      0.0 | 0.145078 |      0.0 | 0.011803 | 0.522836 | 0.522836 | 0.522836 | 2.178485 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.543644       200.0  0.0       200.0 |        0.0   40.102538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.991533   7.351402  1.920499  0.0  0.0 | 14.435393 | 10.616427 | 2.178485 |
    | 02/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.731146  0.358532  0.093664  0.0  0.0 |      0.0 | 0.143635 |      0.0 | 0.011789 | 0.503119 | 0.503119 | 0.503119 | 2.096329 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.583644       200.0  0.0       200.0 |        0.0   40.062538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.260387    6.99287  1.826835  0.0  0.0 | 14.291758 | 10.604637 | 2.096329 |
    | 02/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  167.829582       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 |        0.0         0.0       2.288  0.0       1.716 |       0.0        0.0        0.0  0.0        0.0 |        0.0       2.145         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.484        0.0  0.0   0.484  0.0 | 0.230842       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.230842       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.230842       0.0  0.0  0.0  0.0 | 0.701163  0.341046  0.089096  0.0  0.0 |  0.06146 | 0.142512 | 0.007683 | 0.011781 | 0.486492 | 0.486492 | 0.486492 |  2.02705 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.226244       200.0  0.0       200.0 |        0.0   40.603338         0.0  0.0         0.0 | 138.375653  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.790065   6.651823  1.737739  0.0  0.0 | 14.210707 | 10.600539 |  2.02705 |
    | 02/01 11:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  168.625432       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 | 0.057922       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.057922       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.057922       0.0  0.0  0.0  0.0 | 0.673974  0.324413  0.084751  0.0  0.0 | 0.015365 | 0.141475 | 0.001921 | 0.011773 |  0.47115 |  0.47115 |  0.47115 | 1.963125 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.916894       200.0  0.0       200.0 |        0.0   40.708538         0.0  0.0         0.0 | 138.438731  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.174014    6.32741  1.652989  0.0  0.0 | 14.084597 | 10.590687 | 1.963125 |
    | 02/01 12:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  197.276032       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |        0.0         0.0      20.592  0.0      15.444 |       0.0        0.0        0.0  0.0        0.0 |        0.0      19.305         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.356        0.0  0.0   4.356  0.0 | 2.087099       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.008289       0.0  0.0  0.0  0.0 | 1.841299       0.0  0.0  0.0  0.0 | 0.687779  0.308591  0.080617  0.0  0.0 | 0.603524 | 0.143152 | 0.075441 | 0.011803 | 0.475061 | 0.475061 | 0.475061 | 1.979422 |   2.0       2.0  0.0  0.0       2.0 |        0.0  151.380294       200.0  0.0       200.0 |        0.0   45.895738         0.0  0.0         0.0 | 140.707632  121.612683  0.0  0.0  0.0 | 0.078811       0.0  0.0  0.0  0.0 | 14.327534   6.018819  1.572372  0.0  0.0 | 14.544969 | 10.654325 | 1.979422 |
    | 02/01 13:00 |  5.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  42.523945       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.139  15.3164   33.748   7.139   25.311 | 0.0       0.0  0.0  0.0       0.0 |  7.139   15.3164   33.748   7.139    25.311 |        0.0    9.366467      33.748  0.0      25.311 |       0.0   3.225965        0.0  0.0        0.0 |  32.804173         0.0         0.0  0.0         0.0 |  2.580772        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0   7.139  0.0 |      0.0       0.0  0.0   7.139  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.078811       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.078811       0.0  0.0  0.0  0.0 |   0.7007  0.293541  0.076685  0.0  0.0 |  0.90654 | 0.149242 | 0.113317 | 0.011894 | 0.477576 | 0.477576 | 0.477576 |   1.9899 |   2.0       2.0  0.0  0.0       2.0 |   30.45818  148.803428       200.0  0.0       200.0 |  12.065766   51.196572         0.0  0.0         0.0 | 140.707632  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.705645   5.725277  1.495686  0.0  0.0 | 15.302266 | 10.755748 |   1.9899 |
    | 02/01 14:00 |  1.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  48.329573       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.331   2.8556    6.292   1.331    4.719 | 0.0       0.0  0.0  0.0       0.0 |  1.331    2.8556    6.292   1.331     4.719 |        0.0    2.112422       6.292  0.0       4.719 |       0.0   0.743178        0.0  0.0        0.0 |   6.408937         0.0         0.0  0.0         0.0 |  0.594543        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |  2.528852        0.0  0.0   1.331  0.0 | 1.251696       0.0  0.0   1.331  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.251696       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.251696       0.0  0.0  0.0  0.0 |  0.69921  0.279225  0.072945  0.0  0.0 | 0.169016 | 0.153102 | 0.021127 | 0.011956 | 0.476173 | 0.476173 | 0.476173 | 1.984056 |   2.0       2.0  0.0  0.0       2.0 |  34.521124  147.989406       200.0  0.0       200.0 |  13.808449   52.010594         0.0  0.0         0.0 | 141.984788  121.612683  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.258131   5.446052  1.422741  0.0  0.0 |  15.31818 | 10.764919 | 1.984056 |
    | 02/01 15:00 | 20.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 201.988205  191.984574  201.988205  0.0  201.988205 | 0.0  0.0  0.902305  0.0   0.0 | 25.047  53.7372  118.404  25.047   88.803 | 0.0       0.0  0.0  0.0       0.0 | 25.047   53.7372  118.404  25.047    88.803 |        0.0   36.356814     118.404  0.0      88.803 |       0.0  17.380386        0.0  0.0        0.0 | 115.042936    1.778447    1.778447  0.0    1.778447 | 13.568696   0.209758   0.209758  0.0   0.209758 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.209758  0.0  0.209758 |       0.0  10.003631  0.0  25.047  0.0 |      0.0  3.698754  0.0  25.047  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.161658  0.0  0.0  0.0 |      0.0       2.0  0.0  0.0  0.0 | 0.695377  0.314784  0.069388  0.0  0.0 | 3.180571 | 0.168268 | 0.397571 | 0.012175 | 0.541135 | 0.541135 | 0.541135 |  2.25473 |   2.0       2.0  0.0  0.0       2.0 | 147.218067  137.131839  201.988205  0.0  201.988205 |  54.770138   54.852736         0.0  0.0         0.0 | 141.984788  127.917561  0.0  0.0  0.0 |      0.0  1.537096  0.0  0.0  0.0 | 13.562754   7.131268  1.353353  0.0  0.0 | 18.330483 | 11.150315 |  2.25473 |
    | 02/01 16:00 | 37.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 290.938862  302.004664  309.356596  0.0  309.356596 | 0.0  0.0  0.902305  0.0   0.0 | 45.859  98.3884  216.788  45.859  162.591 | 0.0       0.0  0.0  0.0       0.0 | 45.859   98.3884  216.788  45.859   162.591 |  28.420627   56.173848  218.776205  0.0  164.579205 | 19.426578  34.199126        0.0  0.0        0.0 |  91.362027   91.362027   91.362027  0.0   91.362027 | 17.994569  17.994569  17.994569  0.0  17.994569 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 | 18.417734   7.351931  0.0  45.859  0.0 |  9.28239  3.007473  0.0  45.859  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.693011  0.242148  0.0  0.0  0.0 | 1.821862       2.0  0.0  0.0  0.0 |  0.70626  0.396973  0.066004  0.0  0.0 | 5.879917 | 0.211693 |  0.73499 | 0.012791 | 0.859575 | 0.859575 | 0.859575 | 3.581562 |   2.0       2.0  0.0  0.0       2.0 | 207.813473  215.717617  292.002027  0.0  291.702027 |  83.125389   86.287047   17.354569  0.0   17.654569 | 151.120131  132.262019  0.0  0.0  0.0 | 6.589379  2.302421  0.0  0.0  0.0 | 14.678356   8.734295  1.287349  0.0  0.0 | 23.998706 | 11.872514 | 3.581562 |
    | 02/01 17:00 |  8.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 323.906781  323.906781  323.906781  0.0  323.906781 | 0.0  0.0  0.902305  0.0   0.0 |  9.922  21.2872   46.904   9.922   35.178 | 0.0       0.0  0.0  0.0       0.0 |  9.922   21.2872   46.904   9.922    35.178 |  69.667573   85.844423  148.648653  0.0  137.128384 | 31.193289  37.447441   7.611943  0.0   7.406211 |  98.060761   98.060761   98.060761  0.0   98.060761 | 25.846019  25.846019  25.846019  0.0  25.846019 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.922  0.0 |      0.0       0.0  0.0   9.922  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       2.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.436737  0.028779  0.0  0.0  0.0 | 1.766082       2.0  0.0  0.0  0.0 | 0.759297  0.475154  0.062785  0.0  0.0 | 1.334196 |  0.24544 | 0.166775 | 0.013277 | 0.787086 | 0.787086 | 0.787086 | 3.279526 |   2.0       2.0  0.0  0.0       2.0 | 233.860668  237.354756  288.958135  0.0  288.152404 |  90.046113   86.552025   34.948645  0.0   35.754377 | 151.120131  132.262019  0.0  0.0  0.0 | 4.152642  0.273642  0.0  0.0  0.0 | 15.685141  10.259142  1.224564  0.0  0.0 | 25.087463 | 12.026011 | 3.279526 |
    | 02/01 18:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 334.397735  334.397735  334.397735  0.0  334.397735 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |  91.376852   96.554068  129.839697  0.0  124.668627 | 36.885929  36.698313  14.659084  0.0  14.682154 | 104.406788  104.406788  104.406788  0.0  104.406788 | 29.990946  29.990946  29.990946  0.0  29.990946 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   4.356  0.0 |      0.0       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0  0.273642  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.204851       0.0  0.0  0.0  0.0 | 1.715743  0.273642  0.0  0.0  0.0 | 0.807161  0.507073  0.059723  0.0  0.0 | 0.643383 | 0.252831 | 0.080423 | 0.013399 | 0.732983 | 0.732983 | 0.732983 | 3.054094 |   2.0       2.0  0.0  0.0       2.0 | 244.544611  249.365877  284.757227  0.0  283.674565 |  89.853124   85.031858   49.640507  0.0   50.723169 | 151.120131  132.262019  0.0  0.0  0.0 | 1.947791       0.0  0.0  0.0  0.0 | 16.593723  10.025711  1.164842  0.0  0.0 | 25.478015 | 12.093035 | 3.054094 |
    | 02/01 19:00 |  7.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 356.253888  356.253888  356.253888  0.0  356.253888 | 0.0  0.0  0.902305  0.0   0.0 |  9.075    19.47     42.9   9.075   32.175 | 0.0       0.0  0.0  0.0       0.0 |  9.075     19.47     42.9   9.075    32.175 | 102.149255  112.159273  153.970933  0.0  143.523838 | 41.323479  41.708462  23.326802  0.0  23.048897 | 120.370769  120.370769  120.370769  0.0  120.370769 | 35.883119  35.883119  35.883119  0.0  35.883119 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.075  0.0 |      0.0       0.0  0.0   9.075  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.947791       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.670314       0.0  0.0  0.0  0.0 | 0.850356   0.48896   0.05681  0.0  0.0 | 1.240469 | 0.259692 | 0.155059 | 0.013515 | 0.680197 | 0.680197 | 0.680197 | 2.834154 |   2.0       2.0  0.0  0.0       2.0 | 260.420132  266.197374  294.697064  0.0  293.036497 |  95.833757   90.056515   61.556824  0.0   63.217392 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 17.413681   9.536751  1.108032  0.0  0.0 | 26.458792 | 12.234578 | 2.834154 |
    | 02/01 20:00 | 18.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 410.165735  410.165735  410.165735  0.0  410.165735 | 0.0  0.0  0.902305  0.0   0.0 | 22.385   48.026   105.82  22.385   79.365 | 0.0       0.0  0.0  0.0       0.0 | 22.385    48.026   105.82  22.385    79.365 | 122.864197  145.201755  227.160779  0.0   201.42567 | 55.774692  59.078133   34.91311  0.0  34.193218 | 159.848924  159.848924  159.848924  0.0  159.848924 | 50.316811  50.316811  50.316811  0.0  50.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  22.385  0.0 |      0.0       0.0  0.0  22.385  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.849275  0.465113  0.054039  0.0  0.0 |  2.84254 | 0.277435 | 0.355317 | 0.013784 | 0.689734 | 0.689734 | 0.689734 |  2.87389 |   2.0       2.0  0.0  0.0       2.0 | 295.058865  302.048542  333.845209  0.0   331.16475 | 115.106869  108.117192   76.320526  0.0   79.000985 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 16.564406   9.071639  1.053992  0.0  0.0 | 29.023897 | 12.576112 |  2.87389 |
    | 02/01 21:00 | 15.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 455.043704  455.043704  455.043704  0.0  455.043704 | 0.0  0.0  0.902305  0.0   0.0 | 18.634  39.9784   88.088  18.634   66.066 | 0.0       0.0  0.0  0.0       0.0 | 18.634   39.9784   88.088  18.634    66.066 | 157.438041  177.637793  252.568414  0.0  230.408289 | 71.361694  72.506342  45.685321  0.0  45.823446 | 191.219608  191.219608  191.219608  0.0  191.219608 | 63.824096  63.824096  63.824096  0.0  63.824096 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  18.634  0.0 |      0.0       0.0  0.0  18.634  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.807856  0.442429  0.051404  0.0  0.0 | 2.366222 | 0.300584 | 0.295778 |  0.01413 |  0.69128 |  0.69128 |  0.69128 | 2.880335 |   2.0       2.0  0.0  0.0       2.0 | 326.494439  333.287958  361.224403  0.0  358.382069 | 128.549264  121.755746     93.8193  0.0   96.661635 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  15.75655    8.62921  1.002588  0.0  0.0 | 31.089535 |  12.85776 | 2.880335 |
    | 02/01 22:00 |  6.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 473.402873  473.402873  473.402873  0.0  473.402873 | 0.0  0.0  0.902305  0.0   0.0 |  7.623  16.3548   36.036   7.623   27.027 | 0.0       0.0  0.0  0.0       0.0 |  7.623   16.3548   36.036   7.623    27.027 | 185.358526  196.033453  235.469802  0.0  225.511749 | 77.308178  75.365051  55.609902  0.0  56.558955 | 203.295969  203.295969  203.295969  0.0  203.295969 | 70.106904  70.106904  70.106904  0.0  70.106904 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   7.623  0.0 |      0.0       0.0  0.0   7.623  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.768456  0.420852  0.048897  0.0  0.0 |    0.968 |  0.31417 |    0.121 | 0.014346 | 0.684773 | 0.684773 | 0.684773 | 2.853221 |   2.0       2.0  0.0  0.0       2.0 | 342.085889  347.797674   365.72657  0.0  363.533289 | 131.316984  125.605199  107.676303  0.0  109.869584 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.988094   8.208358  0.953692  0.0  0.0 | 31.743365 | 12.964414 | 2.853221 |
    | 02/01 23:00 |  1.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 478.939765  478.939765  478.939765  0.0  478.939765 | 0.0  0.0  0.902305  0.0   0.0 |  2.299   4.9324   10.868   2.299    8.151 | 0.0       0.0  0.0  0.0       0.0 |  2.299    4.9324   10.868   2.299     8.151 | 198.262243  203.642415  221.064023  0.0  217.315563 |  77.43963  74.692858   63.20685  0.0   64.23831 |  206.64527   206.64527   206.64527  0.0   206.64527 | 72.294496  72.294496  72.294496  0.0  72.294496 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.299  0.0 |      0.0       0.0  0.0   2.299  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.730978  0.400326  0.046512  0.0  0.0 | 0.291937 | 0.317307 | 0.036492 | 0.014417 | 0.669324 | 0.669324 | 0.669324 | 2.788851 |   2.0       2.0  0.0  0.0       2.0 | 348.122922  353.014129  362.815817  0.0  361.353996 | 130.816843  125.925637  116.123948  0.0   117.58577 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.257116   7.808032   0.90718  0.0  0.0 | 31.717995 | 12.986489 | 2.788851 |
    | 03/01 00:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 493.219119  493.219119  493.219119  0.0  493.219119 | 0.0  0.0  0.902686  0.0   0.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 | 204.528223  212.738588  236.654993  0.0  229.412995 | 80.340542  78.921577  70.312772  0.0  70.547771 | 216.382573  216.382573  216.382573  0.0  216.382573 | 76.836547  76.836547  76.836547  0.0  76.836547 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   5.929  0.0 |      0.0       0.0  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.695328  0.380802  0.044244  0.0  0.0 | 0.752889 | 0.319351 | 0.094111 | 0.014474 | 0.653742 | 0.653742 | 0.653742 | 2.723924 |   2.0       2.0  0.0  0.0       2.0 | 357.630288  362.303713  371.211396  0.0  369.684574 | 135.588831  130.915406  122.007723  0.0  123.534546 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.561788    7.42723  0.862936  0.0  0.0 | 32.151532 | 13.066126 | 2.723924 |
    | 03/01 01:00 |  2.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 501.087335  501.087335  501.087335  0.0  501.087335 | 0.0  0.0  0.902686  0.0   0.0 |  3.267   7.0092   15.444   3.267   11.583 | 0.0       0.0  0.0  0.0       0.0 |  3.267    7.0092   15.444   3.267    11.583 | 213.565721  219.302218  234.627313  0.0  230.211324 | 82.920398  80.926101  74.035807  0.0  74.590795 | 221.471596  221.471596  221.471596  0.0  221.471596 | 79.615739  79.615739  79.615739  0.0  79.615739 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   3.267  0.0 |      0.0       0.0  0.0   3.267  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.661416   0.36223  0.042086  0.0  0.0 | 0.414857 |  0.32198 | 0.051857 | 0.014539 | 0.639569 | 0.639569 | 0.639569 | 2.664869 |   2.0       2.0  0.0  0.0       2.0 | 363.189179   367.60189  374.139679  0.0  372.867845 | 137.898156  133.485444  126.947655  0.0   128.21949 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.900372   7.064999   0.82085  0.0  0.0 | 32.244409 | 13.103445 | 2.664869 |
    | 03/01 02:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 502.544412  502.544412  502.544412  0.0  502.544412 | 0.0  0.0  0.902686  0.0   0.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 218.403559  221.603573  227.380998  0.0  225.971212 | 83.288776  80.781762  76.566337  0.0  77.261123 | 222.042022  222.042022  222.042022  0.0  222.042022 | 80.502389  80.502389  80.502389  0.0  80.502389 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.629159  0.344564  0.040033  0.0  0.0 | 0.076825 |  0.32122 | 0.009603 | 0.014557 | 0.623002 | 0.623002 | 0.623002 |  2.59584 |   2.0       2.0  0.0  0.0       2.0 | 364.480659   368.65234  372.300704  0.0  371.423655 | 138.063753  133.892072  130.243708  0.0  131.120756 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.271214   6.720435  0.780817  0.0  0.0 | 32.000014 | 13.098491 |  2.59584 |
    | 03/01 03:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 503.127242  503.127242  503.127242  0.0  503.127242 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 219.496367  222.227265  225.160687  0.0  224.375469 | 83.290045  80.836347  78.527724  0.0  79.026943 |  222.01911   222.01911   222.01911  0.0   222.01911 | 81.108133  81.108133  81.108133  0.0  81.108133 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.598474  0.327759  0.038081  0.0  0.0 |  0.03073 | 0.318559 | 0.003841 | 0.014548 |  0.60541 |  0.60541 |  0.60541 |  2.52254 |   2.0       2.0  0.0  0.0       2.0 | 364.656418  368.712985  370.943126  0.0  370.265296 | 138.470825  134.414258  132.184116  0.0  132.861946 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  11.67274   6.392676  0.742736  0.0  0.0 | 31.712186 | 13.087785 |  2.52254 |
    | 03/01 04:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 504.584319  504.584319  504.584319  0.0  504.584319 | 0.0  0.0  0.902686  0.0   0.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 219.874572  222.866862   226.05113  0.0  225.000538 |  83.85767   81.55838  79.936113  0.0  80.271705 | 222.614941  222.614941  222.614941  0.0  222.614941 | 81.969378  81.969378  81.969378  0.0  81.969378 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.569286  0.311774  0.036224  0.0  0.0 | 0.076825 | 0.315924 | 0.009603 | 0.014539 | 0.588579 | 0.588579 | 0.588579 | 2.452414 |   2.0       2.0  0.0  0.0       2.0 | 365.049803  369.073064  371.006938  0.0    370.3647 | 139.534516  135.511255  133.577381  0.0  134.219619 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 11.103453   6.080901  0.706512  0.0  0.0 | 31.473087 | 13.082849 | 2.452414 |
    | 03/01 05:00 |  2.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 511.578288  511.578288  511.578288  0.0  511.578288 | 0.0  0.0  0.902686  0.0   0.0 |  2.904   6.2304   13.728   2.904   10.296 | 0.0       0.0  0.0  0.0       0.0 |  2.904    6.2304   13.728   2.904    10.296 |   221.1845  226.239983  236.278139  0.0   232.79694 | 86.303819  84.574736   82.03418  0.0  82.083379 | 227.116446  227.116446  227.116446  0.0  227.116446 | 84.461843  84.461843  84.461843  0.0  84.461843 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.904  0.0 |      0.0       0.0  0.0   2.904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.541522  0.296569  0.034457  0.0  0.0 | 0.368762 |    0.315 | 0.046095 | 0.014554 | 0.574068 | 0.574068 | 0.574068 | 2.391948 |   2.0       2.0  0.0  0.0       2.0 | 368.634765  372.735126  376.213244  0.0  375.320206 | 142.943524  138.843162  135.365044  0.0  136.258083 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.561931   5.784332  0.672055  0.0  0.0 | 31.526849 |  13.11439 | 2.391948 |
    | 03/01 06:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  512.74395   512.74395   512.74395  0.0   512.74395 | 0.0  0.0  0.902686  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 224.654326  227.589692  231.186173  0.0  230.127809 | 87.407963  85.026997  82.680115  0.0   83.16648 | 227.485902  227.485902  227.485902  0.0  227.485902 | 85.258048  85.258048  85.258048  0.0  85.258048 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.484  0.0 |      0.0       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.515111  0.282105  0.032777  0.0  0.0 |  0.06146 | 0.314004 | 0.007683 | 0.014568 | 0.560154 | 0.560154 | 0.560154 | 2.333974 |   2.0       2.0  0.0  0.0       2.0 | 369.119357  373.128936  375.440973  0.0  374.734299 | 143.624593  139.615014  137.302977  0.0  138.009651 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  10.04682   5.502227  0.639279  0.0  0.0 | 31.274305 | 13.107505 | 2.333974 |
    | 03/01 07:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 225.209233  227.873792  230.021905  0.0  229.334171 | 87.776717  85.389358  83.866045  0.0  84.267779 | 227.457017  227.457017  227.457017  0.0  227.457017 | 85.869764  85.869764  85.869764  0.0  85.869764 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.489989  0.268347  0.031178  0.0  0.0 |  0.03073 | 0.311338 | 0.003841 | 0.014558 | 0.545294 | 0.545294 | 0.545294 | 2.272057 |   2.0       2.0  0.0  0.0       2.0 | 369.020157  372.980961  374.660085  0.0  374.055145 | 144.306624   140.34582  138.666696  0.0  139.271636 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.556831    5.23388  0.608101  0.0  0.0 | 30.993698 | 13.096788 | 2.272057 |
    | 03/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.244235  227.661848   228.68676  0.0  228.317514 | 88.082546  85.664932  84.640021  0.0  85.009267 | 227.018951  227.018951  227.018951  0.0  227.018951 |  86.30783   86.30783   86.30783  0.0   86.30783 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.466092  0.255259  0.029657  0.0  0.0 |      0.0 | 0.308392 |      0.0 | 0.014544 | 0.530772 | 0.530772 | 0.530772 |  2.21155 |   2.0       2.0  0.0  0.0       2.0 |  368.44789  372.378063  373.632276  0.0  373.096582 | 144.878891  140.948718  139.694505  0.0  140.230199 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.090739   4.978621  0.578443  0.0  0.0 | 30.685305 | 13.082244 |  2.21155 |
    | 03/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.894931  227.293849  228.059401  0.0  227.732422 | 88.431849  86.032932   85.26738  0.0  85.594359 | 226.581591  226.581591  226.581591  0.0  226.581591 |  86.74519   86.74519   86.74519  0.0   86.74519 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.443361   0.24281  0.028211  0.0  0.0 |      0.0 | 0.305324 |      0.0 | 0.014528 | 0.516707 | 0.516707 | 0.516707 | 2.152944 |   2.0       2.0  0.0  0.0       2.0 | 367.787565  371.705805  372.794465  0.0  372.285751 | 145.539216  141.620976  140.532315  0.0   141.04103 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.647378   4.735811  0.550232  0.0  0.0 | 30.379981 | 13.067716 | 2.152944 |
    | 03/01 10:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.201027  514.201027  514.201027  0.0  514.201027 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 |  224.59314  227.312585  229.081589  0.0  228.387074 | 89.096641  86.792995  85.961192  0.0  86.226707 | 226.757529  226.757529  226.757529  0.0  226.757529 | 87.443498  87.443498  87.443498  0.0  87.443498 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.363  0.0 |      0.0       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.421738  0.230968  0.026835  0.0  0.0 | 0.046095 | 0.302516 | 0.005762 | 0.014515 | 0.503433 | 0.503433 | 0.503433 | 2.097637 |   2.0       2.0  0.0  0.0       2.0 | 367.604971  371.533949  372.826406  0.0  372.283206 | 146.596056  142.667078  141.374621  0.0  141.917821 | 151.120131  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.225641   4.504842  0.523397  0.0  0.0 | 30.123561 | 13.058963 | 2.097637 |
    | 03/01 11:00 |  2.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 519.961252  521.777827  521.777827  0.0  521.777827 | 0.0  0.0  0.902686  0.0   0.0 |  3.146   6.7496   14.872   3.146   11.154 | 0.0       0.0  0.0  0.0       0.0 |  3.146    6.7496   14.872   3.146    11.154 | 225.493409  230.729522  241.140791  0.0  237.464602 | 91.853618  90.221105  87.932236  0.0  87.890425 | 231.636316  231.636316  231.636316  0.0  231.636316 | 90.141511  90.141511  90.141511  0.0  90.141511 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.816574        0.0  0.0   3.146  0.0 | 1.037141       0.0  0.0   3.146  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.037141       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.037141       0.0  0.0  0.0  0.0 | 0.426671  0.219704  0.025526  0.0  0.0 | 0.399492 | 0.301725 | 0.049937 |  0.01453 | 0.501088 | 0.501088 | 0.501088 | 2.087866 |   2.0       2.0  0.0  0.0       2.0 | 371.400895  375.455142  378.833931  0.0   377.94892 | 148.560358  146.322685  142.943896  0.0  143.828907 | 151.899565  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  8.836111   4.285139  0.497871  0.0  0.0 | 30.221328 |  13.09437 | 2.087866 |
    | 03/01 12:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.773236  523.119052  523.119052  0.0  523.119052 | 0.0  0.0  0.902686  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 228.776083   232.53573  237.211776  0.0  235.766768 |  92.03217  91.059297  88.570051  0.0  89.014059 | 232.417341  232.417341  232.417341  0.0  232.417341 | 90.701711  90.701711  90.701711  0.0  90.701711 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.345816        0.0  0.0   0.847  0.0 | 0.776316       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.776316       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.776316       0.0  0.0  0.0  0.0 | 0.450031  0.208989  0.024281  0.0  0.0 | 0.107556 | 0.301243 | 0.013444 | 0.014549 | 0.505336 | 0.505336 | 0.505336 | 2.105566 |   2.0       2.0  0.0  0.0       2.0 | 372.695169  376.177553  378.683497  0.0  377.942492 | 149.078068  146.941499  144.435556  0.0   145.17656 | 152.469064  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.162396    4.07615  0.473589  0.0  0.0 |  30.02764 | 13.093266 | 2.105566 |
    | 03/01 13:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.992672  523.475677  523.475677  0.0  523.475677 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 229.937343  232.782522  235.439858  0.0  234.597488 | 92.198893   91.11533  89.395194  0.0  89.808564 | 232.441067  232.441067  232.441067  0.0  232.441067 |  91.03461   91.03461   91.03461  0.0   91.03461 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.483005        0.0  0.0   0.363  0.0 | 0.861878       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.861878       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 0.861878       0.0  0.0  0.0  0.0 | 0.468048  0.198796  0.023097  0.0  0.0 | 0.046095 |  0.29901 | 0.005762 | 0.014543 | 0.506297 | 0.506297 | 0.506297 | 2.109571 |   2.0       2.0  0.0  0.0       2.0 | 372.851909  376.219298  378.040705  0.0  377.413071 | 149.140763  147.256379  145.434972  0.0  146.062606 | 153.090191  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  9.556227   3.877354  0.450492  0.0  0.0 | 29.774726 | 13.084485 | 2.109571 |
    | 03/01 14:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 522.008863  523.779537  523.779537  0.0  523.779537 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 230.094041  232.906119  235.140283  0.0  234.368133 | 92.261631  91.348357  90.051394  0.0  90.394544 |  232.45259   232.45259   232.45259  0.0   232.45259 | 91.326947  91.326947  91.326947  0.0  91.326947 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.770674        0.0  0.0   0.363  0.0 | 1.037465       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.037465       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.037465       0.0  0.0  0.0  0.0 | 0.491572  0.189101  0.021971  0.0  0.0 | 0.046095 | 0.296493 | 0.005762 | 0.014533 |  0.50897 |  0.50897 |  0.50897 | 2.120708 |   2.0       2.0  0.0  0.0       2.0 | 372.863473  376.148968  377.709011  0.0  377.124528 | 149.145389  147.630568  146.070525  0.0  146.655009 |   153.8234  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.102119   3.688253  0.428521  0.0  0.0 | 29.524328 | 13.075713 | 2.120708 |
    | 03/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.265413  523.098508  523.098508  0.0  523.098508 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 230.006331  232.520231  233.484587  0.0  233.123282 | 92.002532  91.259306   90.29495  0.0  90.656254 | 231.822278  231.822278  231.822278  0.0  231.822278 |  91.27623   91.27623   91.27623  0.0   91.27623 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.833096        0.0  0.0     0.0  0.0 | 1.084351       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.084351       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.084351       0.0  0.0  0.0  0.0 | 0.519349  0.179878  0.020899  0.0  0.0 |      0.0 | 0.293772 |      0.0 | 0.014521 | 0.513003 | 0.513003 | 0.513003 | 2.137514 |   2.0       2.0  0.0  0.0       2.0 | 372.332438  375.491016  376.686703  0.0  376.163524 | 148.932975  147.607492  146.411805  0.0  146.934984 | 154.572144  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 10.667122   3.508375  0.407622  0.0  0.0 | 29.230556 | 13.061193 | 2.137514 |
    | 03/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 520.375077  522.393471  522.393471  0.0  522.393471 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 229.475295  231.926846  232.665377  0.0  232.342229 | 91.790118  91.171662  90.433131  0.0  90.756279 | 231.186324  231.186324  231.186324  0.0  231.186324 | 91.207147  91.207147  91.207147  0.0  91.207147 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.018394        0.0  0.0     0.0  0.0 | 1.205615       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.205615       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.205615       0.0  0.0  0.0  0.0 | 0.549886  0.171105   0.01988  0.0  0.0 |      0.0 | 0.290849 |      0.0 | 0.014504 | 0.517896 | 0.517896 | 0.517896 | 2.157899 |   2.0       2.0  0.0  0.0       2.0 | 371.696484  374.790495   375.84765  0.0  375.347619 | 148.678593  147.602977  146.545821  0.0  147.045852 | 155.384924  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  11.32285   3.337269  0.387742  0.0  0.0 | 28.939707 | 13.046688 | 2.157899 |
    | 03/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  519.45772  521.617166  521.617166  0.0  521.617166 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.839341  231.300763  231.953183  0.0  231.644591 | 91.535736  91.092708  90.440288  0.0  90.748881 | 230.531069  230.531069  230.531069  0.0  230.531069 | 91.086097  91.086097  91.086097  0.0  91.086097 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.159446        0.0  0.0     0.0  0.0 | 1.303467       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.303467       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.303467       0.0  0.0  0.0  0.0 | 0.584272  0.162761   0.01891  0.0  0.0 |      0.0 | 0.287955 |      0.0 | 0.014488 | 0.524223 | 0.524223 | 0.524223 | 2.184261 |   2.0       2.0  0.0  0.0       2.0 | 371.041228    374.0608  375.065536  0.0  374.574097 | 148.416491  147.556365   146.55163  0.0  147.043069 | 156.240903  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.042046   3.174509  0.368832  0.0  0.0 | 28.651752 |   13.0322 | 2.184261 |
    | 03/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.519709   520.78661   520.78661  0.0   520.78661 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.184085    230.6373  231.256796  0.0  230.953787 | 91.273634  90.979866   90.36037  0.0  90.663379 | 229.861062  229.861062  229.861062  0.0  229.861062 | 90.925548  90.925548  90.925548  0.0  90.925548 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |    2.2669        0.0  0.0     0.0  0.0 | 1.383445       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.383445       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.383445       0.0  0.0  0.0  0.0 | 0.621314  0.154823  0.017988  0.0  0.0 |      0.0 |  0.28509 |      0.0 | 0.014472 |  0.53158 |  0.53158 |  0.53158 | 2.214915 |   2.0       2.0  0.0  0.0       2.0 | 370.371221  373.324563  374.309801  0.0  373.821372 | 148.148488  147.462047  146.476809  0.0  146.965237 | 157.124358  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 12.804176   3.019686  0.350844  0.0  0.0 | 28.366662 | 13.017728 | 2.214915 |
    | 03/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 517.565962  519.914725  519.914725  0.0  519.914725 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 227.514078  229.955069  230.561942  0.0  230.261087 | 91.005631  90.831541  90.224668  0.0  90.525523 | 229.179814  229.179814  229.179814  0.0  229.179814 | 90.734911  90.734911  90.734911  0.0  90.734911 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.348763        0.0  0.0     0.0  0.0 |  1.44966       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.44966       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  1.44966       0.0  0.0  0.0  0.0 | 0.660112  0.147272  0.017111  0.0  0.0 |      0.0 | 0.282253 |      0.0 | 0.014456 |  0.53966 |  0.53966 |  0.53966 | 2.248585 |   2.0       2.0  0.0  0.0       2.0 | 369.689973  372.589307  373.567673  0.0  373.080099 | 147.875989  147.325418  146.347052  0.0  146.834625 | 158.023461  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 13.593724   2.872414  0.333733  0.0  0.0 | 28.084409 | 13.003272 | 2.248585 |
    | 03/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 516.600219  519.011355  519.011355  0.0  519.011355 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  226.83283  229.262223  229.864233  0.0  229.564218 | 90.733132  90.652502  90.050492  0.0  90.350507 | 228.489997  228.489997  228.489997  0.0  228.489997 | 90.521358  90.521358  90.521358  0.0  90.521358 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.411136        0.0  0.0     0.0  0.0 | 1.505237       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.505237       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.505237       0.0  0.0  0.0  0.0 | 0.699985  0.140089  0.016276  0.0  0.0 |      0.0 | 0.279445 |      0.0 |  0.01444 | 0.548233 | 0.548233 | 0.548233 | 2.284305 |   2.0       2.0  0.0  0.0       2.0 | 369.000156  371.857081  372.833437  0.0  372.345878 | 147.600062  147.154274  146.177917  0.0  146.665476 |  158.92936  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 14.398976   2.732325  0.317456  0.0  0.0 | 27.804965 | 12.988832 | 2.284305 |
    | 03/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 515.625325  518.083995  518.083995  0.0  518.083995 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 226.143013   228.56269  229.162809  0.0   228.86313 | 90.457205  90.448665  89.848546  0.0  90.148225 | 227.793645  227.793645  227.793645  0.0  227.793645 |  90.29035   90.29035   90.29035  0.0   90.29035 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.458669        0.0  0.0     0.0  0.0 |  1.55256       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.55256       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  1.55256       0.0  0.0  0.0  0.0 | 0.740421  0.133257  0.015483  0.0  0.0 |      0.0 | 0.276664 |      0.0 | 0.014424 | 0.557121 | 0.557121 | 0.557121 | 2.321338 |   2.0       2.0  0.0  0.0       2.0 | 368.303804  371.128036  372.104273  0.0  371.616393 | 147.321522  146.955959  145.979722  0.0  146.467601 | 159.835469  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 15.211114   2.599068  0.301974  0.0  0.0 | 27.528301 | 12.974408 | 2.321338 |
    | 03/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.643451  517.138353  517.138353  0.0  517.138353 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.446661  227.858589  228.457962  0.0  228.158422 | 90.178664  90.225405  89.626033  0.0  89.925572 | 227.092305  227.092305  227.092305  0.0  227.092305 | 90.046047  90.046047  90.046047  0.0  90.046047 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.494902        0.0  0.0     0.0  0.0 | 1.593455       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.593455       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.593455       0.0  0.0  0.0  0.0 | 0.781035  0.126758  0.014727  0.0  0.0 |      0.0 | 0.273911 |      0.0 | 0.014408 | 0.566191 | 0.566191 | 0.566191 | 2.359128 |   2.0       2.0  0.0  0.0       2.0 | 367.602465  370.401752  371.378616  0.0  370.890276 | 147.040986  146.736601  145.759736  0.0  146.248076 | 160.736916  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 16.023534    2.47231  0.287246  0.0  0.0 |  27.25439 |     12.96 | 2.359128 |
    | 03/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.656246  516.178775  516.178775  0.0  516.178775 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.745322  227.151208  227.750276  0.0  227.450799 | 89.898129  89.987145  89.388076  0.0  89.687554 | 226.387159  226.387159  226.387159  0.0  226.387159 | 89.791616  89.791616  89.791616  0.0  89.791616 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.522529        0.0  0.0     0.0  0.0 | 1.629324       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.629324       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.629324       0.0  0.0  0.0  0.0 |  0.82154  0.120576  0.014009  0.0  0.0 |      0.0 | 0.271186 |      0.0 | 0.014392 | 0.575341 | 0.575341 | 0.575341 | 2.397255 |   2.0       2.0  0.0  0.0       2.0 | 366.897318  369.677703  370.655499  0.0  370.166637 | 146.758927  146.501072  145.523276  0.0  146.012138 | 161.630121  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 16.831319   2.351734  0.273237  0.0  0.0 | 26.983204 | 12.945608 | 2.397255 |
    | 04/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.663479  515.208571  515.208571  0.0  515.208571 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0  0.000576  0.0  0.0  0.001152 |    0.0       0.0      0.0     0.0       0.0 | 224.040175  226.441398  227.040334  0.0  226.740888 |  89.61607  89.737377   89.13844  0.0  89.437887 | 225.679105  225.679105  225.679105  0.0  225.679105 | 89.529467  89.529467  89.529467  0.0  89.529467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.545093        0.0  0.0     0.0  0.0 | 1.662219       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.662219       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.658434       0.0  0.0  0.0  0.0 | 0.861651  0.114695  0.013326  0.0  0.0 | 0.001202 | 0.268493 |  0.00015 | 0.014376 | 0.584479 | 0.584479 | 0.584479 |  2.43533 |   2.0  1.999424  0.0  0.0  1.998848 | 366.188199   368.95541   369.93427  0.0  369.444854 |  146.47528  146.253161  145.274302  0.0  145.763718 | 162.512995  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 17.628101   2.237039  0.259911  0.0  0.0 | 26.715912 | 12.931382 |  2.43533 |
    | 04/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.668518   514.22969   514.22969  0.0   514.22969 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.331056  225.729761  226.328635  0.0  226.029206 | 89.332422   89.47881  88.879937  0.0  89.179365 | 224.968418  224.968418  224.968418  0.0  224.968418 | 89.261271  89.261271  89.261271  0.0  89.261271 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.561172        0.0  0.0     0.0  0.0 | 1.691044       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.691044       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.618595       0.0  0.0  0.0  0.0 | 0.899531  0.109102  0.012676  0.0  0.0 |    0.023 | 0.265942 | 0.002875 | 0.014362 |  0.59309 |  0.59309 |  0.59309 | 2.471209 |   2.0  1.999424  0.0  0.0  1.998848 | 365.477513  368.234067  369.214053  0.0  368.724065 | 146.191005  145.995622  145.015637  0.0  145.505624 | 163.383123  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 18.347165   2.127937  0.247235  0.0  0.0 | 26.472969 | 12.919895 | 2.471209 |
    | 04/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.671177  513.244623  513.244623  0.0  513.244623 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  222.62037  225.016328  225.615167  0.0  225.315751 | 89.048148  89.213361  88.614523  0.0  88.913939 | 224.256032  224.256032  224.256032  0.0  224.256032 | 88.988592  88.988592  88.988592  0.0  88.988592 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.573447        0.0  0.0     0.0  0.0 | 1.717393       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.717393       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.582642       0.0  0.0  0.0  0.0 | 0.933717  0.103781  0.012058  0.0  0.0 | 0.042778 | 0.263624 | 0.005347 |  0.01435 | 0.600768 | 0.600768 | 0.600768 | 2.503199 |   2.0  1.999424  0.0  0.0  1.998848 | 364.765126  367.513771  368.494918  0.0  368.004346 | 145.906051  145.730853  144.749706  0.0  145.240277 | 164.239177  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  18.99609   2.024156  0.235177  0.0  0.0 | 26.252124 | 12.910892 | 2.503199 |
    | 04/01 03:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.766676   516.04297   516.04297  0.0   516.04297 | 0.0  0.0  0.903096  0.0   0.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |  1.573  3.374224    7.436   1.573  5.575848 | 222.346669  226.148508  231.531107  0.0   229.56884 | 89.897507  90.470339  89.149517  0.0  89.251632 | 226.191503  226.191503  226.191503  0.0  226.191503 | 89.851467  89.851467  89.851467  0.0  89.851467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  3.276294        0.0  0.0   1.573  0.0 |  2.20941       0.0  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019928       0.0  0.0  0.0  0.0 | 1.550196       0.0  0.0  0.0  0.0 | 0.964567  0.098719   0.01147  0.0  0.0 | 0.342541 |  0.26292 | 0.042818 | 0.014361 |  0.61558 |  0.61558 |  0.61558 | 2.564917 |   2.0       2.0  0.0  0.0       2.0 | 366.261912  369.083712  371.231314  0.0  370.542858 | 146.504765  146.959259  144.811656  0.0  145.500113 | 165.306061  132.262019  0.0  0.0  0.0 | 0.189482       0.0  0.0  0.0  0.0 | 19.581718   1.925437  0.223708  0.0  0.0 | 26.331745 | 12.939348 | 2.564917 |
    | 04/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.270864  514.782857  514.782857  0.0  514.782857 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.404769  226.039922  227.355189  0.0  226.933554 | 89.361908  90.003049  88.687781  0.0  89.109416 | 225.398666  225.398666  225.398666  0.0  225.398666 | 89.384192  89.384192  89.384192  0.0  89.384192 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.511993        0.0  0.0     0.0  0.0 | 1.716074       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.905557       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.520914       0.0  0.0  0.0  0.0 | 0.992409  0.093905   0.01091  0.0  0.0 | 0.122109 | 0.262614 | 0.015264 | 0.014378 | 0.616551 | 0.616551 | 0.616551 | 2.568964 |   2.0       2.0  0.0  0.0       2.0 |  365.90776  368.482456   369.91479  0.0  369.347969 | 146.363104  146.300402  144.868067  0.0  145.434888 |  166.10198  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 20.110224   1.831532  0.212797  0.0  0.0 |  26.19124 | 12.940234 | 2.568964 |
    | 04/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  511.28054  513.816706  513.816706  0.0  513.816706 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.050617  225.322111  226.197965  0.0  225.851361 | 89.220247  89.460746  88.584892  0.0  88.931496 | 224.691291  224.691291  224.691291  0.0  224.691291 | 89.125415  89.125415  89.125415  0.0  89.125415 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536166        0.0  0.0     0.0  0.0 | 1.749312       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.749312       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.494489       0.0  0.0  0.0  0.0 | 1.017534  0.089325  0.010378  0.0  0.0 | 0.080896 |  0.26101 | 0.010112 | 0.014376 | 0.622101 | 0.622101 | 0.622101 | 2.592086 |   2.0       2.0  0.0  0.0       2.0 | 365.200386  367.891636  369.048117  0.0  368.527899 | 146.080154   145.92507  144.768589  0.0  145.288807 | 166.888834  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 20.587178   1.742207  0.202419  0.0  0.0 | 26.011126 | 12.935971 | 2.592086 |
    | 04/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.286928  512.841258  512.841258  0.0  512.841258 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 222.343243  224.692074  225.398402  0.0  225.080676 | 88.937297  89.124632  88.418304  0.0   88.73603 | 223.981569  223.981569  223.981569  0.0  223.981569 | 88.859689  88.859689  88.859689  0.0  88.859689 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554329        0.0  0.0     0.0  0.0 | 1.778572       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.778572       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.470641       0.0  0.0  0.0  0.0 | 1.040209  0.084968  0.009872  0.0  0.0 | 0.097756 | 0.259302 | 0.012219 | 0.014372 | 0.626798 | 0.626798 | 0.626798 |  2.61166 |   2.0       2.0  0.0  0.0       2.0 | 364.490663   367.22113  368.271283  0.0  367.768792 | 145.796265  145.620127  144.569974  0.0  145.072466 | 167.664591  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  21.01761   1.657239  0.192547  0.0  0.0 | 25.849579 | 12.933818 |  2.61166 |
    | 04/01 07:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.955818  513.898731  513.898731  0.0  513.898731 | 0.0  0.0  0.903096  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 221.870249  225.005946  228.218517  0.0  227.018407 | 89.263679  89.652512  88.626741  0.0  88.825851 | 224.696076  224.696076  224.696076  0.0  224.696076 | 89.202655  89.202655  89.202655  0.0  89.202655 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.942913        0.0  0.0   0.847  0.0 | 2.068236       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.006494       0.0  0.0  0.0  0.0 |  1.44912       0.0  0.0  0.0  0.0 | 1.060673  0.080824  0.009391  0.0  0.0 | 0.282438 | 0.258615 | 0.035305 | 0.014383 | 0.633951 | 0.633951 | 0.633951 | 2.641463 |   2.0       2.0  0.0  0.0       2.0 | 364.968442   367.75206  369.392842  0.0  368.789461 | 145.987377  146.146671  144.505889  0.0   145.10927 | 168.539268  132.262019  0.0  0.0  0.0 | 0.061743       0.0  0.0  0.0  0.0 | 21.406057   1.576414  0.183156  0.0  0.0 | 25.873403 |  12.95474 | 2.641463 |
    | 04/01 08:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.181898  513.932503  513.932503  0.0  513.932503 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 222.246491  225.198159  227.670615  0.0   226.79139 | 89.193327  89.738972  88.516116  0.0  88.823341 | 224.756025  224.756025  224.756025  0.0  224.756025 | 89.176478  89.176478  89.176478  0.0  89.176478 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.750605        0.0  0.0   0.484  0.0 | 1.953307       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.001432       0.0  0.0  0.0  0.0 | 1.429697       0.0  0.0  0.0  0.0 |  1.07914  0.076883  0.008933  0.0  0.0 | 0.242509 | 0.258653 | 0.030314 | 0.014403 | 0.637332 | 0.637332 | 0.637332 | 2.655548 |   2.0       2.0  0.0  0.0       2.0 | 365.129927  367.807526  369.406253  0.0  368.810097 | 146.051971  146.124977   144.52625  0.0  145.122406 | 169.336566  132.262019  0.0  0.0  0.0 | 0.013617       0.0  0.0  0.0  0.0 | 21.756614   1.499532  0.174224  0.0  0.0 | 25.857258 | 12.970651 | 2.655548 |
    | 04/01 09:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.581776  513.165993  513.165993  0.0  513.165993 | 0.0  0.0  0.903096  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 | 222.306591  224.815129  226.159005  0.0  225.667272 | 88.996307  89.376974  88.345498  0.0  88.694231 | 224.225981  224.225981  224.225981  0.0  224.225981 | 88.940012  88.940012  88.940012  0.0  88.940012 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.584217        0.0  0.0   0.121  0.0 | 1.852552       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.86617       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.412169       0.0  0.0  0.0  0.0 | 1.095806  0.073133  0.008497  0.0  0.0 | 0.159492 | 0.258079 | 0.019937 | 0.014415 | 0.640798 | 0.640798 | 0.640798 |  2.66999 |   2.0       2.0  0.0  0.0       2.0 | 364.701269  367.372779  368.685229  0.0  368.137806 | 145.880507  145.793214  144.480764  0.0  145.028188 | 170.068231  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.072978   1.426399  0.165727  0.0  0.0 | 25.758672 | 12.976172 |  2.66999 |
    | 04/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.579336  513.337725  513.337725  0.0  513.337725 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 221.979417  224.762609  227.032845  0.0   226.18834 | 89.086359  89.441784  88.421148  0.0  88.693653 | 224.325723  224.325723  224.325723  0.0  224.325723 | 89.012002  89.012002  89.012002  0.0  89.012002 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.758389        0.0  0.0   0.484  0.0 | 1.994536       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.994536       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.396351       0.0  0.0  0.0  0.0 | 1.110846  0.069566  0.008083  0.0  0.0 |  0.25136 | 0.257556 |  0.03142 | 0.014427 |   0.6443 |   0.6443 |   0.6443 | 2.684585 |   2.0       2.0  0.0  0.0       2.0 | 364.699526  367.433493  368.906107  0.0  368.331188 |  145.87981  145.904232  144.431618  0.0  145.006537 | 170.832084  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.358483   1.356832  0.157644  0.0  0.0 | 25.752476 | 12.993165 | 2.684585 |
    | 04/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  509.74056  512.276806  512.276806  0.0  512.276806 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.842383  224.278811  225.177685  0.0  224.826758 | 88.736953  89.058914  88.160041  0.0  88.510967 | 223.591306  223.591306  223.591306  0.0  223.591306 | 88.685501  88.685501  88.685501  0.0  88.685501 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536246        0.0  0.0     0.0  0.0 |  1.85042       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.85042       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.382076       0.0  0.0  0.0  0.0 | 1.124419  0.066174  0.007688  0.0  0.0 | 0.148681 | 0.256982 | 0.018585 | 0.014439 | 0.647317 | 0.647317 | 0.647317 | 2.697155 |   2.0       2.0  0.0  0.0       2.0 |   364.1004  366.785987  367.959728  0.0  367.435736 |  145.64016  145.490819  144.317078  0.0  144.841071 |  171.51791  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.616139   1.290659  0.149956  0.0  0.0 | 25.644174 | 12.997311 | 2.697155 |
    | 04/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 508.746874  511.301327  511.301327  0.0  511.301327 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.243257  223.587629  224.303125  0.0  223.983707 | 88.497303  88.689177  87.973681  0.0  88.293099 |  222.88153   222.88153   222.88153  0.0   222.88153 | 88.419797  88.419797  88.419797  0.0  88.419797 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554453        0.0  0.0     0.0  0.0 | 1.878698       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.878698       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.369193       0.0  0.0  0.0  0.0 | 1.136669  0.062946  0.007313  0.0  0.0 | 0.161748 |  0.25597 | 0.020218 | 0.014445 | 0.649525 | 0.649525 | 0.649525 | 2.706353 |   2.0       2.0  0.0  0.0       2.0 | 363.390624  366.119888  367.178132  0.0  366.673558 |  145.35625  145.181439  144.123195  0.0  144.627769 | 172.193665  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 22.848664   1.227713  0.142642  0.0  0.0 | 25.549952 | 13.003085 | 2.706353 |
    | 04/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 507.750567  510.318845  510.318845  0.0  510.318845 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 220.533481  222.908881  223.553184  0.0  223.245979 | 88.213393  88.392446  87.748143  0.0  88.055348 | 222.169882  222.169882  222.169882  0.0  222.169882 | 88.148963  88.148963  88.148963  0.0  88.148963 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.568278        0.0  0.0     0.0  0.0 | 1.903778       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.903778       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.357567       0.0  0.0  0.0  0.0 | 1.147723  0.059876  0.006957  0.0  0.0 | 0.173401 |  0.25509 | 0.021675 | 0.014452 | 0.651505 | 0.651505 | 0.651505 | 2.714606 |   2.0       2.0  0.0  0.0       2.0 | 362.678976  365.420888   366.43483  0.0  365.937461 | 145.071591  144.897957  143.884015  0.0  144.381384 | 172.858165  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 23.058508   1.167837  0.135686  0.0  0.0 | 25.468262 | 13.010308 | 2.714606 |
    | 04/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 506.752198  509.331046  509.331046  0.0  509.331046 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.821833  222.208114   222.82468  0.0  222.522236 | 87.928733  88.110731  87.494165  0.0  87.796609 | 221.456761  221.456761  221.456761  0.0  221.456761 | 87.874284  87.874284  87.874284  0.0  87.874284 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.578848        0.0  0.0     0.0  0.0 | 1.926396       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.926396       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.347075       0.0  0.0  0.0  0.0 | 1.157699  0.056956  0.006617  0.0  0.0 | 0.183911 |  0.25433 | 0.022989 | 0.014461 | 0.653283 | 0.653283 | 0.653283 | 2.722011 |   2.0       2.0  0.0  0.0       2.0 | 361.965856  364.709536  365.706911  0.0  365.211987 | 144.786342   144.62151  143.624134  0.0  144.119059 | 173.510617  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 23.247883    1.11088  0.129068  0.0  0.0 | 25.397844 | 13.018836 | 2.722011 |
    | 04/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 505.752249  508.339181  508.339181  0.0  508.339181 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.108713  221.498342  222.104076  0.0  221.803495 | 87.643485  87.832704   87.22697  0.0  87.527551 | 220.742512  220.742512  220.742512  0.0  220.742512 | 87.596669  87.596669  87.596669  0.0  87.596669 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.586932        0.0  0.0     0.0  0.0 |  1.94705       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.94705       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.337606       0.0  0.0  0.0  0.0 | 1.166702  0.054178  0.006295  0.0  0.0 | 0.193474 | 0.253677 | 0.024184 | 0.014471 | 0.654878 | 0.654878 | 0.654878 |  2.72866 |   2.0       2.0  0.0  0.0       2.0 | 361.251606  363.993706  364.985347  0.0  364.491004 | 144.500643  144.345475  143.353834  0.0  143.848177 | 174.150499  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 23.418787   1.056702  0.122773  0.0  0.0 | 25.337641 |  13.02855 |  2.72866 |
    | 04/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 504.751087  507.344207  507.344207  0.0  507.344207 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 218.394464  220.784715  221.386207  0.0  221.086357 | 87.357785  87.554466  86.952974  0.0  87.252824 | 220.027397  220.027397  220.027397  0.0  220.027397 | 87.316811  87.316811  87.316811  0.0  87.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.59312        0.0  0.0     0.0  0.0 | 1.966129       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.966129       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.329061       0.0  0.0  0.0  0.0 | 1.174827  0.051536  0.005988  0.0  0.0 | 0.202244 | 0.253122 |  0.02528 | 0.014482 | 0.656313 | 0.656313 | 0.656313 | 2.734638 |   2.0       2.0  0.0  0.0       2.0 | 360.536491  363.276387  364.266537  0.0  363.772044 | 144.214596   144.06782   143.07767  0.0  143.572164 |  174.77749  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  23.57302   1.005166  0.116786  0.0  0.0 | 25.286764 | 13.039348 | 2.734638 |
    | 04/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 503.748995  506.346853  506.346853  0.0  506.346853 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 217.679348  220.069317   220.66914  0.0  220.369581 | 87.071739   87.27489  86.675067  0.0  86.974626 | 219.311616  219.311616  219.311616  0.0  219.311616 | 87.035237  87.035237  87.035237  0.0  87.035237 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.597858        0.0  0.0     0.0  0.0 | 1.983931       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.983931       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 1.321349       0.0  0.0  0.0  0.0 |  1.18216  0.049023  0.005696  0.0  0.0 | 0.210343 | 0.252656 | 0.026293 | 0.014495 | 0.657605 | 0.657605 | 0.657605 | 2.740023 |   2.0       2.0  0.0  0.0       2.0 | 359.820711  362.558686  363.549013  0.0  363.054079 | 143.928284  143.788167  142.797841  0.0  143.292775 | 175.391417  132.262019  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 | 23.712209   0.956144   0.11109  0.0  0.0 | 25.244451 | 13.051146 | 2.740023 |
    | 04/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 502.746187  505.347677  505.347677  0.0  505.347677 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.963568  219.353023  219.952184  0.0  219.652742 | 86.785427  86.993831   86.39467  0.0  86.694112 | 218.595325  218.595325  218.595325  0.0  218.595325 | 86.752352  86.752352  86.752352  0.0  86.752352 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.60149        0.0  0.0     0.0  0.0 | 2.000686       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.000065       0.0  0.0  0.0  0.0 |  1.31439       0.0  0.0  0.0  0.0 | 1.188777  0.046632  0.005418  0.0  0.0 | 0.217654 | 0.252271 | 0.027207 | 0.014508 | 0.658794 | 0.658794 | 0.658794 | 2.744973 |   2.0       2.0  0.0  0.0       2.0 | 359.104419  361.840989  362.832154  0.0  362.336662 | 143.641768  143.506688  142.515523  0.0  143.011015 | 175.992222  132.262019  0.0  0.0  0.0 |  0.00062       0.0  0.0  0.0  0.0 | 23.837822   0.909512  0.105672  0.0  0.0 | 25.209834 | 13.063845 | 2.744973 |
    | 04/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 501.742828  504.347104  504.347104  0.0  504.347104 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.247276  218.636219  219.235114  0.0  218.935721 | 86.498911  86.711458  86.112564  0.0  86.411956 |  217.87864   217.87864   217.87864  0.0   217.87864 | 86.468464  86.468464  86.468464  0.0  86.468464 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.604276        0.0  0.0     0.0  0.0 | 2.016573       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.001636       0.0  0.0  0.0  0.0 | 1.308109       0.0  0.0  0.0  0.0 | 1.194749  0.044357  0.005154  0.0  0.0 | 0.219648 | 0.251937 | 0.027456 | 0.014523 |  0.66035 |  0.66035 |  0.66035 | 2.751457 |   2.0       2.0  0.0  0.0       2.0 | 358.387734   361.12341   362.11568  0.0  361.619581 | 143.355094  143.223694  142.231424  0.0  142.727523 | 176.579925  132.262019  0.0  0.0  0.0 | 0.015557       0.0  0.0  0.0  0.0 | 23.951182   0.865155  0.100518  0.0  0.0 | 25.177546 | 13.076778 | 2.751457 |
    | 04/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 500.739045   503.34546   503.34546  0.0   503.34546 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 215.530592  217.919094  218.517877  0.0  218.218507 | 86.212237   86.42801  85.829227  0.0  86.128597 | 217.161652  217.161652  217.161652  0.0  217.161652 | 86.183808  86.183808  86.183808  0.0  86.183808 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.606415        0.0  0.0     0.0  0.0 | 2.031731       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   0.0045       0.0  0.0  0.0  0.0 | 1.302441       0.0  0.0  0.0  0.0 | 1.200138  0.042194  0.004902  0.0  0.0 | 0.221447 | 0.251624 | 0.027681 | 0.014537 | 0.662196 | 0.662196 | 0.662196 | 2.759148 |   2.0       2.0  0.0  0.0       2.0 | 357.670746  360.405968  361.399455  0.0  360.902725 | 143.068299  142.939492  141.946005  0.0  142.442734 | 177.154609  132.262019  0.0  0.0  0.0 | 0.042788       0.0  0.0  0.0  0.0 | 24.053485    0.82296  0.095616  0.0  0.0 | 25.147369 | 13.089922 | 2.759148 |
    | 04/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 499.734933  502.342992  502.342992  0.0  502.342992 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.813603  217.201748  217.800482  0.0  217.501124 | 85.925441  86.143711  85.544978  0.0  85.844336 | 216.444429  216.444429  216.444429  0.0  216.444429 | 85.898563  85.898563  85.898563  0.0  85.898563 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.608059        0.0  0.0     0.0  0.0 | 2.046267       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.008475       0.0  0.0  0.0  0.0 | 1.297326       0.0  0.0  0.0  0.0 | 1.205002  0.040136  0.004663  0.0  0.0 | 0.223071 | 0.251332 | 0.027884 | 0.014552 | 0.664286 | 0.664286 | 0.664286 | 2.767859 |   2.0       2.0  0.0  0.0       2.0 | 356.953524  359.688648  360.683402  0.0  360.186031 | 142.781409  142.654344   141.65959  0.0  142.156961 | 177.716401  132.262019  0.0  0.0  0.0 | 0.080581       0.0  0.0  0.0  0.0 | 24.145809   0.782824  0.090953  0.0  0.0 | 25.119108 | 13.103254 | 2.767859 |
    | 04/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 498.730567  501.339893  501.339893  0.0  501.339893 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.096381  216.484243  217.082951  0.0    216.7836 | 85.638552   85.85875  85.260042  0.0  85.559392 | 215.727025  215.727025  215.727025  0.0  215.727025 | 85.612867  85.612867  85.612867  0.0  85.612867 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.609325        0.0  0.0     0.0  0.0 | 2.060266       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.013403       0.0  0.0  0.0  0.0 |  1.29271       0.0  0.0  0.0  0.0 | 1.209391  0.038179  0.004436  0.0  0.0 | 0.224537 | 0.251058 | 0.028067 | 0.014567 | 0.666583 | 0.666583 | 0.666583 | 2.777428 |   2.0       2.0  0.0  0.0       2.0 |  356.23612  358.971431  359.967476  0.0  359.469456 | 142.494448  142.368462  141.372416  0.0  141.870437 |  178.26546  132.262019  0.0  0.0  0.0 | 0.127443       0.0  0.0  0.0  0.0 | 24.229128   0.744645  0.086517  0.0  0.0 | 25.092586 | 13.116755 | 2.777428 |
    | 04/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 497.726004  500.336306  500.336306  0.0  500.336306 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 213.378977  215.766617  216.365309  0.0  216.065964 | 85.351591  85.573276  84.974584  0.0  85.273929 |  215.00948   215.00948   215.00948  0.0   215.00948 | 85.326826  85.326826  85.326826  0.0  85.326826 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.610302        0.0  0.0     0.0  0.0 | 2.073792       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      2.0       0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01915       0.0  0.0  0.0  0.0 | 1.288544       0.0  0.0  0.0  0.0 | 1.213352  0.036317  0.004219  0.0  0.0 | 0.225859 | 0.250801 | 0.028232 | 0.014582 | 0.669053 | 0.669053 | 0.669053 |  2.78772 |   2.0       2.0  0.0  0.0       2.0 | 355.518575  358.254294  359.251648  0.0  358.752972 |  142.20743  142.082012  141.084658  0.0  141.583334 |  178.80197  132.262019  0.0  0.0  0.0 | 0.182085       0.0  0.0  0.0  0.0 |  24.30432   0.708329  0.082297  0.0  0.0 | 25.067644 | 13.130405 |  2.78772 |


There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.interfaces import aetinterfaces
from hydpy.interfaces import rconcinterfaces

# ...from hland
from hydpy.models.hland import hland_model
from hydpy.models.hland import hland_masks
from hydpy.models.hland.hland_constants import *


class Model(
    hland_model.Main_AETModel_V1,
    hland_model.Main_RConcModel_V1,
    hland_model.Sub_TempModel_V1,
    hland_model.Sub_PrecipModel_V1,
    hland_model.Sub_IntercModel_V1,
    hland_model.Sub_SoilWaterModel_V1,
    hland_model.Sub_SnowCoverModel_V1,
):
    """|hland_96p.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="H-HBV96-PREVAH", description="fusion of SMHI-IHMS-HBV96 and PREVAH"
    )
    __HYDPY_ROOTMODEL__ = True

    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        hland_model.Calc_TC_V1,
        hland_model.Calc_FracRain_V1,
        hland_model.Calc_RFC_SFC_V1,
        hland_model.Calc_PC_V1,
        hland_model.Calc_TF_Ic_V1,
        hland_model.Calc_SP_WC_V1,
        hland_model.Calc_SPL_WCL_SP_WC_V1,
        hland_model.Calc_SPG_WCG_SP_WC_V1,
        hland_model.Calc_CFAct_V1,
        hland_model.Calc_Melt_SP_WC_V1,
        hland_model.Calc_Refr_SP_WC_V1,
        hland_model.Calc_In_WC_V1,
        hland_model.Calc_SWE_V1,
        hland_model.Calc_SR_V1,
        hland_model.Calc_GAct_V1,
        hland_model.Calc_GlMelt_In_V1,
        hland_model.Calc_EI_Ic_V1,
        hland_model.Calc_R_SM_V1,
        hland_model.Calc_EA_SM_V1,
        hland_model.Calc_SUZ_V1,
        hland_model.Calc_DP_SUZ_V1,
        hland_model.Calc_RS_RI_SUZ_V1,
        hland_model.Calc_GR1_V1,
        hland_model.Calc_RG1_SG1_V1,
        hland_model.Calc_GR2_GR3_V1,
        hland_model.Calc_RG2_SG2_V1,
        hland_model.Calc_RG3_SG3_V1,
        hland_model.Calc_EL_SG2_SG3_V1,
        hland_model.Calc_InRC_V2,
        hland_model.Calc_OutRC_V1,
        hland_model.Calc_RT_V1,
        hland_model.Calc_QT_V1,
    )
    INTERFACE_METHODS = (
        hland_model.Get_Temperature_V1,
        hland_model.Get_MeanTemperature_V1,
        hland_model.Get_Precipitation_V1,
        hland_model.Get_InterceptedWater_V1,
        hland_model.Get_SoilWater_V1,
        hland_model.Get_SnowCover_V1,
    )
    ADD_METHODS = (
        hland_model.Calc_EI_Ic_AETModel_V1,
        hland_model.Calc_EA_SM_AETModel_V1,
        hland_model.Calc_EL_SG2_SG3_AETModel_V1,
        hland_model.Calc_OutRC_RConcModel_V1,
    )
    OUTLET_METHODS = (hland_model.Pass_Q_V1,)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (aetinterfaces.AETModel_V1, rconcinterfaces.RConcModel_V1)
    SUBMODELS = ()

    aetmodel = modeltools.SubmodelProperty(aetinterfaces.AETModel_V1)
    rconcmodel = modeltools.SubmodelProperty(
        rconcinterfaces.RConcModel_V1, optional=True
    )

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            \Sigma In_{hru} - \Sigma Out_{hru} - \Sigma Out_{basin}
            + \Delta Vol_{hru} + \Delta Vol_{snow} + \Delta Vol_{basin}
            - \Delta Vol_{rconc}
            \\ \\
            \Sigma In_{hru} =
            \sum_{k=1}^{N_{hru}} A_Z^k \cdot \sum_{t=t0}^{t1} PC_t^k + GLMelt_t^k
            \\
            \Sigma Out_{hru} =
            \sum_{k=1}^{N_{hru}} A_Z^k \cdot \sum_{t=t0}^{t1} EI_t^k + EA_t^k + EL_t^k
            \\ \\
            \Sigma Out_{basin} = \sum_{t=t0}^{t1} RT_t
            \\
            \Delta Vol_{hru} = \sum_{k=1}^{N_{hru}} A_Z^k \cdot \Big(
            \big(IC_{t0}^k - IC_{t1}^k\big) + \big(SM_{t0}^k - SM_{t1}^k\big) +
            \big( SUZ_{t0}^k - SUZ_{t1}^k \big) + \big( SG1_{t0}^k - SG1_{t1}^k \big)
            \Big)
            \\
            \Delta Vol_{snow} = \sum_{k=1}^{N_{hru}} A_Z^k \cdot \frac{1}{N_{snow}}
            \cdot \sum_{c=1}^{N_{snow}} \left(SP_{t0}^{k,s} - SP_{t1}^{k,s}\right) +
            \left(WC_{t0}^{k,s} - WC_{t1}^{k,s}\right)
            \\
            \Delta Vol_{basin} = A_L \cdot \Big(
            \big( SG2_{t0} - SG2_{t1} \big) + \big( SG3_{t0} - SG3_{t1} \big) \Big)
            \\
            \Delta Vol_{rconc} = \begin{cases}
            rconcmodel.get\_waterbalance &|\ rconcmodel \\
            0 &|\ \overline{rconcmodel} \end{cases}
            \\ \\
            N_{hru} = NmbZones \\
            N_{snow} = SClass \\
            A_Z = RelZoneAreas \\
            A_L = RelLowerZoneArea

        The returned error should always be in scale with numerical precision so that
        it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run via
        property |Sequences.conditions|.  See the integration tests of the application
        model |hland_96p| for some examples.
        """
        derived = self.parameters.derived
        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["model"]["states"]
        areas = derived.relzoneareas.value
        zonetypes = self.parameters.control.zonetype.values
        idxs_lake = zonetypes == ILAKE
        idxs_glac = zonetypes == GLACIER
        idxs_seal = zonetypes == SEALED
        idxs_land = ~idxs_lake
        idxs_upper = idxs_land * ~idxs_seal
        idxs_soil = idxs_upper * ~idxs_glac
        return (
            numpy.sum(fluxes.pc.series * areas)
            + numpy.sum((fluxes.glmelt.series * areas)[:, idxs_glac])
            - numpy.sum((fluxes.ei.series * areas)[:, idxs_land])
            - numpy.sum((fluxes.ea.series * areas)[:, idxs_soil])
            - numpy.sum((fluxes.el.series * areas)[:, idxs_lake])
            - numpy.sum(fluxes.rt.series)
            - numpy.sum(((last.ic - first["ic"]) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.wc - first["wc"], axis=0) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.sp - first["sp"], axis=0) * areas)[idxs_land])
            - numpy.sum(((last.sm - first["sm"]) * areas)[idxs_soil])
            - numpy.sum(((last.suz - first["suz"]) * areas)[idxs_upper])
            - numpy.sum(((last.sg1 - first["sg1"]) * areas)[idxs_upper])
            - (last.sg2 - first["sg2"]) * derived.rellowerzonearea
            - (last.sg3 - first["sg3"]) * derived.rellowerzonearea
            - self._get_rconcmodel_waterbalance(initial_conditions)
        )


class Masks(masktools.Masks):
    """Masks applicable to |hland_96p|."""

    CLASSES = hland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
