#  Copyright 2022 Diagnostic Image Analysis Group, Radboudumc, Nijmegen, The Netherlands
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.

import argparse
from pathlib import Path
from typing import Union

import pandas as pd

from dragon_prep.utils import split_and_save_data


def generate_mednli_dataset() -> pd.DataFrame:
    # Define some basic medical premises and hypotheses.
    # Thse examples were generated by ChatGPT (GPT-4):
    # https://chat.openai.com/share/ea3a4e49-8632-4b3d-9e0c-106cd1f3b24f
    premises = [
        "The patient has a high fever.",
        "Blood tests show a high white cell count.",
        "The X-ray confirmed a fracture in the left leg.",
        "The patient has been diagnosed with diabetes.",
        "MRI results indicate a brain tumor.",
        "The patient presents with chest pain.",
        "ECG indicates abnormal heart rhythms.",
        "The patient is short of breath and has swollen ankles.",
        "Blood tests indicate low iron levels.",
        "The patient has been diagnosed with hypertension.",
        "The biopsy indicates malignant cells.",
        "The patient has reported frequent headaches.",
        "Blood tests show elevated cholesterol levels.",
        "The patient's bone density is lower than average.",
        "The patient reports a persistent cough for over three weeks.",
        "The ultrasound shows gallstones.",
        "The patient has been diagnosed with asthma.",
        "The patient's blood pressure is consistently low.",
        "The patient experiences frequent episodes of dizziness.",
        "The patient has a history of seizures.",
        "The patient tested positive for the flu.",
        "The CT scan indicates a blockage in the patient's arteries.",
        "The patient's thyroid levels are elevated.",
        "The patient has symptoms consistent with a urinary tract infection.",
        "The skin test revealed an allergic reaction to pollen.",
        "The patient is experiencing episodes of memory loss.",
        "Prenatal tests indicate the fetus is developing normally.",
        "The patient has a history of recurrent tonsillitis.",
        "The patient complains of severe abdominal pain after eating.",
        "Hearing tests reveal a decline in the patient's right ear.",
        "The patient frequently experiences episodes of blurred vision.",
        "Blood tests indicate signs of kidney dysfunction.",
        "The patient shows symptoms of clinical depression.",
        "The patient's spinal MRI reveals a herniated disc.",
        "The patient tested positive for tuberculosis.",
        "Pulmonary tests indicate decreased lung capacity.",
        "The patient presents symptoms of hypoglycemia.",
        "The patient is recovering from a recent appendectomy.",
        "The patient has a persistent ringing in the ears.",
        "The patient's liver enzymes are elevated.",
        "The patient's blood sample revealed a malaria parasite.",
        "The patient is showing signs of Vitamin D deficiency.",
        "The EEG test indicates irregular brainwave activity.",
        "The patient is showing classic signs of chickenpox.",
        "The patient's blood sugar levels are consistently high.",
        "Biopsy results indicate benign growth on the skin.",
        "The patient's ankle swelling is accompanied by redness and heat.",
        "Lab results show the patient has high levels of uric acid.",
        "The patient reports consistent discomfort in the right shoulder.",
        "The patient's skin is yellow, suggesting jaundice.",
        "The patient complains of persistent insomnia.",
        "The endoscopy revealed inflammation in the patient's esophagus.",
        "The patient's reflexes are slower than normal.",
        "The patient displays symptoms of hypothyroidism.",
        "Lab results show low platelet count in the patient's blood.",
        "The patient reports sudden weight gain and facial puffiness.",
        "The patient displays signs of dehydration.",
        "There's an abnormal growth in the patient's pancreas, as shown by ultrasound.",
        "The patient complains of chronic fatigue.",
        "Lab tests show elevated levels of calcium in the patient's blood.",
        "The patient's stool sample shows signs of a parasitic infection.",
        "The patient has consistent ringing in their left ear.",
        "PET scan suggests abnormal activity in the patient's lungs.",
        "The patient has wounds on their feet that are slow to heal.",
        "The patient displays erratic behavior and mood swings.",
        "The patient's vitamin B12 levels are considerably low.",
        "The patient has been experiencing bouts of nausea and vomiting.",
        "The patient has a history of irregular menstrual cycles.",
        "There's a lump palpable on the patient's neck.",
        "The patient's potassium levels are higher than average.",
        "The patient complains of recurrent ear infections.",
        "Lab results indicate an imbalance in the patient's electrolytes.",
        "The patient presents with decreased sensitivity in their fingertips.",
        "The patient's echocardiogram shows a weakened heart muscle.",
        "The patient has decreased appetite and has lost weight.",
        "A mole on the patient's back has irregular borders.",
        "The patient frequently experiences shortness of breath during light activity.",
        "The patient shows symptoms consistent with polycystic ovary syndrome (PCOS).",
        "The patient has elevated bilirubin levels.",
        "The patient's visual field test indicates a blind spot.",
        "The patient has been diagnosed with rheumatoid arthritis.",
        "The X-ray shows a fracture in the patient's forearm.",
        "The patient's recent blood work indicates anemia.",
        "The patient has been experiencing frequent migraines.",
        "The MRI reveals a lesion in the patient's brain.",
        "The patient's blood pressure readings have been consistently elevated.",
        "The patient has shown an allergic reaction to penicillin.",
        "The skin test confirms a positive reaction to dust mites.",
        "The patient is on anticoagulant therapy.",
        "The patient's respiratory rate is higher than average.",
        "The patient has a family history of glaucoma.",
        "There's a high concentration of ketones in the patient's urine.",
        "The patient presents with swollen and bleeding gums.",
        "The patient's bone density test indicates osteoporosis.",
        "The patient has reported frequent bouts of vertigo.",
        "The patient's lymph nodes are enlarged.",
        "The patient suffers from chronic sinusitis.",
        "The patient's pulse rate is irregular.",
        "The patient's creatinine levels are above the normal range.",
        "The patient complains of persistent dry eyes.",
    ]

    entailments = [
        "The patient is ill.",
        "The patient's immune system is active.",
        "The patient has a broken bone.",
        "The patient has a chronic condition.",
        "There's an abnormal growth in the patient's brain.",
        "The patient may have a heart issue.",
        "There are irregularities in the patient's heart activity.",
        "The patient may have heart failure.",
        "The patient might be anemic.",
        "The patient suffers from a high blood pressure condition.",
        "The patient has cancer cells.",
        "The patient is experiencing pain in the head.",
        "The patient has high lipid levels in the blood.",
        "The patient has weak bones.",
        "The patient has respiratory symptoms.",
        "The patient has stones in the gallbladder.",
        "The patient has respiratory issues.",
        "The patient's blood pressure is not high.",
        "The patient might have issues with balance or circulation.",
        "The patient experiences abnormal brain activity.",
        "The patient is infected with an influenza virus.",
        "There might be a restriction in blood flow in the patient's arteries.",
        "The patient has an overactive thyroid.",
        "The patient likely has an infection in the urinary system.",
        "The patient is allergic to something in the environment.",
        "The patient might be experiencing cognitive challenges.",
        "The fetus is not showing signs of abnormalities.",
        "The patient has issues with their tonsils.",
        "The patient has digestive complaints.",
        "The patient has hearing problems.",
        "The patient might need eye care.",
        "The patient's kidneys might not be working optimally.",
        "The patient may be struggling with their mental health.",
        "The patient has a spine issue.",
        "The patient has an infectious disease.",
        "The patient may have respiratory problems.",
        "The patient often has low blood sugar levels.",
        "The patient recently had a surgical procedure.",
        "The patient might have tinnitus.",
        "There might be issues with the patient's liver function.",
        "The patient might have malaria.",
        "The patient needs more exposure to sunlight or Vitamin D supplements.",
        "The patient's brain activity is not normal.",
        "The patient has a contagious skin disease.",
        "The patient may have diabetes.",
        "The growth on the patient's skin is not cancerous.",
        "The patient's ankle might be infected.",
        "The patient could have gout.",
        "There might be an injury or issue in the patient's shoulder.",
        "The patient might have a liver problem.",
        "The patient struggles with sleep.",
        "The patient might have esophagitis.",
        "The patient's nervous system might not be functioning optimally.",
        "The patient's thyroid is underactive.",
        "The patient is at risk of bleeding easily.",
        "The patient might have Cushing's syndrome.",
        "The patient needs to intake more fluids.",
        "The patient may have a pancreatic tumor.",
        "The patient feels tired most of the time.",
        "The patient might have hypercalcemia.",
        "The patient might be suffering from a gastrointestinal parasite.",
        "The patient may have tinnitus in the left ear.",
        "The patient might have an issue with their respiratory system.",
        "The patient's foot wounds aren't healing properly.",
        "The patient may be facing a mental health issue.",
        "The patient may have a deficiency that affects nerve function.",
        "The patient might have a digestive system issue.",
        "The patient may have a hormonal imbalance.",
        "There might be a growth or tumor in the patient's neck area.",
        "The patient may have hyperkalemia.",
        "The patient has issues related to the ears.",
        "The patient may have a hydration or kidney problem.",
        "The patient might have nerve damage or neuropathy.",
        "The patient has a heart condition.",
        "The patient may have a metabolic or digestive issue.",
        "The mole might be a potential skin cancer risk.",
        "The patient may have a respiratory or cardiac condition.",
        "The patient may have a reproductive health issue.",
        "The patient might have a liver condition.",
        "The patient may have an eye or neural issue.",
        "The patient might suffer from joint pain and inflammation.",
        "The patient has a broken bone in their arm.",
        "The patient has a lower-than-normal number of red blood cells.",
        "The patient frequently experiences severe headaches.",
        "There's an abnormality in the patient's brain.",
        "The patient might be at risk for cardiovascular diseases.",
        "The patient should avoid medications containing penicillin.",
        "The patient may have allergies related to household dust.",
        "The patient is on medication to prevent blood clots.",
        "The patient might be experiencing breathing difficulties.",
        "The patient may be at risk of developing eye problems.",
        "The patient may be in a state of ketosis.",
        "The patient might have a gum disease.",
        "The patient's bones are weaker than they should be.",
        "The patient may feel dizzy or off-balance frequently.",
        "The patient might have an infection or immune response.",
        "The patient has inflammation of the sinuses.",
        "The patient might have a heart-related condition.",
        "The patient may have a kidney function issue.",
        "The patient might have a tear film dysfunction.",
    ]

    contradictions = [
        "The patient is in perfect health.",
        "The patient's immune system is not responding.",
        "The patient has no injuries.",
        "The patient is free from any diseases.",
        "There's no issue found in the MRI scan.",
        "The patient has no cardiovascular complaints.",
        "The ECG results are perfectly normal.",
        "The patient is showing no symptoms of a cardiac condition.",
        "The patient has optimal iron levels.",
        "The patient's blood pressure is normal.",
        "There are no signs of cancer.",
        "The patient has never complained about headaches.",
        "The patient's cholesterol levels are low.",
        "The patient's bones are of average density.",
        "The patient has no respiratory issues.",
        "The ultrasound shows a clear gallbladder.",
        "The patient does not have any breathing issues.",
        "The patient's blood pressure is elevated.",
        "The patient is always stable and never dizzy.",
        "The patient has never had a seizure.",
        "The patient doesn't have the flu.",
        "The CT scan came back perfectly clear.",
        "The patient's thyroid levels are perfectly normal.",
        "The patient's urinary system is perfectly healthy.",
        "The patient has no allergies.",
        "The patient has a sharp memory.",
        "There are complications with the fetus's development.",
        "The patient has never had issues with their tonsils.",
        "The patient digests food without any problems.",
        "The patient has perfect hearing in both ears.",
        "The patient's vision is crystal clear.",
        "The patient's kidney functions are optimal.",
        "The patient is in a happy and positive mental state.",
        "The patient's spine is in perfect condition.",
        "The patient doesn't have any infectious diseases.",
        "The patient's lung functions are above average.",
        "The patient always maintains a balanced blood sugar level.",
        "The patient has never undergone surgery.",
        "The patient has never complained about any auditory issues.",
        "The patient's liver functions are within normal range.",
        "The patient has no signs of infection.",
        "The patient's Vitamin D levels are optimal.",
        "The EEG test results are normal.",
        "The patient has clear skin without any spots.",
        "The patient's blood sugar levels are normal.",
        "The skin growth is malignant.",
        "The patient's ankle looks completely normal.",
        "The patient's uric acid levels are within a normal range.",
        "The patient's shoulder is in perfect condition.",
        "The patient's skin coloration is perfectly normal.",
        "The patient sleeps soundly every night.",
        "The patient's esophagus is in perfect condition.",
        "The patient has quick and responsive reflexes.",
        "The patient has hyperthyroidism.",
        "The patient's platelet count is high.",
        "The patient has maintained a consistent weight.",
        "The patient is well-hydrated.",
        "The patient's pancreas is clear of any growths.",
        "The patient feels energetic all the time.",
        "The patient's calcium levels are low.",
        "The patient's digestive system is free from any parasites.",
        "The patient has never had any auditory issues.",
        "The patient's lungs are in perfect condition.",
        "The patient's wounds heal rapidly.",
        "The patient's behavior is consistent and stable.",
        "The patient's vitamin B12 levels are perfectly normal.",
        "The patient has a strong stomach and never vomits.",
        "The patient's menstrual cycles are perfectly regular.",
        "The patient's neck is clear of any lumps or growths.",
        "The patient's potassium levels are extremely low.",
        "The patient has perfectly healthy ears.",
        "The patient's electrolyte levels are perfectly balanced.",
        "The patient has a heightened sensitivity in their fingertips.",
        "The patient's heart muscle is robust and strong.",
        "The patient has a hearty appetite and is gaining weight.",
        "The mole on the patient's back is perfectly round and consistent.",
        "The patient never experiences breathing difficulties.",
        "The patient shows no signs related to PCOS.",
        "The patient's bilirubin levels are extremely low.",
        "The patient's vision is impeccable without any blind spots.",
        "The patient has perfectly healthy joints.",
        "The patient's forearm is completely intact.",
        "The patient has a high red blood cell count.",
        "The patient has never experienced a headache.",
        "The patient's brain MRI came back clear.",
        "The patient always has low blood pressure.",
        "The patient tolerates penicillin very well.",
        "The patient showed no reaction to dust mites.",
        "The patient is not on any blood-related medication.",
        "The patient's breathing rate is exceptionally slow.",
        "No one in the patient's family has ever had eye problems.",
        "The patient's urine is free of ketones.",
        "The patient's gums are in pristine condition.",
        "The patient's bone density is exceptionally high.",
        "The patient has never felt dizzy in their life.",
        "The patient's lymph nodes are of regular size.",
        "The patient has perfectly clear sinuses.",
        "The patient's pulse rate is perfectly steady.",
        "The patient's creatinine levels are exceptionally low.",
        "The patient frequently complains about watery eyes.",
    ]

    neutrals = [
        "The patient recently visited Africa.",
        "The patient is allergic to penicillin.",
        "The patient is 45 years old.",
        "The patient has a family history of heart disease.",
        "The patient is concerned about potential side effects.",
        "The patient is planning a vacation next month.",
        "The patient participates in regular exercise.",
        "The patient follows a vegetarian diet.",
        "The patient has two children.",
        "The patient works in a high-stress job.",
        "The patient enjoys reading in their spare time.",
        "The patient has a cat at home.",
        "The patient doesn't consume alcohol.",
        "The patient had surgery two years ago.",
        "The patient lives in an urban area.",
        "The patient's favorite color is blue.",
        "The patient frequently travels for work.",
        "The patient recently celebrated a birthday.",
        "The patient is considering changing their diet.",
        "The patient has been practicing meditation.",
        "The patient is considering a career change.",
        "The patient recently read a book about meditation.",
        "The patient plans to attend a wedding next week.",
        "The patient is interested in learning a new language.",
        "The patient recently adopted a pet rabbit.",
        "The patient is a fan of classical music.",
        "The patient expressed interest in traveling to Asia.",
        "The patient recently painted their living room blue.",
        "The patient loves to bake during weekends.",
        "The patient has taken up swimming lessons.",
        "The patient plans to buy a new car soon.",
        "The patient enjoys attending theater performances.",
        "The patient recently took a pottery class.",
        "The patient's favorite holiday is Christmas.",
        "The patient's childhood dream was to become an astronaut.",
        "The patient is learning to play the guitar.",
        "The patient is a fan of mystery novels.",
        "The patient volunteers at a local animal shelter.",
        "The patient recently climbed a mountain.",
        "The patient enjoys watching historical documentaries.",
        "The patient enjoys playing chess.",
        "The patient recently started a book club.",
        "The patient's favorite movie genre is comedy.",
        "The patient is thinking of adopting a dog.",
        "The patient is planning to learn how to cook Italian dishes.",
        "The patient recently went on a hiking trip.",
        "The patient is a fan of jazz music.",
        "The patient is thinking of moving to a coastal city.",
        "The patient wants to take a photography course.",
        "The patient's favorite season is autumn.",
        "The patient likes collecting vintage postcards.",
        "The patient recently bought a bicycle.",
        "The patient has always dreamed of visiting Greece.",
        "The patient recently started gardening.",
        "The patient loves watching detective series on TV.",
        "The patient recently attended a pottery workshop.",
        "The patient enjoys morning runs in the park.",
        "The patient's favorite fruit is mango.",
        "The patient is learning to play the piano.",
        "The patient is considering writing a novel.",
        "The patient has recently started practicing yoga.",
        "The patient expressed interest in astronomy.",
        "The patient's favorite hobby is bird watching.",
        "The patient is planning to take a culinary class.",
        "The patient has recently started playing board games.",
        "The patient is thinking of renovating their kitchen.",
        "The patient is an avid fan of tennis.",
        "The patient recently attended a jazz concert.",
        "The patient plans to take a road trip next month.",
        "The patient enjoys camping during summers.",
        "The patient is a fan of impressionist art.",
        "The patient enjoys cooking Thai food.",
        "The patient is considering adopting a parrot.",
        "The patient has started learning ballroom dancing.",
        "The patient is an active member of a local debate club.",
        "The patient enjoys watching documentaries about nature.",
        "The patient has expressed interest in sailing.",
        "The patient recently visited a vineyard.",
        "The patient collects vintage stamps.",
        "The patient enjoys writing poetry in their free time.",
        "The patient is an avid mountain climber.",
        "The patient recently baked a chocolate cake.",
        "The patient is interested in ancient history.",
        "The patient is planning a trip to the Maldives.",
        "The patient enjoys listening to classical music.",
        "The patient has two cats and a goldfish.",
        "The patient is reading a mystery novel.",
        "The patient recently planted roses in their garden.",
        "The patient is learning to speak Spanish.",
        "The patient frequently attends yoga classes.",
        "The patient loves winter sports, especially skiing.",
        "The patient is thinking of starting a podcast.",
        "The patient is passionate about environmental causes.",
        "The patient recently attended a wedding.",
        "The patient enjoys beach volleyball.",
        "The patient is a fan of gourmet coffee.",
        "The patient's favorite color is turquoise.",
        "The patient recently visited a modern art museum.",
        "The patient is planning to get a new tattoo.",
        "The patient enjoys DIY home projects.",
    ]

    dataset = []

    assert len(premises) == len(entailments) == len(contradictions) == len(neutrals)
    assert len(premises) == len(set(premises))

    # Generating for entailment
    for premise, entailment in zip(premises, entailments):
        dataset.append({
            "text_parts": [premise, entailment],
            "single_label_multi_class_classification_target": "entailment",
            "premise": premise,
        })

    # Generating for contradiction
    for premise, contradiction in zip(premises, contradictions):
        dataset.append({
            "text_parts": [premise, contradiction],
            "single_label_multi_class_classification_target": "contradiction",
            "premise": premise,
        })

    # Generating for neutral
    for premise, neutral in zip(premises, neutrals):
        dataset.append({
            "text_parts": [premise, neutral],
            "single_label_multi_class_classification_target": "neutral",
            "premise": premise,
        })

    dataset = pd.DataFrame(dataset)
    return dataset


def main(
    output_dir: Union[Path, str] = "/output",
    task_name="Task103_Example_mednli",
) -> None:
    """
    Generate an example multi-label regression dataset for NLP
    """
    # generate the data
    df = generate_mednli_dataset()
    df["uid"] = [f"{task_name.split('_')[0]}_case{idx}" for idx in range(len(df))]

    # make test and cross-validation splits
    split_and_save_data(
        df=df,
        task_name=task_name,
        output_dir=output_dir,
        split_by="premise",
    )


if __name__ == '__main__':
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('--output_dir', type=str, default="/output",
                        help="Directory to store data.")
    args = parser.parse_args()

    main(
        output_dir=args.output_dir,
    )
