from functools import wraps, lru_cache
import inspect
import re
import json
import textwrap

from openai import OpenAI

# This is a global variable that will store the OpenAI client. This
# enables any user to set the client  under their own terms (key, project...)
# via `gpt_function.SETTINGS["openai_client"] = OpenAI()`.
# If the user doesn't set up the client, it gets set up automatically with
# OpenAI() the first time a client is needed
SETTINGS = {"openai_client": None}


# This is the universal template we'll use. Look how simple it is!
# It mostly works because ChatGPT is great at understanding Python functions.
SYSTEM_PROMPT_TEMPLATE = """
For the following python function, evaluate the user-provided input.

```python
{code}
```

{thinking}
Provide the final output at the end as follows,
where FUNCTION_OUTPUT is in JSON format:

<ANSWER>
{{"result": FUNCTION_OUTPUT}}
</ANSWER>
"""

report_template = """
SYSTEM:
-------
{system_prompt}

INPUT:
------

{user_input}

RESPONSE:
---------

{response_content}
"""


GPT_PARAMETER_DOCS = """
Parameters
-----------

model: str
    Any OpenAI model, e.g. `gpt-4o`, `gpt-4o-mini`. `gpt-4o-mini` is faster,
    and 10 times less expensive, but the answers can be less good.

think_through: bool
    If true, this will ask the GPT to go through the steps of its reasoning
    before giving an answer. It does greatly improve the final answer (this
    is the way GPTs work best) and it makes troubleshooting easier, but also
    results in longer and more expensive function calls.

retries
    It might happen that the GPT, for some reason, produces an answer that
    python cannot parse. In that case retrying often solves the problem.
    Once the maximum number of retries is reached without success, a ValueError
    exception is raised with the content of the response.

debug
    If True, the full answer from the GPT will be printed. This is practical
    to troubleshoot cases where GPT returns a "valid" answer which is not
    exactly what you expected, and you want to see its reasoning.
"""

ADDITIONAL_DOCS = """

Function auto-generated by @gpt_function.
- The execution happens on a chatbot, and may require an API key.
- Answers can be changing, their quality and validity are not guaranteed.
- The following auto-added parameters can control the GPT execution:

{GPT_PARAMETER_DOCS}
"""


def gpt_function(
    gpt_model: str = "gpt-4o-mini",
    think_through: bool = False,
    retries: int = 0,
    debug: bool = False,
):
    f"""Decorator that runs a function on a GPT model.

    {GPT_PARAMETER_DOCS}
    """

    if hasattr(gpt_model, "__call__"):
        # Case where the user wrote @gpt_function without argument nor
        # parenthesis. In this case we understand that the first argument
        # is the function itself and that the user meant to use the decorator
        # with the default parameter values.
        func = gpt_model
        decorator = gpt_function("gpt-4o-mini")
        return decorator(func)

    def decorator(func):
        """A decorator that will run the user-defined function on a GPT"""

        @add_kwargs(
            gpt_model=gpt_model,
            think_through=think_through,
            retries=retries,
            debug=debug,
        )
        @wraps(func)  # This preserves the docstring and other attributes
        def wrapper(*args, **kwargs):

            # This next block enables users of already-decorated functions to pass
            # extra arguments to the function relative to the GPT execution.
            # This is not the most elegant solution, but it works.
            gpt_model = kwargs.pop("gpt_model")
            think_through = kwargs.pop("think_through")
            retries = kwargs.pop("retries")
            debug = kwargs.pop("debug")

            # Build the system prompt: add the function's code (definition and
            # docstring) and the method of thinking ("think as you go" vs
            # write the result directly)

            code = inspect.getsource(func)
            code
            if think_through:
                thinking = (
                    "Think carefully through the answer. "
                    "If the function docstring suggests steps, take these steps"
                )
            else:
                thinking = (
                    "Write the result directly without providing any explanation."
                )
            system_prompt = SYSTEM_PROMPT_TEMPLATE.format(code=code, thinking=thinking)

            # Generate the user input. For instance, if the user ran
            # `func(1, 2, c=3)`, this will generate the string "1, 2, c=3"
            # We use json.dumps so that strings are properly quoted and escaped
            args_str = [json.dumps(a) for a in args]
            kwargs_str = [f"{k}={json.dumps(v)}" for (k, v) in kwargs.items()]
            user_input = ", ".join(args_str + kwargs_str)

            gpt_messages = [
                {"role": "system", "content": system_prompt},
                {"role": "user", "content": user_input},
            ]

            client = SETTINGS["openai_client"]
            if client is None:
                client = SETTINGS["openai_client"] = OpenAI()
            for retry in range(1 + retries):
                response = client.chat.completions.create(
                    messages=gpt_messages, model=gpt_model
                )
                response_content = response.choices[0].message.content

                full_report = report_template.format(
                    system_prompt=system_prompt,
                    user_input=user_input,
                    response_content=response_content,
                )
                lines = full_report.splitlines()
                wrapped_lines = [textwrap.fill(line, width=80) for line in lines]
                full_report = "\n".join(wrapped_lines)

                if debug:
                    # print a report with the prompt system, user system and response

                    print(full_report)
                try:
                    answer_pattern = "<ANSWER>(.*)</ANSWER>"
                    matches = re.findall(answer_pattern, response_content, re.DOTALL)
                    answer = matches[0]
                    return json.loads(answer)["result"]
                except Exception as e:
                    if retry == retries:
                        raise ValueError(f"GPT transaction: {full_report}") from e

        # Add a text to the docstring so it will be clear to users that the
        # function is actually running on a chatbot.
        wrapper.__doc__ += ADDITIONAL_DOCS
        return wrapper

    return decorator


def add_kwargs(**new_kwargs):
    """Decorator to add keyword arguments to a function's signature."""

    def decorator(func):
        # Get the original function's signature
        sig = inspect.signature(func)
        params = list(sig.parameters.values())

        # Add new kwargs to the parameters
        new_params = [
            inspect.Parameter(name, inspect.Parameter.KEYWORD_ONLY, default=value)
            for name, value in new_kwargs.items()
        ]

        # Update the function's signature
        params.extend(new_params)
        new_sig = sig.replace(parameters=params)

        @wraps(func)
        def wrapper(*args, **kwargs):
            # The new kwargs are already added to the signature, so they will
            # be available in kwargs when the function is called.
            kwargs = {**new_kwargs, **kwargs}
            return func(*args, **kwargs)

        # Update the signature of the wrapper function
        wrapper.__signature__ = new_sig

        return wrapper

    return decorator
