from __future__ import annotations

from dataclasses import dataclass
from typing import TYPE_CHECKING, Any, Collection, Mapping, Sequence, overload

import hypothesis.strategies as st
from hypothesis.errors import InvalidArgument

from polars._utils.deprecation import issue_deprecation_warning
from polars.dataframe import DataFrame
from polars.datatypes import DataType, DataTypeClass, Null
from polars.series import Series
from polars.string_cache import StringCache
from polars.testing.parametric.strategies._utils import flexhash
from polars.testing.parametric.strategies.data import data
from polars.testing.parametric.strategies.dtype import _instantiate_dtype, dtypes

if TYPE_CHECKING:
    from typing import Literal

    from hypothesis.strategies import DrawFn, SearchStrategy

    from polars import LazyFrame
    from polars.type_aliases import PolarsDataType


_ROW_LIMIT = 5  # max generated frame/series length
_COL_LIMIT = 5  # max number of generated cols


@st.composite
def series(  # noqa: D417
    draw: DrawFn,
    /,
    *,
    name: str | SearchStrategy[str] | None = None,
    dtype: PolarsDataType | None = None,
    size: int | None = None,
    min_size: int = 0,
    max_size: int = _ROW_LIMIT,
    strategy: SearchStrategy[Any] | None = None,
    allow_null: bool = True,
    unique: bool = False,
    chunked: bool | None = None,
    allowed_dtypes: Collection[PolarsDataType] | PolarsDataType | None = None,
    excluded_dtypes: Collection[PolarsDataType] | PolarsDataType | None = None,
    **kwargs: Any,
) -> Series:
    """
    Hypothesis strategy for producing polars Series.

    Parameters
    ----------
    name : {str, strategy}, optional
        literal string or a strategy for strings (or None), passed to the Series
        constructor name-param.
    dtype : PolarsDataType, optional
        a valid polars DataType for the resulting series.
    size : int, optional
        if set, creates a Series of exactly this size (ignoring min_size/max_size
        params).
    min_size : int
        if not passing an exact size, can set a minimum here (defaults to 0).
        no-op if `size` is set.
    max_size : int
        if not passing an exact size, can set a maximum value here (defaults to
        MAX_DATA_SIZE). no-op if `size` is set.
    strategy : strategy, optional
        supports overriding the default strategy for the given dtype.
    allow_null : bool
        Allow nulls as possible values and allow the `Null` data type by default.
    unique : bool, optional
        indicate whether Series values should all be distinct.
    chunked : bool, optional
        ensure that Series with more than one element have `n_chunks` > 1.
        if omitted, chunking is applied at random.
    allowed_dtypes : {list,set}, optional
        when automatically generating Series data, allow only these dtypes.
    excluded_dtypes : {list,set}, optional
        when automatically generating Series data, exclude these dtypes.
    **kwargs
        Additional keyword arguments that are passed to the underlying data generation
        strategies.

    null_probability : float
        Percentage chance (expressed between 0.0 => 1.0) that any Series value is null.
        This is applied independently of any None values generated by the underlying
        strategy.

        .. deprecated:: 0.20.26
            Use `allow_null` instead.

    allow_infinities : bool, optional
        Allow generation of +/-inf values for floating-point dtypes.

        .. deprecated:: 0.20.26
            Use `allow_infinity` instead.

    Notes
    -----
    In actual usage this is deployed as a unit test decorator, providing a strategy
    that generates multiple Series with the given dtype/size characteristics for the
    unit test. While developing a strategy/test, it can also be useful to call
    `.example()` directly on a given strategy to see concrete instances of the
    generated data.

    Examples
    --------
    The strategy is generally used to generate series in a unit test:

    >>> from polars.testing.parametric import series
    >>> from hypothesis import given
    >>> @given(s=series(min_size=3, max_size=5))
    ... def test_series_len(s: pl.Series) -> None:
    ...     assert 3 <= s.len() <= 5

    Drawing examples interactively is also possible with the `.example()` method.
    This should be avoided while running tests.

    >>> from polars.testing.parametric import lists
    >>> s = series(strategy=lists(pl.String, select_from=["xx", "yy", "zz"]))
    >>> s.example()  # doctest: +SKIP
    shape: (4,)
    Series: '' [list[str]]
    [
            ["zz", "zz"]
            ["zz", "xx", "yy"]
            []
            ["xx"]
    ]
    """
    if (null_prob := kwargs.pop("null_probability", None)) is not None:
        allow_null = _handle_null_probability_deprecation(null_prob)  # type: ignore[assignment]
    if (allow_inf := kwargs.pop("allow_infinities", None)) is not None:
        issue_deprecation_warning(
            "`allow_infinities` is deprecated. Use `allow_infinity` instead.",
            version="0.20.26",
        )
        kwargs["allow_infinity"] = allow_inf

    if isinstance(allowed_dtypes, (DataType, DataTypeClass)):
        allowed_dtypes = [allowed_dtypes]
    elif allowed_dtypes is not None and not isinstance(allowed_dtypes, Sequence):
        allowed_dtypes = list(allowed_dtypes)
    if isinstance(excluded_dtypes, (DataType, DataTypeClass)):
        excluded_dtypes = [excluded_dtypes]
    elif excluded_dtypes is not None:
        if not isinstance(excluded_dtypes, list):
            excluded_dtypes = list(excluded_dtypes)

    if not allow_null and not (allowed_dtypes is not None and Null in allowed_dtypes):
        if excluded_dtypes is None:
            excluded_dtypes = [Null]
        else:
            excluded_dtypes.append(Null)

    if strategy is None:
        if dtype is None:
            dtype_strat = dtypes(
                allowed_dtypes=allowed_dtypes, excluded_dtypes=excluded_dtypes
            )
        else:
            dtype_strat = _instantiate_dtype(
                dtype,
                allowed_dtypes=allowed_dtypes,
                excluded_dtypes=excluded_dtypes,
            )
        dtype = draw(dtype_strat)

    if size is None:
        size = draw(st.integers(min_value=min_size, max_value=max_size))

    if isinstance(name, st.SearchStrategy):
        name = draw(name)

    if size == 0:
        values = []
    else:
        # Create series using dtype-specific strategy to generate values
        if strategy is None:
            strategy = data(
                dtype,  # type: ignore[arg-type]
                allow_null=allow_null,
                **kwargs,
            )

        values = draw(
            st.lists(
                strategy,
                min_size=size,
                max_size=size,
                unique_by=(flexhash if unique else None),
            )
        )

    s = Series(name=name, values=values, dtype=dtype)

    # Apply chunking
    if size > 1:
        if chunked is None:
            chunk_probability = 0.5
            chunked = draw(st.floats(0.0, 1.0)) < chunk_probability
        if chunked:
            split_at = size // 2
            s = s[:split_at].append(s[split_at:])

    return s


@overload
def dataframes(
    cols: int | column | Sequence[column] | None = None,
    *,
    lazy: Literal[False] = ...,
    min_cols: int = 0,
    max_cols: int = _COL_LIMIT,
    size: int | None = None,
    min_size: int = 0,
    max_size: int = _ROW_LIMIT,
    chunked: bool | None = None,
    include_cols: Sequence[column] | column | None = None,
    allow_null: bool | Mapping[str, bool] = True,
    allowed_dtypes: Collection[PolarsDataType] | PolarsDataType | None = None,
    excluded_dtypes: Collection[PolarsDataType] | PolarsDataType | None = None,
    **kwargs: Any,
) -> SearchStrategy[DataFrame]: ...


@overload
def dataframes(
    cols: int | column | Sequence[column] | None = None,
    *,
    lazy: Literal[True],
    min_cols: int = 0,
    max_cols: int = _COL_LIMIT,
    size: int | None = None,
    min_size: int = 0,
    max_size: int = _ROW_LIMIT,
    chunked: bool | None = None,
    include_cols: Sequence[column] | column | None = None,
    allow_null: bool | Mapping[str, bool] = True,
    allowed_dtypes: Collection[PolarsDataType] | PolarsDataType | None = None,
    excluded_dtypes: Collection[PolarsDataType] | PolarsDataType | None = None,
    **kwargs: Any,
) -> SearchStrategy[LazyFrame]: ...


@st.composite
def dataframes(  # noqa: D417
    draw: DrawFn,
    /,
    cols: int | column | Sequence[column] | None = None,
    *,
    lazy: bool = False,
    min_cols: int = 1,
    max_cols: int = _COL_LIMIT,
    size: int | None = None,
    min_size: int = 0,
    max_size: int = _ROW_LIMIT,
    chunked: bool | None = None,
    include_cols: Sequence[column] | column | None = None,
    allow_null: bool | Mapping[str, bool] = True,
    allowed_dtypes: Collection[PolarsDataType] | PolarsDataType | None = None,
    excluded_dtypes: Collection[PolarsDataType] | PolarsDataType | None = None,
    **kwargs: Any,
) -> DataFrame | LazyFrame:
    """
    Hypothesis strategy for producing polars DataFrames or LazyFrames.

    Parameters
    ----------
    cols : {int, columns}, optional
        integer number of columns to create, or a sequence of `column` objects
        that describe the desired DataFrame column data.
    lazy : bool, optional
        produce a LazyFrame instead of a DataFrame.
    min_cols : int, optional
        if not passing an exact size, can set a minimum here (defaults to 0).
    max_cols : int, optional
        if not passing an exact size, can set a maximum value here (defaults to
        MAX_COLS).
    size : int, optional
        if set, will create a DataFrame of exactly this size (and ignore
        the min_size/max_size len params).
    min_size : int, optional
        if not passing an exact size, set the minimum number of rows in the
        DataFrame.
    max_size : int, optional
        if not passing an exact size, set the maximum number of rows in the
        DataFrame.
    chunked : bool, optional
        ensure that DataFrames with more than row have `n_chunks` > 1. if
        omitted, chunking will be randomised at the level of individual Series.
    include_cols : [column], optional
        a list of `column` objects to include in the generated DataFrame. note that
        explicitly provided columns are appended onto the list of existing columns
        (if any present).
    allow_null : bool or Mapping[str, bool]
        Allow nulls as possible values and allow the `Null` data type by default.
        Accepts either a boolean or a mapping of column names to booleans.
    allowed_dtypes : {list,set}, optional
        when automatically generating data, allow only these dtypes.
    excluded_dtypes : {list,set}, optional
        when automatically generating data, exclude these dtypes.
    **kwargs
        Additional keyword arguments that are passed to the underlying data generation
        strategies.

    null_probability : {float, dict[str,float]}, optional
        percentage chance (expressed between 0.0 => 1.0) that a generated value is
        None. this is applied independently of any None values generated by the
        underlying strategy, and can be applied either on a per-column basis (if
        given as a `{col:pct}` dict), or globally. if null_probability is defined
        on a column, it takes precedence over the global value.

        .. deprecated:: 0.20.26
            Use `allow_null` instead.

    allow_infinities : bool, optional
        optionally disallow generation of +/-inf values for floating-point dtypes.

        .. deprecated:: 0.20.26
            Use `allow_infinity` instead.

    Notes
    -----
    In actual usage this is deployed as a unit test decorator, providing a strategy
    that generates DataFrames or LazyFrames with the given characteristics for
    the unit test. While developing a strategy/test, it can also be useful to
    call `.example()` directly on a given strategy to see concrete instances of
    the generated data.

    Examples
    --------
    The strategy is generally used to generate series in a unit test:

    >>> from polars.testing.parametric import dataframes
    >>> from hypothesis import given
    >>> @given(df=dataframes(min_size=3, max_size=5))
    ... def test_df_height(df: pl.DataFrame) -> None:
    ...     assert 3 <= df.height <= 5

    Drawing examples interactively is also possible with the `.example()` method.
    This should be avoided while running tests.

    >>> df = dataframes(allowed_dtypes=[pl.Datetime, pl.Float64], max_cols=3)
    >>> df.example()  # doctest: +SKIP
    shape: (3, 3)
    ┌─────────────┬────────────────────────────┬───────────┐
    │ col0        ┆ col1                       ┆ col2      │
    │ ---         ┆ ---                        ┆ ---       │
    │ f64         ┆ datetime[ns]               ┆ f64       │
    ╞═════════════╪════════════════════════════╪═══════════╡
    │ NaN         ┆ 1844-07-05 06:19:48.848808 ┆ 3.1436e16 │
    │ -1.9914e218 ┆ 2068-12-01 23:05:11.412277 ┆ 2.7415e16 │
    │ 0.5         ┆ 2095-11-19 22:05:17.647961 ┆ -0.5      │
    └─────────────┴────────────────────────────┴───────────┘

    Use :class:`column` for more control over which exactly which columns are generated.

    >>> from polars.testing.parametric import column
    >>> dfs = dataframes(
    ...     [
    ...         column("x", dtype=pl.Int32),
    ...         column("y", dtype=pl.Float64),
    ...     ],
    ...     size=2,
    ... )
    >>> dfs.example()  # doctest: +SKIP
    shape: (2, 2)
    ┌───────────┬────────────┐
    │ x         ┆ y          │
    │ ---       ┆ ---        │
    │ i32       ┆ f64        │
    ╞═══════════╪════════════╡
    │ -15836    ┆ 1.1755e-38 │
    │ 575050513 ┆ NaN        │
    └───────────┴────────────┘
    """
    if (null_prob := kwargs.pop("null_probability", None)) is not None:
        allow_null = _handle_null_probability_deprecation(null_prob)
    if (allow_inf := kwargs.pop("allow_infinities", None)) is not None:
        issue_deprecation_warning(
            "`allow_infinities` is deprecated. Use `allow_infinity` instead.",
            version="0.20.26",
        )
        kwargs["allow_infinity"] = allow_inf

    if isinstance(include_cols, column):
        include_cols = [include_cols]

    if cols is None:
        n_cols = draw(st.integers(min_value=min_cols, max_value=max_cols))
        cols = [column() for _ in range(n_cols)]
    elif isinstance(cols, int):
        cols = [column() for _ in range(cols)]
    elif isinstance(cols, column):
        cols = [cols]
    else:
        cols = list(cols)

    if include_cols:
        cols.extend(list(include_cols))

    if size is None:
        size = draw(st.integers(min_value=min_size, max_value=max_size))

    # Process columns
    for idx, c in enumerate(cols):
        if c.name is None:
            c.name = f"col{idx}"
        if c.allow_null is None:
            if isinstance(allow_null, Mapping):
                c.allow_null = allow_null.get(c.name, True)
            else:
                c.allow_null = allow_null

    with StringCache():
        data = {
            c.name: draw(
                series(
                    name=c.name,
                    dtype=c.dtype,
                    size=size,
                    allow_null=c.allow_null,  # type: ignore[arg-type]
                    strategy=c.strategy,
                    unique=c.unique,
                    chunked=None if chunked is None else False,
                    allowed_dtypes=allowed_dtypes,
                    excluded_dtypes=excluded_dtypes,
                    **kwargs,
                )
            )
            for c in cols
        }

    df = DataFrame(data)

    # Optionally generate chunked frames
    if size > 1 and chunked:
        split_at = size // 2
        df = df[:split_at].vstack(df[split_at:])

    if lazy:
        return df.lazy()

    return df


@dataclass
class column:
    """
    Define a column for use with the `dataframes` strategy.

    Parameters
    ----------
    name : str
        string column name.
    dtype : PolarsDataType
        a polars dtype.
    strategy : strategy, optional
        supports overriding the default strategy for the given dtype.
    allow_null : bool, optional
        Allow nulls as possible values and allow the `Null` data type by default.
    unique : bool, optional
        flag indicating that all values generated for the column should be unique.

    null_probability : float, optional
        percentage chance (expressed between 0.0 => 1.0) that a generated value is
        None. this is applied independently of any None values generated by the
        underlying strategy.

        .. deprecated:: 0.20.26
            Use `allow_null` instead.

    Examples
    --------
    >>> from polars.testing.parametric import column
    >>> dfs = dataframes(
    ...     [
    ...         column("x", dtype=pl.Int32, allow_null=True),
    ...         column("y", dtype=pl.Float64),
    ...     ],
    ...     size=2,
    ... )
    >>> dfs.example()  # doctest: +SKIP
    shape: (2, 2)
    ┌───────────┬────────────┐
    │ x         ┆ y          │
    │ ---       ┆ ---        │
    │ i32       ┆ f64        │
    ╞═══════════╪════════════╡
    │ null      ┆ 1.1755e-38 │
    │ 575050513 ┆ inf        │
    └───────────┴────────────┘
    """

    name: str | None = None
    dtype: PolarsDataType | None = None
    strategy: SearchStrategy[Any] | None = None
    allow_null: bool | None = None
    unique: bool = False

    null_probability: float | None = None

    def __post_init__(self) -> None:
        if self.null_probability is not None:
            self.allow_null = _handle_null_probability_deprecation(  # type: ignore[assignment]
                self.null_probability
            )


def _handle_null_probability_deprecation(
    null_probability: float | Mapping[str, float],
) -> bool | dict[str, bool]:
    issue_deprecation_warning(
        "`null_probability` is deprecated. Use `include_nulls` instead.",
        version="0.20.26",
    )

    def prob_to_bool(prob: float) -> bool:
        if not (0.0 <= prob <= 1.0):
            msg = f"`null_probability` should be between 0.0 and 1.0, got {prob!r}"
            raise InvalidArgument(msg)

        return bool(prob)

    if isinstance(null_probability, Mapping):
        return {col: prob_to_bool(prob) for col, prob in null_probability.items()}
    else:
        return prob_to_bool(null_probability)
