import { Resource, SecretValue } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IGroup } from './group';
import { IIdentity } from './identity-base';
import { IManagedPolicy } from './managed-policy';
import { Policy } from './policy';
import { PolicyStatement } from './policy-statement';
import { AddToPrincipalPolicyResult, IPrincipal, PrincipalPolicyFragment } from './principals';
/**
 * Represents an IAM user.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_users.html
 * @stability stable
 */
export interface IUser extends IIdentity {
    /**
     * The user's name.
     *
     * @stability stable
     * @attribute true
     */
    readonly userName: string;
    /**
     * The user's ARN.
     *
     * @stability stable
     * @attribute true
     */
    readonly userArn: string;
    /**
     * Adds this user to a group.
     *
     * @stability stable
     */
    addToGroup(group: IGroup): void;
}
/**
 * Properties for defining an IAM user.
 *
 * @stability stable
 */
export interface UserProps {
    /**
     * Groups to add this user to.
     *
     * You can also use `addToGroup` to add this
     * user to a group.
     *
     * @default - No groups.
     * @stability stable
     */
    readonly groups?: IGroup[];
    /**
     * A list of managed policies associated with this role.
     *
     * You can add managed policies later using
     * `addManagedPolicy(ManagedPolicy.fromAwsManagedPolicyName(policyName))`.
     *
     * @default - No managed policies.
     * @stability stable
     */
    readonly managedPolicies?: IManagedPolicy[];
    /**
     * The path for the user name.
     *
     * For more information about paths, see IAM
     * Identifiers in the IAM User Guide.
     *
     * @default /
     * @stability stable
     */
    readonly path?: string;
    /**
     * AWS supports permissions boundaries for IAM entities (users or roles).
     *
     * A permissions boundary is an advanced feature for using a managed policy
     * to set the maximum permissions that an identity-based policy can grant to
     * an IAM entity. An entity's permissions boundary allows it to perform only
     * the actions that are allowed by both its identity-based policies and its
     * permissions boundaries.
     *
     * @default - No permissions boundary.
     * @stability stable
     * @link https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html
     */
    readonly permissionsBoundary?: IManagedPolicy;
    /**
     * A name for the IAM user.
     *
     * For valid values, see the UserName parameter for
     * the CreateUser action in the IAM API Reference. If you don't specify a
     * name, AWS CloudFormation generates a unique physical ID and uses that ID
     * for the user name.
     *
     * If you specify a name, you cannot perform updates that require
     * replacement of this resource. You can perform updates that require no or
     * some interruption. If you must replace the resource, specify a new name.
     *
     * If you specify a name, you must specify the CAPABILITY_NAMED_IAM value to
     * acknowledge your template's capabilities. For more information, see
     * Acknowledging IAM Resources in AWS CloudFormation Templates.
     *
     * @default - Generated by CloudFormation (recommended)
     * @stability stable
     */
    readonly userName?: string;
    /**
     * The password for the user. This is required so the user can access the AWS Management Console.
     *
     * You can use `SecretValue.plainText` to specify a password in plain text or
     * use `secretsmanager.Secret.fromSecretAttributes` to reference a secret in
     * Secrets Manager.
     *
     * @default - User won't be able to access the management console without a password.
     * @stability stable
     */
    readonly password?: SecretValue;
    /**
     * Specifies whether the user is required to set a new password the next time the user logs in to the AWS Management Console.
     *
     * If this is set to 'true', you must also specify "initialPassword".
     *
     * @default false
     * @stability stable
     */
    readonly passwordResetRequired?: boolean;
}
/**
 * Define a new IAM user.
 *
 * @stability stable
 */
export declare class User extends Resource implements IIdentity, IUser {
    /**
     * Import an existing user given a username.
     *
     * @param scope construct scope.
     * @param id construct id.
     * @param userName the username of the existing user to import.
     * @stability stable
     */
    static fromUserName(scope: Construct, id: string, userName: string): IUser;
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The AWS account ID of this principal.
     *
     * Can be undefined when the account is not known
     * (for example, for service principals).
     * Can be a Token - in that case,
     * it's assumed to be AWS::AccountId.
     *
     * @stability stable
     */
    readonly principalAccount: string | undefined;
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     *
     * @stability stable
     */
    readonly assumeRoleAction: string;
    /**
     * An attribute that represents the user name.
     *
     * @stability stable
     * @attribute true
     */
    readonly userName: string;
    /**
     * An attribute that represents the user's ARN.
     *
     * @stability stable
     * @attribute true
     */
    readonly userArn: string;
    /**
     * Returns the permissions boundary attached to this user.
     *
     * @stability stable
     */
    readonly permissionsBoundary?: IManagedPolicy;
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     *
     * @stability stable
     */
    readonly policyFragment: PrincipalPolicyFragment;
    private readonly groups;
    private readonly managedPolicies;
    private readonly attachedPolicies;
    private defaultPolicy?;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: UserProps);
    /**
     * Adds this user to a group.
     *
     * @stability stable
     */
    addToGroup(group: IGroup): void;
    /**
     * Attaches a managed policy to the user.
     *
     * @param policy The managed policy to attach.
     * @stability stable
     */
    addManagedPolicy(policy: IManagedPolicy): void;
    /**
     * Attaches a policy to this user.
     *
     * @stability stable
     */
    attachInlinePolicy(policy: Policy): void;
    /**
     * Adds an IAM statement to the default policy.
     *
     * @returns true
     * @stability stable
     */
    addToPrincipalPolicy(statement: PolicyStatement): AddToPrincipalPolicyResult;
    /**
     * Add to the policy of this principal.
     *
     * @stability stable
     */
    addToPolicy(statement: PolicyStatement): boolean;
    private parseLoginProfile;
}
