"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Certbot = exports.CertificateStorageType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const oneTimeEvents = require("@renovosolutions/cdk-library-one-time-event");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const required_policies_1 = require("./required-policies");
const storage_helpers_1 = require("./storage-helpers");
var CertificateStorageType;
(function (CertificateStorageType) {
    /**
     * Store the certificate in AWS Secrets Manager
     */
    CertificateStorageType["SECRETS_MANAGER"] = "secretsmanager";
    /**
     * Store the certificates in S3
     */
    CertificateStorageType["S3"] = "s3";
    /**
     * Store the certificates as a parameter in AWS Systems Manager Parameter Store  with encryption
     */
    CertificateStorageType["SSM_SECURE"] = "ssm_secure";
    /**
     * Store the certificates in EFS, mounted to the Lambda function
     */
    CertificateStorageType["EFS"] = "efs";
})(CertificateStorageType = exports.CertificateStorageType || (exports.CertificateStorageType = {}));
class Certbot extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Set property defaults
        let layers = props.layers ?? [];
        let runOnDeploy = props.runOnDeploy ?? true;
        let functionDescription = props.functionDescription ?? 'Certbot Renewal Lambda for domain ' + props.letsencryptDomains.split(',')[0];
        let enableInsights = props.enableInsights ?? false;
        let insightsARN = props.insightsARN ?? 'arn:aws:lambda:' + aws_cdk_lib_1.Stack.of(this).region + ':580247275435:layer:LambdaInsightsExtension:14';
        if (props.hostedZoneNames === undefined && props.hostedZones === undefined) {
            throw new Error('You must provide either hostedZoneNames or hostedZones');
        }
        // Create an SNS topic if one is not provided and add the defined email to it
        let snsTopic = props.snsTopic ?? new aws_cdk_lib_1.aws_sns.Topic(this, 'topic');
        if (props.snsTopic === undefined) {
            snsTopic.addSubscription(new aws_cdk_lib_1.aws_sns_subscriptions.EmailSubscription(props.letsencryptEmail));
        }
        let hostedZones = [];
        if (props.hostedZoneNames != undefined) {
            props.hostedZoneNames.forEach((domainName) => {
                hostedZones.push(aws_cdk_lib_1.aws_route53.HostedZone.fromLookup(this, 'zone' + domainName, {
                    domainName,
                    privateZone: false,
                }).hostedZoneArn);
            });
        }
        if (props.hostedZones != undefined) {
            props.hostedZones.forEach((hostedZone) => {
                hostedZones.push(hostedZone.hostedZoneArn);
            });
        }
        const role = new aws_cdk_lib_1.aws_iam.Role(this, 'role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        required_policies_1.assignRequiredPoliciesToRole(this, {
            role,
            snsTopic,
            hostedZones,
        });
        const functionDir = path.join(__dirname, '../function/src');
        const bundlingCmds = [
            'mkdir -p /asset-output',
            'pip install -r /asset-input/requirements.txt -t /asset-output',
            'cp index.py /asset-output/index.py',
        ];
        // Create the Lambda function
        this.handler = new aws_cdk_lib_1.aws_lambda.Function(this, 'handler', {
            runtime: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_10,
            role,
            architecture: props.architecture || aws_cdk_lib_1.aws_lambda.Architecture.X86_64,
            code: aws_cdk_lib_1.aws_lambda.Code.fromAsset(functionDir, {
                bundling: {
                    image: aws_cdk_lib_1.aws_lambda.Runtime.PYTHON_3_10.bundlingImage,
                    command: [
                        'bash', '-c', bundlingCmds.join(' && '),
                    ],
                },
            }),
            handler: 'index.handler',
            functionName: props.functionName,
            description: functionDescription,
            environment: {
                LETSENCRYPT_DOMAINS: props.letsencryptDomains,
                LETSENCRYPT_EMAIL: props.letsencryptEmail,
                OBJECT_PREFIX: props.objectPrefix || '',
                REISSUE_DAYS: (props.reIssueDays === undefined) ? '30' : String(props.reIssueDays),
                PREFERRED_CHAIN: props.preferredChain || 'None',
                KEY_TYPE: props.keyType || 'ecdsa',
                NOTIFICATION_SNS_ARN: snsTopic.topicArn,
                DRY_RUN: 'False',
            },
            layers,
            timeout: props.timeout || aws_cdk_lib_1.Duration.seconds(180),
            filesystem: props.efsAccessPoint ? aws_cdk_lib_1.aws_lambda.FileSystem.fromEfsAccessPoint(props.efsAccessPoint, '/mnt/efs') : undefined,
            vpc: props.vpc,
        });
        let bucket;
        if (props.bucket === undefined && (props.certificateStorage == CertificateStorageType.S3 || props.certificateStorage == undefined)) {
            bucket = new aws_cdk_lib_1.aws_s3.Bucket(this, 'bucket', {
                objectOwnership: aws_cdk_lib_1.aws_s3.ObjectOwnership.BUCKET_OWNER_PREFERRED,
                removalPolicy: props.removalPolicy || aws_cdk_lib_1.RemovalPolicy.RETAIN,
                autoDeleteObjects: props.enableObjectDeletion ?? false,
                versioned: true,
                lifecycleRules: [{
                        enabled: true,
                        abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1),
                    }],
                encryption: aws_cdk_lib_1.aws_s3.BucketEncryption.S3_MANAGED,
                enforceSSL: true,
                blockPublicAccess: aws_cdk_lib_1.aws_s3.BlockPublicAccess.BLOCK_ALL,
            });
            bucket.grantReadWrite(this.handler);
            this.handler.addEnvironment('CERTIFICATE_BUCKET', bucket.bucketName);
            this.handler.addEnvironment('CERTIFICATE_STORAGE', 's3');
        }
        if (props.bucket && (props.certificateStorage == CertificateStorageType.S3 || props.certificateStorage == undefined)) {
            bucket = props.bucket;
            bucket.grantReadWrite(this.handler);
            this.handler.addEnvironment('CERTIFICATE_BUCKET', bucket.bucketName);
            this.handler.addEnvironment('CERTIFICATE_STORAGE', 's3');
        }
        if (props.certificateStorage == CertificateStorageType.SECRETS_MANAGER) {
            this.handler.addEnvironment('CERTIFICATE_STORAGE', 'secretsmanager');
            this.handler.addEnvironment('CERTIFICATE_SECRET_PATH', props.secretsManagerPath || `/certbot/certificates/${props.letsencryptDomains.split(',')[0]}/`);
            if (props.kmsKeyAlias) {
                this.handler.addEnvironment('CUSTOM_KMS_KEY_ID', props.kmsKeyAlias);
            }
            storage_helpers_1.configureSecretsManagerStorage(this, {
                role,
                secretsManagerPath: props.secretsManagerPath || `/certbot/certificates/${props.letsencryptDomains.split(',')[0]}/`,
                kmsKeyAlias: props.kmsKeyAlias,
            });
        }
        ;
        if (props.certificateStorage == CertificateStorageType.SSM_SECURE) {
            this.handler.addEnvironment('CERTIFICATE_STORAGE', 'ssm_secure');
            this.handler.addEnvironment('CERTIFICATE_PARAMETER_PATH', props.ssmSecurePath || `/certbot/certificates/${props.letsencryptDomains.split(',')[0]}/`);
            if (props.kmsKeyAlias) {
                this.handler.addEnvironment('CUSTOM_KMS_KEY_ID', props.kmsKeyAlias);
            }
            storage_helpers_1.configureSSMStorage(this, {
                role,
                parameterStorePath: props.ssmSecurePath || `/certbot/certificates/${props.letsencryptDomains.split(',')[0]}/`,
                kmsKeyAlias: props.kmsKeyAlias,
            });
        }
        if (props.certificateStorage == CertificateStorageType.EFS) {
            if (!props.efsAccessPoint) {
                throw new Error('You must provide an EFS Access Point to use EFS storage');
            }
            else {
                this.handler.addEnvironment('CERTIFICATE_STORAGE', 'efs');
                this.handler.addEnvironment('EFS_PATH', '/mnt/efs');
            }
        }
        if (props.vpc) {
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        if (enableInsights) {
            role.addManagedPolicy(aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('CloudWatchLambdaInsightsExecutionRolePolicy'));
            this.handler.addLayers(aws_cdk_lib_1.aws_lambda.LayerVersion.fromLayerVersionArn(this, 'insightsLayer', insightsARN));
        }
        // Add function triggers
        new aws_cdk_lib_1.aws_events.Rule(this, 'trigger', {
            schedule: props.schedule || aws_cdk_lib_1.aws_events.Schedule.cron({ minute: '0', hour: '0', weekDay: '1' }),
            targets: [new aws_cdk_lib_1.aws_events_targets.LambdaFunction(this.handler)],
        });
        if (runOnDeploy) {
            new aws_cdk_lib_1.aws_events.Rule(this, 'triggerImmediate', {
                schedule: new oneTimeEvents.OnDeploy(this, 'schedule', {
                    offsetMinutes: props.runOnDeployWaitMinutes || 10,
                }).schedule,
                targets: [new aws_cdk_lib_1.aws_events_targets.LambdaFunction(this.handler)],
            });
        }
    }
}
exports.Certbot = Certbot;
_a = JSII_RTTI_SYMBOL_1;
Certbot[_a] = { fqn: "@renovosolutions/cdk-library-certbot.Certbot", version: "2.8.0" };
//# sourceMappingURL=data:application/json;base64,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