from .ufostroker import constant_width_stroke as cws_rust
from ufo2ft.filters import BaseFilter
from fontTools.misc.roundTools import otRound
from beziers.cubicbezier import CubicBezier
from beziers.point import Point


def constant_width_stroke(
    glyph,
    width,
    startcap="round",
    endcap="round",
    jointype="bevel",
    remove_internal=False,
    remove_external=False,
):
    """Applies a constant-width stroke effect to a glyph, in place.

    Parameters:
        glyph: A ufoLib2 or defcon glyph object
        width: The stroke width, in points.
        startcap: Cap to add at the start of the stroke (One of: "round", "square")
        endcap: Cap to add at the end of the stroke (One of: "round", "square")
        jointype: Joining type (One of: "round", "bevel", "mitre")
        remove_internal: Remove the internal path when stroking closed curves
        remove_external: Remove the external path when stroking closed curves

    Returns nothing, but modifies the glyph.
    """

    if not startcap in ["round", "square"]:
        raise ValueError("Unknown start cap type")
    if not endcap in ["round", "square"]:
        raise ValueError("Unknown end cap type")
    if not jointype in ["round", "bevel", "mitre"]:
        raise ValueError("Unknown join type")
    list_of_list_of_points = [list(c) for c in list(glyph)]
    res = cws_rust(
        list_of_list_of_points,
        width,
        startcap,
        endcap,
        jointype,
        remove_internal,
        remove_external,
    )
    contour_class = glyph[0].__class__
    point_class = glyph[0][0].__class__
    contours = []
    glyph.clearContours()
    for contour in res:
        points = []
        for pt in contour:
            x, y, typ = pt
            if not typ:
                typ = None
            # Unfortunately defcon and ufoLib2 have different Point constructors...
            try:
                point = point_class(x, y, typ)
            except Exception:
                point = point_class((x, y), typ)
            points.append(point)
        contour = contour_class()
        # And contour constructors...
        try:
            contour.extend(points)
        except Exception:
            for point in points:
                contour.appendPoint(point)
        glyph.appendContour(contour)


class StrokeFilter(BaseFilter):

    _kwargs = {
        "Width": 10,
        "StartCap": "round",
        "EndCap": "round",
        "JoinType": "bevel",
        "RemoveInternal": False,
        "RemoveExternal": False,
    }

    def filter(self, glyph):
        if not len(glyph):
            return False

        constant_width_stroke(
            glyph,
            self.options.Width,
            startcap=self.options.StartCap,
            endcap=self.options.StartCap,
            jointype=self.options.JoinType,
            remove_external=self.options.RemoveExternal,
            remove_internal=self.options.RemoveInternal,
        )

        # We have to tunnify it...
        for contour in glyph:
            for i in range(0, len(contour)):
                i1 = (i + 1) % len(contour)
                i2 = (i + 2) % len(contour)
                i3 = (i + 3) % len(contour)
                if (
                    contour[i].segmentType
                    and not contour[i1].segmentType
                    and not contour[i2].segmentType
                    and contour[i3].segmentType
                ):
                    cbez = CubicBezier(
                        Point(contour[i].x, contour[i].y),
                        Point(contour[i1].x, contour[i1].y),
                        Point(contour[i2].x, contour[i2].y),
                        Point(contour[i3].x, contour[i3].y),
                    )
                    cbez.balance()
                    contour[i1].x = cbez[1].x
                    contour[i1].y = cbez[1].y
                    contour[i2].x = cbez[2].x
                    contour[i2].y = cbez[2].y
                    pass
        return True
