"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addForwarder = exports.SUBSCRIPTION_FILTER_PREFIX = void 0;
const crypto = require("crypto");
const lambda = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const loglevel_1 = require("loglevel");
exports.SUBSCRIPTION_FILTER_PREFIX = "DatadogSubscriptionFilter";
// Change back to 'import { LambdaDestination } from "@aws-cdk/aws-logs-destinations";'
// once https://github.com/aws/aws-cdk/pull/14222 is merged and released.
const lambdaDestination_1 = require("./lambdaDestination");
function generateForwaderConstructId(forwarderArn) {
    loglevel_1.default.debug("Generating construct Id for Datadog Lambda Forwarder");
    return "forwarder" + crypto.createHash("sha256").update(forwarderArn).digest("hex");
}
function generateSubscriptionFilterName(functionUniqueId, forwarderArn) {
    const subscriptionFilterValue = crypto
        .createHash("sha256")
        .update(functionUniqueId)
        .update(forwarderArn)
        .digest("hex");
    const subscriptionFilterValueLength = subscriptionFilterValue.length;
    const subscriptionFilterName = exports.SUBSCRIPTION_FILTER_PREFIX +
        subscriptionFilterValue.substring(subscriptionFilterValueLength - 8, subscriptionFilterValueLength);
    return subscriptionFilterName;
}
function addForwarder(scope, lambdaFunctions, forwarderArn) {
    const forwarderConstructId = generateForwaderConstructId(forwarderArn);
    let forwarder;
    if (scope.node.tryFindChild(forwarderConstructId)) {
        forwarder = scope.node.tryFindChild(forwarderConstructId);
    }
    else {
        forwarder = lambda.Function.fromFunctionArn(scope, forwarderConstructId, forwarderArn);
    }
    const forwarderDestination = new lambdaDestination_1.LambdaDestination(forwarder);
    lambdaFunctions.forEach((lam) => {
        const subscriptionFilterName = generateSubscriptionFilterName(cdk.Names.uniqueId(lam), forwarderArn);
        loglevel_1.default.debug(`Adding log subscription ${subscriptionFilterName} for ${lam.functionName}`);
        lam.logGroup.addSubscriptionFilter(subscriptionFilterName, {
            destination: forwarderDestination,
            filterPattern: aws_logs_1.FilterPattern.allEvents(),
        });
    });
}
exports.addForwarder = addForwarder;
//# sourceMappingURL=data:application/json;base64,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