import argparse
from git import Repo

def find_md5_for_path(projectdir, artifact_path):
    """
    searches the content of dvc.lock or the `[artifact_path].dvc` file for the md5 has of a given artifact path
    """
    import yaml
    from pathlib import Path

    try:
        dvclock = yaml.safe_load(open(Path(projectdir)/'dvc.lock'))

        for stage in dvclock.get('stages', {}).values():
            for entry in stage.get('outs', []) + stage.get('deps', []):
                if entry.get('path') == artifact_path:
                    return entry.get('md5')

    except FileNotFoundError:

        dvcfile = yaml.safe_load(open(Path(projectdir)/f'{artifact_path}.dvc'))

        for entry in dvcfile.get('outs', []):
                if entry.get('path') == artifact_path:
                    return entry.get('md5')

        raise ValueError(f"There seems to be no DVC-versioned file with the path {artifact_path}.")


def find_artifact_path(dvcyaml, artifact_name):
    artifacts = dvcyaml.get('artifacts', {})
    artifact = artifacts.get(artifact_name)
    if artifact:
        return artifact.get('path')
    return None

def get_gto_tag(repourl, projectdir, artifact_name, rev=None):
    """
    Gets the Git tag generated by GTO for the given artifact. If rev is not provided, gets the latest version. 
    """
    import subprocess

    if not rev:
        rev = "greatest"

    cmd = ["gto", "show", f"{projectdir}:"+artifact_name+f"@{rev}" ,"--ref", "--repo", repourl]
    ref = subprocess.run(cmd, capture_output=True, text=True).stdout

    if ref:
        return ref.strip() #removing newline at the end
    elif rev == "greatest":
        raise ValueError(f"No registered version found for artifact '{artifact_name}'!")
    else:
        raise ValueError(f"Registered version {rev} was not found for artifact '{artifact_name}'!")

def download_artifact(repourl, projectdir, artifact_name, rev=None):
    import os
    import yaml
    import tempfile
    import configparser
    from pathlib import Path

    with tempfile.TemporaryDirectory() as temp_dir:

        original_dir = os.getcwd()
        os.chdir(temp_dir)

        gto_tag = get_gto_tag(repourl, projectdir, artifact_name, rev)   

        Repo.clone_from(repourl, '.', filter='blob:none', sparse=True, depth=1, branch=gto_tag)
        repo = Repo('.')
        repo.git.sparse_checkout('init', '--cone')
        repo.git.sparse_checkout('set', projectdir)

        dvcyaml = yaml.safe_load(open(Path(projectdir)/'dvc.yaml'))
        artifact_path = find_artifact_path(dvcyaml, artifact_name)

        md5 = find_md5_for_path(projectdir, artifact_path)

        dvcconfig = configparser.ConfigParser()
        dvcconfig.read(str(Path(projectdir)/'.dvc'/'config'))

    os.chdir(original_dir)

    remoteurl = dvcconfig[f"""'remote "{dvcconfig['core'].get('remote')}"'"""].get('url')

    # TODO: handle the situation where the remote is a subdirectory in the bucket
    storage = remoteurl.split('://')[0]
    bucket_name = remoteurl.split('://')[1]

    object_key = f"files/md5/{md5[0:2]}/{md5[2:]}"

    if storage=='gs':
        from google.cloud import storage
        storage_client = storage.Client()
        bucket = storage_client.bucket(bucket_name)
        blob = bucket.blob(object_key)
        blob.download_to_filename(Path(artifact_path).name)

    elif storage=='s3':
        import boto3    
        s3 = boto3.client('s3')
        s3.download_file(bucket_name, object_key, Path(artifact_path).name)
    else:
        print(f"Error: Remote storage type {storage} not implemented!")

def main():
    parser = argparse.ArgumentParser(description='Download an artifact from the remote bucket')
    parser.add_argument('repourl', type=str, help='url of the GitHub repository associated with the artifact')
    parser.add_argument('projectdir', type=str, help='project subdirectory in the monorepo where the artifact was created')
    parser.add_argument('artifact_name', type=str, help='Name of the artifact to find')
    parser.add_argument('-r', '--rev', type=str, help='semantic version of the artifact (optional), latest version is used if this is not specified', default=None)


    args = vars(parser.parse_args())
    download_artifact(**args)


if __name__ == '__main__':
    main()