"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return false;
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    /**
     * Returns a new PrincipalWithConditions using this principal as the base, with the
     * passed conditions added.
     *
     * When there is a value for the same operator and key in both the principal and the
     * conditions parameter, the value from the conditions parameter will be used.
     *
     * @returns a new PrincipalWithConditions object.
     */
    withConditions(conditions) {
        return new PrincipalWithConditions(this, conditions);
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * An IAM principal with additional conditions specifying when the policy is in effect.
 *
 * For more information about conditions, see:
 * https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html
 */
class PrincipalWithConditions {
    constructor(principal, conditions) {
        this.principal = principal;
        this.grantPrincipal = this;
        this.assumeRoleAction = this.principal.assumeRoleAction;
        this.additionalConditions = conditions;
    }
    /**
     * Add a condition to the principal
     */
    addCondition(key, value) {
        const existingValue = this.conditions[key];
        this.conditions[key] = existingValue ? { ...existingValue, ...value } : value;
    }
    /**
     * Adds multiple conditions to the principal
     *
     * Values from the conditions parameter will overwrite existing values with the same operator
     * and key.
     */
    addConditions(conditions) {
        Object.entries(conditions).forEach(([key, value]) => {
            this.addCondition(key, value);
        });
    }
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    get conditions() {
        return this.mergeConditions(this.principal.policyFragment.conditions, this.additionalConditions);
    }
    get policyFragment() {
        return new PrincipalPolicyFragment(this.principal.policyFragment.principalJson, this.conditions);
    }
    addToPolicy(statement) {
        return this.principal.addToPolicy(statement);
    }
    toString() {
        return this.principal.toString();
    }
    /**
     * JSON-ify the principal
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
    mergeConditions(principalConditions, additionalConditions) {
        const mergedConditions = {};
        Object.entries(principalConditions).forEach(([operator, condition]) => {
            mergedConditions[operator] = condition;
        });
        Object.entries(additionalConditions).forEach(([operator, condition]) => {
            mergedConditions[operator] = { ...mergedConditions[operator], ...condition };
        });
        return mergedConditions;
    }
}
exports.PrincipalWithConditions = PrincipalWithConditions;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    constructor(principalJson, 
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
class ArnPrincipal extends PrincipalBase {
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
class AccountPrincipal extends ArnPrincipal {
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString()
            ]
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
class FederatedPrincipal extends PrincipalBase {
    constructor(federated, 
    /**
     * The conditions under which the policy is in effect.
     * See [the IAM documentation](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition.html).
     */
    conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    toString() {
        return 'AccountRootPrincipal()';
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return 'AnyPrincipal()';
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error('Cannot add multiple principals with different "assumeRoleAction". ' +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error('Components of a CompositePrincipal must not have conditions. ' +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        return this.fn(cdk.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return '<unresolved-token>';
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || cdk.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, cdk.Aws.URL_SUFFIX);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service
        });
    }
    /**
     * JSON-ify the token
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJpbmNpcGFscy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInByaW5jaXBhbHMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxxQ0FBcUM7QUFDckMsc0RBQTJEO0FBRTNELGlDQUF3QztBQWlEeEM7O0dBRUc7QUFDSCxNQUFzQixhQUFhO0lBQW5DO1FBQ2tCLG1CQUFjLEdBQWUsSUFBSSxDQUFDO1FBT2xEOztXQUVHO1FBQ2EscUJBQWdCLEdBQVcsZ0JBQWdCLENBQUM7SUFvQzlELENBQUM7SUFsQ1EsV0FBVyxDQUFDLFVBQTJCO1FBQzVDLG9FQUFvRTtRQUNwRSxtQ0FBbUM7UUFDbkMsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRU0sUUFBUTtRQUNiLDBFQUEwRTtRQUMxRSxpQ0FBaUM7UUFDakMsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDM0QsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxNQUFNO1FBQ1gsa0ZBQWtGO1FBQ2xGLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxhQUFhLENBQUM7SUFDM0MsQ0FBQztJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksY0FBYyxDQUFDLFVBQXNCO1FBQzFDLE9BQU8sSUFBSSx1QkFBdUIsQ0FBQyxJQUFJLEVBQUUsVUFBVSxDQUFDLENBQUM7SUFDdkQsQ0FBQztDQUNGO0FBL0NELHNDQStDQztBQUVEOzs7OztHQUtHO0FBQ0gsTUFBYSx1QkFBdUI7SUFLbEMsWUFDbUIsU0FBcUIsRUFDdEMsVUFBc0I7UUFETCxjQUFTLEdBQVQsU0FBUyxDQUFZO1FBTHhCLG1CQUFjLEdBQWUsSUFBSSxDQUFDO1FBQ2xDLHFCQUFnQixHQUFXLElBQUksQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLENBQUM7UUFPekUsSUFBSSxDQUFDLG9CQUFvQixHQUFHLFVBQVUsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxZQUFZLENBQUMsR0FBVyxFQUFFLEtBQWdCO1FBQy9DLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDM0MsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsR0FBRyxhQUFhLENBQUMsQ0FBQyxDQUFDLEVBQUUsR0FBRyxhQUFhLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO0lBQ2hGLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGFBQWEsQ0FBQyxVQUFzQjtRQUN6QyxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxFQUFFLEVBQUU7WUFDbEQsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDaEMsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsSUFBVyxVQUFVO1FBQ25CLE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGNBQWMsQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLENBQUM7SUFDbkcsQ0FBQztJQUVELElBQVcsY0FBYztRQUN2QixPQUFPLElBQUksdUJBQXVCLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxjQUFjLENBQUMsYUFBYSxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNuRyxDQUFDO0lBRU0sV0FBVyxDQUFDLFNBQTBCO1FBQzNDLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVNLFFBQVE7UUFDYixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxNQUFNO1FBQ1gsa0ZBQWtGO1FBQ2xGLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxhQUFhLENBQUM7SUFDM0MsQ0FBQztJQUVPLGVBQWUsQ0FBQyxtQkFBK0IsRUFBRSxvQkFBZ0M7UUFDdkYsTUFBTSxnQkFBZ0IsR0FBZSxFQUFFLENBQUM7UUFDeEMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsUUFBUSxFQUFFLFNBQVMsQ0FBQyxFQUFFLEVBQUU7WUFDcEUsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLEdBQUcsU0FBUyxDQUFDO1FBQ3pDLENBQUMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsUUFBUSxFQUFFLFNBQVMsQ0FBQyxFQUFFLEVBQUU7WUFDckUsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLEdBQUcsRUFBRSxHQUFHLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxFQUFFLEdBQUcsU0FBUyxFQUFFLENBQUM7UUFDL0UsQ0FBQyxDQUFDLENBQUM7UUFDSCxPQUFPLGdCQUFnQixDQUFDO0lBQzFCLENBQUM7Q0FDRjtBQXhFRCwwREF3RUM7QUFFRDs7Ozs7R0FLRztBQUNILE1BQWEsdUJBQXVCO0lBQ2xDLFlBQ2tCLGFBQTBDO0lBQzFEOzs7T0FHRztJQUNhLGFBQXlCLEVBQUU7UUFMM0Isa0JBQWEsR0FBYixhQUFhLENBQTZCO1FBSzFDLGVBQVUsR0FBVixVQUFVLENBQWlCO0lBQzdDLENBQUM7Q0FDRjtBQVRELDBEQVNDO0FBRUQsTUFBYSxZQUFhLFNBQVEsYUFBYTtJQUM3QyxZQUE0QixHQUFXO1FBQ3JDLEtBQUssRUFBRSxDQUFDO1FBRGtCLFFBQUcsR0FBSCxHQUFHLENBQVE7SUFFdkMsQ0FBQztJQUVELElBQVcsY0FBYztRQUN2QixPQUFPLElBQUksdUJBQXVCLENBQUMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFTSxRQUFRO1FBQ2IsT0FBTyxnQkFBZ0IsSUFBSSxDQUFDLEdBQUcsR0FBRyxDQUFDO0lBQ3JDLENBQUM7Q0FDRjtBQVpELG9DQVlDO0FBRUQsTUFBYSxnQkFBaUIsU0FBUSxZQUFZO0lBQ2hELFlBQTRCLFNBQWM7UUFDeEMsS0FBSyxDQUFDLElBQUksbUJBQW1CLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxPQUFPLEtBQUssQ0FBQyxTQUFTLFNBQVMsU0FBUyxPQUFPLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBRDFFLGNBQVMsR0FBVCxTQUFTLENBQUs7SUFFMUMsQ0FBQztJQUVNLFFBQVE7UUFDYixPQUFPLG9CQUFvQixJQUFJLENBQUMsU0FBUyxHQUFHLENBQUM7SUFDL0MsQ0FBQztDQUNGO0FBUkQsNENBUUM7QUFxQkQ7O0dBRUc7QUFDSCxNQUFhLGdCQUFpQixTQUFRLGFBQWE7SUFDakQsWUFBNEIsT0FBZSxFQUFtQixPQUE2QixFQUFFO1FBQzNGLEtBQUssRUFBRSxDQUFDO1FBRGtCLFlBQU8sR0FBUCxPQUFPLENBQVE7UUFBbUIsU0FBSSxHQUFKLElBQUksQ0FBMkI7SUFFN0YsQ0FBQztJQUVELElBQVcsY0FBYztRQUN2QixPQUFPLElBQUksdUJBQXVCLENBQUM7WUFDakMsT0FBTyxFQUFFO2dCQUNQLElBQUkscUJBQXFCLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxFQUFFO2FBQzlEO1NBQ0YsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFTSxRQUFRO1FBQ2IsT0FBTyxvQkFBb0IsSUFBSSxDQUFDLE9BQU8sR0FBRyxDQUFDO0lBQzdDLENBQUM7Q0FDRjtBQWhCRCw0Q0FnQkM7QUFFRDs7R0FFRztBQUNILE1BQWEscUJBQXNCLFNBQVEsYUFBYTtJQUN0RCxZQUE0QixjQUFzQjtRQUNoRCxLQUFLLEVBQUUsQ0FBQztRQURrQixtQkFBYyxHQUFkLGNBQWMsQ0FBUTtJQUVsRCxDQUFDO0lBRUQsSUFBVyxjQUFjO1FBQ3ZCLE9BQU8sSUFBSSx1QkFBdUIsQ0FDaEMsRUFBRSxHQUFHLEVBQUUsQ0FBQyxHQUFHLENBQUMsRUFBRSxFQUNkLEVBQUUsWUFBWSxFQUFFLEVBQUUsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLGNBQWMsRUFBRSxFQUFFLENBQ2hFLENBQUM7SUFDSixDQUFDO0lBRU0sUUFBUTtRQUNiLE9BQU8seUJBQXlCLElBQUksQ0FBQyxjQUFjLEdBQUcsQ0FBQztJQUN6RCxDQUFDO0NBQ0Y7QUFmRCxzREFlQztBQUVEOzs7Ozs7Ozs7Ozs7R0FZRztBQUNILE1BQWEsc0JBQXVCLFNBQVEsYUFBYTtJQUN2RCxZQUE0QixlQUF1QjtRQUNqRCxLQUFLLEVBQUUsQ0FBQztRQURrQixvQkFBZSxHQUFmLGVBQWUsQ0FBUTtJQUVuRCxDQUFDO0lBRUQsSUFBVyxjQUFjO1FBQ3ZCLE9BQU8sSUFBSSx1QkFBdUIsQ0FBQyxFQUFFLGFBQWEsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDaEYsQ0FBQztJQUVNLFFBQVE7UUFDYixPQUFPLDBCQUEwQixJQUFJLENBQUMsZUFBZSxHQUFHLENBQUM7SUFDM0QsQ0FBQztDQUNGO0FBWkQsd0RBWUM7QUFFRCxNQUFhLGtCQUFtQixTQUFRLGFBQWE7SUFHbkQsWUFDa0IsU0FBaUI7SUFDakM7OztPQUdHO0lBQ2EsVUFBc0IsRUFDdEMsbUJBQTJCLGdCQUFnQjtRQUMzQyxLQUFLLEVBQUUsQ0FBQztRQVBRLGNBQVMsR0FBVCxTQUFTLENBQVE7UUFLakIsZUFBVSxHQUFWLFVBQVUsQ0FBWTtRQUl0QyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsZ0JBQWdCLENBQUM7SUFDM0MsQ0FBQztJQUVELElBQVcsY0FBYztRQUN2QixPQUFPLElBQUksdUJBQXVCLENBQUMsRUFBRSxTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUUsRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDdkYsQ0FBQztJQUVNLFFBQVE7UUFDYixPQUFPLHNCQUFzQixJQUFJLENBQUMsU0FBUyxHQUFHLENBQUM7SUFDakQsQ0FBQztDQUNGO0FBdkJELGdEQXVCQztBQUVELE1BQWEsb0JBQXFCLFNBQVEsZ0JBQWdCO0lBQ3hEO1FBQ0UsS0FBSyxDQUFDLElBQUksbUJBQW1CLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRU0sUUFBUTtRQUNiLE9BQU8sd0JBQXdCLENBQUM7SUFDbEMsQ0FBQztDQUNGO0FBUkQsb0RBUUM7QUFFRDs7R0FFRztBQUNILE1BQWEsWUFBYSxTQUFRLFlBQVk7SUFDNUM7UUFDRSxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDYixDQUFDO0lBRU0sUUFBUTtRQUNiLE9BQU8sZ0JBQWdCLENBQUM7SUFDMUIsQ0FBQztDQUNGO0FBUkQsb0NBUUM7QUFFRDs7O0dBR0c7QUFDSCxNQUFhLE1BQU8sU0FBUSxZQUFZO0NBQUk7QUFBNUMsd0JBQTRDO0FBRTVDLE1BQWEsa0JBQW1CLFNBQVEsYUFBYTtJQUluRCxZQUFZLEdBQUcsVUFBMkI7UUFDeEMsS0FBSyxFQUFFLENBQUM7UUFITyxlQUFVLEdBQUcsSUFBSSxLQUFLLEVBQWlCLENBQUM7UUFJdkQsSUFBSSxVQUFVLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMzQixNQUFNLElBQUksS0FBSyxDQUFDLHlGQUF5RixDQUFDLENBQUM7U0FDNUc7UUFDRCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDO1FBQ3ZELElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxVQUFVLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRU0sYUFBYSxDQUFDLEdBQUcsVUFBMkI7UUFDakQsS0FBSyxNQUFNLENBQUMsSUFBSSxVQUFVLEVBQUU7WUFDMUIsSUFBSSxDQUFDLENBQUMsZ0JBQWdCLEtBQUssSUFBSSxDQUFDLGdCQUFnQixFQUFFO2dCQUNoRCxNQUFNLElBQUksS0FBSyxDQUNiLG9FQUFvRTtvQkFDcEUsY0FBYyxJQUFJLENBQUMsZ0JBQWdCLFdBQVcsQ0FBQyxDQUFDLGdCQUFnQixHQUFHLENBQUMsQ0FBQzthQUN4RTtZQUVELE1BQU0sUUFBUSxHQUFHLENBQUMsQ0FBQyxjQUFjLENBQUM7WUFDbEMsSUFBSSxRQUFRLENBQUMsVUFBVSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQ3RFLE1BQU0sSUFBSSxLQUFLLENBQ2IsK0RBQStEO29CQUMvRCx3Q0FBd0MsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUM7YUFDdkU7WUFFRCxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUN6QjtRQUVELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVELElBQVcsY0FBYztRQUN2QixNQUFNLGFBQWEsR0FBZ0MsRUFBRSxDQUFDO1FBRXRELEtBQUssTUFBTSxDQUFDLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUMvQixxQkFBYyxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsY0FBYyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQy9EO1FBRUQsT0FBTyxJQUFJLHVCQUF1QixDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ3BELENBQUM7SUFFTSxRQUFRO1FBQ2IsT0FBTyxzQkFBc0IsSUFBSSxDQUFDLFVBQVUsR0FBRyxDQUFDO0lBQ2xELENBQUM7Q0FDRjtBQS9DRCxnREErQ0M7QUFFRDs7R0FFRztBQUNILE1BQU0sbUJBQW1CO0lBRXZCLFlBQTZCLEVBQTZCO1FBQTdCLE9BQUUsR0FBRixFQUFFLENBQTJCO1FBQ3hELElBQUksQ0FBQyxhQUFhLEdBQUcsR0FBRyxDQUFDLGlCQUFpQixFQUFFLENBQUM7SUFDL0MsQ0FBQztJQUVNLE9BQU8sQ0FBQyxPQUE0QjtRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVNLFFBQVE7UUFDYixPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTTtRQUNYLE9BQU8sb0JBQW9CLENBQUM7SUFDOUIsQ0FBQztDQUNGO0FBRUQsTUFBTSxxQkFBcUI7SUFFekIsWUFDbUIsT0FBZSxFQUNmLElBQTBCO1FBRDFCLFlBQU8sR0FBUCxPQUFPLENBQVE7UUFDZixTQUFJLEdBQUosSUFBSSxDQUFzQjtRQUMzQyxJQUFJLENBQUMsYUFBYSxHQUFHLEdBQUcsQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO0lBQy9DLENBQUM7SUFFTSxPQUFPLENBQUMsR0FBd0I7UUFDckMsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQztRQUNsRSxNQUFNLElBQUksR0FBRyx3QkFBVSxDQUFDLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDbkUsT0FBTyxJQUFJLElBQUkscUJBQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLE1BQU0sRUFBRSxHQUFHLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ3BGLENBQUM7SUFFTSxRQUFRO1FBQ2IsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUU7WUFDOUIsV0FBVyxFQUFFLElBQUksQ0FBQyxPQUFPO1NBQzFCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTTtRQUNYLE9BQU8sSUFBSSxJQUFJLENBQUMsT0FBTyxHQUFHLENBQUM7SUFDN0IsQ0FBQztDQUNGIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgRGVmYXVsdCwgUmVnaW9uSW5mbyB9IGZyb20gJ0Bhd3MtY2RrL3JlZ2lvbi1pbmZvJztcbmltcG9ydCB7IENvbmRpdGlvbiwgQ29uZGl0aW9ucywgUG9saWN5U3RhdGVtZW50IH0gZnJvbSAnLi9wb2xpY3ktc3RhdGVtZW50JztcbmltcG9ydCB7IG1lcmdlUHJpbmNpcGFsIH0gZnJvbSAnLi91dGlsJztcblxuLyoqXG4gKiBBbnkgb2JqZWN0IHRoYXQgaGFzIGFuIGFzc29jaWF0ZWQgcHJpbmNpcGFsIHRoYXQgYSBwZXJtaXNzaW9uIGNhbiBiZSBncmFudGVkIHRvXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUdyYW50YWJsZSB7XG4gIC8qKlxuICAgKiBUaGUgcHJpbmNpcGFsIHRvIGdyYW50IHBlcm1pc3Npb25zIHRvXG4gICAqL1xuICByZWFkb25seSBncmFudFByaW5jaXBhbDogSVByaW5jaXBhbDtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgbG9naWNhbCBJQU0gcHJpbmNpcGFsLlxuICpcbiAqIEFuIElQcmluY2lwYWwgZGVzY3JpYmVzIGEgbG9naWNhbCBlbnRpdHkgdGhhdCBjYW4gcGVyZm9ybSBBV1MgQVBJIGNhbGxzXG4gKiBhZ2FpbnN0IHNldHMgb2YgcmVzb3VyY2VzLCBvcHRpb25hbGx5IHVuZGVyIGNlcnRhaW4gY29uZGl0aW9ucy5cbiAqXG4gKiBFeGFtcGxlcyBvZiBzaW1wbGUgcHJpbmNpcGFscyBhcmUgSUFNIG9iamVjdHMgdGhhdCB5b3UgY3JlYXRlLCBzdWNoXG4gKiBhcyBVc2VycyBvciBSb2xlcy5cbiAqXG4gKiBBbiBleGFtcGxlIG9mIGEgbW9yZSBjb21wbGV4IHByaW5jaXBhbHMgaXMgYSBgU2VydmljZVByaW5jaXBhbGAgKHN1Y2ggYXNcbiAqIGBuZXcgU2VydmljZVByaW5jaXBhbChcInNucy5hbWF6b25hd3MuY29tXCIpYCwgd2hpY2ggcmVwcmVzZW50cyB0aGUgU2ltcGxlXG4gKiBOb3RpZmljYXRpb25zIFNlcnZpY2UpLlxuICpcbiAqIEEgc2luZ2xlIGxvZ2ljYWwgUHJpbmNpcGFsIG1heSBhbHNvIG1hcCB0byBhIHNldCBvZiBwaHlzaWNhbCBwcmluY2lwYWxzLlxuICogRm9yIGV4YW1wbGUsIGBuZXcgT3JnYW5pemF0aW9uUHJpbmNpcGFsKCdvLTEyMzQnKWAgcmVwcmVzZW50cyBhbGxcbiAqIGlkZW50aXRpZXMgdGhhdCBhcmUgcGFydCBvZiB0aGUgZ2l2ZW4gQVdTIE9yZ2FuaXphdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJUHJpbmNpcGFsIGV4dGVuZHMgSUdyYW50YWJsZSB7XG4gIC8qKlxuICAgKiBXaGVuIHRoaXMgUHJpbmNpcGFsIGlzIHVzZWQgaW4gYW4gQXNzdW1lUm9sZSBwb2xpY3ksIHRoZSBhY3Rpb24gdG8gdXNlLlxuICAgKi9cbiAgcmVhZG9ubHkgYXNzdW1lUm9sZUFjdGlvbjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIHBvbGljeSBmcmFnbWVudCB0aGF0IGlkZW50aWZpZXMgdGhpcyBwcmluY2lwYWwgaW4gYSBQb2xpY3kuXG4gICAqL1xuICByZWFkb25seSBwb2xpY3lGcmFnbWVudDogUHJpbmNpcGFsUG9saWN5RnJhZ21lbnQ7XG5cbiAgLyoqXG4gICAqIEFkZCB0byB0aGUgcG9saWN5IG9mIHRoaXMgcHJpbmNpcGFsLlxuICAgKlxuICAgKiBAcmV0dXJucyB0cnVlIGlmIHRoZSBzdGF0ZW1lbnQgd2FzIGFkZGVkLCBmYWxzZSBpZiB0aGUgcHJpbmNpcGFsIGluXG4gICAqIHF1ZXN0aW9uIGRvZXMgbm90IGhhdmUgYSBwb2xpY3kgZG9jdW1lbnQgdG8gYWRkIHRoZSBzdGF0ZW1lbnQgdG8uXG4gICAqL1xuICBhZGRUb1BvbGljeShzdGF0ZW1lbnQ6IFBvbGljeVN0YXRlbWVudCk6IGJvb2xlYW47XG59XG5cbi8qKlxuICogQmFzZSBjbGFzcyBmb3IgcG9saWN5IHByaW5jaXBhbHNcbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFByaW5jaXBhbEJhc2UgaW1wbGVtZW50cyBJUHJpbmNpcGFsIHtcbiAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBJUHJpbmNpcGFsID0gdGhpcztcblxuICAvKipcbiAgICogUmV0dXJuIHRoZSBwb2xpY3kgZnJhZ21lbnQgdGhhdCBpZGVudGlmaWVzIHRoaXMgcHJpbmNpcGFsIGluIGEgUG9saWN5LlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHBvbGljeUZyYWdtZW50OiBQcmluY2lwYWxQb2xpY3lGcmFnbWVudDtcblxuICAvKipcbiAgICogV2hlbiB0aGlzIFByaW5jaXBhbCBpcyB1c2VkIGluIGFuIEFzc3VtZVJvbGUgcG9saWN5LCB0aGUgYWN0aW9uIHRvIHVzZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhc3N1bWVSb2xlQWN0aW9uOiBzdHJpbmcgPSAnc3RzOkFzc3VtZVJvbGUnO1xuXG4gIHB1YmxpYyBhZGRUb1BvbGljeShfc3RhdGVtZW50OiBQb2xpY3lTdGF0ZW1lbnQpOiBib29sZWFuIHtcbiAgICAvLyBUaGlzIGJhc2UgY2xhc3MgaXMgdXNlZCBmb3Igbm9uLWlkZW50aXR5IHByaW5jaXBhbHMuIE5vbmUgb2YgdGhlbVxuICAgIC8vIGhhdmUgYSBQb2xpY3lEb2N1bWVudCB0byBhZGQgdG8uXG4gICAgcmV0dXJuIGZhbHNlO1xuICB9XG5cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIC8vIFRoaXMgaXMgYSBmaXJzdCBwYXNzIHRvIG1ha2UgdGhlIG9iamVjdCByZWFkYWJsZS4gRGVzY2VuZGFudCBwcmluY2lwYWxzXG4gICAgLy8gc2hvdWxkIHJldHVybiBzb21ldGhpbmcgbmljZXIuXG4gICAgcmV0dXJuIEpTT04uc3RyaW5naWZ5KHRoaXMucG9saWN5RnJhZ21lbnQucHJpbmNpcGFsSnNvbik7XG4gIH1cblxuICAvKipcbiAgICogSlNPTi1pZnkgdGhlIHByaW5jaXBhbFxuICAgKlxuICAgKiBVc2VkIHdoZW4gSlNPTi5zdHJpbmdpZnkoKSBpcyBjYWxsZWRcbiAgICovXG4gIHB1YmxpYyB0b0pTT04oKSB7XG4gICAgLy8gSGF2ZSB0byBpbXBsZW1lbnQgdG9KU09OKCkgYmVjYXVzZSB0aGUgZGVmYXVsdCB3aWxsIGxlYWQgdG8gaW5maW5pdGUgcmVjdXJzaW9uLlxuICAgIHJldHVybiB0aGlzLnBvbGljeUZyYWdtZW50LnByaW5jaXBhbEpzb247XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyBhIG5ldyBQcmluY2lwYWxXaXRoQ29uZGl0aW9ucyB1c2luZyB0aGlzIHByaW5jaXBhbCBhcyB0aGUgYmFzZSwgd2l0aCB0aGVcbiAgICogcGFzc2VkIGNvbmRpdGlvbnMgYWRkZWQuXG4gICAqXG4gICAqIFdoZW4gdGhlcmUgaXMgYSB2YWx1ZSBmb3IgdGhlIHNhbWUgb3BlcmF0b3IgYW5kIGtleSBpbiBib3RoIHRoZSBwcmluY2lwYWwgYW5kIHRoZVxuICAgKiBjb25kaXRpb25zIHBhcmFtZXRlciwgdGhlIHZhbHVlIGZyb20gdGhlIGNvbmRpdGlvbnMgcGFyYW1ldGVyIHdpbGwgYmUgdXNlZC5cbiAgICpcbiAgICogQHJldHVybnMgYSBuZXcgUHJpbmNpcGFsV2l0aENvbmRpdGlvbnMgb2JqZWN0LlxuICAgKi9cbiAgcHVibGljIHdpdGhDb25kaXRpb25zKGNvbmRpdGlvbnM6IENvbmRpdGlvbnMpOiBJUHJpbmNpcGFsIHtcbiAgICByZXR1cm4gbmV3IFByaW5jaXBhbFdpdGhDb25kaXRpb25zKHRoaXMsIGNvbmRpdGlvbnMpO1xuICB9XG59XG5cbi8qKlxuICogQW4gSUFNIHByaW5jaXBhbCB3aXRoIGFkZGl0aW9uYWwgY29uZGl0aW9ucyBzcGVjaWZ5aW5nIHdoZW4gdGhlIHBvbGljeSBpcyBpbiBlZmZlY3QuXG4gKlxuICogRm9yIG1vcmUgaW5mb3JtYXRpb24gYWJvdXQgY29uZGl0aW9ucywgc2VlOlxuICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb24uaHRtbFxuICovXG5leHBvcnQgY2xhc3MgUHJpbmNpcGFsV2l0aENvbmRpdGlvbnMgaW1wbGVtZW50cyBJUHJpbmNpcGFsIHtcbiAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBJUHJpbmNpcGFsID0gdGhpcztcbiAgcHVibGljIHJlYWRvbmx5IGFzc3VtZVJvbGVBY3Rpb246IHN0cmluZyA9IHRoaXMucHJpbmNpcGFsLmFzc3VtZVJvbGVBY3Rpb247XG4gIHByaXZhdGUgYWRkaXRpb25hbENvbmRpdGlvbnM6IENvbmRpdGlvbnM7XG5cbiAgY29uc3RydWN0b3IoXG4gICAgcHJpdmF0ZSByZWFkb25seSBwcmluY2lwYWw6IElQcmluY2lwYWwsXG4gICAgY29uZGl0aW9uczogQ29uZGl0aW9ucyxcbiAgKSB7XG4gICAgdGhpcy5hZGRpdGlvbmFsQ29uZGl0aW9ucyA9IGNvbmRpdGlvbnM7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgY29uZGl0aW9uIHRvIHRoZSBwcmluY2lwYWxcbiAgICovXG4gIHB1YmxpYyBhZGRDb25kaXRpb24oa2V5OiBzdHJpbmcsIHZhbHVlOiBDb25kaXRpb24pIHtcbiAgICBjb25zdCBleGlzdGluZ1ZhbHVlID0gdGhpcy5jb25kaXRpb25zW2tleV07XG4gICAgdGhpcy5jb25kaXRpb25zW2tleV0gPSBleGlzdGluZ1ZhbHVlID8geyAuLi5leGlzdGluZ1ZhbHVlLCAuLi52YWx1ZSB9IDogdmFsdWU7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBtdWx0aXBsZSBjb25kaXRpb25zIHRvIHRoZSBwcmluY2lwYWxcbiAgICpcbiAgICogVmFsdWVzIGZyb20gdGhlIGNvbmRpdGlvbnMgcGFyYW1ldGVyIHdpbGwgb3ZlcndyaXRlIGV4aXN0aW5nIHZhbHVlcyB3aXRoIHRoZSBzYW1lIG9wZXJhdG9yXG4gICAqIGFuZCBrZXkuXG4gICAqL1xuICBwdWJsaWMgYWRkQ29uZGl0aW9ucyhjb25kaXRpb25zOiBDb25kaXRpb25zKSB7XG4gICAgT2JqZWN0LmVudHJpZXMoY29uZGl0aW9ucykuZm9yRWFjaCgoW2tleSwgdmFsdWVdKSA9PiB7XG4gICAgICB0aGlzLmFkZENvbmRpdGlvbihrZXksIHZhbHVlKTtcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgY29uZGl0aW9ucyB1bmRlciB3aGljaCB0aGUgcG9saWN5IGlzIGluIGVmZmVjdC5cbiAgICogU2VlIFt0aGUgSUFNIGRvY3VtZW50YXRpb25dKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uLmh0bWwpLlxuICAgKi9cbiAgcHVibGljIGdldCBjb25kaXRpb25zKCkge1xuICAgIHJldHVybiB0aGlzLm1lcmdlQ29uZGl0aW9ucyh0aGlzLnByaW5jaXBhbC5wb2xpY3lGcmFnbWVudC5jb25kaXRpb25zLCB0aGlzLmFkZGl0aW9uYWxDb25kaXRpb25zKTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgcG9saWN5RnJhZ21lbnQoKTogUHJpbmNpcGFsUG9saWN5RnJhZ21lbnQge1xuICAgIHJldHVybiBuZXcgUHJpbmNpcGFsUG9saWN5RnJhZ21lbnQodGhpcy5wcmluY2lwYWwucG9saWN5RnJhZ21lbnQucHJpbmNpcGFsSnNvbiwgdGhpcy5jb25kaXRpb25zKTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRUb1BvbGljeShzdGF0ZW1lbnQ6IFBvbGljeVN0YXRlbWVudCk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLnByaW5jaXBhbC5hZGRUb1BvbGljeShzdGF0ZW1lbnQpO1xuICB9XG5cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiB0aGlzLnByaW5jaXBhbC50b1N0cmluZygpO1xuICB9XG5cbiAgLyoqXG4gICAqIEpTT04taWZ5IHRoZSBwcmluY2lwYWxcbiAgICpcbiAgICogVXNlZCB3aGVuIEpTT04uc3RyaW5naWZ5KCkgaXMgY2FsbGVkXG4gICAqL1xuICBwdWJsaWMgdG9KU09OKCkge1xuICAgIC8vIEhhdmUgdG8gaW1wbGVtZW50IHRvSlNPTigpIGJlY2F1c2UgdGhlIGRlZmF1bHQgd2lsbCBsZWFkIHRvIGluZmluaXRlIHJlY3Vyc2lvbi5cbiAgICByZXR1cm4gdGhpcy5wb2xpY3lGcmFnbWVudC5wcmluY2lwYWxKc29uO1xuICB9XG5cbiAgcHJpdmF0ZSBtZXJnZUNvbmRpdGlvbnMocHJpbmNpcGFsQ29uZGl0aW9uczogQ29uZGl0aW9ucywgYWRkaXRpb25hbENvbmRpdGlvbnM6IENvbmRpdGlvbnMpOiBDb25kaXRpb25zIHtcbiAgICBjb25zdCBtZXJnZWRDb25kaXRpb25zOiBDb25kaXRpb25zID0ge307XG4gICAgT2JqZWN0LmVudHJpZXMocHJpbmNpcGFsQ29uZGl0aW9ucykuZm9yRWFjaCgoW29wZXJhdG9yLCBjb25kaXRpb25dKSA9PiB7XG4gICAgICBtZXJnZWRDb25kaXRpb25zW29wZXJhdG9yXSA9IGNvbmRpdGlvbjtcbiAgICB9KTtcbiAgICBPYmplY3QuZW50cmllcyhhZGRpdGlvbmFsQ29uZGl0aW9ucykuZm9yRWFjaCgoW29wZXJhdG9yLCBjb25kaXRpb25dKSA9PiB7XG4gICAgICBtZXJnZWRDb25kaXRpb25zW29wZXJhdG9yXSA9IHsgLi4ubWVyZ2VkQ29uZGl0aW9uc1tvcGVyYXRvcl0sIC4uLmNvbmRpdGlvbiB9O1xuICAgIH0pO1xuICAgIHJldHVybiBtZXJnZWRDb25kaXRpb25zO1xuICB9XG59XG5cbi8qKlxuICogQSBjb2xsZWN0aW9uIG9mIHRoZSBmaWVsZHMgaW4gYSBQb2xpY3lTdGF0ZW1lbnQgdGhhdCBjYW4gYmUgdXNlZCB0byBpZGVudGlmeSBhIHByaW5jaXBhbC5cbiAqXG4gKiBUaGlzIGNvbnNpc3RzIG9mIHRoZSBKU09OIHVzZWQgaW4gdGhlIFwiUHJpbmNpcGFsXCIgZmllbGQsIGFuZCBvcHRpb25hbGx5IGFcbiAqIHNldCBvZiBcIkNvbmRpdGlvblwicyB0aGF0IG5lZWQgdG8gYmUgYXBwbGllZCB0byB0aGUgcG9saWN5LlxuICovXG5leHBvcnQgY2xhc3MgUHJpbmNpcGFsUG9saWN5RnJhZ21lbnQge1xuICBjb25zdHJ1Y3RvcihcbiAgICBwdWJsaWMgcmVhZG9ubHkgcHJpbmNpcGFsSnNvbjogeyBba2V5OiBzdHJpbmddOiBzdHJpbmdbXSB9LFxuICAgIC8qKlxuICAgICAqIFRoZSBjb25kaXRpb25zIHVuZGVyIHdoaWNoIHRoZSBwb2xpY3kgaXMgaW4gZWZmZWN0LlxuICAgICAqIFNlZSBbdGhlIElBTSBkb2N1bWVudGF0aW9uXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbi5odG1sKS5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgY29uZGl0aW9uczogQ29uZGl0aW9ucyA9IHt9KSB7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIEFyblByaW5jaXBhbCBleHRlbmRzIFByaW5jaXBhbEJhc2Uge1xuICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgYXJuOiBzdHJpbmcpIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIGdldCBwb2xpY3lGcmFnbWVudCgpOiBQcmluY2lwYWxQb2xpY3lGcmFnbWVudCB7XG4gICAgcmV0dXJuIG5ldyBQcmluY2lwYWxQb2xpY3lGcmFnbWVudCh7IEFXUzogW3RoaXMuYXJuXSB9KTtcbiAgfVxuXG4gIHB1YmxpYyB0b1N0cmluZygpIHtcbiAgICByZXR1cm4gYEFyblByaW5jaXBhbCgke3RoaXMuYXJufSlgO1xuICB9XG59XG5cbmV4cG9ydCBjbGFzcyBBY2NvdW50UHJpbmNpcGFsIGV4dGVuZHMgQXJuUHJpbmNpcGFsIHtcbiAgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IGFjY291bnRJZDogYW55KSB7XG4gICAgc3VwZXIobmV3IFN0YWNrRGVwZW5kZW50VG9rZW4oc3RhY2sgPT4gYGFybjoke3N0YWNrLnBhcnRpdGlvbn06aWFtOjoke2FjY291bnRJZH06cm9vdGApLnRvU3RyaW5nKCkpO1xuICB9XG5cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiBgQWNjb3VudFByaW5jaXBhbCgke3RoaXMuYWNjb3VudElkfSlgO1xuICB9XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYSBzZXJ2aWNlIHByaW5jaXBhbC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTZXJ2aWNlUHJpbmNpcGFsT3B0cyB7XG4gIC8qKlxuICAgKiBUaGUgcmVnaW9uIGluIHdoaWNoIHRoZSBzZXJ2aWNlIGlzIG9wZXJhdGluZy5cbiAgICpcbiAgICogQGRlZmF1bHQgdGhlIGN1cnJlbnQgU3RhY2sncyByZWdpb24uXG4gICAqL1xuICByZWFkb25seSByZWdpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFkZGl0aW9uYWwgY29uZGl0aW9ucyB0byBhZGQgdG8gdGhlIFNlcnZpY2UgUHJpbmNpcGFsXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY29uZGl0aW9uc1xuICAgKi9cbiAgcmVhZG9ubHkgY29uZGl0aW9ucz86IHsgW2tleTogc3RyaW5nXTogYW55IH07XG59XG5cbi8qKlxuICogQW4gSUFNIHByaW5jaXBhbCB0aGF0IHJlcHJlc2VudHMgYW4gQVdTIHNlcnZpY2UgKGkuZS4gc3FzLmFtYXpvbmF3cy5jb20pLlxuICovXG5leHBvcnQgY2xhc3MgU2VydmljZVByaW5jaXBhbCBleHRlbmRzIFByaW5jaXBhbEJhc2Uge1xuICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgc2VydmljZTogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IG9wdHM6IFNlcnZpY2VQcmluY2lwYWxPcHRzID0ge30pIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIGdldCBwb2xpY3lGcmFnbWVudCgpOiBQcmluY2lwYWxQb2xpY3lGcmFnbWVudCB7XG4gICAgcmV0dXJuIG5ldyBQcmluY2lwYWxQb2xpY3lGcmFnbWVudCh7XG4gICAgICBTZXJ2aWNlOiBbXG4gICAgICAgIG5ldyBTZXJ2aWNlUHJpbmNpcGFsVG9rZW4odGhpcy5zZXJ2aWNlLCB0aGlzLm9wdHMpLnRvU3RyaW5nKClcbiAgICAgIF1cbiAgICB9LCB0aGlzLm9wdHMuY29uZGl0aW9ucyk7XG4gIH1cblxuICBwdWJsaWMgdG9TdHJpbmcoKSB7XG4gICAgcmV0dXJuIGBTZXJ2aWNlUHJpbmNpcGFsKCR7dGhpcy5zZXJ2aWNlfSlgO1xuICB9XG59XG5cbi8qKlxuICogQSBwcmluY2lwYWwgdGhhdCByZXByZXNlbnRzIGFuIEFXUyBPcmdhbml6YXRpb25cbiAqL1xuZXhwb3J0IGNsYXNzIE9yZ2FuaXphdGlvblByaW5jaXBhbCBleHRlbmRzIFByaW5jaXBhbEJhc2Uge1xuICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgb3JnYW5pemF0aW9uSWQ6IHN0cmluZykge1xuICAgIHN1cGVyKCk7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHBvbGljeUZyYWdtZW50KCk6IFByaW5jaXBhbFBvbGljeUZyYWdtZW50IHtcbiAgICByZXR1cm4gbmV3IFByaW5jaXBhbFBvbGljeUZyYWdtZW50KFxuICAgICAgeyBBV1M6IFsnKiddIH0sXG4gICAgICB7IFN0cmluZ0VxdWFsczogeyAnYXdzOlByaW5jaXBhbE9yZ0lEJzogdGhpcy5vcmdhbml6YXRpb25JZCB9IH1cbiAgICApO1xuICB9XG5cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiBgT3JnYW5pemF0aW9uUHJpbmNpcGFsKCR7dGhpcy5vcmdhbml6YXRpb25JZH0pYDtcbiAgfVxufVxuXG4vKipcbiAqIEEgcG9saWN5IHByaW5pY2lwYWwgZm9yIGNhbm9uaWNhbFVzZXJJZHMgLSB1c2VmdWwgZm9yIFMzIGJ1Y2tldCBwb2xpY2llcyB0aGF0IHVzZVxuICogT3JpZ2luIEFjY2VzcyBpZGVudGl0aWVzLlxuICpcbiAqIFNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ2VuZXJhbC9sYXRlc3QvZ3IvYWNjdC1pZGVudGlmaWVycy5odG1sXG4gKlxuICogYW5kXG4gKlxuICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvbkNsb3VkRnJvbnQvbGF0ZXN0L0RldmVsb3Blckd1aWRlL3ByaXZhdGUtY29udGVudC1yZXN0cmljdGluZy1hY2Nlc3MtdG8tczMuaHRtbFxuICpcbiAqIGZvciBtb3JlIGRldGFpbHMuXG4gKlxuICovXG5leHBvcnQgY2xhc3MgQ2Fub25pY2FsVXNlclByaW5jaXBhbCBleHRlbmRzIFByaW5jaXBhbEJhc2Uge1xuICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgY2Fub25pY2FsVXNlcklkOiBzdHJpbmcpIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIGdldCBwb2xpY3lGcmFnbWVudCgpOiBQcmluY2lwYWxQb2xpY3lGcmFnbWVudCB7XG4gICAgcmV0dXJuIG5ldyBQcmluY2lwYWxQb2xpY3lGcmFnbWVudCh7IENhbm9uaWNhbFVzZXI6IFt0aGlzLmNhbm9uaWNhbFVzZXJJZF0gfSk7XG4gIH1cblxuICBwdWJsaWMgdG9TdHJpbmcoKSB7XG4gICAgcmV0dXJuIGBDYW5vbmljYWxVc2VyUHJpbmNpcGFsKCR7dGhpcy5jYW5vbmljYWxVc2VySWR9KWA7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIEZlZGVyYXRlZFByaW5jaXBhbCBleHRlbmRzIFByaW5jaXBhbEJhc2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgYXNzdW1lUm9sZUFjdGlvbjogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIHB1YmxpYyByZWFkb25seSBmZWRlcmF0ZWQ6IHN0cmluZyxcbiAgICAvKipcbiAgICAgKiBUaGUgY29uZGl0aW9ucyB1bmRlciB3aGljaCB0aGUgcG9saWN5IGlzIGluIGVmZmVjdC5cbiAgICAgKiBTZWUgW3RoZSBJQU0gZG9jdW1lbnRhdGlvbl0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb24uaHRtbCkuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGNvbmRpdGlvbnM6IENvbmRpdGlvbnMsXG4gICAgYXNzdW1lUm9sZUFjdGlvbjogc3RyaW5nID0gJ3N0czpBc3N1bWVSb2xlJykge1xuICAgIHN1cGVyKCk7XG5cbiAgICB0aGlzLmFzc3VtZVJvbGVBY3Rpb24gPSBhc3N1bWVSb2xlQWN0aW9uO1xuICB9XG5cbiAgcHVibGljIGdldCBwb2xpY3lGcmFnbWVudCgpOiBQcmluY2lwYWxQb2xpY3lGcmFnbWVudCB7XG4gICAgcmV0dXJuIG5ldyBQcmluY2lwYWxQb2xpY3lGcmFnbWVudCh7IEZlZGVyYXRlZDogW3RoaXMuZmVkZXJhdGVkXSB9LCB0aGlzLmNvbmRpdGlvbnMpO1xuICB9XG5cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiBgRmVkZXJhdGVkUHJpbmNpcGFsKCR7dGhpcy5mZWRlcmF0ZWR9KWA7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIEFjY291bnRSb290UHJpbmNpcGFsIGV4dGVuZHMgQWNjb3VudFByaW5jaXBhbCB7XG4gIGNvbnN0cnVjdG9yKCkge1xuICAgIHN1cGVyKG5ldyBTdGFja0RlcGVuZGVudFRva2VuKHN0YWNrID0+IHN0YWNrLmFjY291bnQpLnRvU3RyaW5nKCkpO1xuICB9XG5cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiAnQWNjb3VudFJvb3RQcmluY2lwYWwoKSc7XG4gIH1cbn1cblxuLyoqXG4gKiBBIHByaW5jaXBhbCByZXByZXNlbnRpbmcgYWxsIGlkZW50aXRpZXMgaW4gYWxsIGFjY291bnRzXG4gKi9cbmV4cG9ydCBjbGFzcyBBbnlQcmluY2lwYWwgZXh0ZW5kcyBBcm5QcmluY2lwYWwge1xuICBjb25zdHJ1Y3RvcigpIHtcbiAgICBzdXBlcignKicpO1xuICB9XG5cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiAnQW55UHJpbmNpcGFsKCknO1xuICB9XG59XG5cbi8qKlxuICogQSBwcmluY2lwYWwgcmVwcmVzZW50aW5nIGFsbCBpZGVudGl0aWVzIGluIGFsbCBhY2NvdW50c1xuICogQGRlcHJlY2F0ZWQgdXNlIGBBbnlQcmluY2lwYWxgXG4gKi9cbmV4cG9ydCBjbGFzcyBBbnlvbmUgZXh0ZW5kcyBBbnlQcmluY2lwYWwgeyB9XG5cbmV4cG9ydCBjbGFzcyBDb21wb3NpdGVQcmluY2lwYWwgZXh0ZW5kcyBQcmluY2lwYWxCYXNlIHtcbiAgcHVibGljIHJlYWRvbmx5IGFzc3VtZVJvbGVBY3Rpb246IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBwcmluY2lwYWxzID0gbmV3IEFycmF5PFByaW5jaXBhbEJhc2U+KCk7XG5cbiAgY29uc3RydWN0b3IoLi4ucHJpbmNpcGFsczogUHJpbmNpcGFsQmFzZVtdKSB7XG4gICAgc3VwZXIoKTtcbiAgICBpZiAocHJpbmNpcGFscy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ29tcG9zaXRlUHJpbmNpcGFscyBtdXN0IGJlIGNvbnN0cnVjdGVkIHdpdGggYXQgbGVhc3QgMSBQcmluY2lwYWwgYnV0IG5vbmUgd2VyZSBwYXNzZWQuJyk7XG4gICAgfVxuICAgIHRoaXMuYXNzdW1lUm9sZUFjdGlvbiA9IHByaW5jaXBhbHNbMF0uYXNzdW1lUm9sZUFjdGlvbjtcbiAgICB0aGlzLmFkZFByaW5jaXBhbHMoLi4ucHJpbmNpcGFscyk7XG4gIH1cblxuICBwdWJsaWMgYWRkUHJpbmNpcGFscyguLi5wcmluY2lwYWxzOiBQcmluY2lwYWxCYXNlW10pOiB0aGlzIHtcbiAgICBmb3IgKGNvbnN0IHAgb2YgcHJpbmNpcGFscykge1xuICAgICAgaWYgKHAuYXNzdW1lUm9sZUFjdGlvbiAhPT0gdGhpcy5hc3N1bWVSb2xlQWN0aW9uKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICAnQ2Fubm90IGFkZCBtdWx0aXBsZSBwcmluY2lwYWxzIHdpdGggZGlmZmVyZW50IFwiYXNzdW1lUm9sZUFjdGlvblwiLiAnICtcbiAgICAgICAgICBgRXhwZWN0aW5nIFwiJHt0aGlzLmFzc3VtZVJvbGVBY3Rpb259XCIsIGdvdCBcIiR7cC5hc3N1bWVSb2xlQWN0aW9ufVwiYCk7XG4gICAgICB9XG5cbiAgICAgIGNvbnN0IGZyYWdtZW50ID0gcC5wb2xpY3lGcmFnbWVudDtcbiAgICAgIGlmIChmcmFnbWVudC5jb25kaXRpb25zICYmIE9iamVjdC5rZXlzKGZyYWdtZW50LmNvbmRpdGlvbnMpLmxlbmd0aCA+IDApIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdDb21wb25lbnRzIG9mIGEgQ29tcG9zaXRlUHJpbmNpcGFsIG11c3Qgbm90IGhhdmUgY29uZGl0aW9ucy4gJyArXG4gICAgICAgICAgYFRyaWVkIHRvIGFkZCB0aGUgZm9sbG93aW5nIGZyYWdtZW50OiAke0pTT04uc3RyaW5naWZ5KGZyYWdtZW50KX1gKTtcbiAgICAgIH1cblxuICAgICAgdGhpcy5wcmluY2lwYWxzLnB1c2gocCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHBvbGljeUZyYWdtZW50KCk6IFByaW5jaXBhbFBvbGljeUZyYWdtZW50IHtcbiAgICBjb25zdCBwcmluY2lwYWxKc29uOiB7IFtrZXk6IHN0cmluZ106IHN0cmluZ1tdIH0gPSB7fTtcblxuICAgIGZvciAoY29uc3QgcCBvZiB0aGlzLnByaW5jaXBhbHMpIHtcbiAgICAgIG1lcmdlUHJpbmNpcGFsKHByaW5jaXBhbEpzb24sIHAucG9saWN5RnJhZ21lbnQucHJpbmNpcGFsSnNvbik7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBQcmluY2lwYWxQb2xpY3lGcmFnbWVudChwcmluY2lwYWxKc29uKTtcbiAgfVxuXG4gIHB1YmxpYyB0b1N0cmluZygpIHtcbiAgICByZXR1cm4gYENvbXBvc2l0ZVByaW5jaXBhbCgke3RoaXMucHJpbmNpcGFsc30pYDtcbiAgfVxufVxuXG4vKipcbiAqIEEgbGF6eSB0b2tlbiB0aGF0IHJlcXVpcmVzIGFuIGluc3RhbmNlIG9mIFN0YWNrIHRvIGV2YWx1YXRlXG4gKi9cbmNsYXNzIFN0YWNrRGVwZW5kZW50VG9rZW4gaW1wbGVtZW50cyBjZGsuSVJlc29sdmFibGUge1xuICBwdWJsaWMgcmVhZG9ubHkgY3JlYXRpb25TdGFjazogc3RyaW5nW107XG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgZm46IChzdGFjazogY2RrLlN0YWNrKSA9PiBhbnkpIHtcbiAgICB0aGlzLmNyZWF0aW9uU3RhY2sgPSBjZGsuY2FwdHVyZVN0YWNrVHJhY2UoKTtcbiAgfVxuXG4gIHB1YmxpYyByZXNvbHZlKGNvbnRleHQ6IGNkay5JUmVzb2x2ZUNvbnRleHQpIHtcbiAgICByZXR1cm4gdGhpcy5mbihjZGsuU3RhY2sub2YoY29udGV4dC5zY29wZSkpO1xuICB9XG5cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiBjZGsuVG9rZW4uYXNTdHJpbmcodGhpcyk7XG4gIH1cblxuICAvKipcbiAgICogSlNPTi1pZnkgdGhlIHRva2VuXG4gICAqXG4gICAqIFVzZWQgd2hlbiBKU09OLnN0cmluZ2lmeSgpIGlzIGNhbGxlZFxuICAgKi9cbiAgcHVibGljIHRvSlNPTigpIHtcbiAgICByZXR1cm4gJzx1bnJlc29sdmVkLXRva2VuPic7XG4gIH1cbn1cblxuY2xhc3MgU2VydmljZVByaW5jaXBhbFRva2VuIGltcGxlbWVudHMgY2RrLklSZXNvbHZhYmxlIHtcbiAgcHVibGljIHJlYWRvbmx5IGNyZWF0aW9uU3RhY2s6IHN0cmluZ1tdO1xuICBjb25zdHJ1Y3RvcihcbiAgICBwcml2YXRlIHJlYWRvbmx5IHNlcnZpY2U6IHN0cmluZyxcbiAgICBwcml2YXRlIHJlYWRvbmx5IG9wdHM6IFNlcnZpY2VQcmluY2lwYWxPcHRzKSB7XG4gICAgdGhpcy5jcmVhdGlvblN0YWNrID0gY2RrLmNhcHR1cmVTdGFja1RyYWNlKCk7XG4gIH1cblxuICBwdWJsaWMgcmVzb2x2ZShjdHg6IGNkay5JUmVzb2x2ZUNvbnRleHQpIHtcbiAgICBjb25zdCByZWdpb24gPSB0aGlzLm9wdHMucmVnaW9uIHx8IGNkay5TdGFjay5vZihjdHguc2NvcGUpLnJlZ2lvbjtcbiAgICBjb25zdCBmYWN0ID0gUmVnaW9uSW5mby5nZXQocmVnaW9uKS5zZXJ2aWNlUHJpbmNpcGFsKHRoaXMuc2VydmljZSk7XG4gICAgcmV0dXJuIGZhY3QgfHwgRGVmYXVsdC5zZXJ2aWNlUHJpbmNpcGFsKHRoaXMuc2VydmljZSwgcmVnaW9uLCBjZGsuQXdzLlVSTF9TVUZGSVgpO1xuICB9XG5cbiAgcHVibGljIHRvU3RyaW5nKCkge1xuICAgIHJldHVybiBjZGsuVG9rZW4uYXNTdHJpbmcodGhpcywge1xuICAgICAgZGlzcGxheUhpbnQ6IHRoaXMuc2VydmljZVxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEpTT04taWZ5IHRoZSB0b2tlblxuICAgKlxuICAgKiBVc2VkIHdoZW4gSlNPTi5zdHJpbmdpZnkoKSBpcyBjYWxsZWRcbiAgICovXG4gIHB1YmxpYyB0b0pTT04oKSB7XG4gICAgcmV0dXJuIGA8JHt0aGlzLnNlcnZpY2V9PmA7XG4gIH1cbn1cbiJdfQ==