"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = '/' + prefix.join('/');
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            scope: options.scope,
            registerPostProcessor(pp) { postProcessor = pp; },
            resolve(x) { return resolve(x, { ...options, prefix: newPrefix }); },
        };
        return [context, { postProcess(x) { return postProcessor ? postProcessor.postProcess(x, context) : x; } }];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error('Unable to resolve object tree with circular reference. Path: ' + pathName);
    }
    //
    // undefined
    //
    if (typeof (obj) === 'undefined') {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof (obj) === 'function') {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof (obj) === 'string') {
        const str = encoding_1.TokenString.forString(obj);
        if (str.test()) {
            const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
            return options.resolver.resolveString(fragments, makeContext()[0]);
        }
        return obj;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof (obj) === 'number') {
        return resolveNumberToken(obj, makeContext()[0]);
    }
    //
    // primitives - as-is
    //
    if (typeof (obj) !== 'object' || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if (encoding_1.containsListTokenElement(obj)) {
            return options.resolver.resolveList(obj, makeContext()[0]);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter(x => typeof (x) !== 'undefined');
        return arr;
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if (encoding_1.unresolved(obj)) {
        const [context, postProcessor] = makeContext();
        return options.resolver.resolveToken(obj, context, postProcessor);
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error('Trying to resolve() a Construct at ' + pathName);
    }
    const result = {};
    for (const key of Object.keys(obj)) {
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof (resolvedKey) !== 'string') {
            throw new Error(`"${key}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}`);
        }
        const value = makeContext(key)[0].resolve(obj[key]);
        // skip undefined
        if (typeof (value) === 'undefined') {
            continue;
        }
        result[resolvedKey] = value;
    }
    return result;
}
exports.resolve = resolve;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
//# sourceMappingURL=data:application/json;base64,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