"use strict";
const lib_1 = require("../lib");
class TaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.STANDARD, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
class AsgTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.AUTOSCALING_GROUP, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
class MapTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.MAP, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
module.exports = {
    'Tag visit all children of the applied node'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.Tag('foo', 'bar'));
        lib_1.ConstructNode.synth(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(map.tags.renderTags(), { foo: 'bar' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'foo', value: 'bar', propagateAtLaunch: true }]);
        test.done();
    },
    'The last aspect applied takes precedence'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.Tag('foo', 'bar'));
        res.node.applyAspect(new lib_1.Tag('foo', 'foobar'));
        res.node.applyAspect(new lib_1.Tag('foo', 'baz'));
        res2.node.applyAspect(new lib_1.Tag('foo', 'good'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'baz' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'good' }]);
        test.done();
    },
    'RemoveTag will remove a tag if it exists'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        root.node.applyAspect(new lib_1.Tag('root', 'was here'));
        res.node.applyAspect(new lib_1.Tag('first', 'there is only 1'));
        res.node.applyAspect(new lib_1.RemoveTag('root'));
        res.node.applyAspect(new lib_1.RemoveTag('doesnotexist'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.deepEqual(map.tags.renderTags(), { first: 'there is only 1' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'first', value: 'there is only 1', propagateAtLaunch: true }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.done();
    },
    'the #visit function is idempotent'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.Tag('foo', 'bar'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.done();
    },
    'removeTag Aspects by default will override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.RemoveTag('key'));
        res2.node.applyAspect(new lib_1.Tag('key', 'value'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), undefined);
        test.done();
    },
    'removeTag Aspects with priority 0 will not override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.RemoveTag('key', { priority: 0 }));
        res2.node.applyAspect(new lib_1.Tag('key', 'value'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'key', value: 'value' }]);
        test.done();
    },
    'Aspects are merged with tags created by L1 Constructor'(test) {
        const root = new lib_1.Stack();
        const aspectBranch = new TaggableResource(root, 'FakeBranchA', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite' },
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        const asgResource = new AsgTaggableResource(aspectBranch, 'FakeAsg', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite', propagateAtLaunch: false },
                    { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                ],
            },
        });
        const mapTaggable = new MapTaggableResource(aspectBranch, 'FakeSam', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: {
                    aspects: 'overwrite',
                    cfn: 'is cool',
                },
            },
        });
        const cfnBranch = new TaggableResource(root, 'FakeBranchB', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        aspectBranch.node.applyAspect(new lib_1.Tag('aspects', 'rule'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(aspectBranch.testProperties().tags, [{ key: 'aspects', value: 'rule' }, { key: 'cfn', value: 'is cool' }]);
        test.deepEqual(asgResource.testProperties().tags, [
            { key: 'aspects', value: 'rule', propagateAtLaunch: true },
            { key: 'cfn', value: 'is cool', propagateAtLaunch: true }
        ]);
        test.deepEqual(mapTaggable.testProperties().tags, {
            aspects: 'rule',
            cfn: 'is cool',
        });
        test.deepEqual(cfnBranch.testProperties().tags, [{ key: 'cfn', value: 'is cool' }]);
        test.done();
    },
    'when invalid tag properties are passed from L1s': {
        'map passed instead of array it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new TaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                        },
                    },
                });
            });
            test.throws(() => {
                new AsgTaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                            propagateAtLaunch: true,
                        },
                    },
                });
            });
            test.done();
        },
        'if array is passed instead of map it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new MapTaggableResource(root, 'FakeSam', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: [
                            { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                            { key: 'aspects', value: 'overwrite' },
                        ],
                    },
                });
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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