"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return false;
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    constructor(principalJson, conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
class ArnPrincipal extends PrincipalBase {
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
class AccountPrincipal extends ArnPrincipal {
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString()
            ]
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
class FederatedPrincipal extends PrincipalBase {
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.account).toString());
    }
    toString() {
        return `AccountRootPrincipal()`;
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return `AnyPrincipal()`;
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error(`Cannot add multiple principals with different "assumeRoleAction". ` +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error(`Components of a CompositePrincipal must not have conditions. ` +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(fn) {
        this.fn = fn;
        this.creationStack = core_1.captureStackTrace();
    }
    resolve(context) {
        return this.fn(core_1.Stack.of(context.scope));
    }
    toString() {
        return cdk.Token.asString(this);
    }
    toJSON() {
        return `<unresolved-token>`;
    }
}
class ServicePrincipalToken {
    constructor(service, opts) {
        this.service = service;
        this.opts = opts;
        this.creationStack = core_1.captureStackTrace();
    }
    resolve(ctx) {
        const region = this.opts.region || core_1.Stack.of(ctx.scope).region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, core_1.Stack.of(ctx.scope).urlSuffix);
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: this.service
        });
    }
    toJSON() {
        return `<${this.service}>`;
    }
}
//# sourceMappingURL=data:application/json;base64,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