# Copyright (C) 2025 Shoal Software LLC. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import enum

from pants.option.option_types import (
    BoolOption,
    DictOption,
    EnumOption,
    FloatOption,
    IntOption,
    StrOption,
)
from pants.option.subsystem import Subsystem
from pants.util.strutil import softwrap


class TracingExporterId(enum.Enum):
    OTLP = "otlp"
    JSON_FILE = "json-file"


class OtelCompression(enum.Enum):
    GZIP = "gzip"
    DEFLATE = "deflate"
    NONE = "none"


class TelemetrySubsystem(Subsystem):
    options_scope = "shoalsoft-opentelemetry"
    help = "Pants OpenTelemetry plugin from Shoal Software LLC"

    enabled = BoolOption(default=False, help="Whether to enable emitting OpenTelemetry spans.")

    exporter = EnumOption(
        default=TracingExporterId.OTLP,
        help=softwrap(
            f"""
            Set the exporter to use when exporting workunits to external tracing systems. Choices are
            `{TracingExporterId.OTLP.value}` (OpenTelemetry OTLP over HTTP),
            `{TracingExporterId.JSON_FILE.value}` (Write OpenTelemetry-style debug output to a file).
            Default is `{TracingExporterId.OTLP.value}`.
            """
        ),
    )

    finish_timeout = FloatOption(
        default=2.0,
        help=softwrap(
            """
            The number of seconds to wait at the end of a session for export of all tracing spans to OpenTelemetry
            to complete.
            """
        ),
    )

    parse_traceparent = BoolOption(
        default=True,
        help=softwrap(
            """
            If `True`, then parse the `TRACEPARENT` environment variable if it is present in the environment.
            `TRACEPARENT` contains the parent trace ID and parent span ID to which to link any trace generated by
            this plugin. This is useful for linking Pants-related traces together with the trace for the CI job.

            The format of the `TRACEPARENT` environment variable is defined in the W3C Trace Context specification:
            https://www.w3.org/TR/trace-context/#traceparent-header
            """
        ),
    )

    async_completion = BoolOption(
        default=False,
        help=softwrap(
            """
            If `True`, allows the plugin to finish asynchronously when Pants is shutting down. This can result in
            faster Pants exit times but may lead to some spans not being exported if the export process is slow.
            If `False`, forces synchronous completion, ensuring all spans are exported before Pants exits but
            potentially slowing down the shutdown process. Defaults to `False`.
            """
        ),
    )

    json_file = StrOption(
        default="dist/otel-json-trace.jsonl",
        help=softwrap(
            f"""
            If set, Pants will write OpenTelemetry tracing spans to a local file for easier debugging. Each line
            will be a tracing span in OpenTelemetry's JSON format. The filename is relative to the build root. Export
            will only occur if the `--shoalsoft-opentelemetry-exporter` is set to `{TracingExporterId.JSON_FILE.value}`.
            """
        ),
    )

    trace_link_template = StrOption(
        default=None,
        help=softwrap(
            """
            Log a link to the URL at which the trace will be available in a trace management system. The following
            replacement variables are available:

            - `{trace_id}` - The OpenTelemetry trace ID

            - `{root_span_id}` - The span ID of the root span of the trace

            - `{trace_start_ms}` - Start time of the root span in milliseconds since the UNIX epoch

            - `{trace_end_ms}` - End time of the root span in milliseconds since the UNIX epoch
            """
        ),
    )

    exporter_endpoint = StrOption(
        default=None,
        help=softwrap(
            """
            The target to which the exporter is going to send traces, metrics, or logs. The endpoint MUST be a valid URL host,
            and MAY contain a scheme (http or https), port and path. The plugin will construct a "signal-specific" URL for
            sending traces by appending the applicable URL path if the signal-specific `[shoalsoft-opentelemetry].exporter_traces_endpoint`
            option is not already set to override this option.

            Corresponds to the `OTEL_EXPORTER_OTLP_ENDPOINT` environment variable.
            """
        ),
    )

    exporter_traces_endpoint = StrOption(
        default=None,
        advanced=True,
        help=softwrap(
            """
            The target to which the exporter is going to send traces. The endpoint MUST be a valid URL host,
            and MAY contain a scheme (http or https), port and path. If this option is set, then the
            `[shoalsoft-opentelemetry].exporter_endpoint` option will not be used. The URL is not modified
            at all since it is specific to the traces endpoint to use.

            You should not normally need to set this option. Prefer using the `[shoalsoft-opentelemetry].exporter_endpoint`
            option instead.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_ENDPOINT`.
            """
        ),
    )

    exporter_certificate_file = StrOption(
        default=None,
        advanced=True,
        help=softwrap(
            """
            The path to the certificate file for TLS credentials of gRPC client for traces.
            Should only be used for a secure connection for tracing.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_CLIENT_KEY` and `OTEL_EXPORTER_OTLP_CERTIFICATE`
            environment variables.
            """
        ),
    )

    exporter_client_key_file = StrOption(
        default=None,
        advanced=True,
        help=softwrap(
            """
            The path to the client private key to use in mTLS communication in PEM format for traces.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_CLIENT_KEY` and `OTEL_EXPORTER_OTLP_CLIENT_KEY`
            environment variables.
            """
        ),
    )

    exporter_client_certificate_file = StrOption(
        default=None,
        advanced=True,
        help=softwrap(
            """
            The path to the client certificate/chain trust for clients private key to use in mTLS
            communication in PEM format for traces.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_CLIENT_CERTIFICATE` and
            `OTEL_EXPORTER_OTLP_CLIENT_CERTIFICATE` environment variables.
            """
        ),
    )

    exporter_headers = DictOption[str](
        advanced=True,
        help=softwrap(
            """
            The key-value pairs to be used as headers for spans associated with gRPC or HTTP requests.

            This option is consumed by both the `{TracingExporterId.HTTP.value}` and `{TracingExporterId.GRPC.value}`
            exporters.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_HEADERS` and `OTEL_EXPORTER_OTLP_HEADERS`
            environment variables.
            """
        ),
    )

    exporter_timeout = IntOption(
        default=None,
        advanced=True,
        help=softwrap(
            """
            The maximum time the OTLP exporter will wait for each batch export for spans.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_TIMEOUT` and `OTEL_EXPORTER_OTLP_TIMEOUT`
            environment variables.
            """
        ),
    )

    exporter_compression = EnumOption(
        default=OtelCompression.NONE,
        advanced=True,
        help=softwrap(
            """
            Specifies a gRPC compression method to be used in the OTLP exporters. Possible values are `gzip`,
            `deflate`, and `none`.

            Corresponds to the `OTEL_EXPORTER_OTLP_TRACES_COMPRESSION` and `OTEL_EXPORTER_OTLP_COMPRESSION`
            environment variables.
            """
        ),
    )
