"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = exports.Protocol = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const k8s = require("./imports/k8s");
const probe = require("./probe");
const utils_1 = require("./utils");
/**
 * Network protocols.
 */
var Protocol;
(function (Protocol) {
    /**
     * TCP.
     */
    Protocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    Protocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? false;
        this.user = props.user ?? 25000;
        this.group = props.group ?? 26000;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-23.ContainerSecurityContext", version: "2.0.0-rc.108" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-23.EnvValue", version: "2.0.0-rc.108" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        this._ports = [];
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        const defaultResourceSpec = {
            cpu: {
                request: Cpu.millis(1000),
                limit: Cpu.millis(1500),
            },
            memory: {
                request: cdk8s_1.Size.mebibytes(512),
                limit: cdk8s_1.Size.mebibytes(2048),
            },
        };
        if (props.port && props.portNumber) {
            throw new Error('Either \'port\' or \'portNumber\' is allowed. Use \'portNumber\' since \'port\' is deprecated');
        }
        const portNumber = props.portNumber ?? props.port;
        const defaultProbeConfiguration = probe.Probe.fromTcpSocket({ port: portNumber });
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.portNumber = portNumber;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup ?? (this.portNumber ? defaultProbeConfiguration : undefined);
        this._lifecycle = props.lifecycle;
        this.resources = props.resources ?? defaultResourceSpec;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
        if (this.portNumber) {
            this.addPort({
                number: this.portNumber,
            });
        }
        for (const port of props.ports ?? []) {
            this.addPort(port);
        }
    }
    /**
     * @deprecated - use `portNumber`.
     */
    get port() {
        return this.portNumber;
    }
    /**
     * Ports exposed by this containers.
     * Returns a copy, use `addPort` to modify.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * Add a port to expose from this container.
     */
    addPort(port) {
        const names = this._ports.map(p => p.name).filter(x => x);
        const numbers = this._ports.map(p => p.number);
        if (port.name && names.includes(port.name)) {
            throw new Error(`Port with name ${port.name} already exists`);
        }
        if (numbers.includes(port.number)) {
            throw new Error(`Port with number ${port.number} already exists`);
        }
        this._ports.push(port);
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        for (const port of this.ports) {
            ports.push({
                containerPort: port.number,
                protocol: port.protocol ? this._portProtocolToKube(port.protocol) : undefined,
                name: port.name,
                hostPort: port.hostPort,
                hostIp: port.hostIp,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this._imagePullPolicyToKube(this.imagePullPolicy),
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            securityContext: this.securityContext._toKube(),
        };
    }
    _imagePullPolicyToKube(imagePullPolicy) {
        switch (imagePullPolicy) {
            case ImagePullPolicy.ALWAYS:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.ALWAYS;
            case ImagePullPolicy.IF_NOT_PRESENT:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.IF_NOT_PRESENT;
            case ImagePullPolicy.NEVER:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.NEVER;
            default:
                throw new Error(`Unsupported image pull policy: ${imagePullPolicy}`);
        }
    }
    _portProtocolToKube(protocol) {
        switch (protocol) {
            case Protocol.SCTP:
                return k8s.IoK8SApiCoreV1ContainerPortProtocol.SCTP;
            case Protocol.TCP:
                return k8s.IoK8SApiCoreV1ContainerPortProtocol.TCP;
            case Protocol.UDP:
                return k8s.IoK8SApiCoreV1ContainerPortProtocol.UDP;
            default:
                throw new Error(`Unsupported port protocol: ${protocol}`);
        }
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-23.Container", version: "2.0.0-rc.108" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-23.Cpu", version: "2.0.0-rc.108" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-23.EnvFrom", version: "2.0.0-rc.108" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers ?? [];
    return containers.flatMap(c => c.ports);
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-23.Env", version: "2.0.0-rc.108" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29udGFpbmVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL2NvbnRhaW5lci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLGlDQUE2QjtBQUc3QixxQ0FBcUM7QUFDckMsaUNBQWlDO0FBRWpDLG1DQUEyQztBQThGM0M7O0dBRUc7QUFDSCxJQUFZLFFBZ0JYO0FBaEJELFdBQVksUUFBUTtJQUVsQjs7T0FFRztJQUNILHVCQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILHVCQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILHlCQUFhLENBQUE7QUFDZixDQUFDLEVBaEJXLFFBQVEsR0FBUixnQkFBUSxLQUFSLGdCQUFRLFFBZ0JuQjtBQUdEOztHQUVHO0FBQ0gsTUFBYSx3QkFBd0I7SUFRbkMsWUFBWSxRQUF1QyxFQUFFO1FBQ25ELElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsSUFBSSxLQUFLLENBQUM7UUFDbEQsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQztRQUM1QyxJQUFJLENBQUMsc0JBQXNCLEdBQUcsS0FBSyxDQUFDLHNCQUFzQixJQUFJLEtBQUssQ0FBQztRQUNwRSxJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLElBQUksS0FBSyxDQUFDO1FBQ2hDLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUM7SUFDcEMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksT0FBTztRQUNaLE9BQU87WUFDTCxVQUFVLEVBQUUsSUFBSSxDQUFDLEtBQUs7WUFDdEIsU0FBUyxFQUFFLElBQUksQ0FBQyxJQUFJO1lBQ3BCLFlBQVksRUFBRSxJQUFJLENBQUMsYUFBYTtZQUNoQyxVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDM0Isc0JBQXNCLEVBQUUsSUFBSSxDQUFDLHNCQUFzQjtTQUNwRCxDQUFDO0lBQ0osQ0FBQzs7QUEzQkgsNERBNkJDOzs7QUFFRCxJQUFZLGFBa0RYO0FBbERELFdBQVksYUFBYTtJQUN2Qjs7T0FFRztJQUNILDJDQUEwQixDQUFBO0lBRTFCOztPQUVHO0lBQ0gscURBQW9DLENBQUE7SUFFcEM7O09BRUc7SUFDSCx5Q0FBd0IsQ0FBQTtJQUV4Qjs7T0FFRztJQUNILDhDQUE2QixDQUFBO0lBRTdCOztPQUVHO0lBQ0gsd0RBQXVDLENBQUE7SUFFdkM7O09BRUc7SUFDSCx3Q0FBdUIsQ0FBQTtJQUV2Qjs7T0FFRztJQUNILGlFQUFnRCxDQUFBO0lBRWhEOztPQUVHO0lBQ0gsNENBQTJCLENBQUE7SUFFM0I7O09BRUc7SUFDSCwwQ0FBeUIsQ0FBQTtJQUV6Qjs7T0FFRztJQUNILDBDQUF5QixDQUFBO0FBQzNCLENBQUMsRUFsRFcsYUFBYSxHQUFiLHFCQUFhLEtBQWIscUJBQWEsUUFrRHhCO0FBRUQsSUFBWSxrQkE4Qlg7QUE5QkQsV0FBWSxrQkFBa0I7SUFDNUI7O09BRUc7SUFDSCw4Q0FBd0IsQ0FBQTtJQUV4Qjs7T0FFRztJQUNILG9EQUE4QixDQUFBO0lBRTlCOztPQUVHO0lBQ0gsa0RBQTRCLENBQUE7SUFFNUI7O09BRUc7SUFDSCx3REFBa0MsQ0FBQTtJQUVsQzs7T0FFRztJQUNILGdFQUEwQyxDQUFBO0lBRTFDOztPQUVHO0lBQ0gsb0VBQThDLENBQUE7QUFDaEQsQ0FBQyxFQTlCVyxrQkFBa0IsR0FBbEIsMEJBQWtCLEtBQWxCLDBCQUFrQixRQThCN0I7QUF5RUQ7O0dBRUc7QUFDSCxNQUFhLFFBQVE7SUErR25CLFlBQW9DLEtBQVcsRUFBa0IsU0FBZTtRQUE1QyxVQUFLLEdBQUwsS0FBSyxDQUFNO1FBQWtCLGNBQVMsR0FBVCxTQUFTLENBQU07SUFBRyxDQUFDO0lBN0dwRjs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsYUFBYSxDQUFDLFNBQStCLEVBQUUsR0FBVyxFQUFFLFVBQXdDLEVBQUc7UUFFbkgsTUFBTSxNQUFNLEdBQXFCO1lBQy9CLGVBQWUsRUFBRTtnQkFDZixJQUFJLEVBQUUsU0FBUyxDQUFDLElBQUk7Z0JBQ3BCLEdBQUc7Z0JBQ0gsUUFBUSxFQUFFLE9BQU8sQ0FBQyxRQUFRO2FBQzNCO1NBQ0YsQ0FBQztRQUNGLE9BQU8sSUFBSSxRQUFRLENBQUMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxlQUFlLENBQUMsV0FBK0IsRUFBRSxVQUFxQyxFQUFFO1FBQ3BHLE1BQU0sTUFBTSxHQUFxQjtZQUMvQixZQUFZLEVBQUU7Z0JBQ1osSUFBSSxFQUFFLFdBQVcsQ0FBQyxNQUFNLENBQUMsSUFBSTtnQkFDN0IsR0FBRyxFQUFFLFdBQVcsQ0FBQyxHQUFHO2dCQUNwQixRQUFRLEVBQUUsT0FBTyxDQUFDLFFBQVE7YUFDM0I7U0FDRixDQUFDO1FBRUYsT0FBTyxJQUFJLFFBQVEsQ0FBQyxTQUFTLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQWE7UUFDbkMsT0FBTyxJQUFJLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUM3QixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxTQUF3QixFQUFFLFVBQXVDLEVBQUU7UUFDNUYsSUFBSSxRQUFRLEdBQVksS0FBSyxDQUFDO1FBRTlCLElBQUksU0FBUyxLQUFLLGFBQWEsQ0FBQyxTQUFTLElBQUksU0FBUyxLQUFLLGFBQWEsQ0FBQyxjQUFjLEVBQUU7WUFDdkYsUUFBUSxHQUFHLElBQUksQ0FBQztTQUNqQjtRQUVELElBQUksUUFBUSxJQUFJLE9BQU8sQ0FBQyxHQUFHLEtBQUssU0FBUyxFQUFFO1lBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxTQUFTLGlCQUFpQixDQUFDLENBQUM7U0FDaEQ7UUFFRCxNQUFNLE1BQU0sR0FBcUI7WUFDL0IsUUFBUSxFQUFFO2dCQUNSLFNBQVMsRUFBRSxDQUFDLEdBQUcsU0FBUyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLEtBQUssT0FBTyxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztnQkFDcEUsR0FBRyxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLEVBQUUsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2FBQ2xFO1NBQ0YsQ0FBQztRQUVGLE9BQU8sSUFBSSxRQUFRLENBQUMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQ3pDLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxZQUFZLENBQUMsUUFBNEIsRUFBRSxVQUF1QyxFQUFFO1FBQ2hHLE1BQU0sTUFBTSxHQUFxQjtZQUMvQixnQkFBZ0IsRUFBRTtnQkFDaEIsUUFBUSxFQUFFLEdBQUcsUUFBUSxFQUFFO2dCQUN2QixHQUFJLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztnQkFDckYsR0FBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsYUFBYSxFQUFFLE9BQU8sQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzthQUN6RTtTQUNGLENBQUM7UUFFRixPQUFPLElBQUksUUFBUSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUN6QyxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxNQUFNLENBQUMsV0FBVyxDQUFDLEdBQVcsRUFBRSxVQUFzQyxFQUFFO1FBRTdFLE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFL0IsSUFBSSxPQUFPLENBQUMsUUFBUSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQzlCLE1BQU0sSUFBSSxLQUFLLENBQUMsV0FBVyxHQUFHLGVBQWUsQ0FBQyxDQUFDO1NBQ2hEO1FBRUQsT0FBTyxRQUFRLENBQUMsU0FBUyxDQUFDLEtBQU0sQ0FBQyxDQUFDO0lBQ3BDLENBQUM7O0FBN0dILDRCQWdIQzs7O0FBRUQsSUFBWSxlQXdCWDtBQXhCRCxXQUFZLGVBQWU7SUFDekI7Ozs7Ozs7O09BUUc7SUFDSCxvQ0FBaUIsQ0FBQTtJQUVqQjs7Ozs7T0FLRztJQUNILGtEQUErQixDQUFBO0lBRS9COztPQUVHO0lBQ0gsa0NBQWUsQ0FBQTtBQUNqQixDQUFDLEVBeEJXLGVBQWUsR0FBZix1QkFBZSxLQUFmLHVCQUFlLFFBd0IxQjtBQWlNRDs7R0FFRztBQUNILE1BQWEsU0FBUztJQTBEcEIsWUFBWSxLQUFxQjtRQU5oQixXQUFNLEdBQW9CLEVBQUUsQ0FBQztRQU81QyxJQUFJLEtBQUssWUFBWSxTQUFTLEVBQUU7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDO1NBQ2hGO1FBRUQsTUFBTSxtQkFBbUIsR0FBdUI7WUFDOUMsR0FBRyxFQUFFO2dCQUNILE9BQU8sRUFBRSxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQztnQkFDekIsS0FBSyxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDO2FBQ3hCO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLE9BQU8sRUFBRSxZQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQztnQkFDNUIsS0FBSyxFQUFFLFlBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDO2FBQzVCO1NBQ0YsQ0FBQztRQUVGLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxLQUFLLENBQUMsVUFBVSxFQUFFO1lBQ2xDLE1BQU0sSUFBSSxLQUFLLENBQUMsK0ZBQStGLENBQUMsQ0FBQztTQUNsSDtRQUVELE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLElBQUksQ0FBQztRQUNsRCxNQUFNLHlCQUF5QixHQUFnQixLQUFLLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsQ0FBQyxDQUFDO1FBRS9GLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksSUFBSSxNQUFNLENBQUM7UUFDakMsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxVQUFVLEdBQUcsVUFBVSxDQUFDO1FBQzdCLElBQUksQ0FBQyxRQUFRLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM5QixJQUFJLENBQUMsS0FBSyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFDeEIsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2xDLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQztRQUNoQyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDM0YsSUFBSSxDQUFDLFVBQVUsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2xDLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsSUFBSSxtQkFBbUIsQ0FBQztRQUN4RCxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLENBQUM7UUFDbkMsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLEVBQUUsQ0FBQztRQUN2QyxJQUFJLENBQUMsZUFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLElBQUksZUFBZSxDQUFDLE1BQU0sQ0FBQztRQUN2RSxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksd0JBQXdCLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQzNFLElBQUksQ0FBQyxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLE9BQU8sSUFBSSxFQUFFLEVBQUUsS0FBSyxDQUFDLFlBQVksSUFBSSxFQUFFLENBQUMsQ0FBQztRQUVsRSxJQUFJLElBQUksQ0FBQyxVQUFVLEVBQUU7WUFDbkIsSUFBSSxDQUFDLE9BQU8sQ0FBQztnQkFDWCxNQUFNLEVBQUUsSUFBSSxDQUFDLFVBQVU7YUFDeEIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxLQUFLLE1BQU0sSUFBSSxJQUFJLEtBQUssQ0FBQyxLQUFLLElBQUksRUFBRSxFQUFFO1lBQ3BDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDcEI7SUFDSCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLElBQUk7UUFDYixPQUFPLElBQUksQ0FBQyxVQUFVLENBQUM7SUFDekIsQ0FBQztJQUVEOzs7T0FHRztJQUNILElBQVcsS0FBSztRQUNkLE9BQU8sQ0FBQyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUMxQixDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsSUFBVyxPQUFPO1FBQ2hCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO0lBQ3hELENBQUM7SUFFRDs7OztPQUlHO0lBQ0gsSUFBVyxJQUFJO1FBQ2IsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7SUFDbEQsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLEtBQUssQ0FBQyxJQUFZLEVBQUUsT0FBd0IsRUFBRSxVQUF3QixFQUFHO1FBQzlFLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxPQUFPLENBQUMsUUFBUSxFQUFFLEVBQUUsR0FBRyxPQUFPLEVBQUUsQ0FBQyxDQUFDO0lBQ3JFLENBQUM7SUFFRDs7T0FFRztJQUNJLE9BQU8sQ0FBQyxJQUFtQjtRQUVoQyxNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUMxRCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUUvQyxJQUFJLElBQUksQ0FBQyxJQUFJLElBQUksS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDMUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxDQUFDLElBQUksaUJBQWlCLENBQUMsQ0FBQztTQUMvRDtRQUVELElBQUksT0FBTyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsSUFBSSxDQUFDLE1BQU0saUJBQWlCLENBQUMsQ0FBQztTQUNuRTtRQUVELElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3pCLENBQUM7SUFFRDs7T0FFRztJQUNJLE9BQU87UUFDWixNQUFNLFlBQVksR0FBc0IsRUFBRSxDQUFDO1FBRTNDLEtBQUssTUFBTSxLQUFLLElBQUksSUFBSSxDQUFDLE1BQU0sRUFBRTtZQUMvQixZQUFZLENBQUMsSUFBSSxDQUFDO2dCQUNoQixJQUFJLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJO2dCQUN2QixTQUFTLEVBQUUsS0FBSyxDQUFDLElBQUk7Z0JBQ3JCLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtnQkFDeEIsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLFdBQVc7Z0JBQ25DLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztnQkFDdEIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO2FBQy9CLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxLQUFLLEVBQXFCLENBQUM7UUFFN0MsS0FBSyxNQUFNLElBQUksSUFBSSxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQzdCLEtBQUssQ0FBQyxJQUFJLENBQUM7Z0JBQ1QsYUFBYSxFQUFFLElBQUksQ0FBQyxNQUFNO2dCQUMxQixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDN0UsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO2dCQUNmLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtnQkFDdkIsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO2FBQ3BCLENBQUMsQ0FBQztTQUNKO1FBRUQsK0JBQStCO1FBQy9CLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUM7UUFDcEQsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLE1BQU0sQ0FBQztRQUN4RCxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsU0FBUyxFQUFFLE1BQU0sRUFBRSxLQUFLLENBQUM7UUFDbEQsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFNBQVMsRUFBRSxNQUFNLEVBQUUsT0FBTyxDQUFDO1FBQ3RELE1BQU0scUJBQXFCLEdBQUcsSUFBSSxDQUFDLFNBQVMsRUFBRSxnQkFBZ0IsRUFBRSxLQUFLLENBQUM7UUFDdEUsTUFBTSx1QkFBdUIsR0FBRyxJQUFJLENBQUMsU0FBUyxFQUFFLGdCQUFnQixFQUFFLE9BQU8sQ0FBQztRQUUxRSxNQUFNLE1BQU0sR0FBb0MsRUFBRSxDQUFDO1FBQ25ELE1BQU0sUUFBUSxHQUFvQyxFQUFFLENBQUM7UUFFckQsSUFBSSxRQUFRLEVBQUU7WUFDWixNQUFNLENBQUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ2hEO1FBQ0QsSUFBSSxXQUFXLEVBQUU7WUFDZixNQUFNLENBQUMsTUFBTSxHQUFHLEdBQUcsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxRQUFRLEVBQUUsR0FBRyxJQUFJLENBQUMsQ0FBQztTQUN0RjtRQUNELElBQUkscUJBQXFCLEVBQUU7WUFDekIsTUFBTSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMscUJBQXFCLENBQUMsV0FBVyxFQUFFLENBQUMsUUFBUSxFQUFFLEdBQUcsSUFBSSxDQUFDLENBQUM7U0FDOUc7UUFDRCxJQUFJLFVBQVUsRUFBRTtZQUNkLFFBQVEsQ0FBQyxHQUFHLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDcEQ7UUFDRCxJQUFJLGFBQWEsRUFBRTtZQUNqQixRQUFRLENBQUMsTUFBTSxHQUFHLEdBQUcsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxRQUFRLEVBQUUsR0FBRyxJQUFJLENBQUMsQ0FBQztTQUMxRjtRQUNELElBQUksdUJBQXVCLEVBQUU7WUFDM0IsUUFBUSxDQUFDLG1CQUFtQixDQUFDLEdBQUcsR0FBRyxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsdUJBQXVCLENBQUMsV0FBVyxFQUFFLENBQUMsUUFBUSxFQUFFLEdBQUcsSUFBSSxDQUFDLENBQUM7U0FDbEg7UUFFRCxJQUFJLG9CQUFvQixHQUF5QyxTQUFTLENBQUM7UUFDM0UsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ3RFLG9CQUFvQixHQUFHO2dCQUNyQixNQUFNLEVBQUUsd0JBQWdCLENBQUMsTUFBTSxDQUFDO2dCQUNoQyxRQUFRLEVBQUUsd0JBQWdCLENBQUMsUUFBUSxDQUFDO2FBQ3JDLENBQUM7U0FDSDtRQUVELE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBTyxFQUFFLENBQUM7UUFFL0IsT0FBTztZQUNMLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtZQUNmLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSztZQUNqQixlQUFlLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxlQUFlLENBQUM7WUFDbEUsS0FBSyxFQUFFLHdCQUFnQixDQUFDLEtBQUssQ0FBQztZQUM5QixZQUFZLEVBQUUsd0JBQWdCLENBQUMsWUFBWSxDQUFDO1lBQzVDLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztZQUNyQixJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7WUFDZixVQUFVLEVBQUUsSUFBSSxDQUFDLFVBQVU7WUFDM0IsR0FBRyxFQUFFLEdBQUcsQ0FBQyxTQUFTO1lBQ2xCLE9BQU8sRUFBRSxHQUFHLENBQUMsSUFBSTtZQUNqQixjQUFjLEVBQUUsSUFBSSxDQUFDLFVBQVUsRUFBRSxPQUFPLENBQUMsSUFBSSxDQUFDO1lBQzlDLGFBQWEsRUFBRSxJQUFJLENBQUMsU0FBUyxFQUFFLE9BQU8sQ0FBQyxJQUFJLENBQUM7WUFDNUMsWUFBWSxFQUFFLElBQUksQ0FBQyxRQUFRLEVBQUUsT0FBTyxDQUFDLElBQUksQ0FBQztZQUMxQyxTQUFTLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7Z0JBQzNCLFNBQVMsRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsSUFBSSxDQUFDO2dCQUNuRCxPQUFPLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLElBQUksQ0FBQzthQUNoRCxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2IsU0FBUyxFQUFFLG9CQUFvQjtZQUMvQixlQUFlLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxPQUFPLEVBQUU7U0FDaEQsQ0FBQztJQUNKLENBQUM7SUFFTyxzQkFBc0IsQ0FBQyxlQUFnQztRQUM3RCxRQUFRLGVBQWUsRUFBRTtZQUN2QixLQUFLLGVBQWUsQ0FBQyxNQUFNO2dCQUN6QixPQUFPLEdBQUcsQ0FBQyxzQ0FBc0MsQ0FBQyxNQUFNLENBQUM7WUFDM0QsS0FBSyxlQUFlLENBQUMsY0FBYztnQkFDakMsT0FBTyxHQUFHLENBQUMsc0NBQXNDLENBQUMsY0FBYyxDQUFDO1lBQ25FLEtBQUssZUFBZSxDQUFDLEtBQUs7Z0JBQ3hCLE9BQU8sR0FBRyxDQUFDLHNDQUFzQyxDQUFDLEtBQUssQ0FBQztZQUMxRDtnQkFDRSxNQUFNLElBQUksS0FBSyxDQUFDLGtDQUFrQyxlQUFlLEVBQUUsQ0FBQyxDQUFDO1NBQ3hFO0lBQ0gsQ0FBQztJQUVPLG1CQUFtQixDQUFDLFFBQWtCO1FBQzVDLFFBQVEsUUFBUSxFQUFFO1lBQ2hCLEtBQUssUUFBUSxDQUFDLElBQUk7Z0JBQ2hCLE9BQU8sR0FBRyxDQUFDLG1DQUFtQyxDQUFDLElBQUksQ0FBQztZQUN0RCxLQUFLLFFBQVEsQ0FBQyxHQUFHO2dCQUNmLE9BQU8sR0FBRyxDQUFDLG1DQUFtQyxDQUFDLEdBQUcsQ0FBQztZQUNyRCxLQUFLLFFBQVEsQ0FBQyxHQUFHO2dCQUNmLE9BQU8sR0FBRyxDQUFDLG1DQUFtQyxDQUFDLEdBQUcsQ0FBQztZQUNyRDtnQkFDRSxNQUFNLElBQUksS0FBSyxDQUFDLDhCQUE4QixRQUFRLEVBQUUsQ0FBQyxDQUFDO1NBQzdEO0lBQ0gsQ0FBQzs7QUE5UkgsOEJBZ1NDOzs7QUE2REQsSUFBWSxnQkFnRFg7QUFoREQsV0FBWSxnQkFBZ0I7SUFDMUI7Ozs7Ozs7OztPQVNHO0lBQ0gsaUNBQWEsQ0FBQTtJQUViOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSCx5REFBcUMsQ0FBQTtJQUVyQzs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FpQkc7SUFDSCxtREFBK0IsQ0FBQTtBQUNqQyxDQUFDLEVBaERXLGdCQUFnQixHQUFoQix3QkFBZ0IsS0FBaEIsd0JBQWdCLFFBZ0QzQjtBQW1CRDs7O0dBR0c7QUFDSCxNQUFhLEdBQUc7SUFRZCxZQUFvQixNQUFjO1FBQ2hDLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO0lBQ3ZCLENBQUM7SUFURCxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQWM7UUFDMUIsT0FBTyxJQUFJLEdBQUcsQ0FBQyxNQUFNLEdBQUcsR0FBRyxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUNELE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBYztRQUN6QixPQUFPLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7O0FBTkgsa0JBV0M7OztBQWtCRDs7R0FFRztBQUNILE1BQWEsT0FBTztJQUVsQixZQUNtQixTQUFnQyxFQUNoQyxNQUFlLEVBQ2YsR0FBb0I7UUFGcEIsY0FBUyxHQUFULFNBQVMsQ0FBdUI7UUFDaEMsV0FBTSxHQUFOLE1BQU0sQ0FBUztRQUNmLFFBQUcsR0FBSCxHQUFHLENBQWlCO0lBQUcsQ0FBQztJQUFBLENBQUM7SUFFNUM7O09BRUc7SUFDSSxPQUFPO1FBQ1osT0FBTztZQUNMLFlBQVksRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztnQkFDN0IsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSTthQUMxQixDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2IsU0FBUyxFQUFFLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO2dCQUNwQixJQUFJLEVBQUUsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJO2FBQ3BCLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDYixNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU07U0FDcEIsQ0FBQztJQUNKLENBQUM7O0FBcEJILDBCQXNCQzs7O0FBRUQsU0FBZ0IscUJBQXFCLENBQUMsUUFBYztJQUVsRCxJQUFJLENBQUMsUUFBUSxFQUFFO1FBQUUsT0FBTyxFQUFFLENBQUM7S0FBRTtJQUU3Qiw0REFBNEQ7SUFDNUQsMEJBQTBCO0lBQzFCLE1BQU0sVUFBVSxHQUFpQixRQUFnQixDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUM7SUFFbkUsT0FBTyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQzFDLENBQUM7QUFURCxzREFTQztBQUVEOztHQUVHO0FBQ0gsTUFBYSxHQUFHO0lBdUJkLFlBQW1CLE9BQWtCLEVBQUUsU0FBdUM7UUFDNUUsSUFBSSxDQUFDLFFBQVEsR0FBRyxPQUFPLENBQUM7UUFDeEIsSUFBSSxDQUFDLFVBQVUsR0FBRyxTQUFTLENBQUM7SUFDOUIsQ0FBQztJQXhCRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLGFBQWEsQ0FBQyxTQUErQixFQUFFLE1BQWU7UUFDMUUsT0FBTyxJQUFJLE9BQU8sQ0FBQyxTQUFTLEVBQUUsTUFBTSxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxJQUFvQjtRQUMzQyxPQUFPLElBQUksT0FBTyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDakQsQ0FBQztJQVVEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsSUFBWSxFQUFFLEtBQWU7UUFDOUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsR0FBRyxLQUFLLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7T0FHRztJQUNILElBQVcsU0FBUztRQUNsQixPQUFPLEVBQUUsR0FBRyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDaEMsQ0FBQztJQUVEOzs7T0FHRztJQUNJLFFBQVEsQ0FBQyxJQUFhO1FBQzNCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzNCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNILElBQVcsT0FBTztRQUNoQixPQUFPLENBQUMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDNUIsQ0FBQztJQUVPLFNBQVMsQ0FBQyxHQUFpQztRQUNqRCxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssRUFBYyxDQUFDO1FBQ3ZDLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQzNDLE1BQU0sQ0FBQyxJQUFJLENBQUM7Z0JBQ1YsSUFBSTtnQkFDSixLQUFLLEVBQUUsQ0FBQyxDQUFDLEtBQUs7Z0JBQ2QsU0FBUyxFQUFFLENBQUMsQ0FBQyxTQUFTO2FBQ3ZCLENBQUMsQ0FBQztTQUNKO1FBQ0QsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQztJQUVEOztPQUVHO0lBQ0ksT0FBTztRQUNaLE9BQU87WUFDTCxJQUFJLEVBQUUsd0JBQWdCLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztZQUMzRCxTQUFTLEVBQUUsd0JBQWdCLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDN0QsQ0FBQztJQUNKLENBQUM7O0FBbkZILGtCQW9GQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IFNpemUgfSBmcm9tICdjZGs4cyc7XG5pbXBvcnQgKiBhcyBjb25maWdtYXAgZnJvbSAnLi9jb25maWctbWFwJztcbmltcG9ydCAqIGFzIGhhbmRsZXIgZnJvbSAnLi9oYW5kbGVyJztcbmltcG9ydCAqIGFzIGs4cyBmcm9tICcuL2ltcG9ydHMvazhzJztcbmltcG9ydCAqIGFzIHByb2JlIGZyb20gJy4vcHJvYmUnO1xuaW1wb3J0ICogYXMgc2VjcmV0IGZyb20gJy4vc2VjcmV0JztcbmltcG9ydCB7IHVuZGVmaW5lZElmRW1wdHkgfSBmcm9tICcuL3V0aWxzJztcbmltcG9ydCAqIGFzIHZvbHVtZSBmcm9tICcuL3ZvbHVtZSc7XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgYENvbnRhaW5lclNlY3VyaXR5Q29udGV4dGBcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb250YWluZXJTZWN1cml0eUNvbnRleHRQcm9wcyB7XG5cbiAgLyoqXG4gICAgKiBUaGUgVUlEIHRvIHJ1biB0aGUgZW50cnlwb2ludCBvZiB0aGUgY29udGFpbmVyIHByb2Nlc3MuXG4gICAgKlxuICAgICogQGRlZmF1bHQgLSAyNTAwMC4gQW4gYXJiaXRyYXJ5IG51bWJlciBiaWdnZXIgdGhhbiA5OTk5IGlzIHNlbGVjdGVkIGhlcmUuXG4gICAgKiBUaGlzIGlzIHNvIHRoYXQgdGhlIGNvbnRhaW5lciBpcyBibG9ja2VkIHRvIGFjY2VzcyBob3N0IGZpbGVzIGV2ZW4gaWZcbiAgICAqIHNvbWVob3cgaXQgbWFuYWdlcyB0byBnZXQgYWNjZXNzIHRvIGhvc3QgZmlsZSBzeXN0ZW0uXG4gICAgKi9cbiAgcmVhZG9ubHkgdXNlcj86IG51bWJlcjtcblxuICAvKipcbiAgICAqIFRoZSBHSUQgdG8gcnVuIHRoZSBlbnRyeXBvaW50IG9mIHRoZSBjb250YWluZXIgcHJvY2Vzcy5cbiAgICAqXG4gICAgKiBAZGVmYXVsdCAtIDI2MDAwLiBBbiBhcmJpdHJhcnkgbnVtYmVyIGJpZ2dlciB0aGFuIDk5OTkgaXMgc2VsZWN0ZWQgaGVyZS5cbiAgICAqIFRoaXMgaXMgc28gdGhhdCB0aGUgY29udGFpbmVyIGlzIGJsb2NrZWQgdG8gYWNjZXNzIGhvc3QgZmlsZXMgZXZlbiBpZlxuICAgICogc29tZWhvdyBpdCBtYW5hZ2VzIHRvIGdldCBhY2Nlc3MgdG8gaG9zdCBmaWxlIHN5c3RlbS5cbiAgICAqL1xuICByZWFkb25seSBncm91cD86IG51bWJlcjtcblxuICAvKipcbiAgICAqIEluZGljYXRlcyB0aGF0IHRoZSBjb250YWluZXIgbXVzdCBydW4gYXMgYSBub24tcm9vdCB1c2VyLlxuICAgICogSWYgdHJ1ZSwgdGhlIEt1YmVsZXQgd2lsbCB2YWxpZGF0ZSB0aGUgaW1hZ2UgYXQgcnVudGltZSB0byBlbnN1cmUgdGhhdCBpdCBkb2VzXG4gICAgKiBub3QgcnVuIGFzIFVJRCAwIChyb290KSBhbmQgZmFpbCB0byBzdGFydCB0aGUgY29udGFpbmVyIGlmIGl0IGRvZXMuXG4gICAgKlxuICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAqL1xuICByZWFkb25seSBlbnN1cmVOb25Sb290PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogUnVuIGNvbnRhaW5lciBpbiBwcml2aWxlZ2VkIG1vZGUuIFByb2Nlc3NlcyBpbiBwcml2aWxlZ2VkIGNvbnRhaW5lcnMgYXJlIGVzc2VudGlhbGx5IGVxdWl2YWxlbnQgdG8gcm9vdCBvbiB0aGUgaG9zdC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHByaXZpbGVnZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgY29udGFpbmVyIGhhcyBhIHJlYWQtb25seSByb290IGZpbGVzeXN0ZW0uXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZWFkT25seVJvb3RGaWxlc3lzdGVtPzogYm9vbGVhbjtcblxufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBuZXR3b3JrIHBvcnQgaW4gYSBzaW5nbGUgY29udGFpbmVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbnRhaW5lclBvcnQge1xuXG4gIC8qKlxuICAgKiBOdW1iZXIgb2YgcG9ydCB0byBleHBvc2Ugb24gdGhlIHBvZCdzIElQIGFkZHJlc3MuIFRoaXMgbXVzdCBiZSBhIHZhbGlkIHBvcnQgbnVtYmVyLCAwIDwgeCA8IDY1NTM2LlxuICAgKi9cbiAgcmVhZG9ubHkgbnVtYmVyOiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFdoYXQgaG9zdCBJUCB0byBiaW5kIHRoZSBleHRlcm5hbCBwb3J0IHRvLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIDEyNy4wLjAuMS5cbiAgICovXG4gIHJlYWRvbmx5IGhvc3RJcD86IHN0cmluZztcblxuICAvKipcbiAgICogTnVtYmVyIG9mIHBvcnQgdG8gZXhwb3NlIG9uIHRoZSBob3N0LiBJZiBzcGVjaWZpZWQsIHRoaXMgbXVzdCBiZSBhIHZhbGlkIHBvcnQgbnVtYmVyLCAwIDwgeCA8IDY1NTM2LlxuICAgKiBNb3N0IGNvbnRhaW5lcnMgZG8gbm90IG5lZWQgdGhpcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhdXRvIGdlbmVyYXRlZCBieSBrdWJlcm5ldGVzIGFuZCBtaWdodCBjaGFuZ2Ugb24gcmVzdGFydHMuXG4gICAqL1xuICByZWFkb25seSBob3N0UG9ydD86IG51bWJlcjtcblxuICAvKipcbiAgICogSWYgc3BlY2lmaWVkLCB0aGlzIG11c3QgYmUgYW4gSUFOQV9TVkNfTkFNRSBhbmQgdW5pcXVlIHdpdGhpbiB0aGUgcG9kLlxuICAgKiBFYWNoIG5hbWVkIHBvcnQgaW4gYSBwb2QgbXVzdCBoYXZlIGEgdW5pcXVlIG5hbWUuXG4gICAqIE5hbWUgZm9yIHRoZSBwb3J0IHRoYXQgY2FuIGJlIHJlZmVycmVkIHRvIGJ5IHNlcnZpY2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHBvcnQgaXMgbm90IG5hbWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogUHJvdG9jb2wgZm9yIHBvcnQuIE11c3QgYmUgVURQLCBUQ1AsIG9yIFNDVFAuIERlZmF1bHRzIHRvIFwiVENQXCIuXG4gICAqXG4gICAqIEBkZWZhdWx0IFByb3RvY29sLlRDUFxuICAgKi9cbiAgcmVhZG9ubHkgcHJvdG9jb2w/OiBQcm90b2NvbDtcblxufVxuXG4vKipcbiAqIE5ldHdvcmsgcHJvdG9jb2xzLlxuICovXG5leHBvcnQgZW51bSBQcm90b2NvbCB7XG5cbiAgLyoqXG4gICAqIFRDUC5cbiAgICovXG4gIFRDUCA9ICdUQ1AnLFxuXG4gIC8qKlxuICAgKiBVRFAuXG4gICAqL1xuICBVRFAgPSAnVURQJyxcblxuICAvKipcbiAgICogU0NUUC5cbiAgICovXG4gIFNDVFAgPSAnU0NUUCdcbn1cblxuXG4vKipcbiAqIENvbnRhaW5lciBzZWN1cml0eSBhdHRyaWJ1dGVzIGFuZCBzZXR0aW5ncy5cbiAqL1xuZXhwb3J0IGNsYXNzIENvbnRhaW5lclNlY3VyaXR5Q29udGV4dCB7XG5cbiAgcHVibGljIHJlYWRvbmx5IGVuc3VyZU5vblJvb3Q6IGJvb2xlYW47XG4gIHB1YmxpYyByZWFkb25seSBwcml2aWxlZ2VkOiBib29sZWFuO1xuICBwdWJsaWMgcmVhZG9ubHkgcmVhZE9ubHlSb290RmlsZXN5c3RlbTogYm9vbGVhbjtcbiAgcHVibGljIHJlYWRvbmx5IHVzZXI/OiBudW1iZXI7XG4gIHB1YmxpYyByZWFkb25seSBncm91cD86IG51bWJlcjtcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogQ29udGFpbmVyU2VjdXJpdHlDb250ZXh0UHJvcHMgPSB7fSkge1xuICAgIHRoaXMuZW5zdXJlTm9uUm9vdCA9IHByb3BzLmVuc3VyZU5vblJvb3QgPz8gZmFsc2U7XG4gICAgdGhpcy5wcml2aWxlZ2VkID0gcHJvcHMucHJpdmlsZWdlZCA/PyBmYWxzZTtcbiAgICB0aGlzLnJlYWRPbmx5Um9vdEZpbGVzeXN0ZW0gPSBwcm9wcy5yZWFkT25seVJvb3RGaWxlc3lzdGVtID8/IGZhbHNlO1xuICAgIHRoaXMudXNlciA9IHByb3BzLnVzZXIgPz8gMjUwMDA7XG4gICAgdGhpcy5ncm91cCA9IHByb3BzLmdyb3VwID8/IDI2MDAwO1xuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLlNlY3VyaXR5Q29udGV4dCB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHJ1bkFzR3JvdXA6IHRoaXMuZ3JvdXAsXG4gICAgICBydW5Bc1VzZXI6IHRoaXMudXNlcixcbiAgICAgIHJ1bkFzTm9uUm9vdDogdGhpcy5lbnN1cmVOb25Sb290LFxuICAgICAgcHJpdmlsZWdlZDogdGhpcy5wcml2aWxlZ2VkLFxuICAgICAgcmVhZE9ubHlSb290RmlsZXN5c3RlbTogdGhpcy5yZWFkT25seVJvb3RGaWxlc3lzdGVtLFxuICAgIH07XG4gIH1cblxufVxuXG5leHBvcnQgZW51bSBFbnZGaWVsZFBhdGhzIHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBwb2QuXG4gICAqL1xuICBQT0RfTkFNRSA9ICdtZXRhZGF0YS5uYW1lJyxcblxuICAvKipcbiAgICogVGhlIG5hbWVzcGFjZSBvZiB0aGUgcG9kLlxuICAgKi9cbiAgUE9EX05BTUVTUEFDRSA9ICdtZXRhZGF0YS5uYW1lc3BhY2UnLFxuXG4gIC8qKlxuICAgKiBUaGUgdWlkIG9mIHRoZSBwb2QuXG4gICAqL1xuICBQT0RfVUlEID0gJ21ldGFkYXRhLnVpZCcsXG5cbiAgLyoqXG4gICAqIFRoZSBsYWJlbHMgb2YgdGhlIHBvZC5cbiAgICovXG4gIFBPRF9MQUJFTCA9ICdtZXRhZGF0YS5sYWJlbHMnLFxuXG4gIC8qKlxuICAgKiBUaGUgYW5ub3RhdGlvbnMgb2YgdGhlIHBvZC5cbiAgICovXG4gIFBPRF9BTk5PVEFUSU9OID0gJ21ldGFkYXRhLmFubm90YXRpb25zJyxcblxuICAvKipcbiAgICogVGhlIGlwQWRkcmVzcyBvZiB0aGUgcG9kLlxuICAgKi9cbiAgUE9EX0lQID0gJ3N0YXR1cy5wb2RJUCcsXG5cbiAgLyoqXG4gICAqIFRoZSBzZXJ2aWNlIGFjY291bnQgbmFtZSBvZiB0aGUgcG9kLlxuICAgKi9cbiAgU0VSVklDRV9BQ0NPVU5UX05BTUUgPSAnc3BlYy5zZXJ2aWNlQWNjb3VudE5hbWUnLFxuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgbm9kZS5cbiAgICovXG4gIE5PREVfTkFNRSA9ICdzcGVjLm5vZGVOYW1lJyxcblxuICAvKipcbiAgICogVGhlIGlwQWRkcmVzcyBvZiB0aGUgbm9kZS5cbiAgICovXG4gIE5PREVfSVAgPSAnc3RhdHVzLmhvc3RJUCcsXG5cbiAgLyoqXG4gICAqIFRoZSBpcEFkZHJlc2VzcyBvZiB0aGUgcG9kLlxuICAgKi9cbiAgUE9EX0lQUyA9ICdzdGF0dXMucG9kSVBzJyxcbn1cblxuZXhwb3J0IGVudW0gUmVzb3VyY2VGaWVsZFBhdGhzIHtcbiAgLyoqXG4gICAqIENQVSBsaW1pdCBvZiB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgQ1BVX0xJTUlUID0gJ2xpbWl0cy5jcHUnLFxuXG4gIC8qKlxuICAgKiBNZW1vcnkgbGltaXQgb2YgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIE1FTU9SWV9MSU1JVCA9ICdsaW1pdHMubWVtb3J5JyxcblxuICAvKipcbiAgICogQ1BVIHJlcXVlc3Qgb2YgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIENQVV9SRVFVRVNUID0gJ3JlcXVlc3RzLmNwdScsXG5cbiAgLyoqXG4gICAqIE1lbW9yeSByZXF1ZXN0IG9mIHRoZSBjb250YWluZXIuXG4gICAqL1xuICBNRU1PUllfUkVRVUVTVCA9ICdyZXF1ZXN0cy5tZW1vcnknLFxuXG4gIC8qKlxuICAgKiBFcGhlbWVyYWwgc3RvcmFnZSBsaW1pdCBvZiB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgU1RPUkFHRV9MSU1JVCA9ICdsaW1pdHMuZXBoZW1lcmFsLXN0b3JhZ2UnLFxuXG4gIC8qKlxuICAgKiBFcGhlbWVyYWwgc3RvcmFnZSByZXF1ZXN0IG9mIHRoZSBjb250YWluZXIuXG4gICAqL1xuICBTVE9SQUdFX1JFUVVFU1QgPSAncmVxdWVzdHMuZXBoZW1lcmFsLXN0b3JhZ2UnLFxufVxuXG4vKipcbiAqIE9wdGlvbnMgdG8gc3BlY2lmeSBhbiBlbnZpb25tZW50IHZhcmlhYmxlIHZhbHVlIGZyb20gYSBDb25maWdNYXAga2V5LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVudlZhbHVlRnJvbUNvbmZpZ01hcE9wdGlvbnMge1xuXG4gIC8qKlxuICAgKiBTcGVjaWZ5IHdoZXRoZXIgdGhlIENvbmZpZ01hcCBvciBpdHMga2V5IG11c3QgYmUgZGVmaW5lZC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG9wdGlvbmFsPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBPcHRpb25zIHRvIHNwZWNpZnkgYW4gZW52aXJvbm1lbnQgdmFyaWFibGUgdmFsdWUgZnJvbSBhIFNlY3JldC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbnZWYWx1ZUZyb21TZWNyZXRPcHRpb25zIHtcblxuICAvKipcbiAgICogU3BlY2lmeSB3aGV0aGVyIHRoZSBTZWNyZXQgb3IgaXRzIGtleSBtdXN0IGJlIGRlZmluZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBvcHRpb25hbD86IGJvb2xlYW47XG5cbn1cblxuLyoqXG4gKiBPcHRpb25zIHRvIHNwZWNpZnkgYW4gZW52aXJvbm1lbnQgdmFyaWFibGUgdmFsdWUgZnJvbSB0aGUgcHJvY2VzcyBlbnZpcm9ubWVudC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbnZWYWx1ZUZyb21Qcm9jZXNzT3B0aW9ucyB7XG5cbiAgLyoqXG4gICAqIFNwZWNpZnkgd2hldGhlciB0aGUga2V5IG11c3QgZXhpc3QgaW4gdGhlIGVudmlyb25tZW50LlxuICAgKiBJZiB0aGlzIGlzIHNldCB0byB0cnVlLCBhbmQgdGhlIGtleSBkb2VzIG5vdCBleGlzdCwgYW4gZXJyb3Igd2lsbCB0aHJvd24uXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZXF1aXJlZD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogT3B0aW9ucyB0byBzcGVjaWZ5IGFuIGVudmlyb25tZW50IHZhcmlhYmxlIHZhbHVlIGZyb20gYSBmaWVsZCByZWZlcmVuY2UuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRW52VmFsdWVGcm9tRmllbGRSZWZPcHRpb25zIHtcbiAgLyoqXG4gICAqIFZlcnNpb24gb2YgdGhlIHNjaGVtYSB0aGUgRmllbGRQYXRoIGlzIHdyaXR0ZW4gaW4gdGVybXMgb2YuXG4gICAqL1xuICByZWFkb25seSBhcGlWZXJzaW9uPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUga2V5IHRvIHNlbGVjdCB0aGUgcG9kIGxhYmVsIG9yIGFubm90YXRpb24uXG4gICAqL1xuICByZWFkb25seSBrZXk/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogT3B0aW9ucyB0byBzcGVjaWZ5IGFuIGVudmlyb25tZW50IHZhcmlhYmxlIHZhbHVlIGZyb20gYSByZXNvdXJjZS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbnZWYWx1ZUZyb21SZXNvdXJjZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIGNvbnRhaW5lciB0byBzZWxlY3QgdGhlIHZhbHVlIGZyb20uXG4gICAqL1xuICByZWFkb25seSBjb250YWluZXI/OiBDb250YWluZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBvdXRwdXQgZm9ybWF0IG9mIHRoZSBleHBvc2VkIHJlc291cmNlLlxuICAgKi9cbiAgcmVhZG9ubHkgZGl2aXNvcj86IHN0cmluZztcbn1cblxuLyoqXG4gKiBVdGlsaXR5IGNsYXNzIGZvciBjcmVhdGluZyByZWFkaW5nIGVudiB2YWx1ZXMgZnJvbSB2YXJpb3VzIHNvdXJjZXMuXG4gKi9cbmV4cG9ydCBjbGFzcyBFbnZWYWx1ZSB7XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHZhbHVlIGJ5IHJlYWRpbmcgYSBzcGVjaWZpYyBrZXkgaW5zaWRlIGEgY29uZmlnIG1hcC5cbiAgICpcbiAgICogQHBhcmFtIGNvbmZpZ01hcCAtIFRoZSBjb25maWcgbWFwLlxuICAgKiBAcGFyYW0ga2V5IC0gVGhlIGtleSB0byBleHRyYWN0IHRoZSB2YWx1ZSBmcm9tLlxuICAgKiBAcGFyYW0gb3B0aW9ucyAtIEFkZGl0aW9uYWwgb3B0aW9ucy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUNvbmZpZ01hcChjb25maWdNYXA6IGNvbmZpZ21hcC5JQ29uZmlnTWFwLCBrZXk6IHN0cmluZywgb3B0aW9uczogRW52VmFsdWVGcm9tQ29uZmlnTWFwT3B0aW9ucyA9IHsgfSk6IEVudlZhbHVlIHtcblxuICAgIGNvbnN0IHNvdXJjZTogazhzLkVudlZhclNvdXJjZSA9IHtcbiAgICAgIGNvbmZpZ01hcEtleVJlZjoge1xuICAgICAgICBuYW1lOiBjb25maWdNYXAubmFtZSxcbiAgICAgICAga2V5LFxuICAgICAgICBvcHRpb25hbDogb3B0aW9ucy5vcHRpb25hbCxcbiAgICAgIH0sXG4gICAgfTtcbiAgICByZXR1cm4gbmV3IEVudlZhbHVlKHVuZGVmaW5lZCwgc291cmNlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWZpbmVzIGFuIGVudmlyb25tZW50IHZhbHVlIGZyb20gYSBzZWNyZXQgSlNPTiB2YWx1ZS5cbiAgICpcbiAgICogQHBhcmFtIHNlY3JldFZhbHVlIFRoZSBzZWNyZXQgdmFsdWUgKHNlY3JlbnQgKyBrZXkpXG4gICAqIEBwYXJhbSBvcHRpb25zIEFkZGl0aW9uYWwgb3B0aW9uc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tU2VjcmV0VmFsdWUoc2VjcmV0VmFsdWU6IHNlY3JldC5TZWNyZXRWYWx1ZSwgb3B0aW9uczogRW52VmFsdWVGcm9tU2VjcmV0T3B0aW9ucyA9IHt9KTogRW52VmFsdWUge1xuICAgIGNvbnN0IHNvdXJjZTogazhzLkVudlZhclNvdXJjZSA9IHtcbiAgICAgIHNlY3JldEtleVJlZjoge1xuICAgICAgICBuYW1lOiBzZWNyZXRWYWx1ZS5zZWNyZXQubmFtZSxcbiAgICAgICAga2V5OiBzZWNyZXRWYWx1ZS5rZXksXG4gICAgICAgIG9wdGlvbmFsOiBvcHRpb25zLm9wdGlvbmFsLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgcmV0dXJuIG5ldyBFbnZWYWx1ZSh1bmRlZmluZWQsIHNvdXJjZSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgdmFsdWUgZnJvbSB0aGUgZ2l2ZW4gYXJndW1lbnQuXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSAtIFRoZSB2YWx1ZS5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVZhbHVlKHZhbHVlOiBzdHJpbmcpOiBFbnZWYWx1ZSB7XG4gICAgcmV0dXJuIG5ldyBFbnZWYWx1ZSh2YWx1ZSk7XG4gIH1cblxuICAvKipcbiAgICpcbiAgICogQ3JlYXRlIGEgdmFsdWUgZnJvbSBhIGZpZWxkIHJlZmVyZW5jZS5cbiAgICpcbiAgICogQHBhcmFtIGZpZWxkUGF0aDogVGhlIGZpZWxkIHJlZmVyZW5jZS5cbiAgICogQHBhcmFtIG9wdGlvbnM6IEFkZGl0aW9uYWwgb3B0aW9ucy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUZpZWxkUmVmKGZpZWxkUGF0aDogRW52RmllbGRQYXRocywgb3B0aW9uczogRW52VmFsdWVGcm9tRmllbGRSZWZPcHRpb25zID0ge30pOiBFbnZWYWx1ZSB7XG4gICAgbGV0IG5lZWRzS2V5OiBib29sZWFuID0gZmFsc2U7XG5cbiAgICBpZiAoZmllbGRQYXRoID09PSBFbnZGaWVsZFBhdGhzLlBPRF9MQUJFTCB8fCBmaWVsZFBhdGggPT09IEVudkZpZWxkUGF0aHMuUE9EX0FOTk9UQVRJT04pIHtcbiAgICAgIG5lZWRzS2V5ID0gdHJ1ZTtcbiAgICB9XG5cbiAgICBpZiAobmVlZHNLZXkgJiYgb3B0aW9ucy5rZXkgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGAke2ZpZWxkUGF0aH0gcmVxdWlyZXMgYSBrZXlgKTtcbiAgICB9XG5cbiAgICBjb25zdCBzb3VyY2U6IGs4cy5FbnZWYXJTb3VyY2UgPSB7XG4gICAgICBmaWVsZFJlZjoge1xuICAgICAgICBmaWVsZFBhdGg6IChgJHtmaWVsZFBhdGh9YCArIChuZWVkc0tleSA/IGBbJyR7b3B0aW9ucy5rZXl9J11gIDogJycpKSxcbiAgICAgICAgLi4uKG9wdGlvbnMuYXBpVmVyc2lvbiA/IHsgYXBpVmVyc2lvbjogb3B0aW9ucy5hcGlWZXJzaW9uIH0gOiB7fSksXG4gICAgICB9LFxuICAgIH07XG5cbiAgICByZXR1cm4gbmV3IEVudlZhbHVlKHVuZGVmaW5lZCwgc291cmNlKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSB2YWx1ZSBmcm9tIGEgcmVzb3VyY2UuXG4gICAqXG4gICAqIEBwYXJhbSByZXNvdXJjZTogUmVzb3VyY2UgdG8gc2VsZWN0IHRoZSB2YWx1ZSBmcm9tLlxuICAgKiBAcGFyYW0gb3B0aW9uczogQWRkaXRpb25hbCBvcHRpb25zLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tUmVzb3VyY2UocmVzb3VyY2U6IFJlc291cmNlRmllbGRQYXRocywgb3B0aW9uczogRW52VmFsdWVGcm9tUmVzb3VyY2VPcHRpb25zID0ge30pOiBFbnZWYWx1ZSB7XG4gICAgY29uc3Qgc291cmNlOiBrOHMuRW52VmFyU291cmNlID0ge1xuICAgICAgcmVzb3VyY2VGaWVsZFJlZjoge1xuICAgICAgICByZXNvdXJjZTogYCR7cmVzb3VyY2V9YCxcbiAgICAgICAgLi4uIChvcHRpb25zLmRpdmlzb3IgPyB7IGRpdmlzb3I6IGs4cy5JbnRPclN0cmluZy5mcm9tU3RyaW5nKG9wdGlvbnMuZGl2aXNvcikgfSA6IHt9KSxcbiAgICAgICAgLi4uIChvcHRpb25zLmNvbnRhaW5lciA/IHsgY29udGFpbmVyTmFtZTogb3B0aW9ucy5jb250YWluZXIubmFtZSB9IDoge30pLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgcmV0dXJuIG5ldyBFbnZWYWx1ZSh1bmRlZmluZWQsIHNvdXJjZSk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgdmFsdWUgZnJvbSBhIGtleSBpbiB0aGUgY3VycmVudCBwcm9jZXNzIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAcGFyYW0ga2V5IC0gVGhlIGtleSB0byByZWFkLlxuICAgKiBAcGFyYW0gb3B0aW9ucyAtIEFkZGl0aW9uYWwgb3B0aW9ucy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVByb2Nlc3Moa2V5OiBzdHJpbmcsIG9wdGlvbnM6IEVudlZhbHVlRnJvbVByb2Nlc3NPcHRpb25zID0ge30pOiBFbnZWYWx1ZSB7XG5cbiAgICBjb25zdCB2YWx1ZSA9IHByb2Nlc3MuZW52W2tleV07XG5cbiAgICBpZiAob3B0aW9ucy5yZXF1aXJlZCAmJiAhdmFsdWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTWlzc2luZyAke2tleX0gZW52IHZhcmlhYmxlYCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIEVudlZhbHVlLmZyb21WYWx1ZSh2YWx1ZSEpO1xuICB9XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgdmFsdWU/OiBhbnksIHB1YmxpYyByZWFkb25seSB2YWx1ZUZyb20/OiBhbnkpIHt9XG59XG5cbmV4cG9ydCBlbnVtIEltYWdlUHVsbFBvbGljeSB7XG4gIC8qKlxuICAgKiBFdmVyeSB0aW1lIHRoZSBrdWJlbGV0IGxhdW5jaGVzIGEgY29udGFpbmVyLCB0aGUga3ViZWxldCBxdWVyaWVzIHRoZSBjb250YWluZXIgaW1hZ2UgcmVnaXN0cnlcbiAgICogdG8gcmVzb2x2ZSB0aGUgbmFtZSB0byBhbiBpbWFnZSBkaWdlc3QuIElmIHRoZSBrdWJlbGV0IGhhcyBhIGNvbnRhaW5lciBpbWFnZSB3aXRoIHRoYXQgZXhhY3RcbiAgICogZGlnZXN0IGNhY2hlZCBsb2NhbGx5LCB0aGUga3ViZWxldCB1c2VzIGl0cyBjYWNoZWQgaW1hZ2U7IG90aGVyd2lzZSwgdGhlIGt1YmVsZXQgZG93bmxvYWRzXG4gICAqIChwdWxscykgdGhlIGltYWdlIHdpdGggdGhlIHJlc29sdmVkIGRpZ2VzdCwgYW5kIHVzZXMgdGhhdCBpbWFnZSB0byBsYXVuY2ggdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogRGVmYXVsdCBpcyBBbHdheXMgaWYgSW1hZ2VQdWxsUG9saWN5IGlzIG9taXR0ZWQgYW5kIGVpdGhlciB0aGUgaW1hZ2UgdGFnIGlzIDpsYXRlc3Qgb3JcbiAgICogdGhlIGltYWdlIHRhZyBpcyBvbWl0dGVkLlxuICAgKi9cbiAgQUxXQVlTID0gJ0Fsd2F5cycsXG5cbiAgLyoqXG4gICAqIFRoZSBpbWFnZSBpcyBwdWxsZWQgb25seSBpZiBpdCBpcyBub3QgYWxyZWFkeSBwcmVzZW50IGxvY2FsbHkuXG4gICAqXG4gICAqIERlZmF1bHQgaXMgSWZOb3RQcmVzZW50IGlmIEltYWdlUHVsbFBvbGljeSBpcyBvbWl0dGVkIGFuZCB0aGUgaW1hZ2UgdGFnIGlzIHByZXNlbnQgYnV0XG4gICAqIG5vdCA6bGF0ZXN0XG4gICAqL1xuICBJRl9OT1RfUFJFU0VOVCA9ICdJZk5vdFByZXNlbnQnLFxuXG4gIC8qKlxuICAgKiBUaGUgaW1hZ2UgaXMgYXNzdW1lZCB0byBleGlzdCBsb2NhbGx5LiBObyBhdHRlbXB0IGlzIG1hZGUgdG8gcHVsbCB0aGUgaW1hZ2UuXG4gICAqL1xuICBORVZFUiA9ICdOZXZlcicsXG59XG5cbi8qKlxuICogQ29udGFpbmVyIGxpZmVjeWNsZSBwcm9wZXJ0aWVzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbnRhaW5lckxpZmVjeWNsZSB7XG5cbiAgLyoqXG4gICAqIFRoaXMgaG9vayBpcyBleGVjdXRlZCBpbW1lZGlhdGVseSBhZnRlciBhIGNvbnRhaW5lciBpcyBjcmVhdGVkLiBIb3dldmVyLFxuICAgKiB0aGVyZSBpcyBubyBndWFyYW50ZWUgdGhhdCB0aGUgaG9vayB3aWxsIGV4ZWN1dGUgYmVmb3JlIHRoZSBjb250YWluZXIgRU5UUllQT0lOVC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwb3N0IHN0YXJ0IGhhbmRsZXIuXG4gICAqL1xuICByZWFkb25seSBwb3N0U3RhcnQ/OiBoYW5kbGVyLkhhbmRsZXI7XG5cbiAgLyoqXG4gICAqIFRoaXMgaG9vayBpcyBjYWxsZWQgaW1tZWRpYXRlbHkgYmVmb3JlIGEgY29udGFpbmVyIGlzIHRlcm1pbmF0ZWQgZHVlIHRvIGFuIEFQSSByZXF1ZXN0IG9yIG1hbmFnZW1lbnRcbiAgICogZXZlbnQgc3VjaCBhcyBhIGxpdmVuZXNzL3N0YXJ0dXAgcHJvYmUgZmFpbHVyZSwgcHJlZW1wdGlvbiwgcmVzb3VyY2UgY29udGVudGlvbiBhbmQgb3RoZXJzLlxuICAgKiBBIGNhbGwgdG8gdGhlIFByZVN0b3AgaG9vayBmYWlscyBpZiB0aGUgY29udGFpbmVyIGlzIGFscmVhZHkgaW4gYSB0ZXJtaW5hdGVkIG9yIGNvbXBsZXRlZCBzdGF0ZVxuICAgKiBhbmQgdGhlIGhvb2sgbXVzdCBjb21wbGV0ZSBiZWZvcmUgdGhlIFRFUk0gc2lnbmFsIHRvIHN0b3AgdGhlIGNvbnRhaW5lciBjYW4gYmUgc2VudC5cbiAgICogVGhlIFBvZCdzIHRlcm1pbmF0aW9uIGdyYWNlIHBlcmlvZCBjb3VudGRvd24gYmVnaW5zIGJlZm9yZSB0aGUgUHJlU3RvcCBob29rIGlzIGV4ZWN1dGVkLFxuICAgKiBzbyByZWdhcmRsZXNzIG9mIHRoZSBvdXRjb21lIG9mIHRoZSBoYW5kbGVyLCB0aGUgY29udGFpbmVyIHdpbGwgZXZlbnR1YWxseSB0ZXJtaW5hdGVcbiAgICogd2l0aGluIHRoZSBQb2QncyB0ZXJtaW5hdGlvbiBncmFjZSBwZXJpb2QuIE5vIHBhcmFtZXRlcnMgYXJlIHBhc3NlZCB0byB0aGUgaGFuZGxlci5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy93b3JrbG9hZHMvcG9kcy9wb2QtbGlmZWN5Y2xlLyNwb2QtdGVybWluYXRpb25cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBwcmUgc3RvcCBoYW5kbGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgcHJlU3RvcD86IGhhbmRsZXIuSGFuZGxlcjtcblxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNyZWF0aW5nIGEgY29udGFpbmVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbnRhaW5lclByb3BzIHtcblxuICAvKipcbiAgICogRG9ja2VyIGltYWdlIG5hbWUuXG4gICAqL1xuICByZWFkb25seSBpbWFnZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBjb250YWluZXIgc3BlY2lmaWVkIGFzIGEgRE5TX0xBQkVMLiBFYWNoIGNvbnRhaW5lciBpbiBhIHBvZCBtdXN0IGhhdmUgYSB1bmlxdWUgbmFtZSAoRE5TX0xBQkVMKS4gQ2Fubm90IGJlIHVwZGF0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0ICdtYWluJ1xuICAgKi9cbiAgcmVhZG9ubHkgbmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgLSB1c2UgYHBvcnROdW1iZXJgLlxuICAgKi9cbiAgcmVhZG9ubHkgcG9ydD86IG51bWJlcjtcblxuICAvKipcbiAgICogTnVtYmVyIG9mIHBvcnQgdG8gZXhwb3NlIG9uIHRoZSBwb2QncyBJUCBhZGRyZXNzLiBUaGlzIG11c3QgYmUgYSB2YWxpZCBwb3J0IG51bWJlciwgMCA8IHggPCA2NTUzNi5cbiAgICpcbiAgICogVGhpcyBpcyBhIGNvbnZpbmllbmNlIHByb3BlcnR5IGlmIGFsbCB5b3UgbmVlZCBhIHNpbmdsZSBUQ1AgbnVtYmVyZWQgcG9ydC5cbiAgICogSW4gY2FzZSBtb3JlIGFkdmFuY2VkIGNvbmZpZ3VhcnRpb24gaXMgcmVxdWlyZWQsIHVzZSB0aGUgYHBvcnRzYCBwcm9wZXJ0eS5cbiAgICpcbiAgICogVGhpcyBwb3J0IGlzIGFkZGVkIHRvIHRoZSBsaXN0IG9mIHBvcnRzIG1lbnRpb25lZCBpbiB0aGUgYHBvcnRzYCBwcm9wZXJ0eS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBPbmx5IHRoZSBwb3J0cyBtZW50aW9uZCBpbiB0aGUgYHBvcnRzYCBwcm9wZXJ0eSBhcmUgZXhwb3NlZC5cbiAgICovXG4gIHJlYWRvbmx5IHBvcnROdW1iZXI/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcG9ydHMgdG8gZXhwb3NlIGZyb20gdGhpcyBjb250YWluZXIuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gT25seSB0aGUgcG9ydCBtZW50aW9uZWQgaW4gdGhlIGBwb3J0TnVtYmVyYCBwcm9wZXJ0eSBpcyBleHBvc2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgcG9ydHM/OiBDb250YWluZXJQb3J0W107XG5cbiAgLyoqXG4gICAqIEVudHJ5cG9pbnQgYXJyYXkuIE5vdCBleGVjdXRlZCB3aXRoaW4gYSBzaGVsbC4gVGhlIGRvY2tlciBpbWFnZSdzIEVOVFJZUE9JTlQgaXMgdXNlZCBpZiB0aGlzIGlzIG5vdCBwcm92aWRlZC4gVmFyaWFibGUgcmVmZXJlbmNlcyAkKFZBUl9OQU1FKSBhcmUgZXhwYW5kZWQgdXNpbmcgdGhlIGNvbnRhaW5lcidzIGVudmlyb25tZW50LlxuICAgKiBJZiBhIHZhcmlhYmxlIGNhbm5vdCBiZSByZXNvbHZlZCwgdGhlIHJlZmVyZW5jZSBpbiB0aGUgaW5wdXQgc3RyaW5nIHdpbGwgYmUgdW5jaGFuZ2VkLiBUaGUgJChWQVJfTkFNRSkgc3ludGF4IGNhbiBiZSBlc2NhcGVkIHdpdGggYSBkb3VibGUgJCQsIGllOiAkJChWQVJfTkFNRSkuXG4gICAqIEVzY2FwZWQgcmVmZXJlbmNlcyB3aWxsIG5ldmVyIGJlIGV4cGFuZGVkLCByZWdhcmRsZXNzIG9mIHdoZXRoZXIgdGhlIHZhcmlhYmxlIGV4aXN0cyBvciBub3QuIENhbm5vdCBiZSB1cGRhdGVkLlxuICAgKiBNb3JlIGluZm86IGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL3Rhc2tzL2luamVjdC1kYXRhLWFwcGxpY2F0aW9uL2RlZmluZS1jb21tYW5kLWFyZ3VtZW50LWNvbnRhaW5lci8jcnVubmluZy1hLWNvbW1hbmQtaW4tYS1zaGVsbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFRoZSBkb2NrZXIgaW1hZ2UncyBFTlRSWVBPSU5ULlxuICAgKi9cbiAgcmVhZG9ubHkgY29tbWFuZD86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBBcmd1bWVudHMgdG8gdGhlIGVudHJ5cG9pbnQuIFRoZSBkb2NrZXIgaW1hZ2UncyBDTUQgaXMgdXNlZCBpZiBgY29tbWFuZGAgaXNcbiAgICogbm90IHByb3ZpZGVkLlxuICAgKlxuICAgKiBWYXJpYWJsZSByZWZlcmVuY2VzICQoVkFSX05BTUUpIGFyZSBleHBhbmRlZCB1c2luZyB0aGUgY29udGFpbmVyJ3NcbiAgICogZW52aXJvbm1lbnQuIElmIGEgdmFyaWFibGUgY2Fubm90IGJlIHJlc29sdmVkLCB0aGUgcmVmZXJlbmNlIGluIHRoZSBpbnB1dFxuICAgKiBzdHJpbmcgd2lsbCBiZSB1bmNoYW5nZWQuIFRoZSAkKFZBUl9OQU1FKSBzeW50YXggY2FuIGJlIGVzY2FwZWQgd2l0aCBhXG4gICAqIGRvdWJsZSAkJCwgaWU6ICQkKFZBUl9OQU1FKS4gRXNjYXBlZCByZWZlcmVuY2VzIHdpbGwgbmV2ZXIgYmUgZXhwYW5kZWQsXG4gICAqIHJlZ2FyZGxlc3Mgb2Ygd2hldGhlciB0aGUgdmFyaWFibGUgZXhpc3RzIG9yIG5vdC5cbiAgICpcbiAgICogQ2Fubm90IGJlIHVwZGF0ZWQuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvdGFza3MvaW5qZWN0LWRhdGEtYXBwbGljYXRpb24vZGVmaW5lLWNvbW1hbmQtYXJndW1lbnQtY29udGFpbmVyLyNydW5uaW5nLWEtY29tbWFuZC1pbi1hLXNoZWxsXG4gICAqIEBkZWZhdWx0IFtdXG4gICAqL1xuICByZWFkb25seSBhcmdzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIENvbnRhaW5lcidzIHdvcmtpbmcgZGlyZWN0b3J5LiBJZiBub3Qgc3BlY2lmaWVkLCB0aGUgY29udGFpbmVyIHJ1bnRpbWUncyBkZWZhdWx0IHdpbGwgYmUgdXNlZCwgd2hpY2ggbWlnaHQgYmUgY29uZmlndXJlZCBpbiB0aGUgY29udGFpbmVyIGltYWdlLiBDYW5ub3QgYmUgdXBkYXRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgY29udGFpbmVyIHJ1bnRpbWUncyBkZWZhdWx0LlxuICAgKi9cbiAgcmVhZG9ubHkgd29ya2luZ0Rpcj86IHN0cmluZztcblxuICAvKipcbiAgICogRW52aXJvbm1lbnQgdmFyaWFibGVzIHRvIHNldCBpbiB0aGUgY29udGFpbmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAgICovXG4gIHJlYWRvbmx5IGVudlZhcmlhYmxlcz86IHsgW25hbWU6IHN0cmluZ106IEVudlZhbHVlIH07XG5cbiAgLyoqXG4gICAqIExpc3Qgb2Ygc291cmNlcyB0byBwb3B1bGF0ZSBlbnZpcm9ubWVudCB2YXJpYWJsZXMgaW4gdGhlIGNvbnRhaW5lci5cbiAgICogV2hlbiBhIGtleSBleGlzdHMgaW4gbXVsdGlwbGUgc291cmNlcywgdGhlIHZhbHVlIGFzc29jaWF0ZWQgd2l0aFxuICAgKiB0aGUgbGFzdCBzb3VyY2Ugd2lsbCB0YWtlIHByZWNlZGVuY2UuIFZhbHVlcyBkZWZpbmVkIGJ5IHRoZSBgZW52VmFyaWFibGVzYCBwcm9wZXJ0eVxuICAgKiB3aXRoIGEgZHVwbGljYXRlIGtleSB3aWxsIHRha2UgcHJlY2VkZW5jZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBzb3VyY2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgZW52RnJvbT86IEVudkZyb21bXTtcblxuICAvKipcbiAgICogUG9kIHZvbHVtZXMgdG8gbW91bnQgaW50byB0aGUgY29udGFpbmVyJ3MgZmlsZXN5c3RlbS4gQ2Fubm90IGJlIHVwZGF0ZWQuXG4gICAqL1xuICByZWFkb25seSB2b2x1bWVNb3VudHM/OiBWb2x1bWVNb3VudFtdO1xuXG4gIC8qKlxuICAgKiBJbWFnZSBwdWxsIHBvbGljeSBmb3IgdGhpcyBjb250YWluZXJcbiAgICogQGRlZmF1bHQgSW1hZ2VQdWxsUG9saWN5LkFMV0FZU1xuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2VQdWxsUG9saWN5PzogSW1hZ2VQdWxsUG9saWN5O1xuXG4gIC8qKlxuICAgKiBEZXRlcm1pbmVzIHdoZW4gdGhlIGNvbnRhaW5lciBpcyByZWFkeSB0byBzZXJ2ZSB0cmFmZmljLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHJlYWRpbmVzcyBwcm9iZSBpcyBkZWZpbmVkXG4gICAqL1xuICByZWFkb25seSByZWFkaW5lc3M/OiBwcm9iZS5Qcm9iZTtcblxuICAvKipcbiAgICogUGVyaW9kaWMgcHJvYmUgb2YgY29udGFpbmVyIGxpdmVuZXNzLiBDb250YWluZXIgd2lsbCBiZSByZXN0YXJ0ZWQgaWYgdGhlIHByb2JlIGZhaWxzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGxpdmVuZXNzIHByb2JlIGlzIGRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IGxpdmVuZXNzPzogcHJvYmUuUHJvYmU7XG5cbiAgLyoqXG4gICAqIFN0YXJ0dXBQcm9iZSBpbmRpY2F0ZXMgdGhhdCB0aGUgUG9kIGhhcyBzdWNjZXNzZnVsbHkgaW5pdGlhbGl6ZWQuXG4gICAqIElmIHNwZWNpZmllZCwgbm8gb3RoZXIgcHJvYmVzIGFyZSBleGVjdXRlZCB1bnRpbCB0aGlzIGNvbXBsZXRlcyBzdWNjZXNzZnVsbHlcbiAgICpcbiAgICogQGRlZmF1bHQgLSBJZiBhIHBvcnQgaXMgcHJvdmlkZWQsIHRoZW4ga25vY2tzIG9uIHRoYXQgcG9ydFxuICAgKiB0byBkZXRlcm1pbmUgd2hlbiB0aGUgY29udGFpbmVyIGlzIHJlYWR5IGZvciByZWFkaW5lc3MgYW5kXG4gICAqIGxpdmVuZXNzIHByb2JlIGNoZWNrcy5cbiAgICogT3RoZXJ3aXNlLCBubyBzdGFydHVwIHByb2JlIGlzIGRlZmluZWQuXG4gICAqL1xuICByZWFkb25seSBzdGFydHVwPzogcHJvYmUuUHJvYmU7XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBhY3Rpb25zIHRoYXQgdGhlIG1hbmFnZW1lbnQgc3lzdGVtIHNob3VsZCB0YWtlIGluIHJlc3BvbnNlIHRvIGNvbnRhaW5lciBsaWZlY3ljbGUgZXZlbnRzLlxuICAgKi9cbiAgcmVhZG9ubHkgbGlmZWN5Y2xlPzogQ29udGFpbmVyTGlmZWN5Y2xlO1xuXG4gIC8qKlxuICAgKiBDb21wdXRlIHJlc291cmNlcyAoQ1BVIGFuZCBtZW1vcnkgcmVxdWVzdHMgYW5kIGxpbWl0cykgcmVxdWlyZWQgYnkgdGhlIGNvbnRhaW5lclxuICAgKiBAc2VlIGh0dHBzOi8va3ViZXJuZXRlcy5pby9kb2NzL2NvbmNlcHRzL2NvbmZpZ3VyYXRpb24vbWFuYWdlLXJlc291cmNlcy1jb250YWluZXJzL1xuICAgKlxuICAgKiBAZGVmYXVsdFxuICAgKiAgICBjcHU6XG4gICAqICAgICAgcmVxdWVzdDogMTAwMCBtaWxsaXNcbiAgICogICAgICBsaW1pdDogMTUwMCBtaWxsaXNcbiAgICogICAgbWVtb3J5OlxuICAgKiAgICAgIHJlcXVlc3Q6IDUxMiBtZWJpYnl0ZXNcbiAgICogICAgICBsaW1pdDogMjA0OCBtZWJpYnl0ZXNcbiAgICovXG4gIHJlYWRvbmx5IHJlc291cmNlcz86IENvbnRhaW5lclJlc291cmNlcztcblxuICAvKipcbiAgICogU2VjdXJpdHlDb250ZXh0IGRlZmluZXMgdGhlIHNlY3VyaXR5IG9wdGlvbnMgdGhlIGNvbnRhaW5lciBzaG91bGQgYmUgcnVuIHdpdGguXG4gICAqIElmIHNldCwgdGhlIGZpZWxkcyBvdmVycmlkZSBlcXVpdmFsZW50IGZpZWxkcyBvZiB0aGUgcG9kJ3Mgc2VjdXJpdHkgY29udGV4dC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy90YXNrcy9jb25maWd1cmUtcG9kLWNvbnRhaW5lci9zZWN1cml0eS1jb250ZXh0L1xuICAgKiBAZGVmYXVsdFxuICAgKlxuICAgKiAgIGVuc3VyZU5vblJvb3Q6IGZhbHNlXG4gICAqICAgcHJpdmlsZWdlZDogZmFsc2VcbiAgICogICByZWFkT25seVJvb3RGaWxlc3lzdGVtOiBmYWxzZVxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlDb250ZXh0PzogQ29udGFpbmVyU2VjdXJpdHlDb250ZXh0UHJvcHM7XG59XG5cbi8qKlxuICogQSBzaW5nbGUgYXBwbGljYXRpb24gY29udGFpbmVyIHRoYXQgeW91IHdhbnQgdG8gcnVuIHdpdGhpbiBhIHBvZC5cbiAqL1xuZXhwb3J0IGNsYXNzIENvbnRhaW5lciB7XG5cbiAgLyoqXG4gICAqIFRoZSBwb3J0IG51bWJlciB0aGF0IHdhcyBjb25maWd1cmVkIGZvciB0aGlzIGNvbnRhaW5lci5cbiAgICogSWYgdW5kZWZpbmVkLCBlaXRoZXIgdGhlIGNvbnRhaW5lciBkb2Vzbid0IGV4cG9zZSBhIHBvcnQsIG9yIGl0c1xuICAgKiBwb3J0IGNvbmZpZ3VyYXRpb24gaXMgc3RvcmVkIGluIHRoZSBgcG9ydHNgIGZpZWxkLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHBvcnROdW1iZXI/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFZvbHVtZSBtb3VudHMgY29uZmlndXJlZCBmb3IgdGhpcyBjb250YWluZXIuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbW91bnRzOiBWb2x1bWVNb3VudFtdO1xuXG4gIC8qKlxuICAgKiBJbWFnZSBwdWxsIHBvbGljeSBmb3IgdGhpcyBjb250YWluZXJcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpbWFnZVB1bGxQb2xpY3k6IEltYWdlUHVsbFBvbGljeTtcblxuICAvKipcbiAgICogVGhlIGNvbnRhaW5lciBpbWFnZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpbWFnZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHdvcmtpbmcgZGlyZWN0b3J5IGluc2lkZSB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHdvcmtpbmdEaXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIENvbXB1dGUgcmVzb3VyY2VzIChDUFUgYW5kIG1lbW9yeSByZXF1ZXN0cyBhbmQgbGltaXRzKSByZXF1aXJlZCBieSB0aGUgY29udGFpbmVyXG4gICAqIEBzZWUgaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvY29uZmlndXJhdGlvbi9tYW5hZ2UtcmVzb3VyY2VzLWNvbnRhaW5lcnMvXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VzPzogQ29udGFpbmVyUmVzb3VyY2VzO1xuXG4gIC8qKlxuICAgKiBUaGUgc2VjdXJpdHkgY29udGV4dCBvZiB0aGUgY29udGFpbmVyLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNlY3VyaXR5Q29udGV4dDogQ29udGFpbmVyU2VjdXJpdHlDb250ZXh0O1xuXG4gIC8qKlxuICAgKiBUaGUgZW52aXJvbm1lbnQgb2YgdGhlIGNvbnRhaW5lci5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBlbnY6IEVudjtcblxuICBwcml2YXRlIHJlYWRvbmx5IF9jb21tYW5kPzogcmVhZG9ubHkgc3RyaW5nW107XG4gIHByaXZhdGUgcmVhZG9ubHkgX2FyZ3M/OiByZWFkb25seSBzdHJpbmdbXTtcbiAgcHJpdmF0ZSByZWFkb25seSBfcG9ydHM6IENvbnRhaW5lclBvcnRbXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IF9yZWFkaW5lc3M/OiBwcm9iZS5Qcm9iZTtcbiAgcHJpdmF0ZSByZWFkb25seSBfbGl2ZW5lc3M/OiBwcm9iZS5Qcm9iZTtcbiAgcHJpdmF0ZSByZWFkb25seSBfc3RhcnR1cD86IHByb2JlLlByb2JlO1xuICBwcml2YXRlIHJlYWRvbmx5IF9saWZlY3ljbGU/OiBDb250YWluZXJMaWZlY3ljbGU7XG5cbiAgY29uc3RydWN0b3IocHJvcHM6IENvbnRhaW5lclByb3BzKSB7XG4gICAgaWYgKHByb3BzIGluc3RhbmNlb2YgQ29udGFpbmVyKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0dGVtcHRlZCB0byBjb25zdHJ1Y3QgYSBjb250YWluZXIgZnJvbSBhIENvbnRhaW5lciBvYmplY3QuJyk7XG4gICAgfVxuXG4gICAgY29uc3QgZGVmYXVsdFJlc291cmNlU3BlYzogQ29udGFpbmVyUmVzb3VyY2VzID0ge1xuICAgICAgY3B1OiB7XG4gICAgICAgIHJlcXVlc3Q6IENwdS5taWxsaXMoMTAwMCksXG4gICAgICAgIGxpbWl0OiBDcHUubWlsbGlzKDE1MDApLFxuICAgICAgfSxcbiAgICAgIG1lbW9yeToge1xuICAgICAgICByZXF1ZXN0OiBTaXplLm1lYmlieXRlcyg1MTIpLFxuICAgICAgICBsaW1pdDogU2l6ZS5tZWJpYnl0ZXMoMjA0OCksXG4gICAgICB9LFxuICAgIH07XG5cbiAgICBpZiAocHJvcHMucG9ydCAmJiBwcm9wcy5wb3J0TnVtYmVyKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0VpdGhlciBcXCdwb3J0XFwnIG9yIFxcJ3BvcnROdW1iZXJcXCcgaXMgYWxsb3dlZC4gVXNlIFxcJ3BvcnROdW1iZXJcXCcgc2luY2UgXFwncG9ydFxcJyBpcyBkZXByZWNhdGVkJyk7XG4gICAgfVxuXG4gICAgY29uc3QgcG9ydE51bWJlciA9IHByb3BzLnBvcnROdW1iZXIgPz8gcHJvcHMucG9ydDtcbiAgICBjb25zdCBkZWZhdWx0UHJvYmVDb25maWd1cmF0aW9uOiBwcm9iZS5Qcm9iZSA9IHByb2JlLlByb2JlLmZyb21UY3BTb2NrZXQoeyBwb3J0OiBwb3J0TnVtYmVyIH0pO1xuXG4gICAgdGhpcy5uYW1lID0gcHJvcHMubmFtZSA/PyAnbWFpbic7XG4gICAgdGhpcy5pbWFnZSA9IHByb3BzLmltYWdlO1xuICAgIHRoaXMucG9ydE51bWJlciA9IHBvcnROdW1iZXI7XG4gICAgdGhpcy5fY29tbWFuZCA9IHByb3BzLmNvbW1hbmQ7XG4gICAgdGhpcy5fYXJncyA9IHByb3BzLmFyZ3M7XG4gICAgdGhpcy5fcmVhZGluZXNzID0gcHJvcHMucmVhZGluZXNzO1xuICAgIHRoaXMuX2xpdmVuZXNzID0gcHJvcHMubGl2ZW5lc3M7XG4gICAgdGhpcy5fc3RhcnR1cCA9IHByb3BzLnN0YXJ0dXAgPz8gKHRoaXMucG9ydE51bWJlciA/IGRlZmF1bHRQcm9iZUNvbmZpZ3VyYXRpb24gOiB1bmRlZmluZWQpO1xuICAgIHRoaXMuX2xpZmVjeWNsZSA9IHByb3BzLmxpZmVjeWNsZTtcbiAgICB0aGlzLnJlc291cmNlcyA9IHByb3BzLnJlc291cmNlcyA/PyBkZWZhdWx0UmVzb3VyY2VTcGVjO1xuICAgIHRoaXMud29ya2luZ0RpciA9IHByb3BzLndvcmtpbmdEaXI7XG4gICAgdGhpcy5tb3VudHMgPSBwcm9wcy52b2x1bWVNb3VudHMgPz8gW107XG4gICAgdGhpcy5pbWFnZVB1bGxQb2xpY3kgPSBwcm9wcy5pbWFnZVB1bGxQb2xpY3kgPz8gSW1hZ2VQdWxsUG9saWN5LkFMV0FZUztcbiAgICB0aGlzLnNlY3VyaXR5Q29udGV4dCA9IG5ldyBDb250YWluZXJTZWN1cml0eUNvbnRleHQocHJvcHMuc2VjdXJpdHlDb250ZXh0KTtcbiAgICB0aGlzLmVudiA9IG5ldyBFbnYocHJvcHMuZW52RnJvbSA/PyBbXSwgcHJvcHMuZW52VmFyaWFibGVzID8/IHt9KTtcblxuICAgIGlmICh0aGlzLnBvcnROdW1iZXIpIHtcbiAgICAgIHRoaXMuYWRkUG9ydCh7XG4gICAgICAgIG51bWJlcjogdGhpcy5wb3J0TnVtYmVyLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBwb3J0IG9mIHByb3BzLnBvcnRzID8/IFtdKSB7XG4gICAgICB0aGlzLmFkZFBvcnQocG9ydCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEBkZXByZWNhdGVkIC0gdXNlIGBwb3J0TnVtYmVyYC5cbiAgICovXG4gIHB1YmxpYyBnZXQgcG9ydCgpOiBudW1iZXIgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLnBvcnROdW1iZXI7XG4gIH1cblxuICAvKipcbiAgICogUG9ydHMgZXhwb3NlZCBieSB0aGlzIGNvbnRhaW5lcnMuXG4gICAqIFJldHVybnMgYSBjb3B5LCB1c2UgYGFkZFBvcnRgIHRvIG1vZGlmeS5cbiAgICovXG4gIHB1YmxpYyBnZXQgcG9ydHMoKTogQ29udGFpbmVyUG9ydFtdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX3BvcnRzXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBFbnRyeXBvaW50IGFycmF5ICh0aGUgY29tbWFuZCB0byBleGVjdXRlIHdoZW4gdGhlIGNvbnRhaW5lciBzdGFydHMpLlxuICAgKiBAcmV0dXJucyBhIGNvcHkgb2YgdGhlIGVudHJ5cG9pbnQgYXJyYXksIGNhbm5vdCBiZSBtb2RpZmllZFxuICAgKi9cbiAgcHVibGljIGdldCBjb21tYW5kKCk6IHN0cmluZ1tdIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy5fY29tbWFuZCA/IFsuLi50aGlzLl9jb21tYW5kXSA6IHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBBcmd1bWVudHMgdG8gdGhlIGVudHJ5cG9pbnQuXG4gICAqXG4gICAqIEByZXR1cm5zIGEgY29weSBvZiB0aGUgYXJndW1lbnRzIGFycmF5LCBjYW5ub3QgYmUgbW9kaWZpZWQuXG4gICAqL1xuICBwdWJsaWMgZ2V0IGFyZ3MoKTogc3RyaW5nW10gfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLl9hcmdzID8gWy4uLnRoaXMuX2FyZ3NdIDogdW5kZWZpbmVkO1xuICB9XG5cbiAgLyoqXG4gICAqIE1vdW50IGEgdm9sdW1lIHRvIGEgc3BlY2lmaWMgcGF0aCBzbyB0aGF0IGl0IGlzIGFjY2Vzc2libGUgYnkgdGhlIGNvbnRhaW5lci5cbiAgICogRXZlcnkgcG9kIHRoYXQgaXMgY29uZmlndXJlZCB0byB1c2UgdGhpcyBjb250YWluZXIgd2lsbCBhdXRtb2F0aWNhbGx5IGhhdmUgYWNjZXNzIHRvIHRoZSB2b2x1bWUuXG4gICAqXG4gICAqIEBwYXJhbSBwYXRoIC0gVGhlIGRlc2lyZWQgcGF0aCBpbiB0aGUgY29udGFpbmVyLlxuICAgKiBAcGFyYW0gc3RvcmFnZSAtIFRoZSBzdG9yYWdlIHRvIG1vdW50LlxuICAgKi9cbiAgcHVibGljIG1vdW50KHBhdGg6IHN0cmluZywgc3RvcmFnZTogdm9sdW1lLklTdG9yYWdlLCBvcHRpb25zOiBNb3VudE9wdGlvbnMgPSB7IH0pIHtcbiAgICB0aGlzLm1vdW50cy5wdXNoKHsgcGF0aCwgdm9sdW1lOiBzdG9yYWdlLmFzVm9sdW1lKCksIC4uLm9wdGlvbnMgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgcG9ydCB0byBleHBvc2UgZnJvbSB0aGlzIGNvbnRhaW5lci5cbiAgICovXG4gIHB1YmxpYyBhZGRQb3J0KHBvcnQ6IENvbnRhaW5lclBvcnQpIHtcblxuICAgIGNvbnN0IG5hbWVzID0gdGhpcy5fcG9ydHMubWFwKHAgPT4gcC5uYW1lKS5maWx0ZXIoeCA9PiB4KTtcbiAgICBjb25zdCBudW1iZXJzID0gdGhpcy5fcG9ydHMubWFwKHAgPT4gcC5udW1iZXIpO1xuXG4gICAgaWYgKHBvcnQubmFtZSAmJiBuYW1lcy5pbmNsdWRlcyhwb3J0Lm5hbWUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFBvcnQgd2l0aCBuYW1lICR7cG9ydC5uYW1lfSBhbHJlYWR5IGV4aXN0c2ApO1xuICAgIH1cblxuICAgIGlmIChudW1iZXJzLmluY2x1ZGVzKHBvcnQubnVtYmVyKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBQb3J0IHdpdGggbnVtYmVyICR7cG9ydC5udW1iZXJ9IGFscmVhZHkgZXhpc3RzYCk7XG4gICAgfVxuXG4gICAgdGhpcy5fcG9ydHMucHVzaChwb3J0KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdG9LdWJlKCk6IGs4cy5Db250YWluZXIge1xuICAgIGNvbnN0IHZvbHVtZU1vdW50czogazhzLlZvbHVtZU1vdW50W10gPSBbXTtcblxuICAgIGZvciAoY29uc3QgbW91bnQgb2YgdGhpcy5tb3VudHMpIHtcbiAgICAgIHZvbHVtZU1vdW50cy5wdXNoKHtcbiAgICAgICAgbmFtZTogbW91bnQudm9sdW1lLm5hbWUsXG4gICAgICAgIG1vdW50UGF0aDogbW91bnQucGF0aCxcbiAgICAgICAgcmVhZE9ubHk6IG1vdW50LnJlYWRPbmx5LFxuICAgICAgICBtb3VudFByb3BhZ2F0aW9uOiBtb3VudC5wcm9wYWdhdGlvbixcbiAgICAgICAgc3ViUGF0aDogbW91bnQuc3ViUGF0aCxcbiAgICAgICAgc3ViUGF0aEV4cHI6IG1vdW50LnN1YlBhdGhFeHByLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgY29uc3QgcG9ydHMgPSBuZXcgQXJyYXk8azhzLkNvbnRhaW5lclBvcnQ+KCk7XG5cbiAgICBmb3IgKGNvbnN0IHBvcnQgb2YgdGhpcy5wb3J0cykge1xuICAgICAgcG9ydHMucHVzaCh7XG4gICAgICAgIGNvbnRhaW5lclBvcnQ6IHBvcnQubnVtYmVyLFxuICAgICAgICBwcm90b2NvbDogcG9ydC5wcm90b2NvbCA/IHRoaXMuX3BvcnRQcm90b2NvbFRvS3ViZShwb3J0LnByb3RvY29sKSA6IHVuZGVmaW5lZCxcbiAgICAgICAgbmFtZTogcG9ydC5uYW1lLFxuICAgICAgICBob3N0UG9ydDogcG9ydC5ob3N0UG9ydCxcbiAgICAgICAgaG9zdElwOiBwb3J0Lmhvc3RJcCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIFJlc291cmNlIHJlcXVlc3RzIGFuZCBsaW1pdHNcbiAgICBjb25zdCBjcHVMaW1pdCA9IHRoaXMucmVzb3VyY2VzPy5jcHU/LmxpbWl0Py5hbW91bnQ7XG4gICAgY29uc3QgY3B1UmVxdWVzdCA9IHRoaXMucmVzb3VyY2VzPy5jcHU/LnJlcXVlc3Q/LmFtb3VudDtcbiAgICBjb25zdCBtZW1vcnlMaW1pdCA9IHRoaXMucmVzb3VyY2VzPy5tZW1vcnk/LmxpbWl0O1xuICAgIGNvbnN0IG1lbW9yeVJlcXVlc3QgPSB0aGlzLnJlc291cmNlcz8ubWVtb3J5Py5yZXF1ZXN0O1xuICAgIGNvbnN0IGVwaGVtZXJhbFN0b3JhZ2VMaW1pdCA9IHRoaXMucmVzb3VyY2VzPy5lcGhlbWVyYWxTdG9yYWdlPy5saW1pdDtcbiAgICBjb25zdCBlcGhlbWVyYWxTdG9yYWdlUmVxdWVzdCA9IHRoaXMucmVzb3VyY2VzPy5lcGhlbWVyYWxTdG9yYWdlPy5yZXF1ZXN0O1xuXG4gICAgY29uc3QgbGltaXRzOiB7IFtrZXk6IHN0cmluZ106IGs4cy5RdWFudGl0eSB9ID0ge307XG4gICAgY29uc3QgcmVxdWVzdHM6IHsgW2tleTogc3RyaW5nXTogazhzLlF1YW50aXR5IH0gPSB7fTtcblxuICAgIGlmIChjcHVMaW1pdCkge1xuICAgICAgbGltaXRzLmNwdSA9IGs4cy5RdWFudGl0eS5mcm9tU3RyaW5nKGNwdUxpbWl0KTtcbiAgICB9XG4gICAgaWYgKG1lbW9yeUxpbWl0KSB7XG4gICAgICBsaW1pdHMubWVtb3J5ID0gazhzLlF1YW50aXR5LmZyb21TdHJpbmcobWVtb3J5TGltaXQudG9NZWJpYnl0ZXMoKS50b1N0cmluZygpICsgJ01pJyk7XG4gICAgfVxuICAgIGlmIChlcGhlbWVyYWxTdG9yYWdlTGltaXQpIHtcbiAgICAgIGxpbWl0c1snZXBoZW1lcmFsLXN0b3JhZ2UnXSA9IGs4cy5RdWFudGl0eS5mcm9tU3RyaW5nKGVwaGVtZXJhbFN0b3JhZ2VMaW1pdC50b0dpYmlieXRlcygpLnRvU3RyaW5nKCkgKyAnR2knKTtcbiAgICB9XG4gICAgaWYgKGNwdVJlcXVlc3QpIHtcbiAgICAgIHJlcXVlc3RzLmNwdSA9IGs4cy5RdWFudGl0eS5mcm9tU3RyaW5nKGNwdVJlcXVlc3QpO1xuICAgIH1cbiAgICBpZiAobWVtb3J5UmVxdWVzdCkge1xuICAgICAgcmVxdWVzdHMubWVtb3J5ID0gazhzLlF1YW50aXR5LmZyb21TdHJpbmcobWVtb3J5UmVxdWVzdC50b01lYmlieXRlcygpLnRvU3RyaW5nKCkgKyAnTWknKTtcbiAgICB9XG4gICAgaWYgKGVwaGVtZXJhbFN0b3JhZ2VSZXF1ZXN0KSB7XG4gICAgICByZXF1ZXN0c1snZXBoZW1lcmFsLXN0b3JhZ2UnXSA9IGs4cy5RdWFudGl0eS5mcm9tU3RyaW5nKGVwaGVtZXJhbFN0b3JhZ2VSZXF1ZXN0LnRvR2liaWJ5dGVzKCkudG9TdHJpbmcoKSArICdHaScpO1xuICAgIH1cblxuICAgIGxldCByZXNvdXJjZVJlcXVpcmVtZW50czogazhzLlJlc291cmNlUmVxdWlyZW1lbnRzIHwgdW5kZWZpbmVkID0gdW5kZWZpbmVkO1xuICAgIGlmIChPYmplY3Qua2V5cyhsaW1pdHMpLmxlbmd0aCA+IDAgfHwgT2JqZWN0LmtleXMocmVxdWVzdHMpLmxlbmd0aCA+IDApIHtcbiAgICAgIHJlc291cmNlUmVxdWlyZW1lbnRzID0ge1xuICAgICAgICBsaW1pdHM6IHVuZGVmaW5lZElmRW1wdHkobGltaXRzKSxcbiAgICAgICAgcmVxdWVzdHM6IHVuZGVmaW5lZElmRW1wdHkocmVxdWVzdHMpLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBjb25zdCBlbnYgPSB0aGlzLmVudi5fdG9LdWJlKCk7XG5cbiAgICByZXR1cm4ge1xuICAgICAgbmFtZTogdGhpcy5uYW1lLFxuICAgICAgaW1hZ2U6IHRoaXMuaW1hZ2UsXG4gICAgICBpbWFnZVB1bGxQb2xpY3k6IHRoaXMuX2ltYWdlUHVsbFBvbGljeVRvS3ViZSh0aGlzLmltYWdlUHVsbFBvbGljeSksXG4gICAgICBwb3J0czogdW5kZWZpbmVkSWZFbXB0eShwb3J0cyksXG4gICAgICB2b2x1bWVNb3VudHM6IHVuZGVmaW5lZElmRW1wdHkodm9sdW1lTW91bnRzKSxcbiAgICAgIGNvbW1hbmQ6IHRoaXMuY29tbWFuZCxcbiAgICAgIGFyZ3M6IHRoaXMuYXJncyxcbiAgICAgIHdvcmtpbmdEaXI6IHRoaXMud29ya2luZ0RpcixcbiAgICAgIGVudjogZW52LnZhcmlhYmxlcyxcbiAgICAgIGVudkZyb206IGVudi5mcm9tLFxuICAgICAgcmVhZGluZXNzUHJvYmU6IHRoaXMuX3JlYWRpbmVzcz8uX3RvS3ViZSh0aGlzKSxcbiAgICAgIGxpdmVuZXNzUHJvYmU6IHRoaXMuX2xpdmVuZXNzPy5fdG9LdWJlKHRoaXMpLFxuICAgICAgc3RhcnR1cFByb2JlOiB0aGlzLl9zdGFydHVwPy5fdG9LdWJlKHRoaXMpLFxuICAgICAgbGlmZWN5Y2xlOiB0aGlzLl9saWZlY3ljbGUgPyB7XG4gICAgICAgIHBvc3RTdGFydDogdGhpcy5fbGlmZWN5Y2xlLnBvc3RTdGFydD8uX3RvS3ViZSh0aGlzKSxcbiAgICAgICAgcHJlU3RvcDogdGhpcy5fbGlmZWN5Y2xlLnByZVN0b3A/Ll90b0t1YmUodGhpcyksXG4gICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgcmVzb3VyY2VzOiByZXNvdXJjZVJlcXVpcmVtZW50cyxcbiAgICAgIHNlY3VyaXR5Q29udGV4dDogdGhpcy5zZWN1cml0eUNvbnRleHQuX3RvS3ViZSgpLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIF9pbWFnZVB1bGxQb2xpY3lUb0t1YmUoaW1hZ2VQdWxsUG9saWN5OiBJbWFnZVB1bGxQb2xpY3kpOiBrOHMuSW9LOFNBcGlDb3JlVjFDb250YWluZXJJbWFnZVB1bGxQb2xpY3kge1xuICAgIHN3aXRjaCAoaW1hZ2VQdWxsUG9saWN5KSB7XG4gICAgICBjYXNlIEltYWdlUHVsbFBvbGljeS5BTFdBWVM6XG4gICAgICAgIHJldHVybiBrOHMuSW9LOFNBcGlDb3JlVjFDb250YWluZXJJbWFnZVB1bGxQb2xpY3kuQUxXQVlTO1xuICAgICAgY2FzZSBJbWFnZVB1bGxQb2xpY3kuSUZfTk9UX1BSRVNFTlQ6XG4gICAgICAgIHJldHVybiBrOHMuSW9LOFNBcGlDb3JlVjFDb250YWluZXJJbWFnZVB1bGxQb2xpY3kuSUZfTk9UX1BSRVNFTlQ7XG4gICAgICBjYXNlIEltYWdlUHVsbFBvbGljeS5ORVZFUjpcbiAgICAgICAgcmV0dXJuIGs4cy5Jb0s4U0FwaUNvcmVWMUNvbnRhaW5lckltYWdlUHVsbFBvbGljeS5ORVZFUjtcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5zdXBwb3J0ZWQgaW1hZ2UgcHVsbCBwb2xpY3k6ICR7aW1hZ2VQdWxsUG9saWN5fWApO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgX3BvcnRQcm90b2NvbFRvS3ViZShwcm90b2NvbDogUHJvdG9jb2wpOiBrOHMuSW9LOFNBcGlDb3JlVjFDb250YWluZXJQb3J0UHJvdG9jb2wge1xuICAgIHN3aXRjaCAocHJvdG9jb2wpIHtcbiAgICAgIGNhc2UgUHJvdG9jb2wuU0NUUDpcbiAgICAgICAgcmV0dXJuIGs4cy5Jb0s4U0FwaUNvcmVWMUNvbnRhaW5lclBvcnRQcm90b2NvbC5TQ1RQO1xuICAgICAgY2FzZSBQcm90b2NvbC5UQ1A6XG4gICAgICAgIHJldHVybiBrOHMuSW9LOFNBcGlDb3JlVjFDb250YWluZXJQb3J0UHJvdG9jb2wuVENQO1xuICAgICAgY2FzZSBQcm90b2NvbC5VRFA6XG4gICAgICAgIHJldHVybiBrOHMuSW9LOFNBcGlDb3JlVjFDb250YWluZXJQb3J0UHJvdG9jb2wuVURQO1xuICAgICAgZGVmYXVsdDpcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbnN1cHBvcnRlZCBwb3J0IHByb3RvY29sOiAke3Byb3RvY29sfWApO1xuICAgIH1cbiAgfVxuXG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgbW91bnRzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE1vdW50T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBEZXRlcm1pbmVzIGhvdyBtb3VudHMgYXJlIHByb3BhZ2F0ZWQgZnJvbSB0aGUgaG9zdCB0byBjb250YWluZXIgYW5kIHRoZVxuICAgKiBvdGhlciB3YXkgYXJvdW5kLiBXaGVuIG5vdCBzZXQsIE1vdW50UHJvcGFnYXRpb25Ob25lIGlzIHVzZWQuXG4gICAqXG4gICAqIE1vdW50IHByb3BhZ2F0aW9uIGFsbG93cyBmb3Igc2hhcmluZyB2b2x1bWVzIG1vdW50ZWQgYnkgYSBDb250YWluZXIgdG9cbiAgICogb3RoZXIgQ29udGFpbmVycyBpbiB0aGUgc2FtZSBQb2QsIG9yIGV2ZW4gdG8gb3RoZXIgUG9kcyBvbiB0aGUgc2FtZSBub2RlLlxuICAgKlxuICAgKiBAZGVmYXVsdCBNb3VudFByb3BhZ2F0aW9uLk5PTkVcbiAgICovXG4gIHJlYWRvbmx5IHByb3BhZ2F0aW9uPzogTW91bnRQcm9wYWdhdGlvbjtcblxuICAvKipcbiAgICogTW91bnRlZCByZWFkLW9ubHkgaWYgdHJ1ZSwgcmVhZC13cml0ZSBvdGhlcndpc2UgKGZhbHNlIG9yIHVuc3BlY2lmaWVkKS5cbiAgICogRGVmYXVsdHMgdG8gZmFsc2UuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZWFkT25seT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFBhdGggd2l0aGluIHRoZSB2b2x1bWUgZnJvbSB3aGljaCB0aGUgY29udGFpbmVyJ3Mgdm9sdW1lIHNob3VsZCBiZSBtb3VudGVkLikuXG4gICAqXG4gICAqIEBkZWZhdWx0IFwiXCIgdGhlIHZvbHVtZSdzIHJvb3RcbiAgICovXG4gIHJlYWRvbmx5IHN1YlBhdGg/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEV4cGFuZGVkIHBhdGggd2l0aGluIHRoZSB2b2x1bWUgZnJvbSB3aGljaCB0aGUgY29udGFpbmVyJ3Mgdm9sdW1lIHNob3VsZCBiZVxuICAgKiBtb3VudGVkLiBCZWhhdmVzIHNpbWlsYXJseSB0byBTdWJQYXRoIGJ1dCBlbnZpcm9ubWVudCB2YXJpYWJsZSByZWZlcmVuY2VzXG4gICAqICQoVkFSX05BTUUpIGFyZSBleHBhbmRlZCB1c2luZyB0aGUgY29udGFpbmVyJ3MgZW52aXJvbm1lbnQuIERlZmF1bHRzIHRvIFwiXCJcbiAgICogKHZvbHVtZSdzIHJvb3QpLlxuICAgKlxuICAgKiBgc3ViUGF0aEV4cHJgIGFuZCBgc3ViUGF0aGAgYXJlIG11dHVhbGx5IGV4Y2x1c2l2ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJcIiB2b2x1bWUncyByb290LlxuICAgKi9cbiAgcmVhZG9ubHkgc3ViUGF0aEV4cHI/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogTW91bnQgYSB2b2x1bWUgZnJvbSB0aGUgcG9kIHRvIHRoZSBjb250YWluZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVm9sdW1lTW91bnQgZXh0ZW5kcyBNb3VudE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHZvbHVtZSB0byBtb3VudC5cbiAgICovXG4gIHJlYWRvbmx5IHZvbHVtZTogdm9sdW1lLlZvbHVtZTtcblxuICAvKipcbiAgICogUGF0aCB3aXRoaW4gdGhlIGNvbnRhaW5lciBhdCB3aGljaCB0aGUgdm9sdW1lIHNob3VsZCBiZSBtb3VudGVkLiBNdXN0IG5vdFxuICAgKiBjb250YWluICc6Jy5cbiAgICovXG4gIHJlYWRvbmx5IHBhdGg6IHN0cmluZztcbn1cblxuZXhwb3J0IGVudW0gTW91bnRQcm9wYWdhdGlvbiB7XG4gIC8qKlxuICAgKiBUaGlzIHZvbHVtZSBtb3VudCB3aWxsIG5vdCByZWNlaXZlIGFueSBzdWJzZXF1ZW50IG1vdW50cyB0aGF0IGFyZSBtb3VudGVkXG4gICAqIHRvIHRoaXMgdm9sdW1lIG9yIGFueSBvZiBpdHMgc3ViZGlyZWN0b3JpZXMgYnkgdGhlIGhvc3QuIEluIHNpbWlsYXJcbiAgICogZmFzaGlvbiwgbm8gbW91bnRzIGNyZWF0ZWQgYnkgdGhlIENvbnRhaW5lciB3aWxsIGJlIHZpc2libGUgb24gdGhlIGhvc3QuXG4gICAqXG4gICAqIFRoaXMgaXMgdGhlIGRlZmF1bHQgbW9kZS5cbiAgICpcbiAgICogVGhpcyBtb2RlIGlzIGVxdWFsIHRvIGBwcml2YXRlYCBtb3VudCBwcm9wYWdhdGlvbiBhcyBkZXNjcmliZWQgaW4gdGhlIExpbnV4XG4gICAqIGtlcm5lbCBkb2N1bWVudGF0aW9uXG4gICAqL1xuICBOT05FID0gJ05vbmUnLFxuXG4gIC8qKlxuICAgKiBUaGlzIHZvbHVtZSBtb3VudCB3aWxsIHJlY2VpdmUgYWxsIHN1YnNlcXVlbnQgbW91bnRzIHRoYXQgYXJlIG1vdW50ZWQgdG9cbiAgICogdGhpcyB2b2x1bWUgb3IgYW55IG9mIGl0cyBzdWJkaXJlY3Rvcmllcy5cbiAgICpcbiAgICogSW4gb3RoZXIgd29yZHMsIGlmIHRoZSBob3N0IG1vdW50cyBhbnl0aGluZyBpbnNpZGUgdGhlIHZvbHVtZSBtb3VudCwgdGhlXG4gICAqIENvbnRhaW5lciB3aWxsIHNlZSBpdCBtb3VudGVkIHRoZXJlLlxuICAgKlxuICAgKiBTaW1pbGFybHksIGlmIGFueSBQb2Qgd2l0aCBCaWRpcmVjdGlvbmFsIG1vdW50IHByb3BhZ2F0aW9uIHRvIHRoZSBzYW1lXG4gICAqIHZvbHVtZSBtb3VudHMgYW55dGhpbmcgdGhlcmUsIHRoZSBDb250YWluZXIgd2l0aCBIb3N0VG9Db250YWluZXIgbW91bnRcbiAgICogcHJvcGFnYXRpb24gd2lsbCBzZWUgaXQuXG4gICAqXG4gICAqIFRoaXMgbW9kZSBpcyBlcXVhbCB0byBgcnNsYXZlYCBtb3VudCBwcm9wYWdhdGlvbiBhcyBkZXNjcmliZWQgaW4gdGhlIExpbnV4XG4gICAqIGtlcm5lbCBkb2N1bWVudGF0aW9uXG4gICAqL1xuICBIT1NUX1RPX0NPTlRBSU5FUiA9ICdIb3N0VG9Db250YWluZXInLFxuXG4gIC8qKlxuICAgKiBUaGlzIHZvbHVtZSBtb3VudCBiZWhhdmVzIHRoZSBzYW1lIHRoZSBIb3N0VG9Db250YWluZXIgbW91bnQuIEluIGFkZGl0aW9uLFxuICAgKiBhbGwgdm9sdW1lIG1vdW50cyBjcmVhdGVkIGJ5IHRoZSBDb250YWluZXIgd2lsbCBiZSBwcm9wYWdhdGVkIGJhY2sgdG8gdGhlXG4gICAqIGhvc3QgYW5kIHRvIGFsbCBDb250YWluZXJzIG9mIGFsbCBQb2RzIHRoYXQgdXNlIHRoZSBzYW1lIHZvbHVtZVxuICAgKlxuICAgKiBBIHR5cGljYWwgdXNlIGNhc2UgZm9yIHRoaXMgbW9kZSBpcyBhIFBvZCB3aXRoIGEgRmxleFZvbHVtZSBvciBDU0kgZHJpdmVyXG4gICAqIG9yIGEgUG9kIHRoYXQgbmVlZHMgdG8gbW91bnQgc29tZXRoaW5nIG9uIHRoZSBob3N0IHVzaW5nIGEgaG9zdFBhdGggdm9sdW1lLlxuICAgKlxuICAgKiBUaGlzIG1vZGUgaXMgZXF1YWwgdG8gYHJzaGFyZWRgIG1vdW50IHByb3BhZ2F0aW9uIGFzIGRlc2NyaWJlZCBpbiB0aGUgTGludXhcbiAgICoga2VybmVsIGRvY3VtZW50YXRpb25cbiAgICpcbiAgICogQ2F1dGlvbjogQmlkaXJlY3Rpb25hbCBtb3VudCBwcm9wYWdhdGlvbiBjYW4gYmUgZGFuZ2Vyb3VzLiBJdCBjYW4gZGFtYWdlXG4gICAqIHRoZSBob3N0IG9wZXJhdGluZyBzeXN0ZW0gYW5kIHRoZXJlZm9yZSBpdCBpcyBhbGxvd2VkIG9ubHkgaW4gcHJpdmlsZWdlZFxuICAgKiBDb250YWluZXJzLiBGYW1pbGlhcml0eSB3aXRoIExpbnV4IGtlcm5lbCBiZWhhdmlvciBpcyBzdHJvbmdseSByZWNvbW1lbmRlZC5cbiAgICogSW4gYWRkaXRpb24sIGFueSB2b2x1bWUgbW91bnRzIGNyZWF0ZWQgYnkgQ29udGFpbmVycyBpbiBQb2RzIG11c3QgYmVcbiAgICogZGVzdHJveWVkICh1bm1vdW50ZWQpIGJ5IHRoZSBDb250YWluZXJzIG9uIHRlcm1pbmF0aW9uLlxuICAgKlxuICAgKi9cbiAgQklESVJFQ1RJT05BTCA9ICdCaWRpcmVjdGlvbmFsJyxcbn1cblxuLyoqXG4gKiBDUFUgYW5kIG1lbW9yeSBjb21wdXRlIHJlc291cmNlc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbnRhaW5lclJlc291cmNlcyB7XG4gIHJlYWRvbmx5IGNwdT86IENwdVJlc291cmNlcztcbiAgcmVhZG9ubHkgbWVtb3J5PzogTWVtb3J5UmVzb3VyY2VzO1xuICByZWFkb25seSBlcGhlbWVyYWxTdG9yYWdlPzogRXBoZW1lcmFsU3RvcmFnZVJlc291cmNlcztcbn1cblxuLyoqXG4gKiBDUFUgcmVxdWVzdCBhbmQgbGltaXRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDcHVSZXNvdXJjZXMge1xuICByZWFkb25seSByZXF1ZXN0PzogQ3B1O1xuICByZWFkb25seSBsaW1pdD86IENwdTtcbn1cblxuLyoqXG4gKiBSZXByZXNlbnRzIHRoZSBhbW91bnQgb2YgQ1BVLlxuICogVGhlIGFtb3VudCBjYW4gYmUgcGFzc2VkIGFzIG1pbGxpcyBvciB1bml0cy5cbiAqL1xuZXhwb3J0IGNsYXNzIENwdSB7XG4gIHN0YXRpYyBtaWxsaXMoYW1vdW50OiBudW1iZXIpOiBDcHUge1xuICAgIHJldHVybiBuZXcgQ3B1KGFtb3VudCArICdtJyk7XG4gIH1cbiAgc3RhdGljIHVuaXRzKGFtb3VudDogbnVtYmVyKTogQ3B1IHtcbiAgICByZXR1cm4gbmV3IENwdShhbW91bnQudG9TdHJpbmcoKSk7XG4gIH1cbiAgYW1vdW50OiBzdHJpbmc7XG4gIHByaXZhdGUgY29uc3RydWN0b3IoYW1vdW50OiBzdHJpbmcpIHtcbiAgICB0aGlzLmFtb3VudCA9IGFtb3VudDtcbiAgfVxufVxuXG4vKipcbiAqIE1lbW9yeSByZXF1ZXN0IGFuZCBsaW1pdFxuICovXG5leHBvcnQgaW50ZXJmYWNlIE1lbW9yeVJlc291cmNlcyB7XG4gIHJlYWRvbmx5IHJlcXVlc3Q/OiBTaXplO1xuICByZWFkb25seSBsaW1pdD86IFNpemU7XG59XG5cbi8qKlxuICogRW1waGVtZXJhbCBzdG9yYWdlIHJlcXVlc3QgYW5kIGxpbWl0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRXBoZW1lcmFsU3RvcmFnZVJlc291cmNlcyB7XG4gIHJlYWRvbmx5IHJlcXVlc3Q/OiBTaXplO1xuICByZWFkb25seSBsaW1pdD86IFNpemU7XG59XG5cbi8qKlxuICogQSBjb2xsZWN0aW9uIG9mIGVudiB2YXJpYWJsZXMgZGVmaW5lZCBpbiBvdGhlciByZXNvdXJjZXMuXG4gKi9cbmV4cG9ydCBjbGFzcyBFbnZGcm9tIHtcblxuICBjb25zdHJ1Y3RvcihcbiAgICBwcml2YXRlIHJlYWRvbmx5IGNvbmZpZ01hcD86IGNvbmZpZ21hcC5JQ29uZmlnTWFwLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgcHJlZml4Pzogc3RyaW5nLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgc2VjPzogc2VjcmV0LklTZWNyZXQpIHt9O1xuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdG9LdWJlKCk6IGs4cy5FbnZGcm9tU291cmNlIHtcbiAgICByZXR1cm4ge1xuICAgICAgY29uZmlnTWFwUmVmOiB0aGlzLmNvbmZpZ01hcCA/IHtcbiAgICAgICAgbmFtZTogdGhpcy5jb25maWdNYXAubmFtZSxcbiAgICAgIH0gOiB1bmRlZmluZWQsXG4gICAgICBzZWNyZXRSZWY6IHRoaXMuc2VjID8ge1xuICAgICAgICBuYW1lOiB0aGlzLnNlYy5uYW1lLFxuICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIHByZWZpeDogdGhpcy5wcmVmaXgsXG4gICAgfTtcbiAgfVxuXG59XG5cbmV4cG9ydCBmdW5jdGlvbiBleHRyYWN0Q29udGFpbmVyUG9ydHMoc2VsZWN0b3I/OiBhbnkpOiBDb250YWluZXJQb3J0W10ge1xuXG4gIGlmICghc2VsZWN0b3IpIHsgcmV0dXJuIFtdOyB9XG5cbiAgLy8gd2UgZG9uJ3QgdXNlIGluc3RhbmNlb2YgaW50ZW50aW9uYWxseSBzaW5jZSBpdCBjYW4gY3JlYXRlXG4gIC8vIGN5Y2xpYyBpbXBvcnQgcHJvYmxlbXMuXG4gIGNvbnN0IGNvbnRhaW5lcnM6IENvbnRhaW5lcltdID0gKHNlbGVjdG9yIGFzIGFueSkuY29udGFpbmVycyA/PyBbXTtcblxuICByZXR1cm4gY29udGFpbmVycy5mbGF0TWFwKGMgPT4gYy5wb3J0cyk7XG59XG5cbi8qKlxuICogQ29udGFpbmVyIGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAqL1xuZXhwb3J0IGNsYXNzIEVudiB7XG5cbiAgLyoqXG4gICAqIFNlbGVjdHMgYSBDb25maWdNYXAgdG8gcG9wdWxhdGUgdGhlIGVudmlyb25tZW50IHZhcmlhYmxlcyB3aXRoLlxuICAgKiBUaGUgY29udGVudHMgb2YgdGhlIHRhcmdldCBDb25maWdNYXAncyBEYXRhIGZpZWxkIHdpbGwgcmVwcmVzZW50XG4gICAqIHRoZSBrZXktdmFsdWUgcGFpcnMgYXMgZW52aXJvbm1lbnQgdmFyaWFibGVzLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQ29uZmlnTWFwKGNvbmZpZ01hcDogY29uZmlnbWFwLklDb25maWdNYXAsIHByZWZpeD86IHN0cmluZyk6IEVudkZyb20ge1xuICAgIHJldHVybiBuZXcgRW52RnJvbShjb25maWdNYXAsIHByZWZpeCwgdW5kZWZpbmVkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBTZWxlY3RzIGEgU2VjcmV0IHRvIHBvcHVsYXRlIHRoZSBlbnZpcm9ubWVudCB2YXJpYWJsZXMgd2l0aC5cbiAgICogVGhlIGNvbnRlbnRzIG9mIHRoZSB0YXJnZXQgU2VjcmV0J3MgRGF0YSBmaWVsZCB3aWxsIHJlcHJlc2VudFxuICAgKiB0aGUga2V5LXZhbHVlIHBhaXJzIGFzIGVudmlyb25tZW50IHZhcmlhYmxlcy5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVNlY3JldChzZWNyOiBzZWNyZXQuSVNlY3JldCk6IEVudkZyb20ge1xuICAgIHJldHVybiBuZXcgRW52RnJvbSh1bmRlZmluZWQsIHVuZGVmaW5lZCwgc2Vjcik7XG4gIH1cblxuICBwcml2YXRlIHJlYWRvbmx5IF9zb3VyY2VzOiBFbnZGcm9tW107XG4gIHByaXZhdGUgcmVhZG9ubHkgX3ZhcmlhYmxlczogeyBba2V5OiBzdHJpbmddOiBFbnZWYWx1ZSB9O1xuXG4gIHB1YmxpYyBjb25zdHJ1Y3Rvcihzb3VyY2VzOiBFbnZGcm9tW10sIHZhcmlhYmxlczogeyBbbmFtZTogc3RyaW5nXTogRW52VmFsdWUgfSkge1xuICAgIHRoaXMuX3NvdXJjZXMgPSBzb3VyY2VzO1xuICAgIHRoaXMuX3ZhcmlhYmxlcyA9IHZhcmlhYmxlcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBzaW5nbGUgdmFyaWFibGUgYnkgbmFtZSBhbmQgdmFsdWUuXG4gICAqIFRoZSB2YXJpYWJsZSB2YWx1ZSBjYW4gY29tZSBmcm9tIHZhcmlvdXMgZHluYW1pYyBzb3VyY2VzIHN1Y2ggYSBzZWNyZXRzIG9mIGNvbmZpZyBtYXBzLlxuICAgKiBVc2UgYEVudlZhbHVlLmZyb21YWFhgIHRvIHNlbGVjdCBzb3VyY2VzLlxuICAgKi9cbiAgcHVibGljIGFkZFZhcmlhYmxlKG5hbWU6IHN0cmluZywgdmFsdWU6IEVudlZhbHVlKSB7XG4gICAgdGhpcy5fdmFyaWFibGVzW25hbWVdID0gdmFsdWU7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGVudmlyb25tZW50IHZhcmlhYmxlcyBmb3IgdGhpcyBjb250YWluZXIuXG4gICAqIFJldHVybnMgYSBjb3B5LiBUbyBhZGQgZW52aXJvbm1lbnQgdmFyaWFibGVzIHVzZSBgY29udGFpbmVyLmVudi5hZGRWYXJpYWJsZSgpYC5cbiAgICovXG4gIHB1YmxpYyBnZXQgdmFyaWFibGVzKCk6IHsgW25hbWU6IHN0cmluZ106IEVudlZhbHVlIH0ge1xuICAgIHJldHVybiB7IC4uLnRoaXMuX3ZhcmlhYmxlcyB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGNvbGxlY3Rpb24gb2YgdmFyaWFibGVzIGJ5IGNvcHlpbmcgZnJvbSBhbm90aGVyIHNvdXJjZS5cbiAgICogVXNlIGBFbnYuZnJvbVhYWGAgZnVuY3Rpb25zIHRvIHNlbGVjdCBzb3VyY2VzLlxuICAgKi9cbiAgcHVibGljIGNvcHlGcm9tKGZyb206IEVudkZyb20pIHtcbiAgICB0aGlzLl9zb3VyY2VzLnB1c2goZnJvbSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGxpc3Qgb2Ygc291cmNlcyB1c2VkIHRvIHBvcHVsYXRlIHRoZSBjb250YWluZXIgZW52aXJvbm1lbnQsXG4gICAqIGluIGFkZGl0aW9uIHRvIHRoZSBgdmFyaWFibGVzYC5cbiAgICpcbiAgICogUmV0dXJucyBhIGNvcHkuIFRvIGFkZCBhIHNvdXJjZSB1c2UgYGNvbnRhaW5lci5lbnYuY29weUZyb20oKWAuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHNvdXJjZXMoKTogRW52RnJvbVtdIHtcbiAgICByZXR1cm4gWy4uLnRoaXMuX3NvdXJjZXNdO1xuICB9XG5cbiAgcHJpdmF0ZSByZW5kZXJFbnYoZW52OiB7IFtuYW1lOiBzdHJpbmddOiBFbnZWYWx1ZSB9KTogazhzLkVudlZhcltdIHtcbiAgICBjb25zdCByZXN1bHQgPSBuZXcgQXJyYXk8azhzLkVudlZhcj4oKTtcbiAgICBmb3IgKGNvbnN0IFtuYW1lLCB2XSBvZiBPYmplY3QuZW50cmllcyhlbnYpKSB7XG4gICAgICByZXN1bHQucHVzaCh7XG4gICAgICAgIG5hbWUsXG4gICAgICAgIHZhbHVlOiB2LnZhbHVlLFxuICAgICAgICB2YWx1ZUZyb206IHYudmFsdWVGcm9tLFxuICAgICAgfSk7XG4gICAgfVxuICAgIHJldHVybiByZXN1bHQ7XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpOiB7IHZhcmlhYmxlcz86IGs4cy5FbnZWYXJbXTsgZnJvbT86IGs4cy5FbnZGcm9tU291cmNlW10gfSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGZyb206IHVuZGVmaW5lZElmRW1wdHkodGhpcy5fc291cmNlcy5tYXAocyA9PiBzLl90b0t1YmUoKSkpLFxuICAgICAgdmFyaWFibGVzOiB1bmRlZmluZWRJZkVtcHR5KHRoaXMucmVuZGVyRW52KHRoaXMuX3ZhcmlhYmxlcykpLFxuICAgIH07XG4gIH1cbn1cbiJdfQ==