"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.validatePeerConfig = exports.NetworkPolicy = exports.NetworkPolicyTrafficDefault = exports.NetworkProtocol = exports.NetworkPolicyIpBlock = exports.NetworkPolicyPort = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const k8s = require("./imports/k8s");
const namespace = require("./namespace");
const pod = require("./pod");
const utils_1 = require("./utils");
/**
 * Describes a port to allow traffic on.
 */
class NetworkPolicyPort {
    constructor(port, endPort, protocol) {
        this.port = port;
        this.endPort = endPort;
        this.protocol = protocol;
    }
    /**
     * Distinct TCP ports
     */
    static tcp(port) {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(port), undefined, NetworkProtocol.TCP);
    }
    /**
     * A TCP port range
     */
    static tcpRange(startPort, endPort) {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(startPort), endPort, NetworkProtocol.TCP);
    }
    /**
     * Any TCP traffic
     */
    static allTcp() {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(0), 65535, NetworkProtocol.TCP);
    }
    /**
     * Distinct UDP ports
     */
    static udp(port) {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(port), undefined, NetworkProtocol.UDP);
    }
    /**
     * A UDP port range
     */
    static udpRange(startPort, endPort) {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(startPort), endPort, NetworkProtocol.UDP);
    }
    /**
     * Any UDP traffic
     */
    static allUdp() {
        return new NetworkPolicyPort(k8s.IntOrString.fromNumber(0), 65535, NetworkProtocol.UDP);
    }
    /**
     * Custom port configuration.
     */
    static of(props) {
        return new NetworkPolicyPort(props.port ? k8s.IntOrString.fromNumber(props.port) : undefined, props.endPort, props.protocol);
    }
    /**
     * @internal
     */
    _toKube() {
        return { port: this.port, endPort: this.endPort, protocol: this.protocol };
    }
}
exports.NetworkPolicyPort = NetworkPolicyPort;
_a = JSII_RTTI_SYMBOL_1;
NetworkPolicyPort[_a] = { fqn: "cdk8s-plus-23.NetworkPolicyPort", version: "2.0.0-rc.108" };
/**
 * Describes a particular CIDR (Ex. "192.168.1.1/24","2001:db9::/64") that is
 * allowed to the pods matched by a network policy selector.
 * The except entry describes CIDRs that should not be included within this rule.
 */
class NetworkPolicyIpBlock extends constructs_1.Construct {
    constructor(scope, id, 
    /**
     * A string representing the IP Block Valid examples are "192.168.1.1/24" or "2001:db9::/64".
     */
    cidr, 
    /**
     * A slice of CIDRs that should not be included within an IP Block Valid examples are "192.168.1.1/24" or "2001:db9::/64".
     * Except values will be rejected if they are outside the CIDR range.
     */
    except) {
        super(scope, id);
        this.cidr = cidr;
        this.except = except;
    }
    /**
     * Create an IPv4 peer from a CIDR
     */
    static ipv4(scope, id, cidrIp, except) {
        const cidrMatch = cidrIp.match(/^(\d{1,3}\.){3}\d{1,3}(\/\d+)?$/);
        if (!cidrMatch) {
            throw new Error(`Invalid IPv4 CIDR: "${cidrIp}"`);
        }
        if (!cidrMatch[2]) {
            throw new Error(`CIDR mask is missing in IPv4: "${cidrIp}". Did you mean "${cidrIp}/32"?`);
        }
        return new NetworkPolicyIpBlock(scope, id, cidrIp, except);
    }
    /**
     * Any IPv4 address
     */
    static anyIpv4(scope, id) {
        return new NetworkPolicyIpBlock(scope, id, '0.0.0.0/0');
    }
    /**
     * Create an IPv6 peer from a CIDR
     */
    static ipv6(scope, id, cidrIp, except) {
        const cidrMatch = cidrIp.match(/^([\da-f]{0,4}:){2,7}([\da-f]{0,4})?(\/\d+)?$/);
        if (!cidrMatch) {
            throw new Error(`Invalid IPv6 CIDR: "${cidrIp}"`);
        }
        if (!cidrMatch[3]) {
            throw new Error(`CIDR mask is missing in IPv6: "${cidrIp}". Did you mean "${cidrIp}/128"?`);
        }
        return new NetworkPolicyIpBlock(scope, id, cidrIp, except);
    }
    /**
     * Any IPv6 address
     */
    static anyIpv6(scope, id) {
        return new NetworkPolicyIpBlock(scope, id, '::/0');
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { ipBlock: this };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return undefined;
    }
    /**
     * @internal
     */
    _toKube() {
        return { cidr: this.cidr, except: this.except };
    }
}
exports.NetworkPolicyIpBlock = NetworkPolicyIpBlock;
_b = JSII_RTTI_SYMBOL_1;
NetworkPolicyIpBlock[_b] = { fqn: "cdk8s-plus-23.NetworkPolicyIpBlock", version: "2.0.0-rc.108" };
/**
 * Network protocols.
 */
var NetworkProtocol;
(function (NetworkProtocol) {
    /**
     * TCP.
     */
    NetworkProtocol["TCP"] = "TCP";
    /**
     * UDP.
     */
    NetworkProtocol["UDP"] = "UDP";
    /**
     * SCTP.
     */
    NetworkProtocol["SCTP"] = "SCTP";
})(NetworkProtocol = exports.NetworkProtocol || (exports.NetworkProtocol = {}));
/**
 * Default behaviors of network traffic in policies.
 */
var NetworkPolicyTrafficDefault;
(function (NetworkPolicyTrafficDefault) {
    /**
     * The policy denies all traffic.
     * Since rules are additive, additional rules or policies can allow
     * specific traffic.
     */
    NetworkPolicyTrafficDefault["DENY"] = "DENY";
    /**
     * The policy allows all traffic (either ingress or egress).
     * Since rules are additive, no additional rule or policies can
     * subsequently deny the traffic.
     */
    NetworkPolicyTrafficDefault["ALLOW"] = "ALLOW";
})(NetworkPolicyTrafficDefault = exports.NetworkPolicyTrafficDefault || (exports.NetworkPolicyTrafficDefault = {}));
/**
 * Control traffic flow at the IP address or port level (OSI layer 3 or 4),
 * network policies are an application-centric construct which allow you
 * to specify how a pod is allowed to communicate with various network peers.
 *
 * - Outgoing traffic is allowed if there are no network policies selecting
 *   the pod (and cluster policy otherwise allows the traffic),
 *   OR if the traffic matches at least one egress rule across all of the
 *   network policies that select the pod.
 *
 * - Incoming traffic is allowed to a pod if there are no network policies
 *   selecting the pod (and cluster policy otherwise allows the traffic),
 *   OR if the traffic source is the pod's local node,
 *   OR if the traffic matches at least one ingress rule across all of
 *   the network policies that select the pod.
 *
 * Network policies do not conflict; they are additive.
 * If any policy or policies apply to a given pod for a given
 * direction, the connections allowed in that direction from
 * that pod is the union of what the applicable policies allow.
 * Thus, order of evaluation does not affect the policy result.
 *
 * For a connection from a source pod to a destination pod to be allowed,
 * both the egress policy on the source pod and the ingress policy on the
 * destination pod need to allow the connection.
 * If either side does not allow the connection, it will not happen.
 *
 * @see https://kubernetes.io/docs/concepts/services-networking/network-policies/#networkpolicy-resource
 */
class NetworkPolicy extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'networkpolicies';
        this._egressRules = [];
        this._ingressRules = [];
        this._policyTypes = new Set();
        const podSelector = props.selector ?? pod.Pods.all(this, 'AllPods');
        this._podSelectorConfig = podSelector.toPodSelectorConfig();
        let ns;
        if (!props.metadata?.namespace) {
            if (this._podSelectorConfig.namespaces?.labelSelector && !this._podSelectorConfig.namespaces?.labelSelector.isEmpty()) {
                throw new Error(`Unable to create a network policy for a selector (${podSelector.node.path}) that selects pods in namespaces based on labels`);
            }
            if (this._podSelectorConfig.namespaces?.names && this._podSelectorConfig.namespaces.names.length > 1) {
                throw new Error(`Unable to create a network policy for a selector (${podSelector.node.path}) that selects pods in multiple namespaces`);
            }
            ns = this._podSelectorConfig.namespaces?.names ? this._podSelectorConfig.namespaces?.names[0] : undefined;
        }
        else {
            ns = props.metadata.namespace;
        }
        this.apiObject = new k8s.KubeNetworkPolicy(this, 'Resource', {
            metadata: { ...props.metadata, namespace: ns },
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.configureDefaultBehavior('Egress', props.egress?.default);
        this.configureDefaultBehavior('Ingress', props.ingress?.default);
        for (const rule of props.egress?.rules ?? []) {
            this.addEgressRule(rule.peer, rule.ports);
        }
        for (const rule of props.ingress?.rules ?? []) {
            this.addIngressRule(rule.peer, rule.ports);
        }
    }
    /**
     * Allow outgoing traffic to the peer.
     *
     * If ports are not passed, traffic will be allowed on all ports.
     */
    addEgressRule(peer, ports) {
        this._policyTypes.add('Egress');
        this._egressRules.push({ ports: (ports ?? []).map(p => p._toKube()), to: this.createNetworkPolicyPeers(peer) });
    }
    /**
     * Allow incoming traffic from the peer.
     *
     * If ports are not passed, traffic will be allowed on all ports.
     */
    addIngressRule(peer, ports) {
        this._policyTypes.add('Ingress');
        this._ingressRules.push({ ports: (ports ?? []).map(p => p._toKube()), from: this.createNetworkPolicyPeers(peer) });
    }
    createNetworkPolicyPeers(peer) {
        const config = peer.toNetworkPolicyPeerConfig();
        validatePeerConfig(config);
        if (config.ipBlock) {
            // ip block is a single peer.
            return [{ ipBlock: config.ipBlock._toKube() }];
        }
        if (!config.podSelector.namespaces?.names) {
            // when no explicit namespaces are defined we can just use
            // the selector as is
            return [{
                    namespaceSelector: config.podSelector.namespaces?.labelSelector?._toKube(),
                    podSelector: config.podSelector.labelSelector._toKube(),
                }];
        }
        // when explicit namespaces are defined, we need to create a separate
        // peer for each, since a label selector cannot have multiple name labels. (they will conflict)
        const namespaceSelector = config.podSelector?.namespaces?.labelSelector?._toKube() ?? {};
        return config.podSelector.namespaces.names.map(n => ({
            podSelector: config.podSelector.labelSelector._toKube(),
            namespaceSelector: {
                matchExpressions: namespaceSelector.matchExpressions,
                matchLabels: {
                    ...namespaceSelector.matchLabels,
                    [namespace.Namespace.NAME_LABEL]: n,
                },
            },
        }));
    }
    configureDefaultBehavior(direction, _default) {
        if (!_default) {
            return;
        }
        if (_default === NetworkPolicyTrafficDefault.DENY) {
            // https://kubernetes.io/docs/concepts/services-networking/network-policies/#default-deny-all-egress-traffic
            this._policyTypes.add(direction);
        }
        if (_default === NetworkPolicyTrafficDefault.ALLOW) {
            // https://kubernetes.io/docs/concepts/services-networking/network-policies/#allow-all-egress-traffic
            this._policyTypes.add(direction);
            if (direction === 'Egress') {
                this._egressRules.push({});
            }
            else {
                this._ingressRules.push({});
            }
        }
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            podSelector: this._podSelectorConfig.labelSelector._toKube(),
            egress: utils_1.undefinedIfEmpty(this._egressRules),
            ingress: utils_1.undefinedIfEmpty(this._ingressRules),
            policyTypes: utils_1.undefinedIfEmpty(Array.from(this._policyTypes)),
        };
    }
}
exports.NetworkPolicy = NetworkPolicy;
_c = JSII_RTTI_SYMBOL_1;
NetworkPolicy[_c] = { fqn: "cdk8s-plus-23.NetworkPolicy", version: "2.0.0-rc.108" };
function validatePeerConfig(peerConfig) {
    if (!peerConfig.ipBlock && !peerConfig.podSelector) {
        throw new Error('Inavlid peer: either \'ipBlock\' or \'podSelector\' must be defined');
    }
    if (peerConfig.ipBlock && peerConfig.podSelector) {
        throw new Error('Inavlid peer: only one of \'ipBlock\' and \'podSelector\' must be defined');
    }
}
exports.validatePeerConfig = validatePeerConfig;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmV0d29yay1wb2xpY3kuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvbmV0d29yay1wb2xpY3kudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxpQ0FBd0M7QUFDeEMsMkNBQW1EO0FBQ25ELCtCQUErQjtBQUMvQixxQ0FBcUM7QUFDckMseUNBQXlDO0FBQ3pDLDZCQUE2QjtBQUM3QixtQ0FBMkM7QUE4QjNDOztHQUVHO0FBQ0gsTUFBYSxpQkFBaUI7SUFtRDVCLFlBQ21CLElBQXNCLEVBQ3RCLE9BQWdCLEVBQ2hCLFFBQTBCO1FBRjFCLFNBQUksR0FBSixJQUFJLENBQWtCO1FBQ3RCLFlBQU8sR0FBUCxPQUFPLENBQVM7UUFDaEIsYUFBUSxHQUFSLFFBQVEsQ0FBa0I7SUFBRyxDQUFDO0lBcERqRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxHQUFHLENBQUMsSUFBWTtRQUM1QixPQUFPLElBQUksaUJBQWlCLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLEVBQUUsU0FBUyxFQUFFLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNqRyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUFDLFNBQWlCLEVBQUUsT0FBZTtRQUN2RCxPQUFPLElBQUksaUJBQWlCLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLEVBQUUsT0FBTyxFQUFFLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNwRyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsTUFBTTtRQUNsQixPQUFPLElBQUksaUJBQWlCLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLEVBQUUsS0FBSyxFQUFFLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUMxRixDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsR0FBRyxDQUFDLElBQVk7UUFDNUIsT0FBTyxJQUFJLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLFNBQVMsRUFBRSxlQUFlLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDakcsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxTQUFpQixFQUFFLE9BQWU7UUFDdkQsT0FBTyxJQUFJLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxlQUFlLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDcEcsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLE1BQU07UUFDbEIsT0FBTyxJQUFJLGlCQUFpQixDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxFQUFFLEtBQUssRUFBRSxlQUFlLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDMUYsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxLQUE2QjtRQUM1QyxPQUFPLElBQUksaUJBQWlCLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLE9BQU8sRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDL0gsQ0FBQztJQU9EOztPQUVHO0lBQ0ksT0FBTztRQUNaLE9BQU8sRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksRUFBRSxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU8sRUFBRSxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDO0lBQzdFLENBQUM7O0FBN0RILDhDQStEQzs7O0FBc0REOzs7O0dBSUc7QUFDSCxNQUFhLG9CQUFxQixTQUFRLHNCQUFTO0lBbURqRCxZQUFvQixLQUFnQixFQUFFLEVBQVU7SUFDOUM7O09BRUc7SUFDYSxJQUFZO0lBQzVCOzs7T0FHRztJQUNhLE1BQWlCO1FBQ2pDLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFORCxTQUFJLEdBQUosSUFBSSxDQUFRO1FBS1osV0FBTSxHQUFOLE1BQU0sQ0FBVztJQUVuQyxDQUFDO0lBNUREOztPQUVHO0lBQ0ksTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxNQUFjLEVBQUUsTUFBaUI7UUFDaEYsTUFBTSxTQUFTLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBRWxFLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDZCxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixNQUFNLEdBQUcsQ0FBQyxDQUFDO1NBQ25EO1FBRUQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUNqQixNQUFNLElBQUksS0FBSyxDQUFDLGtDQUFrQyxNQUFNLG9CQUFvQixNQUFNLE9BQU8sQ0FBQyxDQUFDO1NBQzVGO1FBRUQsT0FBTyxJQUFJLG9CQUFvQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQzdELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBZ0IsRUFBRSxFQUFVO1FBQ2hELE9BQU8sSUFBSSxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLFdBQVcsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsTUFBYyxFQUFFLE1BQWlCO1FBRWhGLE1BQU0sU0FBUyxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztRQUVoRixJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ2QsTUFBTSxJQUFJLEtBQUssQ0FBQyx1QkFBdUIsTUFBTSxHQUFHLENBQUMsQ0FBQztTQUNuRDtRQUVELElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDakIsTUFBTSxJQUFJLEtBQUssQ0FBQyxrQ0FBa0MsTUFBTSxvQkFBb0IsTUFBTSxRQUFRLENBQUMsQ0FBQztTQUM3RjtRQUVELE9BQU8sSUFBSSxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsT0FBTyxDQUFDLEtBQWdCLEVBQUUsRUFBVTtRQUNoRCxPQUFPLElBQUksb0JBQW9CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUNyRCxDQUFDO0lBZUQ7O09BRUc7SUFDSSx5QkFBeUI7UUFDOUIsT0FBTyxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQztJQUMzQixDQUFDO0lBRUQ7O09BRUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFFRDs7T0FFRztJQUNJLE9BQU87UUFDWixPQUFPLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQztJQUNsRCxDQUFDOztBQW5GSCxvREFxRkM7OztBQUVEOztHQUVHO0FBQ0gsSUFBWSxlQWFYO0FBYkQsV0FBWSxlQUFlO0lBQ3pCOztPQUVHO0lBQ0gsOEJBQVcsQ0FBQTtJQUNYOztPQUVHO0lBQ0gsOEJBQVcsQ0FBQTtJQUNYOztPQUVHO0lBQ0gsZ0NBQWEsQ0FBQTtBQUNmLENBQUMsRUFiVyxlQUFlLEdBQWYsdUJBQWUsS0FBZix1QkFBZSxRQWExQjtBQUVEOztHQUVHO0FBQ0gsSUFBWSwyQkFhWDtBQWJELFdBQVksMkJBQTJCO0lBQ3JDOzs7O09BSUc7SUFDSCw0Q0FBYSxDQUFBO0lBQ2I7Ozs7T0FJRztJQUNILDhDQUFlLENBQUE7QUFDakIsQ0FBQyxFQWJXLDJCQUEyQixHQUEzQixtQ0FBMkIsS0FBM0IsbUNBQTJCLFFBYXRDO0FBeUVEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBNEJHO0FBQ0gsTUFBYSxhQUFjLFNBQVEsSUFBSSxDQUFDLFFBQVE7SUFjOUMsWUFBbUIsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBNEIsRUFBRTtRQUM3RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBUkgsaUJBQVksR0FBVyxpQkFBaUIsQ0FBQztRQUd4QyxpQkFBWSxHQUFrQyxFQUFFLENBQUM7UUFDakQsa0JBQWEsR0FBbUMsRUFBRSxDQUFDO1FBQ25ELGlCQUFZLEdBQWdCLElBQUksR0FBRyxFQUFFLENBQUM7UUFLckQsTUFBTSxXQUFXLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDcEUsSUFBSSxDQUFDLGtCQUFrQixHQUFHLFdBQVcsQ0FBQyxtQkFBbUIsRUFBRSxDQUFDO1FBRTVELElBQUksRUFBRSxDQUFDO1FBRVAsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsU0FBUyxFQUFFO1lBRTlCLElBQUksSUFBSSxDQUFDLGtCQUFrQixDQUFDLFVBQVUsRUFBRSxhQUFhLElBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsVUFBVSxFQUFFLGFBQWEsQ0FBQyxPQUFPLEVBQUUsRUFBRTtnQkFDckgsTUFBTSxJQUFJLEtBQUssQ0FBQyxxREFBcUQsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLG1EQUFtRCxDQUFDLENBQUM7YUFDaEo7WUFFRCxJQUFJLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxVQUFVLEVBQUUsS0FBSyxJQUFJLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQ3BHLE1BQU0sSUFBSSxLQUFLLENBQUMscURBQXFELFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSw0Q0FBNEMsQ0FBQyxDQUFDO2FBQ3pJO1lBRUQsRUFBRSxHQUFHLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1NBRTNHO2FBQU07WUFDTCxFQUFFLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUM7U0FDL0I7UUFFRCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDM0QsUUFBUSxFQUFFLEVBQUUsR0FBRyxLQUFLLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUU7WUFDOUMsSUFBSSxFQUFFLFlBQUksQ0FBQyxHQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUM7U0FDbEQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHdCQUF3QixDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQy9ELElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsQ0FBQztRQUVqRSxLQUFLLE1BQU0sSUFBSSxJQUFJLEtBQUssQ0FBQyxNQUFNLEVBQUUsS0FBSyxJQUFJLEVBQUUsRUFBRTtZQUM1QyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQzNDO1FBRUQsS0FBSyxNQUFNLElBQUksSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFLEtBQUssSUFBSSxFQUFFLEVBQUU7WUFDN0MsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztTQUM1QztJQUNILENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksYUFBYSxDQUFDLElBQXdCLEVBQUUsS0FBMkI7UUFDeEUsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDaEMsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsRUFBRSxLQUFLLEVBQUUsQ0FBQyxLQUFLLElBQUksRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLE9BQU8sRUFBRSxDQUFDLEVBQUUsRUFBRSxFQUFFLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDbEgsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxjQUFjLENBQUMsSUFBd0IsRUFBRSxLQUEyQjtRQUN6RSxJQUFJLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNqQyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxFQUFFLEtBQUssRUFBRSxDQUFDLEtBQUssSUFBSSxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsSUFBSSxDQUFDLHdCQUF3QixDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUNySCxDQUFDO0lBRU8sd0JBQXdCLENBQUMsSUFBd0I7UUFFdkQsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLHlCQUF5QixFQUFFLENBQUM7UUFFaEQsa0JBQWtCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFM0IsSUFBSSxNQUFNLENBQUMsT0FBTyxFQUFFO1lBQ2xCLDZCQUE2QjtZQUM3QixPQUFPLENBQUMsRUFBRSxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUM7U0FDaEQ7UUFFRCxJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVksQ0FBQyxVQUFVLEVBQUUsS0FBSyxFQUFFO1lBQzFDLDBEQUEwRDtZQUMxRCxxQkFBcUI7WUFDckIsT0FBTyxDQUFDO29CQUNOLGlCQUFpQixFQUFFLE1BQU0sQ0FBQyxXQUFZLENBQUMsVUFBVSxFQUFFLGFBQWEsRUFBRSxPQUFPLEVBQUU7b0JBQzNFLFdBQVcsRUFBRSxNQUFNLENBQUMsV0FBWSxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUU7aUJBQ3pELENBQUMsQ0FBQztTQUNKO1FBRUQscUVBQXFFO1FBQ3JFLCtGQUErRjtRQUMvRixNQUFNLGlCQUFpQixHQUFHLE1BQU0sQ0FBQyxXQUFXLEVBQUUsVUFBVSxFQUFFLGFBQWEsRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLENBQUM7UUFDekYsT0FBTyxNQUFNLENBQUMsV0FBWSxDQUFDLFVBQVUsQ0FBQyxLQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNyRCxXQUFXLEVBQUUsTUFBTSxDQUFDLFdBQVksQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFO1lBQ3hELGlCQUFpQixFQUFFO2dCQUNqQixnQkFBZ0IsRUFBRSxpQkFBaUIsQ0FBQyxnQkFBZ0I7Z0JBQ3BELFdBQVcsRUFBRTtvQkFDWCxHQUFHLGlCQUFpQixDQUFDLFdBQVc7b0JBQ2hDLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsRUFBRSxDQUFDO2lCQUNwQzthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDO0lBRU8sd0JBQXdCLENBQUMsU0FBK0IsRUFBRSxRQUFzQztRQUV0RyxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQUUsT0FBTztTQUFDO1FBRXpCLElBQUksUUFBUSxLQUFLLDJCQUEyQixDQUFDLElBQUksRUFBRTtZQUNqRCw0R0FBNEc7WUFDNUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDbEM7UUFFRCxJQUFJLFFBQVEsS0FBSywyQkFBMkIsQ0FBQyxLQUFLLEVBQUU7WUFDbEQscUdBQXFHO1lBQ3JHLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ2pDLElBQUksU0FBUyxLQUFLLFFBQVEsRUFBRTtnQkFDMUIsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7YUFDNUI7aUJBQU07Z0JBQ0wsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7YUFDN0I7U0FDRjtJQUNILENBQUM7SUFFRDs7T0FFRztJQUNJLE9BQU87UUFDWixPQUFPO1lBQ0wsV0FBVyxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxhQUFhLENBQUMsT0FBTyxFQUFFO1lBQzVELE1BQU0sRUFBRSx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDO1lBQzNDLE9BQU8sRUFBRSx3QkFBZ0IsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO1lBQzdDLFdBQVcsRUFBRSx3QkFBZ0IsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUM3RCxDQUFDO0lBQ0osQ0FBQzs7QUE1SUgsc0NBOElDOzs7QUFFRCxTQUFnQixrQkFBa0IsQ0FBQyxVQUFtQztJQUNwRSxJQUFJLENBQUMsVUFBVSxDQUFDLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLEVBQUU7UUFDbEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxxRUFBcUUsQ0FBQyxDQUFDO0tBQ3hGO0lBQ0QsSUFBSSxVQUFVLENBQUMsT0FBTyxJQUFJLFVBQVUsQ0FBQyxXQUFXLEVBQUU7UUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO0tBQzlGO0FBQ0gsQ0FBQztBQVBELGdEQU9DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXBpT2JqZWN0LCBMYXp5IH0gZnJvbSAnY2RrOHMnO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBJQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgKiBhcyBiYXNlIGZyb20gJy4vYmFzZSc7XG5pbXBvcnQgKiBhcyBrOHMgZnJvbSAnLi9pbXBvcnRzL2s4cyc7XG5pbXBvcnQgKiBhcyBuYW1lc3BhY2UgZnJvbSAnLi9uYW1lc3BhY2UnO1xuaW1wb3J0ICogYXMgcG9kIGZyb20gJy4vcG9kJztcbmltcG9ydCB7IHVuZGVmaW5lZElmRW1wdHkgfSBmcm9tICcuL3V0aWxzJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBgTmV0d29ya1BvbGljeVBvcnRgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE5ldHdvcmtQb2xpY3lQb3J0UHJvcHMge1xuXG4gIC8qKlxuICAgKiBTcGVjaWZpYyBwb3J0IG51bWJlci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBhbGwgcG9ydHMgYXJlIGFsbG93ZWQuXG4gICAqL1xuICByZWFkb25seSBwb3J0PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBFbmQgcG9ydCAocmVsYXRpdmUgdG8gYHBvcnRgKS4gT25seSBhcHBsaWVzIGlmIGBwb3J0YCBpcyBkZWZpbmVkLlxuICAgKiBVc2UgdGhpcyB0byBzcGVjaWZ5IGEgcG9ydCByYW5nZSwgcmF0aGVyIHRoYXQgYSBzcGVjaWZpYyBvbmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm90IGEgcG9ydCByYW5nZS5cbiAgICovXG4gIHJlYWRvbmx5IGVuZFBvcnQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFByb3RvY29sLlxuICAgKlxuICAgKiBAZGVmYXVsdCBOZXR3b3JrUHJvdG9jb2wuVENQXG4gICAqL1xuICByZWFkb25seSBwcm90b2NvbD86IE5ldHdvcmtQcm90b2NvbDtcbn1cblxuLyoqXG4gKiBEZXNjcmliZXMgYSBwb3J0IHRvIGFsbG93IHRyYWZmaWMgb24uXG4gKi9cbmV4cG9ydCBjbGFzcyBOZXR3b3JrUG9saWN5UG9ydCB7XG5cbiAgLyoqXG4gICAqIERpc3RpbmN0IFRDUCBwb3J0c1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyB0Y3AocG9ydDogbnVtYmVyKTogTmV0d29ya1BvbGljeVBvcnQge1xuICAgIHJldHVybiBuZXcgTmV0d29ya1BvbGljeVBvcnQoazhzLkludE9yU3RyaW5nLmZyb21OdW1iZXIocG9ydCksIHVuZGVmaW5lZCwgTmV0d29ya1Byb3RvY29sLlRDUCk7XG4gIH1cblxuICAvKipcbiAgICogQSBUQ1AgcG9ydCByYW5nZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyB0Y3BSYW5nZShzdGFydFBvcnQ6IG51bWJlciwgZW5kUG9ydDogbnVtYmVyKSB7XG4gICAgcmV0dXJuIG5ldyBOZXR3b3JrUG9saWN5UG9ydChrOHMuSW50T3JTdHJpbmcuZnJvbU51bWJlcihzdGFydFBvcnQpLCBlbmRQb3J0LCBOZXR3b3JrUHJvdG9jb2wuVENQKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBbnkgVENQIHRyYWZmaWNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYWxsVGNwKCkge1xuICAgIHJldHVybiBuZXcgTmV0d29ya1BvbGljeVBvcnQoazhzLkludE9yU3RyaW5nLmZyb21OdW1iZXIoMCksIDY1NTM1LCBOZXR3b3JrUHJvdG9jb2wuVENQKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBEaXN0aW5jdCBVRFAgcG9ydHNcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgdWRwKHBvcnQ6IG51bWJlcik6IE5ldHdvcmtQb2xpY3lQb3J0IHtcbiAgICByZXR1cm4gbmV3IE5ldHdvcmtQb2xpY3lQb3J0KGs4cy5JbnRPclN0cmluZy5mcm9tTnVtYmVyKHBvcnQpLCB1bmRlZmluZWQsIE5ldHdvcmtQcm90b2NvbC5VRFApO1xuICB9XG5cbiAgLyoqXG4gICAqIEEgVURQIHBvcnQgcmFuZ2VcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgdWRwUmFuZ2Uoc3RhcnRQb3J0OiBudW1iZXIsIGVuZFBvcnQ6IG51bWJlcikge1xuICAgIHJldHVybiBuZXcgTmV0d29ya1BvbGljeVBvcnQoazhzLkludE9yU3RyaW5nLmZyb21OdW1iZXIoc3RhcnRQb3J0KSwgZW5kUG9ydCwgTmV0d29ya1Byb3RvY29sLlVEUCk7XG4gIH1cblxuICAvKipcbiAgICogQW55IFVEUCB0cmFmZmljXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFsbFVkcCgpIHtcbiAgICByZXR1cm4gbmV3IE5ldHdvcmtQb2xpY3lQb3J0KGs4cy5JbnRPclN0cmluZy5mcm9tTnVtYmVyKDApLCA2NTUzNSwgTmV0d29ya1Byb3RvY29sLlVEUCk7XG4gIH1cblxuICAvKipcbiAgICogQ3VzdG9tIHBvcnQgY29uZmlndXJhdGlvbi5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgb2YocHJvcHM6IE5ldHdvcmtQb2xpY3lQb3J0UHJvcHMpOiBOZXR3b3JrUG9saWN5UG9ydCB7XG4gICAgcmV0dXJuIG5ldyBOZXR3b3JrUG9saWN5UG9ydChwcm9wcy5wb3J0ID8gazhzLkludE9yU3RyaW5nLmZyb21OdW1iZXIocHJvcHMucG9ydCkgOiB1bmRlZmluZWQsIHByb3BzLmVuZFBvcnQsIHByb3BzLnByb3RvY29sKTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoXG4gICAgcHJpdmF0ZSByZWFkb25seSBwb3J0PzogazhzLkludE9yU3RyaW5nLFxuICAgIHByaXZhdGUgcmVhZG9ubHkgZW5kUG9ydD86IG51bWJlcixcbiAgICBwcml2YXRlIHJlYWRvbmx5IHByb3RvY29sPzogTmV0d29ya1Byb3RvY29sKSB7fVxuXG4gIC8qKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHB1YmxpYyBfdG9LdWJlKCk6IGs4cy5OZXR3b3JrUG9saWN5UG9ydCB7XG4gICAgcmV0dXJuIHsgcG9ydDogdGhpcy5wb3J0LCBlbmRQb3J0OiB0aGlzLmVuZFBvcnQsIHByb3RvY29sOiB0aGlzLnByb3RvY29sIH07XG4gIH1cblxufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIG5ldHdvcmsgcGVlcnMuXG4gKiBBIHBlZXIgY2FuIGVpdGhlciBieSBhbiBpcCBibG9jaywgb3IgYSBzZWxlY3Rpb24gb2YgcG9kcywgbm90IGJvdGguXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTmV0d29ya1BvbGljeVBlZXJDb25maWcge1xuXG4gIC8qKlxuICAgKiBUaGUgaXAgYmxvY2sgdGhpcyBwZWVyIHJlcHJlc2VudHMuXG4gICAqL1xuICByZWFkb25seSBpcEJsb2NrPzogTmV0d29ya1BvbGljeUlwQmxvY2s7XG5cbiAgLyoqXG4gICAqIFRoZSBwb2Qgc2VsZWN0b3IgdGhpcyBwZWVyIHJlcHJlc2VudHMuXG4gICAqL1xuICByZWFkb25seSBwb2RTZWxlY3Rvcj86IHBvZC5Qb2RTZWxlY3RvckNvbmZpZztcblxufVxuXG4vKipcbiAqIERlc2NyaWJlcyBhIHBlZXIgdG8gYWxsb3cgdHJhZmZpYyB0by9mcm9tLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElOZXR3b3JrUG9saWN5UGVlciBleHRlbmRzIElDb25zdHJ1Y3Qge1xuICAvKipcbiAgICogUmV0dXJuIHRoZSBjb25maWd1cmF0aW9uIG9mIHRoaXMgcGVlci5cbiAgICovXG4gIHRvTmV0d29ya1BvbGljeVBlZXJDb25maWcoKTogTmV0d29ya1BvbGljeVBlZXJDb25maWc7XG5cbiAgLyoqXG4gICAqIENvbnZlcnQgdGhlIHBlZXIgaW50byBhIHBvZCBzZWxlY3RvciwgaWYgcG9zc2libGUuXG4gICAqL1xuICB0b1BvZFNlbGVjdG9yKCk6IHBvZC5JUG9kU2VsZWN0b3IgfCB1bmRlZmluZWQ7XG59XG5cbi8qKlxuICogRGVzY3JpYmVzIGEgcnVsZSBhbGxvd2luZyB0cmFmZmljIGZyb20gLyB0byBwb2RzIG1hdGNoZWQgYnkgYSBuZXR3b3JrIHBvbGljeSBzZWxlY3Rvci5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOZXR3b3JrUG9saWN5UnVsZSB7XG5cbiAgLyoqXG4gICAqIFRoZSBwb3J0cyBvZiB0aGUgcnVsZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cmFmZmljIGlzIGFsbG93ZWQgb24gYWxsIHBvcnRzLlxuICAgKi9cbiAgcmVhZG9ubHkgcG9ydHM/OiBOZXR3b3JrUG9saWN5UG9ydFtdO1xuXG4gIC8qKlxuICAgKiBQZWVyIHRoaXMgcnVsZSBpbnRlcmFjdHMgd2l0aC5cbiAgICovXG4gIHJlYWRvbmx5IHBlZXI6IElOZXR3b3JrUG9saWN5UGVlcjtcblxufVxuXG4vKipcbiAqIERlc2NyaWJlcyBhIHBhcnRpY3VsYXIgQ0lEUiAoRXguIFwiMTkyLjE2OC4xLjEvMjRcIixcIjIwMDE6ZGI5OjovNjRcIikgdGhhdCBpc1xuICogYWxsb3dlZCB0byB0aGUgcG9kcyBtYXRjaGVkIGJ5IGEgbmV0d29yayBwb2xpY3kgc2VsZWN0b3IuXG4gKiBUaGUgZXhjZXB0IGVudHJ5IGRlc2NyaWJlcyBDSURScyB0aGF0IHNob3VsZCBub3QgYmUgaW5jbHVkZWQgd2l0aGluIHRoaXMgcnVsZS5cbiAqL1xuZXhwb3J0IGNsYXNzIE5ldHdvcmtQb2xpY3lJcEJsb2NrIGV4dGVuZHMgQ29uc3RydWN0IGltcGxlbWVudHMgSU5ldHdvcmtQb2xpY3lQZWVyIHtcblxuICAvKipcbiAgICogQ3JlYXRlIGFuIElQdjQgcGVlciBmcm9tIGEgQ0lEUlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBpcHY0KHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGNpZHJJcDogc3RyaW5nLCBleGNlcHQ/OiBzdHJpbmdbXSk6IE5ldHdvcmtQb2xpY3lJcEJsb2NrIHtcbiAgICBjb25zdCBjaWRyTWF0Y2ggPSBjaWRySXAubWF0Y2goL14oXFxkezEsM31cXC4pezN9XFxkezEsM30oXFwvXFxkKyk/JC8pO1xuXG4gICAgaWYgKCFjaWRyTWF0Y2gpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBJUHY0IENJRFI6IFwiJHtjaWRySXB9XCJgKTtcbiAgICB9XG5cbiAgICBpZiAoIWNpZHJNYXRjaFsyXSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDSURSIG1hc2sgaXMgbWlzc2luZyBpbiBJUHY0OiBcIiR7Y2lkcklwfVwiLiBEaWQgeW91IG1lYW4gXCIke2NpZHJJcH0vMzJcIj9gKTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IE5ldHdvcmtQb2xpY3lJcEJsb2NrKHNjb3BlLCBpZCwgY2lkcklwLCBleGNlcHQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFueSBJUHY0IGFkZHJlc3NcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgYW55SXB2NChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nKTogTmV0d29ya1BvbGljeUlwQmxvY2sge1xuICAgIHJldHVybiBuZXcgTmV0d29ya1BvbGljeUlwQmxvY2soc2NvcGUsIGlkLCAnMC4wLjAuMC8wJyk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGFuIElQdjYgcGVlciBmcm9tIGEgQ0lEUlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBpcHY2KHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGNpZHJJcDogc3RyaW5nLCBleGNlcHQ/OiBzdHJpbmdbXSk6IE5ldHdvcmtQb2xpY3lJcEJsb2NrIHtcblxuICAgIGNvbnN0IGNpZHJNYXRjaCA9IGNpZHJJcC5tYXRjaCgvXihbXFxkYS1mXXswLDR9Oil7Miw3fShbXFxkYS1mXXswLDR9KT8oXFwvXFxkKyk/JC8pO1xuXG4gICAgaWYgKCFjaWRyTWF0Y2gpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBJUHY2IENJRFI6IFwiJHtjaWRySXB9XCJgKTtcbiAgICB9XG5cbiAgICBpZiAoIWNpZHJNYXRjaFszXSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDSURSIG1hc2sgaXMgbWlzc2luZyBpbiBJUHY2OiBcIiR7Y2lkcklwfVwiLiBEaWQgeW91IG1lYW4gXCIke2NpZHJJcH0vMTI4XCI/YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBOZXR3b3JrUG9saWN5SXBCbG9jayhzY29wZSwgaWQsIGNpZHJJcCwgZXhjZXB0KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBbnkgSVB2NiBhZGRyZXNzXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGFueUlwdjYoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZyk6IE5ldHdvcmtQb2xpY3lJcEJsb2NrIHtcbiAgICByZXR1cm4gbmV3IE5ldHdvcmtQb2xpY3lJcEJsb2NrKHNjb3BlLCBpZCwgJzo6LzAnKTtcbiAgfVxuXG4gIHByaXZhdGUgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZyxcbiAgICAvKipcbiAgICAgKiBBIHN0cmluZyByZXByZXNlbnRpbmcgdGhlIElQIEJsb2NrIFZhbGlkIGV4YW1wbGVzIGFyZSBcIjE5Mi4xNjguMS4xLzI0XCIgb3IgXCIyMDAxOmRiOTo6LzY0XCIuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGNpZHI6IHN0cmluZyxcbiAgICAvKipcbiAgICAgKiBBIHNsaWNlIG9mIENJRFJzIHRoYXQgc2hvdWxkIG5vdCBiZSBpbmNsdWRlZCB3aXRoaW4gYW4gSVAgQmxvY2sgVmFsaWQgZXhhbXBsZXMgYXJlIFwiMTkyLjE2OC4xLjEvMjRcIiBvciBcIjIwMDE6ZGI5OjovNjRcIi5cbiAgICAgKiBFeGNlcHQgdmFsdWVzIHdpbGwgYmUgcmVqZWN0ZWQgaWYgdGhleSBhcmUgb3V0c2lkZSB0aGUgQ0lEUiByYW5nZS5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgZXhjZXB0Pzogc3RyaW5nW10pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEBzZWUgSU5ldHdvcmtQb2xpY3lQZWVyLnRvTmV0d29ya1BvbGljeVBlZXJDb25maWcoKVxuICAgKi9cbiAgcHVibGljIHRvTmV0d29ya1BvbGljeVBlZXJDb25maWcoKTogTmV0d29ya1BvbGljeVBlZXJDb25maWcge1xuICAgIHJldHVybiB7IGlwQmxvY2s6IHRoaXMgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBAc2VlIElOZXR3b3JrUG9saWN5UGVlci50b1BvZFNlbGVjdG9yKClcbiAgICovXG4gIHB1YmxpYyB0b1BvZFNlbGVjdG9yKCk6IHBvZC5JUG9kU2VsZWN0b3IgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cblxuICAvKipcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX3RvS3ViZSgpOiBrOHMuSXBCbG9jayB7XG4gICAgcmV0dXJuIHsgY2lkcjogdGhpcy5jaWRyLCBleGNlcHQ6IHRoaXMuZXhjZXB0IH07XG4gIH1cblxufVxuXG4vKipcbiAqIE5ldHdvcmsgcHJvdG9jb2xzLlxuICovXG5leHBvcnQgZW51bSBOZXR3b3JrUHJvdG9jb2wge1xuICAvKipcbiAgICogVENQLlxuICAgKi9cbiAgVENQID0gJ1RDUCcsXG4gIC8qKlxuICAgKiBVRFAuXG4gICAqL1xuICBVRFAgPSAnVURQJyxcbiAgLyoqXG4gICAqIFNDVFAuXG4gICAqL1xuICBTQ1RQID0gJ1NDVFAnLFxufVxuXG4vKipcbiAqIERlZmF1bHQgYmVoYXZpb3JzIG9mIG5ldHdvcmsgdHJhZmZpYyBpbiBwb2xpY2llcy5cbiAqL1xuZXhwb3J0IGVudW0gTmV0d29ya1BvbGljeVRyYWZmaWNEZWZhdWx0IHtcbiAgLyoqXG4gICAqIFRoZSBwb2xpY3kgZGVuaWVzIGFsbCB0cmFmZmljLlxuICAgKiBTaW5jZSBydWxlcyBhcmUgYWRkaXRpdmUsIGFkZGl0aW9uYWwgcnVsZXMgb3IgcG9saWNpZXMgY2FuIGFsbG93XG4gICAqIHNwZWNpZmljIHRyYWZmaWMuXG4gICAqL1xuICBERU5ZID0gJ0RFTlknLFxuICAvKipcbiAgICogVGhlIHBvbGljeSBhbGxvd3MgYWxsIHRyYWZmaWMgKGVpdGhlciBpbmdyZXNzIG9yIGVncmVzcykuXG4gICAqIFNpbmNlIHJ1bGVzIGFyZSBhZGRpdGl2ZSwgbm8gYWRkaXRpb25hbCBydWxlIG9yIHBvbGljaWVzIGNhblxuICAgKiBzdWJzZXF1ZW50bHkgZGVueSB0aGUgdHJhZmZpYy5cbiAgICovXG4gIEFMTE9XID0gJ0FMTE9XJyxcbn1cblxuLyoqXG4gKiBEZXNjcmliZXMgaG93IHRoZSBuZXR3b3JrIHBvbGljeSBzaG91bGQgY29uZmlndXJlIGVncmVzcyAvIGluZ3Jlc3MgdHJhZmZpYy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOZXR3b3JrUG9saWN5VHJhZmZpYyB7XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgZGVmYXVsdCBiZWhhdmlvciBvZiB0aGUgcG9saWN5IHdoZW5cbiAgICogbm8gcnVsZXMgYXJlIGRlZmluZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdW5zZXQsIHRoZSBwb2xpY3kgZG9lcyBub3QgY2hhbmdlIHRoZSBiZWhhdmlvci5cbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHQ/OiBOZXR3b3JrUG9saWN5VHJhZmZpY0RlZmF1bHQ7XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcnVsZXMgdG8gYmUgYXBwbGllZCB0byB0aGUgc2VsZWN0ZWQgcG9kcy5cbiAgICogSWYgZW1wdHksIHRoZSBiZWhhdmlvciBvZiB0aGUgcG9saWN5IGlzIGRpY3RhdGVkIGJ5IHRoZSBgZGVmYXVsdGAgcHJvcGVydHkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gcnVsZXNcbiAgICovXG4gIHJlYWRvbmx5IHJ1bGVzPzogTmV0d29ya1BvbGljeVJ1bGVbXTtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgTmV0d29ya1BvbGljeS5hZGRFZ3Jlc3NSdWxlYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOZXR3b3JrUG9saWN5QWRkRWdyZXNzUnVsZU9wdGlvbnMge1xuXG4gIC8qKlxuICAgKiBQb3J0cyB0aGUgcnVsZSBzaG91bGQgYWxsb3cgb3V0Z29pbmcgdHJhZmZpYyB0by5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBJZiB0aGUgcGVlciBpcyBhIG1hbmFnZWQgcG9kLCB0YWtlIGl0cyBwb3J0cy4gT3RoZXJ3aXNlLCBhbGwgcG9ydHMgYXJlIGFsbG93ZWQuXG4gICAqL1xuICByZWFkb25seSBwb3J0cz86IE5ldHdvcmtQb2xpY3lQb3J0W107XG5cbn1cblxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGBOZXR3b3JrUG9saWN5YC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBOZXR3b3JrUG9saWN5UHJvcHMgZXh0ZW5kcyBiYXNlLlJlc291cmNlUHJvcHMge1xuXG4gIC8qKlxuICAgKiBXaGljaCBwb2RzIGRvZXMgdGhpcyBwb2xpY3kgb2JqZWN0IGFwcGxpZXMgdG8uXG4gICAqXG4gICAqIFRoaXMgY2FuIGVpdGhlciBiZSBhIHNpbmdsZSBwb2QgLyB3b3JrbG9hZCwgb3IgYSBncm91cGluZyBvZiBwb2RzIHNlbGVjdGVkXG4gICAqIHZpYSB0aGUgYFBvZHMuc2VsZWN0YCBmdW5jdGlvbi4gUnVsZXMgaXMgYXBwbGllZCB0byBhbnkgcG9kcyBzZWxlY3RlZCBieSB0aGlzIHByb3BlcnR5LlxuICAgKiBNdWx0aXBsZSBuZXR3b3JrIHBvbGljaWVzIGNhbiBzZWxlY3QgdGhlIHNhbWUgc2V0IG9mIHBvZHMuXG4gICAqIEluIHRoaXMgY2FzZSwgdGhlIHJ1bGVzIGZvciBlYWNoIGFyZSBjb21iaW5lZCBhZGRpdGl2ZWx5LlxuICAgKlxuICAgKiBOb3RlIHRoYXRcbiAgICpcbiAgICogQGRlZmF1bHQgLSB3aWxsIHNlbGVjdCBhbGwgcG9kcyBpbiB0aGUgbmFtZXNwYWNlIG9mIHRoZSBwb2xpY3kuXG4gICAqL1xuICByZWFkb25seSBzZWxlY3Rvcj86IHBvZC5JUG9kU2VsZWN0b3I7XG5cbiAgLyoqXG4gICAqIEVncmVzcyB0cmFmZmljIGNvbmZpZ3VyYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIHBvbGljeSBkb2Vzbid0IGNoYW5nZSBlZ3Jlc3MgYmVoYXZpb3Igb2YgdGhlIHBvZHMgaXQgc2VsZWN0cy5cbiAgICovXG4gIHJlYWRvbmx5IGVncmVzcz86IE5ldHdvcmtQb2xpY3lUcmFmZmljO1xuXG4gIC8qKlxuICAgKiBJbmdyZXNzIHRyYWZmaWMgY29uZmlndXJhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0aGUgcG9saWN5IGRvZXNuJ3QgY2hhbmdlIGluZ3Jlc3MgYmVoYXZpb3Igb2YgdGhlIHBvZHMgaXQgc2VsZWN0cy5cbiAgICovXG4gIHJlYWRvbmx5IGluZ3Jlc3M/OiBOZXR3b3JrUG9saWN5VHJhZmZpYztcbn1cblxuLyoqXG4gKiBDb250cm9sIHRyYWZmaWMgZmxvdyBhdCB0aGUgSVAgYWRkcmVzcyBvciBwb3J0IGxldmVsIChPU0kgbGF5ZXIgMyBvciA0KSxcbiAqIG5ldHdvcmsgcG9saWNpZXMgYXJlIGFuIGFwcGxpY2F0aW9uLWNlbnRyaWMgY29uc3RydWN0IHdoaWNoIGFsbG93IHlvdVxuICogdG8gc3BlY2lmeSBob3cgYSBwb2QgaXMgYWxsb3dlZCB0byBjb21tdW5pY2F0ZSB3aXRoIHZhcmlvdXMgbmV0d29yayBwZWVycy5cbiAqXG4gKiAtIE91dGdvaW5nIHRyYWZmaWMgaXMgYWxsb3dlZCBpZiB0aGVyZSBhcmUgbm8gbmV0d29yayBwb2xpY2llcyBzZWxlY3RpbmdcbiAqICAgdGhlIHBvZCAoYW5kIGNsdXN0ZXIgcG9saWN5IG90aGVyd2lzZSBhbGxvd3MgdGhlIHRyYWZmaWMpLFxuICogICBPUiBpZiB0aGUgdHJhZmZpYyBtYXRjaGVzIGF0IGxlYXN0IG9uZSBlZ3Jlc3MgcnVsZSBhY3Jvc3MgYWxsIG9mIHRoZVxuICogICBuZXR3b3JrIHBvbGljaWVzIHRoYXQgc2VsZWN0IHRoZSBwb2QuXG4gKlxuICogLSBJbmNvbWluZyB0cmFmZmljIGlzIGFsbG93ZWQgdG8gYSBwb2QgaWYgdGhlcmUgYXJlIG5vIG5ldHdvcmsgcG9saWNpZXNcbiAqICAgc2VsZWN0aW5nIHRoZSBwb2QgKGFuZCBjbHVzdGVyIHBvbGljeSBvdGhlcndpc2UgYWxsb3dzIHRoZSB0cmFmZmljKSxcbiAqICAgT1IgaWYgdGhlIHRyYWZmaWMgc291cmNlIGlzIHRoZSBwb2QncyBsb2NhbCBub2RlLFxuICogICBPUiBpZiB0aGUgdHJhZmZpYyBtYXRjaGVzIGF0IGxlYXN0IG9uZSBpbmdyZXNzIHJ1bGUgYWNyb3NzIGFsbCBvZlxuICogICB0aGUgbmV0d29yayBwb2xpY2llcyB0aGF0IHNlbGVjdCB0aGUgcG9kLlxuICpcbiAqIE5ldHdvcmsgcG9saWNpZXMgZG8gbm90IGNvbmZsaWN0OyB0aGV5IGFyZSBhZGRpdGl2ZS5cbiAqIElmIGFueSBwb2xpY3kgb3IgcG9saWNpZXMgYXBwbHkgdG8gYSBnaXZlbiBwb2QgZm9yIGEgZ2l2ZW5cbiAqIGRpcmVjdGlvbiwgdGhlIGNvbm5lY3Rpb25zIGFsbG93ZWQgaW4gdGhhdCBkaXJlY3Rpb24gZnJvbVxuICogdGhhdCBwb2QgaXMgdGhlIHVuaW9uIG9mIHdoYXQgdGhlIGFwcGxpY2FibGUgcG9saWNpZXMgYWxsb3cuXG4gKiBUaHVzLCBvcmRlciBvZiBldmFsdWF0aW9uIGRvZXMgbm90IGFmZmVjdCB0aGUgcG9saWN5IHJlc3VsdC5cbiAqXG4gKiBGb3IgYSBjb25uZWN0aW9uIGZyb20gYSBzb3VyY2UgcG9kIHRvIGEgZGVzdGluYXRpb24gcG9kIHRvIGJlIGFsbG93ZWQsXG4gKiBib3RoIHRoZSBlZ3Jlc3MgcG9saWN5IG9uIHRoZSBzb3VyY2UgcG9kIGFuZCB0aGUgaW5ncmVzcyBwb2xpY3kgb24gdGhlXG4gKiBkZXN0aW5hdGlvbiBwb2QgbmVlZCB0byBhbGxvdyB0aGUgY29ubmVjdGlvbi5cbiAqIElmIGVpdGhlciBzaWRlIGRvZXMgbm90IGFsbG93IHRoZSBjb25uZWN0aW9uLCBpdCB3aWxsIG5vdCBoYXBwZW4uXG4gKlxuICogQHNlZSBodHRwczovL2t1YmVybmV0ZXMuaW8vZG9jcy9jb25jZXB0cy9zZXJ2aWNlcy1uZXR3b3JraW5nL25ldHdvcmstcG9saWNpZXMvI25ldHdvcmtwb2xpY3ktcmVzb3VyY2VcbiAqL1xuZXhwb3J0IGNsYXNzIE5ldHdvcmtQb2xpY3kgZXh0ZW5kcyBiYXNlLlJlc291cmNlIHtcblxuICAvKipcbiAgICogQHNlZSBiYXNlLlJlc291cmNlLmFwaU9iamVjdFxuICAgKi9cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IGFwaU9iamVjdDogQXBpT2JqZWN0O1xuXG4gIHB1YmxpYyByZWFkb25seSByZXNvdXJjZVR5cGU6IHN0cmluZyA9ICduZXR3b3JrcG9saWNpZXMnO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX3BvZFNlbGVjdG9yQ29uZmlnOiBwb2QuUG9kU2VsZWN0b3JDb25maWc7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2VncmVzc1J1bGVzOiBrOHMuTmV0d29ya1BvbGljeUVncmVzc1J1bGVbXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IF9pbmdyZXNzUnVsZXM6IGs4cy5OZXR3b3JrUG9saWN5SW5ncmVzc1J1bGVbXSA9IFtdO1xuICBwcml2YXRlIHJlYWRvbmx5IF9wb2xpY3lUeXBlczogU2V0PHN0cmluZz4gPSBuZXcgU2V0KCk7XG5cbiAgcHVibGljIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBOZXR3b3JrUG9saWN5UHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBwb2RTZWxlY3RvciA9IHByb3BzLnNlbGVjdG9yID8/IHBvZC5Qb2RzLmFsbCh0aGlzLCAnQWxsUG9kcycpO1xuICAgIHRoaXMuX3BvZFNlbGVjdG9yQ29uZmlnID0gcG9kU2VsZWN0b3IudG9Qb2RTZWxlY3RvckNvbmZpZygpO1xuXG4gICAgbGV0IG5zO1xuXG4gICAgaWYgKCFwcm9wcy5tZXRhZGF0YT8ubmFtZXNwYWNlKSB7XG5cbiAgICAgIGlmICh0aGlzLl9wb2RTZWxlY3RvckNvbmZpZy5uYW1lc3BhY2VzPy5sYWJlbFNlbGVjdG9yICYmICF0aGlzLl9wb2RTZWxlY3RvckNvbmZpZy5uYW1lc3BhY2VzPy5sYWJlbFNlbGVjdG9yLmlzRW1wdHkoKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVuYWJsZSB0byBjcmVhdGUgYSBuZXR3b3JrIHBvbGljeSBmb3IgYSBzZWxlY3RvciAoJHtwb2RTZWxlY3Rvci5ub2RlLnBhdGh9KSB0aGF0IHNlbGVjdHMgcG9kcyBpbiBuYW1lc3BhY2VzIGJhc2VkIG9uIGxhYmVsc2ApO1xuICAgICAgfVxuXG4gICAgICBpZiAodGhpcy5fcG9kU2VsZWN0b3JDb25maWcubmFtZXNwYWNlcz8ubmFtZXMgJiYgdGhpcy5fcG9kU2VsZWN0b3JDb25maWcubmFtZXNwYWNlcy5uYW1lcy5sZW5ndGggPiAxKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5hYmxlIHRvIGNyZWF0ZSBhIG5ldHdvcmsgcG9saWN5IGZvciBhIHNlbGVjdG9yICgke3BvZFNlbGVjdG9yLm5vZGUucGF0aH0pIHRoYXQgc2VsZWN0cyBwb2RzIGluIG11bHRpcGxlIG5hbWVzcGFjZXNgKTtcbiAgICAgIH1cblxuICAgICAgbnMgPSB0aGlzLl9wb2RTZWxlY3RvckNvbmZpZy5uYW1lc3BhY2VzPy5uYW1lcyA/IHRoaXMuX3BvZFNlbGVjdG9yQ29uZmlnLm5hbWVzcGFjZXM/Lm5hbWVzWzBdIDogdW5kZWZpbmVkO1xuXG4gICAgfSBlbHNlIHtcbiAgICAgIG5zID0gcHJvcHMubWV0YWRhdGEubmFtZXNwYWNlO1xuICAgIH1cblxuICAgIHRoaXMuYXBpT2JqZWN0ID0gbmV3IGs4cy5LdWJlTmV0d29ya1BvbGljeSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBtZXRhZGF0YTogeyAuLi5wcm9wcy5tZXRhZGF0YSwgbmFtZXNwYWNlOiBucyB9LFxuICAgICAgc3BlYzogTGF6eS5hbnkoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLl90b0t1YmUoKSB9KSxcbiAgICB9KTtcblxuICAgIHRoaXMuY29uZmlndXJlRGVmYXVsdEJlaGF2aW9yKCdFZ3Jlc3MnLCBwcm9wcy5lZ3Jlc3M/LmRlZmF1bHQpO1xuICAgIHRoaXMuY29uZmlndXJlRGVmYXVsdEJlaGF2aW9yKCdJbmdyZXNzJywgcHJvcHMuaW5ncmVzcz8uZGVmYXVsdCk7XG5cbiAgICBmb3IgKGNvbnN0IHJ1bGUgb2YgcHJvcHMuZWdyZXNzPy5ydWxlcyA/PyBbXSkge1xuICAgICAgdGhpcy5hZGRFZ3Jlc3NSdWxlKHJ1bGUucGVlciwgcnVsZS5wb3J0cyk7XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBydWxlIG9mIHByb3BzLmluZ3Jlc3M/LnJ1bGVzID8/IFtdKSB7XG4gICAgICB0aGlzLmFkZEluZ3Jlc3NSdWxlKHJ1bGUucGVlciwgcnVsZS5wb3J0cyk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFsbG93IG91dGdvaW5nIHRyYWZmaWMgdG8gdGhlIHBlZXIuXG4gICAqXG4gICAqIElmIHBvcnRzIGFyZSBub3QgcGFzc2VkLCB0cmFmZmljIHdpbGwgYmUgYWxsb3dlZCBvbiBhbGwgcG9ydHMuXG4gICAqL1xuICBwdWJsaWMgYWRkRWdyZXNzUnVsZShwZWVyOiBJTmV0d29ya1BvbGljeVBlZXIsIHBvcnRzPzogTmV0d29ya1BvbGljeVBvcnRbXSkge1xuICAgIHRoaXMuX3BvbGljeVR5cGVzLmFkZCgnRWdyZXNzJyk7XG4gICAgdGhpcy5fZWdyZXNzUnVsZXMucHVzaCh7IHBvcnRzOiAocG9ydHMgPz8gW10pLm1hcChwID0+IHAuX3RvS3ViZSgpKSwgdG86IHRoaXMuY3JlYXRlTmV0d29ya1BvbGljeVBlZXJzKHBlZXIpIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFsbG93IGluY29taW5nIHRyYWZmaWMgZnJvbSB0aGUgcGVlci5cbiAgICpcbiAgICogSWYgcG9ydHMgYXJlIG5vdCBwYXNzZWQsIHRyYWZmaWMgd2lsbCBiZSBhbGxvd2VkIG9uIGFsbCBwb3J0cy5cbiAgICovXG4gIHB1YmxpYyBhZGRJbmdyZXNzUnVsZShwZWVyOiBJTmV0d29ya1BvbGljeVBlZXIsIHBvcnRzPzogTmV0d29ya1BvbGljeVBvcnRbXSkge1xuICAgIHRoaXMuX3BvbGljeVR5cGVzLmFkZCgnSW5ncmVzcycpO1xuICAgIHRoaXMuX2luZ3Jlc3NSdWxlcy5wdXNoKHsgcG9ydHM6IChwb3J0cyA/PyBbXSkubWFwKHAgPT4gcC5fdG9LdWJlKCkpLCBmcm9tOiB0aGlzLmNyZWF0ZU5ldHdvcmtQb2xpY3lQZWVycyhwZWVyKSB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlTmV0d29ya1BvbGljeVBlZXJzKHBlZXI6IElOZXR3b3JrUG9saWN5UGVlcik6IGs4cy5OZXR3b3JrUG9saWN5UGVlcltdIHtcblxuICAgIGNvbnN0IGNvbmZpZyA9IHBlZXIudG9OZXR3b3JrUG9saWN5UGVlckNvbmZpZygpO1xuXG4gICAgdmFsaWRhdGVQZWVyQ29uZmlnKGNvbmZpZyk7XG5cbiAgICBpZiAoY29uZmlnLmlwQmxvY2spIHtcbiAgICAgIC8vIGlwIGJsb2NrIGlzIGEgc2luZ2xlIHBlZXIuXG4gICAgICByZXR1cm4gW3sgaXBCbG9jazogY29uZmlnLmlwQmxvY2suX3RvS3ViZSgpIH1dO1xuICAgIH1cblxuICAgIGlmICghY29uZmlnLnBvZFNlbGVjdG9yIS5uYW1lc3BhY2VzPy5uYW1lcykge1xuICAgICAgLy8gd2hlbiBubyBleHBsaWNpdCBuYW1lc3BhY2VzIGFyZSBkZWZpbmVkIHdlIGNhbiBqdXN0IHVzZVxuICAgICAgLy8gdGhlIHNlbGVjdG9yIGFzIGlzXG4gICAgICByZXR1cm4gW3tcbiAgICAgICAgbmFtZXNwYWNlU2VsZWN0b3I6IGNvbmZpZy5wb2RTZWxlY3RvciEubmFtZXNwYWNlcz8ubGFiZWxTZWxlY3Rvcj8uX3RvS3ViZSgpLFxuICAgICAgICBwb2RTZWxlY3RvcjogY29uZmlnLnBvZFNlbGVjdG9yIS5sYWJlbFNlbGVjdG9yLl90b0t1YmUoKSxcbiAgICAgIH1dO1xuICAgIH1cblxuICAgIC8vIHdoZW4gZXhwbGljaXQgbmFtZXNwYWNlcyBhcmUgZGVmaW5lZCwgd2UgbmVlZCB0byBjcmVhdGUgYSBzZXBhcmF0ZVxuICAgIC8vIHBlZXIgZm9yIGVhY2gsIHNpbmNlIGEgbGFiZWwgc2VsZWN0b3IgY2Fubm90IGhhdmUgbXVsdGlwbGUgbmFtZSBsYWJlbHMuICh0aGV5IHdpbGwgY29uZmxpY3QpXG4gICAgY29uc3QgbmFtZXNwYWNlU2VsZWN0b3IgPSBjb25maWcucG9kU2VsZWN0b3I/Lm5hbWVzcGFjZXM/LmxhYmVsU2VsZWN0b3I/Ll90b0t1YmUoKSA/PyB7fTtcbiAgICByZXR1cm4gY29uZmlnLnBvZFNlbGVjdG9yIS5uYW1lc3BhY2VzLm5hbWVzIS5tYXAobiA9PiAoe1xuICAgICAgcG9kU2VsZWN0b3I6IGNvbmZpZy5wb2RTZWxlY3RvciEubGFiZWxTZWxlY3Rvci5fdG9LdWJlKCksXG4gICAgICBuYW1lc3BhY2VTZWxlY3Rvcjoge1xuICAgICAgICBtYXRjaEV4cHJlc3Npb25zOiBuYW1lc3BhY2VTZWxlY3Rvci5tYXRjaEV4cHJlc3Npb25zLFxuICAgICAgICBtYXRjaExhYmVsczoge1xuICAgICAgICAgIC4uLm5hbWVzcGFjZVNlbGVjdG9yLm1hdGNoTGFiZWxzLFxuICAgICAgICAgIFtuYW1lc3BhY2UuTmFtZXNwYWNlLk5BTUVfTEFCRUxdOiBuLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH1cblxuICBwcml2YXRlIGNvbmZpZ3VyZURlZmF1bHRCZWhhdmlvcihkaXJlY3Rpb246ICdJbmdyZXNzJyB8ICdFZ3Jlc3MnLCBfZGVmYXVsdD86IE5ldHdvcmtQb2xpY3lUcmFmZmljRGVmYXVsdCkge1xuXG4gICAgaWYgKCFfZGVmYXVsdCkgeyByZXR1cm47fVxuXG4gICAgaWYgKF9kZWZhdWx0ID09PSBOZXR3b3JrUG9saWN5VHJhZmZpY0RlZmF1bHQuREVOWSkge1xuICAgICAgLy8gaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc2VydmljZXMtbmV0d29ya2luZy9uZXR3b3JrLXBvbGljaWVzLyNkZWZhdWx0LWRlbnktYWxsLWVncmVzcy10cmFmZmljXG4gICAgICB0aGlzLl9wb2xpY3lUeXBlcy5hZGQoZGlyZWN0aW9uKTtcbiAgICB9XG5cbiAgICBpZiAoX2RlZmF1bHQgPT09IE5ldHdvcmtQb2xpY3lUcmFmZmljRGVmYXVsdC5BTExPVykge1xuICAgICAgLy8gaHR0cHM6Ly9rdWJlcm5ldGVzLmlvL2RvY3MvY29uY2VwdHMvc2VydmljZXMtbmV0d29ya2luZy9uZXR3b3JrLXBvbGljaWVzLyNhbGxvdy1hbGwtZWdyZXNzLXRyYWZmaWNcbiAgICAgIHRoaXMuX3BvbGljeVR5cGVzLmFkZChkaXJlY3Rpb24pO1xuICAgICAgaWYgKGRpcmVjdGlvbiA9PT0gJ0VncmVzcycpIHtcbiAgICAgICAgdGhpcy5fZWdyZXNzUnVsZXMucHVzaCh7fSk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aGlzLl9pbmdyZXNzUnVsZXMucHVzaCh7fSk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHVibGljIF90b0t1YmUoKTogazhzLk5ldHdvcmtQb2xpY3lTcGVjIHtcbiAgICByZXR1cm4ge1xuICAgICAgcG9kU2VsZWN0b3I6IHRoaXMuX3BvZFNlbGVjdG9yQ29uZmlnLmxhYmVsU2VsZWN0b3IuX3RvS3ViZSgpLFxuICAgICAgZWdyZXNzOiB1bmRlZmluZWRJZkVtcHR5KHRoaXMuX2VncmVzc1J1bGVzKSxcbiAgICAgIGluZ3Jlc3M6IHVuZGVmaW5lZElmRW1wdHkodGhpcy5faW5ncmVzc1J1bGVzKSxcbiAgICAgIHBvbGljeVR5cGVzOiB1bmRlZmluZWRJZkVtcHR5KEFycmF5LmZyb20odGhpcy5fcG9saWN5VHlwZXMpKSxcbiAgICB9O1xuICB9XG5cbn1cblxuZXhwb3J0IGZ1bmN0aW9uIHZhbGlkYXRlUGVlckNvbmZpZyhwZWVyQ29uZmlnOiBOZXR3b3JrUG9saWN5UGVlckNvbmZpZykge1xuICBpZiAoIXBlZXJDb25maWcuaXBCbG9jayAmJiAhcGVlckNvbmZpZy5wb2RTZWxlY3Rvcikge1xuICAgIHRocm93IG5ldyBFcnJvcignSW5hdmxpZCBwZWVyOiBlaXRoZXIgXFwnaXBCbG9ja1xcJyBvciBcXCdwb2RTZWxlY3RvclxcJyBtdXN0IGJlIGRlZmluZWQnKTtcbiAgfVxuICBpZiAocGVlckNvbmZpZy5pcEJsb2NrICYmIHBlZXJDb25maWcucG9kU2VsZWN0b3IpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ0luYXZsaWQgcGVlcjogb25seSBvbmUgb2YgXFwnaXBCbG9ja1xcJyBhbmQgXFwncG9kU2VsZWN0b3JcXCcgbXVzdCBiZSBkZWZpbmVkJyk7XG4gIH1cbn1cbiJdfQ==