"use strict";
var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PodConnections = exports.PodConnectionsIsolation = exports.PodScheduling = exports.Topology = exports.Node = exports.NamedNode = exports.TaintedNode = exports.LabeledNode = exports.Pods = exports.NodeTaintQuery = exports.TaintEffect = exports.LabelExpression = exports.NodeLabelQuery = exports.DnsPolicy = exports.FsGroupChangePolicy = exports.RestartPolicy = exports.PodSecurityContext = exports.PodDns = exports.Pod = exports.LabelSelector = exports.AbstractPod = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const networkpolicy = require("./network-policy");
const utils_1 = require("./utils");
class AbstractPod extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._containers = [];
        this._initContainers = [];
        this._hostAliases = [];
        this._volumes = new Map();
        this.restartPolicy = props.restartPolicy;
        this.serviceAccount = props.serviceAccount;
        this.securityContext = new PodSecurityContext(props.securityContext);
        this.dns = new PodDns(props.dns);
        this.dockerRegistryAuth = props.dockerRegistryAuth;
        this.automountServiceAccountToken = props.automountServiceAccountToken ?? true;
        if (props.containers) {
            props.containers.forEach(c => this.addContainer(c));
        }
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        if (props.initContainers) {
            props.initContainers.forEach(c => this.addInitContainer(c));
        }
        if (props.hostAliases) {
            props.hostAliases.forEach(c => this.addHostAlias(c));
        }
    }
    get containers() {
        return [...this._containers];
    }
    get initContainers() {
        return [...this._initContainers];
    }
    get volumes() {
        return Array.from(this._volumes.values());
    }
    get hostAliases() {
        return [...this._hostAliases];
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        const podAddress = this.podMetadata.getLabel(Pod.ADDRESS_LABEL);
        if (!podAddress) {
            // shouldn't happen because we add this label automatically in both pods and workloads.
            throw new Error(`Unable to create a label selector since ${Pod.ADDRESS_LABEL} label is missing`);
        }
        return {
            labelSelector: LabelSelector.of({ labels: { [Pod.ADDRESS_LABEL]: podAddress } }),
            namespaces: this.metadata.namespace ? {
                names: [this.metadata.namespace],
            } : undefined,
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
    addContainer(cont) {
        const impl = new container.Container(cont);
        this._containers.push(impl);
        return impl;
    }
    addInitContainer(cont) {
        // https://kubernetes.io/docs/concepts/workloads/pods/init-containers/#differences-from-regular-containers
        if (cont.readiness) {
            throw new Error('Init containers must not have a readiness probe');
        }
        if (cont.liveness) {
            throw new Error('Init containers must not have a liveness probe');
        }
        if (cont.startup) {
            throw new Error('Init containers must not have a startup probe');
        }
        const impl = new container.Container({
            ...cont,
            name: cont.name ?? `init-${this._initContainers.length}`,
        });
        this._initContainers.push(impl);
        return impl;
    }
    addHostAlias(hostAlias) {
        this._hostAliases.push(hostAlias);
    }
    addVolume(vol) {
        const existingVolume = this._volumes.get(vol.name);
        if (existingVolume) {
            throw new Error(`Volume with name ${vol.name} already exists`);
        }
        this._volumes.set(vol.name, vol);
    }
    /**
     * @see ISubect.toSubjectConfiguration()
     */
    toSubjectConfiguration() {
        if (!this.serviceAccount && !this.automountServiceAccountToken) {
            throw new Error(`${this.name} cannot be converted to a role binding subject:`
                + ' You must either assign a service account to it, or use \'automountServiceAccountToken: true\'');
        }
        // 'default' is assumed to be the name of the default service account
        // in the cluster.
        const serviceAccountName = this.serviceAccount?.name ?? 'default';
        return {
            kind: 'ServiceAccount',
            name: serviceAccountName,
            apiGroup: '',
        };
    }
    /**
     * @internal
     */
    _toPodSpec() {
        if (this.containers.length === 0) {
            throw new Error('PodSpec must have at least 1 container');
        }
        const volumes = new Map();
        const containers = [];
        const initContainers = [];
        for (const cont of this.containers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            containers.push(cont._toKube());
        }
        for (const cont of this.initContainers) {
            // automatically add volume from the container mount
            // to this pod so thats its available to the container.
            for (const mount of cont.mounts) {
                addVolume(mount.volume);
            }
            initContainers.push(cont._toKube());
        }
        for (const vol of this.volumes) {
            addVolume(vol);
        }
        function addVolume(vol) {
            const existingVolume = volumes.get(vol.name);
            // its ok to call this function twice on the same volume, but its not ok to
            // call it twice on a different volume with the same name.
            if (existingVolume && existingVolume !== vol) {
                throw new Error(`Invalid mount configuration. At least two different volumes have the same name: ${vol.name}`);
            }
            volumes.set(vol.name, vol);
        }
        const dns = this.dns._toKube();
        return {
            restartPolicy: this.restartPolicy ? this._restartPolicyToKube(this.restartPolicy) : undefined,
            serviceAccountName: this.serviceAccount?.name,
            containers: containers,
            securityContext: utils_1.undefinedIfEmpty(this.securityContext._toKube()),
            initContainers: utils_1.undefinedIfEmpty(initContainers),
            hostAliases: utils_1.undefinedIfEmpty(this.hostAliases),
            volumes: utils_1.undefinedIfEmpty(Array.from(volumes.values()).map(v => v._toKube())),
            dnsPolicy: dns.policy,
            dnsConfig: utils_1.undefinedIfEmpty(dns.config),
            hostname: dns.hostname,
            subdomain: dns.subdomain,
            setHostnameAsFqdn: dns.hostnameAsFQDN,
            imagePullSecrets: this.dockerRegistryAuth ? [{ name: this.dockerRegistryAuth.name }] : undefined,
            automountServiceAccountToken: this.automountServiceAccountToken,
        };
    }
    _restartPolicyToKube(restartPolicy) {
        switch (restartPolicy) {
            case RestartPolicy.ALWAYS:
                return k8s.IoK8SApiCoreV1PodSpecRestartPolicy.ALWAYS;
            case RestartPolicy.NEVER:
                return k8s.IoK8SApiCoreV1PodSpecRestartPolicy.NEVER;
            case RestartPolicy.ON_FAILURE:
                return k8s.IoK8SApiCoreV1PodSpecRestartPolicy.ON_FAILURE;
            default:
                throw new Error(`Unsupported restart policy: ${restartPolicy}`);
        }
    }
}
exports.AbstractPod = AbstractPod;
_a = JSII_RTTI_SYMBOL_1;
AbstractPod[_a] = { fqn: "cdk8s-plus-23.AbstractPod", version: "2.0.0-rc.108" };
/**
 * Match a resource by labels.
 */
class LabelSelector {
    constructor(expressions, labels) {
        this.expressions = expressions;
        this.labels = labels;
    }
    static of(options = {}) {
        return new LabelSelector(options.expressions ?? [], options.labels ?? {});
    }
    isEmpty() {
        return this.expressions.length === 0 && Object.keys(this.labels).length === 0;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.isEmpty()) {
            return {};
        }
        return {
            matchExpressions: utils_1.undefinedIfEmpty(this.expressions.map(q => ({ key: q.key, operator: q.operator, values: q.values }))),
            matchLabels: utils_1.undefinedIfEmpty(this.labels),
        };
    }
}
exports.LabelSelector = LabelSelector;
_b = JSII_RTTI_SYMBOL_1;
LabelSelector[_b] = { fqn: "cdk8s-plus-23.LabelSelector", version: "2.0.0-rc.108" };
/**
 * Pod is a collection of containers that can run on a host. This resource is
 * created by clients and scheduled onto hosts.
 */
class Pod extends AbstractPod {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.resourceType = 'pods';
        this.apiObject = new k8s.KubePod(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.metadata.addLabel(Pod.ADDRESS_LABEL, cdk8s_1.Names.toLabelValue(this));
        this.scheduling = new PodScheduling(this);
        this.connections = new PodConnections(this);
    }
    get podMetadata() {
        return this.metadata;
    }
    /**
     * @internal
     */
    _toKube() {
        const scheduling = this.scheduling._toKube();
        return {
            ...this._toPodSpec(),
            affinity: scheduling.affinity,
            nodeName: scheduling.nodeName,
            tolerations: scheduling.tolerations,
        };
    }
}
exports.Pod = Pod;
_c = JSII_RTTI_SYMBOL_1;
Pod[_c] = { fqn: "cdk8s-plus-23.Pod", version: "2.0.0-rc.108" };
/**
 * This label is autoamtically added by cdk8s to any pod. It provides
 * a unique and stable identifier for the pod.
 */
Pod.ADDRESS_LABEL = 'cdk8s.io/metadata.addr';
/**
 * Holds dns settings of the pod.
 */
class PodDns {
    constructor(props = {}) {
        this.hostname = props.hostname;
        this.subdomain = props.subdomain;
        this.policy = props.policy ?? DnsPolicy.CLUSTER_FIRST;
        this.hostnameAsFQDN = props.hostnameAsFQDN ?? false;
        this._nameservers = props.nameservers ?? [];
        this._searches = props.searches ?? [];
        this._options = props.options ?? [];
    }
    /**
     * Nameservers defined for this pod.
     */
    get nameservers() {
        return [...this._nameservers];
    }
    /**
     * Search domains defined for this pod.
     */
    get searches() {
        return [...this._searches];
    }
    /**
     * Custom dns options defined for this pod.
     */
    get options() {
        return [...this._options];
    }
    /**
     * Add a nameserver.
     */
    addNameserver(...nameservers) {
        this._nameservers.push(...nameservers);
    }
    /**
     * Add a search domain.
     */
    addSearch(...searches) {
        this._searches.push(...searches);
    }
    /**
     * Add a custom option.
     */
    addOption(...options) {
        this._options.push(...options);
    }
    /**
     * @internal
     */
    _toKube() {
        if (this.policy === DnsPolicy.NONE && this.nameservers.length === 0) {
            throw new Error('When dns policy is set to NONE, at least one nameserver is required');
        }
        if (this.nameservers.length > 3) {
            throw new Error('There can be at most 3 nameservers specified');
        }
        if (this.searches.length > 6) {
            throw new Error('There can be at most 6 search domains specified');
        }
        return {
            hostname: this.hostname,
            subdomain: this.subdomain,
            hostnameAsFQDN: this.hostnameAsFQDN,
            policy: this._dnsPolicyToKube(this.policy),
            config: {
                nameservers: utils_1.undefinedIfEmpty(this.nameservers),
                searches: utils_1.undefinedIfEmpty(this.searches),
                options: utils_1.undefinedIfEmpty(this.options),
            },
        };
    }
    _dnsPolicyToKube(dnsPolicy) {
        switch (dnsPolicy) {
            case DnsPolicy.CLUSTER_FIRST:
                return k8s.IoK8SApiCoreV1PodSpecDnsPolicy.CLUSTER_FIRST;
            case DnsPolicy.CLUSTER_FIRST_WITH_HOST_NET:
                return k8s.IoK8SApiCoreV1PodSpecDnsPolicy.CLUSTER_FIRST_WITH_HOST_NET;
            case DnsPolicy.DEFAULT:
                return k8s.IoK8SApiCoreV1PodSpecDnsPolicy.DEFAULT;
            case DnsPolicy.NONE:
                return k8s.IoK8SApiCoreV1PodSpecDnsPolicy.NONE;
            default:
                throw new Error(`Unsupported dns policy: ${dnsPolicy}`);
        }
    }
}
exports.PodDns = PodDns;
_d = JSII_RTTI_SYMBOL_1;
PodDns[_d] = { fqn: "cdk8s-plus-23.PodDns", version: "2.0.0-rc.108" };
/**
 * Holds pod-level security attributes and common container settings.
 */
class PodSecurityContext {
    constructor(props = {}) {
        this._sysctls = [];
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.fsGroupChangePolicy = props.fsGroupChangePolicy ?? FsGroupChangePolicy.ALWAYS;
        this.user = props.user;
        this.group = props.group;
        this.fsGroup = props.fsGroup;
        for (const sysctl of props.sysctls ?? []) {
            this._sysctls.push(sysctl);
        }
    }
    get sysctls() {
        return [...this._sysctls];
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            fsGroup: this.fsGroup,
            runAsNonRoot: this.ensureNonRoot,
            fsGroupChangePolicy: this.fsGroupChangePolicy,
            sysctls: utils_1.undefinedIfEmpty(this._sysctls),
        };
    }
}
exports.PodSecurityContext = PodSecurityContext;
_e = JSII_RTTI_SYMBOL_1;
PodSecurityContext[_e] = { fqn: "cdk8s-plus-23.PodSecurityContext", version: "2.0.0-rc.108" };
/**
 * Restart policy for all containers within the pod.
 */
var RestartPolicy;
(function (RestartPolicy) {
    /**
     * Always restart the pod after it exits.
     */
    RestartPolicy["ALWAYS"] = "Always";
    /**
     * Only restart if the pod exits with a non-zero exit code.
     */
    RestartPolicy["ON_FAILURE"] = "OnFailure";
    /**
     * Never restart the pod.
     */
    RestartPolicy["NEVER"] = "Never";
})(RestartPolicy = exports.RestartPolicy || (exports.RestartPolicy = {}));
var FsGroupChangePolicy;
(function (FsGroupChangePolicy) {
    /**
     * Only change permissions and ownership if permission and ownership of root directory does
     * not match with expected permissions of the volume.
     * This could help shorten the time it takes to change ownership and permission of a volume
     */
    FsGroupChangePolicy["ON_ROOT_MISMATCH"] = "OnRootMismatch";
    /**
     * Always change permission and ownership of the volume when volume is mounted.
     */
    FsGroupChangePolicy["ALWAYS"] = "Always";
})(FsGroupChangePolicy = exports.FsGroupChangePolicy || (exports.FsGroupChangePolicy = {}));
/**
 * Pod DNS policies.
 */
var DnsPolicy;
(function (DnsPolicy) {
    /**
     * Any DNS query that does not match the configured cluster domain suffix,
     * such as "www.kubernetes.io", is forwarded to the
     * upstream nameserver inherited from the node.
     * Cluster administrators may have extra stub-domain and upstream DNS servers configured.
     */
    DnsPolicy["CLUSTER_FIRST"] = "ClusterFirst";
    /**
     * For Pods running with hostNetwork, you should
     * explicitly set its DNS policy "ClusterFirstWithHostNet".
     */
    DnsPolicy["CLUSTER_FIRST_WITH_HOST_NET"] = "ClusterFirstWithHostNet";
    /**
     * The Pod inherits the name resolution configuration
     * from the node that the pods run on.
     */
    DnsPolicy["DEFAULT"] = "Default";
    /**
     * It allows a Pod to ignore DNS settings from the Kubernetes environment.
     * All DNS settings are supposed to be provided using the dnsConfig
     * field in the Pod Spec.
     */
    DnsPolicy["NONE"] = "None";
})(DnsPolicy = exports.DnsPolicy || (exports.DnsPolicy = {}));
/**
 * Represents a query that can be performed against nodes with labels.
 */
class NodeLabelQuery {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to equal `value`.
     */
    static is(key, value) {
        return NodeLabelQuery.in(key, [value]);
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.IN, values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.NOT_IN, values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.EXISTS, undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.DOES_NOT_EXIST, undefined);
    }
    /**
     * Requires value of label `key` to greater than all elements in `values`.
     */
    static gt(key, values) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.GT, values);
    }
    /**
     * Requires value of label `key` to less than all elements in `values`.
     */
    static lt(key, values) {
        return new NodeLabelQuery(key, k8s.IoK8SApiCoreV1NodeSelectorRequirementOperator.LT, values);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            key: this.key,
            operator: this.operator,
            values: this.values,
        };
    }
}
exports.NodeLabelQuery = NodeLabelQuery;
_f = JSII_RTTI_SYMBOL_1;
NodeLabelQuery[_f] = { fqn: "cdk8s-plus-23.NodeLabelQuery", version: "2.0.0-rc.108" };
/**
 * Represents a query that can be performed against resources with labels.
 */
class LabelExpression {
    constructor(key, operator, values) {
        this.key = key;
        this.operator = operator;
        this.values = values;
    }
    /**
     * Requires value of label `key` to be one of `values`.
     */
    static in(key, values) {
        return new LabelExpression(key, 'In', values);
    }
    /**
     * Requires value of label `key` to be none of `values`.
     */
    static notIn(key, values) {
        return new LabelExpression(key, 'NotIn', values);
    }
    /**
     * Requires label `key` to exist.
     */
    static exists(key) {
        return new LabelExpression(key, 'Exists', undefined);
    }
    /**
     * Requires label `key` to not exist.
     */
    static doesNotExist(key) {
        return new LabelExpression(key, 'DoesNotExist', undefined);
    }
}
exports.LabelExpression = LabelExpression;
_g = JSII_RTTI_SYMBOL_1;
LabelExpression[_g] = { fqn: "cdk8s-plus-23.LabelExpression", version: "2.0.0-rc.108" };
/**
 * Taint effects.
 */
var TaintEffect;
(function (TaintEffect) {
    /**
     * This means that no pod will be able to schedule
     * onto the node unless it has a matching toleration.
     */
    TaintEffect["NO_SCHEDULE"] = "NoSchedule";
    /**
     * This is a "preference" or "soft" version of `NO_SCHEDULE` -- the system
     * will try to avoid placing a pod that does not tolerate the taint on the node,
     * but it is not required
     */
    TaintEffect["PREFER_NO_SCHEDULE"] = "PreferNoSchedule";
    /**
     * This affects pods that are already running on the node as follows:
     *
     * - Pods that do not tolerate the taint are evicted immediately.
     * - Pods that tolerate the taint without specifying `duration` remain bound forever.
     * - Pods that tolerate the taint with a specified `duration` remain bound for
     *   the specified amount of time.
     */
    TaintEffect["NO_EXECUTE"] = "NoExecute";
})(TaintEffect = exports.TaintEffect || (exports.TaintEffect = {}));
/**
 * Taint queries that can be perfomed against nodes.
 */
class NodeTaintQuery {
    constructor(operator, key, value, effect, evictAfter) {
        this.operator = operator;
        this.key = key;
        this.value = value;
        this.effect = effect;
        this.evictAfter = evictAfter;
        if (evictAfter && effect !== TaintEffect.NO_EXECUTE) {
            throw new Error('Only \'NO_EXECUTE\' effects can specify \'evictAfter\'');
        }
    }
    /**
     * Matches a taint with a specific key and value.
     */
    static is(key, value, options = {}) {
        return new NodeTaintQuery(k8s.IoK8SApiCoreV1TolerationOperator.EQUAL, key, value, options.effect, options.evictAfter);
    }
    /**
     * Matches a tain with any value of a specific key.
     */
    static exists(key, options = {}) {
        return new NodeTaintQuery(k8s.IoK8SApiCoreV1TolerationOperator.EXISTS, key, undefined, options.effect, options.evictAfter);
    }
    /**
     * Matches any taint.
     */
    static any() {
        return new NodeTaintQuery(k8s.IoK8SApiCoreV1TolerationOperator.EXISTS);
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            effect: this.effect ? this._taintEffectToKube(this.effect) : undefined,
            key: this.key,
            operator: this.operator,
            tolerationSeconds: this.evictAfter?.toSeconds(),
            value: this.value,
        };
    }
    _taintEffectToKube(taintEffect) {
        switch (taintEffect) {
            case TaintEffect.NO_EXECUTE:
                return k8s.IoK8SApiCoreV1TolerationEffect.NO_EXECUTE;
            case TaintEffect.NO_SCHEDULE:
                return k8s.IoK8SApiCoreV1TolerationEffect.NO_SCHEDULE;
            case TaintEffect.PREFER_NO_SCHEDULE:
                return k8s.IoK8SApiCoreV1TolerationEffect.PREFER_NO_SCHEDULE;
            default:
                throw new Error(`Unsupported taint effect: ${taintEffect}`);
        }
    }
}
exports.NodeTaintQuery = NodeTaintQuery;
_h = JSII_RTTI_SYMBOL_1;
NodeTaintQuery[_h] = { fqn: "cdk8s-plus-23.NodeTaintQuery", version: "2.0.0-rc.108" };
/**
 * Represents a group of pods.
 */
class Pods extends constructs_1.Construct {
    constructor(scope, id, expressions, labels, namespaces) {
        super(scope, id);
        this.expressions = expressions;
        this.labels = labels;
        this.namespaces = namespaces;
    }
    /**
     * Select pods in the cluster with various selectors.
     */
    static select(scope, id, options) {
        return new Pods(scope, id, options.expressions, options.labels, options.namespaces);
    }
    /**
     * Select all pods.
     */
    static all(scope, id, options = {}) {
        return Pods.select(scope, id, { namespaces: options.namespaces });
    }
    /**
     * @see IPodSelector.toPodSelectorConfig()
     */
    toPodSelectorConfig() {
        return {
            labelSelector: LabelSelector.of({ expressions: this.expressions, labels: this.labels }),
            namespaces: this.namespaces?.toNamespaceSelectorConfig(),
        };
    }
    /**
     * @see INetworkPolicyPeer.toNetworkPolicyPeerConfig()
     */
    toNetworkPolicyPeerConfig() {
        return { podSelector: this.toPodSelectorConfig() };
    }
    /**
     * @see INetworkPolicyPeer.toPodSelector()
     */
    toPodSelector() {
        return this;
    }
}
exports.Pods = Pods;
_j = JSII_RTTI_SYMBOL_1;
Pods[_j] = { fqn: "cdk8s-plus-23.Pods", version: "2.0.0-rc.108" };
/**
 * A node that is matched by label selectors.
 */
class LabeledNode {
    constructor(labelSelector) {
        this.labelSelector = labelSelector;
    }
    ;
}
exports.LabeledNode = LabeledNode;
_k = JSII_RTTI_SYMBOL_1;
LabeledNode[_k] = { fqn: "cdk8s-plus-23.LabeledNode", version: "2.0.0-rc.108" };
/**
 * A node that is matched by taint selectors.
 */
class TaintedNode {
    constructor(taintSelector) {
        this.taintSelector = taintSelector;
    }
    ;
}
exports.TaintedNode = TaintedNode;
_l = JSII_RTTI_SYMBOL_1;
TaintedNode[_l] = { fqn: "cdk8s-plus-23.TaintedNode", version: "2.0.0-rc.108" };
/**
 * A node that is matched by its name.
 */
class NamedNode {
    constructor(name) {
        this.name = name;
    }
    ;
}
exports.NamedNode = NamedNode;
_m = JSII_RTTI_SYMBOL_1;
NamedNode[_m] = { fqn: "cdk8s-plus-23.NamedNode", version: "2.0.0-rc.108" };
/**
 * Represents a node in the cluster.
 */
class Node {
    /**
     * Match a node by its labels.
     */
    static labeled(...labelSelector) {
        return new LabeledNode(labelSelector);
    }
    /**
     * Match a node by its name.
     */
    static named(nodeName) {
        return new NamedNode(nodeName);
    }
    /**
     * Match a node by its taints.
     */
    static tainted(...taintSelector) {
        return new TaintedNode(taintSelector);
    }
}
exports.Node = Node;
_o = JSII_RTTI_SYMBOL_1;
Node[_o] = { fqn: "cdk8s-plus-23.Node", version: "2.0.0-rc.108" };
/**
 * Available topology domains.
 */
class Topology {
    constructor(key) {
        this.key = key;
    }
    /**
     * Custom key for the node label that the system uses to denote the topology domain.
     */
    static custom(key) {
        return new Topology(key);
    }
    ;
}
exports.Topology = Topology;
_p = JSII_RTTI_SYMBOL_1;
Topology[_p] = { fqn: "cdk8s-plus-23.Topology", version: "2.0.0-rc.108" };
/**
 * A hostname represents a single node in the cluster.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#kubernetesiohostname
 */
Topology.HOSTNAME = new Topology('kubernetes.io/hostname');
/**
 * A zone represents a logical failure domain. It is common for Kubernetes clusters to
 * span multiple zones for increased availability. While the exact definition of a zone is
 * left to infrastructure implementations, common properties of a zone include very low
 * network latency within a zone, no-cost network traffic within a zone, and failure
 * independence from other zones. For example, nodes within a zone might share a network
 * switch, but nodes in different zones should not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesiozone
 */
Topology.ZONE = new Topology('topology.kubernetes.io/zone');
/**
 * A region represents a larger domain, made up of one or more zones. It is uncommon
 * for Kubernetes clusters to span multiple regions. While the exact definition of a
 * zone or region is left to infrastructure implementations, common properties of a region
 * include higher network latency between them than within them, non-zero cost for network
 * traffic between them, and failure independence from other zones or regions.
 *
 * For example, nodes within a region might share power infrastructure (e.g. a UPS or generator), but
 * nodes in different regions typically would not.
 *
 * @see https://kubernetes.io/docs/reference/labels-annotations-taints/#topologykubernetesioregion
 */
Topology.REGION = new Topology('topology.kubernetes.io/region');
/**
 * Controls the pod scheduling strategy.
 */
class PodScheduling {
    constructor(instance) {
        this.instance = instance;
        this._nodeAffinityPreferred = [];
        this._nodeAffinityRequired = [];
        this._podAffinityPreferred = [];
        this._podAffinityRequired = [];
        this._podAntiAffinityPreferred = [];
        this._podAntiAffinityRequired = [];
        this._tolerations = [];
    }
    /**
     * Assign this pod a specific node by name.
     *
     * The scheduler ignores the Pod, and the kubelet on the named node
     * tries to place the Pod on that node. Overrules any affinity rules of the pod.
     *
     * Some limitations of static assignment are:
     *
     * - If the named node does not exist, the Pod will not run, and in some
     *   cases may be automatically deleted.
     * - If the named node does not have the resources to accommodate the Pod,
     *   the Pod will fail and its reason will indicate why, for example OutOfmemory or OutOfcpu.
     * - Node names in cloud environments are not always predictable or stable.
     *
     * Will throw is the pod is already assigned to named node.
     *
     * Under the hood, this method utilizes the `nodeName` property.
     */
    assign(node) {
        if (this._nodeName) {
            // disallow overriding an static node assignment
            throw new Error(`Cannot assign ${this.instance.podMetadata.name} to node ${node.name}. It is already assigned to node ${this._nodeName}`);
        }
        else {
            this._nodeName = node.name;
        }
    }
    /**
     * Allow this pod to tolerate taints matching these tolerations.
     *
     * You can put multiple taints on the same node and multiple tolerations on the same pod.
     * The way Kubernetes processes multiple taints and tolerations is like a filter: start with
     * all of a node's taints, then ignore the ones for which the pod has a matching toleration;
     * the remaining un-ignored taints have the indicated effects on the pod. In particular:
     *
     * - if there is at least one un-ignored taint with effect NoSchedule then Kubernetes will
     *   not schedule the pod onto that node
     * - if there is no un-ignored taint with effect NoSchedule but there is at least one un-ignored
     *   taint with effect PreferNoSchedule then Kubernetes will try to not schedule the pod onto the node
     * - if there is at least one un-ignored taint with effect NoExecute then the pod will be evicted from
     *   the node (if it is already running on the node), and will not be scheduled onto the node (if it is
     *   not yet running on the node).
     *
     * Under the hood, this method utilizes the `tolerations` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/taint-and-toleration/
     */
    tolerate(node) {
        for (const query of node.taintSelector) {
            this._tolerations.push(query._toKube());
        }
    }
    /**
     * Attract this pod to a node matched by selectors.
     * You can select a node by using `Node.labeled()`.
     *
     * Attracting to multiple nodes (i.e invoking this method multiple times) acts as
     * an OR condition, meaning the pod will be assigned to either one of the nodes.
     *
     * Under the hood, this method utilizes the `nodeAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity
     */
    attract(node, options = {}) {
        const term = this.createNodeAffinityTerm(node);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._nodeAffinityPreferred.push({ weight: options.weight, preference: term });
        }
        else {
            this._nodeAffinityRequired.push(term);
        }
    }
    /**
     * Co-locate this pod with a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Co-locating with multiple selections ((i.e invoking this method multiple times)) acts as
     * an AND condition. meaning the pod will be assigned to a node that satisfies all
     * selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    colocate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAffinityRequired.push(term);
        }
    }
    /**
     * Seperate this pod from a scheduling selection.
     *
     * A selection can be one of:
     *
     * - An instance of a `Pod`.
     * - An instance of a `Workload` (e.g `Deployment`, `StatefulSet`).
     * - An un-managed pod that can be selected via `Pods.select()`.
     *
     * Seperating from multiple selections acts as an AND condition. meaning the pod
     * will not be assigned to a node that satisfies all selections (i.e runs at least one pod that satisifies each selection).
     *
     * Under the hood, this method utilizes the `podAntiAffinity` property.
     *
     * @see https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#inter-pod-affinity-and-anti-affinity
     */
    separate(selector, options = {}) {
        const topology = options.topology ?? Topology.HOSTNAME;
        const term = this.createPodAffinityTerm(topology, selector);
        if (options.weight) {
            this.validateWeight(options.weight);
            this._podAntiAffinityPreferred.push({ weight: options.weight, podAffinityTerm: term });
        }
        else {
            this._podAntiAffinityRequired.push(term);
        }
    }
    createPodAffinityTerm(topology, selector) {
        const config = selector.toPodSelectorConfig();
        return {
            topologyKey: topology.key,
            labelSelector: config.labelSelector._toKube(),
            namespaceSelector: config.namespaces?.labelSelector?._toKube(),
            namespaces: config.namespaces?.names,
        };
    }
    createNodeAffinityTerm(node) {
        return { matchExpressions: node.labelSelector.map(s => s._toKube()) };
    }
    validateWeight(weight) {
        if (weight < 1 || weight > 100) {
            // https://kubernetes.io/docs/concepts/scheduling-eviction/assign-pod-node/#node-affinity-weight
            throw new Error(`Invalid affinity weight: ${weight}. Must be in range 1-100`);
        }
    }
    /**
     * @internal
     */
    _toKube() {
        const atLeastOne = (...arrays) => {
            return arrays.flat().length > 0;
        };
        const hasNodeAffinity = atLeastOne(this._nodeAffinityPreferred, this._nodeAffinityRequired);
        const hasPodAffinity = atLeastOne(this._podAffinityPreferred, this._podAffinityRequired);
        const hasPodAntiAffinty = atLeastOne(this._podAntiAffinityPreferred, this._podAntiAffinityRequired);
        const hasAffinity = hasNodeAffinity || hasPodAffinity || hasPodAntiAffinty;
        return {
            affinity: hasAffinity ? {
                nodeAffinity: hasNodeAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._nodeAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: this._nodeAffinityRequired.length > 0 ? {
                        nodeSelectorTerms: this._nodeAffinityRequired,
                    } : undefined,
                } : undefined,
                podAffinity: hasPodAffinity ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAffinityRequired),
                } : undefined,
                podAntiAffinity: hasPodAntiAffinty ? {
                    preferredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityPreferred),
                    requiredDuringSchedulingIgnoredDuringExecution: utils_1.undefinedIfEmpty(this._podAntiAffinityRequired),
                } : undefined,
            } : undefined,
            nodeName: this._nodeName,
            tolerations: utils_1.undefinedIfEmpty(this._tolerations),
        };
    }
}
exports.PodScheduling = PodScheduling;
_q = JSII_RTTI_SYMBOL_1;
PodScheduling[_q] = { fqn: "cdk8s-plus-23.PodScheduling", version: "2.0.0-rc.108" };
/**
 * Isolation determines which policies are created
 * when allowing connections from a a pod / workload to peers.
 */
var PodConnectionsIsolation;
(function (PodConnectionsIsolation) {
    /**
     * Only creates network policies that select the pod.
     */
    PodConnectionsIsolation["POD"] = "POD";
    /**
     * Only creates network policies that select the peer.
     */
    PodConnectionsIsolation["PEER"] = "PEER";
})(PodConnectionsIsolation = exports.PodConnectionsIsolation || (exports.PodConnectionsIsolation = {}));
/**
 * Controls network isolation rules for inter-pod communication.
 */
class PodConnections {
    constructor(instance) {
        this.instance = instance;
    }
    /**
     * Allow network traffic from this pod to the peer.
     *
     * By default, this will create an egress network policy for this pod, and an ingress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * web.connections.allowTo(redis, { isolation: Isolation.POD })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * web.connections.allowTo(redis, { isolation: Isolation.PEER })
     *
     */
    allowTo(peer, options = {}) {
        return this.allow('Egress', peer, { ports: this.extractPorts(peer), ...options });
    }
    /**
     * Allow network traffic from the peer to this pod.
     *
     * By default, this will create an ingress network policy for this pod, and an egress
     * network policy for the peer. This is required if both sides are already isolated.
     * Use `options.isolation` to control this behavior.
     *
     * @example
     *
     * // create only an egress policy that selects the 'web' pod to allow outgoing traffic
     * // to the 'redis' pod. this requires the 'redis' pod to not be isolated for ingress.
     * redis.connections.allowFrom(web, { isolation: Isolation.PEER })
     *
     * // create only an ingress policy that selects the 'redis' peer to allow incoming traffic
     * // from the 'web' pod. this requires the 'web' pod to not be isolated for egress.
     * redis.connections.allowFrom(web, { isolation: Isolation.POD })
     *
     */
    allowFrom(peer, options = {}) {
        return this.allow('Ingress', peer, { ports: this.extractPorts(this.instance), ...options });
    }
    allow(direction, peer, options = {}) {
        const config = peer.toNetworkPolicyPeerConfig();
        networkpolicy.validatePeerConfig(config);
        const peerAddress = utils_1.address(peer);
        if (!options.isolation || options.isolation === PodConnectionsIsolation.POD) {
            const src = new networkpolicy.NetworkPolicy(this.instance, `Allow${direction}${peerAddress}`, {
                selector: this.instance,
                // the policy must be defined in the namespace of the pod
                // so it can select it.
                metadata: { namespace: this.instance.metadata.namespace },
            });
            switch (direction) {
                case 'Egress':
                    src.addEgressRule(peer, options.ports);
                    break;
                case 'Ingress':
                    src.addIngressRule(peer, options.ports);
            }
        }
        if (!options.isolation || options.isolation === PodConnectionsIsolation.PEER) {
            if (config.ipBlock) {
                // for an ip block we don't need to create the opposite policies
                return;
            }
            const podSelector = peer.toPodSelector();
            if (!podSelector) {
                throw new Error(`Unable to create policies for peer '${peer.node.addr}' since its not a pod selector`);
            }
            const oppositeDirection = direction === 'Egress' ? 'Ingress' : 'Egress';
            const podSelectorConfig = podSelector.toPodSelectorConfig();
            let namespaces;
            if (!podSelectorConfig.namespaces) {
                // if the peer doesn't specify namespaces, we assume the same namespace.
                namespaces = [this.instance.metadata.namespace];
            }
            else {
                // a peer cannot specify namespaces by labels because
                // we won't be able to extract the names of those namespaces.
                if (podSelectorConfig.namespaces.labelSelector && !podSelectorConfig.namespaces.labelSelector.isEmpty()) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespaces only by name`);
                }
                // a peer must specify namespaces by name.
                if (!podSelectorConfig.namespaces.names) {
                    throw new Error(`Unable to create an ${oppositeDirection} policy for peer '${peer.node.path}' (pod=${this.instance.name}). Peer must specify namespace names`);
                }
                namespaces = podSelectorConfig.namespaces.names;
            }
            for (const name of namespaces) {
                switch (direction) {
                    case 'Egress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowIngress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            ingress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    case 'Ingress':
                        new networkpolicy.NetworkPolicy(this.instance, `AllowEgress${name}${peerAddress}`, {
                            selector: podSelector,
                            metadata: { namespace: name },
                            egress: { rules: [{ peer: this.instance, ports: options.ports }] },
                        });
                        break;
                    default:
                        throw new Error(`Unsupported direction: ${direction}`);
                }
            }
        }
    }
    extractPorts(selector) {
        return container.extractContainerPorts(selector).map(n => networkpolicy.NetworkPolicyPort.tcp(n.number));
    }
}
exports.PodConnections = PodConnections;
_r = JSII_RTTI_SYMBOL_1;
PodConnections[_r] = { fqn: "cdk8s-plus-23.PodConnections", version: "2.0.0-rc.108" };
//# sourceMappingURL=data:application/json;base64,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