"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Service = exports.ServiceType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const base = require("./base");
const container = require("./container");
const k8s = require("./imports/k8s");
const ingress = require("./ingress");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'services';
        this.apiObject = new k8s.KubeService(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.externalName = props.externalName;
        if (props.externalName !== undefined) {
            this.type = ServiceType.EXTERNAL_NAME;
        }
        else {
            this.type = props.type ?? ServiceType.CLUSTER_IP;
        }
        this._externalIPs = props.externalIPs ?? [];
        this._ports = [];
        this._selector = {};
        this._loadBalancerSourceRanges = props.loadBalancerSourceRanges;
        if (props.selector) {
            this.select(props.selector);
        }
        for (const port of props.ports ?? []) {
            this.bind(port.port, port);
        }
    }
    /**
     * Expose a service via an ingress using the specified path.
     *
     * @param path The path to expose the service under.
     * @param options Additional options.
     *
     * @returns The `Ingress` resource that was used.
     */
    exposeViaIngress(path, options = {}) {
        const ingr = options.ingress ?? new ingress.Ingress(this, 'Ingress');
        ingr.addRule(path, ingress.IngressBackend.fromService(this), options.pathType);
        return ingr;
    }
    /**
     * Ports for this service.
     *
     * Use `bind()` to bind additional service ports.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    bind(port, options = {}) {
        this._ports.push({ ...options, port });
    }
    /**
     * Require this service to select pods matching the selector.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    select(selector) {
        const labels = selector.toPodSelectorConfig().labelSelector._toKube().matchLabels ?? {};
        for (const [key, value] of Object.entries(labels)) {
            this._selector[key] = value;
        }
    }
    /**
     * Require this service to select pods with this label.
     *
     * Note that invoking this method multiple times acts as an AND operator
     * on the resulting labels.
     */
    selectLabel(key, value) {
        this._selector[key] = value;
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0 && this.type !== ServiceType.EXTERNAL_NAME) {
            throw new Error('A service must be configured with a port');
        }
        if (this.type === ServiceType.EXTERNAL_NAME && this.externalName === undefined) {
            throw new Error('A service with type EXTERNAL_NAME requires an externalName prop');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                name: port.name,
                port: port.port,
                targetPort: port.targetPort ? k8s.IntOrString.fromNumber(port.targetPort) : undefined,
                nodePort: port.nodePort,
                protocol: port.protocol ? this._portProtocolToKube(port.protocol) : undefined,
            });
        }
        const serviceType = this._serviceTypeToKube(this.type);
        return this.type !== ServiceType.EXTERNAL_NAME ? {
            clusterIp: this.clusterIP,
            externalIPs: this._externalIPs,
            externalName: this.externalName,
            type: serviceType,
            selector: this._selector,
            ports: ports,
            loadBalancerSourceRanges: this._loadBalancerSourceRanges,
        } : {
            type: serviceType,
            externalName: this.externalName,
        };
    }
    _portProtocolToKube(protocol) {
        switch (protocol) {
            case container.Protocol.SCTP:
                return k8s.IoK8SApiCoreV1ServicePortProtocol.SCTP;
            case container.Protocol.TCP:
                return k8s.IoK8SApiCoreV1ServicePortProtocol.TCP;
            case container.Protocol.UDP:
                return k8s.IoK8SApiCoreV1ServicePortProtocol.UDP;
            default:
                throw new Error(`Unsupported port protocol: ${protocol}`);
        }
    }
    _serviceTypeToKube(serviceType) {
        switch (serviceType) {
            case ServiceType.CLUSTER_IP:
                return k8s.IoK8SApiCoreV1ServiceSpecType.CLUSTER_IP;
            case ServiceType.EXTERNAL_NAME:
                return k8s.IoK8SApiCoreV1ServiceSpecType.EXTERNAL_NAME;
            case ServiceType.LOAD_BALANCER:
                return k8s.IoK8SApiCoreV1ServiceSpecType.LOAD_BALANCER;
            case ServiceType.NODE_PORT:
                return k8s.IoK8SApiCoreV1ServiceSpecType.NODE_PORT;
            default:
                throw new Error(`Unsupported service type: ${serviceType}`);
        }
    }
}
exports.Service = Service;
_a = JSII_RTTI_SYMBOL_1;
Service[_a] = { fqn: "cdk8s-plus-23.Service", version: "2.0.0-rc.108" };
//# sourceMappingURL=data:application/json;base64,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