"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const client_codedeploy_1 = require("@aws-sdk/client-codedeploy");
const logger_1 = require("./logger");
/**
 * The lambda function called from CloudFormation for this custom resource.
 *
 * @param event
 * @returns attribues of the deployment that was created
 */
async function handler(event) {
    const logger = new logger_1.Logger();
    logger.appendKeys({
        stackEvent: event.RequestType,
    });
    switch (event.RequestType) {
        case 'Create':
        case 'Update': {
            // create deployment
            const props = event.ResourceProperties;
            let autoRollbackConfiguration;
            if (props.autoRollbackConfigurationEnabled === 'true') {
                autoRollbackConfiguration = {
                    enabled: true,
                    events: props.autoRollbackConfigurationEvents.split(','),
                };
            }
            else if (props.autoRollbackConfigurationEnabled === 'false') {
                autoRollbackConfiguration = {
                    enabled: false,
                };
            }
            const resp = await codedeployClient().send(new client_codedeploy_1.CreateDeploymentCommand({
                applicationName: props.applicationName,
                deploymentConfigName: props.deploymentConfigName,
                deploymentGroupName: props.deploymentGroupName,
                autoRollbackConfiguration,
                description: props.description,
                revision: {
                    revisionType: 'AppSpecContent',
                    appSpecContent: {
                        content: props.revisionAppSpecContent,
                    },
                },
            }));
            if (!resp.deploymentId) {
                throw new Error('No deploymentId received from call to CreateDeployment');
            }
            logger.appendKeys({
                deploymentId: resp.deploymentId,
            });
            logger.info('Created new deployment');
            return {
                PhysicalResourceId: resp.deploymentId,
                Data: {
                    deploymentId: resp.deploymentId,
                },
            };
        }
        case 'Delete':
            logger.appendKeys({
                deploymentId: event.PhysicalResourceId,
            });
            // cancel deployment and rollback
            try {
                const resp = await codedeployClient().send(new client_codedeploy_1.StopDeploymentCommand({
                    deploymentId: event.PhysicalResourceId,
                    autoRollbackEnabled: true,
                }));
                logger.info(`Stopped deployment: ${resp.status} ${resp.statusMessage}`);
            }
            catch (e) {
                logger.warn('Ignoring error', { error: e });
            }
            return {
                PhysicalResourceId: event.PhysicalResourceId,
                Data: {
                    deploymentId: event.PhysicalResourceId,
                },
            };
        default:
            logger.error('Unknown stack event');
            throw new Error(`Unknown request type: ${event.RequestType}`);
    }
}
exports.handler = handler;
function codedeployClient() {
    return new client_codedeploy_1.CodeDeployClient({});
}
;
//# sourceMappingURL=data:application/json;base64,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