from pydantic import BaseModel, Field
from typing import List, Any


class CommonAll(BaseModel):
    field_context: str | List[Any] | None = Field(
        None,
        alias="@context",
        examples=[
            "https://www.w3.org/ns/activitystreams",
            ["https://www.w3.org/ns/activitystreams", {"Hashtag": "as:Hashtag"}],
        ],
    )
    id: str = Field(
        ...,
        examples=["https://actor.example/some_id"],
        description="""
    id of the activity or object, can be assumed to be globally unique. Some activities such as a Follow request will require an id to be valid. Servers may assume an id to be required. As assigning an id is 'trivial', one should assign one.
    """,
    )


class Common(CommonAll):
    to: List[str] = Field(
        ...,
        examples=[
            ["https://bob.example"],
            ["https://alice.example", "https://bob.example"],
        ],
        min_length=1,
        description="""
    Array of actors this activity or object is addressed to. It is sane to assume that an activity is addressed to at least one person.
    """,
    )
    cc: List[str] | None = Field(
        None,
        examples=[
            ["https://bob.example"],
            ["https://alice.example", "https://bob.example"],
        ],
        description="""
    Array of actors this activity or object is carbon copied to.
    """,
    )
    published: str | None = Field(
        None,
        description="""
    Moment of this activity or object being published
    """,
    )
    type: str = Field(
        ...,
        examples=["Follow", "Accept", "Create", "Undo", "Like", "Note"],
        description="""
    Type of the activity or activity. Side effects of this activity are determine by this type.
    """,
    )
