"""
A simple streaming helper that takes
minimal configuration as arguments and starts
a stream to stdout.
"""

import signal
import logging
import time
import json

import tweepy
from listener import JsonStreamListener
from checker import FileTermChecker
from stream import DynamicTwitterStream

logger = logging.getLogger('twitter_monitor')

class PrintingListener(JsonStreamListener):
    def __init__(self, api=None):
        super(PrintingListener, self).__init__(api)
        self.terminate = False

    def on_status(self, status):
        print json.dumps(status)
        return not self.terminate

    def set_terminate(self):
        self.terminate = True


def start(track_file,
          twitter_api_key,
          twitter_api_secret,
          twitter_access_token,
          twitter_access_token_secret,
          poll_interval=15):

    auth = tweepy.OAuthHandler(twitter_api_key, twitter_api_secret)
    auth.set_access_token(twitter_access_token, twitter_access_token_secret)

    listener = PrintingListener()
    checker = FileTermChecker(track_file)

    if not checker.update_tracking_terms():
        logger.error("No terms in track file %s", checker.filename)
        exit(1)

    logger.info("Monitoring track file %s", track_file)

    def stop(signum, frame):
        """
        Register stream's death and exit.
        """
        logger.info("Stopping because of signal")

        # Let the tweet listener know it should be quitting asap
        listener.set_terminate()

        raise SystemExit()

    # Installs signal handlers for handling SIGINT and SIGTERM
    # gracefully.
    signal.signal(signal.SIGINT, stop)
    signal.signal(signal.SIGTERM, stop)

    # Start and maintain the streaming connection...
    stream = DynamicTwitterStream(auth, listener, checker)
    while True:
        try:
            stream.start_polling(poll_interval)
        except Exception as e:
            logger.error("Exception while polling", exc_info=True)
            time.sleep(1)  # to avoid craziness
