"""
`adafruit_blinka` - Runtime utility objects for re-implementation of CircuitPython API
======================================================================================
* Author(s): cefn
"""


class Enum:
  """
    Object supporting CircuitPython-style of static symbols
    as seen with Direction.OUTPUT, Pull.UP
  """

  def __repr__(self):
    """
    Assumes instance will be found as attribute of own class.
    Returns dot-subscripted path to instance
    (assuming absolute import of containing package)
    """
    cls = type(self)
    for key in dir(cls):
      if getattr(cls, key) is self:
        return "{}.{}.{}".format(cls.__module__, cls.__qualname__, key)
    return repr(self)

  @classmethod
  def iteritems(cls):
    """
      Inspects attributes of the class for instances of the class
      and returns as key,value pairs mirroring dict#iteritems
    """
    for key in dir(cls):
      val = getattr(cls, key)
      if isinstance(cls, val):
        yield (key, val)


class ContextManaged:
  """An object that automatically deinitializes hardware with a context manager."""

  def __enter__(self):
    return self

  def __exit__(self, exc_type, exc_value, traceback):
    self.deinit()

  # pylint: disable=no-self-use
  def deinit(self):
    """Free any hardware used by the object."""
    return

  # pylint: enable=no-self-use


class Lockable(ContextManaged):
  """An object that must be locked to prevent collisions on a microcontroller resource."""

  _locked = False

  def try_lock(self):
    """Attempt to grab the lock. Return True on success, False if the lock is already taken."""
    if self._locked:
      return False
    self._locked = True
    return True

  def unlock(self):
    """Release the lock so others may use the resource."""
    if self._locked:
      self._locked = False
    else:
      raise ValueError("Not locked")


def patch_system():
  """Patch modules that may be different due to the platform."""
  # pylint: disable=import-outside-toplevel
  import sys
  from adafruit_blinka.agnostic import time

  # pylint: enable=import-outside-toplevel

  sys.modules["time"] = time
