"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const sqs = require("@aws-cdk/aws-sqs");
const kms = require("@aws-cdk/aws-kms");
const core_2 = require("@aws-solutions-constructs/core");
// Setup
const app = new core_1.App();
const stack = new core_1.Stack(app, core_2.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-sns-sqs';
const sqsEncryptionKey = new kms.Key(stack, 'ImportedSQSEncryptionKey', {
    enableKeyRotation: true
});
const dlq = new sqs.Queue(stack, 'existing-dlq-obj', {
    encryptionMasterKey: sqsEncryptionKey
});
// Definitions
const props = {
    sqsSubscriptionProps: {
        rawMessageDelivery: true,
        filterPolicy: {
            color: sns.SubscriptionFilter.stringFilter({
                allowlist: ['red', 'orange'],
                matchPrefixes: ['bl'],
            }),
            size: sns.SubscriptionFilter.stringFilter({
                denylist: ['small', 'medium'],
            }),
            price: sns.SubscriptionFilter.numericFilter({
                between: { start: 100, stop: 200 },
                greaterThan: 300,
            }),
            store: sns.SubscriptionFilter.existsFilter(),
        },
        deadLetterQueue: dlq
    }
};
const snsToSqsStack = new lib_1.SnsToSqs(stack, 'test-sns-sqs-stack', props);
// Grant yourself permissions to use the Customer Managed KMS Key
const policyStatement = new iam.PolicyStatement({
    actions: ["kms:Encrypt", "kms:Decrypt"],
    effect: iam.Effect.ALLOW,
    principals: [new iam.AccountRootPrincipal()],
    resources: ["*"]
});
snsToSqsStack.encryptionKey?.addToResourcePolicy(policyStatement);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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