"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-solutions-konstruk'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    const paths = mod.paths.map(stripNodeModules);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
    /**
     * @param s a path.
     * @returns ``s`` with any terminating ``/node_modules``
     *      (or ``\\node_modules``) stripped off.)
     */
    function stripNodeModules(s) {
        if (s.endsWith('/node_modules') || s.endsWith('\\node_modules')) {
            // /node_modules is 13 characters
            return s.substr(0, s.length - 13);
        }
        return s;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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