import os
import logging
from pathlib import Path
from PIL import Image, ImageChops
import numpy

log = logging.getLogger(f'vt.{os.path.basename(__file__)}')

class ImageTools:

    @staticmethod
    def getMaxImagePixels():
        return Image.MAX_IMAGE_PIXELS

    @staticmethod
    def setMaxImagePixels(pixels):
        if type(pixels) != int:
            raise Exception(f'MAX_IMAGE_PIXELS must be an integer')
        if pixels < 0:
            raise Exception(f'MAX_IMAGE_PIXELS must be greater than 0')
            
        Image.MAX_IMAGE_PIXELS = pixels
            

    @staticmethod
    def mergeImages(image1, image2):
        """Merge two images into one, displayed on top of one another
        :param image1: first buffered Image (top image)
        :param image2: second buffered Image (bottom image)
        :return: the merged Image object
        """

        (width1, height1) = image1.size
        (width2, height2) = image2.size

        result_width = max(width1, width2)
        result_height = height1 +  height2

        result = Image.new('RGB', (result_width, result_height))
        result.paste(im=image1, box=(0, 0))
        result.paste(im=image2, box=(0, height1))
        return result

    @staticmethod
    def stitchImages(images, outputFile):

        if len(images) == 0:
            log.error('Could not stitch files as none were found.')
            return

        bigImage = Image.open(images[0])

        for i in range(1,len(images)):
            if os.path.exists(images[i]):
                nextImage = Image.open(images[i])
            else:
                raise Exception(f'Image reference did not exist in temp images folder: {images[i]}')
            bigImage = ImageTools.mergeImages(bigImage, nextImage)
            nextImage.close()

        bigImage.save(outputFile)
        bigImage.close()

    @staticmethod
    def getImageSize(imagePath):
        image = Image.open(imagePath)
        size = image.size
        image.close()
        return size

    @staticmethod
    def cropTop(imagePath, numPixels):
        if numPixels == 0:
            return imagePath

        image = Image.open(imagePath)
        width, height = image.size
        croppedImagePath = imagePath.replace('.png', '_cropped_top.png')
        image.crop((0, numPixels, width, height)).save(croppedImagePath)
        image.close()
        return croppedImagePath

    @staticmethod
    def cropBottom(imagePath, numPixels):
        if numPixels == 0:
            return imagePath

        image = Image.open(imagePath)
        width, height = image.size
        croppedImagePath = imagePath.replace('.png', '_cropped_bottom.png')
        image.crop((0, 0, width, height-numPixels)).save(croppedImagePath)
        image.close()
        return croppedImagePath

    @staticmethod
    def imagesMatch(pathOne, pathTwo):
        imageOne = Image.open(pathOne)
        imageTwo = Image.open(pathTwo)

        diff = ImageChops.difference(imageOne, imageTwo)
        imageOne.close()
        imageTwo.close()

        if diff.getbbox():
            return False
        else:
            return True

    @staticmethod
    def imageDiff(pathOne, pathTwo):
        imageOne = Image.open(pathOne)
        imageTwo = Image.open(pathTwo)

        diff = ImageChops.difference(imageOne, imageTwo)
        imageOne.close()
        imageTwo.close()

        return diff

    """
    Given an browser screenshot image with toolbars and
    a viewport set to a single color throughout the image.
    Find the beginning and ending Y coordinates (or offsets)
    of the toolbars.
    """
    @staticmethod
    def getToolbarHeights(imagePath):

        toolbarHeights = {
            'top': 0,
            'bottom': 0
        }

        log.debug(f'Finding center RGB color in image: {imagePath}')
        # Load image, ensure not palettised, and make into Numpy array
        image = Image.open(imagePath)
        imageWidth, imageHeight = image.size
        log.debug(f'Image Dimensions: {imageWidth}x{imageHeight}')

        pim = image.convert('RGB')
        im  = numpy.array(pim)

        #get actual pixel value as some screenshots don't keep RGB of exact red
        color = pim.getpixel((imageWidth/2,imageHeight/2))
        log.debug(f'RGB value at center is {color}')

        # Define the colour we want to find - PIL uses RGB ordering
        # color = rgb

        # Get X and Y coordinates of all matching pixels
        Y,X = numpy.where(numpy.all(im==color,axis=2))
        coords = numpy.column_stack((X,Y))
        image.close()

        # log.debug(f'Finding RGB color in image.. results:')
        log.debug(coords)
        if len(coords) == 0:
            return toolbarHeights

        firstX, firstY = coords[0]
        lastX, lastY = coords[-1]

        hasTopToolbar = firstY > 0 # has top toolbar if first Y coordinate > 0
        hasBottomToolbar = lastY != imageHeight-1 # has bottom toolbar if last Y coordinate is not at the bottom of the image

        log.debug(f'hasTopToolbar={hasTopToolbar}, hasBottomToolbar={hasBottomToolbar}')
        log.debug(f'color starts at {firstY} and ends at {lastY} for total height of {lastY-firstY}')

        #ensure matches across full width
        firstRow = [pos for pos in coords if pos[1] == firstY]
        lastRow =  [pos for pos in coords if pos[1] == lastY]
        log.debug(f'firstRow length: {len(firstRow)}')
        log.debug(f'lastRow length: {len(lastRow)}')
        

        if len(firstRow) == imageWidth and len(lastRow) == imageWidth:
            log.debug(f'color pixel found across full width t first and last row')
        else:
            log.debug(f'color pixel IS NOT full width at first and last row')

        toolbarHeights['top'] = int(firstY) if hasTopToolbar else 0
        toolbarHeights['bottom'] = int(imageHeight - lastY - 1) if hasBottomToolbar else 0

        log.debug(f'Final toolbarHeights returned: {toolbarHeights}')

        return toolbarHeights


