"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const index_1 = require("../lib/index");
const cdk = require("aws-cdk-lib");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
function deployNewStateMachine(stack) {
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 's3stp-test')
        }
    };
    return new index_1.S3ToStepfunctions(stack, 'test-s3-stepfunctions', props);
}
test('override eventRuleProps', () => {
    const stack = new cdk.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket');
    const props = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 's3stp-test')
        },
        existingBucketObj: mybucket,
        eventRuleProps: {
            eventPattern: {
                source: ['aws.s3'],
                detailType: ['Object Created'],
                detail: {
                    bucket: {
                        name: [mybucket.bucketName]
                    }
                }
            }
        },
        deployCloudTrail: false // Testing warning
    };
    new index_1.S3ToStepfunctions(stack, 'test-s3-stepfunctions', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Events::Rule', {
        EventPattern: {
            "source": [
                "aws.s3"
            ],
            "detail-type": [
                "Object Created"
            ],
            "detail": {
                bucket: {
                    name: [{
                            Ref: "mybucket160F8132"
                        }
                    ]
                }
            }
        },
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "tests3stepfunctionstests3stepfunctionseventrulestepfunctionconstructStateMachine67197269"
                },
                Id: "Target0",
                RoleArn: {
                    "Fn::GetAtt": [
                        "tests3stepfunctionstests3stepfunctionseventrulestepfunctionconstructEventsRuleRoleE7CAD359",
                        "Arn"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStateMachine(stack);
    expect(construct.stateMachine).toBeDefined();
    expect(construct.s3Bucket).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    expect(construct.stateMachineLogGroup).toBeDefined();
    expect(construct.s3LoggingBucket).toBeDefined();
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Confirm that CheckS3Props is getting called", () => {
    // Stack
    const stack = new cdk.Stack();
    const testBucket = new aws_s3_1.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new index_1.S3ToStepfunctions(stack, "bad-s3-args", {
            stateMachineProps: {
                definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 's3stp-test')
            },
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: cdk.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// s3 bucket with bucket, loggingBucket, and auto delete objects
// --------------------------------------------------------------
test('s3 bucket with bucket, loggingBucket, and auto delete objects', () => {
    const stack = new cdk.Stack();
    const testProps = {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 's3stp-test')
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY
        },
        loggingBucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    };
    new index_1.S3ToStepfunctions(stack, 'test-s3-stepfunctions', testProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("Custom::S3BucketNotifications", {});
    template.resourceCountIs("AWS::S3::Bucket", 2);
    template.hasResourceProperties("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "tests3stepfunctionsS3LoggingBucketF7586A92"
        }
    });
});
// --------------------------------------------------------------
// s3 bucket with one content bucket and no logging bucket
// --------------------------------------------------------------
test('s3 bucket with no logging bucket', () => {
    const stack = new cdk.Stack();
    const construct = new index_1.S3ToStepfunctions(stack, 's3-stepfunctions', {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 's3stp-test')
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        },
        logS3AccessLogs: false
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("Custom::S3BucketNotifications", {});
    expect(construct.s3LoggingBucket).toEqual(undefined);
});
test('Deploy 2 constructs', () => {
    const stack = new cdk.Stack();
    new index_1.S3ToStepfunctions(stack, 'test-new-eventbridge-stepfunctions', {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'test'),
        },
    });
    new index_1.S3ToStepfunctions(stack, 'test-second-eventbridge-stepfunctions', {
        stateMachineProps: {
            definitionBody: defaults.CreateTestStateMachineDefinitionBody(stack, 'seconttest'),
        },
    });
    assertions_1.Template.fromStack(stack);
    // No checks, as our main concern is this has no collisions
});
//# sourceMappingURL=data:application/json;base64,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