"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib/core");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    static bundle(options) {
        const bundling = new Bundling(options);
        return aws_lambda_1.Code.fromAsset(path.dirname(options.moduleDir), {
            assetHashType: options.assetHashType ?? cdk.AssetHashType.OUTPUT,
            assetHash: options.assetHash,
            bundling: {
                image: bundling.image,
                command: bundling.command,
                environment: bundling.environment,
                local: bundling.local,
                entrypoint: bundling.entrypoint,
                volumes: bundling.volumes,
                volumesFrom: bundling.volumesFrom,
                workingDirectory: bundling.workingDirectory,
                user: bundling.user,
                securityOpt: bundling.securityOpt,
                network: bundling.network,
                bundlingFileAccess: bundling.bundlingFileAccess,
            },
        });
    }
    static clearRunsLocallyCache() {
        this.runsLocally = undefined;
    }
    constructor(props) {
        this.props = props;
        Bundling.runsLocally = Bundling.runsLocally
            ?? (0, util_1.getGoBuildVersion)()
            ?? false;
        const projectRoot = path.dirname(props.moduleDir);
        this.relativeEntryPath = `./${path.relative(projectRoot, path.resolve(props.entry))}`;
        const cgoEnabled = props.cgoEnabled ? '1' : '0';
        const environment = {
            CGO_ENABLED: cgoEnabled,
            GO111MODULE: 'on',
            GOARCH: props.architecture.dockerPlatform.split('/')[1],
            GOOS: 'linux',
            ...props.environment,
        };
        if (props.goProxies) {
            environment.GOPROXY = props.goProxies.join(',');
        }
        // Docker bundling
        const shouldBuildImage = props.forcedDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? props.dockerImage ?? cdk.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
                buildArgs: {
                    ...props.buildArgs ?? {},
                    IMAGE: aws_lambda_1.Runtime.GO_1_X.bundlingImage.image, // always use the GO_1_X build image
                },
                platform: props.architecture.dockerPlatform,
            })
            : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR);
        this.command = props.command ?? ['bash', '-c', bundlingCommand];
        this.environment = environment;
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.workingDirectory = props.workingDirectory;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
        // Local bundling
        if (!props.forcedDockerBundling) { // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(projectRoot, outputDir, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    if (Bundling.runsLocally == false) {
                        process.stderr.write('go build cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    (0, util_1.exec)(osPlatform === 'win32' ? 'cmd' : 'bash', [
                        osPlatform === 'win32' ? '/c' : '-c',
                        localCommand,
                    ], {
                        env: { ...process.env, ...environment ?? {} },
                        stdio: [
                            'ignore',
                            process.stderr,
                            'inherit', // inherit stderr
                        ],
                        cwd: path.dirname(props.moduleDir),
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    createBundlingCommand(inputDir, outputDir, osPlatform = 'linux') {
        const pathJoin = osPathJoin(osPlatform);
        const hasVendor = (0, util_1.findUp)('vendor', path.dirname(this.props.entry));
        const goBuildCommand = [
            'go', 'build',
            hasVendor ? '-mod=vendor' : '',
            '-o', `${pathJoin(outputDir, 'bootstrap')}`,
            `${this.props.goBuildFlags ? this.props.goBuildFlags.join(' ') : ''}`,
            `${this.relativeEntryPath.replace(/\\/g, '/')}`,
        ].filter(c => !!c).join(' ');
        return chain([
            ...this.props.commandHooks?.beforeBundling(inputDir, outputDir) ?? [],
            goBuildCommand,
            ...this.props.commandHooks?.afterBundling(inputDir, outputDir) ?? [],
        ]);
    }
}
exports.Bundling = Bundling;
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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