/**
 * Represents a length of time.
 *
 * The amount can be specified either as a literal value (e.g: `10`) which
 * cannot be negative, or as an unresolved number token.
 *
 * Whent he amount is passed as an token, unit conversion is not possible.
 */
export declare class Duration {
    /**
     * @param amount the amount of Milliseconds the `Duration` will represent.
     * @returns a new `Duration` representing `amount` ms.
     */
    static millis(amount: number): Duration;
    /**
     * @param amount the amount of Seconds the `Duration` will represent.
     * @returns a new `Duration` representing `amount` Seconds.
     */
    static seconds(amount: number): Duration;
    /**
     * @param amount the amount of Minutes the `Duration` will represent.
     * @returns a new `Duration` representing `amount` Minutes.
     */
    static minutes(amount: number): Duration;
    /**
     * @param amount the amount of Hours the `Duration` will represent.
     * @returns a new `Duration` representing `amount` Hours.
     */
    static hours(amount: number): Duration;
    /**
     * @param amount the amount of Days the `Duration` will represent.
     * @returns a new `Duration` representing `amount` Days.
     */
    static days(amount: number): Duration;
    /**
     * Parse a period formatted according to the ISO 8601 standard (see https://www.iso.org/fr/standard/70907.html).
     *
     * @param duration an ISO-formtted duration to be parsed.
     * @returns the parsed `Duration`.
     */
    static parse(duration: string): Duration;
    private readonly amount;
    private readonly unit;
    private constructor();
    /**
     * @returns the value of this `Duration` expressed in Milliseconds.
     */
    toMilliseconds(opts?: TimeConversionOptions): number;
    /**
     * @returns the value of this `Duration` expressed in Seconds.
     */
    toSeconds(opts?: TimeConversionOptions): number;
    /**
     * @returns the value of this `Duration` expressed in Minutes.
     */
    toMinutes(opts?: TimeConversionOptions): number;
    /**
     * @returns the value of this `Duration` expressed in Hours.
     */
    toHours(opts?: TimeConversionOptions): number;
    /**
     * @returns the value of this `Duration` expressed in Days.
     */
    toDays(opts?: TimeConversionOptions): number;
    /**
     * @returns an ISO 8601 representation of this period (see https://www.iso.org/fr/standard/70907.html).
     */
    toISOString(): string;
    /**
     * Returns a string representation of this `Duration` that is also a Token that cannot be successfully resolved. This
     * protects users against inadvertently stringifying a `Duration` object, when they should have called one of the
     * `to*` methods instead.
     */
    toString(): string;
    private fractionDuration;
}
/**
 * Options for how to convert time to a different unit.
 */
export interface TimeConversionOptions {
    /**
     * If `true`, conversions into a larger time unit (e.g. `Seconds` to `Mintues`) will fail if the result is not an
     * integer.
     *
     * @default true
     */
    readonly integral?: boolean;
}
