"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
const util_1 = require("../lib/util");
const util_2 = require("./util");
class StackWithPostProcessor extends lib_1.Stack {
    // ...
    _toCloudFormation() {
        const template = super._toCloudFormation();
        // manipulate template (e.g. rename "Key" to "key")
        template.Resources.myResource.Properties.Environment.key =
            template.Resources.myResource.Properties.Environment.Key;
        delete template.Resources.myResource.Properties.Environment.Key;
        return template;
    }
}
module.exports = {
    'a stack can be serialized into a CloudFormation template, initially it\'s empty'(test) {
        const stack = new lib_1.Stack();
        test.deepEqual(util_2.toCloudFormation(stack), {});
        test.done();
    },
    'stack objects have some template-level propeties, such as Description, Version, Transform'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.templateFormatVersion = 'MyTemplateVersion';
        stack.templateOptions.description = 'This is my description';
        stack.templateOptions.transform = 'SAMy';
        test.deepEqual(util_2.toCloudFormation(stack), {
            Description: 'This is my description',
            AWSTemplateFormatVersion: 'MyTemplateVersion',
            Transform: 'SAMy'
        });
        test.done();
    },
    'Stack.isStack indicates that a construct is a stack'(test) {
        const stack = new lib_1.Stack();
        const c = new lib_1.Construct(stack, 'Construct');
        test.ok(lib_1.Stack.isStack(stack));
        test.ok(!lib_1.Stack.isStack(c));
        test.done();
    },
    'stack.id is not included in the logical identities of resources within it'(test) {
        const stack = new lib_1.Stack(undefined, 'MyStack');
        new lib_1.CfnResource(stack, 'MyResource', { type: 'MyResourceType' });
        test.deepEqual(util_2.toCloudFormation(stack), { Resources: { MyResource: { Type: 'MyResourceType' } } });
        test.done();
    },
    'stack.templateOptions can be used to set template-level options'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.description = 'StackDescription';
        stack.templateOptions.templateFormatVersion = 'TemplateVersion';
        stack.templateOptions.transform = 'DeprecatedField';
        stack.templateOptions.transforms = ['Transform'];
        stack.templateOptions.metadata = {
            MetadataKey: 'MetadataValue'
        };
        test.deepEqual(util_2.toCloudFormation(stack), {
            Description: 'StackDescription',
            Transform: ['DeprecatedField', 'Transform'],
            AWSTemplateFormatVersion: 'TemplateVersion',
            Metadata: { MetadataKey: 'MetadataValue' }
        });
        test.done();
    },
    // This approach will only apply to TypeScript code, but at least it's a temporary
    // workaround for people running into issues caused by SDK-3003.
    // We should come up with a proper solution that involved jsii callbacks (when they exist)
    // so this can be implemented by jsii languages as well.
    'Overriding `Stack._toCloudFormation` allows arbitrary post-processing of the generated template during synthesis'(test) {
        const stack = new StackWithPostProcessor();
        new lib_1.CfnResource(stack, 'myResource', {
            type: 'AWS::MyResource',
            properties: {
                MyProp1: 'hello',
                MyProp2: 'howdy',
                Environment: {
                    Key: 'value'
                }
            }
        });
        test.deepEqual(stack._toCloudFormation(), { Resources: { myResource: { Type: 'AWS::MyResource',
                    Properties: { MyProp1: 'hello',
                        MyProp2: 'howdy',
                        Environment: { key: 'value' } } } } });
        test.done();
    },
    'Stack.getByPath can be used to find any CloudFormation element (Parameter, Output, etc)'(test) {
        const stack = new lib_1.Stack();
        const p = new lib_1.CfnParameter(stack, 'MyParam', { type: 'String' });
        const o = new lib_1.CfnOutput(stack, 'MyOutput', { value: 'boom' });
        const c = new lib_1.CfnCondition(stack, 'MyCondition');
        test.equal(stack.node.findChild(p.node.path), p);
        test.equal(stack.node.findChild(o.node.path), o);
        test.equal(stack.node.findChild(c.node.path), c);
        test.done();
    },
    'Stack names can have hyphens in them'(test) {
        const root = new lib_1.App();
        new lib_1.Stack(root, 'Hello-World');
        // Did not throw
        test.done();
    },
    'Stacks can have a description given to them'(test) {
        const stack = new lib_1.Stack(new lib_1.App(), 'MyStack', { description: 'My stack, hands off!' });
        const output = util_2.toCloudFormation(stack);
        test.equal(output.Description, 'My stack, hands off!');
        test.done();
    },
    'Stack descriptions have a limited length'(test) {
        const desc = `Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor
     incididunt ut labore et dolore magna aliqua. Consequat interdum varius sit amet mattis vulputate
     enim nulla aliquet. At imperdiet dui accumsan sit amet nulla facilisi morbi. Eget lorem dolor sed
     viverra ipsum. Diam volutpat commodo sed egestas egestas. Sit amet porttitor eget dolor morbi non.
     Lorem dolor sed viverra ipsum. Id porta nibh venenatis cras sed felis. Augue interdum velit euismod
     in pellentesque. Suscipit adipiscing bibendum est ultricies integer quis. Condimentum id venenatis a
     condimentum vitae sapien pellentesque habitant morbi. Congue mauris rhoncus aenean vel elit scelerisque
     mauris pellentesque pulvinar.
     Faucibus purus in massa tempor nec. Risus viverra adipiscing at in. Integer feugiat scelerisque varius
     morbi. Malesuada nunc vel risus commodo viverra maecenas accumsan lacus. Vulputate sapien nec sagittis
     aliquam malesuada bibendum arcu vitae. Augue neque gravida in fermentum et sollicitudin ac orci phasellus.
     Ultrices tincidunt arcu non sodales neque sodales.`;
        test.throws(() => new lib_1.Stack(new lib_1.App(), 'MyStack', { description: desc }));
        test.done();
    },
    'Include should support non-hash top-level template elements like "Description"'(test) {
        const stack = new lib_1.Stack();
        const template = {
            Description: 'hello, world'
        };
        new lib_1.CfnInclude(stack, 'Include', { template });
        const output = util_2.toCloudFormation(stack);
        test.equal(typeof output.Description, 'string');
        test.done();
    },
    'Pseudo values attached to one stack can be referenced in another stack'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        // THEN
        const assembly = app.synth();
        const template1 = assembly.getStackByName(stack1.stackName).template;
        const template2 = assembly.getStackByName(stack2.stackName).template;
        test.deepEqual(template1, {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.done();
    },
    'Cross-stack references are detected in resource properties'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const resource1 = new lib_1.CfnResource(stack1, 'Resource', { type: 'BLA' });
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another resource
        new lib_1.CfnResource(stack2, 'SomeResource', { type: 'AWS::Some::Resource', properties: {
                someProperty: new intrinsic_1.Intrinsic(resource1.ref),
            } });
        // THEN
        const assembly = app.synth();
        const template2 = assembly.getStackByName(stack2.stackName).template;
        test.deepEqual(template2, {
            Resources: {
                SomeResource: {
                    Type: 'AWS::Some::Resource',
                    Properties: {
                        someProperty: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefResource1D5D905A' }
                    }
                }
            }
        });
        test.done();
    },
    'cross-stack references in lazy tokens work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: lib_1.Lazy.stringValue({ produce: () => account1 }) });
        const assembly = app.synth();
        const template1 = assembly.getStackByName(stack1.stackName).template;
        const template2 = assembly.getStackByName(stack2.stackName).template;
        // THEN
        test.deepEqual(template1, {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }
                }
            }
        });
        test.done();
    },
    'Cross-stack use of Region and account returns nonscoped intrinsic because the two stacks must be in the same region anyway'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnOutput(stack2, 'DemOutput', { value: stack1.region });
        new lib_1.CfnOutput(stack2, 'DemAccount', { value: stack1.account });
        // THEN
        const assembly = app.synth();
        const template2 = assembly.getStackByName(stack2.stackName).template;
        test.deepEqual(template2, {
            Outputs: {
                DemOutput: {
                    Value: { Ref: 'AWS::Region' },
                },
                DemAccount: {
                    Value: { Ref: 'AWS::AccountId' },
                }
            }
        });
        test.done();
    },
    'cross-stack references in strings work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: `TheAccountIs${account1}` });
        const assembly = app.synth();
        const template2 = assembly.getStackByName(stack2.stackName).template;
        // THEN
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::Join': ['', ['TheAccountIs', { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }]] }
                }
            }
        });
        test.done();
    },
    'CfnSynthesisError is ignored when preparing cross references'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'my-stack');
        // WHEN
        class CfnTest extends lib_1.CfnResource {
            _toCloudFormation() {
                return new util_1.PostResolveToken({
                    xoo: 1234
                }, props => {
                    lib_1.validateString(props).assertSuccess();
                });
            }
        }
        new CfnTest(stack, 'MyThing', { type: 'AWS::Type' });
        // THEN
        lib_1.ConstructNode.prepare(stack.node);
        test.done();
    },
    'Stacks can be children of other stacks (substack) and they will be synthesized separately'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const parentStack = new lib_1.Stack(app, 'parent');
        const childStack = new lib_1.Stack(parentStack, 'child');
        new lib_1.CfnResource(parentStack, 'MyParentResource', { type: 'Resource::Parent' });
        new lib_1.CfnResource(childStack, 'MyChildResource', { type: 'Resource::Child' });
        // THEN
        const assembly = app.synth();
        test.deepEqual(assembly.getStackByName(parentStack.stackName).template, { Resources: { MyParentResource: { Type: 'Resource::Parent' } } });
        test.deepEqual(assembly.getStackByName(childStack.stackName).template, { Resources: { MyChildResource: { Type: 'Resource::Child' } } });
        test.done();
    },
    'cross-stack reference (substack references parent stack)'(test) {
        // GIVEN
        const app = new lib_1.App();
        const parentStack = new lib_1.Stack(app, 'parent');
        const childStack = new lib_1.Stack(parentStack, 'child');
        // WHEN (a resource from the child stack references a resource from the parent stack)
        const parentResource = new lib_1.CfnResource(parentStack, 'MyParentResource', { type: 'Resource::Parent' });
        new lib_1.CfnResource(childStack, 'MyChildResource', {
            type: 'Resource::Child',
            properties: {
                ChildProp: parentResource.getAtt('AttOfParentResource')
            }
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(assembly.getStackByName(parentStack.stackName).template, {
            Resources: { MyParentResource: { Type: 'Resource::Parent' } },
            Outputs: { ExportsOutputFnGetAttMyParentResourceAttOfParentResourceC2D0BB9E: {
                    Value: { 'Fn::GetAtt': ['MyParentResource', 'AttOfParentResource'] },
                    Export: { Name: 'parent:ExportsOutputFnGetAttMyParentResourceAttOfParentResourceC2D0BB9E' }
                }
            }
        });
        test.deepEqual(assembly.getStackByName(childStack.stackName).template, {
            Resources: {
                MyChildResource: {
                    Type: 'Resource::Child',
                    Properties: {
                        ChildProp: {
                            'Fn::ImportValue': 'parent:ExportsOutputFnGetAttMyParentResourceAttOfParentResourceC2D0BB9E'
                        }
                    }
                }
            }
        });
        test.done();
    },
    'cross-stack reference (parent stack references substack)'(test) {
        // GIVEN
        const app = new lib_1.App();
        const parentStack = new lib_1.Stack(app, 'parent');
        const childStack = new lib_1.Stack(parentStack, 'child');
        // WHEN (a resource from the child stack references a resource from the parent stack)
        const childResource = new lib_1.CfnResource(childStack, 'MyChildResource', { type: 'Resource::Child' });
        new lib_1.CfnResource(parentStack, 'MyParentResource', {
            type: 'Resource::Parent',
            properties: {
                ParentProp: childResource.getAtt('AttributeOfChildResource')
            }
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(assembly.getStackByName(parentStack.stackName).template, {
            Resources: {
                MyParentResource: {
                    Type: 'Resource::Parent',
                    Properties: {
                        ParentProp: { 'Fn::ImportValue': 'parentchild13F9359B:childExportsOutputFnGetAttMyChildResourceAttributeOfChildResource420052FC' }
                    }
                }
            }
        });
        test.deepEqual(assembly.getStackByName(childStack.stackName).template, {
            Resources: {
                MyChildResource: { Type: 'Resource::Child' }
            },
            Outputs: {
                ExportsOutputFnGetAttMyChildResourceAttributeOfChildResource52813264: {
                    Value: { 'Fn::GetAtt': ['MyChildResource', 'AttributeOfChildResource'] },
                    Export: { Name: 'parentchild13F9359B:childExportsOutputFnGetAttMyChildResourceAttributeOfChildResource420052FC' }
                }
            }
        });
        test.done();
    },
    'cannot create cyclic reference between stacks'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        const account2 = new lib_1.ScopedAws(stack2).accountId;
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        new lib_1.CfnParameter(stack1, 'SomeParameter', { type: 'String', default: account2 });
        test.throws(() => {
            lib_1.ConstructNode.prepare(app.node);
            // tslint:disable-next-line:max-line-length
        }, "'Stack2' depends on 'Stack1' (Stack2/SomeParameter -> Stack1.AWS::AccountId). Adding this dependency (Stack1/SomeParameter -> Stack2.AWS::AccountId) would create a cyclic reference.");
        test.done();
    },
    'stacks know about their dependencies'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        lib_1.ConstructNode.prepare(app.node);
        // THEN
        test.deepEqual(stack2.dependencies.map(s => s.node.id), ['Stack1']);
        test.done();
    },
    'cannot create references to stacks in other regions/accounts'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2', { env: { account: '123456789012', region: 'es-norst-2' } });
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        test.throws(() => {
            lib_1.ConstructNode.prepare(app.node);
        }, /Stack "Stack2" cannot consume a cross reference from stack "Stack1"/);
        test.done();
    },
    'urlSuffix does not imply a stack dependency'(test) {
        // GIVEN
        const app = new lib_1.App();
        const first = new lib_1.Stack(app, 'First');
        const second = new lib_1.Stack(app, 'Second');
        // WHEN
        new lib_1.CfnOutput(second, 'Output', {
            value: first.urlSuffix
        });
        // THEN
        app.synth();
        test.equal(second.dependencies.length, 0);
        test.done();
    },
    'stack with region supplied via props returns literal value'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        // THEN
        test.equal(stack.resolve(stack.region), 'es-norst-1');
        test.done();
    },
    'overrideLogicalId(id) can be used to override the logical ID of a resource'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const bonjour = new lib_1.CfnResource(stack, 'BonjourResource', { type: 'Resource::Type' });
        // { Ref } and { GetAtt }
        new lib_1.CfnResource(stack, 'RefToBonjour', { type: 'Other::Resource', properties: {
                RefToBonjour: bonjour.ref,
                GetAttBonjour: bonjour.getAtt('TheAtt').toString()
            } });
        bonjour.overrideLogicalId('BOOM');
        // THEN
        test.deepEqual(util_2.toCloudFormation(stack), { Resources: { BOOM: { Type: 'Resource::Type' },
                RefToBonjour: { Type: 'Other::Resource',
                    Properties: { RefToBonjour: { Ref: 'BOOM' },
                        GetAttBonjour: { 'Fn::GetAtt': ['BOOM', 'TheAtt'] } } } } });
        test.done();
    },
    'Stack name can be overridden via properties'(test) {
        // WHEN
        const stack = new lib_1.Stack(undefined, 'Stack', { stackName: 'otherName' });
        // THEN
        test.deepEqual(stack.stackName, 'otherName');
        test.done();
    },
    'Stack name is inherited from App name if available'(test) {
        // WHEN
        const root = new lib_1.App();
        const app = new lib_1.Construct(root, 'Prod');
        const stack = new lib_1.Stack(app, 'Stack');
        // THEN
        test.deepEqual(stack.stackName, 'ProdStackD5279B22');
        test.done();
    },
    'stack construct id does not go through stack name validation if there is an explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const stack = new lib_1.Stack(app, 'invalid as : stack name, but thats fine', {
            stackName: 'valid-stack-name'
        });
        // THEN
        const session = app.synth();
        test.deepEqual(stack.stackName, 'valid-stack-name');
        test.ok(session.tryGetArtifact(stack.artifactId));
        test.done();
    },
    'stack validation is performed on explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // THEN
        test.throws(() => new lib_1.Stack(app, 'boom', { stackName: 'invalid:stack:name' }), /Stack name must match the regular expression/);
        test.done();
    },
    'Stack.of(stack) returns the correct stack'(test) {
        const stack = new lib_1.Stack();
        test.same(lib_1.Stack.of(stack), stack);
        const parent = new lib_1.Construct(stack, 'Parent');
        const construct = new lib_1.Construct(parent, 'Construct');
        test.same(lib_1.Stack.of(construct), stack);
        test.done();
    },
    'Stack.of() throws when there is no parent Stack'(test) {
        const root = new lib_1.Construct(undefined, 'Root');
        const construct = new lib_1.Construct(root, 'Construct');
        test.throws(() => lib_1.Stack.of(construct), /No stack could be identified for the construct at path/);
        test.done();
    },
    'Stack.of() works for substacks'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const parentStack = new lib_1.Stack(app, 'ParentStack');
        const parentResource = new lib_1.CfnResource(parentStack, 'ParentResource', { type: 'parent::resource' });
        // we will define a substack under the /resource/... just for giggles.
        const childStack = new lib_1.Stack(parentResource, 'ChildStack');
        const childResource = new lib_1.CfnResource(childStack, 'ChildResource', { type: 'child::resource' });
        // THEN
        test.same(lib_1.Stack.of(parentStack), parentStack);
        test.same(lib_1.Stack.of(parentResource), parentStack);
        test.same(lib_1.Stack.of(childStack), childStack);
        test.same(lib_1.Stack.of(childResource), childStack);
        test.done();
    },
    'stack.availabilityZones falls back to Fn::GetAZ[0],[2] if region is not specified'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'MyStack');
        // WHEN
        const azs = stack.availabilityZones;
        // THEN
        test.deepEqual(stack.resolve(azs), [
            { "Fn::Select": [0, { "Fn::GetAZs": "" }] },
            { "Fn::Select": [1, { "Fn::GetAZs": "" }] }
        ]);
        test.done();
    },
    'stack.templateFile is the name of the template file emitted to the cloud assembly (default is to use the stack name)'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const stack1 = new lib_1.Stack(app, 'MyStack1');
        const stack2 = new lib_1.Stack(app, 'MyStack2', { stackName: 'MyRealStack2' });
        // THEN
        test.deepEqual(stack1.templateFile, 'MyStack1.template.json');
        test.deepEqual(stack2.templateFile, 'MyRealStack2.template.json');
        test.done();
    },
    'when feature flag is enabled we will use the artifact id as the template name'(test) {
        // GIVEN
        const app = new lib_1.App({
            context: {
                [cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: 'true'
            }
        });
        // WHEN
        const stack1 = new lib_1.Stack(app, 'MyStack1');
        const stack2 = new lib_1.Stack(app, 'MyStack2', { stackName: 'MyRealStack2' });
        // THEN
        test.deepEqual(stack1.templateFile, 'MyStack1.template.json');
        test.deepEqual(stack2.templateFile, 'MyStack2.template.json');
        test.done();
    },
    '@aws-cdk/core:enableStackNameDuplicates': {
        'disabled (default)': {
            'artifactId and templateFile use the stack name'(test) {
                // GIVEN
                const app = new lib_1.App();
                // WHEN
                const stack1 = new lib_1.Stack(app, 'MyStack1', { stackName: 'thestack' });
                const assembly = app.synth();
                // THEN
                test.deepEqual(stack1.artifactId, 'thestack');
                test.deepEqual(stack1.templateFile, 'thestack.template.json');
                test.deepEqual(assembly.getStackArtifact(stack1.artifactId).templateFile, 'thestack.template.json');
                test.done();
            }
        },
        'enabled': {
            'allows using the same stack name for two stacks (i.e. in different regions)'(test) {
                // GIVEN
                const app = new lib_1.App({ context: { [cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: 'true' } });
                // WHEN
                const stack1 = new lib_1.Stack(app, 'MyStack1', { stackName: 'thestack' });
                const stack2 = new lib_1.Stack(app, 'MyStack2', { stackName: 'thestack' });
                const assembly = app.synth();
                // THEN
                test.deepEqual(assembly.getStackArtifact(stack1.artifactId).templateFile, 'MyStack1.template.json');
                test.deepEqual(assembly.getStackArtifact(stack2.artifactId).templateFile, 'MyStack2.template.json');
                test.deepEqual(stack1.templateFile, 'MyStack1.template.json');
                test.deepEqual(stack2.templateFile, 'MyStack2.template.json');
                test.done();
            },
            'artifactId and templateFile use the unique id and not the stack name'(test) {
                // GIVEN
                const app = new lib_1.App({ context: { [cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: 'true' } });
                // WHEN
                const stack1 = new lib_1.Stack(app, 'MyStack1', { stackName: 'thestack' });
                const assembly = app.synth();
                // THEN
                test.deepEqual(stack1.artifactId, 'MyStack1');
                test.deepEqual(stack1.templateFile, 'MyStack1.template.json');
                test.deepEqual(assembly.getStackArtifact(stack1.artifactId).templateFile, 'MyStack1.template.json');
                test.done();
            }
        }
    },
    'metadata is collected at the stack boundary'(test) {
        // GIVEN
        const app = new lib_1.App({
            context: {
                [cxapi.DISABLE_METADATA_STACK_TRACE]: 'true'
            }
        });
        const parent = new lib_1.Stack(app, 'parent');
        const child = new lib_1.Stack(parent, 'child');
        // WHEN
        child.node.addMetadata('foo', 'bar');
        // THEN
        const asm = app.synth();
        test.deepEqual(asm.getStackByName(parent.stackName).findMetadataByType('foo'), []);
        test.deepEqual(asm.getStackByName(child.stackName).findMetadataByType('foo'), [
            { path: '/parent/child', type: 'foo', data: 'bar' }
        ]);
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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