import logging

import sqlalchemy as sa
from alembic.autogenerate import produce_migrations, render_python_code
from alembic.operations import Operations
from alembic.runtime.migration import MigrationContext

from mitm_tooling.utilities.sql_utils import AnyDBBind, use_nested_conn

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
# logger.addHandler(logging.FileHandler('db_migration.log', mode='w'))


def run_ephemeral_migration(
    bind: AnyDBBind,
    target_metadata: sa.MetaData,
    *,
    compare_type: bool = True,
    compare_server_default: bool = True,
    include_default_schema: bool = False,
) -> bool:
    """
    Autogenerate and APPLY schema changes directly to the database,
    without creating Alembic files or version history.

    Returns True if changes were applied, False if no diff.
    """

    target_schema = target_metadata.schema

    def only_my_schema(name, type_, parent_names):
        """Return True if Alembic should include this object in diffs."""
        x = None if include_default_schema else target_schema
        y = name is x or parent_names.get('schema_name') is x
        return y

    # For SQLite, ALTER support is limited; batch mode helps
    render_as_batch = bind.dialect.name == 'sqlite'
    with use_nested_conn(bind) as conn:
        # Build a MigrationContext bound to the live connection
        ctx = MigrationContext.configure(
            conn,
            opts={
                'target_metadata': target_metadata,
                'compare_type': compare_type,
                'compare_server_default': compare_server_default,
                'include_schemas': True,
                'include_name': only_my_schema,
                'render_as_batch': render_as_batch,
            },
        )

        # Compute the autogenerate diff
        script = produce_migrations(ctx, target_metadata)
        up_ops = script.upgrade_ops

        if up_ops.is_empty():
            logger.info('No schema changes detected.')
            return False

        # Log a pretty “migration script” (for audit/debug), no files written
        logger.info('Autogenerated migration (python):\n%s', render_python_code(up_ops))

        # Apply the operations directly
        ops = Operations(ctx)
        logger.info('Applying migration operations...')
        skipped_ops = []
        for m_op in up_ops.ops:
            # Invoke each Alembic operation against the current connection
            try:
                logger.info('Invoking %s', m_op)
                ops.invoke(m_op)
            except Exception:
                skipped_ops.append(
                    (
                        type(m_op),
                        repr(m_op),
                    )
                )
        if skipped_ops:
            logger.warning('Migration applied. Skipped the following operations:')
            for op in skipped_ops:
                logger.warning('  %s', op)
        else:
            logger.info('Migration applied successfully.')

        return True
