/**
 * Bundling options
 *
 * @experimental
 */
export interface BundlingOptions {
    /**
     * The Docker image where the command will run.
     */
    readonly image: BundlingDockerImage;
    /**
     * The command to run in the Docker container.
     *
     * @example ['npm', 'install']
     *
     * @see https://docs.docker.com/engine/reference/run/
     *
     * @default - run the command defined in the image
     */
    readonly command?: string[];
    /**
     * Additional Docker volumes to mount.
     *
     * @default - no additional volumes are mounted
     */
    readonly volumes?: DockerVolume[];
    /**
     * The environment variables to pass to the Docker container.
     *
     * @default - no environment variables.
     */
    readonly environment?: {
        [key: string]: string;
    };
    /**
     * Working directory inside the Docker container.
     *
     * @default /asset-input
     */
    readonly workingDirectory?: string;
    /**
     * The user to use when running the Docker container.
     *
     *   user | user:group | uid | uid:gid | user:gid | uid:group
     *
     * @see https://docs.docker.com/engine/reference/run/#user
     *
     * @default - uid:gid of the current user or 1000:1000 on Windows
     */
    readonly user?: string;
    /**
     * Local bundling provider.
     *
     * The provider implements a method `tryBundle()` which should return `true`
     * if local bundling was performed. If `false` is returned, docker bundling
     * will be done.
     *
     * @default - bundling will only be performed in a Docker container
     *
     * @experimental
     */
    readonly local?: ILocalBundling;
}
/**
 * Local bundling
 *
 * @experimental
 */
export interface ILocalBundling {
    /**
     * This method is called before attempting docker bundling to allow the
     * bundler to be executed locally. If the local bundler exists, and bundling
     * was performed locally, return `true`. Otherwise, return `false`.
     *
     * @param outputDir the directory where the bundled asset should be output
     * @param options bundling options for this asset
     */
    tryBundle(outputDir: string, options: BundlingOptions): boolean;
}
/**
 * A Docker image used for asset bundling
 */
export declare class BundlingDockerImage {
    readonly image: string;
    /**
     * Reference an image on DockerHub or another online registry.
     *
     * @param image the image name
     */
    static fromRegistry(image: string): BundlingDockerImage;
    /**
     * Reference an image that's built directly from sources on disk.
     *
     * @param path The path to the directory containing the Docker file
     * @param options Docker build options
     */
    static fromAsset(path: string, options?: DockerBuildOptions): BundlingDockerImage;
    /** @param image The Docker image */
    private constructor();
}
/**
 * A Docker volume
 */
export interface DockerVolume {
    /**
     * The path to the file or directory on the host machine
     */
    readonly hostPath: string;
    /**
     * The path where the file or directory is mounted in the container
     */
    readonly containerPath: string;
    /**
     * Mount consistency. Only applicable for macOS
     *
     * @default DockerConsistency.DELEGATED
     * @see https://docs.docker.com/storage/bind-mounts/#configure-mount-consistency-for-macos
     */
    readonly consistency?: DockerVolumeConsistency;
}
/**
 * Supported Docker volume consistency types. Only valid on macOS due to the way file storage works on Mac
 */
export declare enum DockerVolumeConsistency {
    /**
     * Read/write operations inside the Docker container are applied immediately on the mounted host machine volumes
     */
    CONSISTENT = "consistent",
    /**
     * Read/write operations on mounted Docker volumes are first written inside the container and then synchronized to the host machine
     */
    DELEGATED = "delegated",
    /**
     * Read/write operations on mounted Docker volumes are first applied on the host machine and then synchronized to the container
     */
    CACHED = "cached"
}
/**
 * Docker build options
 */
export interface DockerBuildOptions {
    /**
     * Build args
     *
     * @default - no build args
     */
    readonly buildArgs?: {
        [key: string]: string;
    };
}
