"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnReference = void 0;
const reference_1 = require("../reference");
const CFN_REFERENCE_SYMBOL = Symbol.for('@aws-cdk/core.CfnReference');
/**
 * A Token that represents a CloudFormation reference to another resource
 *
 * If these references are used in a different stack from where they are
 * defined, appropriate CloudFormation `Export`s and `Fn::ImportValue`s will be
 * synthesized automatically instead of the regular CloudFormation references.
 *
 * Additionally, the dependency between the stacks will be recorded, and the toolkit
 * will make sure to deploy producing stack before the consuming stack.
 *
 * This magic happens in the prepare() phase, where consuming stacks will call
 * `consumeFromStack` on these Tokens and if they happen to be exported by a different
 * Stack, we'll register the dependency.
 */
class CfnReference extends reference_1.Reference {
    constructor(value, displayName, target) {
        // prepend scope path to display name
        super(value, target, displayName);
        this.replacementTokens = new Map();
        this.targetStack = stack_1.Stack.of(target);
        Object.defineProperty(this, CFN_REFERENCE_SYMBOL, { value: true });
    }
    /**
     * Check whether this is actually a Reference
     */
    static isCfnReference(x) {
        return CFN_REFERENCE_SYMBOL in x;
    }
    /**
     * Return the CfnReference for the indicated target
     *
     * Will make sure that multiple invocations for the same target and intrinsic
     * return the same CfnReference. Because CfnReferences accumulate state in
     * the prepare() phase (for the purpose of cross-stack references), it's
     * important that the state isn't lost if it's lazily created, like so:
     *
     *     Lazy.stringValue({ produce: () => new CfnReference(...) })
     *
     * If fnSub is true, then this reference will resolve as ${logicalID}.
     * This allows cloudformation-include to correctly handle Fn::Sub.
     */
    static for(target, attribute, fnSub = false) {
        return CfnReference.singletonReference(target, attribute, fnSub, () => {
            const cfnIntrinsic = fnSub
                ? ('${' + target.logicalId + (attribute === 'Ref' ? '' : `.${attribute}`) + '}')
                : (attribute === 'Ref' ? { Ref: target.logicalId } : { 'Fn::GetAtt': [target.logicalId, attribute] });
            return new CfnReference(cfnIntrinsic, attribute, target);
        });
    }
    /**
     * Return a CfnReference that references a pseudo referencd
     */
    static forPseudo(pseudoName, scope) {
        return CfnReference.singletonReference(scope, `Pseudo:${pseudoName}`, false, () => {
            const cfnIntrinsic = { Ref: pseudoName };
            return new CfnReference(cfnIntrinsic, pseudoName, scope);
        });
    }
    /**
     * Get or create the table.
     * Passing fnSub = true allows cloudformation-include to correctly handle Fn::Sub.
     */
    static singletonReference(target, attribKey, fnSub, fresh) {
        let attribs = CfnReference.referenceTable.get(target);
        if (!attribs) {
            attribs = new Map();
            CfnReference.referenceTable.set(target, attribs);
        }
        const cacheKey = attribKey + (fnSub ? 'Fn::Sub' : '');
        let ref = attribs.get(cacheKey);
        if (!ref) {
            ref = fresh();
            attribs.set(cacheKey, ref);
        }
        return ref;
    }
    resolve(context) {
        // If we have a special token for this consuming stack, resolve that. Otherwise resolve as if
        // we are in the same stack.
        const consumingStack = stack_1.Stack.of(context.scope);
        const token = this.replacementTokens.get(consumingStack);
        // if (!token && this.isCrossStackReference(consumingStack) && !context.preparing) {
        // eslint-disable-next-line max-len
        //   throw new Error(`Cross-stack reference (${context.scope.node.path} -> ${this.target.node.path}) has not been assigned a value--call prepare() first`);
        // }
        if (token) {
            return token.resolve(context);
        }
        else {
            return super.resolve(context);
        }
    }
    hasValueForStack(stack) {
        if (stack === this.targetStack) {
            return true;
        }
        return this.replacementTokens.has(stack);
    }
    assignValueForStack(stack, value) {
        if (stack === this.targetStack) {
            throw new Error('cannot assign a value for the same stack');
        }
        if (this.hasValueForStack(stack)) {
            throw new Error('Cannot assign a reference value twice to the same stack. Use hasValueForStack to check first');
        }
        this.replacementTokens.set(stack, value);
    }
    /**
     * Implementation of toString() that will use the display name
     */
    toString() {
        return token_1.Token.asString(this, {
            displayHint: `${this.target.node.id}.${this.displayName}`,
        });
    }
}
exports.CfnReference = CfnReference;
/**
 * Static table where we keep singleton CfnReference instances
 */
CfnReference.referenceTable = new Map();
const stack_1 = require("../stack");
const token_1 = require("../token");
//# sourceMappingURL=data:application/json;base64,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