"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
require("@aws-cdk/assert/jest");
function deployStack(stack) {
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler'
        },
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    return new lib_1.IotToLambdaToDynamoDB(stack, 'test-iot-lambda-dynamodb-stack', props);
}
test('check lambda function properties', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: "index.handler",
        Role: {
            "Fn::GetAtt": [
                "testiotlambdadynamodbstackIotToLambdaLambdaFunctionServiceRoleC57F7FDA",
                "Arn"
            ]
        },
        Runtime: "nodejs10.x",
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                DDB_TABLE_NAME: {
                    Ref: "testiotlambdadynamodbstackLambdaToDynamoDBDynamoTableE17E5733"
                }
            }
        }
    });
});
test('check lambda function permission', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::Lambda::Permission', {
        Action: "lambda:InvokeFunction",
        FunctionName: {
            "Fn::GetAtt": [
                "testiotlambdadynamodbstackIotToLambdaLambdaFunctionDFEAF894",
                "Arn"
            ]
        },
        Principal: "iot.amazonaws.com",
        SourceArn: {
            "Fn::GetAtt": [
                "testiotlambdadynamodbstackIotToLambdaIotTopic74F5E3BB",
                "Arn"
            ]
        }
    });
});
test('check iot lambda function role', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "lambda.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/lambda/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "LambdaFunctionServiceRolePolicy"
            }
        ]
    });
});
test('check iot topic rule properties', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    Lambda: {
                        FunctionArn: {
                            "Fn::GetAtt": [
                                "testiotlambdadynamodbstackIotToLambdaLambdaFunctionDFEAF894",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'connectedcar/dtc/#'"
        }
    });
});
test('check dynamo table properties', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('check lambda function policy ', () => {
    const stack = new cdk.Stack();
    deployStack(stack);
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "dynamodb:BatchGetItem",
                        "dynamodb:GetRecords",
                        "dynamodb:GetShardIterator",
                        "dynamodb:Query",
                        "dynamodb:GetItem",
                        "dynamodb:Scan",
                        "dynamodb:ConditionCheckItem",
                        "dynamodb:BatchWriteItem",
                        "dynamodb:PutItem",
                        "dynamodb:UpdateItem",
                        "dynamodb:DeleteItem"
                    ],
                    Effect: "Allow",
                    Resource: [
                        {
                            "Fn::GetAtt": [
                                "testiotlambdadynamodbstackLambdaToDynamoDBDynamoTableE17E5733",
                                "Arn"
                            ]
                        },
                        {
                            Ref: "AWS::NoValue"
                        }
                    ]
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployStack(stack);
    expect(construct.lambdaFunction !== null);
    expect(construct.dynamoTable !== null);
    expect(construct.iotTopicRule !== null);
});
test('check exception for Missing existingObj from props for deploy = false', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "Processing of DTC messages from the AWS Connected Vehicle Solution.",
                sql: "SELECT * FROM 'connectedcar/dtc/#'",
                actions: []
            }
        }
    };
    try {
        new lib_1.IotToLambdaToDynamoDB(stack, 'test-iot-lambda-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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