"""
Management node

This model provides data gap-filled data from cycles in the form of a list of management nodes
(https://www.hestia.earth/schema/Management).

It includes products of type crop, forage, landCover (gap-filled with a value of 100) and practices of type waterRegime,
tillage, cropResidueManagement and landUseManagement.

All values are copied from the source node, except for crop and forage terms in which case the dates are copied from the
cycle.

When nodes are chronologically consecutive with "% area" or "boolean" units and the same term and value, they are
condensed into a single node to aid readability.
"""
from functools import reduce
from hestia_earth.schema import TermTermType, SiteSiteType
from hestia_earth.utils.model import filter_list_term_type
from hestia_earth.utils.tools import safe_parse_float, flatten
from hestia_earth.utils.blank_node import get_node_value

from hestia_earth.models.log import logRequirements, logShouldRun, log_as_table
from hestia_earth.models.utils import _include, _omit
from hestia_earth.models.utils.management import _new_management
from hestia_earth.models.utils.term import get_lookup_value
from hestia_earth.models.utils.blank_node import condense_nodes
from hestia_earth.models.utils.crop import get_landCover_term_id
from hestia_earth.models.utils.site import (
    related_cycles, get_land_cover_term_id as get_landCover_term_id_from_site_type
)
from . import MODEL

REQUIREMENTS = {
    "Site": {
        "related": {
            "Cycle": [{
                "@type": "Cycle",
                "startDate": "",
                "endDate": "",
                "products": [
                    {
                        "@type": "Product",
                        "term.termType": ["crop", "forage", "landCover"]
                    }
                ],
                "practices": [
                    {
                        "term.termType": [
                            "waterRegime",
                            "tillage",
                            "cropResidueManagement",
                            "landUseManagement",
                            "system"
                        ],
                        "value": ""
                    }
                ],
                "inputs": [
                    {
                        "@type": "Input",
                        "term.termType": [
                            "inorganicFertiliser",
                            "organicFertiliser",
                            "soilAmendment"
                        ]
                    }
                ]
            }]
        }
    }
}
RETURNS = {
    "Management": [{
        "@type": "Management",
        "term.termType": [
            "landCover", "waterRegime", "tillage", "cropResidueManagement", "landUseManagement", "system"
        ],
        "value": "",
        "endDate": "",
        "startDate": ""
    }]
}
LOOKUPS = {
    "crop": ["landCoverTermId"],
    "forage": ["landCoverTermId"],
    "inorganicFertiliser": "nitrogenContent",
    "organicFertiliser": "ANIMAL_MANURE",
    "soilAmendment": "PRACTICE_INCREASING_C_INPUT",
    "landUseManagement": "GAP_FILL_TO_MANAGEMENT",
    "property": "GAP_FILL_TO_MANAGEMENT"
}
MODEL_KEY = 'management'

_ANIMAL_MANURE_USED_TERM_ID = "animalManureUsed"
_INORGANIC_NITROGEN_FERTILISER_USED_TERM_ID = "inorganicNitrogenFertiliserUsed"
_ORGANIC_FERTILISER_USED_TERM_ID = "organicFertiliserUsed"
_AMENDMENT_INCREASING_C_USED_TERM_ID = "amendmentIncreasingSoilCarbonUsed"
_INPUT_RULES = {
    TermTermType.INORGANICFERTILISER.value: (
        (
            TermTermType.INORGANICFERTILISER.value,  # Lookup column
            lambda x: safe_parse_float(x) > 0,  # Condition
            _INORGANIC_NITROGEN_FERTILISER_USED_TERM_ID  # New term.
        ),
    ),
    TermTermType.SOILAMENDMENT.value: (
        (
            TermTermType.SOILAMENDMENT.value,
            lambda x: bool(x) is True,
            _AMENDMENT_INCREASING_C_USED_TERM_ID
        ),
    ),
    TermTermType.ORGANICFERTILISER.value: (
        (
            TermTermType.SOILAMENDMENT.value,
            lambda x: bool(x) is True,
            _ORGANIC_FERTILISER_USED_TERM_ID
        ),
        (
            TermTermType.ORGANICFERTILISER.value,
            lambda x: bool(x) is True,
            _ANIMAL_MANURE_USED_TERM_ID
        )
    )
}
_SKIP_LAND_COVER_SITE_TYPES = [
    SiteSiteType.CROPLAND.value
]


def management(data: dict):
    node = _new_management(data.get('id'))
    node['value'] = data['value']
    node['endDate'] = data['endDate']
    if data.get('startDate'):
        node['startDate'] = data['startDate']
    if data.get('properties'):
        node['properties'] = data['properties']
    return node


def _should_gap_fill(term: dict):
    value = get_lookup_value(lookup_term=term, column='GAP_FILL_TO_MANAGEMENT')
    return bool(value)


def _filter_properties(blank_node: dict):
    properties = list(filter(lambda p: _should_gap_fill(p.get('term', {})), blank_node.get('properties', [])))
    return _omit(blank_node, ['properties']) | ({'properties': properties} if properties else {})


def _map_to_value(value: dict):
    return {
        'id': value.get('term', {}).get('@id'),
        'value': value.get('value'),
        'startDate': value.get('startDate'),
        'endDate': value.get('endDate'),
        'properties': value.get('properties')
    }


def _extract_node_value(node: dict) -> dict:
    return node | {'value': get_node_value(node)}


def _copy_item_if_exists(source: dict, keys: list[str] = None, dest: dict = None) -> dict:
    return reduce(lambda p, c: p | ({c: source[c]} if source.get(c) else {}), keys or [], dest or {})


def _get_landCover_term_id(product: dict) -> str:
    term = product.get('term', {})
    return get_landCover_term_id(term, model=MODEL, term=term.get('@id'), model_key=MODEL_KEY)


def _get_relevant_items(cycle: dict, item_name: str, relevant_terms: list):
    """
    Get items from the list of cycles with any of the relevant terms.
    Also adds dates from Cycle.
    """
    return [
        _include(cycle, ["startDate", "endDate"]) | item
        for item in filter_list_term_type(cycle.get(item_name, []), relevant_terms)
    ]


def _process_rule(node: dict, term: dict) -> list:
    relevant_terms = []
    for column, condition, new_term in _INPUT_RULES[term.get('termType')]:
        lookup_result = get_lookup_value(term, LOOKUPS[column], model=MODEL, term=term.get('@id'), model_key=MODEL_KEY)

        if condition(lookup_result):
            relevant_terms.append(node | {'id': new_term})

    return relevant_terms


def _run_from_inputs(site: dict, cycle: dict) -> list:
    inputs = flatten([
        _process_rule(node={
            'value': True,
            'startDate': cycle.get('startDate'),
            'endDate': cycle.get('endDate')
        }, term=input.get('term'))
        for input in cycle.get('inputs', [])
        if input.get('term', {}).get('termType') in _INPUT_RULES
    ])
    return inputs


def _run_from_siteType(site: dict, cycle: dict):
    site_type = site.get('siteType')
    site_type_id = get_landCover_term_id_from_site_type(site_type) if site_type not in _SKIP_LAND_COVER_SITE_TYPES \
        else None

    should_run = all([site_type_id])
    return [{
        'id': site_type_id,
        'value': 100,
        'startDate': cycle.get('startDate'),
        'endDate': cycle.get('endDate')
    }] if should_run else []


def _run_products(cycle: dict, products: list, total_products: int = None, use_cycle_dates: bool = False):
    default_dates = _include(cycle, ["startDate", "endDate"])
    return [
        _map_to_value(default_dates | _copy_item_if_exists(
            source=product,
            keys=['properties', 'startDate', 'endDate'],
            dest={
                "term": {'@id': _get_landCover_term_id(product)},
                "value": round(100 / (total_products or len(products)), 2)
            }
        ) | (
            default_dates if use_cycle_dates else {}
        ))
        for product in products
    ]


def _run_from_landCover(cycle: dict, crop_forage_products: list):
    """
    Copy landCover items, and include crop/forage landCover items with properties to count in ratio.
    """
    products = [
        _map_to_value(_extract_node_value(
            _include(
                value=product,
                keys=["term", "value", "startDate", "endDate", "properties"]
            )
        )) for product in _get_relevant_items(
            cycle=cycle,
            item_name="products",
            relevant_terms=[TermTermType.LANDCOVER]
        )
    ]
    return products + _run_products(
        cycle,
        crop_forage_products,
        total_products=len(crop_forage_products) + len(products),
        use_cycle_dates=True
    )


def _run_from_crop_forage(cycle: dict):
    products = _get_relevant_items(
        cycle=cycle,
        item_name="products",
        relevant_terms=[TermTermType.CROP, TermTermType.FORAGE]
    )
    # only take products with a matching landCover term
    products = list(filter(_get_landCover_term_id, products))
    # remove any properties that should not get gap-filled
    products = list(map(_filter_properties, products))

    # split products with properties and those without
    products_with_gap_filled_props = [p for p in products if p.get('properties')]
    products_without_gap_filled_props = [p for p in products if not p.get('properties')]

    return _run_from_landCover(
        cycle=cycle,
        crop_forage_products=products_with_gap_filled_props
    ) + _run_products(cycle, products_without_gap_filled_props, use_cycle_dates=False)


def _should_run_practice(practice: dict):
    """
    Include only landUseManagement practices where GAP_FILL_TO_MANAGEMENT = True
    """
    term = practice.get('term', {})
    return term.get('termType') != TermTermType.LANDUSEMANAGEMENT.value or _should_gap_fill(term)


def _run_from_practices(cycle: dict):
    practices = [
        _extract_node_value(
            _include(
                value=practice,
                keys=["term", "value", "startDate", "endDate"]
            )
        ) for practice in _get_relevant_items(
            cycle=cycle,
            item_name="practices",
            relevant_terms=[
                TermTermType.WATERREGIME,
                TermTermType.TILLAGE,
                TermTermType.CROPRESIDUEMANAGEMENT,
                TermTermType.LANDUSEMANAGEMENT,
                TermTermType.SYSTEM
            ]
        )
    ]
    practices = list(map(_map_to_value, filter(_should_run_practice, practices)))
    return practices


def _run_cycle(site: dict, cycle: dict):
    inputs = _run_from_inputs(site, cycle)
    products = _run_from_crop_forage(cycle)
    site_types = _run_from_siteType(site, cycle)
    practices = _run_from_practices(cycle)
    return [
        node | {'cycle-id': cycle.get('@id')}
        for node in inputs + products + site_types + practices
    ]


def run(site: dict):
    cycles = related_cycles(site)
    nodes = flatten([_run_cycle(site, cycle) for cycle in cycles])

    # group nodes with same `id` to display as a single log per node
    grouped_nodes = reduce(lambda p, c: p | {c['id']: p.get(c['id'], []) + [c]}, nodes, {})
    for id, values in grouped_nodes.items():
        logRequirements(
            site,
            model=MODEL,
            term=id,
            model_key=MODEL_KEY,
            details=log_as_table(values, ignore_keys=['id', 'properties']),
        )
        logShouldRun(site, MODEL, id, True, model_key=MODEL_KEY)

    management_nodes = condense_nodes(list(map(management, nodes)))
    return management_nodes
