import uuid
from dataclasses import asdict, dataclass
from typing import Iterable, Mapping

from django.utils.translation import gettext as _

from arches.app.datatypes.base import BaseDataType
from arches.app.models.models import Node
from arches.app.models.graph import GraphValidationError

from arches_controlled_lists.models import ListItem


@dataclass(kw_only=True)
class ReferenceLabel:
    id: uuid.UUID
    value: str
    language_id: str
    valuetype_id: str
    list_item_id: uuid.UUID


@dataclass(kw_only=True)
class Reference:
    uri: str
    labels: list[ReferenceLabel]
    list_id: uuid.UUID


class ReferenceDataType(BaseDataType):
    def to_python(
        self, value: Iterable[Mapping] | None, **kwargs
    ) -> list[Reference] | None:
        if not value:
            return None

        references = []
        for reference in value:
            incoming_args = {**reference}
            if labels := incoming_args.get("labels"):
                incoming_args["labels"] = [
                    ReferenceLabel(**label) for label in incoming_args["labels"]
                ]
            elif labels == []:
                incoming_args.pop("labels")
            references.append(Reference(**incoming_args))

        return references or None

    def serialize(self, value):
        """
        Unlike to_json() which is concerned with calculating display values, this
        method simply serializes all the information held in the Reference(s).
        """
        if value is None:
            return None
        return [
            asdict(reference) if isinstance(reference, Reference) else {**reference}
            for reference in value
        ]

    def validate(
        self,
        value,
        row_number=None,
        source="",
        node=None,
        nodeid=None,
        strict=False,
        **kwargs,
    ):
        try:
            parsed = self.to_python(value)
            self.validate_pref_labels(parsed)
            self.validate_list_item_consistency(parsed)
            self.validate_multivalue(parsed, node, nodeid)
        except Exception as e:
            return [self.transform_exception(e)]
        return []

    def validate_pref_labels(self, references: list[Reference] | None):
        if not references:
            return
        for reference in references:
            pref_label_languages = [
                label.language_id
                for label in reference.labels
                if label.valuetype_id == "prefLabel"
            ]
            if len(set(pref_label_languages)) < len(pref_label_languages):
                msg = _("A reference can have only one prefLabel per language")
                raise ValueError(msg)

    def validate_list_item_consistency(self, references: list[Reference] | None):
        if not references:
            return
        for reference in references:
            list_item_ids = {ref.list_item_id for ref in reference.labels}
            if len(list_item_ids) != 1:
                msg = _("Found multiple list items among labels: {reference}")
                raise ValueError(msg)

    def validate_multivalue(self, parsed: list[Reference] | None, node, nodeid):
        if not parsed:
            return
        if not node:
            if not nodeid:
                raise ValueError
            try:
                node = Node.objects.get(nodeid=nodeid)
            except Node.DoesNotExist:
                return
        if not node.config.get("multiValue") and len(parsed) > 1:
            raise ValueError(_("This node does not allow multiple references."))

    @staticmethod
    def transform_exception(e):
        message = _("Unknown error")
        if isinstance(e, TypeError) and e.args:
            # Localize the error raised by the dataclass constructor.
            if "__init__() missing" in e.args[0]:
                message = _("Missing required value(s): {}").format(
                    e.args[0].split(": ")[-1]
                )
            elif "unexpected keyword argument" in e.args[0]:
                message = _("Unexpected value: {}").format(
                    e.args[0].split("argument ")[-1]
                )
        elif isinstance(e, ValueError) and e.args:
            message = e.args[0]
        return {
            "type": "ERROR",
            "message": message,
            "title": _("Invalid Reference Datatype Value"),
        }

    def transform_value_for_tile(self, value, **kwargs):
        if value is None:
            return None
        if not isinstance(value, list):
            value = [value]

        # Pre-process to discard display values generated by to_json().
        pre_processed_values = []
        for single_value in value:
            if isinstance(single_value, dict) and (
                list_item_id := single_value.get("list_item_id")
            ):
                pre_processed_values.append(list_item_id)
            else:
                pre_processed_values.append(single_value)

        final_tile_values = []
        for single_value in pre_processed_values:
            found_item: ListItem | None = None
            match single_value:
                case Reference():
                    found_item = ListItem.objects.filter(
                        pk=single_value.labels[0].list_item_id
                    ).first()
                case uuid.UUID():
                    found_item = ListItem.objects.filter(pk=list_item_id).first()
                case str():
                    try:
                        list_item_id = uuid.UUID(single_value)
                    except ValueError:
                        list_id = kwargs.get("controlledList")
                        found_item = self.lookup_listitem_from_label(
                            single_value, list_id
                        )
                    else:
                        found_item = ListItem.objects.filter(pk=list_item_id).first()
                case _:
                    raise TypeError(type(single_value))

            if found_item:
                final_tile_values.append(found_item.build_tile_value())

        return final_tile_values

    def lookup_listitem_from_label(self, value, list_id):
        if not value or not list_id:
            return None
        return (
            ListItem.objects.filter(list_id=list_id, list_item_values__value=value)
            .order_by("sortorder")
            .first()
        )

    def clean(self, tile, nodeid):
        super().clean(tile, nodeid)
        if tile.data[nodeid] == []:
            tile.data[nodeid] = None

    def transform_export_values(self, value, *args, **kwargs):
        return ",".join(value)

    def get_display_value(self, tile, node, **kwargs):
        requested_language = kwargs.pop("language", None)
        node_data = self.get_tile_data(tile)
        value = node_data.get(str(node.pk), None)
        references = self.to_python(value)
        if not references:
            return _("(Empty)")
        else:
            best_labels = [
                ListItem.find_best_label_from_set(reference.labels, requested_language)
                for reference in references
            ]
            return ", ".join(best_labels)

    def get_interchange_value(self, value, **kwargs):
        """
        Expects tile representation of reference datatype:
        [
            {
                "uri": "",
                "labels": [
                    {
                        "id": "uuid",
                        "value": "label",
                        "language_id": "en",
                        "valuetype_id": "prefLabel",
                        "list_item_id": "uuid"
                    }
                ],
                "list_id": "uuid"
            }
        ]
        Returns list item transformed for use in dropdown pickers:
        [
            {
                "list_item_id": "uuid",
                "uri": "",
                "list_item_values": [
                    {
                        "id": "uuid",
                        "value": "Parent",
                        "language_id": "en",
                        "valuetype_id": "prefLabel",
                        "list_item_id": "uuid"
                    }
                ],
                "display_value": "",
                "sortorder": 0,
                "children": [
                    {
                        "list_item_id": "uuid",
                        "uri": "",
                        "list_item_values": [
                            {
                                "id": "uuid",
                                "value": "Child",
                                "language_id": "en",
                                "valuetype_id": "prefLabel",
                                "list_item_id": "uuid"
                            }
                        ],
                        "display_value": "",
                        "sortorder": 0,
                        "children": []
                    }
                ]
            }
        ]
        """
        list_item_ids = []
        if value:
            for default_val in value:
                list_item_ids.append(default_val["labels"][0]["list_item_id"])
        else:
            return None
        items = ListItem.objects.filter(id__in=list_item_ids)
        transformed_items = [item.build_select_option() for item in items]

        return transformed_items

    def collects_multiple_values(self):
        return True

    def default_es_mapping(self):
        return {
            "properties": {
                "uri": {"type": "keyword"},
                "id": {"type": "keyword"},
                "labels": {
                    "properties": {},
                },
            }
        }

    def validate_node(self, node):
        try:
            uuid.UUID(node.config["controlledList"])
        except (TypeError, KeyError):
            raise GraphValidationError(
                _("A reference datatype node requires a controlled list")
            )

    def append_to_document(self, document, nodevalue, nodeid, tile, provisional=False):
        if "references" not in document:
            document["references"] = []
        for reference in self.get_nodevalues(nodevalue):
            for label in reference["labels"]:
                document["references"].append(
                    {
                        "id": label["id"],
                        "uri": reference["uri"],
                        "list_id": reference["list_id"],
                        "nodegroup_id": tile.nodegroup_id,
                        "provisional": provisional,
                    }
                )
                document["strings"].append(
                    {
                        "string": label["value"],
                        "nodegroup_id": tile.nodegroup_id,
                        "provisional": provisional,
                    }
                )
