"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenizedStringFragments = void 0;
const token_1 = require("./token");
/**
 * Fragments of a concatenated string containing stringified Tokens
 */
class TokenizedStringFragments {
    constructor() {
        this.fragments = new Array();
    }
    get firstToken() {
        const first = this.fragments[0];
        if (first.type === 'token') {
            return first.token;
        }
        return undefined;
    }
    get firstValue() {
        return fragmentValue(this.fragments[0]);
    }
    get length() {
        return this.fragments.length;
    }
    addLiteral(lit) {
        this.fragments.push({ type: 'literal', lit });
    }
    addToken(token) {
        this.fragments.push({ type: 'token', token });
    }
    addIntrinsic(value) {
        this.fragments.push({ type: 'intrinsic', value });
    }
    /**
     * Return all Tokens from this string
     */
    get tokens() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === 'token') {
                ret.push(f.token);
            }
        }
        return ret;
    }
    /**
     * Apply a transformation function to all tokens in the string
     */
    mapTokens(mapper) {
        const ret = new TokenizedStringFragments();
        for (const f of this.fragments) {
            switch (f.type) {
                case 'literal':
                    ret.addLiteral(f.lit);
                    break;
                case 'token':
                    const mapped = mapper.mapToken(f.token);
                    if (token_1.isResolvableObject(mapped)) {
                        ret.addToken(mapped);
                    }
                    else {
                        ret.addIntrinsic(mapped);
                    }
                    break;
                case 'intrinsic':
                    ret.addIntrinsic(f.value);
                    break;
            }
        }
        return ret;
    }
    /**
     * Combine the string fragments using the given joiner.
     *
     * If there are any
     */
    join(concat) {
        if (this.fragments.length === 0) {
            return concat.join(undefined, undefined);
        }
        if (this.fragments.length === 1) {
            return this.firstValue;
        }
        const values = this.fragments.map(fragmentValue);
        while (values.length > 1) {
            const prefix = values.splice(0, 2);
            values.splice(0, 0, concat.join(prefix[0], prefix[1]));
        }
        return values[0];
    }
}
exports.TokenizedStringFragments = TokenizedStringFragments;
/**
 * Resolve the value from a single fragment
 *
 * If the fragment is a Token, return the string encoding of the Token.
 */
function fragmentValue(fragment) {
    switch (fragment.type) {
        case 'literal': return fragment.lit;
        case 'token': return fragment.token.toString();
        case 'intrinsic': return fragment.value;
    }
}
//# sourceMappingURL=data:application/json;base64,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