"use strict";
const child_process = require("child_process");
const sinon = require("sinon");
const lib_1 = require("../lib");
module.exports = {
    'tearDown'(callback) {
        sinon.restore();
        callback();
    },
    'bundling with image from registry'(test) {
        const spawnSyncStub = sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        image._run({
            command: ['cool', 'command'],
            environment: {
                VAR1: 'value1',
                VAR2: 'value2',
            },
            volumes: [{ hostPath: '/host-path', containerPath: '/container-path' }],
            workingDirectory: '/working-directory',
            user: 'user:group',
        });
        test.ok(spawnSyncStub.calledWith('docker', [
            'run', '--rm',
            '-u', 'user:group',
            '-v', '/host-path:/container-path:delegated',
            '--env', 'VAR1=value1',
            '--env', 'VAR2=value2',
            '-w', '/working-directory',
            'alpine',
            'cool', 'command',
        ]));
        test.done();
    },
    'bundling with image from asset'(test) {
        const imageId = 'abcdef123456';
        const spawnSyncStub = sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from(`sha256:${imageId}`),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromAsset('docker-path', {
            buildArgs: {
                TEST_ARG: 'cdk-test',
            },
        });
        image._run();
        test.ok(spawnSyncStub.firstCall.calledWith('docker', [
            'build', '-q',
            '--build-arg', 'TEST_ARG=cdk-test',
            'docker-path',
        ]));
        test.ok(spawnSyncStub.secondCall.calledWith('docker', [
            'run', '--rm',
            imageId,
        ]));
        test.done();
    },
    'throws if image id cannot be extracted from build output'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        test.throws(() => lib_1.BundlingDockerImage.fromAsset('docker-path'), /Failed to extract image ID from Docker build output/);
        test.done();
    },
    'throws in case of spawnSync error'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: 0,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
            error: new Error('UnknownError'),
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        test.throws(() => image._run(), /UnknownError/);
        test.done();
    },
    'throws if status is not 0'(test) {
        sinon.stub(child_process, 'spawnSync').returns({
            status: -1,
            stderr: Buffer.from('stderr'),
            stdout: Buffer.from('stdout'),
            pid: 123,
            output: ['stdout', 'stderr'],
            signal: null,
        });
        const image = lib_1.BundlingDockerImage.fromRegistry('alpine');
        test.throws(() => image._run(), /\[Status -1\]/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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