"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiPipelineLib = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codebuild = require("@aws-cdk/aws-codebuild");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const codepipeline_actions = require("@aws-cdk/aws-codepipeline-actions");
const ec2 = require("@aws-cdk/aws-ec2");
const events = require("@aws-cdk/aws-events");
const event_targets = require("@aws-cdk/aws-events-targets");
const iam = require("@aws-cdk/aws-iam");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const kms = require("@aws-cdk/aws-kms");
const sns = require("@aws-cdk/aws-sns");
const cdk = require("@aws-cdk/core");
const cleaner_1 = require("./cleaner");
const component_builder_1 = require("./component_builder");
const image_recipe_version_bump_1 = require("./image_recipe_version_bump");
const slack_notification_1 = require("./slack_notification");
const source_action_builder_1 = require("./source_action_builder");
const ssmupdate = require("./ssm_update");
const utils = require("./utils");
/**
 * Construct for creating a Codepipeline, EC2 Image builder pipeline from 1 pipeline configuration.
 *
 * @stability stable
 */
class AmiPipelineLib extends cdk.Construct {
    /**
     * Constructor.
     *
     * @stability stable
     */
    constructor(scope, id, pipelineConfig, componentDepsConfig, componentBuilder, optionalParams) {
        super(scope, id);
        this.id = id;
        this.pipelineConfig = pipelineConfig;
        this.pipelineConfig.name = this.id;
        this.componentDepsConfig = componentDepsConfig;
        this.slackConfig = {
            channel: optionalParams.channel,
            slackWebhookUrl: optionalParams.slackWebhookUrl,
            username: optionalParams.username,
        };
        this.componentBuilder = componentBuilder !== null && componentBuilder !== void 0 ? componentBuilder : new component_builder_1.ComponentBuilder(this, componentDepsConfig);
        this.sourceActionBuilder = new source_action_builder_1.SourceActionBuilder(this, pipelineConfig.sources, this.id);
        this.extraParameters = optionalParams.extraParams;
        this.createImagebuilderPipeline();
        this.createCodepipelineProject();
        this.createScheduledTask();
        this.createCleanerTask();
        if (this.topic) {
            new ssmupdate.SsmUpdateConstruct(this, 'SSMUpdate', this.topic, this.pipelineConfig);
            if (this.slackConfig && this.slackConfig.channel && this.slackConfig.slackWebhookUrl && this.slackConfig.username) {
                new slack_notification_1.SlackNotification(this, 'SlackNotification', this.topic, this.slackConfig, `${this.id}Recipe`);
            }
        }
    }
    /**
     * @stability stable
     */
    createCleanerTask() {
        if (this.pipelineConfig.cleaning_schedule) {
            new cleaner_1.Cleaner(this, 'Cleaner', this.pipelineConfig.cleaning_schedule, this.id);
        }
    }
    /**
     * @stability stable
     */
    createScheduledTask() {
        if (this.codepipeline && this.pipelineConfig.schedule) {
            const pipelineTarget = new event_targets.CodePipeline(this.codepipeline);
            new events.Rule(this, 'ScheduleRule', {
                schedule: events.Schedule.expression(this.pipelineConfig.schedule),
                targets: [pipelineTarget],
            });
        }
    }
    /**
     * @stability stable
     */
    createImagebuilderPipeline() {
        let image_id;
        let parentImage;
        if ('disk_size' in this.pipelineConfig && this.pipelineConfig.disk_size) {
            this.diskSize = this.pipelineConfig.disk_size;
        }
        if ('image_id' in this.pipelineConfig && this.pipelineConfig.image_id) {
            image_id = this.pipelineConfig.image_id;
        }
        else {
            switch (this.pipelineConfig.parent_image) {
                case 'AmazonLinux2':
                    parentImage = ec2.MachineImage.latestAmazonLinux({
                        generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
                        edition: ec2.AmazonLinuxEdition.STANDARD,
                        virtualization: ec2.AmazonLinuxVirt.HVM,
                        storage: ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
                        cpuType: ec2.AmazonLinuxCpuType.X86_64,
                    });
                    break;
                default:
                    parentImage = ec2.MachineImage.lookup(this.getLookupCriteria(this.pipelineConfig.parent_image));
                    break;
            }
        }
        // Use a custom block device mapping if encryption is required.
        let blockDeviceMappings = this.createBlockDeviceMapping();
        this.recipe = new imagebuilder.CfnImageRecipe(this, 'ImageRecipe', {
            name: `${this.id}Recipe`,
            parentImage: image_id ? image_id : parentImage ? parentImage.getImage(this).imageId : null,
            version: this.getNextRecipeVersion(`${this.id}Recipe`),
            components: this.getComponents(this.pipelineConfig.recipe.components),
            blockDeviceMappings: blockDeviceMappings,
            tags: {
                ShortName: `${this.id}Recipe`,
            },
        });
        const builderRole = new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilderECRContainerBuilds'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('EC2InstanceProfileForImageBuilder'),
            ],
            inlinePolicies: this.getInlinePolicies(this.pipelineConfig),
        });
        const instanceProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [builderRole.roleName],
        });
        this.topic = new sns.Topic(this, 'AmiPipelineTopic');
        this.infrastructure = new imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: `${this.id} - EC2 IB Infrastructure`,
            instanceProfileName: instanceProfile.ref,
            snsTopicArn: this.topic.topicArn,
            instanceTypes: this.pipelineConfig.instance_type ? [this.pipelineConfig.instance_type] : undefined,
            terminateInstanceOnFailure: !('terminate_on_failure' in this.pipelineConfig) || this.pipelineConfig.terminate_on_failure,
            subnetId: this.pipelineConfig.subnet_id,
            securityGroupIds: this.pipelineConfig.security_group_ids ? this.pipelineConfig.security_group_ids.split(',') : undefined,
        });
        this.distributionConfig = this.createDistributionConfig();
        this.imagePipeline = new imagebuilder.CfnImagePipeline(this, 'AmiPipeline', {
            name: `${this.id} - AMI Pipeline`,
            imageRecipeArn: this.recipe.ref,
            infrastructureConfigurationArn: this.infrastructure.ref,
            distributionConfigurationArn: this.distributionConfig ? this.distributionConfig.ref : undefined,
        });
    }
    getComponents(components) {
        let compMap = this.componentBuilder.componentDependenciesMap;
        if (this.extraParameters) {
            for (let param of this.extraParameters) {
                let component = components.find((c) => c.name === param.componentName);
                if (component) {
                    component.parameters = component.parameters || {};
                    component.parameters = {
                        ...component.parameters,
                        ...param.parameters,
                    };
                }
            }
        }
        let compList = components.map((c) => ({
            componentArn: compMap[c.name].ref,
            parameters: 'parameters' in c ? Object.keys(c.parameters).map((k) => ({
                name: k,
                value: [c.parameters[k]],
            })) : [],
        }));
        for (let c of compList) {
            if (c.parameters && c.parameters.length === 0) {
                delete c.parameters;
            }
        }
        return compList;
    }
    createDistributionConfig() {
        // Get current region
        const region = cdk.Stack.of(this).region;
        const accountId = cdk.Stack.of(this).account;
        const amiTags = {
            Name: this.id,
            CreatedBy: 'EC2 Image Builder',
            Ec2ImageBuilderArn: '{{imagebuilder:buildVersion}}',
            AutoClean: 'True',
        };
        let sharedWithDistributions = [];
        if (this.pipelineConfig.shared_with) {
            sharedWithDistributions = this.pipelineConfig.shared_with.map((s) => ({
                region: s.region,
                amiDistributionConfiguration: {
                    Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                    Description: `${this.id} Shared image`,
                    AmiTags: amiTags,
                    TargetAccountIds: s.account_ids,
                },
            }));
            // current region might already be included
            if (sharedWithDistributions.some(s => s.region === region)) {
                const distribution = sharedWithDistributions.find(s => s.region === region);
                //Push current account ID as distribute
                distribution === null || distribution === void 0 ? void 0 : distribution.amiDistributionConfiguration.TargetAccountIds.push(accountId);
            }
        }
        if (sharedWithDistributions.length == 0) {
            sharedWithDistributions = [{
                    region: region,
                    amiDistributionConfiguration: {
                        Name: `${this.id} - {{ imagebuilder:buildDate }}`,
                        Description: `${this.id} Shared image`,
                        AmiTags: amiTags,
                        TargetAccountIds: [accountId],
                    },
                }];
        }
        return new imagebuilder.CfnDistributionConfiguration(this, 'DistributionConfig', {
            name: `${this.id} - Distribution`,
            distributions: sharedWithDistributions,
        });
    }
    getInlinePolicies(pipelineConfig) {
        let result = {};
        const statements = pipelineConfig.policy_statements;
        if (statements) {
            for (let statement in statements) {
                result[statement] = new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            sid: statement,
                            actions: statements[statement].actions,
                            effect: iam.Effect.ALLOW,
                            resources: statements[statement].resources,
                        }),
                    ],
                });
            }
        }
        return result;
    }
    createBlockDeviceMapping() {
        const key = this.pipelineConfig.encrypted ? new kms.Key(this, 'PipelineEncryption', {
            description: `Image encryption for ${this.pipelineConfig.name}`,
            alias: `${utils.pascalize(this.pipelineConfig.name)}_key`,
        }) : undefined;
        this.ebsEncryptionKey = key;
        let blockDeviceMappings = [
            {
                deviceName: this.pipelineConfig.parent_image.includes('AmazonLinux') ? '/dev/xvda' : '/dev/sda1',
                ebs: {
                    encrypted: key ? true : false,
                    volumeSize: this.pipelineConfig.disk_size ? this.pipelineConfig.disk_size : 8,
                    volumeType: 'gp2',
                    kmsKeyId: key === null || key === void 0 ? void 0 : key.keyArn,
                },
            },
        ];
        return blockDeviceMappings;
    }
    /**
     * @stability stable
     */
    getNextRecipeVersion(recipeName) {
        const recipeBumpCustomResource = new image_recipe_version_bump_1.ImageRecipeVersionBump(this, 'ImageRecipeVersionBump', recipeName);
        return recipeBumpCustomResource.nextVersion;
    }
    /**
     * @stability stable
     */
    createCodepipelineProject() {
        var _b;
        if (this.imagePipeline) {
            // Create CodePipeline project
            let envVariables = {
                IMAGE_PIPELINE_ARN: { value: this.imagePipeline.ref },
                PIPELINE_NAME: { value: this.imagePipeline.name },
            };
            if (this.slackConfig && this.slackConfig.slackWebhookUrl && this.slackConfig.channel && this.slackConfig.username) {
                envVariables.SLACK_WEBHOOK_URL = { value: this.slackConfig.slackWebhookUrl };
                envVariables.SLACK_CHANNEL = { value: this.slackConfig.channel };
                envVariables.SLACK_USERNAME = { value: this.slackConfig.username };
            }
            let imagebuilderBuild = new codebuild.PipelineProject(this, 'AmiPipelineBuild', {
                buildSpec: codebuild.BuildSpec.fromObject({
                    version: '0.2',
                    phases: {
                        install: {
                            commands: [
                                'apt-get update',
                                'apt-get install awscli -y',
                            ],
                        },
                        pre_build: {
                            commands: [
                                'if [ -n "$SLACK_WEBHOOK_URL" ]; then curl -X POST -H \'Content-type: application/json\' --data "{\\"text\\":\\"$PIPELINE_NAME build has started.\\"}" $SLACK_WEBHOOK_URL; fi',
                            ],
                        },
                        build: {
                            commands: 'aws imagebuilder start-image-pipeline-execution --image-pipeline-arn $IMAGE_PIPELINE_ARN',
                        },
                    },
                }),
                environment: {
                    buildImage: codebuild.LinuxBuildImage.STANDARD_4_0,
                    environmentVariables: envVariables,
                },
            });
            imagebuilderBuild.addToRolePolicy(iam.PolicyStatement.fromJson({
                Sid: 'imagebuilderstart',
                Effect: 'Allow',
                Action: ['imagebuilder:StartImagePipelineExecution'],
                Resource: '*',
            }));
            const imageBuilderOutput = new codepipeline.Artifact('CdkBuildOutput');
            let sourceActions = this.sourceActionBuilder.createPipelineSources();
            const buildAction = new codepipeline_actions.CodeBuildAction({
                actionName: 'StartImageBuilder',
                project: imagebuilderBuild,
                input: (_b = sourceActions.find(a => a && a.sourceOutput)) === null || _b === void 0 ? void 0 : _b.sourceOutput,
                extraInputs: sourceActions.length > 1 ? sourceActions.slice(1).map(a => a.sourceOutput) : [],
                outputs: [imageBuilderOutput],
            });
            // Complete Pipeline Project
            this.codepipeline = new codepipeline.Pipeline(this, 'Pipeline', {
                pipelineName: this.pipelineConfig.name,
                restartExecutionOnUpdate: true,
                stages: [
                    {
                        stageName: 'Source',
                        actions: sourceActions.map(c => c.action).filter((c) => c != null),
                    },
                    {
                        stageName: 'Build',
                        actions: [buildAction],
                    },
                ],
            });
        }
    }
    /**
     * @stability stable
     */
    getLookupCriteria(parentImage) {
        switch (parentImage) {
            case 'Ubuntu1804':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-bionic-18.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'Ubuntu2004':
                return {
                    name: 'ubuntu/images/hvm-ssd/ubuntu-focal-20.04-amd64*',
                    owners: ['099720109477'],
                };
            case 'CentOS7':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['cvugziknvmxgqna9noibqnnsy'],
                    },
                };
            case 'CentOS8':
                return {
                    name: '*',
                    owners: ['aws-marketplace'],
                    filters: {
                        'product-code': ['47k9ia2igxpcce2bzo8u3kj03'],
                    },
                };
            default:
                return {
                    name: 'amzn-linux',
                };
        }
    }
}
exports.AmiPipelineLib = AmiPipelineLib;
_a = JSII_RTTI_SYMBOL_1;
AmiPipelineLib[_a] = { fqn: "halloumi-ami-pipelines.AmiPipelineLib", version: "0.0.28" };
//# sourceMappingURL=data:application/json;base64,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