"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComponentBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const imagebuilder = require("@aws-cdk/aws-imagebuilder");
const yaml = require("yaml");
const Validator = require("yaml-validator");
const component_version_bump_1 = require("./component_version_bump");
const index_1 = require("./index");
const utils = require("./utils");
/**
 * @stability stable
 */
class ComponentBuilder {
    /**
     * @stability stable
     */
    constructor(scope, componentDefs) {
        /**
         * @stability stable
         */
        this.cacheDir = 'component_cache';
        this.componentDeps = componentDefs;
        this.scope = scope;
        if (!fs.existsSync(this.cacheDir)) {
            fs.mkdirSync(this.cacheDir);
        }
        this.componentDependenciesMap = this.createComponentDependenciesMap();
    }
    createComponentDependenciesMap() {
        const components = this.componentDeps.map(c => this.createComponent(c)).filter(c => c);
        return components.reduce((a, el) => ({
            ...a,
            [el.name]: el,
        }), {}); // remove the prefix
    }
    createComponent(componentConfig) {
        let componentYaml;
        switch (componentConfig.type) {
            case 'local':
                componentYaml = this.assembleComponent(componentConfig, 'local_components');
                break;
            case 'git':
                componentYaml = this.assembleComponent(componentConfig, this.cacheDir);
                break;
            case 'aws_arn':
                return new index_1.ArnComponentRef(componentConfig.arn, componentConfig.name);
            default:
                componentYaml = '';
        }
        if (componentConfig && componentYaml) {
            return new imagebuilder.CfnComponent(this.scope, `${utils.pascalize(componentConfig.name)}Component`, componentYaml);
        }
        return undefined;
    }
    assembleComponent(componentConfig, basedir) {
        let componentPath = componentConfig.name;
        if (componentConfig.path) {
            componentPath = `${componentPath}/${componentConfig.path}`;
        }
        if (this.checkComponentYaml(basedir, componentPath)) {
            // Get metadata YAML
            const metadata = fs.readFileSync(path.join(basedir, componentPath, 'metadata.yaml'), 'utf-8');
            const metayaml = yaml.parse(metadata);
            // Get base component.yaml
            const baseComponent = fs.readFileSync(path.join(basedir, componentPath, 'component.yaml'), 'utf-8');
            const baseYaml = yaml.parse(baseComponent);
            let assembledYaml = baseYaml;
            if (metayaml.distribution_configs) {
                const distributionconfigcomp = metayaml.distribution_configs[componentConfig.platform];
                // if there is a platform specific component yaml specified, load it and add steps to the assembled component
                if (distributionconfigcomp) {
                    const componentExtension = fs.readFileSync(path.join(basedir, componentPath, distributionconfigcomp), 'utf-8');
                    const componentExtYaml = yaml.parse(componentExtension);
                    const phase_extensions = componentExtYaml.phase_extensions;
                    // Add phase extensions to base component
                    this.addPhaseExtensions(phase_extensions, assembledYaml);
                }
            }
            this.addDependencyConstants(componentConfig, assembledYaml);
            const name = componentConfig.name;
            const componentVersionBump = new component_version_bump_1.ComponentVersionBump(this.scope, `${utils.pascalize(componentConfig.name)}VersionBump`, name);
            return {
                name: name,
                platform: metayaml.platform,
                version: componentVersionBump.nextVersion,
                data: yaml.stringify(assembledYaml),
                tags: {
                    ShortName: utils.pascalize(componentConfig.name),
                },
            };
        }
    }
    addDependencyConstants(componentConfig, assembledYaml) {
        if ('constants' in componentConfig) {
            let constantsDict = componentConfig.constants;
            let compConstants = Object.keys(constantsDict).map((k) => {
                let constantObj = {};
                constantObj[k] = {
                    type: 'string',
                    value: constantsDict[k],
                };
                return constantObj;
            });
            if ('constants' in assembledYaml) {
                // Override the keys from the component itself with the ones from the component dependencies.
                let compConstantsKeys = Object.keys(constantsDict);
                assembledYaml.constants = assembledYaml.constants.filter(function (c) {
                    // See if their is an intersection between the keys from component dependency and
                    // the ones defined in the original component.yaml file and remove those from component.yaml file.
                    return Object.keys(c).filter(d => compConstantsKeys.includes(d)).length == 0;
                });
                assembledYaml.constants.splice(assembledYaml.constants.length, 0, ...compConstants);
            }
            else {
                assembledYaml.constants = compConstants;
            }
        }
    }
    addPhaseExtensions(phase_extensions, assembledYaml) {
        for (let phase_extension of phase_extensions) {
            // For prebuild phases, copy the steps before the build step
            if (phase_extension.name === 'prebuild') {
                const buildphase = assembledYaml.phases.find((el) => el.name === 'build');
                if (buildphase) {
                    // Insert the prebuild before the build steps.
                    buildphase.steps.splice(0, 0, ...phase_extension.steps);
                }
                else {
                    // In case the original componen yaml doesn't include a build phase,
                    // copy the phase from the extension directly in the assembled YAML
                    const copied_phase = { ...phase_extension };
                    copied_phase.name = 'build';
                    assembledYaml.phases.push(copied_phase);
                }
            }
            else { // Else just copy the build phase.
                const extphase = assembledYaml.phases.find((el) => el.name === phase_extension.name);
                if (extphase) {
                    extphase.steps.push(...phase_extension.steps);
                }
            }
        }
    }
    checkComponentYaml(baseDir, componentPath) {
        const metadatafile = path.join(baseDir, componentPath, 'metadata.yaml');
        const componentfile = path.join(baseDir, componentPath, 'component.yaml');
        const metadataExists = fs.existsSync(metadatafile);
        const componentYamlExists = fs.existsSync(componentfile);
        const yamlValidator = new Validator({});
        yamlValidator.validate([
            metadatafile,
            componentfile,
        ]);
        return metadataExists && componentYamlExists && yamlValidator.report() == 0;
    }
}
exports.ComponentBuilder = ComponentBuilder;
_a = JSII_RTTI_SYMBOL_1;
ComponentBuilder[_a] = { fqn: "halloumi-ami-pipelines.ComponentBuilder", version: "0.0.28" };
class PhaseExtension {
}
class Step {
}
class Inputs {
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcG9uZW50X2J1aWxkZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvY29tcG9uZW50X2J1aWxkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLDBEQUEwRDtBQUUxRCw2QkFBNkI7QUFDN0IsNENBQTRDO0FBQzVDLHFFQUFnRTtBQUNoRSxtQ0FBcUQ7QUFDckQsaUNBQWlDOzs7O0FBRWpDLE1BQWEsZ0JBQWdCOzs7O0lBTTNCLFlBQVksS0FBb0IsRUFBRSxhQUFtQjs7OztRQUhyRCxhQUFRLEdBQVcsaUJBQWlCLENBQUM7UUFJbkMsSUFBSSxDQUFDLGFBQWEsR0FBRyxhQUFhLENBQUM7UUFDbkMsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7UUFFbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1lBQ2pDLEVBQUUsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQzdCO1FBRUQsSUFBSSxDQUFDLHdCQUF3QixHQUFHLElBQUksQ0FBQyw4QkFBOEIsRUFBRSxDQUFDO0lBQ3hFLENBQUM7SUFFTyw4QkFBOEI7UUFDcEMsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFnQixDQUFDO1FBRXRHLE9BQU8sVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQ2xDO1lBQ0UsR0FBRyxDQUFDO1lBQ0osQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEVBQUUsRUFBRTtTQUNkLENBQ0YsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLG9CQUFvQjtJQUM5QixDQUFDO0lBRU8sZUFBZSxDQUFDLGVBQW9CO1FBRTFDLElBQUksYUFBYSxDQUFDO1FBRWxCLFFBQVEsZUFBZSxDQUFDLElBQUksRUFBRTtZQUM1QixLQUFLLE9BQU87Z0JBQ1YsYUFBYSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxlQUFlLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztnQkFDNUUsTUFBTTtZQUNSLEtBQUssS0FBSztnQkFDUixhQUFhLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLGVBQWUsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7Z0JBQ3ZFLE1BQU07WUFDUixLQUFLLFNBQVM7Z0JBQ1osT0FBTyxJQUFJLHVCQUFlLENBQUMsZUFBZSxDQUFDLEdBQUcsRUFBRSxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDeEU7Z0JBQ0UsYUFBYSxHQUFHLEVBQUUsQ0FBQztTQUN0QjtRQUVELElBQUksZUFBZSxJQUFJLGFBQWEsRUFBRTtZQUNwQyxPQUFPLElBQUksWUFBWSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxhQUFhLENBQUMsQ0FBQztTQUN0SDtRQUVELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxlQUFvQixFQUFFLE9BQWU7UUFDN0QsSUFBSSxhQUFhLEdBQUcsZUFBZSxDQUFDLElBQUksQ0FBQztRQUV6QyxJQUFJLGVBQWUsQ0FBQyxJQUFJLEVBQUU7WUFDeEIsYUFBYSxHQUFHLEdBQUcsYUFBYSxJQUFJLGVBQWUsQ0FBQyxJQUFJLEVBQUUsQ0FBQztTQUM1RDtRQUVELElBQUksSUFBSSxDQUFDLGtCQUFrQixDQUFDLE9BQU8sRUFBRSxhQUFhLENBQUMsRUFBRTtZQUNuRCxvQkFBb0I7WUFDcEIsTUFBTSxRQUFRLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxhQUFhLEVBQUUsZUFBZSxDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDOUYsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUV0QywwQkFBMEI7WUFDMUIsTUFBTSxhQUFhLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxhQUFhLEVBQUUsZ0JBQWdCLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNwRyxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO1lBRTNDLElBQUksYUFBYSxHQUFHLFFBQVEsQ0FBQztZQUU3QixJQUFJLFFBQVEsQ0FBQyxvQkFBb0IsRUFBRTtnQkFDakMsTUFBTSxzQkFBc0IsR0FBRyxRQUFRLENBQUMsb0JBQW9CLENBQUMsZUFBZSxDQUFDLFFBQVEsQ0FBQyxDQUFDO2dCQUV2Riw2R0FBNkc7Z0JBQzdHLElBQUksc0JBQXNCLEVBQUU7b0JBQzFCLE1BQU0sa0JBQWtCLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxhQUFhLEVBQUUsc0JBQXNCLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztvQkFDL0csTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUM7b0JBRXhELE1BQU0sZ0JBQWdCLEdBQUcsZ0JBQWdCLENBQUMsZ0JBQW9DLENBQUM7b0JBRS9FLHlDQUF5QztvQkFDekMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixFQUFFLGFBQWEsQ0FBQyxDQUFDO2lCQUMxRDthQUNGO1lBRUQsSUFBSSxDQUFDLHNCQUFzQixDQUFDLGVBQWUsRUFBRSxhQUFhLENBQUMsQ0FBQztZQUU1RCxNQUFNLElBQUksR0FBRyxlQUFlLENBQUMsSUFBSSxDQUFDO1lBRWxDLE1BQU0sb0JBQW9CLEdBQUcsSUFBSSw2Q0FBb0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRSxJQUFJLENBQUMsQ0FBQztZQUUvSCxPQUFPO2dCQUNMLElBQUksRUFBRSxJQUFJO2dCQUNWLFFBQVEsRUFBRSxRQUFRLENBQUMsUUFBUTtnQkFDM0IsT0FBTyxFQUFFLG9CQUFvQixDQUFDLFdBQVc7Z0JBQ3pDLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQztnQkFDbkMsSUFBSSxFQUFFO29CQUNKLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUyxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUM7aUJBQ2pEO2FBQ0YsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVPLHNCQUFzQixDQUFDLGVBQW9CLEVBQUUsYUFBa0I7UUFDckUsSUFBSSxXQUFXLElBQUksZUFBZSxFQUFFO1lBQ2xDLElBQUksYUFBYSxHQUFHLGVBQWUsQ0FBQyxTQUFTLENBQUM7WUFDOUMsSUFBSSxhQUFhLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFTLEVBQUUsRUFBRTtnQkFDL0QsSUFBSSxXQUFXLEdBQVEsRUFBRSxDQUFDO2dCQUMxQixXQUFXLENBQUMsQ0FBQyxDQUFDLEdBQUc7b0JBQ2YsSUFBSSxFQUFFLFFBQVE7b0JBQ2QsS0FBSyxFQUFFLGFBQWEsQ0FBQyxDQUFDLENBQUM7aUJBQ3hCLENBQUM7Z0JBQ0YsT0FBTyxXQUFXLENBQUM7WUFDckIsQ0FBQyxDQUFDLENBQUM7WUFFSCxJQUFJLFdBQVcsSUFBSSxhQUFhLEVBQUU7Z0JBQ2hDLDZGQUE2RjtnQkFDN0YsSUFBSSxpQkFBaUIsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO2dCQUNuRCxhQUFhLENBQUMsU0FBUyxHQUFHLGFBQWEsQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBSztvQkFDdEUsaUZBQWlGO29CQUNqRixrR0FBa0c7b0JBQ2xHLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLElBQUksQ0FBQyxDQUFDO2dCQUMvRSxDQUFDLENBQUMsQ0FBQztnQkFDSCxhQUFhLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLE1BQU0sRUFBRSxDQUFDLEVBQUUsR0FBRyxhQUFhLENBQUMsQ0FBQzthQUNyRjtpQkFBTTtnQkFDTCxhQUFhLENBQUMsU0FBUyxHQUFHLGFBQWEsQ0FBQzthQUN6QztTQUVGO0lBQ0gsQ0FBQztJQUVPLGtCQUFrQixDQUFDLGdCQUFrQyxFQUFFLGFBQWtCO1FBQy9FLEtBQUssSUFBSSxlQUFlLElBQUksZ0JBQWdCLEVBQUU7WUFFNUMsNERBQTREO1lBQzVELElBQUksZUFBZSxDQUFDLElBQUksS0FBSyxVQUFVLEVBQUU7Z0JBQ3ZDLE1BQU0sVUFBVSxHQUFHLGFBQWEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsRUFBb0IsRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksS0FBSyxPQUFPLENBQUMsQ0FBQztnQkFDNUYsSUFBSSxVQUFVLEVBQUU7b0JBQ2QsOENBQThDO29CQUM5QyxVQUFVLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEdBQUcsZUFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDO2lCQUN6RDtxQkFBTTtvQkFDTCxvRUFBb0U7b0JBQ3BFLG1FQUFtRTtvQkFDbkUsTUFBTSxZQUFZLEdBQUcsRUFBRSxHQUFHLGVBQWUsRUFBRSxDQUFDO29CQUM1QyxZQUFZLENBQUMsSUFBSSxHQUFHLE9BQU8sQ0FBQztvQkFDNUIsYUFBYSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7aUJBQ3pDO2FBQ0Y7aUJBQU0sRUFBRSxrQ0FBa0M7Z0JBQ3pDLE1BQU0sUUFBUSxHQUFHLGFBQWEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsRUFBb0IsRUFBRSxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksS0FBSyxlQUFlLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3ZHLElBQUksUUFBUSxFQUFFO29CQUNaLFFBQVEsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEdBQUcsZUFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDO2lCQUMvQzthQUNGO1NBQ0Y7SUFDSCxDQUFDO0lBRU8sa0JBQWtCLENBQUMsT0FBZSxFQUFFLGFBQXFCO1FBQy9ELE1BQU0sWUFBWSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLGFBQWEsRUFBRSxlQUFlLENBQUMsQ0FBQztRQUN4RSxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxhQUFhLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUUxRSxNQUFNLGNBQWMsR0FBRyxFQUFFLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQ25ELE1BQU0sbUJBQW1CLEdBQUcsRUFBRSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUV6RCxNQUFNLGFBQWEsR0FBRyxJQUFJLFNBQVMsQ0FBQyxFQUNuQyxDQUFDLENBQUM7UUFDSCxhQUFhLENBQUMsUUFBUSxDQUFDO1lBQ3JCLFlBQVk7WUFDWixhQUFhO1NBQ2QsQ0FBQyxDQUFDO1FBRUgsT0FBTyxjQUFjLElBQUksbUJBQW1CLElBQUksYUFBYSxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQztJQUM5RSxDQUFDOztBQTNLSCw0Q0E0S0M7OztBQUVELE1BQU0sY0FBYztDQUduQjtBQUVELE1BQU0sSUFBSTtDQUtUO0FBRUQsTUFBTSxNQUFNO0NBRVgiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgaW1hZ2VidWlsZGVyIGZyb20gJ0Bhd3MtY2RrL2F3cy1pbWFnZWJ1aWxkZXInO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgeWFtbCBmcm9tICd5YW1sJztcbmltcG9ydCAqIGFzIFZhbGlkYXRvciBmcm9tICd5YW1sLXZhbGlkYXRvcic7XG5pbXBvcnQgeyBDb21wb25lbnRWZXJzaW9uQnVtcCB9IGZyb20gJy4vY29tcG9uZW50X3ZlcnNpb25fYnVtcCc7XG5pbXBvcnQgeyBDb21wb25lbnQsIEFybkNvbXBvbmVudFJlZiB9IGZyb20gJy4vaW5kZXgnO1xuaW1wb3J0ICogYXMgdXRpbHMgZnJvbSAnLi91dGlscyc7XG5cbmV4cG9ydCBjbGFzcyBDb21wb25lbnRCdWlsZGVyIHtcbiAgY29tcG9uZW50RGVwczogYW55W107XG4gIHNjb3BlOiBjZGsuQ29uc3RydWN0O1xuICBjYWNoZURpcjogc3RyaW5nID0gJ2NvbXBvbmVudF9jYWNoZSc7XG4gIGNvbXBvbmVudERlcGVuZGVuY2llc01hcDogeyBbbmFtZTogc3RyaW5nXTogQ29tcG9uZW50IH07XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGNvbXBvbmVudERlZnM6YW55W10pIHtcbiAgICB0aGlzLmNvbXBvbmVudERlcHMgPSBjb21wb25lbnREZWZzO1xuICAgIHRoaXMuc2NvcGUgPSBzY29wZTtcblxuICAgIGlmICghZnMuZXhpc3RzU3luYyh0aGlzLmNhY2hlRGlyKSkge1xuICAgICAgZnMubWtkaXJTeW5jKHRoaXMuY2FjaGVEaXIpO1xuICAgIH1cblxuICAgIHRoaXMuY29tcG9uZW50RGVwZW5kZW5jaWVzTWFwID0gdGhpcy5jcmVhdGVDb21wb25lbnREZXBlbmRlbmNpZXNNYXAoKTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlQ29tcG9uZW50RGVwZW5kZW5jaWVzTWFwKCk6IHsgW25hbWU6IHN0cmluZ106IENvbXBvbmVudCB9IHtcbiAgICBjb25zdCBjb21wb25lbnRzID0gdGhpcy5jb21wb25lbnREZXBzLm1hcChjID0+IHRoaXMuY3JlYXRlQ29tcG9uZW50KGMpKS5maWx0ZXIoYyA9PiBjKSBhcyBDb21wb25lbnRbXTtcblxuICAgIHJldHVybiBjb21wb25lbnRzLnJlZHVjZSgoYSwgZWwpID0+IChcbiAgICAgIHtcbiAgICAgICAgLi4uYSxcbiAgICAgICAgW2VsLm5hbWVdOiBlbCxcbiAgICAgIH1cbiAgICApLCB7fSk7IC8vIHJlbW92ZSB0aGUgcHJlZml4XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZUNvbXBvbmVudChjb21wb25lbnRDb25maWc6IGFueSk6IENvbXBvbmVudCB8IHVuZGVmaW5lZCB7XG5cbiAgICBsZXQgY29tcG9uZW50WWFtbDtcblxuICAgIHN3aXRjaCAoY29tcG9uZW50Q29uZmlnLnR5cGUpIHtcbiAgICAgIGNhc2UgJ2xvY2FsJzpcbiAgICAgICAgY29tcG9uZW50WWFtbCA9IHRoaXMuYXNzZW1ibGVDb21wb25lbnQoY29tcG9uZW50Q29uZmlnLCAnbG9jYWxfY29tcG9uZW50cycpO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgJ2dpdCc6XG4gICAgICAgIGNvbXBvbmVudFlhbWwgPSB0aGlzLmFzc2VtYmxlQ29tcG9uZW50KGNvbXBvbmVudENvbmZpZywgdGhpcy5jYWNoZURpcik7XG4gICAgICAgIGJyZWFrO1xuICAgICAgY2FzZSAnYXdzX2Fybic6XG4gICAgICAgIHJldHVybiBuZXcgQXJuQ29tcG9uZW50UmVmKGNvbXBvbmVudENvbmZpZy5hcm4sIGNvbXBvbmVudENvbmZpZy5uYW1lKTtcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIGNvbXBvbmVudFlhbWwgPSAnJztcbiAgICB9XG5cbiAgICBpZiAoY29tcG9uZW50Q29uZmlnICYmIGNvbXBvbmVudFlhbWwpIHtcbiAgICAgIHJldHVybiBuZXcgaW1hZ2VidWlsZGVyLkNmbkNvbXBvbmVudCh0aGlzLnNjb3BlLCBgJHt1dGlscy5wYXNjYWxpemUoY29tcG9uZW50Q29uZmlnLm5hbWUpfUNvbXBvbmVudGAsIGNvbXBvbmVudFlhbWwpO1xuICAgIH1cblxuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cblxuICBwcml2YXRlIGFzc2VtYmxlQ29tcG9uZW50KGNvbXBvbmVudENvbmZpZzogYW55LCBiYXNlZGlyOiBzdHJpbmcpOiBhbnkge1xuICAgIGxldCBjb21wb25lbnRQYXRoID0gY29tcG9uZW50Q29uZmlnLm5hbWU7XG5cbiAgICBpZiAoY29tcG9uZW50Q29uZmlnLnBhdGgpIHtcbiAgICAgIGNvbXBvbmVudFBhdGggPSBgJHtjb21wb25lbnRQYXRofS8ke2NvbXBvbmVudENvbmZpZy5wYXRofWA7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuY2hlY2tDb21wb25lbnRZYW1sKGJhc2VkaXIsIGNvbXBvbmVudFBhdGgpKSB7XG4gICAgICAvLyBHZXQgbWV0YWRhdGEgWUFNTFxuICAgICAgY29uc3QgbWV0YWRhdGEgPSBmcy5yZWFkRmlsZVN5bmMocGF0aC5qb2luKGJhc2VkaXIsIGNvbXBvbmVudFBhdGgsICdtZXRhZGF0YS55YW1sJyksICd1dGYtOCcpO1xuICAgICAgY29uc3QgbWV0YXlhbWwgPSB5YW1sLnBhcnNlKG1ldGFkYXRhKTtcblxuICAgICAgLy8gR2V0IGJhc2UgY29tcG9uZW50LnlhbWxcbiAgICAgIGNvbnN0IGJhc2VDb21wb25lbnQgPSBmcy5yZWFkRmlsZVN5bmMocGF0aC5qb2luKGJhc2VkaXIsIGNvbXBvbmVudFBhdGgsICdjb21wb25lbnQueWFtbCcpLCAndXRmLTgnKTtcbiAgICAgIGNvbnN0IGJhc2VZYW1sID0geWFtbC5wYXJzZShiYXNlQ29tcG9uZW50KTtcblxuICAgICAgbGV0IGFzc2VtYmxlZFlhbWwgPSBiYXNlWWFtbDtcblxuICAgICAgaWYgKG1ldGF5YW1sLmRpc3RyaWJ1dGlvbl9jb25maWdzKSB7XG4gICAgICAgIGNvbnN0IGRpc3RyaWJ1dGlvbmNvbmZpZ2NvbXAgPSBtZXRheWFtbC5kaXN0cmlidXRpb25fY29uZmlnc1tjb21wb25lbnRDb25maWcucGxhdGZvcm1dO1xuXG4gICAgICAgIC8vIGlmIHRoZXJlIGlzIGEgcGxhdGZvcm0gc3BlY2lmaWMgY29tcG9uZW50IHlhbWwgc3BlY2lmaWVkLCBsb2FkIGl0IGFuZCBhZGQgc3RlcHMgdG8gdGhlIGFzc2VtYmxlZCBjb21wb25lbnRcbiAgICAgICAgaWYgKGRpc3RyaWJ1dGlvbmNvbmZpZ2NvbXApIHtcbiAgICAgICAgICBjb25zdCBjb21wb25lbnRFeHRlbnNpb24gPSBmcy5yZWFkRmlsZVN5bmMocGF0aC5qb2luKGJhc2VkaXIsIGNvbXBvbmVudFBhdGgsIGRpc3RyaWJ1dGlvbmNvbmZpZ2NvbXApLCAndXRmLTgnKTtcbiAgICAgICAgICBjb25zdCBjb21wb25lbnRFeHRZYW1sID0geWFtbC5wYXJzZShjb21wb25lbnRFeHRlbnNpb24pO1xuXG4gICAgICAgICAgY29uc3QgcGhhc2VfZXh0ZW5zaW9ucyA9IGNvbXBvbmVudEV4dFlhbWwucGhhc2VfZXh0ZW5zaW9ucyBhcyBQaGFzZUV4dGVuc2lvbltdO1xuXG4gICAgICAgICAgLy8gQWRkIHBoYXNlIGV4dGVuc2lvbnMgdG8gYmFzZSBjb21wb25lbnRcbiAgICAgICAgICB0aGlzLmFkZFBoYXNlRXh0ZW5zaW9ucyhwaGFzZV9leHRlbnNpb25zLCBhc3NlbWJsZWRZYW1sKTtcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICB0aGlzLmFkZERlcGVuZGVuY3lDb25zdGFudHMoY29tcG9uZW50Q29uZmlnLCBhc3NlbWJsZWRZYW1sKTtcblxuICAgICAgY29uc3QgbmFtZSA9IGNvbXBvbmVudENvbmZpZy5uYW1lO1xuXG4gICAgICBjb25zdCBjb21wb25lbnRWZXJzaW9uQnVtcCA9IG5ldyBDb21wb25lbnRWZXJzaW9uQnVtcCh0aGlzLnNjb3BlLCBgJHt1dGlscy5wYXNjYWxpemUoY29tcG9uZW50Q29uZmlnLm5hbWUpfVZlcnNpb25CdW1wYCwgbmFtZSk7XG5cbiAgICAgIHJldHVybiB7XG4gICAgICAgIG5hbWU6IG5hbWUsXG4gICAgICAgIHBsYXRmb3JtOiBtZXRheWFtbC5wbGF0Zm9ybSxcbiAgICAgICAgdmVyc2lvbjogY29tcG9uZW50VmVyc2lvbkJ1bXAubmV4dFZlcnNpb24sXG4gICAgICAgIGRhdGE6IHlhbWwuc3RyaW5naWZ5KGFzc2VtYmxlZFlhbWwpLFxuICAgICAgICB0YWdzOiB7XG4gICAgICAgICAgU2hvcnROYW1lOiB1dGlscy5wYXNjYWxpemUoY29tcG9uZW50Q29uZmlnLm5hbWUpLFxuICAgICAgICB9LFxuICAgICAgfTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGFkZERlcGVuZGVuY3lDb25zdGFudHMoY29tcG9uZW50Q29uZmlnOiBhbnksIGFzc2VtYmxlZFlhbWw6IGFueSkge1xuICAgIGlmICgnY29uc3RhbnRzJyBpbiBjb21wb25lbnRDb25maWcpIHtcbiAgICAgIGxldCBjb25zdGFudHNEaWN0ID0gY29tcG9uZW50Q29uZmlnLmNvbnN0YW50cztcbiAgICAgIGxldCBjb21wQ29uc3RhbnRzID0gT2JqZWN0LmtleXMoY29uc3RhbnRzRGljdCkubWFwKChrOiBzdHJpbmcpID0+IHtcbiAgICAgICAgbGV0IGNvbnN0YW50T2JqOiBhbnkgPSB7fTtcbiAgICAgICAgY29uc3RhbnRPYmpba10gPSB7XG4gICAgICAgICAgdHlwZTogJ3N0cmluZycsXG4gICAgICAgICAgdmFsdWU6IGNvbnN0YW50c0RpY3Rba10sXG4gICAgICAgIH07XG4gICAgICAgIHJldHVybiBjb25zdGFudE9iajtcbiAgICAgIH0pO1xuXG4gICAgICBpZiAoJ2NvbnN0YW50cycgaW4gYXNzZW1ibGVkWWFtbCkge1xuICAgICAgICAvLyBPdmVycmlkZSB0aGUga2V5cyBmcm9tIHRoZSBjb21wb25lbnQgaXRzZWxmIHdpdGggdGhlIG9uZXMgZnJvbSB0aGUgY29tcG9uZW50IGRlcGVuZGVuY2llcy5cbiAgICAgICAgbGV0IGNvbXBDb25zdGFudHNLZXlzID0gT2JqZWN0LmtleXMoY29uc3RhbnRzRGljdCk7XG4gICAgICAgIGFzc2VtYmxlZFlhbWwuY29uc3RhbnRzID0gYXNzZW1ibGVkWWFtbC5jb25zdGFudHMuZmlsdGVyKGZ1bmN0aW9uIChjOmFueSkge1xuICAgICAgICAgIC8vIFNlZSBpZiB0aGVpciBpcyBhbiBpbnRlcnNlY3Rpb24gYmV0d2VlbiB0aGUga2V5cyBmcm9tIGNvbXBvbmVudCBkZXBlbmRlbmN5IGFuZFxuICAgICAgICAgIC8vIHRoZSBvbmVzIGRlZmluZWQgaW4gdGhlIG9yaWdpbmFsIGNvbXBvbmVudC55YW1sIGZpbGUgYW5kIHJlbW92ZSB0aG9zZSBmcm9tIGNvbXBvbmVudC55YW1sIGZpbGUuXG4gICAgICAgICAgcmV0dXJuIE9iamVjdC5rZXlzKGMpLmZpbHRlcihkID0+IGNvbXBDb25zdGFudHNLZXlzLmluY2x1ZGVzKGQpKS5sZW5ndGggPT0gMDtcbiAgICAgICAgfSk7XG4gICAgICAgIGFzc2VtYmxlZFlhbWwuY29uc3RhbnRzLnNwbGljZShhc3NlbWJsZWRZYW1sLmNvbnN0YW50cy5sZW5ndGgsIDAsIC4uLmNvbXBDb25zdGFudHMpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgYXNzZW1ibGVkWWFtbC5jb25zdGFudHMgPSBjb21wQ29uc3RhbnRzO1xuICAgICAgfVxuXG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBhZGRQaGFzZUV4dGVuc2lvbnMocGhhc2VfZXh0ZW5zaW9uczogUGhhc2VFeHRlbnNpb25bXSwgYXNzZW1ibGVkWWFtbDogYW55KSB7XG4gICAgZm9yIChsZXQgcGhhc2VfZXh0ZW5zaW9uIG9mIHBoYXNlX2V4dGVuc2lvbnMpIHtcblxuICAgICAgLy8gRm9yIHByZWJ1aWxkIHBoYXNlcywgY29weSB0aGUgc3RlcHMgYmVmb3JlIHRoZSBidWlsZCBzdGVwXG4gICAgICBpZiAocGhhc2VfZXh0ZW5zaW9uLm5hbWUgPT09ICdwcmVidWlsZCcpIHtcbiAgICAgICAgY29uc3QgYnVpbGRwaGFzZSA9IGFzc2VtYmxlZFlhbWwucGhhc2VzLmZpbmQoKGVsOiB7IG5hbWU6IHN0cmluZyB9KSA9PiBlbC5uYW1lID09PSAnYnVpbGQnKTtcbiAgICAgICAgaWYgKGJ1aWxkcGhhc2UpIHtcbiAgICAgICAgICAvLyBJbnNlcnQgdGhlIHByZWJ1aWxkIGJlZm9yZSB0aGUgYnVpbGQgc3RlcHMuXG4gICAgICAgICAgYnVpbGRwaGFzZS5zdGVwcy5zcGxpY2UoMCwgMCwgLi4ucGhhc2VfZXh0ZW5zaW9uLnN0ZXBzKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAvLyBJbiBjYXNlIHRoZSBvcmlnaW5hbCBjb21wb25lbiB5YW1sIGRvZXNuJ3QgaW5jbHVkZSBhIGJ1aWxkIHBoYXNlLFxuICAgICAgICAgIC8vIGNvcHkgdGhlIHBoYXNlIGZyb20gdGhlIGV4dGVuc2lvbiBkaXJlY3RseSBpbiB0aGUgYXNzZW1ibGVkIFlBTUxcbiAgICAgICAgICBjb25zdCBjb3BpZWRfcGhhc2UgPSB7IC4uLnBoYXNlX2V4dGVuc2lvbiB9O1xuICAgICAgICAgIGNvcGllZF9waGFzZS5uYW1lID0gJ2J1aWxkJztcbiAgICAgICAgICBhc3NlbWJsZWRZYW1sLnBoYXNlcy5wdXNoKGNvcGllZF9waGFzZSk7XG4gICAgICAgIH1cbiAgICAgIH0gZWxzZSB7IC8vIEVsc2UganVzdCBjb3B5IHRoZSBidWlsZCBwaGFzZS5cbiAgICAgICAgY29uc3QgZXh0cGhhc2UgPSBhc3NlbWJsZWRZYW1sLnBoYXNlcy5maW5kKChlbDogeyBuYW1lOiBzdHJpbmcgfSkgPT4gZWwubmFtZSA9PT0gcGhhc2VfZXh0ZW5zaW9uLm5hbWUpO1xuICAgICAgICBpZiAoZXh0cGhhc2UpIHtcbiAgICAgICAgICBleHRwaGFzZS5zdGVwcy5wdXNoKC4uLnBoYXNlX2V4dGVuc2lvbi5zdGVwcyk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGNoZWNrQ29tcG9uZW50WWFtbChiYXNlRGlyOiBzdHJpbmcsIGNvbXBvbmVudFBhdGg6IHN0cmluZykge1xuICAgIGNvbnN0IG1ldGFkYXRhZmlsZSA9IHBhdGguam9pbihiYXNlRGlyLCBjb21wb25lbnRQYXRoLCAnbWV0YWRhdGEueWFtbCcpO1xuICAgIGNvbnN0IGNvbXBvbmVudGZpbGUgPSBwYXRoLmpvaW4oYmFzZURpciwgY29tcG9uZW50UGF0aCwgJ2NvbXBvbmVudC55YW1sJyk7XG5cbiAgICBjb25zdCBtZXRhZGF0YUV4aXN0cyA9IGZzLmV4aXN0c1N5bmMobWV0YWRhdGFmaWxlKTtcbiAgICBjb25zdCBjb21wb25lbnRZYW1sRXhpc3RzID0gZnMuZXhpc3RzU3luYyhjb21wb25lbnRmaWxlKTtcblxuICAgIGNvbnN0IHlhbWxWYWxpZGF0b3IgPSBuZXcgVmFsaWRhdG9yKHtcbiAgICB9KTtcbiAgICB5YW1sVmFsaWRhdG9yLnZhbGlkYXRlKFtcbiAgICAgIG1ldGFkYXRhZmlsZSxcbiAgICAgIGNvbXBvbmVudGZpbGUsXG4gICAgXSk7XG5cbiAgICByZXR1cm4gbWV0YWRhdGFFeGlzdHMgJiYgY29tcG9uZW50WWFtbEV4aXN0cyAmJiB5YW1sVmFsaWRhdG9yLnJlcG9ydCgpID09IDA7XG4gIH1cbn1cblxuY2xhc3MgUGhhc2VFeHRlbnNpb24ge1xuICBuYW1lITogc3RyaW5nO1xuICBzdGVwcyE6IFN0ZXBbXVxufVxuXG5jbGFzcyBTdGVwIHtcbiAgbmFtZSE6IHN0cmluZztcbiAgYWN0aW9uITogc3RyaW5nXG4gIGlucHV0cyE6IElucHV0c1xuXG59XG5cbmNsYXNzIElucHV0cyB7XG4gIGNvbW1hbmRzITogc3RyaW5nW11cbn0iXX0=