# (C) Copyright IBM Corp. 2021.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except in compliance with
# the License. You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
# an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
# specific language governing permissions and limitations under the License.

from typing import Dict, List, TypeVar, Type, Optional
from .models import Feature
from .models import SegmentRules
from .models import Segment
from app_configuration.core.internal import Logger
from app_configuration.core import BaseRequest
from .internal.utils.file_manager import FileManager
from .internal.utils.validators import Validators
from .internal.utils.metering import Metering
from .internal.utils.socket import Socket
from .internal.utils.url_builder import URLBuilder
from threading import Timer
from urllib.parse import urlparse
import json

try:
    import thread
except ImportError:
    import _thread as thread


class FeatureHandler:
    __instance = None
    T = TypeVar('T')

    @staticmethod
    def get_instance():
        """ Static access method. """
        if FeatureHandler.__instance is None:
            return FeatureHandler()
        return FeatureHandler.__instance

    def __init__(self):
        """ Virtually private constructor. """
        self.__retryCount = 3
        if FeatureHandler.__instance is not None:
            raise Exception("This class is a FeatureHandler singleton!")
        else:
            self.__collectionId = ''
            self.__clientAttributes = dict()
            self.__apikey = ''
            self.__guid = ''
            self.__isInitialized = False
            self.__is_initialized_feature = False
            self.__featuresUpdateListener = None
            self.__featureMap = dict()
            self.__segmentMap = dict()
            self.__live_feature_update_enabled = True
            FeatureHandler.__instance = self
            self.__retryCount = 3
            self.__retry_interval = 600000
            self.__feature_file = None
            self.__on_socket_retry = False

    def init(self, collection_id: str,
             apikey: str,
             guid: str,
             region: str,
             override_server_host=str,
             live_feature_update_enabled: Optional[bool] = True,
             feature_file: Optional[str] = None):

        self.__live_feature_update_enabled = live_feature_update_enabled
        self.__collectionId = collection_id
        self.__apikey = apikey
        self.__guid = guid
        URLBuilder.init_with_collection_id(collection_id=collection_id,
                                           guid=guid,
                                           region=region,
                                           override_server_host=override_server_host)
        self.__featureMap = dict()
        self.__segmentMap = dict()
        self.__feature_file = feature_file
        self.__live_feature_update_enabled = live_feature_update_enabled
        Metering.get_instance().set_metering_url(URLBuilder.get_metering_url(), self.__apikey)
        self.__isInitialized = True

    def load_data(self):
        if self.__feature_file:
            self.__get_file_data(self.__feature_file)
        self.__loadFeatures()
        if self.__live_feature_update_enabled:
            thread.start_new_thread(self.__fetch_features_data, ())

    def register_features_update_listener(self, listener):
        if callable(listener):
            if self.__isInitialized:
                self.__featuresUpdateListener = listener
            else:
                Logger.error('Invalid action in FeatureHandler. '
                             'This action can be performed only after a successful initialization.'
                             ' Please check the initialization section for errors.')
        else:
            Logger.error('Invalid action in FeatureHandler. Should be a method/function')

    def set_client_attributes(self, attributes: dict = None):
        if dict is not None:
            self.__clientAttributes = attributes

    def get_features(self) -> Dict[str, Feature]:
        return self.__featureMap

    def get_feature(self, feature_id: str):
        if feature_id in self.__featureMap:
            return self.__featureMap.get(feature_id)
        else:
            self.__loadFeatures()
            if feature_id in self.__featureMap:
                return self.__featureMap.get(feature_id)
            else:
                return None

    def __fetch_features_data(self):
        if self.__isInitialized:
            self.__fetch_from_api()
            self.__start_web_socket()

    def __start_web_socket(self):
        headers = {
            'Authorization': self.__apikey
        }
        _ = Socket(
            url=URLBuilder.get_web_socket_url(),
            headers=headers,
            callback=self.__on_web_socket_callback
        )

    def __get_file_data(self, file_path: str):
        data = FileManager.read_files(file_path=file_path)
        if data is not None:
            self.__write_to_file(json=data)

    def __loadFeatures(self):
        all_feature: dict = FileManager.read_files()
        if all_feature:
            if 'features' in all_feature:
                self.__featureMap = dict()
                try:
                    all_feature_list: List = all_feature.get('features')
                    for i in range(0, len(all_feature_list)):
                        feature: dict = all_feature_list[i]
                        feature_obj = Feature(feature)
                        self.__featureMap[feature_obj.get_feature_id()] = feature_obj
                except Exception as err:
                    Logger.debug(err)

            if 'segments' in all_feature:
                self.__segmentMap = dict()
                try:
                    segment_list: List = all_feature.get('segments')
                    for i in range(0, len(segment_list)):
                        segment: dict = segment_list[i]
                        segment_obj = Segment(segment)
                        self.__segmentMap[segment_obj.get_segment_id()] = segment_obj
                except Exception as err:
                    Logger.debug(err)

    def record_valuation(self, feature_id):
        Metering.get_instance().add_metering(
            guid=self.__guid,
            feature_id=feature_id,
            collection_id=self.__collectionId
        )

    def feature_evaluation(self, feature: Feature, request_object, type_v: Type[T]) -> T:

        if len(self.__clientAttributes) <= 0 and request_object is None:
            return Validators.convertValue(feature.get_enabled_value(), type_v)

        rules_map = self.__parse_rules(feature.get_segment_rules())

        for i in range(1, len(rules_map) + 1):
            segment_rule = rules_map[i]
            if not (segment_rule is None):
                for level in range(0, len(segment_rule.get_rules())):
                    try:
                        rule: dict = segment_rule.get_rules()[level]
                        segments: List = rule.get('segments')
                        for inner_level in range(0, len(segments)):
                            segment_key = segments[inner_level]
                            if self.__evaluate_segment(segment_key, request_object):
                                if segment_rule.get_value() == "$default":
                                    return Validators.convertValue(feature.get_enabled_value(), type_v)
                                else:
                                    return Validators.convertValue(segment_rule.get_value(), type_v)

                    except Exception as err:
                        Logger.debug(err)

        return Validators.convertValue(feature.get_enabled_value(), type_v)

    def __fetch_queries(self, request) -> dict:
        query_map = dict()

        ## BaseHTTPRequestHandler request
        try:
            queries = urlparse(request.path).query
            query_map = dict(single.split("=") for single in queries.split("&"))
            return query_map
        except Exception as error:
            Logger.error(f'Error while getting request query data {error}')

        ## Django http request
        try:
            query_map = request.GET
            if not query_map:
                query_map = request.POST
            return query_map
        except Exception as error:
            Logger.error(f'Error while getting request query data {error}')

        ## Flask http request
        try:
            query_map = request.args
            return query_map
        except Exception as error:
            Logger.error(f'Error while getting request query data {error}')

        return query_map

    def __fetch_headers(self, request) -> dict:
        header_map = dict()

        try:
            header_map = request.headers
            return header_map
        except Exception as error:
            Logger.error(f'Error while getting request header data {error}')

        return header_map

    def __fetch_body(self, request) -> dict:
        body_map = dict()

        ## BaseHTTPRequestHandler
        try:
            length = int(request.headers['content-length'])
            body_map = json.loads(request.rfile.read(length))
            return body_map
        except Exception as error:
            Logger.error(f'Error while getting request body data {error}')

        ## Django
        try:
            body_map = json.loads(request.body)
            return body_map
        except Exception as error:
            Logger.error(f'Error while getting request body data {error}')

        ## Flask
        try:
            body_map = request.json
            return body_map
        except Exception as error:
            Logger.error(f'Error while getting request body data {error}')

        return body_map

    def __evaluate_segment(self, segment_key: str, request_object=None) -> bool:

        query_map = dict()
        header_map = dict()
        body_map = dict()

        if request_object is not None:

            query_map = self.__fetch_queries(request_object)
            header_map = self.__fetch_headers(request_object)

            try:
                content_type = request_object.headers['content-type']
                if content_type == 'application/json':
                    body_map = self.__fetch_body(request_object)
            except Exception as error:
                Logger.error(f'Error while getting request body data {error}')

        if segment_key in self.__segmentMap:
            segment: Segment = self.__segmentMap[segment_key]
            return segment.evaluateRule(self.__clientAttributes, query_map, header_map, body_map)
        return False

    def __parse_rules(self, segment_rules: List) -> dict:
        rule_map = dict()
        for i in range(0, len(segment_rules)):
            try:
                rules = segment_rules[i]
                rules_obj = SegmentRules(rules)
                rule_map[rules_obj.get_order()] = rules_obj
            except Exception as err:
                Logger.debug(err)
        return rule_map

    def __write_to_file(self, json: dict):
        FileManager.store_files(json)
        self.__loadFeatures()
        if self.__featuresUpdateListener and callable(self.__featuresUpdateListener):
            self.__featuresUpdateListener()

    def __fetch_from_api(self):
        if self.__isInitialized:
            self.__retryCount -= 1
            config_url = URLBuilder.get_config_url()
            service = BaseRequest()
            header = {
                'Authorization': self.__apikey,
                'Content-Type': 'application/json'
            }

            request = service.prepare_request(
                method='GET',
                url=config_url,
                headers=header
            )

            response = service.send(request)
            status_code = response.get_status_code()

            if 200 <= status_code <= 299:
                response_data = response.get_result()
                try:
                    response_data = dict(response_data)
                    if response_data:
                        self.__write_to_file(response_data)
                except:
                    if response_data:
                        self.__write_to_file(response_data)
            else:
                if self.__retryCount > 0:
                    self.__fetch_from_api()
                else:
                    self.__retryCount = 3
                    Timer(self.__retry_interval, lambda: self.__fetch_from_api()).start()
        else:
            Logger.debug("fetchFromAPI() - Feature SDK not initialized with call to initialize()")

    def __on_web_socket_callback(self, message=None, error_state=None, closed_state=None, open_state=None):
        if message:
            self.__fetch_from_api()
            Logger.debug(f'Received message from socket {message}')
        elif error_state:
            Logger.debug(f'Received error from socket {error_state}')
        elif closed_state:
            Logger.debug(f'Received close connection from socket')
            self.__on_socket_retry = True
            Timer(self.__retry_interval, lambda: self.__start_web_socket()).start()
        elif open_state:
            if self.__on_socket_retry:
                self.__on_socket_retry = False
                self.__fetch_from_api()
            Logger.debug(f'Received opened connection from socket')
        else:
            Logger.debug('Unknown Error')
