"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lambda = require("@aws-cdk/aws-lambda");
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const builder_1 = require("./builder");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        const entry = findEntry(id, props.entry);
        const handler = props.handler || 'handler';
        const buildDir = props.buildDir || path.join(path.dirname(entry), '.build');
        const handlerDir = path.join(buildDir, crypto.createHash('sha256').update(entry).digest('hex'));
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = props.runtime || defaultRunTime;
        // Build with Parcel
        const builder = new builder_1.Builder({
            entry,
            outDir: handlerDir,
            global: handler,
            minify: props.minify,
            sourceMaps: props.sourceMaps,
            cacheDir: props.cacheDir,
            nodeVersion: extractVersion(runtime),
        });
        builder.build();
        super(scope, id, {
            ...props,
            runtime,
            code: lambda.Code.fromAsset(handlerDir),
            handler: `index.${handler}`,
        });
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(js|ts)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
/**
 * Extracts the version from the runtime
 */
function extractVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        return undefined;
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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